/*
 * Copyright 2005 Tridium, Inc. All Rights Reserved.
 */
package appliance.ui;

import java.text.*;
import java.util.*;
import javax.baja.sys.*;
import javax.baja.file.*;
import javax.baja.naming.*;
import javax.baja.security.*;
import javax.baja.gx.*;
import javax.baja.util.*;
import javax.baja.ui.*;
import javax.baja.ui.enums.*;
import javax.baja.ui.pane.*;
import javax.baja.ui.text.*;
import javax.baja.workbench.*;
import javax.baja.workbench.fieldeditor.*;
import javax.baja.platform.*;

/**
 * PlatformUtil used the javax.baja.platform API to implement
 * the "Open Offline" and "Install Site" commands.  See the
 * openOffline() and installSite() methods for more details.
 *
 * @author    Brian Frank
 * @creation  26 Feb 05
 * @version   $Revision$ $Date$
 * @since     Baja 1.0
 */
public class PlatformUtil
{ 

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////
  
  /**
   * Constructor with parent shell.
   */
  public PlatformUtil(BWbShell shell)
  {
    this.shell = shell;
  }

////////////////////////////////////////////////////////////////
// Open Offline 
////////////////////////////////////////////////////////////////
  
  /**
   * Open offline is used to boot up and connect to a station
   * running on the local machine using the local platform daemon:
   *   1) discover our local stations
   *   2) prompt user to choose local station and for daemon credentials
   *   3) if necessary shutdown running stations
   *   4) startup the chosen station
   *   5) connect to it using fox
   */
  public void openOffline()
    throws Exception
  { 
    // check for local stations                                       
    LocalStation[] locals = getLocalStations();
    if (locals.length == 0)
    {
      BDialog.error(shell, "No stations found on local machine");
      return;
    }
    
    // build a drop down list of local stations
    BListDropDown localField = new BListDropDown();
    for(int i=0; i<locals.length; ++i)
      localField.getList().addItem(stationIcon, locals[i]);
    localField.setSelectedIndex(0);
    
    // we have to input username/password for local daemon
    BTextField user = new BTextField(lastLocalUser, 20);
    BTextField pass = new BTextField(lastLocalPass, 20); 
    pass.setRenderer(new PasswordRenderer());
       
    // prompt user to select which station is desired
    BGridPane pane = new BGridPane(2);
    pane.add("a", new BLabel(lex.getText("openOffline.select")));
    pane.add("b", localField);    
    pane.add("c", new BLabel(lex.getText("openOffline.user")));
    pane.add("d", user);
    pane.add("e", new BLabel(lex.getText("openOffline.password")));
    pane.add("f", pass);
    int r = BDialog.open(shell, lex.getText("openOffline.label"), new BBorderPane(pane), BDialog.OK_CANCEL, BDialog.QUESTION_ICON);
    if (r != BDialog.OK) return;
    
    // get the station the user selected
    LocalStation local = (LocalStation)localField.getSelectedItem(); 

    // cache local user and password
    lastLocalUser = user.getText();
    lastLocalPass = pass.getText();
    
    // get PlatformDaemon instance with appropiate credentials                     
    BUsernameAndPassword credentials;
    credentials = new BUsernameAndPassword(user.getText(), pass.getText());
    plat = PlatformDaemon.make(BLocalHost.INSTANCE, 3011, credentials);
    
    // check if this station is already running
    RemoteStation station = plat.getStationManager().getStation(local.name);
    
    // if not running, then stop all the others, and start selected station
    if (station.getStatus() != BStationStatus.running)
    {
      // first make sure that no other stations are running
      new StopAllStations().execute();
    
      // then start up the selected station
      new StartStation(local.name).execute();
    }
    
    // now connect
    station = plat.getStationManager().getStation(local.name);
    if (station.getStatus() == BStationStatus.running)
    {
      shell.hyperlink(BOrd.make(station.getFoxOrd(), "station:|slot:/App"));
    }
  }

////////////////////////////////////////////////////////////////
// Install Site 
////////////////////////////////////////////////////////////////

  /**
   * Install site is used to install a local station database to
   * a remote box via the remote platform daemon:
   *   1) discover local station databases
   *   2) prompt for local db, remote address, and remote daemon credentials
   *   3) stop all remote stations
   *   4) optionally backup the current station to a local directory
   *   5) delete the existing station
   *   6) create station by copying local directory to remote machine
   *   7) start the station
   *   8) connect to it via fox
   *
   * Note: this code doesn't do anything to guarantee that the
   *  correct modules are installed.  It assumes the station already
   *  has the required modules pre-installed.
   */
  public void installSite()
    throws Exception
  {
    // check for local stations                                       
    LocalStation[] locals = getLocalStations();
    if (locals.length == 0)
    {
      BDialog.error(shell, "No stations found on local machine");
      return;
    }

    // build a drop down list of local stations
    BListDropDown localField = new BListDropDown();
    for(int i=0; i<locals.length; ++i)
      localField.getList().addItem(stationIcon, locals[i]);
    localField.setSelectedIndex(0); 
    
    // build text field for remote IP address, but as a 
    // convenience provide a drop down with all the cached hosts
    BTextDropDown remoteField = new BTextDropDown(lastRemoteAddr, 18, true);
    BHost[] hosts = BHost.getRemoteHosts();
    for(int i=0; i<hosts.length; ++i)
      remoteField.getList().addItem(hosts[i].getHostname());    
    
    // we have to input username/password for local daemon
    BTextField user = new BTextField(lastRemoteUser, 20);
    BTextField pass = new BTextField(lastRemotePass, 20); 
    pass.setRenderer(new PasswordRenderer());      
    
    // prompt user to select which station is desired
    BGridPane pane = new BGridPane(2);
    pane.add("a", new BLabel(lex.getText("installSite.local")));
    pane.add("b", localField);    
    pane.add("c", new BLabel(lex.getText("installSite.remote")));
    pane.add("d", remoteField);    
    pane.add("e", new BLabel(lex.getText("installSite.user")));
    pane.add("f", user);
    pane.add("g", new BLabel(lex.getText("installSite.password")));
    pane.add("h", pass);                
    int r = BDialog.open(shell, lex.getText("installSite.label"), new BBorderPane(pane), BDialog.OK_CANCEL, BDialog.QUESTION_ICON);
    if (r != BDialog.OK) return;
    
    // get the local station the user selected
    LocalStation local = (LocalStation)localField.getSelectedItem();
    
    // get remote IP address and log info and cache for next time
    String remoteAddr = lastRemoteAddr = remoteField.getText();
    String remoteUser = lastRemoteUser = user.getText();
    String remotePass = lastRemotePass = pass.getText();

    // get PlatformDaemon instance with appropiate credentials                     
    plat = PlatformDaemon.make(remoteAddr, 3011, user.getText(), pass.getText());
    
    // first stop all the stations
    new StopAllStations().execute();
    
    // second get the existing installed station(s)
    RemoteStation[] existing = plat.getStationManager().getAllStations();
    if (existing.length > 0)
    {                                             
      // ask the user if they wish to perform a backup; under
      // normal circumstances if this is Jace there should only
      // be one station, so assume the first is the active station           
      RemoteStation old = existing[0];
      
      // create a unique directory to backup to
      SimpleDateFormat format = new SimpleDateFormat("yyMMdd_HHmm");
      format.setTimeZone(TimeZone.getDefault());
      String ts = format.format(new Date());
      String dir = "!backups/" + old.getName() + "_" + ts;
      
      // ask the user to backup or to skip it                 
      String msg = "Backup existing station: " + old.getName() + " to\n" +
                   dir + "?";
      r = BDialog.confirm(shell, msg);
      
      // if the user wants a backup then do it
      if (r == BDialog.YES)
      {                          
        BDirectory d = BFileSystem.INSTANCE.makeDir(new FilePath(dir));
        new BackupStation(old, d).execute();
      }
      
      // now nuke it so we start off with a clean slate
      new DeleteStation(old).execute();
    }                                                   
    
    // TODO at this point we might need to configure auto-start
    
    // create the station by copying from a local directory (this does 
    // NOT do anything to make sure the required modules are installed)
    new CreateStation(local.dir, local.name).execute();      

    if (!plat.getStationManager().getStation(local.name).canStart())
    {
      // reboot the host if the station can't be started once stopped
      new RebootHost().execute();
    }
    new StartStation(local.name).execute();
    
    // now connect
    RemoteStation station = plat.getStationManager().getStation(local.name);
    shell.hyperlink(station.getFoxOrd());
  }               
      
////////////////////////////////////////////////////////////////
// PlatformOperation
////////////////////////////////////////////////////////////////
  
  /**
   * PlatformOperation is the base class for the various platform
   * operations we run on a background thread for cancelability.
   */
  abstract class PlatformOperation 
    extends Thread
    implements IPlatformOperationListener
  { 
    /**
     * Run a platform operation on a background thread while we
     * display a dialog allowing the user to cancel the task.
     */
    public void execute()
      throws Exception
    {          
      // build dialog status pane
      BTextEditorPane statusPane = new BTextEditorPane("", 10, 80, false);
      this.status = statusPane.getEditor();
      
      // description title
      BLabel descr = new BLabel(clockIcon, description());            
      descr.setHalign(BHalign.left);
      
      // build displayPane -> description + status
      BEdgePane displayPane = new BEdgePane();
      displayPane.setTop(new BBorderPane(descr, 0, 0, 10, 0));
      displayPane.setCenter(statusPane);
      
      // build buttonPane
      BGridPane buttonPane = new BGridPane(1);
      buttonPane.add(null, new BButton(new CancelCommand(this)));
      
      // put the whole dialog together
      BEdgePane edge = new BEdgePane();
      edge.setCenter(displayPane);
      edge.setBottom(new BBorderPane(buttonPane, 10, 0, 0, 0));
      this.dialog = new BDialog(shell, "Please Wait", true, new BBorderPane(edge));
      
      // start operation on a background thread
      start();     
            
      // open cancel dialog           
      dialog.setBoundsCenteredOnOwner();
      dialog.open();  
      
      // check for exception on background thread
      if (exception != null)
        throw exception;
    }
    
    /**
     * Long running operations will poll the cancel flag.
     */
    public boolean isCanceled()
    {  
      return canceled;
    }         
    
    /**
     * Callback during long operations telling us what is currently happening.
     */
    public void notifyStatus(String msg)                         
    {
      if (status == null) return;
      status.getModel().insert(status.getModel().getEndPosition(), msg + "\n");
      status.moveCaretPosition(status.getModel().getEndPosition());
      status.relayoutSync();
    }
    
    /**
     * Thread.run is used to invoke doRun() on background thread.
     */
    public void run() 
    {        
      // do operation
      try 
      {               
        doRun();    
      }
      catch(Exception e)
      {
        exception = e;
      }
      
      // now close the dialog
      dialog.close();
    }          

    /**
     * Get human readable description of the operation.
     */
    abstract String description();    

    /**
     * This is the operation to actual run on the background thread.
     */
    abstract void doRun() throws Exception;
        
    Exception exception;  
    BDialog dialog; 
    BTextEditor status; 
    boolean canceled;
  }                

  class CancelCommand extends Command
  {                  
    CancelCommand(PlatformOperation op) { super(shell, "Cancel"); this.op = op; }
    public CommandArtifact doInvoke() { op.canceled = true; return null; }
    PlatformOperation op;
  } 

////////////////////////////////////////////////////////////////
// PlatformOperation Implementations
////////////////////////////////////////////////////////////////

  class StopAllStations extends PlatformOperation
  {              
    String description() { return "Ensuring all stations are stopped..."; }
    
    void doRun() throws Exception
    {
      plat.getStationManager().stopAllStations(this);
    }                                  
  }                

  class StartStation extends PlatformOperation
  {     
    StartStation(String name) { this.name = name; }         

    String description() { return "Starting station \"" + name + "\"..."; }
    
    void doRun() throws Exception
    {
      plat.getStationManager().getStation(name).start(this);
    }      
    
    String name;                                
  }     

  class BackupStation extends PlatformOperation
  {     
    BackupStation(RemoteStation station, BDirectory dir) 
    { 
      this.station = station; 
      this.dir = dir;
    }         

    String description() { return "Backing up station \"" + station.getName() + "\"..."; }
    
    void doRun() throws Exception
    {   
      station.makeLocalCopy(dir, this);
    }      
    
    RemoteStation station;
    BDirectory dir;
  }                

  class DeleteStation extends PlatformOperation
  {     
    DeleteStation(RemoteStation station) { this.station = station; }         

    String description() { return "Deleting existing station \"" + station.getName() + "\"..."; }
    
    void doRun() throws Exception
    {                 
      station.delete(this);
    }      
    
    RemoteStation station;
  }  

  class RebootHost extends PlatformOperation
  {
    String description() { return "Rebooting host..."; }

    void doRun() throws Exception 
    {
      plat.getStationManager().rebootSync(this);
    }

    String name;
  }

  class CreateStation extends PlatformOperation
  {     
    CreateStation(BDirectory dir, String name) 
    { 
      this.dir  = dir;
      this.name = name; 
    }         

    String description() { return "Creating station \"" + name + "\"..."; }
    
    void doRun() throws Exception
    {      
      plat.getStationManager().createStation(dir, name, this);           
    }      
    
    BDirectory dir;
    String name;
  }                

////////////////////////////////////////////////////////////////
// LocalStation
////////////////////////////////////////////////////////////////
  
  /**
   * Search the local machine for all the station databases.
   */
  LocalStation[] getLocalStations()
  {                 
    Array acc = new Array(LocalStation.class);
    
    // search all the sub-directories under file:!stations; this
    // isn't perfect because we don't actually know if the daemon
    // is running against the same version as this workbench
    BFileSystem fs = BFileSystem.INSTANCE;
    BDirectory stationsDir = (BDirectory)fs.resolveFile(new FilePath("!stations"));
    BIFile[] dirs = stationsDir.listFiles();
    for(int i=0; i<dirs.length; ++i)
    {                                                
      // skip non-directories
      if (!dirs[i].isDirectory()) continue;
      BDirectory dir = (BDirectory)dirs[i];
        
      // check for config.bog
      BIFile config = fs.findFile(dir.getFilePath().merge("config.bog"));
      if (config == null) continue;
      
      // this looks like a local station
      acc.add(new LocalStation(dir, config));
    }
        
    return (LocalStation[])acc.trim();
  }

  static class LocalStation
  {                        
    LocalStation(BDirectory dir, BIFile config) 
    {
      this.name   = dir.getFileName();
      this.dir    = dir;
      this.config = config;
    }     
    
    public String toString() { return name; }
     
    String name;
    BDirectory dir; 
    BIFile config;
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  static final Lexicon lex = Lexicon.make(PlatformUtil.class);
  static final BImage stationIcon = BImage.make("module://icons/x16/database.png");
  static final BImage clockIcon = BImage.make("module://icons/x32/clock.png");
  
  static String lastLocalUser = "", lastLocalPass = "";
  static String lastRemoteAddr = "", lastRemoteUser = "", lastRemotePass = "";

  BWbShell shell;
  PlatformDaemon plat;
}
