/*
 * Copyright 2004 Tridium, Inc. All Rights Reserved.
 */
package appliance.ui;

import javax.baja.sys.*;
import javax.baja.naming.*;
import javax.baja.user.*;
import javax.baja.util.*;
import javax.baja.workbench.*;
import javax.baja.workbench.mgr.*;
import javax.baja.schedule.*;
import com.tridium.workbench.transfer.*;
import appliance.*;

/**
 * BApplianceUserManager is a simple tool for adding/removing/configuring users.
 *
 * @author    Andy Frank       
 * @creation  23 Mar 05
 * @version   $Revision$ $Date$
 * @since     Niagara 3.0
 */
public class BApplianceUserManager  
  extends BAbstractManager
{   

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BApplianceUserManager.class);
  
////////////////////////////////////////////////////////////////
// AbstractManager
////////////////////////////////////////////////////////////////

  protected MgrModel makeModel() { return new Model(this); }

////////////////////////////////////////////////////////////////
// Model
////////////////////////////////////////////////////////////////
  
  class Model extends MgrModel
  {                                  
    Model(BApplianceUserManager mgr) { super(mgr); }
    
    protected String makeTableTitle()
    {        
      return "User Manager";
    }  
    
    protected MgrColumn[] makeColumns()
    {
      return new MgrColumn[]
      { 
        new MgrColumn.Name(),
        new MgrColumn.Prop(BUser.password, MgrColumn.EDITABLE | MgrColumn.UNSEEN),
        new RoleColumn()
      };
    }

    public int getSubscribeDepth()
    {
      return 2;
    }
  
    public Type[] getIncludeTypes()
    {                                   
      return new Type[] { BUser.TYPE };
    }                                                              
    
    public MgrTypeInfo[] getNewTypes()
    {
      return new MgrTypeInfo[] { MgrTypeInfo.make(new BUser()) };
    }

    public BComponent newInstance(MgrTypeInfo type)
      throws Exception
    { 
      BObject base = getWbShell().getActiveOrdTarget().get();
      BUser[] protos = getProtos(base);
      BUser user = (BUser)type.newInstance();
      user.add("role", BString.make(protos[0].getName()), Flags.READONLY);
      return copyFromProto(user, protos[0]);
    }
    
    public boolean accept(BComponent c)
    {
      if (c.getName().equals("admin")) return false;
      if (c.getName().equals("guest")) return false;
      return true;
    }
  }
  
////////////////////////////////////////////////////////////////
// Columns
////////////////////////////////////////////////////////////////

  class RoleColumn extends MgrColumn
  {
    public RoleColumn()
    {
      super("Role", MgrColumn.EDITABLE);
    }
    
    public Object get(Object row)
    {                                                   
      BComplex complex = (BComplex)row;
      return complex.get("role");
    }                        

    public BValue load(MgrEditRow row)
    {                       
      return ((BComplex)row.getTarget()).get("role").newCopy();
    }

    public void save(MgrEditRow row, BValue value, Context cx)
    {                          
      BComponent target = row.getTarget();
      Property prop = target.getProperty("role");
      BValue old = target.get(prop);
      if (!old.equivalent(value)) 
      {
        target.set(prop, value.newCopy(), cx);
        String role = ((BString)value).getString();
        BObject base = getWbShell().getActiveOrdTarget().get();
        BUser[] protos = getProtos(base);
        for (int i=0; i<protos.length; i++)
        {
          String name = SlotPath.unescape(protos[i].getName());
          if (role.equals(name)) 
            copyFromProto((BUser)target, protos[i]);
        }        
      }
    }

    public BWbEditor toEditor(MgrEditRow[] rows, int colIndex, BWbEditor currentEditor)
    {               
      if (currentEditor == null)
        currentEditor = new BUserRoleFE();
      currentEditor.loadValue(rows[0].getCell(colIndex));
      return currentEditor;      
    }
    
    public void fromEditor(MgrEditRow[] rows, int colIndex, BWbEditor editor)
      throws Exception
    {
      BValue val = (BValue)editor.saveValue();
      for(int i=0; i<rows.length; ++i)
        rows[i].setCell(colIndex, val);
    }
  }
  
////////////////////////////////////////////////////////////////
// Protos
////////////////////////////////////////////////////////////////
  
  /**
   * Return the prototypes defined for this database.
   */
  public static BUser[] getProtos(BObject base)
  {
    BOrd ord = BOrd.make("station:|slot:/Services/UserService/protos");
    BComponent proto = ord.get(base).asComponent();
    return (BUser[])proto.getChildren(BUser.class);
  }
  
  /**
   * Configure the user based on the prototype.
   */
  public static BUser copyFromProto(BUser user, BUser proto)
  {
    SlotCursor c = proto.getProperties();
    while (c.next())
    {
      String name = c.property().getName();
      if (name.equals("name")) continue;
      if (name.equals("password")) continue;
      
      BValue value = proto.get(name).newCopy();
      Property prop = user.getProperty(name);      
      if (prop == null) user.add(name, value);
      else user.set(name, value);
    }  
    return user;
  }
}

