/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Niagara Mobile Utils - functions relating to the 
 * "available views" button and associated dialog in our JQM page headers
 *
 * @author Gareth Johnson
 */

// JsLint options (see http://www.jslint.com )
/*jslint white: true, vars: true */

/*global $, baja, niagara*/ 

////////////////////////////////////////////////////////////////
// Views Command Management
////////////////////////////////////////////////////////////////

(function () {
  "use strict";
  
  niagara.util.require(
    'niagara.util.flow',
    'niagara.util.mobile.dialogs',
    'niagara.util.mobile.commands'
  );
  
  var util = niagara.util,
      dialogs = util.mobile.dialogs,
      commands = util.mobile.commands,
      Command = commands.Command,
      AsyncCommand = commands.AsyncCommand;
  
  /**
   * Finds out which of the available views can be linked to. A view will only
   * be linked to if it is not the same as the current view.
   * 
   * @memberOf niagara.util.mobile.views
   * @private
   * @param {baja.Component} list the list of all available views retrieved
   * from a <code>ComponentServerSideHandler.getMobileAgentList()</code>
   * server-side call
   * @return {Array} an array of linkable view objects - each object has
   * <code>displayName</code> and <code>url</code> properties - may be empty
   * if no available linkable views
   */
  function getLinkableViews(list) {
    var slots = list.getSlots(),
        linkables = [];
    
    baja.iterate(slots, function (slot) {
      var prop = list.get(slot);

      linkables.push({
        displayName: prop.get('displayName'),
        url: prop.get('ord').toUri()                                                               
      });
    });
    
    
    return linkables;
  }
  
  /**
   * Assembles the list of alternate views for a component into a listview
   * dialog. If no alternate views are available, an OK dialog saying so will 
   * be shown instead.
   * 
   * @memberOf niagara.util.mobile.views
   * @private
   * @param {baja.Component} list the list of all available views retrieved
   * from a <code>ComponentServerSideHandler.getMobileAgentList()</code>
   * server-side call
   * @return {Object} a dialog handler for the shown dialog
   */
  function showListDialog(list, callbacks) {
    var views = getLinkableViews(list),
        viewCommands = [];
    if (views.length) {
      baja.iterate(views, function (view) {
        var ord = baja.Ord.make(view.url),
            viewQuery = ord.parse().get('view');
        viewCommands.push(new Command(view.displayName, view.url));
      });
      commands.showCommandsDialog(viewCommands, undefined, callbacks);
    } else {
      return dialogs.ok(baja.lex("mobile").get('views.noneAvailable'));
    }
  }
  
  /**
   * The asynchronous process for performing the necessary server side
   * calls to display the alternate views dialog.
   * 
   * @memberOf niagara.util.mobile.views
   * @field
   * @private
   * @type niagara.util.flow.SequentialWorkflow
   */
  var showViewsDialogWorkflow = util.flow.sequential(
      
    //first, retrieve the component in question from the station
    function showViewsDialogWorkflow__step1__retrieveComponent(cx) {
      var ord = cx.ord;
      if (ord instanceof baja.Component && ord.isMounted()) {
        //already have a mounted component so skip the initial retrieve
        this.ok(ord);
      } else {
        delete this.batch; //ORD resolve doesn't like batches
        baja.Ord.make(ord).get(this);
      }
    },
    
    //then query the station again to get the list of alternate views
    function showViewsDialogWorkflow__step2__getMobileAgentList(cx, comp) {
      comp.serverSideCall($.extend(this, {
        typeSpec: 'mobile:ComponentServerSideHandler',
        methodName: 'getMobileAgentList'
      }));
    },
    
    //finally, assemble them into a listview
    function showViewsDialogWorkflow__step3__showListDialog(cx, list) {
      this.ok(showListDialog(list, cx.callbacks));
    }
  );
  
  /**
   * Shows a listview dialog with other views available for the specified ORD.
   * The entries in the listview will link to alternate views/apps for the given
   * ORD (for example, when viewing an alarm console recipient component in the 
   * Alarm Console app, can switch to view it in the Property Sheet.
   * 
   * @memberOf niagara.util.mobile.views
   * @param {baja.Ord|String|baja.Component} [ord] the ORD we wish to find 
   * alternate views for. You can also pass a mounted Component here - this
   * saves a network call to the station to resolve the ORD, and the alternate
   * views will be found for that component. If undefined/null, 
   * <code>niagara.view.ord</code> will be used.
   * @param {Object} [callbacks] an object literal with ok/fail callbacks to
   * run once the dialog has been created and displayed -
   * the handler object for the listview dialog will be passed as the first
   * argument to the ok callback
   */
  function showViewsDialog(ord, callbacks) {
    callbacks = baja.objectify(callbacks, 'ok');
    
    ord = ord || niagara.view.ord;
    
    $.mobile.showPageLoadingMsg();
    
    showViewsDialogWorkflow.invoke({ord: ord, callbacks: callbacks}, {
      ok: function (dialog) {
        $.mobile.hidePageLoadingMsg();
      },
      fail: dialogs.error
    });
  }
  
  // View Command for the Commands dialog
  var viewsCmdOrd = "",
      viewsCmd = new AsyncCommand("%lexicon(mobile:views.selectView)%", function (callbacks) {
        // When the link is clicked, show the views dialog
        showViewsDialog(viewsCmdOrd, callbacks);
      });
  
  // Set the ORD for the views command
  viewsCmd.setOrd = function(ord) {
    viewsCmdOrd = ord;
  };
  
  /**
   * Return the option object for selecting views.
   *
   * @see niagara.util.mobile.commands   
   *
   * @memberOf niagara.util.mobile.views
   * @returns {Object}
   */
  function getViewsCommand() {
    return viewsCmd;
  }
  
  // Automatically add the views command if we can
  if (niagara.view.profile.showViews && util.mobile.commands) {
    util.mobile.commands.prependDefaultCommand(getViewsCommand());
  }
    
  /**
   * @namespace
   * @name niagara.util.mobile.views
   */
  util.api('niagara.util.mobile.views', {
    'public': {
      getViewsCommand: getViewsCommand
    },
    'private': {
      getLinkableViews: getLinkableViews,
      showListDialog: showListDialog
    }
  });
}());