/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Niagara Mobile Component Selection
 *
 * @author Gareth Johnson
 * @version 0.1.0.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint white: true, plusplus: true */

/*global $, baja, niagara*/ 

////////////////////////////////////////////////////////////////
// Component Selection View
////////////////////////////////////////////////////////////////

(function utilMobileSelect() {

  // Use ECMAScript 5 Strict Mode
  "use strict";

  var util = niagara.util,
      mobileLex = util.lazyLex('mobile');
  
////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////

  /**
   * Default ok callback handler.
   *
   * @private
   */
  function defOk() {    
    $.mobile.hidePageLoadingMsg();
  }
 
  /**
   * Default fail callback handler.
   *
   * @private
   */
  function defFail(err) {
    baja.error(err);

    // Hide any loading title      
    $.mobile.hidePageLoadingMsg();
     
    // Load error page     
    $("<a href='#error' data-rel='dialog' data-transition='flip' />").click();  
  }
  
////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////  
  
  /**
   * A collection of component display names and their corresponding nav ords.
   * 
   * @class 
   * @name niagara.util.mobile.select.NamesAndOrds
   * @private
   * @inner
   */
  function NamesAndOrds() {
    this.names = [];
    this.ords = [];
  }
  
  /**
   * Adds a display name and nav ord to the collection.
   * 
   * @name niagara.util.mobile.select.NamesAndOrds#add
   * @function
   * @param {String} name the display name to add
   * @param {baja.Ord|String} ord the ord to add
   */
  NamesAndOrds.prototype.add = function(name, ord) {
    this.names.push(name);
    this.ords.push(ord);
  };
  
  /**
   * Calls a function on each name/ord pair in the collection.
   * 
   * @name niagara.util.mobile.select.NamesAndOrds#each
   * @function
   * @param {Function} the function to call - the first parameter to this
   * function will be the display name, the second will be the ord
   */
  NamesAndOrds.prototype.each = function (func) {
    var names = this.names,
        ords = this.ords,
        len = names.length,
        i;
    
    for (i = 0; i < len; i++) {
      func(names[i], ords[i]);
    }
  };
  
  /**
   * Retrieves a collection of display names and nav ords from the station,
   * for all components that are of the given type spec.
   * 
   * @name niagara.util.mobile.select.getNamesAndOrds
   * @function
   * @private
   * @param {String} typeSpec the type spec we want to retrieve components for
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {niagara.util.mobile.select.NamesAndOrds} the collection of
   * display names and nav ords (to be passed as a parameter to the ok
   * callback)
   */
  function getNamesAndOrds(typeSpec, callbacks) {
    var namesAndOrds = new NamesAndOrds(),
        bql = "station:|slot:/|bql:select displayName, navOrd " +
              "from " + typeSpec + " stop";

    baja.Ord.make(bql).get({
      cursor: {            
        each: function () {
          var displayName = this.get("displayName"),
              ord = baja.Ord.make(this.get("navOrd")).relativizeToSession();
          namesAndOrds.add(displayName, ord);
        },
        after: function () {
          callbacks.ok(namesAndOrds);
        },
        fail: callbacks.fail,
        limit: -1
      },
      fail: callbacks.fail
    });
  }

  /**
   * Collapses a set of NamesAndOrds objects into a single NamesAndOrds.
   * 
   * @name niagara.util.mobile.select.collapseNamesAndOrds
   * @function
   * @param {Array} namesAndOrdses
   * @returns {niagara.util.mobile.select.NamesAndOrds}
   */
  function collapseNamesAndOrds(namesAndOrdses) {
    var result = new NamesAndOrds();
    
    baja.iterate(namesAndOrdses, function (namesAndOrds) {
      namesAndOrds.each(function (name, ord) {
        result.add(name, ord);
      });
    });
    
    return result;
  }

  /**
   * Iterates an array of type specs, retrieves names and ords for each one,
   * and returns a single collection of display names and ords for all
   * components in the station that match one of the given type specs.
   * 
   * @name niagara.util.mobile.select.getAllNamesAndOrds
   * @function
   * @private
   * @param {Array} typeSpecs an array of type specs
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {niagara.util.mobile.select.NamesAndOrds} a complete collection
   * of component display names and ords
   */
  function getAllNamesAndOrds(typeSpecs, callbacks) {
    var invocations = [];
    
    baja.iterate(typeSpecs, function (typeSpec) {
      invocations.push(function (callbacks) {
        getNamesAndOrds(typeSpec, callbacks);
      });
    });
    
    util.flow.runParallel(invocations, {
      ok: function (namesAndOrdses) {
        callbacks.ok(collapseNamesAndOrds(namesAndOrdses));
      },
      fail: callbacks.fail
    });
  }
  
  /**
   * Takes a collection of display names and ords and builds up a string of
   * html (button links).
   * 
   * @name niagara.util.mobile.select.buildSelectHtml
   * @private
   * @function
   * @param {niagara.util.mobile.select.NamesAndOrds} namesAndOrds
   * @returns {String} the string of html to be appended to the DOM (will be
   * an empty string if the names/ords collection contained no elements)
   */
  function buildSelectHtml(namesAndOrds) {
    var html = "";
    
    namesAndOrds.each(function (displayName, ord) {
      html += "<a data-role='button' data-theme='a' rel='external' href='" + 
        ord.toUri() + "'>" + displayName + "</a>\n";
    });
    
    return html;
  }
  
  /**
   * Builds up a list of link buttons by querying the station for all the
   * components that match the given list of type specs, then assembling those
   * components' display names and ords into a list of JQM link buttons linking
   * to those ords.
   * 
   * @name niagara.util.mobile.select.loadLinkButtons
   * @private
   * @function
   * @param {Array} typeSpecs the type specs to load
   */
  function loadLinkButtons(typeSpecs) {
    // Query the Station for all the Component Types and update the DOM
    baja.started(function () {
      $.mobile.showPageLoadingMsg();
      getAllNamesAndOrds(typeSpecs, {
        ok: function (namesAndOrds) {
          var html = buildSelectHtml(namesAndOrds);
          
          //html will be an empty string if there were no components found
          if (!html) {
            html = "<h3>" + mobileLex.get({
              key: "select.noComponentsFound", 
              args: typeSpecs.join(" / ")
            }) + "</h3>";
          }
          
          $("#selectCompContent").append($(html)).trigger("create");
          defOk();
        },
        fail: defFail
      });
    });
  }
  
////////////////////////////////////////////////////////////////
// Export
////////////////////////////////////////////////////////////////
      
  /**
   * @namespace
   * @name niagara.util.mobile.select
   */
  util.api('niagara.util.mobile.select', {
    'public': {
      loadLinkButtons: loadLinkButtons
    },
    'private': {
      NamesAndOrds: NamesAndOrds,
      buildSelectHtml: buildSelectHtml,
      collapseNamesAndOrds: collapseNamesAndOrds,
      getAllNamesAndOrds: getAllNamesAndOrds,
      getNamesAndOrds: getNamesAndOrds
    }
  });
}());