/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Niagara Mobile Utils - Pages
 *
 * @author Gareth Johnson
 */

//JsLint options (see http://www.jslint.com )
/*jslint white: true, plusplus: true */

/*global $, baja, niagara, document, setTimeout*/ 

////////////////////////////////////////////////////////////////
// jQuery Mobile Pages Architecture
////////////////////////////////////////////////////////////////

/**
 *@namespace An event framework for managing internal Pages in jQuery Mobile.
 */
(function mobilePages(mobile) {

  // Use ECMAScript 5 Strict Mode
  "use strict";
 
  var first = true,         
      prevPage = null,
      currentPage = null,
      reg = [],
      util = niagara.util,
      loading = false;
  
  /**
   * Register a Page.
   * <p>
   * This registers a Page handler object with the pages framework. After an object is
   * registered, it will receive event notifications and callbacks.
   * <p>
   * For instance, if the page wanted to receive a 'pageshow' event, the handler object would
   * need a function called 'pageshow'.
   * <p>
   * Handler objects also support dynamic page creation. This is done by defining a 
   * 'createPage' method on the handler. This method returns a String, jQuery DOM or 
   * DOM element that's then injected into the page container. The method is passed
   * a jQuery Mobile URL object.
   *
   * @name niagara.util.mobile.pages.register
   * @function
   *
   * @param {String|Function} matcher a function or string for matching a page name.
   * @param {Object} handler an object that will attempt to have notification methods called upon it.
   */
  function register(matcher, handler) {
    var m = matcher;
    if (typeof m === "string") {
      m = function (s) {
        return matcher === s;
      };
    }
    
    // Register page so it can be looked up
    reg.push({
      isMatch: m,
      handler: handler
    });  
    
    return handler;
  }
    
  /**
   * Return the specified Page Handler object.
   *
   * @name niagara.util.mobile.pages.get
   * @function
   *
   * @param {String} pageName.
   * @return {Object} the page handler Object or null if the page can't be found.
   */    
  function getHandler(pageName) {
    if (!pageName) {
      return null;
    }
    var i;
    for (i = 0; i < reg.length; ++i) {
      if (reg[i].isMatch(pageName)) {
        return reg[i].handler;
      }
    }
    return null;
  }
             
  /**
   * Fire an event to the specified registered Page Handler.
   *
   * @name niagara.util.mobile.pages.fire
   * @function
   *
   * @private
   */
  function fire(obj) {
    //baja.outln(obj);
    var nm = obj.page.attr("id"),
        handler;   
    
    if (nm) {
      handler = getHandler(nm);
      if (handler && typeof handler[obj.eventName] === "function") {
        handler[obj.eventName](obj);
      }
    }    
  }
  
  /**
   * Fire an event to all the registered Page Handlers.
   *
   * @name niagara.util.mobile.pages.fireAll
   * @function
   *
   * @param {Object} obj the Object Literal for the method's arguments.
   * @param {String} obj.eventName the name of the event to fire
   */
  function fireAll(obj) {
    obj = baja.objectify(obj, "eventName");
    var i;
    for (i = 0; i < reg.length; ++i) {
      if (typeof reg[i].handler[obj.eventName] === "function") {
        reg[i].handler[obj.eventName](obj);
      }
    }
  }
    
  /**
   * Return the previous Page's name.
   *
   * @name niagara.util.mobile.pages.getPrevName
   * @function
   *
   * @return {String}
   */  
  function getPrevName() {
    return prevPage.attr("id") || "";
  }
  
  /**
   * Return the previous Page's handler.
   *
   * @name niagara.util.mobile.pages.getPrevHandler
   * @function
   *
   * @return {Object}
   */  
  function getPrevHandler() {
    return getHandler(getPrevName());
  }
  
  /**
   * Return the previous page DOM object.
   *
   * @name niagara.util.mobile.pages.getPrev
   * @function
   *
   * @return {Object}
   */  
  function getPrev() {
    return prevPage;
  }
    
  /**
   * Return the current Page name.
   *
   * @name niagara.util.mobile.pages.getCurrentName
   * @function
   *
   * @return {String}
   */     
  function getCurrentName() {
    return currentPage.attr("id") || "";
  }
  
  /**
   * Return the current page DOM object.
   *
   * @name niagara.util.mobile.pages.getCurrent
   * @function
   *
   * @return {Object}
   */ 
  function getCurrent() {
    return currentPage;
  }
  
  /**
   * Return the current page handler object.
   *
   * @name niagara.util.mobile.pages.getCurrentHandler
   * @function
   *
   * @return {Object}
   */ 
  function getCurrentHandler() {
    return getHandler(getCurrentName());
  }
  
  /**
   * Is the current page the very first page being shown?
   *
   * @name niagara.util.mobile.pages.isFirst
   * @function
   *
   * @return {Boolean}
   */
  function isFirst() {
    return first;
  }
        
  /*
  The jQuery Mobile event sequence
  
  On start up... 
  pagebeforeshow   Current Page: alarmConsole    Page Name: undefined Next: false
  pageshow         Current Page: alarmConsole    Page Name: undefined Next: false

  On page change...
  pagebeforehide   Current Page: alarmSource     Page Name: alarmSource Next: true
  pagebeforeshow   Current Page: alarmSource     Page Name: alarmConsole Next: false
  pagehide         Current Page: alarmSource     Page Name: alarmSource Next: true
  pageshow         Current Page: alarmSource     Page Name: alarmConsole Next: false
  */  
       
  // Register for the events   
      
  $("div").live("pageshow", function (event, ui) {   
    var activePage = $(this),
        pPage;
        
    if (ui && ui.prevPage) {
      pPage = $(ui.prevPage);   
    }
    if (!prevPage) {
      pPage = activePage;
    }
          
    fire({
      page: activePage,  
      prevPage: pPage,
      event: event,
      eventName: event.type
    });
    
    if (activePage.jqmData('role') !== 'dialog') {
      prevPage = currentPage;
      currentPage = activePage;
    }
  });
  
  $("div").live("pagebeforeshow", function (event, ui) { 
    var activePage = $(this),
        pPage;

    if (!activePage) {
      return;
    }
  
    if (ui && ui.prevPage) {
      pPage = $(ui.prevPage);   
    }
    if (!prevPage) {
      pPage = activePage;
    }

    fire({
      page: activePage,        
      prevPage: pPage,
      event: event,
      eventName: event.type
    });
  });
  
  $("div").live("pagehide", function (event, ui) { 
    var nextPage;
    if (ui && ui.nextPage) {
      nextPage = $(ui.nextPage);   
    }
          
    fire({
      page: currentPage,  
      nextPage: nextPage,
      event: event,
      eventName: event.type
    });
  });
  
  $("div").live("pagebeforehide", function (event, ui) {  
    first = false;  
    var nextPage;
    if (ui && ui.nextPage) {
      nextPage = $(ui.nextPage);   
    }
    
    fire({
      page: currentPage, 
      nextPage: nextPage,
      event: event,
      eventName: event.type 
    });
  });
  
  $("div").live("pagebeforecreate pagecreate pageinit pagelayout", function (event, ui) { 
    fire({
      page: $(this), 
      event: event,
      eventName: event.type
    });
  });
  
//commented out due to JQM 1.0 wonkiness
//https://github.com/jquery/jquery-mobile/issues/142#issuecomment-2983197
//  var swipeTicket = baja.clock.expiredTicket;
//  $("div").live("swiperight swipeleft", function (event, ui) {  
//    // Throttle the swipe events into one
//    if (swipeTicket.isExpired()) {
//      swipeTicket = baja.clock.schedule(function () {
//        fire({
//          page: currentPage, 
//          eventName: event.type
//        });
//      }, 20);
//    }
//  });
    
  function normalizeOrdUrl(url) {
    url = decodeURI(url);
    if (url.charAt(0) === '/' || url.charAt(0) === '#') {
      url = url.substring(1);
    }
    if (url.match(/^ord[\?\/]/)) {
      return 'ord?' + encodeURI(url.substring(4));
    } else {
      return encodeURI(url); 
    }
  }

  function toId(path) {
    return util.mobile.encodePageId(normalizeOrdUrl(path));
  }
  
  function createPageDom(handler, id, options) {
    if (!handler || typeof handler.createPage !== "function") {
      throw new Error("No page handler (or no createPage function) " +
          "found for id " + id);
    }
    
    handler.createPage(options);
  }
  
  function load(path, options) {
    if (loading) {
      return;
    }
    
    var handler = getHandler(path),
        id;
    
    loading = true;
    
    if (options.pageData && handler && typeof handler.encodeUrl === "function") {
      path = handler.encodeUrl(options.pageData);
    }
    
    id = toId(path);
    
    if (!$.mobile.pageContainer.children("#" + id).length) {
      createPageDom(handler, id, $.extend(options, {
        ok: function (page) {
          page.attr('id', id);
          $.mobile.pageContainer.append(page);
          page.jqmData('pageData', options.pageData);
          page.page();
          loading = false;
          $.mobile.changePage(page, options);
        },
        fail: function (err) {
          loading = false;
          baja.fail(err);
        }
      }));
    }
  }

  $(document).bind("pagebeforechange", function pagebeforechange(e, data) {
    
    // We only want to handle changePage() calls where the caller is asking us to load a page by URL.
    if (typeof data.toPage !== "string") {
      if (data.toPage instanceof $) {
        //if we are going from one JQM page to another (i.e. no dialogs
        //involved), let the pages handler have a shot at doing its own
        //pagebeforechange event.
        if (data.toPage.jqmData('role') !== 'dialog') {
          if (data.options.fromPage && 
              data.options.fromPage.jqmData('role') !== 'dialog') {
            fire({
              page: data.options.fromPage,
              nextPage: data.toPage,
              event: e,
              eventName: 'pagebeforechange',
              options: data.options
            });
          }
          
          if (!e.isDefaultPrevented()) {
            setTimeout(function () {
              data.toPage.trigger('pagelayout');
            }, 0);
          }
        }
      }
      return;
    }
    
    // Attempt to decode page information from the URL    
    var url = $.mobile.path.parseUrl(data.toPage),
        href = normalizeOrdUrl(url.pathname + url.search),
        hash = normalizeOrdUrl(url.hash || ""),
        pathname = normalizeOrdUrl(url.pathname || ""),
        pageName = hash,
        handler, 
        pageDom,
        pageData;
    
    // If we detect a dialog in the hash then bail
    if (hash.indexOf("&ui-state=dialog") > -1) {
      return;
    }
    
    // Get the page handler
    handler = getHandler(pageName);
    if (!handler) {
      pageName = pathname;
      handler = getHandler(pageName);
    }
    if (!handler) {
      pageName = href;
      handler = getHandler(pageName);
    }
    
    // If there's no page handler then bail
    if (!handler) {
      return;
    }

    // Look up the page name via its id
    pageDom = $.mobile.pageContainer.children("#" + toId(pageName));
    if (!pageDom.length && hash) {
      pageDom = $.mobile.pageContainer.children("#" + toId(hash));
    }
    if (!pageDom.length) {
      pageDom = $.mobile.pageContainer.children("#" + toId(href));
    }
    
    if (!hash || !pageDom.length) {
      /*
       * if we're intercepting a pagechange to an ORD so we can dynamically
       * inject the necessary page, then force dataUrl to the ord we want.
       * otherwise (e.g. we are doing hash-based navigation to a static page),
       * we want to keep the dataUrl JQM gives us, so don't.
       */
      data.options.dataUrl = '/' + href;
    }

    if (!pageDom.length) {
      if (typeof handler.createPage === "function") {
        pageData = handler.decodeUrl && handler.decodeUrl(url);
        data.options.pageData = pageData;
        data.options.url = url;
        
        /*
         * only do the dynamic page load AFTER baja starts because it may
         * (will) need to perform network calls to the station
         */
        baja.started(function () {
          load(pageName, data.options);
        });
      } else {
        // If the handler doesn't support dynamic page creation then bail
        return;
      }
    }
    
    // If a dialog then bail (currently we only support views)
    if ((data.options.role && data.options.role === "dialog") || pageDom.is(".ui-dialog")) {
      return;
    }
    
    if (pageDom.length) {
      // Change to this new page
      $.mobile.changePage(pageDom, data.options);
    }
    
    // Prevent default change handler
    e.preventDefault();
  });
            
  // Functions exported as public
  niagara.util.api("niagara.util.mobile.pages", {
    register: register,
    getPrevName: getPrevName,
    getPrev: getPrev,
    getPrevHandler: getPrevHandler,
    getCurrentName: getCurrentName,
    getCurrent: getCurrent,
    getCurrentHandler: getCurrentHandler,
    getHandler: getHandler,
    fire: fire,
    fireAll: fireAll,
    isFirst: isFirst
  });
}(niagara.util.mobile));
 