/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions related to the management and display of JQM
 * dialog boxes.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, bitwise: true, browser: true */

/*global $, baja, niagara */

(function () {
  "use strict";
  
  niagara.util.require(
    'niagara.util.flow',
    'niagara.util.mobile.pages'
  );
  
  var util = niagara.util,
      pages = util.mobile.pages,
      callbackify = util.callbackify,
      currentInvocation,
      mobileLex = null,
      
      dialogHtml,
      buttonHtml,
      
      validateAndSaveWorkflow,
      
      //exports
      error;
  
  dialogHtml = 
    '<div data-role="dialog" class="dynamicDialog">\n' +
      '<div class="dynamicDialogHeader" data-role="header" data-theme="a">\n' +
        '<h1 class="title" />\n' +
      '</div>\n' +
      '<div class="dynamicDialogContent" data-role="content" data-theme="c">\n' +
        '<div class="dialogDisplay"></div>\n' +
        '<div class="dialogButtons" data-role="controlgroup" data-type="horizontal"></div>\n' +
      '</div>\n' +
    '</div>';
  
  buttonHtml = 
    '<a data-value="{value}" data-role="button" data-theme="a"></a>';

  validateAndSaveWorkflow = util.flow.sequential(
    function validateAndSaveWorkflow__step1__validate(editor) {
      editor.validate(this);
    },
    function validateAndSaveWorkflow__step2__saveValue(editor) {
      editor.saveValue(this);
    }
  );
  
  
  ////////////////////////////////////////////////////////////////
  // DialogInvocation utility functions
  ////////////////////////////////////////////////////////////////
  
  
  /**
   * Loads the dialog's content div. That content may be one of several 
   * different types.
   * <p>If it is a jQuery object, it will be inserted directly into the dialog's
   * content div.
   * <p>If it is a string or other type of object, it will be converted into
   * a string and inserted into the dialog's content div.
   * <p>If it is a function, it will be executed with the (empty) content div 
   * as the first parameter, and an object with ok/fail callbacks as the
   * second parameter.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @private
   * @param {String|jQuery|Function} content the main dialog display content
   * @param {jQuery} contentDiv the dialog's content div
   * @param {Object} callbacks an object with ok/fail callbacks to execute
   * after the content has fully loaded
   */
  function loadContent(content, contentDiv, callbacks) {
    callbacks = baja.objectify(callbacks, 'ok');
    
    if (content instanceof $) {
      contentDiv.html(content);
      callbacks.ok();
    } else if (typeof content === 'object' || typeof content === 'string') {
      contentDiv.text(String(content));
      callbacks.ok();
    } else if (typeof content === 'function') {
      try {
        content(contentDiv.empty(), callbacks);
      } catch (err) {
        callbacks.fail(err);
      }
    }
  } 
  
  ////////////////////////////////////////////////////////////////
  // DialogInvocation
  ////////////////////////////////////////////////////////////////


  /**
   * @name niagara.util.mobile.dialogs.DialogInvocation
   * @class
   * 
   * @param {Object} [params] an object literal containing parameters to be
   * used by this dialog. Can also be a function, in which case it will be
   * treated as <code>params.content</code>.
   * @param {String} [params.title] the dialog title to be shown
   * @param {String|jQuery|Function} [params.content] the main dialog display 
   * content
   * @param {jQuery|String} [params.returnPage] The page to return to after 
   * the dialog is closed. This can be changed by calling redirect(). If
   * omitted, when this dialog is closed it will simply return to whatever
   * page is currently referenced by <code>location.href</code>.
   * @param {Object} [params.callbacks] an object containing ok/fail callbacks.
   * These callbacks will be called when the dialog is closed by calling 
   * <code>invoke()</code> (e.g., when the user clicks Yes/No/Cancel/etc). Note
   * that they are called even if this dialog itself does not return us to the
   * previously viewed JQM page - they will still fire if this dialog opens
   * another dialog.
   * 
   * @see niagara.util.mobile.dialogs.loadContent
   */
  function DialogInvocation(params, page) {
    /**
     * the parameters of the currently viewed dialog - title, content, handlers
     * etc.
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$params
     * @field
     * @type {Object}
     * @private
     */
    this.$params = baja.objectify(params, 'content');
    
    /**
     * The page to return to after the dialog is closed. This can be changed
     * by calling redirect().
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$returnPage
     * @field
     * @type {jQuery|String}
     * @private
     */
    this.$returnPage = params.returnPage;
    
    /**
     * The JQM page holding the actual dialog. Should already be constructed
     * and appended to the DOM when the DialogInvocation is created.
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$page
     * @field
     * @type {jQuery}
     * @private
     * @see niagara.util.mobile.dialogs.createDialogPage
     */
    this.$page = page;
    
    var activePage = $.mobile.activePage;
    /**
     * Set to false if this dialog is opened when we are NOT already looking at
     * another dialog; set to true if this dialog is being opened while another
     * dialog is already active. When this dialog is closed, we will return
     * to the JQM page we were previously looking at if and only if this is
     * false. 
     * 
     * @name niagara.util.mobile.dialogs.DialogInvocation#$dialogActive
     * @field
     * @type {Boolean}
     * @private
     * @see niagara.util.mobile.dialogs.DialogInvocation#invoke
     */
    this.$dialogActive = activePage && activePage.jqmData('role') === 'dialog';
  }
  
  /**
   * @memberOf niagara.util.mobile.dialogs
   * @private
   */
  function isRedirect(returnPage) {
    var currentPage = util.mobile.pages.getCurrent();
    
    if (typeof returnPage === 'string') {
      returnPage = util.mobile.encodePageId(returnPage);
      
      if (returnPage.charAt(0) === '#') {
        returnPage = $(returnPage);
      } else {
        returnPage = $('#' + returnPage);
      }
    }
    
    // when changing to returnPage, should the URL change as well?
    // if the current URL references an existing JQM page, and that
    // JQM page is the same as the one referenced by returnPage, NO.
    // otherwise yes.
    if (currentPage && returnPage && currentPage[0] === returnPage[0]) {
      return false;
    } else {
      return true;
    }
  }
  
  /**
   * Closes a dialog and returns to another JQM pages, invoking a 
   * function after the dialog has fully closed.
   *  
   * @memberOf niagara.util.mobile.dialogs
   * @private
   * @param {String} [clickedValue] the value of the button that was clicked
   * to close the dialog - defaults to 'cancel' if omitted
   */
  DialogInvocation.prototype.close = function close(clickedValue) {
    clickedValue = clickedValue || 'cancel';
    
    var that = this,
        page = that.$page,
        params = that.$params,
        options = that.$options,
        returnPage = that.$returnPage;
    
    if (typeof returnPage === 'string') {
      //if we're going to localhost/ord?.... then we don't want a # if we can
      //help it - we want to update the entire URL. otherwise, we're going
      //directly to a JQM page referenced by a hash
      if (returnPage.indexOf('ord?') !== 0 && returnPage.charAt(0) !== '#') {
        returnPage = '#' + returnPage;
      }
    }
    
    if (returnPage instanceof $ || typeof returnPage === 'string') {
      $.mobile.changePage(returnPage, $.extend({
        transition: 'none', 
        changeHash: isRedirect(returnPage)
      }, options));
    } else if (typeof returnPage === 'function') {
      returnPage.call(this, clickedValue);
    } else {
      // Since we're never changing the hash for these dialogs, simply change
      // back to the current page to get rid of the dialogs
      $.mobile.changePage(location.href, {
        transition: "none",
        changeHash: false
      });
    }
    
    currentInvocation = undefined;
  };
  
  /**
   * Performs the action corresponding to one of the dialog buttons (e.g.
   * 'ok', 'yes', 'no') and then closes the dialog.
   * 
   * @name niagara.util.mobile.dialogs.DialogInvocation#invoke
   * @function
   * @private
   * 
   * @param {String|Function} value the value corresponding to the button 
   * whose action you wish to invoke. This can also be a function to be
   * invoked directly. The function should take a single argument of an ok/fail
   * callback object literal - it should call <code>ok()</code> or
   * <code>fail()</code> on this object to signal that the handler has completed
   * and the dialog should be closed.
   * 
   * @param {Object} [callbacks] an object containing ok/fail callbacks. 
   * <code>callbacks.ok</code> will be called when the dialog closes (or 
   * another dialog is shown). <code>callbacks.fail</code> will be called
   * if the click handler calls its own <code>fail()</code> or throws an error.
   */
  DialogInvocation.prototype.invoke = function invoke(value, callbacks) {
    callbacks = callbackify(callbacks, baja.ok, error);
    
    var that = this,
        params = that.$params,
        closeCallbacks = callbackify(params.callbacks, baja.ok, error),
        handler = typeof value === 'function' ? value : params[value];
    
    if (typeof handler !== 'function') {
      handler = function (callbacks) {
        callbacks.ok();
      };
    }
    
    function doFail(err) {
      //run the fail callbacks first in case one of them shows another dialog.
      closeCallbacks.fail(err);
      callbacks.fail(err);

      //if no more dialogs were shown, then we can close.
      if (!that.$dialogActive) {
        that.close(value);
      }
    }
    
    try {
      handler.call(that, {
        ok: function () {
          //if the ok/yes/no/whatever click handler causes another dialog page,
          //we don't want to return to the previous page. 
          if (!that.$dialogActive) {
            that.close(value);
          }
          closeCallbacks.ok();
          callbacks.ok();
        },
        fail: doFail
      });
    } catch (err) {
      doFail(err);
    }
  };
  
  /**
   * Allows a dialog to redirect to another page instead of the one that was
   * in view when the dialog was initially invoked. This method may be called
   * inside of a dialog click handler e.g. 
   * <code>this.redirect('#otherPage')</code>. The <code>#otherPage</code>
   * JQM page will be shown after the dialog is closed. 
   * 
   * @name niagara.util.mobile.dialogs.DialogInvocation#redirect
   * @function
   * @param {jQuery|String|Function} returnPage the new return page. If 
   * <code>returnPage</code> is a function, then this function will be executed
   * <i>instead</i> of doing <code>$.mobile.changePage()</code> or
   * <code>$('.ui-dialog').dialog('close')</code>. This allows you to perform your own page
   * management when the dialog is closed - performing an asynchronous
   * action before redirecting to another page, for instance.
   * 
   * @param {Object} [options] options to be passed directly to 
   * <code>$.mobile.changePage</code> - only to be used if 
   * <code>returnPage</code> is a jQuery object or string.
   */
  DialogInvocation.prototype.redirect = function redirect(returnPage, options) {
    this.$returnPage = returnPage;
    this.$options = options || {};
  };
  
  /**
   * Populates a JQM dialog page with the dialog content. At the point this
   * function is called, the JQM page has already been created, with all
   * necessary buttons, and inserted into the DOM.
   * 
   * <p>This function runs on the output of <code>createDialogPage</code>,
   * just before it is shown by JQM.
   * 
   * @name niagara.util.mobile.dialogs.DialogInvocation#load
   * @private
   * @param {jQuery} page the JQM page to be populated
   * @param {Object} callbacks an object with ok/fail callbacks to execute 
   * after the dialog content has fully loaded
   */
  DialogInvocation.prototype.load = function load() {
    var that = this,
        page = that.$page,
        params = that.$params,
        header = page.children(':jqmData(role=header)'),
        content = page.children(':jqmData(role=content)'),
        display = content.children('.dialogDisplay'),
        xButton = header.find('a:jqmData(icon=delete)'),
        buttons = content.find('.dialogButtons a');

    header.find('h1.title').text(params.title || '');
    
    buttons.unbind('click').addClass('ui-disabled');
    
    loadContent(params.content || '', display, {
      ok: function () {
        buttons.bind('click', function () {
          var value = $(this).jqmData('value');
          
          that.invoke(value);
        });
        buttons.removeClass('ui-disabled');
      },
      fail: function (err) {
        niagara.util.mobile.dialogs.error("Failed to load dynamic dialog " +
            "content: " + err);
      }
    });
    
    //arm our own handler on the X button
    xButton
      .unbind() //remove any previous click handlers we already armed
      .bind('click', function (e) {
        that.invoke('cancel'); //we will do it ourselves thank you
        return false;
      });
  };
  
  
  
  
  ////////////////////////////////////////////////////////////////
  // Dialog utility functions
  ////////////////////////////////////////////////////////////////
  
  function getMobileLex() {
    if (!mobileLex) {
      mobileLex = baja.lex('mobile');
    }
    return mobileLex;
  }
  

  
  /**
   * Dynamically builds up the HTML for a dialog page, appending all necessary
   * buttons, and inserts it into the DOM.
   * 
   * <p>This function will run immediately before showing a particular type
   * of dialog for the first time.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @private
   * @param {String} id the desired HTML id of the page
   * @param {Array} buttons an array of button values (e.g. 
   * <code>[ 'yes', 'no', 'cancel' ]</code>
   * @returns {jQuery} a JQM dialog page
   */
  function createDialogPage(id, buttons) {
    var page = $(dialogHtml.patternReplace({
      theme: 'a'
    })),
    buttonsDiv = page.children(':jqmData(role=content)')
      .children('.dialogButtons');
    
    baja.iterate(buttons, function (buttonValue, i) {
      var button = $(buttonHtml.patternReplace({
        value: buttonValue
      }));
      
      button.text(getMobileLex().get({
        key: buttonValue,
        def: buttonValue
      }));
      button.appendTo(buttonsDiv);
    });

    page.attr('id', id);
    if (id.indexOf('error') === 0) {
      page.addClass('error');
    }
    page.appendTo($.mobile.pageContainer);
    return page;
  }
  
  /**
   * Attempts to center the dialog vertically on the screen. If the dialog is
   * too tall it will simply butt the top against the upper edge of the screen.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @param {jQuery} page the dialog page to position onscreen
   */
  function repositionDialog(page) {
    var headerDiv = page.children('div:jqmData(role=header)'),
        headerTop = headerDiv.offset().top,
        contentDiv = page.children('div:jqmData(role=content)'),
        dialogHeight = headerDiv.outerHeight() + contentDiv.outerHeight(),
        windowHeight = $(window).height(),
        headerAdjust = (windowHeight - dialogHeight) / 2;
    
    headerDiv.css('margin-top', Math.max(headerAdjust, 0));
  }
  
  
  
  
  
  ////////////////////////////////////////////////////////////////
  // Public dialog functions
  ////////////////////////////////////////////////////////////////
  
  /**
   * Shortcut to closing whatever dialog is currently being displayed.
   * 
   * @name niagara.util.mobile.dialogs.closeCurrent
   * @function
   */
  function closeCurrent() {
    if (currentInvocation) {
      currentInvocation.close();
    }
  }
  
  /**
   * Registers a new dialog type in Pages. Once registered, can be displayed
   * by calling the <code>load</code> method on the Pages handler. The
   * <code>load</code> method, and all built-in dialog methods like 
   * <code>yesNoCancel</code> and <code>okCancel</code>, return an object
   * reference that can be used to programmatically close the dialog or 
   * click buttons.
   * <p>
   * For example:
   * 
   * <pre>
   * registerDialog('leftRightDialog', [ 'left', 'right' ]);
   * var dialog = pages.getHandler('leftRightDialog').load({
   *   content: 'please click "left" or "right" below',
   *   title: 'left or right',
   *   //function names correspond to button titles
   *   left: function (cb) { console.log('you clicked "left"'); cb.ok(); }
   *   right: function (cb) { console.log('you clicked "right"'); cb.ok(); }
   *   callbacks: function () { console.log('all done. thanks for clicking'); }
   * });
   * //now you can call methods on the returned object directly, like
   * //dialog.close(); or dialog.left(); or dialog.right();
   * </pre>
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @param {String} pageId the HTML ID to use for the dialog page
   * @param {Array} buttons an array of button values (e.g. 'ok', 'yes', 'no').
   * These should unique within this dialog (cannot have two 'ok', for example)
   * and should correspond to entries in the <code>mobile</code> lexicon
   * whose values will be used to display button titles.
   * @returns {Object} an object reference with a close() method as well as
   * methods corresponding to each button value
   */
  function registerDialog(pageId, buttons) {
    var handler;

    function pagebeforeshow(obj) {
      if (currentInvocation) {
        currentInvocation.load();
      } else {
        obj.page.find('.dialogDisplay').empty();
      }
    }
    
    function load(params) {
      var page = $('#' + pageId);

      if (!page.length) {
        page = createDialogPage(pageId, buttons);
      }
      
      params.returnPage = (currentInvocation && currentInvocation.$returnPage) || params.returnPage;
      
      currentInvocation = new DialogInvocation(params, page);

      $.mobile.changePage(page, { 
        changeHash: false,
        allowSamePageTransition: true,
        transition: "none"
      });
      
      repositionDialog(page);
      
      return currentInvocation;
    }
    
    handler = {
      load: load,
      pagebeforeshow: pagebeforeshow
    };

    pages.register(pageId, handler);
  }
  
  
  
  
  ////////////////////////////////////////////////////////////////
  // Functions for displaying predefined dialog types
  ////////////////////////////////////////////////////////////////
  

  (function registerPages() {
    registerDialog('yesNoCancelDialog', [ 'yes', 'no', 'cancel' ]);
    registerDialog('yesNoDialog', [ 'yes', 'no' ]);
    registerDialog('okCancelDialog', [ 'ok', 'cancel' ]);
    registerDialog('okDialog', [ 'ok' ]);
    registerDialog('errorDialog', [ 'ok' ]);
    registerDialog('cancelDialog', [ 'cancel' ]);
  }());

  /**
   * Shows a standard dialog with 'Yes', 'No', and 'Cancel' buttons.
   * 
   * <p>With this and all dialogs, you can simply pass in the content directly
   * rather than with an object literal with properties. This will cause all
   * the buttons to do nothing except close the dialog. Obviously, this is
   * most appropriate when used with a one-button dialog like 'ok' or 'cancel'.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * 
   * @param {Object} params an object containing the behavior to execute on
   * each button click
   * @param {String} [params.title] the dialog title
   * @param {String|Function|jQuery} [params.content] the content of the dialog
   * @see niagara.util.mobile.dialogs.loadContent
   * @param {Function} [params.yes] a function to execute when 'yes' is clicked
   * @param {Function} [params.no] a function to execute when 'no' is clicked
   * @param {Function} [params.cancel] a function to execute when 'cancel' is 
   * clicked
   * @param {Function} [params.onClose] a function to execute after the
   * dialog closes (this will not execute if your dialog triggers the opening
   * of another dialog). The value of the clicked button will be passed as an
   * argument to this function (if the X button is clicked, 'cancel' will
   * always be the value passed).
   */
  function yesNoCancel(params) {
    return pages.getHandler('yesNoCancelDialog').load(params);
  }
  
  /**
   * Shows a standard dialog with an 'OK' button.
   * @memberOf niagara.util.mobile.dialogs
   * @see niagara.util.mobile.dialogs.yesNoCancel
   */
  function ok(params) {
    return pages.getHandler('okDialog').load(params);
  }
  
  /**
   * Shows a standard dialog with 'OK' and 'Cancel' buttons.
   * @memberOf niagara.util.mobile.dialogs
   * @see niagara.util.mobile.dialogs.yesNoCancel
   */
  function okCancel(params) {
    return pages.getHandler('okCancelDialog').load(params);
  }
  
  /**
   * Shows a standard dialog with a 'Cancel' button.
   * @memberOf niagara.util.mobile.dialogs
   * @see niagara.util.mobile.dialogs.yesNoCancel
   */
  function cancel(params) {
    return pages.getHandler('cancelDialog').load(params);
  }
  
  /**
   * Shows a standard dialog with 'Yes' and 'No' buttons.
   * @memberOf niagara.util.mobile.dialogs
   * @see niagara.util.mobile.dialogs.yesNoCancel
   */
  function yesNo(params) {
    return pages.getHandler('yesNoDialog').load(params);
  }
  
  /**
   * Displays the error in a standard OK dialog with 'Application Error' as the
   * dialog title.
   * @memberOf niagara.util.mobile.dialogs
   * @param {String|Error|Function} err the error to be displayed
   * @param {Function} [okHandler] a function to be executed when 'OK' is clicked
   */
  error = function error(err, okHandler) {
    okHandler = okHandler || function (callbacks) { callbacks.ok(); };
    
    if (typeof err !== 'function') {
      baja.error(err);
    }
    
    $.mobile.hidePageLoadingMsg();
    return pages.getHandler('errorDialog').load({
      title: getMobileLex().get('appError'),
      content: err,
      ok: okHandler
    });
  };
  
  /**
   * Displays the error message - but the OK button will
   * refresh the entire page instead of continuing on. This represents an error
   * that should render the entire app unusable, forcing a page reload.
   *
   * @memberOf niagara.util.mobile.dialogs
   * @param {String|Error|Function} err the error to be displayed
   */
  function unrecoverableError(err) {
    var content,
        okHandler = function () { location.reload(); };
        
    if (typeof err === 'function') {
      content = err;
    } else {
      content = function (contentDiv, callbacks) {
        var errDiv = $('<div/>').css({ "margin-bottom": "1em" }).text(String(err)),
            msgDiv = $('<div/>').text(getMobileLex().get('message.clickOKToReload'));
        
        contentDiv.append(errDiv);
        contentDiv.append(msgDiv);
        callbacks.ok();
      };
      baja.error(err);
    }
        
    $.mobile.hidePageLoadingMsg();
    return pages.getHandler('errorDialog').load({
      title: getMobileLex().get('unrecoverableError'),
      content: content,
      ok: okHandler
    });
  }
  
  /**
   * Displays a single field editor in a popup dialog. The field editor will
   * be shown with OK and Cancel buttons. Cancel will simply cause the dialog
   * to close, and OK will save the field editor and pass the resulting value
   * into the input <code>ok</code> callback.
   * 
   * <p>Example:
   * 
   * <pre>
   *   niagara.util.mobile.dialogs.fieldEditor({
   *     title: 'Please select a stooge',
   *     value: baja.DynamicEnum.make({
   *       range: baja.EnumRange.make({
   *         ordinals: [0, 1, 2],
   *         tags: ['Moe', 'Larry', 'Curly']
   *       })
   *     }),
   *     ok: function (selectedValue) {
   *       alert('you selected "' + selectedValue.getTag() + '".');
   *     }
   *   });
   * </pre>
   * 
   * @memberOf niagara.util.mobile.dialogs
   * 
   * @param {Object} params an object literal containing dialog parameters
   * 
   * @param {String} [params.title] the dialog title
   * 
   * @param {baja.Value} params.value the value to be edited using a field
   * editor
   * 
   * @param {Object} [params.editorParams] any extra field editor parameters
   * to be passed into <code>niagara.fieldEditors.getFieldEditor</code> when
   * constructing the field editor
   * 
   * @param {Function} params.ok a function to be executed when the field
   * editor is saved (the new value from the field editor will be passed in
   * as the first parameter)
   * 
   * @param {Function} [params.fail] a function to be executed if the field
   * editor fails to load
   * 
   * @param {Function} [params.cancel] a function to be executed when 'Cancel'
   * is clicked
   */
  function fieldEditor(params, callbacks) {
    callbacks = callbackify(callbacks);
    
    var fe = niagara.fieldEditors,
        value = params.value,
        AllSlotsEditor;
    
    function showEditor(editor) {
      okCancel({
        title: params.title || String(value.getType()),
    
        content: function (contentDiv, contentCallbacks) {
          editor.buildAndLoad(contentDiv, {
            ok: function () {
              callbacks.ok(editor);
              contentCallbacks.ok(editor);
            },
            fail: function (err) {
              callbacks.fail(err);
              contentCallbacks.fail(err);
            }
          });
        },
        
        ok: function (callbacks) {
          validateAndSaveWorkflow.invoke(editor, {
            ok: function (value) {
              if (params.ok) {
                params.ok(value, callbacks);
              } else {
                callbacks.ok();
              }
            },
            fail: function (err) {
              callbacks.fail(err);
            } 
          });
        },
        
        cancel: params.cancel,
        callbacks: params.callbacks
      });
    }
    
    if (fe.isRegistered(value)) {
      fe.makeFor(params, {
        ok: showEditor,
        fail: callbacks.fail
      });
    } else {
      AllSlotsEditor = fe.composite.allSlots();
      showEditor(new AllSlotsEditor(value));
    }
  }

  
  /**
   * Dialog to fire an action on a component. Note that because this dialog
   * type is asynchronous by nature, it will therefore <i>not</i> return
   * a handler object the same way <code>yesNoCancel</code> etc. will.
   * 
   * <p>If the specified action requires an argument, <code>action()<code> will
   * pop up a <code>fieldEditor</code> for that argument before firing the
   * action. If no argument is required, <code>action()</code> will simply
   * fire the action.
   * 
   * <p>In either case, if the action slot has the <code>CONFIRM_REQUIRED</code>
   * flag set, a yes/no dialog will be shown for confirmation before firing
   * the action.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @param {Object} params an object containing parameters for the dialog
   * @param {baja.Component} params.component the component on which to fire
   * an action
   * @param {String|baja.Slot} params.slot the slot for the action to fire
   * @param {baja.Value} [params.parameter] the parameter to give to the action -
   * if omitted, will retrieve the default action parameter asynchronously from
   * the server - or will just not use one if not needed
   * @param {Object} [params.editorParams] parameters to be passed to the
   * field editor for the action parameter, if one is used
   */
  function action(params) {
    params = baja.objectify(params);
    
    var actionArg = null,
        slot = params.slot,
        actionSlot,
        arg = baja.def(params.parameter, ""),
        comp = params.component,
        callbacks = callbackify(params.callbacks),
        paramType,
        showDialog;
        
    baja.strictArg(comp, baja.Component);
    
    // Make sure we have an Action
    actionSlot = comp.getSlot(slot);
    if (!actionSlot.isAction()) {
      throw new Error("slot '" + slot + "' is not an action");
    }
    
    paramType = actionSlot.getParamType();
    
    if (!arg.equals("") && paramType !== null) {
      if (arg.getType().is(paramType)) {
        actionArg = arg;
      }
      else if (typeof arg === "string" && paramType.isSimple()) {
        actionArg = paramType.getInstance().decodeFromString(arg);
      }
    }
    
    // Invoke the Action
    function invoke(callbacks) {
      callbacks = callbackify(callbacks);
      
      comp.invoke({
        slot: actionSlot, 
        value: actionArg,
        ok: callbacks.ok,
        fail: callbacks.fail
      });
      
      // Manually do a sync so we can a response ASAP.
      baja.clock.schedule(function () {
        baja.station.sync();
      }, 500);
    }
    
    showDialog = actionArg === null && paramType !== null;
    
    // Check for dialog input and invoke the dialog
    function dialogInvoke() {
      
      // Dialog for Action?
      if (showDialog) {
        // Make network call to get the default argument for the Action invocation
        comp.getActionParameterDefault({
          slot: actionSlot,
          ok: function (actionDefArg) {
            actionArg = actionDefArg;
            
            // Use facets if not already provided
            if (params && !params.facets) {
              params.facets = util.slot.getFacets(comp);
            }
            
            // Ask the user to enter a value for the Action
            fieldEditor($.extend(params, {
              title: comp.getDisplayName(actionSlot),
              value: actionArg,
              ok: function (val, callbacks) {
                actionArg = val;
                invoke(callbacks);
              },
              callbacks: callbacks
            }));
          }
        });
      }
      else {
        invoke(callbacks);
      }
    }
          
    // Confirm required?
    if ((comp.getFlags(actionSlot) & baja.Flags.CONFIRM_REQUIRED) === baja.Flags.CONFIRM_REQUIRED) {
      yesNo({
        content: getMobileLex().get({
                   key: "invoke.confirm", 
                   args: comp.getDisplayName(actionSlot)
                 }),
        yes: dialogInvoke,
        callbacks: callbacks
      });
    }
    else {
      dialogInvoke();
      // If there are going to be no further dialogs then close the current one
      if (!showDialog) {
        closeCurrent();
      }
    }    
  }
  
  /**
   * To be called when navigating away from a page that still has
   * unsaved changes. Will display a yes/no/cancel dialog asking the user to
   * confirm whether or not to save the outstanding changes. The dialog title
   * and content will automatically use the appropriate mobile lexicon entries.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @see niagara.util.mobile.dialogs.yesNoCancel
   * @param {String} params.viewName the name of the view that has unsaved
   * changes
   * @param {String|jQuery|Function} [params.redirect] the page to navigate to
   * if the user choose to save or abandon changes (i.e. clicks 'yes' or 'no'
   * and not 'cancel')
   */
  function confirmAbandonChanges(params) {
    params = baja.objectify(params);

    var lex = getMobileLex(),
        noop = function (callbacks) { callbacks.ok(); },
        yes = params.yes || noop,
        no = params.no || noop,
        redirect = params.redirect;
    
    params.title = lex.get('abandonChanges');
    params.content = lex.get({
      key: 'message.confirmUnsavedChanges',
      args: [params.viewName]
    });
    
    //this is the page with unsaved changes - if we don't redirect in yes()
    //or no(), we go back here
    if (!params.returnPage) {
      params.returnPage = util.mobile.decodePageId(pages.getCurrent());
    }
    params.yes = function (callbacks) {
      if (redirect) {
        this.redirect(redirect);
      }
      yes.call(this, callbacks);
    };
    params.no = function (callbacks) {
      if (redirect) {
        this.redirect(redirect);
      }
      no.call(this, callbacks);
    };
    
    yesNoCancel(params);
  }
  
  
  /**
   * Displays a registered dialog page.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @param {String} pageId the page ID of the dialog to show (must have
   * already be registered via <code>registerDialog</code>)
   * @param {Object} params an object containing dialog parameters - content,
   * button handlers, etc.
   */
  function showDialog(pageId, params) {
    var handler = pages.getHandler(pageId);
    if (handler) {
      handler.load(params);
    } else {
      error(getMobileLex().get({
        key: 'dialogs.noPageRegistered',
        args: [pageId]
      }));
    }
  }
  
  /**
   * Binds to the window resize event - always attempts to re-center the
   * dialog vertically whenever the window is resized (or, on mobile devices,
   * orientation changed). Throttled to 500ms. 
   * 
   * <p>This function is self-executing - will be run just
   * by including the js file.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @private
   */
  function repositionOnResize() {
    $(window).bind('resize', baja.throttle(function () {
      var page = $.mobile.activePage;
      if (page && (page.jqmData('role') === 'dialog')) {
        repositionDialog(page);
      }
    }, 500));
  }
  
  /**
   * Prevent Pages handler from navigating us to the wrong page in case
   * we're trying to initially navigate to a dialog on first page load. If
   * the window hash contains <code>&ui-state=dialog</code> when we first
   * load the page, have JQM redirect us to <code>#</code>. (This doesn't
   * work on Android, so you'll be left with a <code>&ui-state=dialog</code>
   * hanging on the URL, but won't do any harm.)
   * 
   * <p>This function is self-executing - will be run just
   * by including the js file.
   * 
   * @memberOf niagara.util.mobile.dialogs
   * @private
   */
  function noDialogOnLoad() {
    var hash = window.location.hash,
        dialogStr = '&ui-state=dialog';
    if (hash.indexOf(dialogStr) > 0) {
      baja.started(function () {
        $.mobile.changePage(hash.replace(dialogStr, ''), {
          changeHash: false,
          transition: 'none'
        });
      });
    }
  }

  repositionOnResize();
  noDialogOnLoad();
  
  /**
   * @namespace
   * @name niagara.util.mobile.dialogs
   */
  util.api('niagara.util.mobile.dialogs', {
    'public': {
      DialogInvocation: DialogInvocation,
      
      registerDialog: registerDialog,
      repositionDialog: repositionDialog,
      showDialog: showDialog,
      
      action: action,
      cancel: cancel,
      closeCurrent: closeCurrent,
      confirmAbandonChanges: confirmAbandonChanges,
      error: error,
      fieldEditor: fieldEditor,
      ok: ok,
      okCancel: okCancel,
      unrecoverableError: unrecoverableError,
      yesNoCancel: yesNoCancel,
      yesNo: yesNo
    },
    'private': {
      createDialogPage: createDialogPage,
      loadContent: loadContent,
      noDialogOnLoad: noDialogOnLoad,
      repositionOnResize: repositionOnResize
    }
  });
}());