/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to mobile Views in Niagara web apps.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true */

/*globals $, baja, niagara */



(function () {
  "use strict";

  niagara.util.require(
    '$.mobile',
    'niagara.util.flow',
    'niagara.util.nav',
    'niagara.util.View',
    'niagara.util.mobile.pages',
    'niagara.util.mobile.dialogs',
    'niagara.util.mobile.commands'
  );
  
  var util = niagara.util,
      mobileUtil = util.mobile,
      pages = mobileUtil.pages,
      dialogs = mobileUtil.dialogs,
      commands = mobileUtil.commands,
      
      encodePageId = mobileUtil.encodePageId,
      decodePageId = mobileUtil.decodePageId,
      
      callbackify = util.callbackify,
      
      AsyncCommand = commands.AsyncCommand,
      View = util.View,
      MobileView,
      ListView,
      RadioButtonView,
      PageView,
      ListPageView;
  
////////////////////////////////////////////////////////////////
//Utility functions
////////////////////////////////////////////////////////////////


  
  /**
   * Check to see if clicking this property in the property sheet should link
   * new a new property sheet for this property.
   * 
   * @memberOf niagara.util.mobile
   * @private
   * 
   * @param {baja.Property} prop property to check for linkability
   * @returns {Boolean} true if this is a subcomponent we can show on its own
   * property sheet
   */
  function isLinkable(prop) {
    return prop.getType().isComponent() || 
      prop.getType().isStruct();
  }
  
  
  
  

  
  ////////////////////////////////////////////////////////////////
  // MobileView
  ////////////////////////////////////////////////////////////////
  
  /**
   * A View subclass that includes behavior specific to jQuery Mobile.
   * 
   * @class
   * @memberOf niagara.util.mobile
   * @extends niagara.util.View
   */
  MobileView = View.subclass();
  
  
  /**
   * Simply calls <code>trigger('create')</code> to tell JQM to enhance any
   * widgets contained in this view.
   * 
   * @name niagara.util.mobile.MobileView#refreshWidgets
   * @function
   */
  MobileView.prototype.refreshWidgets = function refreshWidgets() {
    this.$dom.trigger('create');
  };
  
  ////////////////////////////////////////////////////////////////
  // ListView
  ////////////////////////////////////////////////////////////////
  
  /**
   * A Complex list view widget, displaying component slots in the form of
   * a JQM <code>listview</code>. Supports dynamic listing of component
   * properties and inline expansion of properties.
   * <p>
   * Note that only Property slots will be displayed - Actions and Topics
   * will not.
   * 
   * @class
   * @memberOf niagara.util.mobile
   * @extends niagara.util.mobile.MobileView
   */
  ListView = MobileView.subclass(function ListView() {
    this.elements = {};
  });
  
  /**
   * A function used to restrict which slots are displayed in the list view.
   * By default, only returns slots which are Properties.
   * 
   * @name niagara.util.mobile.ListView#shouldIncludeSlot
   * @function 
   * @param {baja.Slot} slot 
   * @returns {Boolean} true if the given slot should be included.
   */
  ListView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
    return slot.isProperty() && !slot.isAction() && !slot.isTopic();
  };
  
  /**
   * Arms a handler on expandable slots (slots for which 
   * <code>makeListItemParams</code> returns <code>expandable === true</code>)
   * that will call <code>this.doExpand</code> when they are clicked.
   * 
   * @name niagara.util.mobile.ListView#armHandlers</code>
   * @function
   * @see niagara.util.mobile.View#armHandlers
   */
  ListView.prototype.armHandlers = function armHandlers() {
    var that = this;
    
    that.$dom.delegate('a.expandable', 'click', function () {
      var slotName = $(this).children('span.slotName').html();
      
      that.doExpand(slotName);
      
      return false;
    });
    
    that.$dom.delegate('a.link', 'click', function (e) {
      var a = $(this),
          slotName = a.parents('li.property').attr('id');
      that.linkClicked(decodePageId(slotName), a);
    });
  };
  
  
  /**
   * Loads a Complex into the listview, building up the HTML necessary to
   * display its slots. Delegates the actual construction of the
   * <code>&lt;ul&gt;</code> to <code>makeListView</code>.
   * 
   * @name niagara.util.mobile.ListView#doLoadValue
   * @function
   * @param {baja.Complex} the Complex value to be displayed in the list
   * @param {Object} callbacks an object with ok/fail callbacks to execute once 
   * HTML is fully built
   * 
   * @see niagara.util.mobile.View#doLoadValue
   */
  ListView.prototype.doLoadValue = function doLoadValue(complex, callbacks) {

    baja.strictArg(complex, baja.Complex);
    
    var that = this,
        highlightedSlot = that.$highlightedSlot;
    
    this.makeListView(complex, {
      ok: function (ul) {
        that.elements.ul = ul;
        that.$dom.empty().append(ul).trigger('updatelayout');
        if (highlightedSlot) {
          that.highlight(highlightedSlot);
        }
        callbacks.ok();
      },
      fail: callbacks.fail
    });
  };
  
  /**
   * When expanding an entry in a ListView, this will scroll the screen far
   * enough that the entire expanded div will be in view (but not so far that
   * the list entry itself will be scrolled off the top of the screen). This
   * will be called from <code>doExpand()</code>.
   * 
   * @name niagara.util.mobile.ListView#scrollExpandedDivIntoView
   * @function
   * @private
   * @param {jQuery} expandedDiv the div that has just been expanded
   * @see niagara.util.mobile.ListView#doExpand
   */
  ListView.prototype.scrollExpandedDivIntoView = function scrollExpandedDivIntoView(expandedDiv) {
    //after we expand out the slot, let's check to see if the expanded
    //div goes past the bottom of our viewable area - and if it does,
    //scroll down a bit so we can see it all
    var activePage = $.mobile.activePage,
        divBottom = expandedDiv.offset().top + expandedDiv.height(),
        headerHeight = activePage.children(':jqmData(role=header)').height(),
        footerHeight = activePage.children(':jqmData(role=footer)').height(),
        $window = $(window),
        scrollTop = $window.scrollTop(),
        
        //the bottom of the content area / top edge of the footer bar
        scrollBottom = scrollTop + $window.height() - footerHeight,
        //don't autoscroll the property li off the top of the screen
        maxTop = expandedDiv.prev().offset().top - headerHeight;

    if (divBottom > scrollBottom) {
      //do the scroll
      $.mobile.silentScroll(Math.min(maxTop, scrollTop + (divBottom - scrollBottom)));
    } else {
      //no scrolling necessary, but trigger a scroll event anyway so
      //our persistent header/footer bars reappear in the right place
      expandedDiv.trigger('updatelayout');
    }
  };
  
  
  /**
   * Upon clicking a component property, will load up a div (populated using
   * the override hook <code>populateExpandedDiv</code>) and display it
   * inline after the <code>&lt;li&gt;</code> for the selected slot. Calling
   * this function a second time will hide that div.
   * 
   * @name niagara.util.mobile.ListView#doExpand
   * @function
   * 
   * @param {String|baja.Slot} slotName the property slot to show an expansion
   * for
   * @param {Function} callback a function that will be executed after the
   * jQuery show/hide animation is complete
   */
  ListView.prototype.doExpand = function doExpand(slotName, callback) {
    var that = this,
        slot = that.value.getSlot(slotName),
        slotId = encodePageId(slot.getName()),
        li = that.elements.ul.children('li#' + slotId),
        expandedDiv = li.next('.expanded'),
        arrowSpan = li.find('span.ui-icon')
          .removeClass('ui-icon-arrow-u ui-icon-arrow-d ' +
                       'ui-icon-arrow-r ui-icon-arrow-l');
    
    function after() {
      that.scrollExpandedDivIntoView(expandedDiv);
      if (callback) {
        callback();
      }
    }
    
    if (expandedDiv.length) {
      if (expandedDiv.is(':visible')) {
        arrowSpan.addClass('ui-icon-arrow-d');
        expandedDiv.hide('fast', after);
      } else {
        arrowSpan.addClass('ui-icon-arrow-u');
        expandedDiv.show('fast', after);
      }
    } else {
      arrowSpan.addClass('ui-icon-arrow-u');
      expandedDiv = $('<div class="expanded"/>')
        .insertAfter(li)
        .hide();
      this.populateExpandedDiv(slot, expandedDiv);
      expandedDiv.show('fast', after);
    }
  };
  
  /**
   * Highlights the selected property in blue (or whatever color is dictated 
   * by the JQM theme).
   * 
   * @name niagara.util.mobile.ListView#highlight
   * @function
   * 
   * @param {String|baja.Property} prop the name of the property to highlight
   */
  ListView.prototype.highlight = function highlight(prop) {
    this.removeHighlight();
    
    var ul = this.elements.ul,
        pageId = encodePageId(prop);
    this.$highlightedSlot = prop;
    ul.children('li#' + pageId).addClass('ui-btn-active');
  };
  
  /**
   * Removes highlighting from all list items.
   * @name niagara.util.mobile.ListView#removeHighlight
   * @function
   */
  ListView.prototype.removeHighlight = function () {
    delete this.$highlightedSlot;
    this.elements.ul.children().removeClass('ui-btn-active');
  };
  
  /**
   * Executed whenever a linkable list item is clicked.
   * 
   * @name niagara.util.mobile.ListView#linkClicked
   * @function
   */
  ListView.prototype.linkClicked = function linkClicked(prop, a) {
    this.highlight(prop);
  };
  
  /**
   * Given an expanded property, populate the expanded div with any information,
   * form elements, etc. that should be shown. Default behavior is to do
   * nothing (and may be left that way if you do not plan to do any inline
   * property expansion in your view).
   * 
   * @name niagara.util.mobile.ListView#populateExpandedDiv
   * @function
   * 
   * @param {baja.Property} prop the property being expanded
   * @param {jQuery} expandedDiv the div that is expanded to show property
   * details
   */
  ListView.prototype.populateExpandedDiv = function populateExpandedDiv(prop, expandedDiv) {

  };
  
  /**
   * Creates the <code>&lt;ul&gt;</code> that contains the list of all this
   * component's properties. Each item is built using 
   * <code>makeListItem()</code>. Note that the output object is raw
   * HTML and has not yet been enhanced with jQuery Mobile widgets.
   * 
   * @name niagara.util.mobile.ListView#makeListView
   * @function
   * 
   * @param {baja.Complex} component the child component of the view's main
   * component that we are creating a list item for
   * @param {Object} callbacks an object with ok/fail callbacks
   * @param {Function} callbacks.ok a (non-optional!) function that will 
   * receive the created <code>&lt;ul&gt; element as the first parameter
   */
  ListView.prototype.makeListView = function makeListView(complex, callbacks) {
    callbacks = callbackify(callbacks);
    var that = this,
        ul = $('<ul data-role="listview" data-theme="c" />');
    
    complex.getSlots(function filter(slot) {
      return that.shouldIncludeSlot(slot);
    }).each(function (prop) {
      if (!util.slot.isHidden(prop)) {
        var params = that.makeListItemParams(complex, prop);
        ul.append(that.makeListItem(complex, prop, params));
      }
    });
    
    callbacks.ok(ul);
  };
  
  /**
   * Adds icons to a <code>&lt;li&gt;</code> element for display. Any existing
   * icons will be removed and replaced with the new ones.
   * 
   * @name niagara.util.mobile.ListView#prependIcons
   * @function
   * @param {jQuery} li the listview item to add icons to
   * @param {baja.Icon} icon the icon from a Baja component
   */
  ListView.prototype.prependIcons = function prependIcons(li, icon) {
    if (!icon) {
      return;
    }
    
    var icons = $(),
        existingImages = li.find('img.ui-li-icon'),
        updateNeeded = false;
    
    baja.iterate(icon.getImageUris(), function (url, i) {
      icons = icons.add($('<img/>').attr('src', url).addClass('ui-li-icon'));
      if (existingImages.eq(i).attr('src') !== url) {
        updateNeeded = true;
      }
    });
    
    if (updateNeeded) {
      existingImages.remove();
      li.prepend(icons);
    }
  };
  
  /**
   * Creates an <code>&lt;li&gt;</code> element to be added to the view's
   * property list.
   * 
   * @name niagara.util.mobile.ListView#makeListItem
   * @function
   * 
   * @param {baja.Complex} component the component being displayed in this
   * view (this will usually just be <code>this.value</code> - but not
   * always, we may want to custom-create list items from the output of
   * a server-side call, etc)
   * @param {baja.Property} prop the property on <code>component</code> we are
   * creating a list item for
   * @param {Object} [params] optional parameters to be used when building
   * the list item - if omitted it will default to the output of
   * <code>this.makeListItemParams</code>
   */
  ListView.prototype.makeListItem = function makeListItem(complex, prop, params) {
    var li = $('<li class="property"/>')
          .attr('id', encodePageId(String(prop))),
        a;

    params = params || this.makeListItemParams(complex, prop);

    this.prependIcons(li, params.icon);

    $('<label class="title" />') //label with property name
      .text(params.title || '')
      .appendTo(li);

    //  holds the original slot name - hold on to a copy for use when handling
    //  slot renames
    $('<span class="hidden slotName" />')
      .text(String(prop))
      .appendTo(li);

    //display output of this slot
    $('<div class="display" />') 
      .text(params.display || '')
      .appendTo(li);

    if (params.linkable || params.expandable) {
      a = $('<a />');
      a.jqmData('ord', params.ord);

      if (params.expandable) {
        a.addClass('expandable');
        li.attr('data-icon', 'arrow-d');
      } else {
        a.addClass('link');
      }
      li.wrapInner(a);
    }

    return li;
  };
  
  /**
   * Creates the necessary parameters to build up a <code>&lt;li&gt;</code>
   * element for a given property. The returned object can include the following
   * properties:
   * 
   * <ul>
   * <li><code>title</code>: the string title (commonly the property display
   * name)</li>
   * <li><code>display</code>: the display value to show on the right hand
   * side, if any</li>
   * <li><code>icon</code>: a <code>baja.Icon</code> icon to display for this 
   * property</li>
   * <li><code>expandable</code>: true if this list item should respond to a
   * click by expanding a div below it (populated using
   * <code>populateExpandedDiv()</code>)</li>
   * <li><code>linkable</code>: true if this property should respond to a click
   * by linking to a subcomponent</li>
   * <li><code>ord</code>: the ORD (baja.Ord or String) that this list item
   * should link to - only used if <code>linkable</code> is also true
   * </ul>
   * 
   * These flags will cause the <code>&lt;a&gt;</code> tags within the 
   * <code>&lt;li&gt;</code> to be given CSS classes of <code>expandable</code> 
   * or <code>link</code>. Basic click handlers will be added by default by
   * <code>PageViewManager.armHandlers()</code> - any additional functionality
   * should be handled by your own application.
   * 
   * @name niagara.util.mobile.ListView#makeListItemParams
   * @function
   * 
   * @param {baja.Complex} component the component whose property is displayed
   * in the list item (the same one passed to <code>makeListItem</code>
   * @param {baja.Property} prop the property on <code>component</code> being 
   * displayed
   * @returns {Object} an object literal with parameters for the list item
   */
  ListView.prototype.makeListItemParams = function makeListItemParams(complex, prop) {
    var subComponent = complex.get(prop),
        ord = util.ord.deriveOrd(subComponent);
    
    if (prop.getType().is('baja:VirtualGateway')) {
      ord += '|virtual:/';
    }
    
    return {
      ord: ord,
      title: complex.getDisplayName(prop),
      display: baja.SlotPath.unescape(complex.getDisplay(prop)),
      icon: subComponent.getIcon(),
      expandable: util.slot.isEditable(prop),
      linkable: isLinkable(prop) && !util.slot.isReadonly(prop)
    };
  };
  
  /**
   * Updates the display value on the <code>&lt;li&gt;</code> in this view
   * for the specified property.
   * 
   * @name niagara.util.mobile.ListView#updateDisplay
   * @function
   * 
   * @param {baja.Property} prop the property to be updated
   */
  ListView.prototype.updateDisplay = function updateDisplay(prop) {
    var id = encodePageId(String(prop)),
        li = this.elements.ul.find('li.property#' + id),
        displayDiv = li.find('div.display'),
        value = this.value.get(prop);
    if (value && value.getType().isComponent()) {
      this.prependIcons(li, value.getIcon());
    }
    if (displayDiv.length) {
      displayDiv.text(this.value.getDisplay(prop));
    }
  };
  
  /**
   * Arms ListView with additional responses to Baja events.
   * Keeps list items updated on <code>changed</code> events, adjusts
   * display titles and ORD links on <code>renamed</code> events, and
   * disables list items on <code>removed</code> events.
   * 
   * @name niagara.util.mobile.ListView#makeSubscriber
   * @see niagara.util.mobile.View#makeSubscriber
   * @function
   */
  ListView.prototype.makeSubscriber = function makeSubscriber() {
    var that = this,
        sub = new baja.Subscriber();
    
    sub.attach("subscribed changed", function (prop, cx) {
      that.updateDisplay(prop);
    });
    
    sub.attach('renamed', function (prop, oldName, cx) {
      var propId = encodePageId(prop.getName()),
          oldPropId = encodePageId(oldName),
          li = that.elements.ul.children('li.property#' + oldPropId),
          newOrd = util.ord.deriveOrd(that.value.get(prop));
      
      li.find('label.title')
        .addClass('renamed')
        .text(that.value.getDisplayName(prop));
      li.find('span.slotName').text(prop.getName());
      li.find('a.link').jqmData('ord', newOrd);
      li.attr('id', propId);
    });
    
    sub.attach("removed", function (prop, val, cx) {
      var propId = encodePageId(prop.getName()),
          li = that.elements.ul.children('li.property#' + propId);
      
      li.find('a').removeClass('link expandable');
      li.find('label.title').addClass('removed').append(' (removed)');
    });
    
    return sub;
  };
  
  
  
  
  ////////////////////////////////////////////////////////////////
  // RadioButtonView
  ////////////////////////////////////////////////////////////////
  
  /**
   * A View that displays the slots on a Complex in the form of a set of
   * JQM radio buttons.
   * 
   * @class
   * @name niagara.util.mobile.RadioButtonView
   * @extends niagara.util.mobile.MobileView
   */
  RadioButtonView = MobileView.subclass();
  
  /**
   * Builds up the set of radio buttons. Iterates over the slots on the
   * component (filtered by <code>shouldIncludeSlot</code>) and appends the
   * results of <code>makeInput</code> and <code>makeLabel</code> for each one.
   * 
   * @name niagara.util.mobile.RadioButtonView#doLoadValue
   * @function
   * @param {baja.Complex} value a Complex to assemble into radio buttons
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  RadioButtonView.prototype.doLoadValue = function doLoadValue(value, callbacks) {
    baja.strictArg(value, baja.Complex);
    
    var that = this,
        fieldset = $('<fieldset data-role="controlgroup" />'),
        filteredSlots = value.getSlots(function (slot) {
          return that.shouldIncludeSlot(slot);
        });
    
    baja.iterate(filteredSlots, function (slot) {
      fieldset
        .append(that.makeInput(value, slot))
        .append(that.makeLabel(value, slot));
    });
    
    this.$dom.empty().append(fieldset);
    callbacks.ok();
  };
  
  /**
   * Arms an event listener to call <code>selectionChanged</code> whenever the
   * user selects a different radio button.
   * 
   * @name niagara.util.mobile.RadioButtonView#armHandlers
   * @function
   */
  RadioButtonView.prototype.armHandlers = function () {
    var that = this;
    this.$dom.delegate('input[type=radio]', 'change', function () {
      that.selectionChanged($(this).jqmData('slot'));
    });
  };
  
  /**
   * Called whenever the user selects a different radio button.
   * 
   * @name niagara.util.mobile.RadioButtonView#selectionChanged
   * @function
   * @param {baja.Slot} slot the slot corresponding to the selected radio
   * button
   */
  RadioButtonView.prototype.selectionChanged = function selectionChanged(slot) {
    this.$selectedSlot = slot;
  };
  
  /**
   * Programmatically set the selected radio button. If the given slot does
   * not correspond to an available radio button, all radio buttons will be
   * unchecked.
   * 
   * @name niagara.util.mobile.RadioButtonView#setSelectedSlot
   * @function
   * @param {baja.Slot|String} slot the slot to select
   */
  RadioButtonView.prototype.setSelectedSlot = function setSelectedSlot(slot) {
    slot = String(slot);
    
    var pageId = encodePageId(slot),
        input =  this.$dom.find('input#' + pageId);
    
    slot = this.value.getSlot(slot);
    
    if (slot && input.length) {
      this.$selectedSlot = slot;
      input.attr('checked', true).checkboxradio('refresh');
    } else {
      delete this.$selectedSlot;
      this.$dom.find('input[type=radio]').attr('checked', false);
    }
    
    this.selectionChanged(slot);
  };
  
  /**
   * Returns the currently selected slot.
   * 
   * @name niagara.util.mobile.RadioButtonView#getSelectedSlot
   * @function
   * @returns {baja.Slot} slot the currently selected slot
   */
  RadioButtonView.prototype.getSelectedSlot = function getSelectedSlot() {
    return this.$selectedSlot || this.$dom.find('input:checked').jqmData('slot');
  };
  
  /**
   * Returns the currently selected value (equivalent to
   * <code>this.value.get(this.getSelectedSlot())</code>).
   * 
   * @name niagara.util.mobile.RadioButtonView#getSelectedValue
   * @function
   * @returns {baja.Value} the currently selected value
   */
  RadioButtonView.prototype.getSelectedValue = function getSelectedValue() {
    var slot = this.getSelectedSlot();
    if (slot) {
      return this.value.get(slot);
    }
  };
  
  /**
   * Creates the <code>&lt;input&gt;</code> element to be appended for each
   * slot on the <code>Complex</code> to be displayed.
   * 
   * @name niagara.util.mobile.RadioButtonView#makeInput
   * @function
   * @param {baja.Complex} complex the Complex whose slots are to be assembled
   * into radio buttons
   * @param {baja.Slot} slot the slot to make into a radio button input
   * @returns {jQuery} the assembled <code>&lt;input&gt;</code> element
   */
  RadioButtonView.prototype.makeInput = function (complex, slot) {
    slot = String(slot);
    var pageId = encodePageId(slot),
        nameId = encodePageId(complex.getName()),
        input = $('<input type="radio" />')
            .attr('name', nameId)
            .attr('id', pageId)
            .attr('value', pageId);
    if (slot === String(this.$selectedSlot)) {
      input.attr('checked', true);
    }
    input.jqmData('slot', slot);
    return input;
  };
  
  /**
   * Creates the <code>&lt;label&gt;</code> element to be used for display
   * after a certain <code>&lt;input&gt;</code> radio button.
   * 
   * @name niagara.util.mobile.RadioButtonView#makeLabel
   * @function
   * @param {baja.Complex} complex the Complex whose slots are to be assembled
   * into radio buttons
   * @param {baja.Slot} slot the slot to make into a label
   * @returns {jQuery} the assembled <code>&lt;label&gt;</code> element
   */
  RadioButtonView.prototype.makeLabel = function (complex, slot) {
    var pageId = encodePageId(String(slot)),
        display = complex.getDisplay(slot),
        displayName = baja.SlotPath.unescape(String(slot)),
        label = $('<label data-theme="c"/>').attr('for', pageId);
    
    label
      .append($('<span/>').text(displayName))
      .append($('<br/>'))
      .append($('<span/>').text(display));
    
    return label;
  };
  
  /**
   * Decides whether or not a certain slot should be included in the set of
   * radio buttons. By default, only slots that are Properties are included.
   * 
   * @name niagara.util.mobile.RadioButtonView#shouldIncludeSlot
   * @function
   * @param {baja.Slot} slot a slot to test for inclusion
   * @returns {Boolean} true if the slot should be displayed
   */
  RadioButtonView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
    return slot.isProperty() && !slot.isAction() && !slot.isTopic();
  };
  
  ////////////////////////////////////////////////////////////////
  // PageView
  ////////////////////////////////////////////////////////////////
  
  /**
   * A view designed to take up an entire jQuery mobile page. Its job is
   * twofold: 1) to create the HTML necessary to create a new JQM page, and
   * 2) to populate that page's content div with another View (e.g.
   * ListView).
   * <p>
   * In most cases there will be no need to call any functions on this class
   * directly - if subclassing, simply override them as needed and
   * <code>PageViewManager</code> should do the work for you.
   * 
   * @class
   * @memberOf niagara.util.mobile
   * @extends niagara.util.mobile.MobileView
   * 
   * @param {baja.Component} component the component to be displayed
   * @param {Object} [options] options to customize this particular view
   * @param {Function} [options.contentViewConstructor] the constructor for
   * the View subclass to display in the page's content div
   * @param {String} [options.title] a title to display in the page's header
   * div - if omitted will fall back to <code>View.getName()</code>.
   */
  PageView = MobileView.subclass(function PageView(options) {
    options = baja.objectify(options, 'contentViewConstructor');
    this.contentViewConstructor = options.contentViewConstructor;
    this.$title = options.title;
    this.$ignoreProfileClasses = "";
  });
  

  
  ////////////////////////////////////////////////////////////////
  // PageView basic behavior
  ////////////////////////////////////////////////////////////////
  
  /**
   * Ignores profile-related CSS classes in the page view's header div. This
   * essentially turns off the functionality whereby you can disable the
   * header / back button / etc using the user's web profile - by ignoring
   * one of the following CSS classes, the corresponding header element will
   * <i>always</i> be shown. You may include any or all of the applicable
   * CSS classes in a single string separated by spaces.
   * 
   * <ul>
   *  <li><code>profileHeader</code> - always show the page header</li>
   *  <li><code>profileHeaderBack</code> - always show the back button whenever
   *  the page header is visible (hiding the entire header will still hide the
   *  back button)</li>
   *  <li><code>profileHeaderCommands</code> - always show the commands button
   *  whenever the page header is visible (hiding the entire header will still
   *  hide the commands button)</li>
   * </ul>
   * 
   * <p>If subclassing PageView, <code>this.$ignoreProfileClasses</code> will
   * be available as an instance variable after calling this method.
   * 
   * @name niagara.util.mobile.PageView#ignoreProfileClasses
   * @function
   * @param {String} classString a string list of CSS classes to ignore, 
   * separated by spaces
   */
  PageView.prototype.ignoreProfileClasses = function ignoreProfileClasses(classString) {
    this.$ignoreProfileClasses = classString || "";
  };

  /**
   * Creates a jQuery mobile page to hold a component view. When instantiating
   * a new view, this page's <code>div[data-role="content"]</code> element will
   * be used as its target div.
   * <p>
   * Profile-related CSS classes <code>profileHeader</code> and
   * <code>profileHeaderBack</code> will be applied to the appropriate elements
   * in the JQM page's header div (unless ignored via
   * <code>ignoreProfileClasses</code>).
   * 
   * @name niagara.util.mobile.PageView#createPage
   * @function
   * @see niagara.util.mobile.PageViewManager#initializeView
   * @see niagara.util.mobile.PageView#ignoreProfileClasses
   * @returns {jQuery} an initialized JQM page
   */
  PageView.prototype.createPage = function createPage() {
    var pageDiv = $('<div data-role="page" />'),
        headerDiv = $('<div data-role="header" data-position="fixed" data-theme="a" />')
          .appendTo(pageDiv),
        backButton = $('<a href="#" class="profileHeader profileHeaderBack" data-rel="back" data-icon="arrow-l" data-iconpos="notext"></a>')
          .appendTo(headerDiv),
        headerTitle = $('<h1 class="profileHeader viewName"/>')
          .appendTo(headerDiv),
        contentDiv = $('<div data-role="content" />')
          .appendTo(pageDiv);
    
    backButton.removeClass(this.$ignoreProfileClasses);
    headerTitle.removeClass(this.$ignoreProfileClasses);

    return pageDiv;
  };
  
  /**
   * Returns the header title for this page. If passed in as the
   * <code>title</code> parameter to the PageView's constructor, will use that
   * - otherwise falls back to <code>View.getName()</code>.
   * 
   * @name niagara.util.mobile.PageView#getName
   * @function
   * @returns {String}
   */
  PageView.prototype.getName = function getName() {
    return this.$title || (this.contentView && this.contentView.getName());
  };
  
  /**
   * Finds and returns this view's enclosing JQM page.
   * 
   * @name niagara.util.mobile.PageView#getPage
   * @function
   * @return {jQuery} the enclosing JQM page
   */
  PageView.prototype.getPage = function getPage() {
    return this.$dom;
  };
  
  /**
   * Returns the header div for this JQM page.
   * 
   * @name niagara.util.mobile.PageView#getHeaderDiv
   * @function
   * @return {jQuery} the JQM page's header div
   */
  PageView.prototype.getHeaderDiv = function getHeaderDiv() {
    return this.$dom.children(':jqmData(role=header)');
  };
  
  /**
   * Returns the content div for this JQM page.
   * 
   * @name niagara.util.mobile.PageView#getContentDiv
   * @function
   * @return {jQuery} the JQM page's content div
   */
  PageView.prototype.getContentDiv = function getContentDiv() {
    return this.$dom.children(':jqmData(role=content)');
  };
  
  /**
   * Returns the footer div for this JQM page.
   * 
   * @name niagara.util.mobile.PageView#getFooterDiv
   * @function
   * @return {jQuery} the JQM page's footer div
   */
  PageView.prototype.getFooterDiv = function getFooterDiv() {
    return this.$dom.children(':jqmData(role=footer)');
  };
  
  /**
   * Returns any view parameters associated with the PageView's containing
   * JQM page.
   * 
   * @name niagara.util.mobile.PageView#getViewQuery
   * @function
   * @return {baja.ViewQuery} an object containing the view params
   */
  PageView.prototype.getViewQuery = function getViewQuery() {
    return this.viewQuery;
  };
  
  /**
   * Simply proxies through to this.contentView.activated.
   * 
   * @name niagara.util.mobile.PageView#activated
   * @function
   */
  PageView.prototype.activated = function activated() {
    this.contentView.activated();
  };
  
  /**
   * Simply proxies through to this.contentView.deactivated.
   * 
   * @name niagara.util.mobile.PageView#deactivated
   * @function
   */
  PageView.prototype.deactivated = function deactivated() {
    this.contentView.deactivated();
  };

  
  ////////////////////////////////////////////////////////////////
  // PageView override points
  ////////////////////////////////////////////////////////////////

  /**
   * This is where the PageView instantiates the sub-View that will display
   * inside its page's content div. A ListPageView would return a ListView,
   * a TablePageView would return a TableView, etc. Note that your subclass
   * might display a different type of content view depending on the type of
   * the value being displayed.
   * <p>
   * A content view constructor can be passed in directly to
   * <code>PageView</code>'s constructor as 
   * <code>options.contentViewConstructor</code>. If so, by default, that
   * constructor will be used to instantiate the content view. If no
   * constructor was provided, then this function <i>must</i> be implemented
   * by a subclass, or an error will be thrown.
   * 
   * @name niagara.util.mobile.PageView#instantiateContentView
   * @function
   * @param {baja.Value} value the value to be displayed in the page's content
   * div
   * @return {View} the content view to be displayed
   * @throws {Error} if not implemented in a subclass
   */
  PageView.prototype.instantiateContentView = function instantiateContentView(value) {
    var Ctor = this.contentViewConstructor;
    if (Ctor) {
      return new Ctor();
    } else {
      throw new Error("instantiateContentView not implemented");
    }
  };
  
  /**
   * A PageView will not perform any enhancement after loading itself - its
   * content View will enhance itself and we don't want to trigger two
   * enhancement passes.
   * 
   * @name niagara.util.mobile.PageView#refreshWidgets
   * @see niagara.util.mobile.View#refreshWidgets
   */
  PageView.prototype.refreshWidgets = function refreshWidgets() {
  };

  /**
   * Readies a JQM page div for loading in a value. Since a PageView consists 
   * of a full JQM page (which JQM will enhance on its own), it just retrieves 
   * the <code>viewQuery</code> (if any) and the <code>pageId</code> from the
   * JQM page being displayed. No alterations will be made to the page
   * structure.
   * 
   * @name niagara.util.mobile.PageView#doInitializeDOM
   * @function
   * @param {jQuery} targetElement the div in which to insert this component
   * view's display div
   * @param {Object} [callbacks] a function with ok/fail callbacks to execute 
   * after the HTML is fully built
   * @throws {Error} if attempting to build inside of a div that is not a
   * JQM page (data-role=page)
   * @see niagara.util.mobile.View#doInitializeDOM
   */
  PageView.prototype.doInitializeDOM = function doInitializeDOM(targetElement, callbacks) {
    var that = this,
        pageData;
    
    if (targetElement.jqmData('role') !== 'page') {
      throw new Error("PageView requires a JQM page (data-role='page')");
    }
    
    pageData = targetElement.data('pageData');
    that.viewQuery = pageData && pageData.viewQuery;
    that.pageId = targetElement.attr('id');
    callbacks.ok();
  };
  
  /**
   * Binds to any <code>editorchange</code> events (triggered by field editors
   * when the user enters data) and calls <code>setModified()</code>
   * accordingly.
   * 
   * @name niagara.util.mobile.PageView#armHandlers
   * @function
   * @see niagara.util.mobile.View#armHandlers
   */
  util.aop.after(PageView.prototype, 'armHandlers', function afterArmHandlers() {
    var that = this;
    that.$dom.bind('editorchange', function (editor) {
      that.setModified(true);
    });
  });
  
  PageView.$doLoadValueWorkflow = util.flow.sequential(
    function pageViewDoLoadValue__step1__buildContentViewHtml(cx) {
      //instantiate the content view, and build its html in our content div
      var pageView = cx.pageView,
          contentView = pageView.contentView;
      
      if (!contentView) {
        contentView = pageView.instantiateContentView(cx.value);
      }

      contentView.initializeDOM(pageView.getContentDiv(), this);
    },
    function pageViewDoLoadValue__step2__loadContentViewValue(cx, contentView) {
      //load the value into the content view
      cx.pageView.contentView = contentView;
      contentView.loadValue(cx.value, this);
    },
    function pageViewDoLoadValue__step3__updateHeaderTitle(cx) {
      //update our title display
      var pageView = cx.pageView,
          name = pageView.getName();
      pageView.getHeaderDiv().find('h1').text(name);
      this.ok();
    }
  );
  
  /**
   * When loading a new value, the PageView will instantiate its content view,
   * build content View's HTML into the PageView's content div, and then load
   * the content View with the desired value.
   * 
   * @name niagara.util.mobile.PageView#doLoadValue
   * @function
   * @param {baja.Value} value the value to be displayed
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   * @see niagara.util.mobile.View#doLoadValue
   */
  PageView.prototype.doLoadValue = function doLoadValue(value, callbacks) {
    PageView.$doLoadValueWorkflow.invoke({
      pageView: this,
      value: value
    }, callbacks);
  };
  
  /**
   * Simply proxies to the content view's <code>start</code> function.
   * @name niagara.util.mobile.PageView#start
   * @function
   * @see niagara.util.mobile.View#start
   */
  PageView.prototype.start = function start(callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    if (that.contentView.hasMixin('subscribable')) {
      that.contentView.start({
        ok: function () {
          that.started = true;
          callbacks.ok();
        },
        fail: callbacks.fail
      });
    } else {
      callbacks.ok();
    }
  };
  
  /**
   * Simply proxies to the content view's <code>stop</code> function.
   * @name niagara.util.mobile.PageView#stop
   * @function
   * @see niagara.util.mobile.View#stop
   */
  PageView.prototype.stop = function stop(callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    if (that.contentView.hasMixin('subscribable')) {
      that.contentView.stop({
        ok: function() {
          that.started = false;
          callbacks.ok();
        },
        fail: callbacks.fail
      });
    } else {
      callbacks.ok();
    }
  };
  
  /**
   * Simply proxies to the content view's <code>refresh</code> function.
   * @name niagara.util.mobile.PageView#refresh
   * @function
   * @see niagara.util.mobile.View#refresh
   */
  PageView.prototype.refresh = function refresh(callbacks) {
    if (this.contentView.hasMixin('subscribable')) {
      this.contentView.refresh(callbacks);
    } else {
      callbacks.ok();
    }
  };
  
  /**
   * Saves the content view. After the content view saves successfully, 
   * <code>this.isModified</code> will be reset to false as well.
   * 
   * @name niagara.util.mobile.PageView#save
   * @function
   * @see niagara.util.mobile.View#save
   */
  PageView.prototype.save = function save(callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    that.contentView.save({
      ok: function () {
        that.setModified(false);
        callbacks.ok();
      },
      fail: callbacks.fail
    });
  };


  ////////////////////////////////////////////////////////////////
  // ListPageView
  ////////////////////////////////////////////////////////////////
  
  /**
   * A full-JQM-page view for displaying a ListView.
   * 
   * @class
   * @memberOf niagara.util.mobile
   * @extends niagara.util.mobile.PageView
   */
  ListPageView = PageView.subclass();

  ListPageView.prototype.highlight = function highlight(prop) {
    this.contentView.highlight(prop);
  };
  
  ListPageView.prototype.instantiateContentView = function instantiateContentView() {
    return new ListView();
  };
  
  
  /**
   * A manager for view widgets. This class, in conjunction with
   * <code>niagara.util.mobile.nav.NavModel</code>, manages the creation of and
   * navigation between views within a component hierarchy. jQuery Mobile pages
   * will, where possible, be dynamically created and inserted into the DOM
   * to hold <code>PageView</code>s, then removed when no longer
   * needed/accessible.
   * <p>
   * Note that this class does not perform dynamic linking between different
   * apps, e.g. a property sheet and a schedule app. It is used for
   * navigation between components using the same app. If linking to a
   * completely different app (e.g. property sheet to History), a full page
   * refresh will occur and this PageViewManager will be left behind.
   * 
   * @class
   * @memberOf niagara.util.mobile
   * 
   * @param {niagara.util.nav.NavModel} navModel a NavModel this
   * page manager should use - if omitted just defaults to a new NavModel()
   */
  function PageViewManager(navModel) {
    var that = this;
    
    that.navModel = navModel || new niagara.util.nav.NavModel();
    
    that.navModel.attach('cut', function (node) {
      var pageView = node.getValue();
      if (!(pageView instanceof PageView)) {
        return;
      }
      
      pageView.stop();
      
      //remove the old page at a point in the future - JQM barfs if we
      //remove it before the animation scrolling away from it is complete
      //TODO: this locks up the app if you navigate between two components not
      //in the same branch of the component tree and hit back/forward too fast
      setTimeout(function () {
        $('#' + pageView.pageId).remove();
      }, 1000);
    });
    
    window.onbeforeunload = function () {
      var selectedView = that.selectedView;
      
      if (selectedView.isModified()) {
        return baja.lex('mobile').get({
          key: 'message.viewModified',
          args: [selectedView.getName()]
        });
      }
    };
  }
  
  
  /**
   * Performs resolution of an ORD (the same ORD from the URL currently
   * viewed in the browser). By default, this simply resolves the ord - but
   * if your view manager needs to incorporate specialized ORD resolution
   * logic (e.g. ORDs matching a certain pattern should resolve to the output
   * of a server side call instead), it may do so by overriding this method.
   * 
   * @name niagara.util.mobile.PageViewManager#doResolveOrd
   * @function
   * @param {String} ord the ORD to resolve
   * @param {Object} callbacks an object with ok/fail callbacks
   */
  PageViewManager.prototype.doResolveOrd = function doResolveOrd(ord, callbacks) {
    util.ord.get(ord, callbacks);
  };

  /**
   * Registers navigation behavior for the Pages framework. The default
   * behavior is as follows:
   * <ul>
   * <li>ORDs will be encoded/decoded using the usual Niagara station url
   * scheme of /ord?station:|slot:/etcetc</li>
   * <li>on <code>pagecreate</code>, the ORD will be retrieved from the URL,
   * the component will be retrieved from the server via the usual Bajascript
   * calls, and a new <code>PageView</code> will be created and displayed
   * </li>
   * <li>upon every <code>pageshow</code> the page's corresponding
   * <code>PageView</code> will be retrieved and set as the currently
   * selected view (for saving etc)</li>
   * <li>the Pages <code>createPage</code> function will delegate to
   * <code>this.createPage</code>
   * </ul>
   * 
   * @name niagara.util.mobile.PageViewManager#registerPages
   * @function
   * @returns {Object} an object literal containing Pages handlers
   */
  PageViewManager.prototype.createPagesHandlers = function createPagesHandlers() {
    var that = this,
        saveCommand = new AsyncCommand("%lexicon(mobile:save)%", function (callbacks) {
          that.saveSelected(callbacks);
        }),
        refreshCommand = new AsyncCommand("%lexicon(mobile:refresh)%", function (callbacks) {
          that.refreshSelected(callbacks);
        });
    
    function createPage(obj) {
      
      var spinner = util.mobile.spinnerTicket(1000, 5000),
          pageData = obj.pageData,
          ord = pageData.ord;
      that.doResolveOrd(ord, {
        ok: function (value) {
          that.instantiateView(value, pageData, {
            ok: function (view) {
              var page = view.createPage();
              page.jqmData('view', view);
              page.jqmData('value', value);
              spinner.hide();
              obj.ok(page);
            },
            fail: function (err) {
              baja.error(err);
              spinner.hide();
            }
          });
        },
        fail: function (err) {
          baja.error(err);
          spinner.hide();
        }
      });
    }
    
    function encodeUrl(pageData) {
      var ord = pageData.ord || 'station:|slot',
          url = '/ord/' + ord;
      
      return url;
    }
    
    function decodeUrl(url) {
      //clean up the href as best we can
      var href = decodeURI(url.pathname + url.search).replace('&ui-state=dialog', ''),
          ord,
          queryList,
          viewQuery;
      
      if (href.indexOf('/ord?') === 0 || href.indexOf('/ord/') === 0) {
        ord = href.substring(5);
      } else {
        return {
          ord: 'station:|slot:'
        };
      }
      
      viewQuery = baja.Ord.make(ord).parse().get('view');

      return {
        ord: ord,
        viewQuery: viewQuery
      };
    }
    
    function pagecreate(obj) {
      var page = obj.page,
          view = page.jqmData('view'),
          value = page.jqmData('value');

      //so on page create, we want to instantiate and draw the component view
      //being shown on that page. BUT! we want that to happen immediately
      //after $.mobile.pageChange - e.g. when the page can be properly laid out
      //but not waiting until scrolling animation is fully complete, so it's
      //drawn properly before it starts to scroll onto the screen. the 
      //solution is to bind to our custom pagelayout event (see 
      //pagebeforechange in util.mobile.pages) - just be sure that the addView
      //function only executes ONCE or stuff goes haywire.
      function doAddView(e) {
        that.addView(view, page, value);
        
        //we don't want to add the view on every pageshow
        page.unbind('pagelayout', doAddView);
//        return false; //don't call the regular pagelayout method
      }
      
      page.bind('pagelayout', doAddView);
    }
    
    function pagelayout(obj) {
      var page = obj.page,
          view = page.jqmData('view'),
          value = view.value,
          type = value && value.getType(),
          selectedView = that.selectedView;
      
//      if (selectedView) {
//        selectedView.deactivated();
//      }
//      if (view) {
//        view.activated();
//      }
      
      /*
       * For virtual components, we want to refresh the view each time the
       * page is shown, as the virtual component could have gone unsubscribed
       * server-side. Perform a check to make sure we don't do a refresh on
       * the very first pageload, since it will be fresh from the server at
       * that point.
       */
      if (type && view.$layoutDone) {
        if (type.isComponent() && type.is('baja:VirtualComponent')) {
          that.refresh(view);
        }
      }
      
      view.$layoutDone = true;
    }
    
    function pageshow(obj) {
      var page = obj.page,
          view = page.jqmData('view'); //set in pagecreate
      if (view && view.pageId) {
        that.setSelectedView(view);
      }
    }
    
    function pagebeforechange(obj) {
      var selectedView = that.selectedView,
          targetUrl;
      
      if (selectedView && selectedView.isModified()) {
        targetUrl = decodePageId(obj.nextPage.attr('id'));
        
        that.confirmAbandonChanges(targetUrl);
        obj.event.preventDefault();
      }
    }
    
    function getCommands(obj) {
      var commands = that.selectedView.isModified() 
        ? [ saveCommand, refreshCommand ]
        : [ refreshCommand ];
      return commands.concat(obj.commands);
    }
    
    return {
      createPage: createPage,
      encodeUrl: encodeUrl,
      decodeUrl: decodeUrl,
      pagelayout: pagelayout,
      pageshow: pageshow,
      pagecreate: pagecreate,
      pagebeforechange: pagebeforechange,
      
      getCommands: getCommands
    };
  };
  
  /**
   * Registers the handlers created in <code>createPagesHandlers</code> with
   * the Pages framework.
   * 
   * @name niagara.util.mobile.PageViewManager#registerPages
   * @function
   * @param {String|Regex|Function} matcher a matcher to be passed to the
   * <code>niagara.util.mobile.pages.register</code> function. If omitted,
   * performs the default matching on <code>^ord</code>. 
   */
  PageViewManager.prototype.registerPages = function registerPages(matcher) {
    pages.register(matcher || function ordMatch(str) {
      return str.match(/^ord/);
    }, this.createPagesHandlers());
  };
  
  PageViewManager.$initializeViewWorkflow = util.flow.sequential(
    function step1__initializeDOM(cx) {
      cx.view.initializeDOM(cx.page, this);
    },
    function step2__loadViewValue(cx) {
      cx.view.loadValue(cx.value, this);
    },
    function step3__armHandlersAndStartView(cx) {
      cx.pageViewManager.armHandlers(cx.view);
      cx.pageViewManager.attachSubscriberEvents(cx.view);
      cx.view.start(this);
    }
  );
  /**
   * Creates a new PageView object and adds it to the screen. A JQM page
   * to hold this view must have already been created and appended to the DOM.
   * The view will be returned fully initialized, subscribed, and ready to use.
   * 
   * @private
   * @name niagara.util.mobile.PageViewManager#initializeView
   * @function
   * @param {baja.Component} component the component to create a new view for
   * @param {jQuery} page the JQM page in which the view will live
   * @param {Object} callbacks an object holding callbacks
   * @param {Function} callbacks.ok the ok handler once the view has been fully
   * constructed and initialized. The new view will be returned as an argument
   * to this function.
   * @param {Function} [callbacks.fail] the failure handler if the view could not be
   * initialized
   * @returns {niagara.util.mobile.PageView} the view added (will be an
   * instance of whatever view constructor was passed into the
   * <code>PageViewManager</code> constructor). This is passed as an argument to
   * <code>callbacks.ok</code>, not returned directly from 
   * <code>initializeView</code>.
   */
  PageViewManager.prototype.initializeView = function initializeView(view, page, value, callbacks) {
    
    PageViewManager.$initializeViewWorkflow.invoke({
      view: view,
      pageViewManager: this,
      page: page,
      value: value
    }, callbackify(callbacks));
  };
  
  /**
   * Creates a new instance of <code>PageView</code>. This is this
   * function's only job - it does not build HTML, subscribe or perform any
   * actions upon the instantiated view. This function may be overridden to
   * perform additional logic, possibly returning an instance of a different
   * <code>PageView</code> subclass depending on the type of the given
   * component.
   * 
   * <p>This function <i>must</i> be implemented in your subclass or your
   * PageViewManager will not work.
   * 
   * @name niagara.util.mobile.PageViewManager#instantiateView
   * @function
   * 
   * @param {baja.Complex} component the component we are creating a new view
   * for
   * @param {Object} [pageData] any page data the <code>PageViewManager</code>
   * may have parsed from the currently viewed URL
   * @param {Object} callbacks an object holding callbacks
   * @param {Function} callbacks.ok a function to be called once the view has
   * successfully instantiated - the view will be passed as an argument to
   * this function
   * @param {Function} [callbacks.fail] a function to be called if a view cannot
   * be instantiated
   * @return {niagara.util.mobile.PageView} a new instance of a
   * <code>PageView</code> or subclass. This will be passed as an arugment
   * to <code>callbacks.ok</code>, not returned directly from 
   * <code>instantiateView</code>.
   * @throws {Error} if not implemented in a subclass
   */
  PageViewManager.prototype.instantiateView = function instantiateView(component, pageData, callbacks) {
    throw new Error("instantiateView not implemented");
  };

  
  /**
   * Adds a component to our list of views. If the <code>component</code> 
   * parameter is an ORD, this method will retrieve that ORD and add the 
   * resulting component; otherwise the input component will be added directly.
   * After the view is added it will be set as the selected view and scrolled
   * to.
   * 
   * @name niagara.util.mobile.PageViewManager#addView
   * @function
   * @private
   * 
   * @param {String|baja.Component} component the component to be added
   * @param {jQuery} the JQM page this component is being added to. This is
   * an actual <code>:jqmData[role=page]</code> page, not just a target div.
   * @param {Function} callback a function to be called once the view has been
   * instantiated, initialized, and added to the view.
   * @return {niagara.util.mobile.PageView} the newly added view. This
   * will be passed as an argument to <code>callback</code>, not returned
   * directly from <code>addView</code>.
   */
  PageViewManager.prototype.addView = function addView(view, page, value, callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    that.initializeView(view, page, value, {
      ok: function () {
        that.navModel.add(view);
        that.setSelectedView(view);
        callbacks.ok(view);
      },
      fail: callbacks.fail
    });
  };
  
  /**
   * Arms basic necessary event handlers on a component view. The only default
   * behavior is that when a list item with class <code>expandable</code> is 
   * clicked, the corresponding component view will have its 
   * <code>doExpand()</code> function called.</li>
   * 
   * <p>Also subscribes to basic Bajascript events.
   *
   * @name niagara.util.mobile.PageViewManager#armHandlers
   * @function
   * 
   * @param {niagara.util.mobile.PageView} view the view to arm click handlers
   * on
   */
  PageViewManager.prototype.armHandlers = function armHandlers(view) {
    var that = this;
    
    view.$dom.delegate('a.link', 'click', function () {
      that.linkToOrd($(this).jqmData('ord'));
    });
  };
  
  PageViewManager.prototype.attachSubscriberEvents = function (view) {
    var that = this,
        navModel = that.navModel,
        sub = view.contentView.subscriber;
    
    if (!sub) {
      return;
    }
    
    sub.attach('added reordered flagsChanged facetsChanged', function (prop) {
      that.refresh(view);
    });
    
    sub.attach('renamed', function (prop, oldName, cx) {
      var propId = encodePageId(prop.getName()),
          propOrdId = view.pageId + '_2F' + propId,
          oldPropId = encodePageId(oldName),
          oldPropOrdId = view.pageId + '_2F' + oldPropId;

      $('#' + oldPropOrdId).attr('id', propOrdId);
    });
    
    sub.attach('unmount', function () {
      var prev = navModel.prev(view),
          prevIndex = navModel.indexOf(prev),
          selectedIndex = navModel.indexOf(that.selectedView),
          msg;

      //only need to hassle the user if he is currently looking at the unmounted
      //component, or one of its children.
      if (selectedIndex > prevIndex) {
        msg = baja.lex('mobile').get({
          key: 'propsheet.message.unmounted',
          def: 'Component "{0}" has been unmounted.',
          args: [ view.getName() ]
        });
        
        dialogs.ok({
          content: msg,
          ok: function (callbacks) {
            //iterate backwards until I find a view whose component has NOT
            //been unmounted
            while (prev && !prev.getNavOrd()) {
              prev = navModel.prev(prev);
              view = navModel.prev(view);
            }
            
            var ord = util.ord.deriveOrd(prev.getValue().value);
            
            if (ord && ord !== 'null') {
              that.linkToOrdInternal(ord);
            }
            
            navModel.cut(view);
            
            callbacks.ok();
          }
        });
      } else {
        navModel.cut(view);
      }
    });
  };
  
  /**
   * If the currently selected view is modified, this function delegates to 
   * <code>niagara.util.mobile.dialogs.confirmAbandonChanges</code>. It 
   * displays the name of the selected view in the message to confirm 
   * abandoning changes; it will save this view if the user clicks 'yes',
   * abandon changes and refresh the view if the user clicks 'no', and just
   * stays put if the user clicks 'cancel'.
   *  
   * @name niagara.util.mobile.PageViewManager#confirmAbandonChanges
   * @function
   * @param {String|jQuery|Function} [redirect] the redirect to be passed to
   * the dialog invocation - this will be the page/url we should navigate to if
   * 'yes' or 'no' is clicked (or a function that does the page navigation
   * for us).
   */
  PageViewManager.prototype.confirmAbandonChanges = function confirmAbandonChanges(redirect, callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this,
        selectedView = this.selectedView;
    
    if (selectedView.isModified()) {
      dialogs.confirmAbandonChanges({
        yes: function (cb) {
          var dialog = this;
          
          selectedView.save({
            ok: function () {
              selectedView.refresh(cb);
            },
            fail: function (err) {
              dialog.redirect(decodePageId(selectedView.pageId));
              cb.fail(err);
            }
          });
        },
        no: function (cb) {
          selectedView.setModified(false);
          that.refresh(selectedView, cb);
        },
        viewName: selectedView.getName(),
        redirect: redirect,
        callbacks: callbacks
      });
    }
  };
  
  /**
   * Saves the given view. This is different from just calling
   * <code>view.save()</code> only in that view will also be be refreshed and
   * resubscribed after the save is complete.
   * 
   * @name niagara.util.mobile.PageViewManager#save
   * @function
   * @param {niagara.util.mobile.PageView} view the PageView to save
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   */
  PageViewManager.prototype.save = function save(view, callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    
    view.save(callbacks);
  };
  
  PageViewManager.prototype.saveSelected = function saveSelected(callbacks) {
    this.save(this.selectedView, callbacks);
  };
  
  /**
   * Sets a given view as our currently selected view.
   * 
   * @name niagara.util.mobile.PageViewManager#setSelectedView
   * @function
   * @param {niagara.util.mobile.PageView} view the view to select 
   */
  PageViewManager.prototype.setSelectedView = function setSelectedView(view) {
    if (this.selectedView === view) {
      return;
    }

    var that = this,
        name = view.getName(),
        page;
    
    if (that.selectedView) {
      that.selectedView.deactivated();
    }
    
    view.activated();
    

    document.title = name;
    
    page = $('#' + view.pageId.replace(/(:|\.)/g, '\\$1'));
    page.children(':jqmData(role=header)').children('h1.viewName').text(name);

    that.selectedView = view;
    that.currentPage = page;
    that.navModel.setSelectedNode(view);
  };
  
  /**
   * Refreshes a given view by calling its <code>refresh</code> method. Re-arms
   * event handlers once the view's HTML has been completely rebuilt. Note
   * that calling <code>refresh</code> on a view will always perform a refresh,
   * potentially throwing out user-entered changes. <code>refreshSelected</code>
   * will confirm with the user if there are any outstanding changes to the
   * current view.
   * 
   * @name niagara.util.mobile.PageViewManager#refresh
   * @function
   * @param {niagara.util.mobile.PageView} view the view to refresh
   * @param {Object} [callbacks] an object holding callbacks
   * @param {Function} [callbacks.ok] a function to execute once the view is done
   * refreshing
   * @param {Function} [callbacks.fail] a function to execute if there is a
   * problem refreshing the view
   */
  PageViewManager.prototype.refresh = function refresh(view, callbacks) {
    var that = this;
    callbacks = callbackify(callbacks);
    
    view.refresh({
      ok: function () {
        callbacks.ok(view);
      },
      fail: callbacks.fail
    });
  };
  
  /**
   * Refreshes the currently selected view. If this view has any outstanding
   * user-entered changes, will pop up a dialog confirming that the user wishes
   * to abandon these changes before refreshing.
   * 
   * @name niagara.util.mobile.PageViewManager#refreshSelected
   * @function
   * @param {Object} [callbacks] an object holding callbacks
   * @param {Function} [callbacks.ok] a function to be called once the selected view
   * is done refreshing
   * @param {Function} [callbacks.fail] a function to be called if there is an error
   * during refresh
   */
  PageViewManager.prototype.refreshSelected = function refreshSelected(callbacks) {
    var that = this,
        selectedView = that.selectedView;
    
    if (selectedView.isModified()) {
      that.confirmAbandonChanges(undefined, callbacks);
    } else {
      that.refresh(selectedView, callbacks);
    }
  };
  
  /**
   * Performs a refresh on <i>all</i> views this view manager knows about. The
   * refreshes will occur in parallel and a callback will be fired once all
   * the views are done refreshing.
   * 
   * @name niagara.util.mobile.PageViewManager#refreshAll
   * @function
   * @param {Object} [callbacks] an object holding callbacks
   * @param {Function} [callbacks.ok] a function to be called once all views are
   * done refreshing
   * @param {Function} [callbacks.fail] a function to be called if there is an error
   * while refreshing any of the views
   */
  PageViewManager.prototype.refreshAll = function refreshAll(callbacks) {
    callbacks = callbackify(callbacks);
    
    var refreshes = [], that = this;
    
    util.flow.parallel(function (node) {
      var view = node.getValue();
      
      if (view instanceof PageView) {
        that.refresh(view, this);
      } else {
        this.ok();
      }
    }).invoke(that.navModel.getNodes(), callbacks);
  };
  
  /**
   * @memberOf niagara.util.mobile
   * @private
   */
  function shouldScrollReverse(pageViewManager, ord) {
    var that = pageViewManager,
        selectedView = that.selectedView,
        ords, selectedIndex, targetIndex;
    
    if (selectedView) {
      ords = that.navModel.getOrds();
      selectedIndex = util.indexOf(ords,
          util.ord.deriveOrd(selectedView.value));
      targetIndex = util.indexOf(ords, ord);
      
      if (targetIndex >= 0 && selectedIndex >= 0 && targetIndex < selectedIndex) {
        return true;
      }
    }
    
    return false;
  }
  
  /**
   * Delegates to <code>niagara.util.mobile.linkToOrd</code>.
   * 
   * <p>If the current view has outstanding changes, will confirm with the user
   * whether to abandon those changes before linking to the new view.
   * 
   * @name niagara.util.mobile.PageViewManager#linkToOrd
   * @function
   * @see niagara.util.mobile.linkToOrd
   * 
   * @param {baja.Ord|String} ord the Ord to navigate to
   * @param {baja.ViewQuery} viewQuery any view parameters to append to the ORD in the
   * case of an internal link
   */
  PageViewManager.prototype.linkToOrd = function pageViewManagerLinkToOrd(ord, viewQuery) {
    var that = this,
        selectedView = that.selectedView;
    
    function doLink() {
      util.mobile.linkToOrd(ord, { 
        viewQuery: viewQuery,
        reverse: shouldScrollReverse(that, ord)
      });
    }
    
    if (selectedView.isModified()) {
      that.confirmAbandonChanges(doLink);
    } else {
      doLink();
    }
  };
  
  /**
   * Delegates to <code>niagara.util.mobile.linkToOrdInternal</code>, but
   * performs some extra logic to cause JQM to scroll left or right depending
   * on whether we're moving up or down in the component tree.
   * 
   * @name niagara.util.mobile.PageViewManager#linkToOrdInternal
   * @function
   * @private
   * @see niagara.util.mobile.linkToOrdInternal
   * 
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition -
   * will be passed directly into <code>$.mobile.changePage()</code>
   * @param {Object} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   */
  PageViewManager.prototype.linkToOrdInternal = function pageViewManagerLinkToOrdInternal(ord, options) {
    options = baja.objectify(options);
    options.reverse = shouldScrollReverse(this, ord);
    util.mobile.linkToOrdInternal(ord, options);
  };
  
  /**
   * Delegates to <code>niagara.util.mobile.linkToOrdExternal</code>.
   * 
   * @name niagara.util.mobile.PageViewManager#linkToOrdExternal
   * @function
   * @private
   * @see niagara.util.mobile.linkToOrdExternal
   * 
   * @param {String|baja.Ord} ord the ORD to link to
   * @param {Object} [options] additional options used in the page transition
   * @param {Object} [options.viewQuery] can specify any view parameters to be
   * encoded into the ORD
   */
  PageViewManager.prototype.linkToOrdExternal = function pageViewManagerLinkToOrdExternal(ord, options) {
    util.mobile.linkToOrdExternal(ord, options);
  };
  
  /**
   * @namespace
   * @name niagara.util.mobile
   */
  util.api('niagara.util.mobile', {
    'public': {
      ListView: ListView,
      ListPageView: ListPageView,
      MobileView: MobileView,
      PageView: PageView,
      RadioButtonView: RadioButtonView,
      PageViewManager: PageViewManager
    },
    'private': {
      shouldScrollReverse: shouldScrollReverse
    }
  });
}());