/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Flow control functions for Niagara web apps.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, plusplus: true, browser: true */
/*global niagara, $, baja */


(function () {
  "use strict";
  
  var util = niagara.util,
      callbackify = util.callbackify;
  
  /**
   * Waits for a condition to become true, and then executes a function.
   * 
   * @memberOf niagara.util.flow
   * 
   * @param {Function} testFunc a function to test for truthiness. This function
   * will not be passed any arguments.
   * @param {Object} [obj] an object with ok/fail callbacks
   * @param {Function} [obj.ok] a function to execute once <code>testFunc</code>
   * returns true. This function will not be passed any arguments, unless the
   * timeout elapsed before the test function ever returned true, in which case
   * an error message will be passed as the first argument.
   * @param {Number} timeout number of milliseconds to wait (if omitted, 
   * 10 seconds)
   */
  function waitFor(testFunc, obj) {
    obj = callbackify(obj);
    
    var start = baja.clock.ticks(),
        timeout = obj.timeout || 10000,
        testSucceeded = false;
    function doWait() {
      try {
        testSucceeded = testFunc();
      } catch (e) {
        return obj.fail(e);
      }
      
      if (!testSucceeded) {
        if (baja.clock.ticks() - start < timeout) {
          setTimeout(doWait, 25);
        } else {
          obj.fail("Timeout exceeded");
        }
      } else {
        obj.ok();
      }
    }
    
    doWait();
  }
  
  /**
   * Runs a series of functions in sequential order, waiting for the callback
   * to each one to execute before moving on to the next.
   * <p>
   * The input functions and callback will not run in any particular
   * context, so if <code>this</code> needs to be meaningful then the caller
   * must perform its own binding.
   * 
   * @memberOf niagara.util.flow
   * 
   * @param {Array} funcs an array of functions to execute. These functions
   * must take a single argument of a callback function. It is up to the
   * individual function to call this callback's <code>ok</code> or
   * <code>fail</code> function to signal that it has completed execution. It
   * may also call the callback's <code>exit</code> function to jump out of the
   * workflow early - any arguments to <code>callbacks.exit</code> will be
   * passed to the workflow's <code>ok</code> callback, and no further steps
   * in the workflow will be executed.
   * @param {Object} [obj] an object literal containing additional parameters
   * @param {Function} [obj.ok] a callback function to be executed once
   * all of the input functions have successfully completed execution. 
   * @param {Function} [obj.fail] a callback function to be executed if any
   * of the functions running in parallel throws an error (or calls its own
   * <code>fail</code> callback).
   * @param {Number} [obj.timeout] how long, in milliseconds, to wait for the
   * input functions to finish executing. Note that if the timeout expires,
   * the functions may or may not finish execution anyway. If omitted, will
   * wait forever.
   */
  function runSequential(funcs, obj) {
    baja.strictArg(funcs, Array);
    obj = callbackify(obj);
    
    if (funcs.length === 0) {
      return obj.ok();
    }
    
    var done,
        errorCondition,
        timeout = obj.timeout,
        stepNumber = 1;
    
    function runHead(funcArray, callbacks) {
      if (!funcArray || !funcArray.length) {
        return callbacks.ok();
      }
      
      var headFunc = funcArray[0],
          theRest = funcArray.slice(1);

      try {
        headFunc({
          ok: function () {
            //run the next function on ok() from the current one
            stepNumber++;
            runHead(theRest, callbacks);
          },
          fail: function (err) {
            //bail on the rest of the steps
            callbacks.fail(err);
          },
          exit: callbacks.ok
        });
      } catch (err) {
        callbacks.fail(err);
      }
    }

    runHead(funcs, {
      ok: function () {
        //all of our sequential functions completed
        
        if (timeout) {
          //if we specified a timeout, then we're just waiting for done = true
          done = true; 
        } else if (!errorCondition) {
          //otherwise, ok will get called whenever it's called - we don't care.
          //we DON'T want to call obj.ok if we have an error condition (caused
          //by waitFor() timing out, below)
          obj.ok();
        }
      },
      fail: function (err) {
        //one of our sequential functions failed - set the error condition to
        //prevent obj.fail being called a second time when waitFor() inevitably
        //times out
        errorCondition = err;
        obj.fail(err);
      }
    });
    
    if (timeout) {
      waitFor(function () {
        return done;
      }, {
        ok: obj.ok,
        fail: function (err) {
          if (!errorCondition) {
            //if no error, then our sequential functions are still chugging 
            //away - we just timed out. set errorCondition to the timeout error 
            //to prevent obj.ok being called in addition to obj.fail in case
            //they ever do complete
            errorCondition = err;
            obj.fail(err);
          } else {
            //if we do have an error it was caused by a failure of one of our
            //sequential functions - obj.fail was already called above, let's
            //not call it twice
            baja.outln('util.flow.waitFor timed out naturally after ' +
                'exception on step ' + stepNumber);
          }
        },
        timeout: timeout
      });
    }
  }

  /**
   * Runs a series of functions in parallel. There is no guarantee of what
   * order the functions will complete in.
   * <p>
   * Each function must take in a single parameter: an object literal with
   * <code>ok</code> and <code>fail</code> callbacks.
   * <p>
   * The input functions and callback will not run in any particular
   * context, so if <code>this</code> needs to be meaningful then the caller
   * must perform its own binding.
   * <p>
   * The <code>ok</code> callback will receive a single parameter of Array type.
   * This will contain any arguments the input functions pass to their own
   * callback handlers. This allows you to run a set of async functions in 
   * parallel but still handle their returned results in the same order. As an
   * example:
   * 
   * <pre>
   * var invocations = [];
   *  
   *  //run 5 functions in parallel
   *  baja.iterate(0, 5, function (i) {
   *    invocations.push(function (callbacks) {
   *      setTimeout(function () {
   *          callbacks.ok(i);
   *      }, Math.random() * 1000); //wait a random amount of time
   *    });
   *  });
   *  
   *  niagara.util.flow.runParallel(invocations, {
   *    ok: function (results) {
   *      //this is guaranteed to be [0, 1, 2, 3, 4] because that is the
   *      //order of the input functions
   *      console.log(results); 
   *    }
   *  });
   * </pre>
   * 
   * @memberOf niagara.util.flow
   * 
   * @param {Array} funcs an array of functions to execute. These functions
   * must take a single argument of a callback function. It is up to the
   * individual function to call this callback to signal that it has
   * completed execution.
   * @param {Object} [callbacks] an object containing ok/fail callbacks (and
   * an optional timeout parameter)
   * @param {Function} [callbacks.ok] a callback function to be executed once
   * all of the input functions have successfully completed execution. The 
   * single parameter to this function will be an array containing any values
   * passed to the individual functions' own <code>ok</code> handlers.
   * @param {Function} [callbacks.fail] a callback function to be executed if any
   * of the functions running in parallel throws an error (or calls its own
   * <code>fail</code> callback).
   * @param {Number} [callbacks.timeout] how long, in milliseconds, to wait for 
   * the input functions to finish executing. Note that if the timeout expires,
   * the functions may or may not finish execution anyway. If omitted, will
   * wait forever.
   */
  function runParallel(funcs, callbacks) {
    baja.strictArg(funcs, Array);
    callbacks = callbackify(callbacks);

    if (funcs.length === 0) {
      return callbacks.ok();
    }
    
    var results = [],
        NOT_DONE = {},
        timeout = callbacks.timeout,
        complete = false,
        error;
    
    function allDone() {
      if (complete || error) {
        return true;
      }
      
      complete = !baja.iterate(results, function (result) {
        if (result === NOT_DONE) {
          return true; //stop iterating, at least one dude is still waiting
        }
      });
      
      return complete;
    }
    
    function doComplete() {
      if (error) { //one of our functions threw an error
        callbacks.fail(error);
      } else {
        callbacks.ok(results);
      }
    }
    
    function tryComplete() {
      //if we specified a timeout, don't even bother - the waitFor call
      //at the bottom of this function will be continually checking. otherwise,
      //check after each individual step completes.
      if (!timeout && allDone()) {
        doComplete();
      }
    }
    
    //preload results array
    baja.iterate(funcs.length, function (i) {
      results[i] = NOT_DONE;
    });
    
    baja.iterate(funcs, function (func, i) {
      results[i] = NOT_DONE;
      try {
        func({
          ok: function (result) {
            results[i] = result;
            tryComplete();
          },
          fail: function (err) {
            results[i] = error = err;
            tryComplete();
          }
        });
      } catch (err) {
        results[i] = error = err;
        tryComplete();
      }
    });
    
    if (timeout) {
      waitFor(allDone, {
        timeout: timeout,
        ok: doComplete,
        fail: callbacks.fail
      });
    }
  }
  
  function iterateToParallel(obj, func, params) {
    baja.strictAllArgs([obj, func], [Object, Function]);
    
    params = baja.objectify(params, 'ok');
    params.fail = params.fail || util.noop;
    params.timeout = params.timeout || 10000;
    
    var subCallbacks = [];
    
    function queueOperation(obj, func) {
      subCallbacks.push(function (callbacks) {
        func(obj, callbacks);
      });
    }
    
    baja.iterate(obj, function (prop, propName) {
      queueOperation(prop, func);
    });
    
    runParallel(subCallbacks, params);
  }

  
  
  /**
   * Represents one step in an asynchronous workflow.
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   * @param {Function} func the function to be run in this step
   */
  function Step(func, name) {
    this.func = func;
    this.name = name || func.toString().split(/[\s\(]/g)[1];
  }
  
  /**
   * Runs the function associated with this step.
   * 
   * @name niagara.util.flow.Step#doCall
   * @function
   * @private
   * @param {Object} the context currently running in this step
   * @param {Array} [okArgs] any arguments passed to the <code>ok()</code>
   * callback of the previous step, if there was one
   * @param {Object} callbacks the scope that will be bound to this step's
   * function (the <code>this</code> that will be passed in place of a
   * callback object)
   */
  Step.prototype.doCall = function doCall(cx, okArgs, callbacks) {
    var args = [cx],
        that = this,
        execContext;
    
    execContext = {
      ok: callbacks.ok,
      exit: callbacks.exit,
      fail: function (err) {
        if (that.name) {
          baja.error("Step failed: " + that.name);
        }
        callbacks.fail(err);
      },
      batch: callbacks.batch
    };
    
    if (okArgs) {
      args = args.concat(okArgs);
    }
    
    try {
      this.func.apply(execContext, args);
    } catch (e) {
      execContext.fail(e);
    }
  };
  
  /**
   * Runs a series of steps sequentially. The ok callback of each step must be
   * called before the next step can begin.
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   * @param {Object} context a context object to be passed as the first
   * parameter to each step function
   * @param {niagara.util.flow.SequentialWorkflow} workflow
   */
  function SequentialInvocation(context, workflow) {
    this.context = context;
    this.workflow = workflow;
  }
  
  /**
   * Performs a series of operations in sequential order. The ok callback of
   * each step must be called before the next step can begin. Each step will
   * have its own batch object it can use - batches are created at the beginning
   * of each step, and the invocation will commit the batch itself. They are
   * available as <code>this.batch</code>. If you have
   * your own batch you wish to use instead, set a <code>batch</code> property
   * on your context object instead, and that batch will be used (you will need
   * to commit this batch yourself).
   * 
   * @name niagara.util.flow.SequentialInvocation#invoke
   * @function
   * @param {Object} invocationCallbacks an object with ok/fail callbacks and
   * a timeout (in milliseconds) (if omitted, 10000)
   */
  SequentialInvocation.prototype.invoke = function invoke(invocationCallbacks) {
    invocationCallbacks = callbackify(invocationCallbacks);
    
    var cx = this.context,
        steps = this.workflow.steps,
        stepCalls = [],
        okArgs = [],
        timeout = invocationCallbacks.timeout || 10000;
    
    baja.iterate(steps, function (step) {
      
      stepCalls.push(function (stepCallbacks) {
        
        var stepBatch = new baja.comm.Batch();
        
        step.doCall(cx, okArgs, {
          ok: function () {
            //pass any arguments to the ok handler as arguments to the
            //next step, after the context argument
            okArgs = Array.prototype.slice.call(arguments);
            stepCallbacks.ok();
          },
          exit: function () {
            okArgs = Array.prototype.slice.call(arguments);
            stepCallbacks.exit();
          },
          fail: function (err) {
            stepCallbacks.fail(err);
          },
          batch: cx.batch || stepBatch
        });
        
        stepBatch.commit();
      });
    });
    
    runSequential(stepCalls, {
      ok: function () {
        (invocationCallbacks.ok).apply(this, okArgs);
      },
      fail: invocationCallbacks.fail,
      timeout: timeout
    });
  };
  
  /**
   * Runs steps for multiple contexts, all in parallel. The callback
   * will be fired after the execution is complete for all contexts.
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   * @param {Array} contexts an array of contexts - each will be passed as the 
   * first parameter to the step function
   * @param {niagara.util.flow.SequentialWorkflow} workflow
   */
  function ParallelInvocation(contexts, workflow) {
    this.contexts = contexts;
    this.workflow = workflow;
  }
  
  /**
   * Performs steps for multiple contexts, all in parallel. The ok callback
   * will be fired only after all executions have completed, calling their own
   * callbacks. A batch will be created and available to the step functions as
   * <code>this.batch</code>. This batch will be shared across all invocations
   * of all steps. If you have your own batch you wish to use
   * instead, set a <code>batch</code> property on your execution contexts.
   * 
   * @name niagara.util.flow.ParallelInvocation#invoke
   * @function
   * @param {Object} invocationCallbacks an object with ok/fail callbacks and
   * a timeout (in milliseconds) (if omitted, 10000)
   */
  ParallelInvocation.prototype.invoke = function invoke(invocationCallbacks) {
    invocationCallbacks = callbackify(invocationCallbacks);
    
    var contexts = this.contexts,
        steps = this.workflow.steps,
        contextCalls = [],
        batch = new baja.comm.Batch();
    
        
    //queue up one execution of the steps for each of our contexts - 
    //these executions will run in parallel
    baja.iterate(contexts, function (cx) {
      baja.iterate(steps, function (step) {
        contextCalls.push(function (contextCallbacks) {
          contextCallbacks.batch = cx.batch || batch;
          step.doCall(cx, undefined, contextCallbacks);
        });
      });
    });

    invocationCallbacks.timeout = invocationCallbacks.timeout || 10000;

    //run all executions of the current step in parallel
    runParallel(contextCalls, invocationCallbacks);
    batch.commit();
  };
  
  /**
   * A workflow in which steps are executed in parallel. A callback is fired
   * when all steps are complete.
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   */
  function ParallelWorkflow(funcs) {
    var steps = [];
    
    baja.iterate(funcs, function (func) {
      steps.push(new Step(func));
    });
    
    this.steps = steps;
  }
  
  /**
   * @name niagara.util.flow.ParallelWorkflow#createInvocation
   * @function
   * @param {Array|Object} contexts an array of context objects (or a single
   * object that will be converted to an array of length 1)
   * @returns {niagara.util.flow.ParallelInvocation}
   */
  ParallelWorkflow.prototype.createInvocation = function createInvocation(contexts) {
    if (!$.isArray(contexts)) {
      contexts = [contexts];
    }
    
    return new ParallelInvocation(contexts, this);
  };
  
  ParallelWorkflow.prototype.invoke = function invoke(contexts, callbacks) {
    this.createInvocation(contexts).invoke(callbacks);
  };
  
  /**
   * A workflow in which steps are executed in sequence. The ok callback of 
   * one step must be called before the next step will be executed.
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   */
  function SequentialWorkflow(funcs) {
    var steps = [];
    
    baja.iterate(funcs, function (func) {
      steps.push(new Step(func));
    });
    
    this.steps = steps;
  }
  
  /**
   * @name niagara.util.flow.ParallelWorkflow#createInvocation
   * @function
   * @param {Array} contexts an array of context objects
   * @returns {niagara.util.flow.SequentialInvocation}
   */
  SequentialWorkflow.prototype.createInvocation = function createInvocation(context) {
    return new SequentialInvocation(context, this);
  };
  
  SequentialWorkflow.prototype.invoke = function invoke(context, callbacks) {
    this.createInvocation(context).invoke(callbacks);
  };
  
  /**
   * Runs a series of steps in sequential-parallel order. This means that all
   * the step 1s can run in parallel, but all of them must return before the
   * step 2s can start running, and so on.
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   * @param {Array} contexts an array of context objects - each of these will be
   * passed to each step in the workflow
   * @param {niagara.util.flow.SequentialWorkflow} workflow the workflow being
   * invoked
   */
  function SequentialParallelInvocation(contexts, workflow) {
    this.contexts = contexts;
    this.workflow = workflow;
  }
  
  /**
   * Performs the work on this invocation's workflow, using the context objects
   * it was created with.
   * 
   * @name niagara.util.flow.SequentialParallelInvocation#invoke
   * @function
   * @param {Object} invocationCallbacks an object with ok/fail callbacks. The
   * ok callback will only be executed once all the steps have completed for
   * all the context objects. If any step fails for any context, the fail
   * callback will be executed instead.
   */
  SequentialParallelInvocation.prototype.invoke = function invoke(invocationCallbacks) {
    invocationCallbacks = baja.objectify(invocationCallbacks, 'ok');
    
    var contexts = this.contexts,
        steps = this.workflow.steps,
        sequentialFuncs = [],
        stepCount = steps.length,
        stepCalls = [],
        contextOkArgs = [],
        timeout = invocationCallbacks.timeout || 10000;
    
    baja.iterate(contexts, function (cx, i) {
      cx.$contextIndex = i;
    });
    
    baja.iterate(steps, function (step) {
      sequentialFuncs.push(function (cx) {
        var that = this,
            parallelFunc;
        
        parallelFunc = function (cx) {
          var contextIndex = cx.$contextIndex,
              that = this;
          
          step.doCall(cx, contextOkArgs[contextIndex], {
            ok: function () {
              //pass any arguments to the ok handler as arguments to the
              //next step, after the context argument
              var okArgs = Array.prototype.slice.call(arguments);
              contextOkArgs[contextIndex] = okArgs;
              that.ok();
            },
            fail: that.fail,
            batch: that.batch
          });
        };
        
        new ParallelWorkflow([parallelFunc])
          .createInvocation(cx.contexts) //cx.contexts is passed into the sequential invocation
          .invoke(that);
      });
    });
    
    //now run all the parallel step operations in sequence
    new SequentialWorkflow(sequentialFuncs)
      .createInvocation({contexts: contexts})
      .invoke({
        ok: function () { invocationCallbacks.ok(contextOkArgs); },
        fail: invocationCallbacks.fail
      });
  };
  
  /**
   * A workflow in which each step may have multiple executions going at one
   * time in a number of contexts, but all executions of an individual step
   * must complete before moving to the next. The parallel executions of each
   * step will be delegated to <code>niagara.util.flow.ParallelWorkflow</code>
   * and the flow from one step to the next will be delegated to
   * <code>niagara.util.flow.SequentialWorkflow</code>.
   * 
   * <p>
   * Each step function receives a context object as the first parameter - these
   * context objects are provided to the <code>createInvocation</code> function.
   * Each function is responsible for triggering its own callback by passing 
   * <code>this</code> in where an object for ok/fail callbacks would be
   * expected. It may also call <code>this.ok()</code> or 
   * <code>this.fail(err)</code> directly if you choose.
   * <p>
   * <code>this</code> will also have a <code>batch</code> property. A new
   * <code>baja.comm.Batch</code> will be created at the start of each step,
   * and committed at the end of it.
   * <p>
   * An example workflow:
   * 
   * <pre>
   * var workflow = niagara.util.flow.toSequentialParallelWorkflow([
   *   function (cx) {
   *     console.log("step 1: " + cx.name);
   *     this.ok("i am an argument to step 2: " + cx.name);
   *   },
   *   function (cx, okArgFromPrevStep) {
   *     console.log("step 2: " + cx.name);
   *     console.log("step 1 passed this to ok(): " + okArgFromPrevStep);
   *     this.ok();
   *   }
   * ]);
   * 
   * var invocation = workflow.createInvocation([
   *   {name: 'context1'},
   *   {name: 'context2'}
   * ]);
   * 
   * invocation.invoke({
   *   ok: function () {
   *     console.log('ok');
   *   },
   *   fail: function (err) {
   *     console.log(err);
   *   }
   * });
   * </pre>
   * 
   * @class
   * @memberOf niagara.util.flow
   * @private
   * @param {Array} funcs an array of functions
   */
  function SequentialParallelWorkflow(funcs) {
    var steps = [];
    
    baja.iterate(funcs, function (func) {
      steps.push(new Step(func));
    });
    
    this.steps = steps;
  }
  
  /**
   * @name niagara.util.flow.SequentialParallelWorkflow#createInvocation
   * @function
   * @param {Array} contexts an array of context objects
   * @returns {niagara.util.flow.SequentialParallelInvocation}
   */
  SequentialParallelWorkflow.prototype.createInvocation = function createInvocation(contexts) {
    return new SequentialParallelInvocation(contexts, this);
  };
  
  SequentialParallelWorkflow.prototype.invoke = function invoke(contexts, callbacks) {
    this.createInvocation(contexts).invoke(callbacks);
  };
  
  
  /**
   * Creates a sequential-parallel workflow from the given functions.
   * 
   * @memberOf niagara.util.flow
   * @param funcs a variable number of functions passed in directly
   * as arguments
   */
  function sequentialParallel() {
    var funcs = Array.prototype.slice.call(arguments);
    return new SequentialParallelWorkflow(funcs);
  }
  
  /**
   * Creates a sequential workflow from the given functions.
   * 
   * @memberOf niagara.util.flow
   * @param funcs a variable number of functions passed in directly
   * as arguments
   */
  function sequential() {
    var funcs = Array.prototype.slice.call(arguments);
    return new SequentialWorkflow(funcs);
  }
  
  /**
   * Creates a parallel workflow from the given functions.
   * 
   * @memberOf niagara.util.flow
   * @param funcs a variable number of functions passed in directly
   * as arguments
   */
  function parallel() {
    var funcs = Array.prototype.slice.call(arguments);
    return new ParallelWorkflow(funcs);
  }
  
  
  
  /**
   * @namespace
   * @name niagara.util.flow
   */
  util.api('niagara.util.flow', {
    iterateToParallel: iterateToParallel,
    runParallel: runParallel,
    runSequential: runSequential,
    parallel: parallel,
    sequentialParallel: sequentialParallel,
    sequential: sequential,
    waitFor: waitFor
  });
}());
