/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview The base View class and mixins, used for backing displayed
 * DOM elements with Baja values.
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true */

/*globals $, baja, niagara */

(function () {
  "use strict";
  
  niagara.util.require(
    'niagara.util.flow'
  );
  
  var util = niagara.util,
      callbackify = util.callbackify,
      View;
  
  /**
   * Get the display name for a component - try <code>getDisplayName</code>
   * first, then <code>getName</code>, then undefined if neither exists.
   * 
   * @memberOf niagara.util
   * @private
   * 
   * @param {baja.Complex} component
   * @returns {String}
   */
  function getComponentName(component) {
    if (!(component instanceof baja.Complex)) {
      return undefined;
    }
    
    return component.getDisplayName() || 
           component.getName();
  }
  
  


  ////////////////////////////////////////////////////////////////
  // View
  ////////////////////////////////////////////////////////////////

  /**
   * A View contains the most basic mechanisms necessary for displaying a
   * BValue inside of a DOM element. It should not be instantiated directly,
   * but rather subclassed with the specific functionality you need. 
   * <p>
   * At minimum, the <code>doLoadValue</code> function will need to be 
   * implemented by your subclass.
   * 
   * @class
   * @memberOf niagara.util
   */
  View = function View(options) {
    
    var that = this;
    
    /**
     * whether any field editors in this view have been changed
     * @type {Boolean}
     */
    that.$modified = false;
    
    that.$mixins = {};
  };
  
  View.prototype.hasMixin = function (name) {
    return !!this.$mixins[name];
  };
  
  /**
   * Indicates that a view has been loaded and/or scrolled into view. For
   * example, this will typically be called by a PageViewManager when 
   * scrolling a JQM page into view. By default, does nothing.
   * 
   * 
   * @name niagara.util.View#activated
   * @function
   * @see niagara.util.View#deactivated
   */
  View.prototype.activated = function () {
  };
  
  /**
   * Indicates that a view has been unloaded and/or scrolled out of view. For
   * example, this will typically be called by a PageViewManager when the
   * view is being scrolled out of view. By default, does nothing.
   * 
   * @name niagara.util.View#deactivated
   * @function
   * @see niagara.util.View#activated
   */
  View.prototype.deactivated = function () {
  };
  
  View.$initializeDOMWorkflow = util.flow.sequential(
    function initializeDOM__step1__doInitializeDOM(cx) {
//      cx.tmpElement = $('<div/>');
//      cx.targetElement.after(cx.tmpElement);
//      cx.targetElement.detach();
      cx.view.doInitializeDOM(cx.targetElement, this);
    },
    function initializeDOM__step2__postInitializeDOM(cx) {
      cx.view.$dom = cx.targetElement;
      cx.targetElement.data('view', cx.view);
      cx.view.postInitializeDOM(this);
    },
    function initializeDOM__step3__armHandlersAndTriggerEvent(cx) {
//      cx.tmpElement.replaceWith(cx.targetElement);
      cx.view.armHandlers();
      cx.view.$dom.trigger('viewinitializedom', [ cx.view ]);
      this.ok(cx.view);
    }
  );
  
  /**
   * Initializes the DOM element to be bound to this view. Most commonly, this
   * will involve building up the HTML structure necessary to load in a value.
   * If this View will display a baja:String, for example, initializeDOM might
   * append a text input element to the target field. A baja:DynamicEnum might
   * include a select dropdown. 
   * 
   * <p>In some cases, <i>no</i> initialization may be required at all. This 
   * might be the case if you are binding the View to an HTML element that is
   * already pre-populated with all the necessary structure to load a value,
   * or maybe <code>doLoadValue</code> will empty out the element completely
   * and rebuild it from scratch every time a new value is loaded.
   * 
   * <p>In a nutshell, <code>initializeDOM</code> defines the following 
   * contract:
   * 
   * <ul>
   * <li>After <code>initializeDOM</code> completes and calls 
   * <code>callbacks.ok</code>, the target element will be fully initialized,
   * structured, and ready to load in a value. It will be accessible as an
   * instance variable as <code>this.$dom</code>.</li>
   * 
   * <li><code>loadValue</code> may not be called until 
   * <code>initializeDOM</code> completes. Attempting to load a value prior to
   * initialization will result in failure.</li>
   * </ul>
   * 
   * <p><code>initializeDOM</code> delegates the actual work of building the
   * HTML structure (if any) to the <code>doIntializeDOM</code> function. When
   * subclassing View, it is best not to override <code>initializeDOM</code>,
   * but to override <code>doInitializeDOM</code> instead.
   * 
   * @name niagara.util.mobile.View#initializeDOM
   * @function
   * @param {jQuery} targetElement the div in which this view should build its
   * HTML (will be passed directly to <code>doInitializeDOM</code>)
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   * @param {Function} [callbacks.ok] a function to be called when the HTML is
   * done building - <code>this</code> will be passed as the first argument
   * @param {Function} [callbacks.fail] a function to be called if there is
   * a problem building the HTML
   */
  View.prototype.initializeDOM = function initializeDOM(targetElement, callbacks) {
    callbacks = callbackify(callbacks);

    View.$initializeDOMWorkflow.invoke({
      view: this,
      targetElement: targetElement
    }, callbacks);
  };
  
  /**
   * Runs upon successful completion of <code>doInitializeDOM</code>.
   * <p>
   * This is an override hook and by default does nothing.
   * 
   * @name niagara.util.View#postInitializeDOM
   * @function
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  View.prototype.postInitializeDOM = function postInitializeDOM(callbacks) {
    callbacks.ok();
  };
  
  /**
   * Performs the actual work of initializing the DOM element in which this
   * view will live. This function should be overridden by subclasses - the 
   * subclass function should append elements to <code>targetElement</code> as 
   * necessary and then call <code>callbacks.ok</code> or 
   * <code>callbacks.fail</code> as appropriate.
   * 
   * <p>By default, <code>callbacks.ok</code> will be called and no other action
   * will be taken. 
   * 
   * @name niagara.util.mobile.View#doInitializeDOM
   * @function
   * @see niagara.util.mobile.View#initializeDOM
   * @param {jQuery} targetElement the div in which this view should build its HTML
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   */
  View.prototype.doInitializeDOM = function doInitializeDOM(targetElement, callbacks) {
    callbackify(callbacks).ok();
  };
  
  /**
   * Returns the display name of this view.
   * 
   * @name niagara.util.mobile.View#getName
   * @function
   * 
   * @returns {String} the view's display name, equivalent to the
   * display name of this view's component value. If the view has no loaded
   * value, or the loaded value is not a component, returns 
   * <code>undefined</code>.
   */
  View.prototype.getName = function getName() {
    return getComponentName(this.value);
  };
  
  /**
   * Returns this view's modified status. 
   * 
   * @name niagara.util.mobile.View#isModified
   * @function
   */
  View.prototype.isModified = function isModified() {
    return this.$modified;
  };
  
  /**
   * Sets this view's modified status. If <code>modified</code> is
   * <code>true</code>, a <code>viewmodified</code> event will be triggered 
   * on the view's div.
   * <p>
   * The default implementation of <code>View</code> never calls this method - 
   * subclasses of <code>View</code> should call <code>setModified</code> as 
   * appropriate.
   * 
   * @name niagara.util.mobile.View#setModified
   * @function
   * @param {Boolean} modified
   */
  View.prototype.setModified = function setModified(modified) {
    baja.strictArg(modified, Boolean);
    this.$modified = modified;
    if (modified) {
      this.$dom.trigger('viewmodified', [ this ]);
    }
  };
  
  View.$loadValueWorkflow = util.flow.sequential(
    function loadValue__step1__doLoadValue(cx) {
//      cx.tmpElement = $('<div/>');
//      cx.view.$dom.after(cx.tmpElement);
//      cx.view.$dom.detach();
      cx.view.doLoadValue(cx.value, this);
    },
    function loadValue__step2__postLoadValue(cx) {
      cx.view.value = cx.value;
      cx.view.postLoadValue(this);
    },
    function loadValue__step3__triggerAndRefresh(cx) {
//      cx.tmpElement.replaceWith(cx.view.$dom);
      cx.view.$dom.trigger('viewloadvalue', [ cx.view ]);
      cx.view.refreshWidgets();
      this.ok(cx.view);
    }
  );
  
  /**
   * Updates the view's HTML with the given value. Delegates the work of loading
   * the HTML values to override point <code>doLoadValue</code> - subclasses
   * will typically not override <code>loadValue</code>. After the values are
   * loaded, any enhancement necessary will take place by calling 
   * <code>this.refreshWidgets</code>. The value will hence be accessible at
   * <code>this.value</code>, and a <code>viewloadvalue</code> event will be
   * triggered on this view's div.
   * 
   * @name niagara.util.mobile.View#loadValue
   * @function
   * @param {baja.Value} value the value to be loaded
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   */
  View.prototype.loadValue = function loadValue(value, callbacks) {
    callbacks = callbackify(callbacks);

    if (!this.$dom) {
      return callbacks.fail(
          'this.$dom does not exist - call initializeDOM first');
    }
    
    
    View.$loadValueWorkflow.invoke({
      view: this,
      value: value
    }, callbacks);
  };
  
  /**
   * Runs upon successful completion of <code>doLoadValue</code>. The loaded
   * value will be accessible as <code>this.value</code>.
   * <p>
   * This is an override hook and by default does nothing.
   * 
   * @name niagara.util.View#postLoadValue
   * @function
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  View.prototype.postLoadValue = function postLoadValue(callbacks) {
    callbacks.ok();
  };
  
  /**
   * Performs the actual work of populating the view's HTML to reflect the
   * input Baja value. This function should be overridden by subclasses - 
   * the subclass function should manipulate <code>this.$dom</code> and then
   * call <code>callbacks.ok</code> or <code>callbacks.fail</code> as
   * appropriate.
   * <p>
   * If not implemented by a subclass, an error will be thrown.
   * 
   * @name niagara.util.mobile.View#doLoadValue
   * @function
   * @param {baja.Value} value the value to be loaded
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   * @throws {Error} if not implemented by a subclass
   */
  View.prototype.doLoadValue = function doLoadValue(value, callbacks) {
    throw new Error("doLoadValue not implemented");
  };
  
  /**
   * Arms any jQuery event listeners needed on the view's DOM elements. Will
   * be called after <code>doInitializeDOM</code> completes. By default, does
   * nothing.
   * 
   * @name niagara.util.mobile.View#armHandlers
   * @function
   */
  View.prototype.armHandlers = function armHandlers() {
    
  };
  
  /**
   * After building the main HTML for this view, will refresh any JQM widgets
   * to keep their displays up to date. May also be used to refresh (or create)
   * widgets that rely on the HTML being fully built, such as FLOT charts.
   * <p>
   * Default behavior is to do nothing.
   * 
   * @name niagara.util.mobile.View#refreshWidgets
   * @function
   */
  View.prototype.refreshWidgets = function refreshWidgets() {
    this.$dom.trigger('create');
  };
  
  /**
   * Saves any user-entered changes to the view. Views by default are not
   * editable so calling this function will throw an error - views intended
   * for user editing should provide their own implementation of
   * this function.
   * 
   * @name niagara.util.mobile.View#save
   * @function
   * @param {Object} [callbacks] an object with ok/fail callbacks
   * @throws {Error} if not overridden by a subclass
   */
  View.prototype.save = function save(callbacks) {
    throw new Error("save not implemented");
  };
  
  
  /**
   * Creates a subclass of a View, or a View subclass. e.g.:
   * 
   * <pre>
   *  var SubView = View.subclass();
   *  var SubSubView = SubView.subclass();
   *  var SubscribableView = View.subclass('subscribable');
   * </pre>
   * @name niagara.util.View.subclass
   * @function
   * 
   * @param {Function} func a constructor function. The code in this function
   * will run <i>after</i> the View superclass code runs, and it may do
   * whatever it wishes with <code>this</code> etc. This may be omitted if
   * you don't need any special behavior in the constructor of your View
   * subclass.
   * 
   * @param {Array|String} [mixins] any mixins you want applied to your View
   * subclass - you may either pass in an array or an arbitrary number of
   * extra arguments, e.g. <code>ListView.subclass(function(){}, 'mixin1', 
   * 'mixin2', 'mixin3')</code>
   */
  View.subclass = function subclass(func, mixins) {
    var subClass,
        args = Array.prototype.slice.call(arguments);
    
    if (typeof func !== 'function') {
      func = util.noop;
      mixins = func;
    } 
    
    
    subClass = util.aop.after(function (args) {
      func.apply(this, args);
    }, this).$extend(this);
  
    
    if (mixins) {
      if (!$.isArray(mixins)) {
        mixins = Array.prototype.slice.call(args, util.indexOf(args, mixins));
      }
      
      baja.iterate(mixins, function (mixin) {
        subClass = util.mixins.addMixin(subClass, mixin);
      });
    }
    
    subClass.subclass = View.subclass;
    
    return subClass;
  };
  
  
  (function mixinFunctions() {
    /**
     * @namespace
     * @name niagara.util.mobile.mixins.subscribable
     * @private
     */
    var subscribable = {
      $name: 'subscribable',
      
      /**
       * Sets <code>this.subscriber</code> to the output of 
       * <code>makeSubscriber()</code>.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribable
       */
      postConstructor: function postConstructor() {
        this.subscriber = this.makeSubscriber();
        util.aop.before(this, 'loadValue', function (args) {
          var value = args[0];
          if (value !== this.value && this.started) {
            throw new Error("Cannot load a new value into a subscribed View." +
                "Call stop() first.");
          }
        });
      },
      
      /**
       * Creates a subscriber to handle component events on this view's
       * component. Default behavior is to simply return a new 
       * <code>baja.Subscriber</code> - subclasses can augment or replace this
       * function to attach event listeners.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribableMixin
       * 
       * @returns {baja.Subscriber} a subscriber object that can be attached to
       * events on this view's component
       */
      makeSubscriber: function makeSubscriber() {
        return new baja.Subscriber();
      },
      
      /**
       * Re-retrieves a fresh copy of the view's value from the server,
       * reinitializes subscriptions on it, and rebuilds the view's HTML. If
       * the value is not a mounted Component, it will skip the 
       * retrieve/subscribe step and just rebuild the HTML.
       * <p>
       * Once the HTML is rebuilt, a <code>viewrefresh</code> event will be
       * triggered on the view's div.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribableMixin
       * 
       * @param {Object} [callbacks] an object with ok/fail callbacks to be 
       * executed once the view is fully reinitialized
       */
      refresh: function (callbacks) {
        callbacks = callbackify(callbacks);
        
        this.$refreshWorkflow.invoke(this, callbacks);
      },
      
      /**
       * Starts up subscriptions on the view's component.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribableMixin
       * 
       * @param {Function} [callback] a function to be executed once this view's
       * component and its child components have fully completed subscription
       */
      start: function (callbacks) {
        callbacks = callbackify(callbacks);
        
        var that = this,
            myValue = that.value;
        
        if (!myValue.getType().isComponent() || !myValue.isMounted()) {
          //can't subscribe a non-Component
          return callbacks.ok();
        }
        
        this.$startWorkflow.invoke(that, callbacks);
      },
      
      /**
       * Stops subscriptions on the view's component.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribableMixin
       * 
       * @param {Object} [callbacks] an object with ok/fail callbacks to be 
       * executed once this view's component and its child components have fully 
       * finished unsubscribing
       */
      stop: function (callbacks) {
        callbacks = callbackify(callbacks);
        
        var that = this,
            myValue = that.value, 
            batch;
        
        if (!myValue.getType().isComponent() || !myValue.isMounted()) {
          //can't subscribe a non-Component
          return callbacks.ok();
        }
        
        this.$stopWorkflow.invoke(that, callbacks);
      },
      
      /**
       * Subscribes to all the child components of this view's component. Will 
       * be called during the <code>start</code> function, using the same
       * <code>baja.comm.Batch</code> used to subscribe to the main component.
       * Since gaining subscription to the component's children is not always
       * desired behavior, this function serves only as an override point to be
       * overwritten by child classes.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribableMixin
       * 
       * @param {baja.comm.Batch} batch the batch used to subscribe to the main
       * component. This batch will be committed <i>for</i> you in the
       * <code>start()</code> method.
       */
      subscribeChildren: function (batch) {
        
      },
      
      /**
       * Unsubscribes from all this component's child components. The inverse of
       * <code>subscribeChildren</code>, this function also serves only as an
       * override point by default.
       * 
       * @memberOf niagara.util.mobile.mixins.subscribableMixin
       * 
       * @param {baja.comm.Batch} batch a batch to use to unsubscribe children 
       * (should be the same batch used to unsubscribe the parent component). 
       * This batch will be committed <i>for</i> you in the <code>stop()</code> 
       * method.
       */
      unsubscribeChildren: function (batch) {
        
      },
      
      $refreshWorkflow: util.flow.sequential(
        function refreshWorkflow__step1__loadSlots(view) {
          var value = view.value,
              type = value.getType();
          if (type.isComponent() && value.isMounted()) {
            value.loadSlots(this);
          } else {
            this.ok();
          }
        },
        function refreshWorkflow__step2__reloadValue(view) {
          view.loadValue(view.value, this);
        },
        function refreshWorkflow__step3__triggerViewrefreshEvent(view) {
          view.$dom.trigger('viewrefresh', [ view ]);
          view.setModified(false);
          this.ok();
        }
      ),
        
      $startWorkflow: util.flow.sequential(
        //first fully subscribe our component
        function step1__subscribeComponent(view) {
          var sub = view.subscriber;
          
          if (view.started) {
            throw new Error("Attempted to start a view that was already " +
              "started");
          }
          if (sub.$comps.length && baja.iterate(sub.$handlers, util.noop)) {
            throw new Error("Attempted to start a view but the view's " +
                "subscriber had live events attached - call stop() first");
          }
          view.subscriber.subscribe($.extend(this, { comps: view.value }));
        },
        //then batch together our child subscribes
        function step2__subscribeChildren(view) {
          var batch = this.batch;
          //we need to add our own callback because subscribeChildren won't
          //necessarily do it for us
          view.subscribeChildren(this.batch);
          batch.addCallback(this.ok, this.fail);
        },
        //wrap up - mark as started and trigger event
        function step3__triggerViewStartEvent(view) {
          view.started = true;
          view.$dom.trigger('viewstart', [ view ]);
          this.ok();
        }
      ),
      
      $stopWorkflow: util.flow.sequential(
        function step1__unsubscribe(view) {
          this.batch.addCallback(this.ok, this.fail);
          view.subscriber.unsubscribeAll({batch: this.batch});
          view.unsubscribeChildren(this.batch);
        },
        function step2__triggerViewStopEvent(view) {
          view.started = false;
          view.$dom.trigger('viewstop', [ view ]);
          this.ok();
        }
      )
    },
    
    mixinReg = {
      subscribable: subscribable
    };
    
    /**
     * Adds the mixin properties to the given object.
     * 
     * @memberOf niagara.util.mobile.mixins
     * @private
     * @param {Object} obj the object to apply the mixin's properties to
     * @param {Object} mixin
     */
    function doMixinExtend(obj, mixin) {
      baja.iterate(mixin, function (prop, propName) {
        if (propName !== '$name' &&
            propName !== 'postConstructor' &&
            obj[propName] === undefined) {
          obj[propName] = prop;
        }
      });
    }
    
    /**
     * Runs the mixin's postConstructor function on the given view and registers
     * the mixin in the view's <code>$mixins</code> field.
     * 
     * @memberOf niagara.util.mobile.mixins
     * @private
     * @param {niagara.util.mobile.View} view
     * @param {Object} mixin
     */
    function applyMixin(view, mixin) {
      mixin.postConstructor.call(view);
      view.$mixins[mixin.$name] = mixin;
    }
    
    /**
     * Adds a mixin to a given View.
     *
     * @memberOf niagara.util.mobile.mixins
     * @private
     * 
     * @param {View|Function} view the View to be made subscribable. This may
     * be either an existing instance of a View, or a constructor function
     * for a View subclass.
     * @param {Object|String} mixin the mixin to be added - this may either be
     * an actual mixin object (member of 
     * <code>niagara.util.mobile.mixins</code>) or the String name of a
     * mixin (e.g. <code>'subscribable'</code>).
     * @returns {View|Function} the input View with the mixin's behavior added.
     */
    function addMixin(view, mixin) {
      if (typeof mixin === 'string') {
        mixin = mixinReg[mixin];
      }
      
      baja.strictArg(mixin, Object);
      baja.strictArg(mixin.$name, String);
      baja.strictArg(mixin.postConstructor, Function);
      
      if (typeof view === 'function') {
        //view is a View subclass function, so wrap it in some AOP advice to 
        //cause the mixin's postConstructor function to run each time a new 
        //View is created
        view = util.aop.after(function () {
          applyMixin(this, mixin);
        }, view);
        
        //and then stick the mixin properties onto the prototype
        doMixinExtend(view.prototype, mixin);
      } else {
        //we have a constructed instance of a View so just stick the mixin
        //properties onto it directly
        doMixinExtend(view, mixin);
        applyMixin(view, mixin);
      }
      
      return view;
    }
    
    /**
     * Adds subscribable behavior - all members of the
     * <code>niagara.util.mobile.mixins.subscribableMixin</code> field - to
     * the given view.
     * 
     * @memberOf niagara.util.mobile.mixins
     * @param {View|Function} view the View to be made subscribable. This may
     * be either an existing instance of a View, or a constructor function
     * for a View subclass.
     */
    function toSubscribable(view) {
      return addMixin(view, subscribable);
    }
    
    util.api('niagara.util.mixins', {
      addMixin: addMixin,
      toSubscribable: toSubscribable,
      
      subscribable: subscribable
    });
  }());
  
  
  util.api('niagara.util', {
    View: View,
    
    getComponentName: getComponentName
  });
}());