/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to page navigation and interaction within
 * the mobile scheduler app.
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true, plusplus: true, undef: false, bitwise: true */
/*global niagara, baja, $ */


(function () {
  "use strict";
  
  niagara.util.require(
    'jQuery.mobile',
    'niagara.fieldEditors',
    'niagara.fieldEditors.schedule.DayEditor',
    'niagara.util.mobile.ListView',
    'niagara.util.mobile.dialogs',
    'niagara.util.mobile.pages',
    'niagara.util.mobile.views',
    'niagara.util.mobile.commands',
    'niagara.util.schedule',
    'niagara.schedule.Schedule',
    'niagara.schedule.ui.day',
    'niagara.schedule.ui.calendar'
  );
  
      //imports
  var util = niagara.util,
      scheduleUtil = util.schedule,
      dateboxUtil = scheduleUtil.datebox,
      mobileUtil = util.mobile,
      escapeHtml = mobileUtil.escapeHtml,
      views = mobileUtil.views,
      timeUtil = util.time,
      commands = mobileUtil.commands,
      fe = niagara.fieldEditors,
      pages = util.mobile.pages,
      dialogs = util.mobile.dialogs,
      callbackify = util.callbackify,
      
      Schedule = niagara.schedule.Schedule,
      ScheduleBlock = niagara.schedule.ScheduleBlock,
      calendarUI = niagara.schedule.ui.calendar,
      ListView = mobileUtil.ListView,
      RadioButtonView = mobileUtil.RadioButtonView,
      Command = commands.Command,
      AsyncCommand = commands.AsyncCommand,
      
      //constants
      WEEKLY_SCHEDULE_TYPE = 'schedule:WeeklySchedule',
      CALENDAR_SCHEDULE_TYPE = 'schedule:CalendarSchedule',
      TRIGGER_SCHEDULE_TYPE = 'schedule:TriggerSchedule',
      ABSTRACT_SCHEDULE_TYPE = 'schedule:AbstractSchedule',
      DAILY_SCHEDULE_TYPE = 'schedule:DailySchedule',
      DATE_SCHEDULE_TYPE = 'schedule:DateSchedule',
      SCHEDULE_SSC_TYPE = 'mobile:ScheduleServerSideCallHandler',
      DISABLED_CLASS = 'ui-disabled',
      BTN_ACTIVE_CLASS = 'ui-btn-active',
      JQM_HEADER_SELECTOR = ':jqmData(role=header)',
      NAV_BAR_LIST_ITEM_HTML = 
        '<li>' +
          '<a id="{pageId}-{tabId}" ' +
             'class="to-{tabId}" ' +
             'href="#{tabId}" ' +
             'data-theme="b">' +
             '{text}' +
           '</a>' +
         '</li>',

      //private vars
      bajaLex = util.lazyLex('baja'),
      mobileLex = util.lazyLex('mobile'),
      scheduleLex = util.lazyLex('schedule'),
      
      reretrieveScheduleWorkflow,
      saveWorkflow,
      
      scheduleReadonly = false,
      specialEventsRead = false,
      specialEventsReadonly = false,

      currentSchedule, //currently edited niagara.schedule.Schedule
      scheduleComponent, //keep a version subscribed for bajascript events and SSCs
      scheduleSnapshot,
      specialEventsComponent,
      specialEventsSnapshot,
      editedDaySchedule, //day schedule being edited on editDay or editSpecialEvent
      dayEditor, //current day editor on editDay or editSpecialEvent
      copiedDayEditor, //copied day editor, to be used later when Paste chosen
      createMode = true, //whether day editor button should create new block or edit existing one
      blockCommands;
  
  /**
   * Starts up a page-loading spinner, and tells the given callback literal to
   * hide it once ok or fail is called.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {Object} callbacks
   * @param {Number} [timeout] how long to wait until the spinner appears
   * (default 1 second)
   * @returns {Object} new callbacks
   */
  function spinUntilCallback(callbacks, timeout) {
    callbacks = callbackify(callbacks, baja.ok, dialogs.error);
    
    if (timeout === undefined || timeout === null) {
      timeout = 1000;
    }
    var spinner = util.mobile.spinnerTicket(timeout);
    
    function stopSpin() {
      spinner.hide();
    }
    
    util.aop.before(callbacks, 'ok', stopSpin);
    util.aop.before(callbacks, 'fail', stopSpin);
    
    return callbacks;
  }

  /**
   * Repositions/resizes a schedule block div to occupy the correct amount of
   * time/space within a day's div.
   * 
   * @memberOf niagara.schedule.ui
   * 
   * @param {jQuery} div the schedule block div
   */
  function updateScheduleBlockDiv(containerDiv) {
    var block = containerDiv.data('scheduleBlock'),
        start = block.start,
        finish = block.finish,
        value = block.value,
        displayDiv = containerDiv.find('div.display'),
        parentDiv = containerDiv.parent(),
        blockDiv = containerDiv.children('.scheduleBlock'),
        startms = timeUtil.millisOfDay(start),
        stopms = timeUtil.millisOfDay(finish),
        startPct = scheduleUtil.percentageOfDay(startms),
        stopPct = scheduleUtil.percentageOfDay(stopms),
        newTop = Math.floor(startPct * parentDiv.height()),  
        newBottom = Math.floor(stopPct * parentDiv.height()),
        offset = containerDiv.outerHeight() - containerDiv.height() - 2,
        newHeight;

    if (stopms === 0) { //ending at midnight so stretch to bottom
      newBottom = parentDiv.height() + 1;
    }
    
    currentSchedule.stringify(value, function (str) {
      displayDiv.children('span.valueDisplay').text(str);
    });
    
    displayDiv.children('span.timeDisplay')
      .text('(' + start.toString({textPattern: 'HH:mm'}) + ' - ' + 
          finish.toString({textPattern: 'HH:mm'}) + ')');
    
    newHeight = (newBottom - newTop) - offset;

    containerDiv.css({
      top: newTop,
      height: newHeight
    });
    
    block.$colors.apply(blockDiv);
  }
  
  /**
   * Redraws the time labels at the left side of a day/schedule editor ensuring
   * they align properly with the grid lines.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {jQuery} labelsDiv the div containing the time labels
   * @param {Number} height the height to set the labels div to (should be
   * equivalent to the total height of the day editor's blocks div)
   */
  function relayoutLabelsDiv(labelsDiv, height) {
    var blockHeight = height / 8;
    
    labelsDiv.height(height);
    
    baja.iterate(0, 7, function (i) {
      var label = labelsDiv.children('div.scheduleLabel' + i);
      label.height(blockHeight);
      label.css('bottom', blockHeight * (6 - i));
    });
  }
  
  /**
   * Redraws a field editor for a day so it displays correctly given the
   * current visible height.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * 
   * @param {jQuery} dayDiv the div containing a
   * <code>niagara.fieldEditors.schedule.DayEditor</code>
   */
  function relayoutDayDiv(dayDiv) {
    var blocksDisplayHeight = dayDiv.find('div.blocksDisplay').height(),
        blockHeight = blocksDisplayHeight / 8,
        labelsDiv = dayDiv.siblings('div.scheduleLabels');
    
    baja.iterate(0, 8, function (i) {
      var blocks = dayDiv.find('div.block.block' + i);
      blocks.css('top', blockHeight * i);
    });
    
    if (labelsDiv.length) {
      relayoutLabelsDiv(labelsDiv, blocksDisplayHeight);
    }
    
    $('.scheduleBlockContainer', dayDiv).each(function () {
      updateScheduleBlockDiv($(this));
    });
  }
  
  /**
   * Completely redraws a schedule display so it reflects the current status
   * of its underlying <code>schedule:WeeklySchedule</code> component.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   */
  function redrawSchedule() {
      var scheduleDiv = $('#schedule'),
          mainPage = $('#main'),
          labelsDiv = scheduleDiv.find('div.scheduleLabels'),
          blocksDisplayHeight;
      
      mobileUtil.setContentHeight(mainPage);

      currentSchedule.loadValue(currentSchedule.value, function () {
        scheduleDiv.find('div.day').each(function () {
          relayoutDayDiv($(this));
        });
        
        blocksDisplayHeight = scheduleDiv.find('div.blocksDisplay').height();
        
        relayoutLabelsDiv(labelsDiv, blocksDisplayHeight);
      });
  }
  
  /**
   * @memberOf niagara.schedule.ui
   * @private
   */
  function setCreateMode(footer, isCreate) {
    createMode = isCreate;
    $('.edit .ui-icon', footer)
      .toggleClass('ui-icon-grid', !isCreate)
      .toggleClass('ui-icon-plus', isCreate);
    $('.edit .ui-btn-text', footer)
      .text(mobileLex.get(isCreate ? 'create' : 'edit'));
    $('.delete', footer)
      .toggleClass(DISABLED_CLASS, isCreate);
  }
  
  /**
   * Returns a 'snapshot' of a mounted Schedule - all slots are fully
   * populated all the way down the component tree. The component returned
   * will be <b>unmounted</b>.
   * 
   * @inner
   * @private
   * @memberOf niagara.schedule.ui
   *
   * @param {baja.Component} component a mounted Component
   * @param {Object} callbacks an object containing ok/fail callbacks (and
   * optional batch)
   */
  function getSnapshot(component, callbacks) {
    callbacks = util.callbackify(callbacks);
    
    baja.strictArg(component, baja.Component);
    
    if (!component.isMounted()) {
      return callbacks.fail(
          "Cannot call 'getSnapshot' on an unmounted Component");
    }
    
    component.serverSideCall({
      typeSpec: SCHEDULE_SSC_TYPE,
      methodName: 'getSnapshot',
      ok: callbacks.ok,
      fail: callbacks.fail,
      batch: callbacks.batch
    });
  }
  
  
  /**
   * When the current schedule is unmounted from the station, there's nothing
   * left we can do - just show an error dialog and redirect the user back
   * to their home.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   */
  function redirectToHomeOnUnmount() {
    currentSchedule.setModified(false);
    dialogs.ok({
      title: mobileLex.get("schedule.scheduleUnmounted"),
      content: mobileLex.get({
        key: 'schedule.message.scheduleUnmounted',
        args: scheduleComponent.getDisplayName()
      }),
      ok: function (cb) {
        util.mobile.linkToOrd(baja.getUserHome());
        //don't call cb.ok() because this dialog should never close
      }
    });
  }
  
  function setLiveComponent(component) {
    var type = component.getType(),
        cPerm = component.getPermissions(),
        schedule, events, sePerm;
    
    if (type.is(WEEKLY_SCHEDULE_TYPE)) {
      schedule = component.get('schedule');
      events = schedule.get('specialEvents');
    } else if (type.is(CALENDAR_SCHEDULE_TYPE)) {
      schedule = component;
      events = component;
    } else if (type.is(TRIGGER_SCHEDULE_TYPE)) {
      schedule = component;
      events = component.get('dates');
    }
    
    sePerm = events && events.getPermissions();
     
    scheduleComponent = component;
    specialEventsComponent = events;
    scheduleReadonly = !cPerm.hasOperatorWrite() || 
                       (component.getFlags() & baja.Flags.READONLY) ||
                       component.has("ext");
    specialEventsRead = sePerm && sePerm.hasOperatorRead();
    specialEventsReadonly = (sePerm && !sePerm.hasOperatorWrite()) || 
                            (events && events.getFlags() & baja.Flags.READONLY) ||
                            component.has("ext");
    
    
    $('#main-save, #editDay-edit, #editDay-delete, #editDay-actions, ' +
        '#specialEvents-add, #editSpecialEvent-edit, ' +
        '#editSpecialEvent-delete, #editSpecialEvent-actions')
      .toggleClass(DISABLED_CLASS, scheduleReadonly);
  }
  
  function setSnapshot(snapshot, callbacks) {
    currentSchedule = new Schedule(snapshot);
    scheduleSnapshot = snapshot;
    
    var type = snapshot.getType();
    
    if (type.is(WEEKLY_SCHEDULE_TYPE)) {
      specialEventsSnapshot = snapshot.getSchedule().get('specialEvents');
    } else if (type.is(CALENDAR_SCHEDULE_TYPE)) {
      specialEventsSnapshot = snapshot;
    } else if (type.is(TRIGGER_SCHEDULE_TYPE)) {
      specialEventsSnapshot = snapshot.get('dates');
    }
    
    util.aop.after(currentSchedule, 'setModified', function (args) {
      var modified = args[0];
      $('.commandsButton').toggleClass('red', modified);
      $('#main-save').toggleClass(DISABLED_CLASS, !modified);
    });
    currentSchedule.setModified(false);

    currentSchedule.initializeDOM($('#schedule').empty(), callbacks);
  }
  
  
  reretrieveScheduleWorkflow = util.flow.sequential(
      
    function reretrieveSchedule__step1__retrieveLiveComponent(cx) {
      var subscriber = new baja.Subscriber();
      subscriber.attach('unmount', redirectToHomeOnUnmount);

      baja.Ord.make(cx.ord).get({
        ok: this.ok,
        fail: this.fail,
        subscriber: subscriber
      });
    },
    
    function reretrieveSchedule__step2__loadScheduleSlots(cx, component) {
      cx.component = component;
      if (component.has('schedule')) {
        component.get('schedule').loadSlots(this);
      } else {
        this.ok();
      }
    },
    
    function reretrieveSchedule__step3__setReadonlyAndGetSnapshot(cx) {
      var component = cx.component;
      setLiveComponent(component);
      getSnapshot(component, this.ok, this.fail);
    },
    
    function reretrieveSchedule__step4__buildAndLoadEditor(cx, snapshot) {
      setSnapshot(snapshot, this);
    }
  );

  
  /**
   * Retrieves a snapshot of the currently viewed schedule from the server,
   * overwrites the last downloaded instance of it, and updates the
   * schedule display.
   * 
   * @memberOf niagara.schedule.ui
   */
  function reretrieveSchedule(callbacks) {
    reretrieveScheduleWorkflow.invoke({ ord: niagara.view.ord }, callbacks);
  }
  
  /**
   * Show a "schedule saved successfully" dialog.
   * 
   * @memberOf niagara.schedule.ui
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  function confirmSaveAndRedraw(callbacks) {
    callbacks = callbackify(callbacks);
    
    dialogs.ok({
      title: mobileLex.get('schedule.saved'),
      content: mobileLex.get('schedule.message.savedSuccessfully'),
      ok: function (cb) {
        cb.ok(); //close dialog
        callbacks.ok(); //call callback
      }
    });
  }
  
  /**
   * @memberOf niagara.schedule.ui
   * @private
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   */
  function saveDayEditor(callbacks) {
    callbacks = callbackify(callbacks);
    
    if (!dayEditor || !dayEditor.isModified()) {
      return callbacks.ok();
    }
    
    currentSchedule.setModified(true);
    dayEditor.saveValue(callbacks);
  }
  
  saveWorkflow = util.flow.sequential(
    function save__step1__saveDayEditor(cx) {
      saveDayEditor(this);
    },
    function save__step2__saveSchedule(cx) {
      currentSchedule.saveValue(this);
    },
    function save__step3__returnNewSnapshot(cx) {
      var snapshot = currentSchedule.snapshot;
      delete currentSchedule.snapshot;
      this.ok(snapshot);
    }
  );
  
  /**
   * Sends our instance of the schedule up to the server to be saved.
   * 
   * @memberOf niagara.schedule.ui
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  function save(callbacks) {
    callbacks = callbackify(callbacks, baja.ok, function (err) {
      dialogs.error(mobileLex.get('schedule.message.saveFailed'));
    });
    callbacks.timeout = 30000;
    
    saveWorkflow.invoke({}, function (snapshot) {
      confirmSaveAndRedraw(function () {
        setSnapshot(snapshot, function () {
          redrawSchedule();
          callbacks.ok();
        });
      });
    });
  }

  /**
   * @memberOf niagara.schedule.ui
   * @private
   */
  function confirmAbandonChanges(callbacks) {
    callbacks = callbackify(callbacks);
    
    if (currentSchedule.isModified()) {
      dialogs.confirmAbandonChanges({
        viewName: scheduleLex.get('scheduler.weeklySchedule'),
        yes: function (cb) {
          save(function () {
            cb.ok();
            callbacks.ok();
          });
        },
        no: function (cb) {
          reretrieveSchedule(function () {
            cb.ok();
            callbacks.ok();
          });
        },
        cancel: function (cb) {
          dialogs.closeCurrent();
        }
      });
    } else {
      callbacks.ok();
    }
  }
  
  /**
   * Shows JQM loading message.
   * @memberOf niagara.schedule.ui
   * @private
   */
  function showLoading() {
    $.mobile.showPageLoadingMsg();
  }
  
  /**
   * Hides JQM loading message.
   * @memberOf niagara.schedule.ui
   * @private
   */
  function hideLoading() {
    $.mobile.hidePageLoadingMsg();
  }

  /**
   * Adds the navigation header (weekly schedule, special events, etc) to a 
   * JQM page.
   * @memberOf niagara.schedule.ui
   * @private
   * @param {jQuery} page the JQM page to append the navbar to
   */
  function appendNavbar(page) {
    var pageId = page.attr('id'),
        headerDiv = page.children(JQM_HEADER_SELECTOR),
        navbar = $('<div data-role="navbar"/>').appendTo(headerDiv),
        ul = $('<ul/>').appendTo(navbar),
        type = scheduleComponent.getType(),
        isWeekly = type.is(WEEKLY_SCHEDULE_TYPE),
        isTrigger = type.is(TRIGGER_SCHEDULE_TYPE);
    
    if (isWeekly) {
      ul.append(NAV_BAR_LIST_ITEM_HTML.patternReplace({
        pageId: pageId,
        tabId: 'main',
        text: escapeHtml(scheduleLex.get('scheduler.weeklySchedule'))
      }));
    }
    
    ul.append(NAV_BAR_LIST_ITEM_HTML.patternReplace({
      pageId: pageId,
      tabId: 'specialEvents',
      text: escapeHtml(scheduleLex.get('scheduler.specialEvents'))
    }));
    
    if (isTrigger) {
      ul.append(NAV_BAR_LIST_ITEM_HTML.patternReplace({
        pageId: pageId,
        tabId: 'triggers',
        text: escapeHtml(mobileLex.get('schedule.triggers'))
      }));
    }
    
    if (isWeekly) {
      ul.append(NAV_BAR_LIST_ITEM_HTML.patternReplace({
        pageId: pageId,
        tabId: 'properties',
        text: escapeHtml(scheduleLex.get('scheduler.properties'))
      }));
    }
    
    ul.append(NAV_BAR_LIST_ITEM_HTML.patternReplace({
      pageId: pageId,
      tabId: 'summary',
      text: escapeHtml(scheduleLex.get('summary'))
    }));
    
    if (!specialEventsRead) {
      headerDiv.find('.to-specialEvents').addClass(DISABLED_CLASS);
    }
  }
  
  /**
   * Displays a field editor for a <code>schedule:TimeSchedule</code> in a
   * dialog. Used on both the <code>editDay</code> and 
   * <code>editSpecialEvent</code> pages when editing a draggable schedule block
   * directly using the datebox field editors.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {niagara.fieldEditors.BaseFieldEditor} timeScheduleEditor the field
   * editor for the <code>schedule:TimeSchedule</code>
   */
  function showTimeScheduleDialog(timeScheduleEditor, callbacks) {
    callbacks = callbackify(callbacks);
    
    dialogs.okCancel({
      
      title: mobileLex.get('schedule.' + 
          (createMode ? 'createBlock' : 'editBlock')),
      
      content: function (targetElement, callbacks) {
        timeScheduleEditor.buildAndLoad(targetElement, {
          ok: function () {
            if (createMode) {
              return callbacks.ok();
            }
            
            //arm an event listener on the time editors so we validate the
            //time range each time a new value is entered
            timeScheduleEditor.$dom.delegate('input:jqmData(role="datebox")', 
                'datebox', function (e, passed) {
              if (passed.method === 'offset') {
                var changems;
                switch (passed.type) {
                case 'h':
                  changems = timeUtil.MILLIS_IN_HOUR * passed.amount;
                  break;
                case 'i':
                  changems = timeUtil.MILLIS_IN_MINUTE * passed.amount;
                  break;
                }
                dateboxUtil.validateTimeSchedule(
                    $(this), dayEditor, timeScheduleEditor, changems);
              }
              return false; //validateTimeSchedule sets the datebox content
            });
            callbacks.ok();
          },
          fail: callbacks.fail
        });
      },
      
      //after the OK button is clicked
      ok: function (cb) {
        timeScheduleEditor.validate({
          ok: function () {
            timeScheduleEditor.saveValue({
              ok: function (timeSchedule) {
                //set last entered value - this will be used in
                //niagara.schedule.ui.day so new blocks will be have this value
                //as default
                niagara.schedule.lastEnteredValue = timeSchedule.get('effectiveValue');
                dayEditor.setModified(true);
                callbacks.ok(timeSchedule, cb);
              }
            });
          },
          fail: function (err) {
            cb.fail(err);
            callbacks.fail(err);
          }
        });
      },
      
      //after the dialog is closed
      callbacks: {
        ok: function () {
          
          //Upon hiding the block editor, delete any cruft the datebox plugin
          //left lying around the DOM since datebox doesn't know how to clean
          //up after itself
          $('#okCancelDialog')
            .find('.ui-datebox-screen, .ui-datebox-container')
            .remove();
          $.mobile.pageContainer.children('.ui-dialog-datebox').remove();
        },
        fail: dialogs.error
      }
    });
  }
  
  function validateTimeSchedule(saveData, callbacks) {
    var timeSchedule = saveData,
        startTime = timeSchedule.getStart(),
        finishTime = timeSchedule.getFinish(),
        mschange = 0;
    
    try {
      dateboxUtil.validateTimeChange(
          dayEditor, startTime, finishTime, 'start', mschange);
      dateboxUtil.validateTimeChange(
          dayEditor, startTime, finishTime, 'finish', mschange);
      callbacks.ok();
    } catch (e) {
      callbacks.fail(e);
    }
  }
  
  /**
   * Attempts to edit the currently selected schedule block. Shows a field
   * editor for the block's day schedule, with editors for start time, end
   * time, and effective value. Also arms handlers on the field editor to
   * perform validation on the entered time range (prevent overlapping with
   * another block etc).
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {niagara.schedule.ScheduleBlock} block the block to edit
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {niagara.schedule.ScheduleBlock} the block after changes are
   * saved (to be passed to callbacks' ok handler)
   */
  function editBlock(block, callbacks) {
    callbacks = callbackify(callbacks);
    
    if (!block) {
      return;
    }
    
    fe.makeFor({
      value: block.createTimeSchedule(),
      facets: currentSchedule.value.get('facets')
    }, function (timeScheduleEditor) {
      timeScheduleEditor.addValidateStep(validateTimeSchedule);
      showTimeScheduleDialog(timeScheduleEditor, function (timeSchedule, cb) {
        //re-run constructor to overwrite properties
        ScheduleBlock.call(block, timeSchedule);
        callbacks.ok(block);
        cb.ok();
      });
    });
  }
  
  /**
   * Show a time schedule edit dialog. If we are in create mode (no block
   * selected), then create a new time schedule and edit it (without the 
   * real-time time range validation as this makes no sense on a brand new
   * time schedule). If not in create mode, then just edit the selected
   * block.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   */
  function createOrEdit(callbacks) {
    callbacks = callbackify(callbacks);
    if (createMode) {
      var block = new ScheduleBlock(baja.Time.DEFAULT, baja.Time.DEFAULT,
          currentSchedule.getNewBlockValue());
      editBlock(block, function (block) {
        dayEditor.addBlock(block.start, block.finish, block.value);
        callbacks.ok();
      });
    } else {
      editBlock(dayEditor.getSelectedBlock(), callbacks.ok);
    }
  }
  
  /**
   * Removes any handlers the last edited day may have registered on
   * <code>$(document)</code> - see 
   * <code>niagara.schedule.ui.day.createHandlers()</code>. Then 
   * instantiates a new editor for the currently edited day.
   * @memberOf niagara.schedule.ui
   * @private
   * @param {Object} params an object literal to be passed into
   * <code>fieldEditors.makeFor</code> to create the day field editor
   * @param {baja.Component} params.value the <code>schedule:DaySchedule</code>
   * we want to start editing
   * @param {Object} [callbacks] an object containing ok/fail callbacks to be
   * called when the editor is instantiated
   * @param {Function} [callbacks.ok] the new field editor will be passed as
   * the first parameter to this callback
   */
  function resetDayEditor(params, callbacks) {
    callbacks = callbackify(callbacks);
    params = baja.objectify(params, 'value');
    
    if (!params.value) {
      return callbacks.ok();
    }
    
    var container = params.value.getParent(),
        slot = params.value.getPropertyInParent();
    params.container = container;
    params.slot = slot;
    
    if (dayEditor && dayEditor.unbindAll) {
      dayEditor.unbindAll();
    }
    
    fe.makeFor(params, function (editor) {
      dayEditor = editor;
      callbacks.ok(editor);
    });
  }
  
  //these are the commands shown when clicking "options" on a day editor,
  //equivalent to right-clicking a schedule block in the Workbench schedule
  //editor
  blockCommands = {
    'delete': new Command("%lexicon(schedule:day.delete)%", function () {
      dayEditor.removeBlock(dayEditor.getSelectedBlock());
    }),
    'all_day': new Command("%lexicon(schedule:day.all_day)%", function () {
      var block = dayEditor.getSelectedBlock();
      dayEditor.allDayEvent(block ? block.value : currentSchedule.getNewBlockValue());
    }),
    'apply_weekdays': new Command("%lexicon(schedule:day.apply_weekdays)%", function () {
      currentSchedule.applyMF(dayEditor);
    }),
    'copy_day': new Command("%lexicon(schedule:day.copy_day)%", function () {
      copiedDayEditor = dayEditor;
    }),
    'clear_day': new Command("%lexicon(schedule:day.clear_day)%", function () {
      dayEditor.empty();
    }),
    'clear_week': new Command("%lexicon(schedule:day.clear_week)%", function () {
      currentSchedule.empty();
      resetDayEditor(editedDaySchedule);
    }),
    'paste_day': new Command("%lexicon(schedule:day.paste_day)%", function () {
      if (copiedDayEditor) {
        copiedDayEditor.copyTo(dayEditor);
      }
    })
  };
  
  /**
   * Shows a dialog with options (apply M-F, Clear Day, etc) for a schedule
   * block or day. If the current day editor does not have a schedule block
   * selected, then no block-specific actions (e.g. delete event) will be 
   * shown, only day-specific ones.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {Object} optionsToShow a mapping from the different option keys
   * (corresponding to day.* entries in the schedule lexicon) to boolean
   * values indicating whether or not that option should be shown.
   */
  function showBlockActions(optionsToShow) {
    var block = dayEditor.getSelectedBlock(),
        filteredCommands = [],
        title = dayEditor.$dayDisplay +
          (block ? ' (' + block.toString() + ')' : '');
    
    baja.iterate(blockCommands, function (command, commandName) {
      if (optionsToShow[commandName]) {
        filteredCommands.push(command);
      }
    });
    
    commands.showCommandsDialog(filteredCommands, title);
  }
  
  /**
   * Registers event listeners using the pages framework.
   * 
   * @namespace
   * @name niagara.schedule.ui.pages
   * @function
   * @private
   */
  function registerPages() {
    
    var saveCommand = new AsyncCommand("%lexicon(mobile:save)%", save),
        refreshCommand = new AsyncCommand("%lexicon(mobile:refresh)%", function (callbacks) {
          confirmAbandonChanges(function () {
            reretrieveSchedule(callbacks);
          });
        });
    
    (function addConfirmToHomeCommand() {
      var homeCmd = commands.getHomeCmd(),
          defaultCommands = commands.getDefaultCommands(),
          index = util.indexOf(defaultCommands, homeCmd);
      
      defaultCommands[index] = new AsyncCommand(homeCmd.getDisplayName(), function (callbacks) {
        if (currentSchedule.isModified() || (dayEditor && dayEditor.isModified())) {
          dialogs.confirmAbandonChanges({
            viewName: scheduleLex.get('scheduler.weeklySchedule'),
            yes: function (cb) {
              save(function () {
                homeCmd.invoke();
              });
            },
            no: function (cb) {
              currentSchedule.setModified(false); //prevent onbeforeunload
              homeCmd.invoke();
            },
            callbacks: callbacks
          });
        } else {
          homeCmd.invoke();
        }
      });
      
      commands.setDefaultCommands(defaultCommands);
      
    }());
    
    /**
     * Adds a save command (only if the current schedule is modified) and a
     * refresh command to the list of default commands.
     * 
     * @memberOf niagara.schedule.ui.pages
     * @private
     */
    function getDefaultCommands(obj) {
      var cmds = currentSchedule.isModified()
        ? [saveCommand, refreshCommand]
        : [refreshCommand];
      return cmds.concat(obj.commands);
    }

    pages.register("main", (function main() {
      
      /**
       * Arm save/refresh buttons in footer nav bar, and enable linking to
       * day editor when clicking a day in the main schedule display.
       * @memberOf niagara.schedule.ui.pages.main
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);
        
        mobileUtil.preventNavbarHighlight($('#main-footer'));

        $('#main-save').click(function () {
          var $this = $(this);
          if (!$this.hasClass(DISABLED_CLASS)) {
            $this.addClass(DISABLED_CLASS);
            save();
          }
        });
        
        $('#main-refresh').click(function () {
          confirmAbandonChanges(function () {
            reretrieveSchedule(function () {
              redrawSchedule();
            });
          });
        });
        
        $('#schedule').delegate('div.day', 'click', function () {
          pages.getHandler('editDay').load($(this).data('daySchedule'));
        });
      }
      
      /**
       * Sets the height of the <code>#main</code> page's content div to fill
       * the screen, then redraws the weekly schedule display to fill that
       * space. Called when the page is loaded or when the window is resized.
       * @memberOf niagara.schedule.ui.pages.main
       * @see niagara.schedule.ui.initializeUI 
       */
      function redraw() {
        mobileUtil.setContentHeight($('#main'));
        if (currentSchedule) {
          redrawSchedule();
        }
      }
      
      /**
       * On every page view, if we have a Schedule object loaded, call
       * <code>redraw()</code> to update the display.
       * @memberOf niagara.schedule.ui.pages.main
       */
      function pagelayout() {
        $('#main-main').addClass(BTN_ACTIVE_CLASS);
        $('#main-title').text(scheduleComponent.getDisplayName());
        redraw();
      }
      
      /**
       * If we're not actually viewing a weekly schedule (i.e. a calendar or
       * trigger schedule), this page shouldn't even be accessible - so redirect
       * the user directly to the special events page.
       * @memberOf niagara.schedule.ui.pages.main
       */
      function pagebeforeshow(obj) {
        if (scheduleComponent && !scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
          $.mobile.changePage('#specialEvents', {
            changeHash: false,
            transition: 'none'
          });
          obj.event.preventDefault();
        }
      }
      
      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.main
       */
      return {
        pagebeforeshow: pagebeforeshow,
        pagebeforecreate: pagebeforecreate,
        pagelayout: pagelayout,
        redraw: redraw,
        
        getCommands: getDefaultCommands
      };
    }()));
    
    pages.register("editDay", (function editDay() {
      /**
       * Returns a click handler to be bound to a back button. If the currently
       * edited day is in a modified state, will pop up a dialog asking the user
       * whether or not to save changes before allowing the backward navigation
       * to take place.
       * 
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       */
      function checkForChangesFunction(targetPage) {

        return function () {
          if (dayEditor.isModified()) {
            dialogs.confirmAbandonChanges({
              yes: saveDayEditor,
              redirect: targetPage,
              viewName: mobileLex.get('schedule.dayEditor')
            });
            return false;
          }
        };
      }
      
      
      /**
       * Loads the available actions for the currently edited day (copy/paste,
       * all day, apply MF, etc.). The actions shown will differ depending on 
       * whether or not a schedule block is currently selected.
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       */
      function loadBlockActions() {
        var selectedBlock = dayEditor.getSelectedBlock(),
            hasBlocks = !!dayEditor.scheduleBlocks.length,
            hasSelected = !!selectedBlock,
            hasCopied = !!copiedDayEditor;
        
        showBlockActions({
          'delete': hasSelected,
          'paste_day': hasCopied,
          'all_day': true,
          'apply_weekdays': hasBlocks,
          'copy_day': hasBlocks,
          'clear_day': hasBlocks,
          'clear_week': true
        });
      }
      
      /**
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagecreate(obj) {
        var backButton = obj.page
          .children(JQM_HEADER_SELECTOR)
          .find('a.profileHeaderBack');
        backButton.bind('click', checkForChangesFunction('#main'));
      }
      
      /**
       * Sets the <code>#editDay</code> page's content div to fill the screen,
       * then redraws the day editor to fill that space. Does not add/remove
       * blocks to reflect the structure of the underlying day schedule - only
       * moves the existing divs around to be correctly laid out.
       * <p>
       * Runs when the page is shown or when the window is resized.
       * 
       * @memberOf niagara.schedule.ui.pages.editDay
       * @see niagara.schedule.ui.initializeUI
       */
      function relayout() {
        mobileUtil.setContentHeight($('#editDay'));
        relayoutDayDiv($('#editDay-day > div.schedule-DaySchedule'));
      }
      
      /**
       * Completely rebuilds the structure of the day editor to reflect the
       * current state of the day schedule, throwing away any user edited
       * changes.
       * <p>
       * Runs each time the page is loaded (when clicking on a day on the main
       * schedule tab), and after clicking Refresh. <code>relayout</code> 
       * will be called after the day editor is rebuilt.
       * 
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       * @param {Object} callbacks an object containing ok/fail callbacks
       */
      function rebuild(callbacks) {
        callbacks = callbackify(callbacks);
        
        var dayDiv = $('#editDay-day').empty();
        scheduleUtil.appendTimeLabels(dayDiv);
        
        dayEditor.buildAndLoad(dayDiv, function () {
          dayEditor.setModified(false);
          setCreateMode($('#editDay-footer'), true);
          relayout();
          callbacks.ok();
        });
      }
      
      /**
       * Repaints the day editor to ensure that any user entered changes
       * (from clicking the create/delete/edit footer buttons) are reflected
       * on the screen.
       * 
       * @memberOf niagara.schedule.ui.pages.editDay
       * @private
       */
      function refresh() {
        dayEditor.refreshWidgets();
        setCreateMode($('#editDay-footer'), true);
        relayout();
      }
      
      /**
       * Arms click handlers on the edited day.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagebeforecreate() {
        //toggle edit/create when we select/deselect a block
        $('#editDay-day').bind('selectionchange', function () {
          var selected = !!dayEditor.getSelectedBlock();
          setCreateMode($('#editDay-footer'), !selected);
        });
        
        //edit a block by double clicking
        $('#editDay-day').bind('dblclick', function () {
          editBlock(dayEditor.getSelectedBlock());
        });
        
        //prevent footer buttons from staying highlighted
        mobileUtil.preventNavbarHighlight($('#editDay-footer'));
        
        $('#editDay-footer').delegate('a', 'click', function () {
          var $this = $(this);
          
          switch ($this.attr('id')) {
          
          case 'editDay-save':
            saveDayEditor();
            break;
            
          case 'editDay-refresh':
            if (dayEditor.isModified()) {
              dialogs.confirmAbandonChanges({
                yes: saveDayEditor,
                no: rebuild,
                viewName: mobileLex.get('schedule.dayEditor')
              });
            } else {
              rebuild();
            }
            break;
            
          case 'editDay-actions':
            loadBlockActions();
            break;
            
          case 'editDay-edit':
            createOrEdit(refresh);
            break;
            
          case 'editDay-delete':
            dayEditor.removeBlock(dayEditor.getSelectedBlock());
            refresh();
            break;
          }
        });
      }
      
      /**
       * Updates the day editor div to show the current state of the edited
       * day.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagebeforeshow() {
        refresh();
        setCreateMode($('#editDay-footer'), true);
      }

      /**
       * Loads, and instigates a page change to, the page to edit the specified
       * day. The current day editor will be completely reloaded and refreshed
       * to begin editing the day.
       *  
       * @memberOf niagara.schedule.ui.pages.editDay
       * @param {baja.Component} day the <code>schedule:DaySchedule</code> to 
       * edit
       */
      function load(daySchedule) {
        editedDaySchedule = daySchedule;
        
        var parent = daySchedule.getParent(),
            parentName = parent && parent.getName(),
            label = parentName && baja.lex('baja').get(parentName),
            params = {
              value: daySchedule,
              readonly: scheduleReadonly,
              label: label
            };
        
        resetDayEditor(params, function (dayEditor) {
          rebuild();
          $.mobile.changePage('#editDay');
        });
      }
            
      /**
       * Ensures that the day editor paints correctly given the current
       * visible height.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagelayout(obj) {
        relayout();
      }
      
      /**
       * If the day editor is modified, prompts for changes before navigating
       * away.
       * @memberOf niagara.schedule.ui.pages.editDay
       */
      function pagebeforechange(obj) {
        if (dayEditor.isModified()) {
          dialogs.confirmAbandonChanges({
            yes: function (cb) {
              saveDayEditor(cb);
            },
            no: function (cb) {
              resetDayEditor(editedDaySchedule, cb);
            },
            redirect: obj.nextPage,
            viewName: mobileLex.get('schedule.dayEditor')
          });
          obj.event.preventDefault();
        }
      }

      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.editDay
       */
      return {
        load: load,
        rebuild: rebuild,
        relayout: relayout,
        pagebeforechange: pagebeforechange,
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        pagecreate: pagecreate,
        pagelayout: pagelayout
      };
    }()));
    
    pages.register("properties", (function () {
      var propertyEditors,
          effectiveRangeEditor,
          validateAndSaveWorkflow,
          loadPropertiesEditorWorkflow,
          loadPropertiesEditorsWorkflow;
      
      /**
       * The asynchronous sequential-parallel workflow for validating and
       * saving the properties tab field editors.
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @field
       */
      validateAndSaveWorkflow = util.flow.sequentialParallel(
        function validateAndSaveWorkflow__step1__validate(editor) {
          editor.validate(this);
        },
        function validateAndSaveWorkflow__step2__saveValue(editor) {
          editor.saveValue(this);
        }
      );
      
      /**
       * Sequential async workflow to instantiate and build one particular
       * field editor for the current schedule's properties tab. Invoked once
       * for each field editor in <code>loadPropertiesEditorsWorkflow</code>.
       * 
       * @memberOf niagara.schedule.ui.pages.properties
       * @field
       * @private
       */
      loadPropertiesEditorWorkflow = util.flow.sequential(
        function loadPropertiesEditorWorkflow__step1__makeFor(cx) {
          fe.makeFor({
            container: cx.schedule,
            slot: cx.schedule.getSlot(cx.slot),
            facets: cx.facets || cx.schedule.get('facets'),
            readonly: scheduleReadonly
          }, this);
        },
        function loadPropertiesEditorWorkflow__step2__buildAndLoad(cx, editor) {
          var targetElement = fe.toLabeledEditorContainer(editor, 
              $('<div class="defaultOutput"/>').appendTo(cx.targetElement));
          editor.buildAndLoad(targetElement, this);
        },
        function loadPropertiesEditorWorkflow__step3__addToEditorList(cx, editor) {
          cx.editors.push(editor);
          this.ok();
        }
      );
      
      /**
       * Sequential async workflow to instantiate the 3 different field editors
       * for the current schedule's properties tab. 
       * @memberOf niagara.schedule.ui.pages.properties
       * @field
       * @private
       */
      loadPropertiesEditorsWorkflow = util.flow.sequential(
        function loadPropertiesEditorsWorkflow__step1__defaultOutput(cx) {
          loadPropertiesEditorWorkflow.invoke({
            schedule: cx.schedule,
            slot: 'defaultOutput',
            targetElement: $('<div class="defaultOutput"/>').appendTo(cx.targetElement),
            editors: cx.editors
          }, this);
        },
        function loadPropertiesEditorsWorkflow__step2__facets(cx) {
          loadPropertiesEditorWorkflow.invoke({
            schedule: cx.schedule,
            slot: 'facets',
            targetElement: $('<div class="facets"/>').appendTo(cx.targetElement),
            editors: cx.editors
          }, this);
        },
        function loadPropertiesEditorsWorkflow__step3__cleanupExpiredEvents(cx) {
          loadPropertiesEditorWorkflow.invoke({
            schedule: cx.schedule,
            slot: 'cleanupExpiredEvents',
            targetElement: $('<div class="cleanup"/>').appendTo(cx.targetElement),
            facets: {
              trueText: bajaLex.get('true'),
              falseText: bajaLex.get('false')
            },
            editors: cx.editors
          }, this);
        }
      );
      
      /**
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @param {jQuery} targetElement the element containing the calendar
       * datebox
       * @param {baja.AbsTime} the new time (month) to load into the calendar
       * datebox
       */
      function doCalendarUpdate(targetElement, newTime) {
        effectiveRangeEditor.getSaveData({
          ok: function (effectiveRange) {
            calendarUI.showEffectiveRangeCalendar(
                targetElement, 
                effectiveRange, 
                newTime);
          }
        });
      }
      
      /**
       * Validates and saves the Properties tab field editors for effective
       * date range, facets, etc.
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @param {Object} editors an object literal containing the field editors
       * to save
       * @param {Object} callbacks an object containing ok/fail callbacks
       */
      function validateAndSaveEditors() {
        var allEditors = [].concat(propertyEditors),
            editorsToSave = [];
        
        allEditors.push(effectiveRangeEditor);
        
        baja.iterate(allEditors, function (editor) {
          if (editor.isModified()) {
            editorsToSave.push(editor);
          }
        });
        
        if (editorsToSave.length > 0) {
          validateAndSaveWorkflow.invoke(editorsToSave, function () {
            currentSchedule.setModified(true);
          });
        }
      }
      
      /**
       * Arms handlers for save/cancel and to ensure the calendar preview
       * updates its display when a different month is shown.
       * @memberOf niagara.schedule.ui.pages.properties
       */
      function pagebeforecreate(obj) {
        //if user clicks +/- a bunch of times in a row, don't go through
        //a round trip to the server until they've settled on a month
        var debouncedCalendarUpdate = util.debounce(doCalendarUpdate, 600);

        appendNavbar(obj.page);
        
        $('#properties-save').click(function () {
          validateAndSaveEditors();
        });

        $('#properties-calendars').delegate('input:jqmData(role="datebox")', 'datebox', function (e, passed) {
          if (passed.method === 'offset') { //+ or - button clicked
            var input = $(this),
                absTime = dateboxUtil.getAbsTime(input),
                newTime = scheduleUtil.addMonths(absTime, passed.amount);
            
            debouncedCalendarUpdate($('#properties-calendars'), newTime);
          }
        });
        
        mobileUtil.preventNavbarHighlight($('#properties-footer'));
      }
      
      /**
       * Instantiates the field editors for the schedule's properties
       * (facets, default output, cleanup special events). After this function
       * (asynchronously) completes, the <code>propertyEditors</code> variable
       * will be populated with an array containing field editor instances for
       * the schedule's <code>defaultOutput</code>, <code>facets</code>, and
       * <code>cleanupExpiredEvents</code> properties. 
       * 
       * @memberOf niagara.schedule.ui.pages.properties
       * @private
       * @param {jQuery} targetElement the div in which to display the field editors
       * @param {Object} [callbacks] an object containing ok/fail callbacks
       */
      function loadPropertiesEditors(targetElement, callbacks) {
        loadPropertiesEditorsWorkflow.invoke({
          schedule: currentSchedule.value,
          editors: propertyEditors = [],
          targetElement: $(targetElement).empty()
        }, callbacks);
      }
      
      /**
       * Loads and shows the necessary field editors and calendar preview
       * for the properties tab.
       * @memberOf niagara.schedule.ui.pages.properties
       */
      function pagebeforeshow() {
        var sched = currentSchedule.value;
        
        loadPropertiesEditors($('#properties-editors'));
        
        calendarUI.loadScheduleEditor($('#properties'), sched.getEffective(), {
          ok: function (editor) {
            calendarUI.bindEditorToCalendar(editor, $('#properties-calendars'));
            effectiveRangeEditor = editor;
          },
          fail: function (err) {
            baja.error("properties: could not load schedule editor: " + err);
          }
        }, scheduleReadonly);
        
        calendarUI.showEffectiveRangeCalendar(
            $('#properties-calendars'), 
            sched.getEffective(), 
            baja.AbsTime.now());
        
        $('#properties-properties').addClass(BTN_ACTIVE_CLASS);
        $('#properties-title').text(scheduleComponent.getDisplayName());
      }
      
      function pageshow() {
        hideLoading();
      }
      
      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.properties
       */
      return {
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        pageshow: pageshow,
        getCommands: getDefaultCommands
      };
    }()));
    
    pages.register("specialEvents", (function specialEvents() {
      var radioView,
          SpecialEventsRadioView,
          redrawSpecialEventsWorkflow,
          events;
      
      redrawSpecialEventsWorkflow = util.flow.sequential(
        function redrawSpecialEventsWorkflow__step1__initializeDOM(listView) {
          listView.initializeDOM($('#specialEvents-eventList').empty(), this);
        },
        function redrawSpecialEventsWorkflow__step2__loadValue(listView) {
          listView.loadValue(events, this);
        },
        function redrawSpecialEventsWorkflow__step3__triggerSlotChange(listView) {
          listView.setSelectedSlot(listView.$selectedSlot);
          this.ok();
        }
      );
      
      /**
       * A radio button view for showing the current schedule's special
       * events on the <code>specialEvents</code> page.
       * 
       * @class
       * @name niagara.schedule.ui.pages.specialEvents.SpecialEventsRadioView
       * @private
       * @extends niagara.util.mobile.RadioButtonView
       */
      SpecialEventsRadioView = RadioButtonView.subclass();
      
      /**
       * Only show those slots that are of type 
       * <code>schedule:AbstractSchedule</code>.
       * @name niagara.schedule.ui.pages.specialEvents.SpecialEventsRadioView#shouldIncludeSlot
       * @function
       */
      SpecialEventsRadioView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
        return slot.isProperty() &&
               slot.getType().is(ABSTRACT_SCHEDULE_TYPE);
      };
      
      SpecialEventsRadioView.prototype.doLoadValue = function(value, callbacks) {
        if (value.getSlots().properties().dynamic().toArray().length) {
          RadioButtonView.prototype.doLoadValue.call(this, value, callbacks);
        } else {
          this.$dom.text(mobileLex.get('schedule.message.noSpecialEvents'));
          callbacks.ok();
        }
      };

      /**
       * Makes multiline labels for the radio buttons - display name on top,
       * display value below.
       * 
       * @name niagara.schedule.ui.pages.specialEvents.SpecialEventsRadioView#makeLabel
       * @function
       */
      SpecialEventsRadioView.prototype.makeLabel = function (complex, slot) {
        var pageId = util.mobile.encodePageId(String(slot)),
            display = complex.getDisplay(slot),
            displayName = baja.SlotPath.unescape(String(slot)),
            label = $('<label data-theme="c"/>').attr('for', pageId);
        
        label
          .append($('<span/>').text(displayName))
          .append($('<br/>'))
          .append($('<span/>').text(display));
        
        return label;
      };
      
      /**
       * After the user selects a special event, enable the footer buttons
       * for editing.
       * 
       * @name niagara.schedule.ui.pages.specialEvents.SpecialEventsRadioView#selectionChanged
       * @function
       */
      util.aop.after(SpecialEventsRadioView.prototype, 'selectionChanged', function (args) {
        if (!specialEventsReadonly) {
          $('#specialEvents-footer a').removeClass(DISABLED_CLASS);        
          var slot = String(args[0]),
              value = this.value,
              slots = this.value.getSlots().is(ABSTRACT_SCHEDULE_TYPE).toArray(),
              firstSlot = slots[0],
              lastSlot = slots[slots.length - 1];
          
          if (firstSlot && slot === String(firstSlot)) {
            $('#specialEvents-priorityUp').addClass(DISABLED_CLASS);
          }
          
          if (lastSlot && slot === String(lastSlot)) {
            $('#specialEvents-priorityDown').addClass(DISABLED_CLASS);
          }
        } else {
          $('#specialEvents-edit').removeClass(DISABLED_CLASS);
        }
      });
      
      /**
       * Refreshes the special events radio buttons to reflect any changes
       * the user may have made on the edit screen.
       * 
       * @memberOf niagara.schedule.ui.pages.specialEvents
       * @private
       */
      function redrawSpecialEvents(callbacks) {
        callbacks = callbackify(callbacks);
        redrawSpecialEventsWorkflow.invoke(radioView, callbacks);
      }
      
      /**
       * Enable click handlers for selecting a special event and manipulating
       * it via the buttons in the footer bar.
       * 
       * @memberOf niagara.schedule.ui.pages.specialEvents
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);
        radioView = new SpecialEventsRadioView();

        mobileUtil.preventNavbarHighlight($('#specialEvents-navbar'));
        
        $('#specialEvents-navbar').delegate('a', 'click', function () {
          var $this = $(this),
              selectedSlot = String(radioView.getSelectedSlot());
          
          if ($this.hasClass(DISABLED_CLASS)) {
            return;
          }
          
          switch ($(this).attr('id')) {
          case "specialEvents-add":
            pages.getHandler('addSpecialEvent').load(events);
            break;
          case "specialEvents-edit": 
            pages.getHandler('editSpecialEvent').load(radioView.getSelectedValue());
            break;
          case "specialEvents-priorityUp":
            util.slot.moveUp(events, selectedSlot);
            redrawSpecialEvents();
            currentSchedule.setModified(true);
            break;
          case "specialEvents-priorityDown":
            util.slot.moveDown(events, selectedSlot);
            redrawSpecialEvents();
            currentSchedule.setModified(true);
            break;
          case "specialEvents-delete":
            dialogs.yesNo({
              content: mobileLex.get({
                key: 'schedule.message.confirmDeleteSpecialEvent',
                def: '{0}',
                args: [ baja.SlotPath.unescape(String(selectedSlot)) ]
              }),
              yes: function (cb) {
                events.remove(selectedSlot);
                radioView.setSelectedSlot(null);
                redrawSpecialEvents();
                currentSchedule.setModified(true);
                cb.ok();
              }
            });
            break;
          case "specialEvents-rename":
            dialogs.fieldEditor({
              title: scheduleLex.get('composite.rename'),
              value: baja.SlotPath.unescape(selectedSlot),
              ok: function (newName, cb) {
                newName = baja.SlotPath.escape(newName);
                if (newName !== selectedSlot) {
                  events.rename({
                    slot: selectedSlot,
                    newName: newName,
                    ok: function () {
                      redrawSpecialEvents();
                      currentSchedule.setModified(true);
                      cb.ok();
                    },
                    fail: cb.fail
                  });
                } else {
                  cb.ok();
                }
              }
            });
            break;
          }
        });
      }
      
      /**
       * Populate the list of special events to be displayed.
       * @memberOf niagara.schedule.ui.pages.specialEvents
       */
      function pagebeforeshow() {
        events = specialEventsSnapshot;
        redrawSpecialEvents(function () {
          if (!radioView.getSelectedSlot()) {
            $('#specialEvents-footer a').addClass(DISABLED_CLASS);
          }
        });
        
        if (!specialEventsReadonly) {
          $('#specialEvents-add').removeClass(DISABLED_CLASS);
        }
        $('#specialEvents-specialEvents').addClass(BTN_ACTIVE_CLASS);
        $('#specialEvents-title').text(scheduleComponent.getDisplayName());
        $('#specialEvents-edit .ui-btn-text').text(mobileLex.get(
          specialEventsReadonly ? 'view' : 'edit'
        ));
      }
      
      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.specialEvents
       */
      return {
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        getCommands: getDefaultCommands
      };
    }()));
    
    pages.register('triggers', (function triggers() {
      var triggersEditor;
      
      /**
       * @memberOf niagara.schedule.ui.pages.triggers
       */
      function pagebeforeshow() {
        $('#triggers-triggers').addClass(BTN_ACTIVE_CLASS);
        $('#triggers-remove').addClass(DISABLED_CLASS);
        
        fe.makeFor({
          value: scheduleSnapshot.get('times'),
          key: 'triggers-list'
        }, function (editor) {
          triggersEditor = editor;
          editor.buildAndLoad($('#triggers-triggerList').empty());
        });
      }
      
      /**
       * @memberOf niagara.schedule.ui.pages.triggers
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);
        
        $('#triggers-triggerList').bind('editorchange', baja.throttle(function () {
          $('#triggers-remove').toggleClass(DISABLED_CLASS,
              !triggersEditor.getSelectedSlots().length);
        }, 100));
        
        $('#triggers-add').click(function () {
          dialogs.fieldEditor({
            title: mobileLex.get('schedule.addTrigger'),
            value: scheduleSnapshot.get('times'),
            key: 'triggers-add',
            ok: function (value, cb) {
              currentSchedule.setModified(true);
              cb.ok();
            }
          });
        });
        
        $('#triggers-remove').click(function () {
          triggersEditor.removeSelected(function () {
            currentSchedule.setModified(true);
            $('#triggers-triggerList').trigger('updatelayout');
            $('#triggers-remove').addClass(DISABLED_CLASS);
          });
        });
        
        $('#triggers-triggerList').delegate('#selectAll', 'change', function () {
          var checked = $(this).is(':checked');
          $('#triggers-triggerList .ui-li-static input[type=checkbox]')
            .attr('checked', checked)
            .checkboxradio('refresh');
        });
        
        util.mobile.preventNavbarHighlight($('#triggers-navbar'));
      }
      
      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.triggers
       */
      return {
        pagebeforeshow: pagebeforeshow,
        pagebeforecreate: pagebeforecreate,
        getCommands: getDefaultCommands
      };
    }()));
    
    pages.register('addSpecialEvent', (function addSpecialEvent() {
      var editedEvents,
          newScheduleEditor,
          buildNewScheduleEditorWorkflow,
          addNewSpecialEventWorkflow;
      
      /**
       * Sequential async workflow to add a new special event. First the
       * special event editor must be saved, then the new special event added
       * to the current list of special events. Finally it must retrieve
       * a display string from the station and update the slot's display
       * value (since toString on an AbstractSchedule is too complicated to
       * really do browserside).
       * 
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       * @field
       */
      addNewSpecialEventWorkflow = util.flow.sequential(
        function addNewSpecialEvent__step1__saveScheduleEditor(cx) {
          newScheduleEditor.saveValue(this);
        },
        function addNewSpecialEvent__step2__addEventToEventList(cx, savedSchedule) {
          cx.slotName = baja.SlotPath.escape(cx.name);
          
          if (scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
            cx.specialEvent = baja.$(DAILY_SCHEDULE_TYPE);
            cx.specialEvent.setDays(savedSchedule);
          } else {
            cx.specialEvent = savedSchedule;
          }
          
          editedEvents.add($.extend(this, {
            slot: cx.slotName,
            value: cx.specialEvent
          }));
        },
        function addNewSpecialEvent__step3__retrieveNewEventDisplayString(cx) {
          scheduleComponent.serverSideCall($.extend(this, {
            typeSpec: SCHEDULE_SSC_TYPE,
            methodName: 'getDailyScheduleDisplayString',
            value: cx.specialEvent
          }));
        },
        function addNewSpecialEvent__step4__setSlotDisplayAndReturn(cx, displayString) {
          //complex.getDisplay(slot) just returns slot.$display
          editedEvents.getSlot(cx.slotName).$display = displayString;
          util.slot.moveToTop(editedEvents, cx.slotName);
          currentSchedule.setModified(true);
          history.back();
          this.ok();
        }
      );
      
      /**
       * When creating a new DateSchedule, set the default values for 
       * day/month/year to match the current date.
       * 
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       */
      function setNewDateScheduleDefaults(div) {
        var date = new Date();
        
        div
          .find('.schedule-DayOfMonthSchedule')
          .find('select')
          .val(String(date.getDate()))
          .selectmenu('refresh');
        
        div
          .find('.schedule-MonthSchedule')
          .find('select')
          .val(String(date.getMonth()))
          .selectmenu('refresh');
        
        div
          .find('.schedule-YearSchedule')
          .find('select')
          .val(String(date.getFullYear()))
          .selectmenu('refresh');
      }
      
      /**
       * Sequential async workflow to display a field editor for a newly
       * created special event.
       * 
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       * @field
       */
      buildNewScheduleEditorWorkflow = util.flow.sequential(
        function buildNewScheduleEditorWorkflow__step1__makeFor(newSchedule) {
          fe.makeFor(newSchedule, this);
        },
        function buildNewScheduleEditorWorkflow__step2__buildAndLoad(newSchedule, editor) {
          editor.buildAndLoad($('#addSpecialEvent-editors').empty(), this);
        },
        function buildNewScheduleEditorWorkflow__step3__setDefaultsAndUpdateLayout(newSchedule, editor) {
          if (newSchedule.getType().is(DATE_SCHEDULE_TYPE)) {
            setNewDateScheduleDefaults(editor.$dom);
          }
          editor.$dom.trigger('updatelayout');
          this.ok(editor);
        }
      );
      
      
      /**
       * When creating a new special event, start with a default name - 
       * "Event" + (an integer as high as necessary to make it a unique slot
       * name).
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @private
       * @returns {String} a default name for a new special event
       */
      function generateDefaultName() {
        var i = 0,
            name = "Event";
        while (editedEvents.has(name)) {
          i++;
          name = "Event" + i;
        }
        return name;
      }
      
      /**
       * Add event handlers to redraw the field editor depending on what
       * type of new special event is selected, and save it once editing is
       * complete.
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       */
      function pagebeforecreate() {
        if (!scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
          $('#addSpecialEvent-typesList option[value="schedule:ScheduleReference"]')
            .remove();
        }
        
        $('#addSpecialEvent-typesList').change(function () {
          var typeSpec = $(this).val(),
              newSchedule = baja.$(typeSpec);
          buildNewScheduleEditorWorkflow.invoke(newSchedule, function (editor) {
            newScheduleEditor = editor;
          });
        });
        
        $('#addSpecialEvent-editors').bind('editorchange', function () {
          $('#addSpecialEvent-save').removeClass(DISABLED_CLASS);
        });
        
        mobileUtil.preventNavbarHighlight($('#addSpecialEvent-footer'));
        
        $('#addSpecialEvent-save').click(function () {
          var name = $('#addSpecialEvent-name').val();
          
          if (editedEvents.has(name)) {
            dialogs.ok({
              content: mobileLex.get({
                key: 'schedule.message.nameInUse',
                args: [ name ]
              })
            });
          } else {
            addNewSpecialEventWorkflow.invoke({ name: name });
          }
        });
      }
      
      /**
       * Loads, and instigates a page change to, the addSpecialEvent page.
       * Shows the default field editor and a default slot name.
       * @memberOf niagara.schedule.ui.pages.addSpecialEvent
       * @param {baja.Component} events the 
       * <code>schedule:CompositeSchedule</code> in the current schedule's 
       * <code>schedule/specialEvents</code> slot
       */
      function load(events) {
        editedEvents = events;
        $.mobile.changePage($('#addSpecialEvent'));
        $('#addSpecialEvent-typesList').trigger('change');
        $('#addSpecialEvent-name').val(generateDefaultName());
      }
      
      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.addSpecialEvent
       */
      return {
        load: load,
        pagebeforecreate: pagebeforecreate
      };
    }()));
    
    pages.register('editSpecialEvent', (function editSpecialEvent() {
      var scheduleEditor,
          editedSpecialEvent,
          dateRangeModified,
          selectedSectionIndex = 0,
          saveScheduleEditorWorkflow,
          saveBothEditorsWorkflow,
          debouncedCalendarUpdate;
      
      /**
       * Sequential async workflow to save the effective date range on an
       * edited special event. Will retrieve an updated toString from the
       * server for proper display back on the Special Events tab.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @field
       */
      saveScheduleEditorWorkflow = util.flow.sequential(
        function saveScheduleEditor__step1__saveEditorValue(cx) {
          scheduleEditor.saveValue(this);
        },
        function saveScheduleEditor__step2__retrieveUpdatedDisplayString(cx) {
          scheduleComponent.serverSideCall($.extend(this, {
            typeSpec: SCHEDULE_SSC_TYPE,
            methodName: 'getDailyScheduleDisplayString',
            value: editedSpecialEvent
          }));
        },
        function saveScheduleEditor__step3__setDisplayString(cx, displayString) {
          var slot = editedSpecialEvent.getPropertyInParent();
          slot.$display = displayString;
          currentSchedule.setModified(true);
          this.ok();
        }
      );
      
      /**
       * Parallel async workflow to save both the day editor and the effective
       * date range editor on the currently edited special event.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @field
       */
      saveBothEditorsWorkflow = util.flow.parallel(
        function saveBothEditors__saveDayEditor(cx) {
          saveDayEditor(this);
        },
        function saveBothEditors__saveScheduleEditorIfModified(cx) {
          if (scheduleEditor.isModified()) {
            saveScheduleEditorWorkflow.invoke({}, this);
          } else {
            this.ok();
          }
        }
      );
      
      /**
       * Updates the editSpecialEvent calendar to show the currently effective
       * date range (based on user input on the Active Dates tab).
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @param {niagara.fieldEditors.BaseFieldEditor} scheduleEditor the field
       * editor for the currently edited special event's date range
       * @param {Number} [changeInMonths] how many months forward/backward to
       * skip - this will only be passed if updating the calendar due to
       * clicking the +/- buttons to change months
       */
      function doCalendarUpdate(targetElement, newTime) {
        
        scheduleEditor.getSaveData({
          ok: function (saveData) {
            calendarUI.showEffectiveRangeCalendar(
                targetElement, 
                saveData,
                newTime);
            dateRangeModified = false;
          },
          fail: function (err) {
            baja.error("editSpecialEvent: failed to load schedule editor: " + err);
          }
        });
      }
      
      debouncedCalendarUpdate = util.debounce(doCalendarUpdate, 600);
      
      /**
       * Saves the editors for the current special event's active date range
       * schedule and day schedule.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function saveEditors(callbacks) {
        saveBothEditorsWorkflow.invoke({}, callbacks);
      }
      
      /**
       * Returns a click handler to be bound to a back button. If the currently
       * edited day is in a modified state, will pop up a dialog asking the user
       * whether or not to save changes before allowing the backward navigation
       * to take place.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function checkForChangesFunction(targetPage) {

        return function () {
          if ((dayEditor && dayEditor.isModified()) || scheduleEditor.isModified()) {
            dialogs.confirmAbandonChanges({
              yes: saveEditors,
              redirect: targetPage,
              viewName: editedSpecialEvent.getDisplayName()
            });
            return false;
          }
        };
      }
      
      
      /**
       * Enables/disables footer bar links depending on which tab is being
       * shown. Edit, Delete, Options, and Refresh will only be available
       * when viewing the Day Editor tab (edit/delete only if a schedule
       * block is selected), while the OK (save) button can be accessible from
       * any tab if the user has entered a change.
       * <p>
       * Called whenever the user switches tabs or makes an edit to a field
       * editor.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function updateLinks() {
        if (selectedSectionIndex === 2) {
          if (specialEventsReadonly) {
            $('#editSpecialEvent-refresh').removeClass(DISABLED_CLASS);
          } else {
            $('#editSpecialEvent-actions, #editSpecialEvent-refresh')
              .removeClass(DISABLED_CLASS);
            
            $('#editSpecialEvent-edit').removeClass(DISABLED_CLASS);
            $('#editSpecialEvent-delete')
              .toggleClass(DISABLED_CLASS, !dayEditor.getSelectedBlock());
          }
        } else {
          $('#editSpecialEvent-footer a[id!="editSpecialEvent-save"]')
            .addClass(DISABLED_CLASS);
        }
      }
      
      /**
       * Redraws the content of the special event editor tabs. The 
       * content div of the #editSpecialEvent page will be set to display
       * full screen, and the day editor drawn accordingly. If the date range
       * editor tab is taller than the day editor tab, then the content div
       * will be expanded to fit.
       * <p>
       * Called upon pageshow and window resize.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @see niagara.schedule.ui.initializeUI
       */
      function relayout() {
        mobileUtil.setContentHeight(
            $('#editSpecialEvent'), $('#editSpecialEvent-dayWrapper'));
        
        $('#editSpecialEvent-content').height(
            Math.max($('#editSpecialEvent-dayWrapper').height(),
                     $('#editSpecialEvent-editors > .editor').height()));
        relayoutDayDiv($('#editSpecialEvent-day > div.schedule-DaySchedule'));
        updateLinks();
      }
      
      /**
       * Blanks out the day editor and redraws to fill available screen space.
       * Will be called when explicitly refreshing the day editor (by clicking
       * Refresh), when the page is shown, or when the window is resized.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function rebuild() {
        if (dayEditor) {
          var dayDiv = $('#editSpecialEvent-day').empty();
          scheduleUtil.appendTimeLabels(dayDiv);
          dayEditor.buildAndLoad(dayDiv, function () {
            $('#editSpecialEvent-edit').addClass(DISABLED_CLASS);
            $('#editSpecialEvent-delete').addClass(DISABLED_CLASS);
            relayout();
          });
        }
      }
      
      /**
       * Repaints the day editor to ensure that any user entered changes
       * (from clicking the create/delete/edit footer buttons) are reflected
       * on the screen.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function refresh() {
        if (dayEditor) {
          dayEditor.refreshWidgets();
        }
        setCreateMode($('#editSpecialEvent-footer'), true);
        relayout();
      }
      
      /**
       * Shows available actions to perform on the currently edited special
       * event's day schedule. Copy/paste and Clear Week functions are always
       * unavailable for a special event.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       */
      function loadBlockActions() {
        var selectedBlock = dayEditor.getSelectedBlock(),
            hasBlocks = !!dayEditor.scheduleBlocks.length,
            hasSelected = !!selectedBlock;
        
        showBlockActions({
          'delete': hasSelected,
          'paste_day': false,
          'all_day': true,
          'apply_weekdays': false,
          'copy_day': false,
          'clear_day': hasBlocks,
          'clear_week': false
        });
      }
      
      /**
       * Performs the scrolling between calendar/date range/day editor tabs.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @private
       * @param {Number} index the tab index. 0 = Calendar Preview, 1 = 
       * Active Dates, 2 = Day Editor.
       */
      function scrollSections(index) {
        var contentDiv = $('#editSpecialEvent-content'),
            calendarsDiv,
            absTime,
            contentWidth = $(document.documentElement).outerWidth();

        //viewing calendar 
        if (index === 0 && dateRangeModified) {
          calendarsDiv = $('#editSpecialEvent-calendars');
          absTime = dateboxUtil.getAbsTime(calendarsDiv);
          doCalendarUpdate(calendarsDiv, absTime);
        }
        
        contentDiv.children().each(function () {
          var $this = $(this),
              elementIndex = $this.index();
          $this.animate({
            left: contentWidth * (elementIndex - index),
            right: contentWidth * (index - elementIndex)
          }, 'fast');
        });
        
        selectedSectionIndex = index;
        
        updateLinks();
      }
      
      /**
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagecreate(obj) {
        var backButton = obj.page
          .children(JQM_HEADER_SELECTOR)
          .find('a.profileHeaderBack');
        backButton.bind('click', checkForChangesFunction('#specialEvents'));
      }
      
      /**
       * Arm handlers for the save button, the <code>DayEditor</code>, and
       * for updating the calendar preview whenever the schedule field editor
       * is changed by the user or the calendar's display month is changed.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagebeforecreate() {  
        if (!scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
          $('#editSpecialEvent-dayEditorLink').parent().remove();
          $('#editSpecialEvent-edit').parent().remove();
          $('#editSpecialEvent-delete').parent().remove();
          $('#editSpecialEvent-actions').parent().remove();
          $('#editSpecialEvent-refresh').parent().remove();
        }
        
        $('#editSpecialEvent-day').bind('selectionchange', function () {
          var selected = !!dayEditor.getSelectedBlock();
          setCreateMode($('#editSpecialEvent-footer'), !selected);
        });
        
        $('#editSpecialEvent-day').bind('dblclick', function () {
          editBlock(dayEditor.getSelectedBlock());
        });
        
        $('#editSpecialEvent-navbar').delegate('a', 'click', function () {
          
          switch ($(this).attr('id')) {
          case 'editSpecialEvent-calendarLink':
            scrollSections(0);
            break;
          case 'editSpecialEvent-activeDatesLink':
            scrollSections(1);
            break;
          case 'editSpecialEvent-dayEditorLink':
            scrollSections(2);
            break;
          }
        });
        
        mobileUtil.preventNavbarHighlight($('#editSpecialEvent-footer'));
        
        $('#editSpecialEvent-footer').delegate('a', 'click', function () {
          var $this = $(this);
          switch ($this.attr('id')) {
          case 'editSpecialEvent-save':
            saveEditors(function () {
              history.back();
            });
            break;
          case 'editSpecialEvent-refresh':
            if (dayEditor.isModified()) {
              dialogs.confirmAbandonChanges({
                yes: saveDayEditor,
                callbacks: function () {
                  var calendarDiv = $('#editSpecialEvent-calendars'),
                      absTime = dateboxUtil.getAbsTime(calendarDiv);
                  rebuild();
                  doCalendarUpdate(calendarDiv, absTime);
                },
                viewName: mobileLex.get('schedule.dayEditor')
              });
            } else {
              rebuild();
            }
            
            break;
          case 'editSpecialEvent-actions':
            loadBlockActions();
            break;
          case 'editSpecialEvent-edit':
            createOrEdit(refresh);
            break;
          case 'editSpecialEvent-delete':
            dayEditor.removeBlock(dayEditor.getSelectedBlock());
            refresh();
            break;
          }
        });
        
        $('#editSpecialEvent-content').bind('editorchange', function () {
          dateRangeModified = true;
        });
        
        $('#editSpecialEvent-calendars').delegate('input:jqmData(role="datebox")', 'datebox', function (e, passed) {
          if (passed.method === 'offset') { //+ or - button clicked
            var calendarDiv = $('#editSpecialEvent-calendars'),
                absTime = dateboxUtil.getAbsTime(calendarDiv),
                newTime = scheduleUtil.addMonths(absTime, passed.amount);
            
            debouncedCalendarUpdate(calendarDiv, newTime);
          }
        });
      }
      
      /**
       * Loads, and instigates page change to, the editSpecialEvent page.
       * Instantiates a new <code>niagara.fieldEditors.schedule.DayEditor</code>
       * for the edited schedule's <code>day</code> property.
       * 
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       * @param {baja.Component} specialEvent the 
       * <code>schedule:DailySchedule</code> representing the edited special 
       * event
       */
      function load(specialEvent) {
        editedSpecialEvent = specialEvent;
        editedDaySchedule = editedSpecialEvent.get('day'); 

        $('#editSpecialEvent-eventName').text(
            baja.SlotPath.unescape(specialEvent.getName()));

        resetDayEditor({ 
          value: editedDaySchedule, 
          label: baja.SlotPath.unescape(specialEvent.getName()),
          readonly: specialEventsReadonly
        }, function () {
          rebuild();
          $.mobile.changePage('#editSpecialEvent');
          calendarUI.loadScheduleEditor(
            $('#editSpecialEvent'), 
            editedSpecialEvent.get('days') || editedSpecialEvent, 
            {
              ok: function (editor) {
                var calendarDiv = $('#editSpecialEvent-calendars'),
                    absTime = dateboxUtil.getAbsTime(calendarDiv);
                
                editor.refreshWidgets();
                scheduleEditor = editor;
                doCalendarUpdate(calendarDiv, absTime);
              },
              fail: function (err) {
                baja.error("editSpecialEvent.pagebeforeshow: failed to load " +
                    "schedule editor: " + err);
              }
            }, specialEventsReadonly);
        });
      }
      
      /**
       * Loads the calendar display preview and special event schedule
       * field editor.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagebeforeshow(obj) {
        if (!editedSpecialEvent) {
          $.mobile.changePage('#specialEvents');
          return;
        }
        
        refresh();
        setCreateMode($('#editSpecialEvent-footer'), true);
      }
      
      /**
       * Loads the <code>niagara.fieldEditors.schedule.DayEditor</code> for
       * the edited special event's <code>day</code> property. (Done in 
       * <code>pagelayout</code> because to draw the editor properly we must 
       * know the div's height - which can't be obtained in 
       * <code>pagebeforeshow</code>.)
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagelayout() {
        if (!dayEditor) {
          return;
        }
        
        document.title = $('#editSpecialEvent-header > h1').text();

        relayout();
        scrollSections(selectedSectionIndex || 0);
      }
      
      /**
       * If the day editor is modified, prompts for changes before navigating
       * away.
       * @memberOf niagara.schedule.ui.pages.editSpecialEvent
       */
      function pagebeforechange(obj) {
        if ((dayEditor && dayEditor.isModified()) || scheduleEditor.isModified()) {
          dialogs.confirmAbandonChanges({
            yes: function (cb) {
              saveEditors(cb);
            },
            no: function (cb) {
              resetDayEditor(editedDaySchedule, cb);
            },
            redirect: obj.nextPage,
            viewName: scheduleLex.get('summary.specialEvent')
          });
          obj.event.preventDefault();
        }
      }
      
      $(window).resize(baja.throttle(function () {
        scrollSections(selectedSectionIndex || 0);
      }));
      
      
      /**
       * @namespace
       * @private
       * @name niagara.schedule.ui.pages.editSpecialEvent
       */
      return {
        pagebeforechange: pagebeforechange,
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        pagecreate: pagecreate,
        pagelayout: pagelayout,
        load: load,
        rebuild: rebuild,
        relayout: relayout
      };
    }()));
    
    pages.register('summary', (function summary() {
      var debouncedCalendarUpdate;
      
      /**
       * Parses the text value of a datebox <code>&lt;input&gt;</code> into a
       * <code>baja.AbsTime</code>.
       * @memberOf niagara.schedule.ui.pages.summary
       * @private
       * @param {String} str the datebox text input value
       * @returns {baja.AbsTime} the parsed <code>AbsTime</code> value
       */
      function dateboxToAbsTime(str) {
        var year = str.substring(0, 4),
            month = str.substring(5, 7),
            day = str.substring(8, 10),
            date = baja.Date.make({ year: Number(year), month: Number(month) - 1, day: Number(day) }),
            //bump the time forward in hopes of not getting results from yesterday
            //TODO: TIME ZONES AHOY
            time = baja.Time.make(timeUtil.MILLIS_IN_HOUR * 9);
        return baja.AbsTime.make({ date: date, time: time });
      }
      
      /**
       * @memberOf niagara.schedule.ui.pages.summary
       * @private
       */
      function doCalendarUpdate(targetElement, newTime) {
        calendarUI.showEffectiveRangeCalendar(
            targetElement, 
            currentSchedule.value, 
            newTime);
      }
      
      debouncedCalendarUpdate = util.debounce(doCalendarUpdate, 600);
      
      /**
       * Arms click handlers on the effective range calendar to show a summary
       * for the clicked day, and to update the effective range display when
       * a new month is selected.
       * @memberOf niagara.schedule.ui.pages.summary
       */
      function pagebeforecreate(obj) {
        appendNavbar(obj.page);
        
        $('#summary-calendars').delegate('input:jqmData(role="datebox")', 'datebox', function (e, passed) {
          if (passed.method === 'offset') { //+ or - button clicked
            var calendarDiv = $('#summary-calendars'),
                absTime = dateboxUtil.getAbsTime(calendarDiv),
                newTime = scheduleUtil.addMonths(absTime, passed.amount);
            
            debouncedCalendarUpdate(calendarDiv, newTime);
          } else if (passed.method === 'set' && scheduleComponent.getType().is(WEEKLY_SCHEDULE_TYPE)) {
            calendarUI.showDaySummary(
                $('#summary-summaryList'),
                currentSchedule.value,
                dateboxToAbsTime(passed.value));
          }
        });
      }
      
      /**
       * Shows the effective range calendar and day summary.
       * @memberOf niagara.schedule.ui.pages.summary
       */
      function pagebeforeshow() {
        calendarUI.showEffectiveRangeCalendar(
            $('#summary-calendars'), 
            currentSchedule.value, 
            baja.AbsTime.now());
        
        $('#summary-summary').addClass(BTN_ACTIVE_CLASS);
        $('#summary-title').text(scheduleComponent.getDisplayName());
      }
      
      /**
       * @namespace
       * @name niagara.schedule.ui.pages.summary
       * @private
       */
      return {
        pagebeforecreate: pagebeforecreate,
        pagebeforeshow: pagebeforeshow,
        getCommands: getDefaultCommands
      };
    }()));
  }
  
  /**
   * Prevent blowups if we hit refresh on a sub-page and thus no schedule
   * currently loaded. <code>editDay</code> redirects to <code>main</code>, 
   * <code>addSpecialEvent</code> and <code>editSpecialEvent</code> redirect to
   * <code>specialEvents</code>, and all other pages just preload the current 
   * schedule before displaying.
   * <p>
   * This function is bound to <code>pagebeforechange</code> in
   * <code>initializeUI</code>.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {jQuery.Event} event the <code>pagebeforechange</code> event
   * @param {Object} data the JQM data passed along with the event 
   */
  function preloadOrRedirect(event, data) {
    if (!currentSchedule) {
      var id;
      
      if (typeof data.toPage === 'string') {
        id = $.mobile.path.parseUrl(data.toPage).hash.replace('#', '') || 'main';
      } else {
        id = data.toPage.attr('id');
      }
      
      event.preventDefault();
      event.stopImmediatePropagation();
      
      baja.started(function () {
        reretrieveSchedule(function () {
          data.options.changeHash = true;
          if (id === 'editDay') {
            $.mobile.changePage('#main', data.options);
          } else if (id === 'addSpecialEvent' || id === 'editSpecialEvent') {
            $.mobile.changePage('#specialEvents', data.options);
          } else {
            data.options.changeHash = false;
            $.mobile.changePage('#' + id, data.options);
          }
        });
      });

    } 
  }
  
  /**
   * Registers jQuery custom selectors and JQM page event listeners. Runs
   * automatically when <code>schedule.ui.js</code> runs.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   */
  function initializeUI() {
    util.customSelectors.installAll($);
    registerPages();

    $(window).bind('resize', baja.throttle(function () {
      var activePage = $.mobile.activePage;
      if (activePage) {
        activePage.trigger('pagelayout');
      }
    }, 500));
    
    window.onbeforeunload = function () {
      if (currentSchedule.isModified()) {
        return mobileLex.get({
          key: 'message.viewModified',
          args: [scheduleLex.get('scheduler.weeklySchedule')]
        });
      }
    };
    
    // When the DOM is ready, register the command handler directly on the commands button...
    $(function () {
      $("#commandsButtonMain, #commandsButtonEditDay, #commandsButtonSpecialEv," +
        "#commandsButtonAddSpecEv, #commandsButtonEditSpecEv, #commandsButtonProperties, #commandsButtonSummary").click(commands.showCommandsHandler);
    });
    
    util.mobile.prependEventHandler($(document), 'pagebeforechange', preloadOrRedirect);
  }
  
  /**
   * Returns the currently viewed Schedule object.
   * 
   * @memberOf niagara.schedule.ui
   * @returns {niagara.schedule.Schedule} the currently viewed Schedule
   */
  function getCurrentSchedule() {
    return currentSchedule;
  }
  

  initializeUI();
  
  /**
   * Functions relating to page navigation and interaction within
   * the mobile scheduler app.
   * @namespace
   * @name niagara.schedule.ui
   */
  util.api('niagara.schedule.ui', {
    'public': {
      getCurrentSchedule: getCurrentSchedule,
      reretrieveSchedule: reretrieveSchedule,
      save: save,
      updateScheduleBlockDiv: updateScheduleBlockDiv
    },
    'private': {
      appendNavbar: appendNavbar,
      editBlock: editBlock,
      initializeUI: initializeUI,
      relayoutDayDiv: relayoutDayDiv,
      relayoutLabelsDiv: relayoutLabelsDiv,
      registerPages: registerPages,
      resetDayEditor: resetDayEditor,
      showBlockActions: showBlockActions,
      showTimeScheduleDialog: showTimeScheduleDialog,
      redrawSchedule: redrawSchedule
    }
  });
}());