/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Utilities for managing color gradients in the schedule app.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, plusplus: true, vars: true */
/*globals niagara, baja, $ */

(function statusGradients() {
  
   "use strict";
   
   var util = niagara.util,
       colorUtil = util.color,
       gradientUtil = util.gradient,
       
       applyLinearGradient = gradientUtil.applyLinearGradientBackground,
       createLinearGradient = gradientUtil.createLinearGradientCss,
       parseRgba = colorUtil.parseRgba,
       hsvToRgb = colorUtil.hsvToRgb,
       
       stopCache = {},
       
       VERTICAL_ANGLE = 270,
       LUMA_THRESHOLD = 150,
       
       BOX_SHADOW_PREFS = [ '-webkit-', '-moz-', '-ms-', '-o-', '' ],
       
       TRUE_BACKGROUND_COLOR = parseRgba("#f065cb65"),
       
       FALSE_BACKGROUND_COLOR = parseRgba("#f0e17171"),
       
       NULL_BACKGROUND_COLOR = parseRgba("#f8cccccc"),
       
       DEFAULT_BACKGROUND_COLOR = parseRgba("#f88fbc8f"),
       
       DEFAULT_ENUM_COLORS = [
         "#f08c0000", //dark red
         "#f0387038", //dark green
         "#f0703870", //dark purple
         "#f025255a", //dark blue
         "#f01f6f70", //dark indigo
         "#f0704308", //dark orange
         "#f0606060", //gray
         
         "#f0e17171", //red
         "#f065cb65", //green
         "#f0cb7fcb", //purple
         "#f06c6ccb", //blue
         "#f054b7b8", //indigo
         "#f0c88435", //orange
         "#f0e1e171"  //yellow
       ];
   
   
   
   
////////////////////////////////////////////////////////////////
// Color parsing / gradients
////////////////////////////////////////////////////////////////
   
   /**
    * Get the background color of the schedule display, to use for alpha
    * blending calculations against the schedule block.
    * 
    * @private
    * @inner
    * @returns {niagara.util.color.Color}
    */
   function getScheduleBackgroundColor() {
     return parseRgba($('#main').css('background-color'));
   }
   
   /**
    * Transforms the given color into a pastelized gradient.
    * 
    * @private
    * @inner
    * @param {String} color
    * @returns {Array} an array of stops to pass to niagara.util.gradient
    */
   function toGradientStops(color) {
     if (stopCache[color.css]) {
       return stopCache[color.css];
     }
     
     var h = color.h,
         s = color.s,
         v = color.v,
         a = color.alpha,
         rgba1, rgba2, rgba3, rgba4,
         stops;
     
     //light/pastel at the top...
     rgba1 = hsvToRgb(h, s * 0.50, v * 1.35);
     rgba2 = hsvToRgb(h, s * 0.60, v * 1.25);
     
     //...darker/more vibrant at bottom
     rgba3 = hsvToRgb(h, s * 0.90, v * 1.05);
     rgba4 = hsvToRgb(h, s,        v);

     //keep alpha
     if (typeof a === 'number') {
       rgba1 = parseRgba([ rgba1.red, rgba1.green, rgba1.blue, a ]);
       rgba2 = parseRgba([ rgba2.red, rgba2.green, rgba2.blue, a ]);
       rgba3 = parseRgba([ rgba3.red, rgba3.green, rgba3.blue, a ]);
       rgba4 = parseRgba([ rgba4.red, rgba4.green, rgba4.blue, a ]);
     }
     
     stops = stopCache[color.css] = [
       [   "0%", rgba1 ],
       [  "30%", rgba2 ],
       [  "90%", rgba3 ],
       [  "97%", rgba4 ],
       [ "100%", rgba3 ]
     ];
     
     return stops;
   }
   
   /**
    * Calculates the luma value of the color. Does alpha blending to take into
    * account the background color and the fact that it might be partially
    * transparent.
    * 
    * @private
    * @inner
    * @param {niagara.util.color.Color} color
    * @return {Number}
    */
   function luma(color) {
     if (color.alpha < 255) {
       color = colorUtil.alphaBlend(color, getScheduleBackgroundColor());
     }
     
     return colorUtil.luma(color.red, color.green, color.blue);
   }
   
   /**
    * Get either black or white depending on which contrasts best with the
    * color's luma value.
    * 
    * @private
    * @inner
    * @param {niagara.util.color.Color} color
    * @returns {Object} a css object with <code>color</code> and
    * <code>text-shadow</code> properties
    */
   function getContrastingTextCss(color) {
     if (luma(color) > LUMA_THRESHOLD) {
       return { 
         color: 'black', 
         'text-shadow': '0 1px 1px rgba(255, 255, 255, 0.7)' 
       };
     }
      
     return { 
       color: 'white', 
       'text-shadow': '0 1px 1px rgba(0, 0, 0, 0.7)' 
     };
   }

   /**
    * Calculates the gradient color stops and contrasting text color and
    * applies the CSS to the given div.
    * 
    * @private
    * @inner
    * @param {jQuery} div
    * @param {niagara.util.color.Color} blockColor
    */
   function applyGradient(div, blockColor) {
     var gradientStops = toGradientStops(blockColor),
         foregroundCss = getContrastingTextCss(gradientStops[0][1]);

     div.css(foregroundCss);

     if (div.parents('.schedule').length) {
       //main weekly schedule view = fancy and shiny
       applyLinearGradient(div, VERTICAL_ANGLE, gradientStops);
     } else {
       //editable day sliders = background color only for speed
       div.css('background-color', 
           createLinearGradient(VERTICAL_ANGLE, gradientStops).background);       
     }
   }
   
   /**
    * Applies box shadow to schedule block div. This is a white inset box shadow
    * 50% of the block's height to give the shiny effect.
    * 
    * @private
    * @inner
    * @param {jQuery} blockDiv
    */
   function applyBoxShadow(blockDiv) {
     //only apply to main schedule div. too slow on interactive sliders
     if (!blockDiv.parents('.schedule').length) {
       return;
     }
     
     var height = blockDiv.height(),
         highlightHeight = height / 2,
         css = '0 2px 6px rgba(0,0,0,0.3), ' +
           'inset 0 0px 0px 1px rgba(255,255,255,0.1)',
         highlightAlpha,
         prefs = BOX_SHADOW_PREFS,
         i;
     
     if (highlightHeight < 160) {
       //begin fading in the glass highlight effect at 160px, for a max of
       //0.06 alpha at 80px and less.
       highlightAlpha = Math.min(160 - highlightHeight, 80)  / 80 * 0.07;
       css += ', inset 0 ' + highlightHeight + 'px ' + 
         'rgba(255,255,255,' + highlightAlpha + ')';
     }
     
     for (i = 0; i < prefs.length; i++) {
       blockDiv.css(prefs[i] + 'box-shadow', css);
     }
   }
   
   
////////////////////////////////////////////////////////////////
// WeeklySchedule
////////////////////////////////////////////////////////////////
   
   /**
    * Given a schedule block value, walk up the parent tree until you find
    * a WeeklySchedule.
    * 
    * @private
    * @inner
    * @param {baja.Complex} statusValue <code>baja:StatusValue</code>
    * @returns {baja.Component} <code>schedule:WeeklySchedule</code> or undefined
    */
   function getWeeklySchedule(statusValue) {
     return niagara.schedule.ui.getCurrentSchedule().value;
   }
   
   /**
    * Gets an enum schedule color configured in the lexicon, if any. Configured
    * as lexicon entries <code>EnumSchedule.colors.0</code> through however
    * many.
    * 
    * @private
    * @inner
    * @param {baja.Enum} en
    * @returns {String} color string from lexicon, or undefined
    */
   function getLexiconColor(key) {
     var value = baja.lex('schedule').get(key);
     
     if (value) {
       try {
         return parseRgba(value);
       } catch (e) {
         baja.error('Invalid color in lexicon: ' + key + ' = ' + value);
       }
     }
   }
   
   /**
    * Retrieve the color configured directly on the schedule (via a 
    * <code>colors</code> facet) for the given ordinal.
    * 
    * @private
    * @inner
    * @param {baja.Component} weeklySchedule a <code>schedule:WeeklySchedule</code>
    * @param {Number} ordinal For an EnumSchedule, use the ordinal from the
    * DynamicEnum value. For a BooleanSchedule, use 0 = false, 1 = true.
    * @returns {niagara.util.color.Color}
    */
   function getConfiguredColor(weeklySchedule, ordinal) {
     var colors;
     
     try {
       if (weeklySchedule) {
         colors = weeklySchedule.get('facets').get('colors');
       }
       
       if (colors && colors.getType().is('baja:EnumRange')) {
         var tag = colors.get(ordinal).getTag();
         return parseRgba(baja.SlotPath.unescape(tag));
       }
     } catch (e) {
       return null;
     }

   }
   
////////////////////////////////////////////////////////////////
// BooleanSchedule
////////////////////////////////////////////////////////////////
   
   function getLexiconColorFromBoolean(b) {
     return getLexiconColor('BooleanSchedule.colors.' + (b ? 1 : 0));
   }
   
   function getDefaultColorFromBoolean(b) {
     return b ? TRUE_BACKGROUND_COLOR : FALSE_BACKGROUND_COLOR;
   }
   
   /**
    * Red for false, green for true.
    * 
    * @private
    * @inner
    * @param {baja.Value} value <code>baja:StatusBoolean</code>
    * @returns {Object} color
    */
   function getColorFromStatusBoolean(statusBoolean) {
     var schedule = getWeeklySchedule(statusBoolean),
         b = statusBoolean.getValue(),
         color;
     
     color = getConfiguredColor(schedule, b ? 1 : 0);
     
     if (!color) {
       color = getLexiconColorFromBoolean(b);
     }
     
     if (!color) {
       color = getDefaultColorFromBoolean(b);
     }
     
     return color;
   }
   
////////////////////////////////////////////////////////////////
// EnumSchedule
////////////////////////////////////////////////////////////////
   
   /**
    * Get the enum's ordinal's index within all ordinals in the enum's range.
    * Protects against gaps in ordinal sequence when picking a color.
    * 
    * @private
    * @inner
    * @param {baja.Enum} en
    * @returns {Number}
    */
   function getOrdinalIndex(en) {
     if (en === baja.DynamicEnum.DEFAULT) {
       //default value was not encoded, so no range. however, must always be 0.
       return 0;
     }
     
     var ordinal = en.getOrdinal(),
         ordinals = en.getRange().getOrdinals(),
         index = util.indexOf(ordinals, ordinal);
     
     if (index === -1) {
       return ordinal;
     }

     return index;
   }
   
   /**
    * Get the default color for an enum schedule block. Pulled from 
    * DEFAULT_ENUM_COLORS using the enum's ordinal. For ordinals 0 to
    * 6, returns a vibrant, gem-like color. For 7 to 13, returns a lighter
    * pastel color to contrast with first 7. Loops back around at 14.
    * 
    * @private
    * @inner
    * @param {baja.Enum} en
    * @returns {niagara.util.color.Color} color
    */
   function getDefaultColorFromEnum(en) {
     var colors = DEFAULT_ENUM_COLORS,
         len = colors.length,
         index = getOrdinalIndex(en),
         ord = index % (len * 2),
         color = colors[ord % len];
     
     return parseRgba(color);
   }
   
   /**
    * Gets an enum schedule color configured in the lexicon, if any. Configured
    * as lexicon entries <code>EnumSchedule.colors.0</code> through however
    * many.
    * 
    * @private
    * @inner
    * @param {baja.Enum} en
    * @returns {String} color string from lexicon, or undefined
    */
   function getLexiconColorFromEnum(en) {
     var index = getOrdinalIndex(en);
     
     if (index >= 0) {
       return getLexiconColor('EnumSchedule.colors.' + index);
     }
   }
   
   /**
    * Calculate the block color for the given StatusEnum. If colors are
    * configured directly on the schedule object, uses those. If not, checks
    * the lexicon for configured colors. If not in the lexicon, pulls
    * from the default set of colors.
    * 
    * @private
    * @inner
    * @param {baja:Value} value <code>baja:StatusEnum</code>
    * @returns {Array} hsva
    */
   function getColorFromStatusEnum(statusEnum) {
     var schedule = getWeeklySchedule(statusEnum),
         en = statusEnum.getValue(),
         range = en.getRange(),
         color;

     /*
      * If you manually type all your ordinals, THEN choose a range, the
      * range does not get saved on your schedule blocks, so we must pull it
      * from the schedule instead.
      */
     if (range === baja.EnumRange.DEFAULT) {
       range = schedule.get('facets').get('range');
       if (range && range !== baja.EnumRange.DEFAULT) {
         en = baja.DynamicEnum.make({
           ordinal: en.getOrdinal(),
           range: range
         });
       }
     }

     color = getConfiguredColor(schedule, en.getOrdinal());
     
     if (!color) {
       color = getLexiconColorFromEnum(en);
     }
     
     if (!color) {
       color = getDefaultColorFromEnum(en);
     }
     
     return color;
   }
   
   
////////////////////////////////////////////////////////////////
// ScheduleColors
////////////////////////////////////////////////////////////////
   
   /**
    * Get the appropriate null color. Check for 
    * <code>EnumSchedule.colors.null</code> or 
    * <code>BooleanSchedule.colors.null</code>, then fall back to gray.
    * 
    * @private
    * @inner
    * @param {baja.Complex} statusValue <code>baja:StatusValue</code>
    * @return {niagara.util.color.Color}
    */
   function getNullColor(statusValue) {
     var color,
         type = statusValue.getType();
     
     if (type.is('baja:StatusEnum')) {
       color = getLexiconColor('EnumSchedule.colors.null');
     } else if (type.is('baja:StatusBoolean')) {
       color = getLexiconColor('BooleanSchedule.colors.null');
     }
     
     if (!color) {
       color = NULL_BACKGROUND_COLOR;
     }
     
     return color;
   }

   /**
    * Default coloring, used for StringSchedule and NumericSchedule blocks.
    * 
    * @private
    * @inner
    * @param {jQuery} div
    */
   function applyDefault(div) {
     applyGradient(div, DEFAULT_BACKGROUND_COLOR);
     div.css(getContrastingTextCss(DEFAULT_BACKGROUND_COLOR));
     applyBoxShadow(div);
   }
   
   /**
    * Calculates color gradients for a mobile schedule block. This could be
    * calculated and reused for multiple blocks that share the same value.
    * 
    * @class
    * @name niagara.schedule.scheduleGradients.ScheduleColors
    * @param {baja.Complex} statusValue a <code>baja:StatusValue</code> from
    * a schedule block
    */
   function ScheduleColors(statusValue) {
     var that = this,
         backgroundColor,
         foregroundCss,
         apply,
         type = statusValue && statusValue.getType();

     if (!type || !type.is('baja:StatusValue')) {
       
       apply = applyDefault;
       
     } else if (statusValue.getStatus() === baja.Status.nullStatus) {
       
       backgroundColor = getNullColor(statusValue);
       foregroundCss = getContrastingTextCss(backgroundColor);
       apply = function (div) {
         div.css({
           'background-color': backgroundColor.css
         });
         div.css(foregroundCss);
       };
       
     } else if (type.is('baja:StatusEnum')) {
       
       backgroundColor = getColorFromStatusEnum(statusValue);
       apply = function (div) {
         applyGradient(div, backgroundColor);
         applyBoxShadow(div);
       };
       
     } else if (type.is('baja:StatusBoolean')) {
       
       backgroundColor = getColorFromStatusBoolean(statusValue);
       apply = function (div) {
         applyGradient(div, backgroundColor);
         applyBoxShadow(div);
       };
       
     } else {
       
       apply = applyDefault;
       
     }
     
     /**
      * Applies the calculated colors to a schedule block div.
      * 
      * @name niagara.schedule.scheduleGradients.ScheduleColors#apply
      * @function
      * @param {jQuery} div
      */
     that.apply = function (div) {
       if (!div.data('$hasGradient')) {
         apply(div);
         div.data('$hasGradient', true);
       }
     };
   }
   
   /**
    * @namespace niagara.schedule.scheduleGradients
    */
   util.api('niagara.schedule.scheduleGradients', {
     ScheduleColors: ScheduleColors
   });
}());