/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to the field editor for trigger times on
 * a <code>BTriggerSchedule</code>'s <code>times</code> slot (a
 * <code>BDaySchedule</code>).
 * 
 * @author Logan Byam
 * @version 0.0.1
 */
/*jslint white: true, sloppy: true */
/*global niagara, $, baja */

(function () {

      //imports
  var util = niagara.util,
      fe = niagara.fieldEditors,
      dialogs = util.mobile.dialogs,
      MobileFieldEditor = fe.mobile.MobileFieldEditor,
      
      //local vars
      mobileLex = util.lazyLex('mobile'),
      scheduleLex = util.lazyLex('schedule'),
      triggersAddEditorsWorkflow,
      triggersSaveEditorsWorkflow,
      checkboxHtml,
      triggerAddHtml,
      
      //constants
      TIME_SCHEDULE_TYPE = 'schedule:TimeSchedule',
      DAY_SCHEDULE_TYPE = 'schedule:DaySchedule',
      
      //exports
      TriggersListEditor,
      TriggersAddEditor;

  
  checkboxHtml = 
    '<div class="checkboxContainer" data-role="fieldcontain">' +
      '<fieldset data-role="controlgroup" data-type="horizontal">' +
        '<input type="checkbox" id="{checkboxId}"/>' +
        '<label for="{checkboxId}"><span class="ui-btn-icon-notext"><span class="ui-icon"></span></span></label>' +
      '</fieldset>' +
    '</div>';
  
  triggerAddHtml =
    '<div>' +
      '<label class="rangeStartLabel">{triggerTime}:</label>' +
      '<div class="triggerStartTime"></div>' +
    '</div>' +
    '<div>' +
      '<div class="triggerRange"></div>' +
    '</div>' +
    '<div class="triggerRangeEditors hidden">' +
      '<div>' +
        '<label class="rangeEndLabel">{rangeEnd}:</label>' +
        '<div class="triggerEndTime"></div>' +
      '</div>' +
      '<div>' +
        '<label class="rangeIntervalLabel">{rangeInterval}:</label>' +
        '<div class="triggerInterval"></div>' +
      '</div>' +
      '<div>' +
        '<label class="triggersToCreate"/>' +
      '</div>' +
    '</div>';
  
  /**
   * Loads individual field editors for the <code>TriggerAddEditor</code>'s
   * start, end, and interval times.
   * 
   * @name niagara.fieldEditors.schedule.TriggerAddEditor.triggerAddEditorsWorkflow
   * @private
   * @inner
   * @function
   * @type niagara.util.flow.SequentialWorkflow
   */
  triggersAddEditorsWorkflow = util.flow.sequential(
      
    function triggerAdd__step1__makeStartEditor(editor) {
      fe.makeFor(baja.Time.DEFAULT, this);
    },
    
    function triggerAdd__step2__loadStartEditor(editor, startEditor) {
      editor.startEditor = startEditor;
      startEditor.buildAndLoad($('.triggerStartTime', editor.$dom), this);
    },
    
    function triggerAdd__step3__makeIsRangeEditor(editor) {
      fe.makeFor({ 
        value: false, 
        label: scheduleLex.get('trigger.range') + '?', 
        key: 'checkbox' 
      }, this);
    },
    
    function triggerAdd__step4__loadIsRangeEditor(editor, rangeEditor) {
      editor.rangeEditor = rangeEditor;
      rangeEditor.buildAndLoad($('.triggerRange', editor.$dom), this);
    },
    
    function triggerAdd__step5__makeEndEditor(editor) {
      //11:59 PM
      fe.makeFor(baja.Time.make(baja.RelTime.MILLIS_IN_DAY - 1), this);
    },
    
    function triggerAdd__step6__loadEndEditor(editor, endEditor) {
      editor.endEditor = endEditor;
      endEditor.buildAndLoad($('.triggerEndTime', editor.$dom), this);
    },
    
    function triggerAdd__step7__makeIntervalEditor(editor) {
      fe.makeFor({
        value: baja.RelTime.make(baja.RelTime.MILLIS_IN_HOUR),
        durationOrder: ['h', 'i'] //don't show seconds/days
      }, this);
    },
    
    function triggerAdd__step8__loadIntervalEditor(editor, intervalEditor) {
      editor.intervalEditor = intervalEditor;
      intervalEditor.buildAndLoad($('.triggerInterval', editor.$dom), this);
    }
  );
  
  /**
   * Saves the individual field editors for start/range/end/interval when 
   * adding a new trigger.
   * 
   * @name niagara.fieldEditors.schedule.TriggerAddEditor.triggersSaveEditorsWorkflow
   * @private
   * @inner
   * @function
   * @type niagara.util.flow.SequentialWorkflow
   * @returns {Object} an object with <code>start</code>, <code>isRange</code>,
   * <code>end</code>, and <code>interval</code> fields (to be passed to the
   * workflow's ok callback)
   */
  triggersSaveEditorsWorkflow = util.flow.sequential(
      
    function triggersSave__step1__saveStart(cx) {
      cx.editor.startEditor.getSaveData(this);
    },
    
    function triggersSave__step2__saveIsRange(cx, start) {
      cx.start = start;
      cx.editor.rangeEditor.getSaveData(this);
    },
    
    function triggersSave__step3__saveEnd(cx, isRange) {
      cx.isRange = isRange;
      cx.editor.endEditor.getSaveData(this);
    },
    
    function triggersSave__step4__saveInterval(cx, end) {
      cx.end = end;
      cx.editor.intervalEditor.getSaveData(this);
    },
    
    function triggersSave__step5__assembleResult(cx, interval) {
      // if the user bumped hours up past 24, datebox will still record a day
      // even though the days spinner is disabled, so let's mod it off
      var intervalMillis = interval.getMillis(),
          newInterval = baja.RelTime.make(intervalMillis % baja.RelTime.MILLIS_IN_DAY);
      
      this.ok({
        start: cx.start,
        isRange: cx.isRange,
        end: cx.end,
        interval: newInterval
      });
    }
  );

  
  /**
   * A field editor that will display a list of the triggers currently 
   * configured on a <code>schedule:TriggerSchedule</code>. Registered
   * using key <code>triggers-list</code>.
   * @class
   * @name niagara.fieldEditors.schedule.TriggersListEditor
   * @extends niagara.fieldEditors.mobile.MobileFieldEditor
   */
  TriggersListEditor = fe.defineEditor(MobileFieldEditor, {
    
    /**
     * Does nothing - all DOM manipulation will be done in 
     * <code>doLoadValue</code>.
     * 
     * @name niagara.fieldEditors.schedule.TriggersListEditor#doInitializeDOM
     * @function
     * @param {jQuery} element
     * @param {Object} callbacks
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#doInitializeDOM
     */
    doInitializeDOM: function doInitializeDOM(element, callbacks) {
      callbacks.ok();
    },
    
    /**
     * Assembles the triggers on a trigger schedule into a listview of the
     * trigger times with checkboxes to select each one.
     * 
     * @name niagara.fieldEditors.schedule.TriggersListEditor#doLoadValue
     * @function
     * @param {baja.Component} value a <code>schedule:DaySchedule</code>
     * existing as the <code>times</code> slot on a
     * <code>schedule:TriggerSchedule</code>.
     * @param {Object} callbacks an object containing ok/fail callbacks
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#doLoadValue
     */
    doLoadValue: function doLoadValue(value, callbacks) {
      var that = this,
          dom = that.$dom,
          ul = $('<ul data-role="listview" data-inset="true" data-theme="c" />'),
          header = $('<li data-role="list-divider"/>'),
          timeSchedules;

      header
        .append(checkboxHtml.patternReplace({
          checkboxId: 'selectAll'
        }))
        .append($('<div class="detailsContainer"/>').text(mobileLex.get('selectAll')));

      ul.append(header);
      
      timeSchedules = value.getSlots().properties().is(TIME_SCHEDULE_TYPE).toValueArray();
      
      // sort by time of day
      timeSchedules.sort(function (timeSchedA, timeSchedB) {
        var millisA = timeSchedA.getStart().getTimeOfDayMillis(),
            millisB = timeSchedB.getStart().getTimeOfDayMillis();
        return millisA - millisB;
      });
      
      // now assemble them into a listview
      baja.iterate(timeSchedules, function (timeSchedule) {
        var li = $('<li/>').appendTo(ul),
            checkboxContainer = $(checkboxHtml.patternReplace({
              checkboxId: timeSchedule.getName()
            })),
            detailsContainer = $('<div class="detailsContainer"/>');
        
        detailsContainer.append(timeSchedule.getStart().toString());
        li
          .append(checkboxContainer)
          .append(detailsContainer);
      });
      
      // enhance and load into DOM
      dom.html(ul).trigger('create');
      
      callbacks.ok();
    },
    
    /**
     * Just returns <code>this.value</code> - other functions on this editor
     * will make changes to the currently edited schedule as the user
     * interacts with the editor.
     * 
     * @name niagara.fieldEditors.schedule.TriggerSchedule#getSaveData
     * @function
     * @param {Object} callbacks
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#getSaveData
     */
    getSaveData: function getSaveData(callbacks) {
      callbacks.ok(this.value);
    }
  });
  
  /**
   * Returns the names of all the slots the user currently has selected
   * (checkboxes are checked).
   * 
   * @name niagara.fieldEditors.schedule.TriggersListEditor#getSelectedSlots
   * @function
   * @returns {Array} an array of slot names
   */
  TriggersListEditor.prototype.getSelectedSlots = function getSelectedSlots() {
    var slots = [];
    
    this.$dom.find('.ui-li-static input[type=checkbox]').each(function () {
      var $this = $(this);
      if ($this.is(':checked')) {
        slots.push($this.attr('id'));
      }
    });
    
    return slots;
  };
  
  /**
   * Removes all the selected slots from the currently edited trigger schedule.
   * 
   * @name niagara.fieldEditors.schedule.TriggersListEditor#removeSelected
   * @function
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  TriggersListEditor.prototype.removeSelected = function removeSelected(callbacks) {
    var that = this,
        component = that.value,
        workflow = util.flow.parallel(function (slot) {
          component.remove($.extend(this, { slot: slot }));
        });
    
    // remove all slots in parallel (if unmounted as in the schedule app, will
    // actually be synchronous)
    workflow.invoke(that.getSelectedSlots(), function () {
      that.loadValue(component, callbacks);
    });
  };
  
  fe.register(DAY_SCHEDULE_TYPE, TriggersListEditor, 'triggers-list');
  
  
  function toggleRangeDisplay(dom, isRange) {
    var method = isRange ? 'show' : 'hide';
    
    dom.find('.triggerRangeEditors').animate({
      height: method,
      opacity: method
    }, {
      step: function (now, fx) {
        dialogs.repositionDialog($.mobile.activePage);
      }
    });
  }
  
  function hasTime(daySchedule, time) {
    return daySchedule
      .getSlots()
      .properties()
      .is(TIME_SCHEDULE_TYPE)
      .eachValue(function (timeSchedule) {
        return timeSchedule.getStart().equals(time); 
      });
  }
  
  function doRange(startTime, endTime, interval) {
    var time = startTime,
        timeMillis = startTime.getTimeOfDayMillis(),
        timeSchedules = [],
        intervalMillis = interval.getMillis();
    
    if (intervalMillis < baja.RelTime.MILLIS_IN_MINUTE) {
      throw new Error(mobileLex.get('schedule.message.intervalRequired'));
    }
    
    function hasMoreTime() {
      return timeMillis < baja.RelTime.MILLIS_IN_DAY &&
             endTime && 
             !time.isAfter(endTime); //inclusive
    }
    
    do {
      timeSchedules.push(baja.$(TIME_SCHEDULE_TYPE, {
        start: time,
        finish: baja.Time.make(timeMillis + 20000) //+ 20secs
      }));
      
      timeMillis += intervalMillis;
      time = baja.Time.make(timeMillis);
    } while (hasMoreTime());

    
    return timeSchedules;
    
  }
  
  /**
   * A field editor used to add new triggers to a 
   * <code>schedule:TriggerSchedule</code>. Registered using key
   * <code>triggers-add</code>.
   * 
   * @class
   * @name niagara.fieldEditors.schedule.TriggersAddEditor
   * @extends niagara.fieldEditors.mobile.MobileFieldEditor
   */
  TriggersAddEditor = fe.defineEditor(MobileFieldEditor, {
    
    /**
     * Instantiates sub-field editors for start time, range toggle, end
     * time, and interval.
     * 
     * @name niagara.fieldEditors.schedule.TriggersAddEditor#doInitializeDOM
     * @function
     * @param {jQuery} element
     * @param {Object} callbacks
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#doInitializeDOM
     */
    doInitializeDOM: function doInitializeDOM(element, callbacks) {
      element.html(triggerAddHtml.patternReplace({
        triggerTime: mobileLex.get('schedule.triggerTime'),
        rangeEnd: scheduleLex.get('trigger.rangeEnd'),
        rangeInterval: scheduleLex.get('trigger.rangeInterval'),
        triggersToCreate: mobileLex.get('schedule.triggersToCreate')
      }));
      triggersAddEditorsWorkflow.invoke(this, callbacks);
    },
    
    /**
     * Does nothing - no actual component state to display.
     * 
     * @name niagara.fieldEditors.schedule.TriggersAddEditor#doLoadValue
     * @function
     * @param {baja.Component} value a <code>schedule:DaySchedule</code>
     * existing as the <code>times</code> slot on a
     * <code>schedule:TriggerSchedule</code>.
     * @param {Object} callbacks an object containing ok/fail callbacks
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#doLoadValue
     */
    doLoadValue: function doLoadValue(value, callbacks) {
      this.updateTriggerCountDisplay();
      
      callbacks.ok();
    },
    
    /**
     * Creates an array of <code>schedule:TimeSchedule</code>s based on the
     * user input. If the range checkbox is unchecked, this array will be of
     * length 1 and only contain an entry for the start time entered by the
     * user. Otherwise, the start time, end time, and interval will be used
     * to assemble a list of triggers.
     * 
     * @name niagara.fieldEditors.schedule.TriggerSchedule#getSaveData
     * @function
     * @param {Object} callbacks
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#getSaveData
     */
    getSaveData: function getSaveData(callbacks) {
      // first save all our field editors...
      triggersSaveEditorsWorkflow.invoke({ editor: this }, {
        ok: function (result) {
          // ...then use the results to assemble an array of time schedules
          var times,
              start = result.start,
              isRange = result.isRange,
              end = result.end,
              interval = result.interval;
          
          
          if (isRange) {
            times = doRange(start, end, interval);
          } else {
            times = [];
            times.push(baja.$(TIME_SCHEDULE_TYPE, {
              start: start,
              finish: baja.Time.make(start.getTimeOfDayMillis() + 20000)
            }));
          }
          
          callbacks.ok(times);
        },
        fail: callbacks.fail
      });
    },
    
    /**
     * @name niagara.fieldEditors.schedule.TriggerSchedule#doSaveValue
     * @function
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#doSaveValue
     */
    doSaveValue: function (value, times, callbacks) {
      baja.iterate(times, function (timeSchedule) {
        if (!hasTime(value, timeSchedule.getStart())) {
          value.add({
            slot: 'time?',
            value: timeSchedule
          });
        }
      });
      
      callbacks.ok(value);
    },
    
    /**
     * @name niagara.fieldEditors.schedule.TriggerSchedule#armHandlers
     * @function
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#armHandlers
     */
    armHandlers: function () {
      var that = this,
          dom = that.$dom;
      
      //arm handler on range checkbox
      dom.delegate('.triggerRange', 'editorchange', function () {
        that.rangeEditor.getSaveData(function (isRange) {
          var startText = mobileLex.get(isRange
              ? 'schedule.rangeStart'
              : 'schedule.triggerTime') + ':';
          
          dom.find('.rangeStartLabel').text(startText);

          //show/hide end/interval editors if range is checked
          toggleRangeDisplay(dom, isRange);
        });
      });
      
      dom.bind('editorchange', function () {
        that.updateTriggerCountDisplay();
      });
    }
  });
  
  TriggersAddEditor.prototype.updateTriggerCountDisplay = function () {
    var dom = this.$dom;
    triggersSaveEditorsWorkflow.invoke({ editor: this }, function (result) {
      var start = result.start,
          end = result.end,
          interval = result.interval,
          totalMillis = end.getTimeOfDayMillis() - start.getTimeOfDayMillis() + baja.RelTime.MILLIS_IN_MINUTE,
          intervalMillis = interval.getMillis(),
          intervalValid = intervalMillis >= baja.RelTime.MILLIS_IN_MINUTE,
          totalTriggers = Math.max(Math.ceil(totalMillis / intervalMillis), 1),
          label = dom.find('.triggersToCreate'),
          text = intervalValid
            ? mobileLex.get('schedule.triggersToCreate') + ': ' + totalTriggers
            : mobileLex.get('schedule.message.intervalRequired');
      
      label.text(text);
      label.toggleClass('error', !intervalValid);
    });
  };
  
  fe.register(DAY_SCHEDULE_TYPE, TriggersAddEditor, 'triggers-add');
  
  util.api('niagara.fieldEditors.schedule', {
    TriggersAddEditor: TriggersAddEditor,
    TriggersListEditor: TriggersListEditor
  });
}());