/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to the DayEditor field editor for a
 * <code>schedule:DaySchedule</code> object.
 * 
 * @author Logan Byam
 * @version 0.0.1
 */
/*jslint white: true */
/*global niagara, $, baja */

(function dayFunctions() {
  "use strict";
  
  niagara.util.require(
    'jQuery',
    'niagara.util.schedule',
    'niagara.util.time',
    'niagara.schedule.ScheduleBlock',
    'niagara.fieldEditors.mobile'
  );
  
      //imports
  var util = niagara.util,
      timeUtil = util.time,
      scheduleUtil = util.schedule,
      ScheduleBlock = niagara.schedule.ScheduleBlock,
      fe = niagara.fieldEditors,
      
      //exports
      DayEditor;
  
  
  /**
   * Represents one day in a schedule object.
   * 
   * @class
   * @name niagara.fieldEditors.schedule.DayEditor
   * @extends niagara.fieldEditors.mobile.MobileFieldEditor
   */
  DayEditor = (function DayEditor() {
    return fe.defineEditor(fe.mobile.MobileFieldEditor, {
      
      /**
       * After instantiation, builds up its set of schedule blocks, one for each
       * <code>schedule:TimeSchedule</code> contained in the
       * <code>DaySchedule</code>'s <code>day</code> slot.
       * 
       * @name niagara.fieldEditors.schedule.DayEditor#postCreate
       * @function
       */
      postCreate: function postCreate() {
        var that = this;
        that.scheduleBlocks = [];
  
        that.$dayDisplay = that.params.label || that.container.getDisplayName();
      },
      
      /**
       * Creates a <code>&lt;div&gt;</code> element to display this day - a vertical
       * stack of eight blocks (representing three hours apiece) on top of which 
       * additional divs for the schedule blocks will be overlaid.
       * 
       * @name niagara.fieldEditors.schedule.DayEditor#doInitializeDOM
       * @function
       * @param {jQuery} targetElement the DOM element in which this day editor
       * will be built
       * @param {Object} callbacks an object containing ok/fail callbacks
       * @returns {jQuery} the built-up div
       */
      doInitializeDOM: function doInitializeDOM(targetElement, callbacks) {
        var dayDiv, 
            blocksDiv, 
            i, 
            container = this.container,
            name = container.getName(),
            displayName = this.$dayDisplay;
        
        dayDiv = $('<div class="day" />')
          .addClass(name) //"monday"
          .append($('<div class="dayTitle"/>').text(displayName)); //"Monday"
        
        blocksDiv = $('<div class="blocksDisplay"/>').appendTo(dayDiv);
        
        baja.iterate(8, function (i) {
          $('<div class="block" />')
            .addClass('block' + i)
            .addClass(name)
            .appendTo(blocksDiv);
        });
        
        dayDiv.data('daySchedule', this.value);
        
        dayDiv.appendTo(targetElement);
        
        callbacks.ok();
      },
      
      /**
       * Given a <code>DaySchedule</code>, converts all of its current
       * <code>schedule:TimeSchedule</code> slots into 
       * <code>ScheduleBlock</code>s to be displayed and edited.
       * 
       * @name niagara.fieldEditors.schedule.DayEditor#doLoadValue
       * @function
       * @param {baja.Component} daySchedule the 
       * <code>schedule:DaySchedule</code> to be edited
       * @param {Object} callbacks an object containing ok/fail callbacks
       */
      doLoadValue: function doLoadValue(daySchedule, callbacks) {
        var that = this;
        
        that.empty();
        
        daySchedule
          .getSlots()
          .is('schedule:TimeSchedule')
          .eachValue(function (timeSchedule) {
            that.addBlock(
                timeSchedule.get('start'), 
                timeSchedule.get('finish'), 
                timeSchedule.get('effectiveValue'));
          });
        
        that.setModified(false);
        
        callbacks.ok();
      },
      
      /**
       * Save data for a DayEditor is an array of all the
       * <code>ScheduleBlocks</code> it currently contains.
       * 
       * @name niagara.fieldEditors.schedule.DayEditor#getSaveData
       * @function
       * 
       * @param {Object} callbacks an object containing ok/fail callbacks
       * @returns {Array} an array of this day editor's schedule blocks (to be
       * passed to the callbacks' ok handler)
       */
      getSaveData: function getSaveData(callbacks) {
        callbacks.ok(this.scheduleBlocks);
      },
      
      /**
       * Sets the <code>schedule:TimeSchedule</code> slots on the currently
       * edited <code>schedule:DaySchedule</code> to reflect the day editor's
       * current set of schedule blocks (returned from 
       * <code>getSaveData()</code>).
       * 
       * @param {baja.Component} value the currently edited
       * <code>DaySchedule</code>
       * @param {Array} scheduleBlocks the array of schedule blocks this day
       * editor contains
       * @param {Object} callbacks an object containing ok/fail callbacks
       * @returns the updated <code>DaySchedule</code> (to be passed to the
       * callbacks' ok handler)
       */
      doSaveValue: function doSaveValue(value, scheduleBlocks, callbacks) {
        value.getSlots().is('schedule:TimeSchedule').each(function (slot) {
          value.remove({
            slot: slot
          });
        });
        
        baja.iterate(this.scheduleBlocks, function (block) {
          value.add({
            slot: 'time?',
            value: block.createTimeSchedule()
          });
        });
        
        callbacks.ok(value);
      },
      
      /**
       * If this editor is not configured as readonly (readonly parameter passed
       * into field editor constructor), then will arm event handlers to enable
       * creating, dragging, and resizing schedule blocks.
       * 
       * @name niagara.fieldEditors.schedule.DayEditor#armHandlers
       * @function
       */
      armHandlers: function armHandlers() {
        if (!this.isReadonly()) {
          niagara.schedule.ui.day.armHandlers(this, this.$dom.children('.day'));
        }
      }
    });
  }());
  
  
  /**
   * Wipes out and redraws all schedule block divs. Will be called once
   * automatically when the editor is instantiated; should be called again after
   * removing or adding new blocks.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#refreshWidgets
   * @function
   */
  DayEditor.prototype.refreshWidgets = function refreshWidgets() {
    var blocksDiv = $('.blocksDisplay', this.$dom);
    $('.scheduleBlockContainer', blocksDiv).remove();
    
    baja.iterate(this.getBlocks(), function (block) {
      block.generateDiv(blocksDiv);
    });
  };
  
  
  /**
   * Adds a new schedule block to this day - does not make any changes to the
   * underlying component. Should only be called when constructing the object
   * or adding new schedule blocks.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#addBlock
   * @function
   * 
   * @param {baja.Time} start the start time
   * @param {baja.Time} finish the finish time
   * @param {baja.Struct} value the effective value 
   * (<code>baja:StatusValue</code>)
   * @return {niagara.schedule.ScheduleBlock} the block that was added
   */
  DayEditor.prototype.addBlock = function (start, finish, value) {
    var block = new ScheduleBlock(start, finish, value);
    this.scheduleBlocks.push(block);
    this.setModified(true);
    return block;
  };
  
  
  /**
   * Empties out any other events from this day and converts the given block
   * to a midnight-to-midnight 24-hour event.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#allDayEvent
   * @function
   * 
   * @param {baja.Struct} value the value to set for the entirety of this day
   * (<code>baja:StatusValue</code>)
   */
  DayEditor.prototype.allDayEvent = function (value) {
    this.empty();
    
    this.addBlock(baja.Time.make(0), baja.Time.make(0), value);
  };

  
  /**
   * Checks to see if the given block is able to set its own start/finish to
   * the given start/finish times. Ensures that setting the times will not
   * result in an overlap/conflict with another schedule block. 
   * 
   * <p>This method should be called upon a drag
   * or resize on a schedule block, and the drag or resize should be cancelled
   * if this method returns false.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#canMoveBlockTo
   * @function
   * 
   * @param {ScheduleBlock} block the block attempting a move/resize
   * @param {Number} startms the block's desired start time (in ms past midnight)
   * @param {Number} finishms the block's desired finish time (in ms past midnight)
   * @return {Boolean} true if the block can move to this date range
   */
  DayEditor.prototype.canMoveBlockTo = function (block, startms, finishms) {
    return !baja.iterate(this.scheduleBlocks, function (myBlock) {
      if (myBlock && myBlock !== block && myBlock.overlaps(startms, finishms)) {
        return true;
      }
    });
  };
  
  
  /**
   * Overwrites all blocks of the given day editor with the blocks from this day
   * editor.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#copyTo
   * @function
   * 
   * @param {niagara.fieldEditors.schedule.DayEditor} destDayEditor the 
   * day editor we wish to overwrite
   */
  DayEditor.prototype.copyTo = function (destDayEditor) {
    var that = this;
    
    destDayEditor.empty();
    
    baja.iterate(that.getBlocks(), function (block) {
      destDayEditor.addBlock(block.start, block.finish, block.value);
    });
  };
  
  
  /**
   * Removes all schedule blocks from this day (and from the underlying
   * <code>DaySchedule</code> component).
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#empty
   * @function
   */
  DayEditor.prototype.empty = function () {
    this.scheduleBlocks = [];
    this.setModified(true);
  };
  
  
  /**
   * Two days are considered equivalent if they have the same number of
   * schedule blocks and every schedule block in one is equivalent with the
   * corresponding schedule block in the other.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#equivalent
   * @function
   * 
   * @param {niagara.fieldEditors.schedule.DayEditor} other the day to check 
   * for equivalence
   * @return {Boolean} true if the two days are equivalent
   */
  DayEditor.prototype.equivalent = function (other) {
    if (!other instanceof DayEditor) {
      return false;
    }
    
    var myBlocks = this.getBlocks(),
        hisBlocks = other.getBlocks();
    
    if (myBlocks.length !== hisBlocks.length) {
      return false;
    }
    
    return !baja.iterate(myBlocks, function (block, i) {
      if (block.equivalent(hisBlocks[i])) {
        return true;
      }
    });
  };
  

  /**
   * Returns the schedule blocks contained in this day.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#getBlocks
   * @function
   * 
   * @return {Array} an array of <code>niagara.schedule.ScheduleBlock</code>s
   */
  DayEditor.prototype.getBlocks = function () {
    return this.scheduleBlocks;
  };
  
  
  /**
   * Removes the specified block from this day (and from the underlying
   * <code>DaySchedule</code> component).
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#removeBlock
   * @function
   * 
   * @param {niagara.schedule.ScheduleBlock} block the block to be removed
   * @return {niagara.schedule.ScheduleBlock} the block that was removed
   */
  DayEditor.prototype.removeBlock = function (block) {
    var scheduleBlocks = this.scheduleBlocks,
        index = util.indexOf(scheduleBlocks, block);
    
    if (index >= 0) {
      scheduleBlocks.splice(index, 1);
      this.setModified(true);
      return block;
    }
  };
  
  
  /**
   * Returns the currently selected/highlighted schedule block.
   * 
   * @name niagara.fieldEditors.schedule.DayEditor#getSelectedBlock
   * @function
   * 
   * @return {niagara.schedule.ScheduleBlock} the currently selected block, or
   * undefined if none selected
   */
  DayEditor.prototype.getSelectedBlock = function () {
    return this.$dom.find('.scheduleBlockContainer.selected')
      .data('scheduleBlock');
  };
  
  
  fe.register('schedule:DaySchedule', DayEditor);

  util.api('niagara.fieldEditors.schedule', {
    DayEditor: DayEditor
  });
}());