//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

/**
 * @fileOverview Utility methods for the Px App.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true, browser: true */

/*global $, baja, BaseBajaObj, window, niagara*/ 

/**
 * @namespace Niagara Util
 */ 
niagara.util.namespace("niagara.util");

/**
 * @namespace Niagara Px Util
 */  
niagara.util.px = (function pxUtil(baja) {
   // Use ECMAScript 5 Strict Mode
  "use strict";
            
  /**
   * Return the color code for the given Status.
   *
   * @name niagara.util.px.getStatusColor
   * @function
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} [obj.statusName] if specified, the color for the given status will be returned.
   * @param {baja.Status} [obj.status] if specified, the color for the status will be returned.
   * @param {String} [obj.defaultColor] returned if no color can be found for the given status.
   * @param {Boolean} obj.isForeground request foreground or background color.
   * @return resolved color value.
   */
  function getStatusColor(obj) {
    var defaultColor = obj.defaultColor || null;
    var isForeground = obj.isForeground || false;
  
    if (obj.status) {
      var status = obj.status;
      var statusName;
      
      if (status.isDisabled()) {
        statusName = "disabled";
      }
      else if (status.isFault()) {
        statusName = "fault";
      }
      else if (status.isDown()) {
        statusName = "down";
      }
      else if (status.isAlarm()) {
        statusName = "alarm";
      }
      else if (status.isStale()) {
        statusName = "stale";
      }
      else if (status.isOverridden()) {
        statusName = "overridden";
      }
      else {
        return defaultColor;
      }
      
      return getStatusColor({
        statusName: statusName,
        isForeground: isForeground,
        defaultColor: defaultColor
      });
    }
  
    return baja.lex("baja").get({
      key: "Status." + obj.statusName + (isForeground ? ".fg" : ".bg"),
      def: defaultColor
    });
  }
  
  /**
   * Return the brush for the given Status.
   *
   * @name niagara.util.px.getStatusBrush
   * @function
   *
   * @param {Object} obj Object Literal for method arguments.
   * @param {String} obj.statusName.
   * @param {Boolean} obj.isForeground request foreground or background color.
   * @param [obj.proto] the prototype brush.
   * @return resolved brush.
   */
  function getStatusBrush(obj) {
    var proto = obj.proto;
    if (!proto) {
      proto = baja.$("gx:Brush");
    }
    var statusColor = getStatusColor(obj);
    return statusColor === null ? null : proto.decodeFromString(statusColor);
  }
  
  ////////////////////////////////////////////////////////////////
  // Layout utility functions
  ////////////////////////////////////////////////////////////////  

  /**
   * Adds the CSS3 transforms to the CSS object, including all vendor prefixes 
   * and IE8 filter malarkey.
   * 
   * @memberOf niagara.util.px
   * @param {Object} cssObj the CSS object to add rules to
   * @param {Number} hScale horizontal scale factor
   * @param {Number} vScale vertical scale factor
   */
  function addScaling(cssObj, hScale, vScale) {
    var i,
        prefixes = ['', '-moz-', '-webkit-', '-ms-'],
        prop;
    
    for (i = 0; i < prefixes.length; i++) {
      cssObj[prefixes[i] + 'transform'] = 'scale(' + hScale + ',' + vScale + ')';
      cssObj[prefixes[i] + 'transform-origin'] = 'left top';
    }
    
    cssObj['-ms-filter'] = cssObj.filter = 
      "progid:DXImageTransform.Microsoft.Matrix(M11=" + 
      hScale + ", M12=0, M21=0, M22=" + vScale + ", SizingMethod='auto expand')";
  }
  
  /**
   * This function applies to widgets that support dynamic scaling of an 
   * internal element - currently, Pictures and CanvasPanes. It calculates
   * the CSS necessary to scale and align the internal element within the
   * absolutely positioned outer frame. 
   * 
   * @memberOf niagara.util.px
   * @param {Number} outerWidth width of absolutely positioned outer element
   * @param {Number} outerHeight height of absolutely positioned outer element
   * @param {Number} innerWidth width of scaled/aligned inner element
   * @param {Number} innerHeight height of scaled/aligned inner element
   * @param {String} halign the horizontal alignment - should correspond to
   * a BHalign tag (left, center, right, fill)
   * @param {String} valign the vertical alignment - should correspond to
   * a BValign tag (top, center, bottom, fill)
   * @param {String} scale the scale mode - should correspond to a BScaleMode
   * tag (none, fit, fitRatio, fitWidth, fitHeight)
   * @return {Object} an object with <code>display</code>, <code>width</code>, 
   * <code>height</code>, <code>margin-top</code>, and <code>margin-left</code> 
   * properties. This should be applied to the inner element to be scaled.
   */
  function getScaledLayoutCSS(outerWidth, outerHeight, innerWidth, innerHeight,
                             halign, valign, scale) {
    var imgRatio,
        desiredWidth, 
        desiredHeight, 
        marginTop, 
        marginLeft,
        cssObj;

    switch(scale) {
      case 'none':
        desiredWidth = innerWidth;
        desiredHeight = innerHeight;
        break;
      case 'fit':
        desiredWidth = outerWidth;
        desiredHeight = outerHeight;
        break;
      case 'fitHeight':
        desiredWidth = innerWidth;
        desiredHeight = outerHeight;
        break;
      case 'fitWidth':
        desiredWidth = outerWidth;
        desiredHeight = innerHeight;
        break;
      case 'fitRatio':
        //scale image down only enough to fit within picture borders
        imgRatio = Math.min(outerWidth / innerWidth, outerHeight / innerHeight);
        desiredWidth = innerWidth * imgRatio;
        desiredHeight = innerHeight * imgRatio;
        break;
    }

    /*
     * align using margins, since text-align won't quite do it if the image is
     * larger than the Picture itself
     */
    switch (halign) {
      case 'left': 
        marginLeft = 0; 
        break;
      case 'center': 
        marginLeft = (outerWidth - desiredWidth) / 2; 
        break;
      case 'right': 
      case 'fill': 
        marginLeft = (outerWidth - desiredWidth); 
        break;
    }
    
    switch (valign) {
      case 'top': 
        marginTop = 0; 
        break;
      case 'center': 
        marginTop = (outerHeight - desiredHeight) / 2; 
        break;
      case 'bottom': 
      case 'fill': 
        marginTop = (outerHeight - desiredHeight); 
        break;
    }
    
    cssObj = { 
      display: 'block',
      width: desiredWidth, 
      height: desiredHeight, 
      'margin-top': marginTop,
      'margin-left': marginLeft
    };
    
    return cssObj;
  }
  
  /**
   * Get a vertical align value for the widget, derived from the widget's
   * valign property.
   * 
   * @memberOf niagara.util.px
   * @param {Widget} widget
   * @return {String} a vertical-align CSS value ('top', 'middle', 'bottom')
   */
  function getValign(widget) {
    if (widget.has('valign')) {
      var valign = widget.getValign().getTag().toLowerCase();
      if (valign === "fill" || valign === "center") {
        valign = "middle";
      }
      return valign;
    }
  }
  
  /**
   * Get a horizontal align value for the widget, derived from the widget's
   * halign property.
   * 
   * @memberOf niagara.util.px
   * @param {Widget} widget
   * @return {String} a text-align CSS value ('left', 'center', 'right')
   */
  function getHalign(widget) {
    if (widget.has('halign')) {
      var halign = widget.getHalign().getTag().toLowerCase();
      if (halign === 'fill') {
        halign = 'center';
      }
      return halign;
    }
  }
  
  /**
   * Asynchronously calculates image width and height and passes them back to
   * callback function.
   * 
   * @memberOf niagara.util.px
   * @param {String} imgSrc the path where the image lives
   * @param {Function} handleDims a function to handle width and height
   */
  function calculateImageDimensions(imgSrc, handleDims) {
    var img = new Image();
    
    img.onload = function () {
      handleDims(this.width, this.height);
      img.onload = null;
      img = null;
    };
    
    img.src = imgSrc; 
  }
    
  return {
    addScaling: addScaling,
    calculateImageDimensions: calculateImageDimensions,
    getHalign: getHalign,
    getScaledLayoutCSS: getScaledLayoutCSS,
    getStatusColor: getStatusColor,
    getStatusBrush: getStatusBrush,
    getValign: getValign
  };
}(baja));


