//
// Copyright 2012, Tridium, Inc. All Rights Reserved.
//

/**
 * @fileOverview Widgets for the Px App.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, 
bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, BaseBajaObj, window, niagara, Image*/ 

/**
 * @namespace Px Mobile
 */ 
niagara.util.namespace("niagara.px.mobile");  
  
(function pxMobileWidgets(baja) {
  // Use ECMAScript 5 Strict Mode
  "use strict";
   
  niagara.util.require(
    "$.mobile",
    "niagara.util.mobile.dialogs",
    "niagara.util.flow",
    "niagara.util.mobile.touchscroll",
    "niagara.util.px",
    "niagara.util.px.mobile",
    "niagara.px.mobile"
  );
   
  var util = niagara.util,
      dialogs = util.mobile.dialogs,
      touchScroll = util.mobile.touchscroll.touchScroll,
      pxUtil = util.px,
      pxUtilMobile = util.px.mobile,
      mobile = niagara.px.mobile,
      noop = baja.noop,
      
      addScaling = pxUtil.addScaling,
      calculateImageDimensions = pxUtil.calculateImageDimensions,
      getHalign = pxUtil.getHalign,
      getScaledLayoutCSS = pxUtil.getScaledLayoutCSS,
      getValign = pxUtil.getValign;
  

  ////////////////////////////////////////////////////////////////
  // Widget
  ////////////////////////////////////////////////////////////////  
   
  /**
   * @class Widget.
   * <p>
   * The base Widget that represents Niagara's 'bajaui:Widget' Type.
   * 
   * @name Widget
   * @extends baja.Component
   */
  var Widget = function() {
    Widget.$super.apply(this, arguments);
    this.$dom = null;
  }.$extend(baja.Component).registerType("bajaui:Widget");
  
  /**
   * Return the DOM element for the Widget.
   *
   * @return {Object} the DOM element
   */
  Widget.prototype.getDomElement = function() {
    return this.$dom;
  };
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   * By default, this method throws an error as it's designed to be overriden.
   * <p>
   * Anything that implements this method should assign the associated DOM element for the
   * Widget to the '$dom' attribute.
   *
   * @param {Object} DOM element to append content too.
   */
  Widget.prototype.load = function(dom) {
    throw new Error("Unsupported Widget: " + this.getType());
  };
  
  /**
   * Called after the Widget has loaded and any enhancements have 
   * been made to the DOM.
   *
   * @function
   */
  Widget.prototype.postLoad = noop;
      
  var widgetType,
      loadChildren = function (dom) {
    // Lazily load Widget Type
    if (!widgetType) {
      widgetType = baja.lt("bajaui:Widget");
    }
    
    dom = dom || this.$dom;
    
    // Load all child Widget content...
    this.getSlots(function (slot) {
      // Iterate through all Widget Properties
      return slot.isProperty() && slot.getType().is(widgetType);
    }).each(function (slot) {
      // Load the Content on all Widget Properties
      var widget = this.get(slot);
      widget.load(dom);
      widget.loadChildren();
    });
  };
  
  /**
   * Load the child Widgets.
   * <p>
   * {@link baja.Widget#load} must be called before this method is invoked.
   *
   * @function
   */
  Widget.prototype.loadChildren = loadChildren;
  
  function hasUpdateAll() {
    return true;
  }
  
  function hasUpdateEnabled(propName) {
    return propName === "enabled";
  }
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   * @param {Boolean} [firstUpdate] set to true on Widgets first update
   *                                when the page is first loaded.
   */
  Widget.prototype.update = function(hasUpdate, firstUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Bail if there's no DOM element available
    if (!this.$dom) {
      return;
    }
    
    if (hasUpdate("visible")) {
      if (this.getVisible()) {
        this.$dom.show();
      }
      else {
        this.$dom.hide();
      }
    }
    
    if (hasUpdate("enabled") && !firstUpdate) {
      // If the enabled state of this Widget has changed then update all of the child Widgets
      this.getSlots(function (slot) {
        return slot.isProperty() && slot.getType().is(widgetType);
      }).each(function (slot) {
        this.get(slot).update(hasUpdateEnabled);
      });
    }
    
    if (hasUpdate("layout") && this.getParent() && this.getDomElement()) {
      // If the parent Widget supports laying out children then call it
      this.getParent().layoutChild(this);
    }
  };
  
  // Check to make sure all parent Widgets are enabled
  function checkParentWidgetsEnabled(widget) {
    if (widget === null) {
      return true;
    }
    if (!widget.getEnabled()) {
      return false;
    }
    return checkParentWidgetsEnabled(widget.getParent());
  }
  
  /**
   * Return true if the Widget (and all parent Widgets) are enabled.
   *
   * @return {Boolean}
   */
  Widget.prototype.isEnabled = function() {
    return checkParentWidgetsEnabled(this);
  };
    
  /**
   * Handle a UI Widget event.
   * <p>
   * By default, this method will return immediately if the Widget is disabled.
   * {@link Widget#doHandleEvent} will then be called for any subclasses.
   * The event will then get passed onto each binding.
   *
   * @param {String} eventName the name of the event being fired.
   */
  Widget.prototype.handleEvent = function(eventName) {
    // If this or any parent Widget aren't enabled then don't process the event
    if (!this.isEnabled(this)) {
      return;
    }
    
    // Let any subclasses handle the event
    if (this.doHandleEvent(eventName)) {
      return;
    }
        
    // Forward event to all registered bindings
    this.getSlots(function (s) {
      return s.isProperty() && s.getType().is("bajaui:Binding");
    }).eachValue(function(binding) {
      if (binding.handleEvent(eventName)) {
        return true;
      }
    });
  };
  
  /**
   * Handle a UI Widget event. 
   * <p>
   * This method is designed to be overriden and will only be called
   * if the Widget is enabled.
   *
   * @see Widget#doHandleEvent
   *
   * @function
   *
   * @param {String} eventName the name of the event.
   * @returns {Boolean} return true if the event shouldn't bubble through to the Widget's bindings.
   */
  Widget.prototype.doHandleEvent = noop;
    
  /**
   * Return a Property's value.
   * <p>
   * This method overrides {@link baja.Complex#get} and checks to see if
   * the Property has been overriden by any associated bindings..
   *
   * @see baja.Complex#get
   *
   * @param {baja.Slot|String} [slot]  the Slot or Slot name
   *
   * @return the value for the Property (or null if the Property doesn't exist)
   */
  Widget.prototype.get = function(slot) {
    var origGet = Widget.$super.prototype.get,
        val = null;
        
    // If there are definately no bindings just use the original get
    if (this.$hasNoBindings) {
      return origGet.apply(this, arguments);
    }
    
    // Get the Slot
    slot = this.getSlot(slot);
    
    // Bail if there's no Slot
    if (!slot) {
      return null;
    }
    
    // Get cached overridden access
    if (typeof slot.$isOverridden === "boolean") {
      if (slot.$isOverridden) {
        val = slot.$binding.getOnWidget(slot);
        if (val === null) {
          return origGet.apply(this, arguments);
        }
      }
      else {
        return origGet.apply(this, arguments);
      }
    }
    
    var binding,
        p,
        map = this.$map.$map;
    
    // Access internal OrderedMap internally for speed
    for (p in map) {
      if (map.hasOwnProperty(p) && map[p].isProperty() && map[p].getType().isComponent() && map[p].getType().is("bajaui:Binding")) {
        binding = origGet.call(this, map[p]);
            
        if (binding.isBound()) {
          // Cache whether the slot is overridden for quicker access in the future
          slot.$isOverridden = binding.isOverridden(slot);
          
          if (slot.$isOverridden) {
            // If the slot is overriden then cache the binding on the Slot
            slot.$binding = binding;
            val = binding.getOnWidget(slot);
            
            if (val !== null) {
              break;
            }
          }
        }
      }
    }
    
    // Cache the fact this Widget has no bindings
    if (!binding) {
      this.$hasNoBindings = true;
    }
            
    if (val !== null) {
      return val;
    }
    
    return origGet.apply(this, arguments);
  };
  
  /**
   * Add mouse over effect to Widget.
   */
  Widget.prototype.addMouseOver = function() {
    if (this.$dom) {
      this.$dom.addClass("widgetMouseOver");
    }  
  };
  
  function findRootContainer(widget) {
    if (widget === null) {
      return null;
    }
    if (widget.getType().is("bajaui:RootContainer")) {
      return widget;
    }
    else {
      return findRootContainer(widget.getParent());
    }
  }
  
  /**
   * Return the root container of the Widget tree.
   */
  Widget.prototype.getRootContainer = function() {
    if (!this.$root) {
      this.$root = findRootContainer(this);
    }
    return this.$root;
  };
  
  /**
   * Layout the child Widget.
   * <p>
   * Gives a parent Widget a chance to layout a child Widget. By default,
   * this does nothing.
   *
   * @function
   *
   * @param {Widget} childWidget the Child Widget to be laid out.
   */
  Widget.prototype.layoutChild = noop;
  
  /**
   * Return the preferred Width of the Widget. By default, this will return the
   * DOM element's width.
   *
   * @returns {Number} the width in pixels.
   */
  Widget.prototype.getPreferredWidth = function() {
    return this.$dom.width();
  };
  
  /**
   * Return the preferred Height of the Widget. By default, this will return 
   * the DOM element's height.
   *
   * @returns {Number} the height in pixels.
   */
  Widget.prototype.getPreferredHeight = function() {
    return this.$dom.height();
  };
  
  /**
   * Return true if this Widget is absolutely positioned.
   *
   * @returns {Boolean} return true this Widget is absolutely positioned.
   */
  Widget.prototype.isAbsolutelyPositioned = function() {
    var parent = this.getParent();
    return parent && parent.getType().is("bajaui:CanvasPane");
  };
  
  ////////////////////////////////////////////////////////////////
  // Root Container
  //////////////////////////////////////////////////////////////// 
  
  /**
   * @class RootContainer.
   * <p>
   * The mobile pane Widget that represents Niagara's 'bajaui:RootContainer' Type.
   * 
   * @name RootContainer
   * @extends Widget
   */  
  var RootContainer = function() {
    RootContainer.$super.apply(this, arguments);
    this.$savableWidgets = [];
  }.$extend(Widget).registerType("bajaui:RootContainer"); 
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   *
   * @param {Object} dom DOM element to append content too.
   */
  RootContainer.prototype.load = function(dom) {
    this.$dom = $("<div></div>");
    this.$dom.appendTo(dom);
  };
  
  /**
   * Add a Widget that can be modified and saved.
   * <p>
   * Any Widget added must have an 'isModified' and 'save' method. 
   *
   * @param {Widget} widget
   */
  RootContainer.prototype.addSavableWidget = function(widget) {
    if (!this.$savableWidgets.contains(widget)) {
    
      // Check we have the correct methods defined on this objects
      if (typeof widget.isModified !== "function") {
        throw new Error("Could not find 'isModified' method on modify widget");
      }
      if (typeof widget.save !== "function") {
        throw new Error("Could not find 'save' method on modify widget");
      }
      if (typeof widget.clearModified !== "function") {
        throw new Error("Cound not find 'clearModified' method on modify widget");
      }
          
      this.$savableWidgets.push(widget);
    }
  };
  
  RootContainer.$saveWidgetsWorkflow = util.flow.sequentialParallel(
    function (widget) {
      if (typeof widget.validate === 'function') {
        widget.validate(this);
      } 
      else {
        this.ok();
      }
    },
    function (widget) {
      widget.save(this);
    }
  );
  
  /**
   * Save any Widgets that have been modified.
   *
   * @param {Object} obj the object literal that holds the method's arguments
   * @param {Function} obj.ok the ok callback handler that's called once everything has been saved
   * @param {Function} obj.fail the fail callback handler that's called if something fails in the save process
   */   
  RootContainer.prototype.saveWidgets = function(callbacks) {
    callbacks = util.callbackify(callbacks, baja.ok, dialogs.error);
   
    var modifiedWidgets = [];
    
    // Find all the modified Widgets
    baja.iterate(this.$savableWidgets, function (widget, i) { 
      if (widget.isModified()) {
        modifiedWidgets.push(widget);
      }
    });
    
    if (modifiedWidgets.length) {
      RootContainer.$saveWidgetsWorkflow.invoke(modifiedWidgets, callbacks);
    } else {
      callbacks.ok();
    }
  };
    
  /**
   * Refresh the page.
   */
  RootContainer.prototype.refresh = function(callbacks) {
    callbacks = util.callbackify(callbacks);
    pxUtilMobile.saveModifyQuestion(this, function yesNo() {
        callbacks.ok();
        baja.runAsync(function() {
          window.location.reload(/*forceGet*/false);
        });
      },
      function cancel() {
        callbacks.ok();
      });
  };
  
  /**
   * Return true if any of the widgets are modified and hence can be saved.
   *
   * @return {Boolean}
   */
  RootContainer.prototype.isWidgetTreeModified = function() {
    var modified = false;    
    baja.iterate(this.$savableWidgets, function (widget) {    
      if (widget.isModified()) {
        modified = true;
        
        // Returning true will break iteration
        return true;
      }
    });
    return modified;
  };
  
  /**
   * Clear down the modified state of all the saveable Widgets in the Widget tree.
   */
  RootContainer.prototype.clearWidgetTreeModified = function() {
    // Clear the modified state of all the Widgets
    baja.iterate(this.$savableWidgets, function (widget) { 
      if (widget.isModified()) {
        widget.clearModified();
      }
    });
  };
  
  /**
   * Return the PxPage for this Widget tree
   *
   * @return PxPage
   */
  RootContainer.prototype.getPxPage = function() {
    return this.$pxPage;
  };
    
  ////////////////////////////////////////////////////////////////
  // Pane/Layout Widgets
  ////////////////////////////////////////////////////////////////  
  
  /**
   * @class BasicMobilePane.
   * <p>
   * The mobile pane Widget that represents Niagara's 'mobile:BasicMobilePane' Type.
   * 
   * @name BasicMobilePane
   * @extends Widget
   */
  var BasicMobilePane = function() {
    BasicMobilePane.$super.apply(this, arguments);
  }.$extend(Widget).registerType("mobile:BasicMobilePane");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   *
   * @param {Object} dom DOM element to append content too.
   */
  BasicMobilePane.prototype.load = function(dom) {
    this.$dom = $("<div class='basicMobilePane'></div>");
    this.$dom.appendTo(dom);
  };
  
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  BasicMobilePane.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    var dom = this.$dom;
    
    // Update padding
    if (hasUpdate("padding")) {
      this.getPadding().update(dom, 0, "padding");
    }
  
    // Update row gap
    if (hasUpdate("rowGap")) {
      var rowGap = this.getRowGap() + "px";
    
      // Set the pane's margin bottom and top to each child
      dom.children().each(function() {
        $(this).css("margin-bottom", rowGap);
      });
    }
  
    // Call Super update method
    BasicMobilePane.$super.prototype.update.apply(this, arguments);
  };
      
 /**
  * @class MobileGridPane.
  * <p>
  * The mobile pane Widget that represents Niagara's 'mobile:MobileGridPane' Type.
  * 
  * @name MobileGridPane
  * @extends Widget
  */
  var MobileGridPane = function() {
    MobileGridPane.$super.apply(this, arguments);
    this.$kids = null;
    this.$tableDom = null;
  }.$extend(Widget).registerType("mobile:MobileGridPane");
       
 /**
  * Append the Widget's DOM content onto the specified DOM element.
  * <p>
  *
  * @param {Object} dom DOM element to append content too.
  */
  MobileGridPane.prototype.load = function(dom) {
   var colCount = this.getColumnCount(),
       colPercent = 100 / colCount,
       count = 0,
       html = "<div data-role='none' class='mobileGridPane'><table data-role='none' class='mobileGridPaneTable' cellpadding='0' cellspacing='0'><tbody>",
       kids = this.$kids = [],
       tableDiv;
   
    // Create html for table...
   this.getSlots().properties().is(widgetType).eachValue(function (widget) {
     kids.push(widget);
     
     if (count === 0) {
       html += "<tr>";
     }
     
     html += "<td class='mobileGridPaneData' style='width: " + colPercent + "%;'></td>";
             
     if (++count === colCount) {
       html += "</tr>";
       count = 0;
     }
   }); 
  
   if (count > 0 && kids.length > 0) {
     html += "</tr>";
   }    
  
   html += "</tbody></table></div>"; 
  
   // Create and attach to the DOM
   tableDiv = $(html);
   tableDiv.appendTo(dom);
   this.$dom = tableDiv;
   this.$tableDom = tableDiv.children(".mobileGridPaneTable");
  };
  
 /**
  * Append the Widget's DOM content onto the specified DOM element.
  */
  MobileGridPane.prototype.loadChildren = function() {
   // Build up the HTML for the jQuery Mobile column grid
   var kids = this.$kids,
       i = 0;
       
   this.$tableDom.find("td.mobileGridPaneData").each(function () {   
     kids[i].load($(this));
     kids[i++].loadChildren();
   });
   
   // Listen for any updates to any vertical alignment changes... 
   baja.iterate(kids, function (k) {
     if (k.getDomElement() && k.has("valign")) {
       var update = k.update;
       k.update = function (hasUpdate) {
         // First call the original update
         update.apply(k, arguments);
         
         if (hasUpdate("valign")) {
           // If the vertical alignment is being updated then
           var kd = k.getDomElement(),
               td = kd.parent();
           
           td.css("vertical-align", kd.css("vertical-align"));
         }
       };
     }
   });
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  MobileGridPane.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    
    var dom = this.$dom,
        tableDom = this.$tableDom,
        rows,
        gap;
    
    // Update Row Gap
    if (hasUpdate("rowGap")) {
      gap = (this.getRowGap() / 2) + "px";
      rows = tableDom.children("tbody").children("tr");
      
      // Apply top and bottom margin to all data
      rows.children("td").css({
        "padding-top": gap,
        "padding-bottom": gap
      });
      
      // No top margin on first row
      rows.first().children("td").css("padding-top", "0px");
      
      // No bottom margin on last row
      rows.last().children("td").css("padding-bottom", "0px");
    }
    
    // Update Column Gap
    if (hasUpdate("columnGap")) {
      gap = (this.getColumnGap() / 2) + "px";
      rows = tableDom.children("tbody").children("tr");
      
      // Apply top and bottom margin to all data
      rows.children("td").css({
        "padding-left": gap,
        "padding-right": gap
      });
      
      // No left margin on first td in a row
      rows.children("td:first-child").css("padding-left", "0px");
      
      // No right margin on last td in a row
      rows.children("td:last-child").css("padding-right", "0px");
    }
    
    // Padding
    if (hasUpdate("padding")) {
      this.getPadding().update(dom);
    }
    
    // Rounded corners
    if (hasUpdate("roundedCorners")) {
      if (this.getRoundedCorners()) {
        dom.css({
          "-moz-border-radius": ".6em",
          "border-radius": ".6em"
        });
      }
      else {
        dom.css({
          "-moz-border-radius": "",
          "border-radius": ""
        });
      }
    }
    
    // Shadow
    if (hasUpdate("shadow")) {
      if (this.getShadow()) {
        dom.css({
          "-moz-box-shadow": "0 0 5px #888",
          "-webkit-box-shadow": "0 0 5px #888",
          "box-shadow": "0 0 5px #888"
        });
      }
      else {
        dom.css({
          "-moz-box-shadow": "",
          "-webkit-box-shadow": "",
          "box-shadow": ""
        });
      }
    }
      
    // Background (will override theme's background color)
    if (hasUpdate("background")) {
      this.getBackground().update({ 
        dom: dom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    // Foreground (will color the border)
    if (hasUpdate("foreground")) {
      this.getForeground().update({ 
        dom: dom, 
        colorSelector: "border",
        prefix: "1px solid "
      });
    }
                                
    // Call Super update method
    MobileGridPane.$super.prototype.update.apply(this, arguments);
  };
    
  /**
   * @class BorderPane.
   * <p>
   * The mobile pane Widget that represents Niagara's 'bajaui:BorderPane' Type.
   * 
   * @name BorderPane
   * @extends Widget
   */
  var BorderPane = function() {
    BorderPane.$super.apply(this, arguments);
  }.$extend(Widget).registerType("bajaui:BorderPane");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   *
   * @param {Object} dom DOM element to append content too.
   */
  BorderPane.prototype.load = function(dom) {
    this.$dom = $("<div class='borderPane'></div>");
    this.$dom.appendTo(dom);
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  BorderPane.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    var dom = this.$dom;
    
    // TODO: label, border

    // Fill
    if (hasUpdate("fill")) {
      this.getFill().update({ 
        dom: dom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    // Margin
    if (hasUpdate("margin")) {
      this.getMargin().update(dom, 0, "margin");
    }
    
    // Padding
    if (hasUpdate("padding")) {
      this.getPadding().update(dom, 0, "padding");
    }
    
    // Call Super update method
    BorderPane.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * @class ScrollPane.
   * <p>
   * The mobile pane Widget that represents Niagara's 'bajaui:ScrollPane' Type.
   * 
   * @name ScrollPane
   * @extends Widget
   */
  var ScrollPane = function() {
    ScrollPane.$super.apply(this, arguments);
  }.$extend(Widget).registerType("bajaui:ScrollPane");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   *
   * @param {Object} dom DOM element to append content too.
   */
  ScrollPane.prototype.load = function(dom) {
    var newDom = $("<div class='scrollPane'></div>"),
        parent = this.getParent();
    
    if (parent && parent.getType().is('bajaui:CanvasPane')) {
      touchScroll(newDom, { 
        horiz: true, 
        vert: true, 
        showBlockade: false
      });
    }
    
    newDom.appendTo(dom);
    this.$dom = newDom;
  };
  
  function getOverflowVal(scrollBarPolicy) {
    var overflowVal;
      
    if (scrollBarPolicy.is("always")) {
      overflowVal = "scroll";
    }
    else if (scrollBarPolicy.is("asNeeded")) {
      overflowVal = "auto";
    }
    else {
      overflowVal = "hidden";
    }
    
    return overflowVal;
  }
  
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  ScrollPane.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    var dom = this.$dom;
    
    // TODO: viewportBackground, borderPolicy
    
    if (hasUpdate("hpolicy")) {
      dom.css("overflow-x", getOverflowVal(this.getHpolicy()));
    }
    
    if (hasUpdate("vpolicy")) {
      dom.css("overflow-y", getOverflowVal(this.getVpolicy()));
    }
        
    // Call Super update method
    ScrollPane.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * Update the positioning of the child Widget
   *
   * @param {Widget} childWidget the Child Widget to be positioned.
   */
  ScrollPane.prototype.layoutChild = function (childWidget) {
    // Set the child Widget to its preferred height and width.
    childWidget.getDomElement().css({
      "width": childWidget.getPreferredWidth(),
      "height": childWidget.getPreferredHeight()
    });
  };
  
  
  
  /**
   * Applies the appropriate dimension layout for the canvas pane's scale mode.
   * Works in both absolute and fluid modes, but as of right now, only
   * 'fitRatio' and 'none' are appropriate values for fluid.
   * 
   * @private
   * @inner
   * @ignore
   * @param {CanvasPane} canvas
   * @param {Boolean} [looping] sometimes scroll bars appear in a fluid layout
   * which necessitates a recalc - safety measure to prevent infinite loop 
   */
  function scaleCanvasPane(canvas, looping) {
    
    var dom = canvas.getDomElement(),
        isAbsolute = canvas.isAbsolutelyPositioned(),
        inner = canvas.$inner,
        parent = dom.parent(),
        domWidth = dom.width(),
        parentWidth = parent.width(),
        viewSize = canvas.getViewSize(),
        layout = canvas.getLayout(),
        outerWidth,
        outerHeight,
        viewWidth = viewSize.getWidth(),
        viewHeight = viewSize.getHeight(),
        viewAspectRatio,
        scale = canvas.getScale().getTag(),
        css;
    
    // If the CanvasPane is absolutely positioned then use the layout
    if (isAbsolute) {
      if (layout === niagara.px.Layout.FILL) {
        outerWidth = viewWidth;
        outerHeight = viewHeight;
      } else {
        outerWidth = layout.getWidth();
        outerHeight = layout.getHeight();
      }
    } else { // otherwise use the DOM for width
      outerWidth = domWidth;

      if (scale === 'fitRatio') {
        viewAspectRatio = viewWidth / viewHeight;
        outerHeight = outerWidth / viewAspectRatio;
      } else {
        // ONLY fitRatio and none are supported in fluid layout.
        scale = 'none';
        outerHeight = viewHeight;
      }
    }
                
    css = getScaledLayoutCSS(
      outerWidth,
      outerHeight,
      viewWidth,  //innerWidth 
      viewHeight, //innerHeight
      canvas.getHalign().getTag(), //halign
      canvas.getValign().getTag(), //valign
      scale);
    
    addScaling(css, css.width / viewWidth, css.height / viewHeight);

    inner.css(css);
    
    /*
     * Sometimes, setting the canvas pane dimensions in a fluid page adds a 
     * scroll bar. this can change our domWidth, but innerCanvasPane
     * must then be recalculated.
     */
    if (!isAbsolute && parentWidth !== parent.width()) {
      if (!looping) { //only try once
        scaleCanvasPane(canvas, true);
      }
    }
  }
  
  /**
   * If the canvas pane is in a fluid layout, its inner div is first set to
   * width/height of 0 (so that its containing grid can re-calc column widths).
   * This is needed because scaling in a fluid layout depends on the canvas
   * pane's parent's dimensions. Then asynchronously performs scaling.
   * 
   * @private
   * @inner
   * @ignore
   * @param {CanvasPane} canvas
   */
  function performCanvasPaneRescale(canvas) {
    if (canvas.isAbsolutelyPositioned()) {
      scaleCanvasPane(canvas);
    } else {
      var inner = canvas.$inner;
      
      inner.css({ width: 0, height: 0 });
      
      //run async so we get a browser reflow. it's a little clunky but should
      //only happen upon window resize
      baja.runAsync(function () {
        scaleCanvasPane(canvas);
      });
    }
  }
  
  /**
   * @class CanvasPane.
   * <p>
   * The mobile pane Widget that represents Niagara's 'bajaui:CanvasPane' Type.
   * <p>
   * This Widget will absolutely position Widgets in a Px page.
   * 
   * @name CanvasPane
   * @extends Widget
   */
  var CanvasPane = function() {
    CanvasPane.$super.apply(this, arguments);
  }.$extend(Widget).registerType("bajaui:CanvasPane");
    
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   *
   * @param {Object} dom DOM element to append content too.
   */
  CanvasPane.prototype.load = function(dom) {
    var that = this,
        autoScale,
        rootDom,
        windowWidth = undefined;
    
    that.$dom = $("<div class='canvasPane'><div class='innerCanvasPane'></div></div>");
    that.$inner = this.$dom.find(".innerCanvasPane");
    that.$dom.appendTo(dom);
    
    autoScale = baja.throttle(function autoScale() {
      var w = $(window).width();
      if (w !== windowWidth) {
        performCanvasPaneRescale(that);
      }
      windowWidth = w;
    }, 500);
      
    // Only do this if the Widget is not absolutely positioned. We do this so the
    // CanvasPane will automatically shrink and contract to the window.    
    if (!that.isAbsolutelyPositioned()) {
      rootDom = that.getRootContainer().getDomElement();
     
      rootDom.one("pxPageLoaded", autoScale);
      
      // When the page is started, register the auto scale function on window resize
      rootDom.bind("pxPageStarted", function () {
        $(window).resize(autoScale);
      });
      
      // When the page is stopped remove this resize handler from the window
      rootDom.bind("pxPageStopped", function () {
        $(window).unbind("resize", autoScale);
      });
    }
  };
  
  /**
   * Called after the Widget has loaded and any enhancements have 
   * been made to the DOM.
   */
  CanvasPane.prototype.postLoad = function() { 
    // Set the z-order of the Widgets according their Slot order
    var kids = this.getSlots().is("bajaui:Widget").toValueArray(),
        i,
        dom;
            
    for (i = 0; i < kids.length; ++i) {
      dom = kids[i].getDomElement();
      if (dom) {
        dom.css("z-index", i + 1); // NCCB-2278
      }
    }
  };
  
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   * @param {Boolean} [firstUpdate] set to true on Widgets first update
   *                                when the page is first loaded.
   */
  CanvasPane.prototype.update = function(hasUpdate, firstUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    
    var that = this,
        inner = that.$inner,
        dom = that.$dom;
    
    // Background color
    if (hasUpdate("background")) {
      that.getBackground().update({ 
        dom: dom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    

    // View Size
    if (hasUpdate("viewSize")) {
      performCanvasPaneRescale(that);
      // If there is a parent then also update the layout
      if (!firstUpdate && that.getParent() && that.getDomElement()) {
        // If the parent Widget supports laying out children then call it
        that.getParent().layoutChild(that);
      }
    } else if (hasUpdate("halign") || 
        hasUpdate("valign") || hasUpdate("scale")) {
      performCanvasPaneRescale(that);
    }

    
    // Call Super update method
    CanvasPane.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * Update the positioning of the child Widget
   *
   * @param {Widget} childWidget the Child Widget to be positioned.
   */
  CanvasPane.prototype.layoutChild = function (childWidget) {
    childWidget.getLayout().update(childWidget.getDomElement());
    
    // TODO: Z-order, change of Z-order etc.
  };
  
  /**
   * Return the preferred Width of the CanvasPane.
   *
   * @returns {Number} the width in pixels.
   */
  CanvasPane.prototype.getPreferredWidth = function() {
    return this.getViewSize().getWidth();
  };
  
  /**
   * Return the preferred Height of the CanvasPane.
   *
   * @returns {Number} the height in pixels.
   */
  CanvasPane.prototype.getPreferredHeight = function() {
    return this.getViewSize().getHeight();
  };
       
  /**
   * Load the child Widgets.
   * <p>
   * {@link baja.Widget#load} must be called before this method is invoked.
   */
  CanvasPane.prototype.loadChildren = function() {    
    loadChildren.call(this, this.$inner);
  };     
       
  /**
   * @class IgnoreWidget.
   * <p>
   * There are several Widgets we don't want to throw errors for when loading and ignore.
   * This JS Constructor is registered on these Widgets.
   *
   * @name IgnoreWidget
   * @extends Widget
   */
  var IgnoreWidget = function() {
    IgnoreWidget.$super.apply(this, arguments);
  }.$extend(Widget).registerType("bajaui:ScrollBar").registerType("bajaui:NullWidget").registerType("bajaui:Shape");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  IgnoreWidget.prototype.load = noop;
  
  /**
   * Load the child Widgets.
   * <p>
   * {@link baja.Widget#load} must be called before this method is invoked.
   */
  IgnoreWidget.prototype.loadChildren = noop;    
  
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  IgnoreWidget.prototype.update = noop;
  
  ////////////////////////////////////////////////////////////////
  // Button/Label Util Methods
  ////////////////////////////////////////////////////////////////  
    
  function processText(str) {
    // Replace newlines with HTML line breaks
    return str.replace(/\n/g, "<br />");
  }
  
  function updateText(widget) {
    var text = processText(widget.getText());
    
    widget.$textDom.text(text);
    
    // The text DOM being hidden or shown is done further in textToIconAlign...
    if (text === "") {
      if (widget.$textDomShown) {
        widget.$textDomShown = false;
        return true;
      }
    }
    else {
      if (!widget.$textDomShown) {
        widget.$textDomShown = true;
        return true;
      }
    }
  }

  /**
   * NCCB-5908: webkit strikes again with poor SVG rescaling. the hide()
   * and offset() force a browser reflow as a workaround for the following
   * bug:
   * https://code.google.com/p/chromium/issues/detail?id=269446
   *
   * @param dom
   * @param src
   * @private
   * @inner
   */
  function setImageSrc(dom, src) {
    var visible = dom.is(':visible'),
        isSvg = src.toLowerCase().lastIndexOf('.svg') === src.length - 4;

    if (isSvg) {
      dom.hide().offset();
    }

    dom.attr("src", src);

    if (isSvg) {
      dom.toggle(visible);
    }
  }
  
  function getImageSrc(widget) {
    var imageStr = widget.getImage().toString();
    return imageStr && imageStr !== "null" ? "/ord?" + imageStr : "";
  }
  
  function updateImage(widget) {
    var imageSrc = getImageSrc(widget),
        imgDom = widget.$imgDom;

    if (imageSrc) {

      setImageSrc(imgDom, imageSrc);

      if (!widget.$imgDomShown) {
        imgDom.show();
        widget.$imgDomShown = true;
        return true;
      }
    }
    else {
      if (widget.$imgDomShown) {
        imgDom.hide();
        widget.$imgDomShown = false;
        return true;
      }
    }
    
    // If the text to icon alignment is center then we also need to indicate
    // the image show state has changed
    return widget.getTextToIconAlign().is("center");
  }
  
  function updateImagePosition(widget, dom) {
    var domContents = dom.children().remove(),
        textDom = widget.$textDom.remove(),
        imgDom = widget.$imgDom.remove(),
        tAlign = widget.getTextToIconAlign(),
        gap = widget.$imgDomShown && widget.$textDomShown ? String(widget.getTextIconGap()) + "px" : "0px",
        imgSrc = getImageSrc(widget),
        img,
        imgCell;
                      
    // Reference Margin CSS: top, right, bottom, left 
    
    if (tAlign.is("center")) {
      textDom.css({
        "margin": 0,
        "display": "block",
        "text-align": "center"
      });
      imgCell = $('<div/>')
        .css({
          "display": "table-cell",
          "vertical-align": "middle",
          "background-image": imgSrc ? "url('" + imgSrc + "')" : "",
          "background-repeat": "no-repeat",
          "background-position": "center"
        })
        .append(textDom);
      domContents = $('<div/>')
        .css({
          display: "inline-block",
          "vertical-align": "inherit"
        })
        .append(imgCell);
      
      if (imgSrc) {
        img = new Image();
        img.onload = function () {
          imgCell.css({ 
            "min-width": img.width,
            height: img.height
          });
        };
        img.src = imgSrc;
      }
    }
    else if (tAlign.is("right")) {
      textDom.css({
        display: widget.$textDomShown ? "inline" : "none",
        margin: "0px 0px 0px " + gap
      });
      domContents = imgDom.add(textDom);
    }
    else if (tAlign.is("left")) {
      textDom.css({
        display: widget.$textDomShown ? "inline" : "none",
        margin: "0px " + gap + " 0px 0px"
      });
      domContents = textDom.add(imgDom);
    }
    else if (tAlign.is("top")) {
      textDom.css({
        display: widget.$textDomShown ? "block" : "none",
        margin: "0px 0px " + gap + " 0px"
      });
      domContents = textDom.add(imgDom);
    }
    else if (tAlign.is("bottom")) {
      textDom.css({
        display:  widget.$textDomShown ? "block" : "none",
        margin: gap + " 0px 0px 0px"
      });
      domContents = imgDom.add(textDom);
    }
    
    dom.html(domContents);
  }
    
  ////////////////////////////////////////////////////////////////
  // Core Widgets
  ////////////////////////////////////////////////////////////////  
  
  /**
   * @class Button.
   *
   * The Widget that represents Niagara's 'bajaui:Button' Type.
   *
   * @name Button
   * @extends Widget
   */
  var Button = function() {
    Button.$super.apply(this, arguments);
    this.$blinkTicket = baja.clock.expiredTicket;
    this.$imgDom = null;
    this.$textDom = null;
    this.$buttonDom = null;
    this.$innerDom = null;
    this.$textDomShown = true;
    this.$imgDomShown = true;
  }.$extend(Widget).registerType("bajaui:Button");
  
  /**
   * Called after the Widget has loaded and any enhancements have 
   * been made to the DOM.
   */
  Button.prototype.postLoad = function() { 
    this.$buttonDom = this.$dom;
    
    // After the enhancement, the top level DOM element has changed
    // so we need to set the parent DOM as the base DOM element for the Widget
    var dom = this.$dom = this.$dom.parent();
    
    dom.addClass("widgetButton");    
    this.$textDom = dom.find("span.ui-btn-text");
    
    // Create image DOM element after page has been enhanced
    this.$imgDom = $("<img src='#'></img>");
    this.$innerDom = dom.find("span.ui-btn-inner").first();
    this.$innerDom.wrapAll("<div class='widgetInnerOuter'></div>");
    this.$innerDom.prepend(this.$imgDom);
        
    if (this.isAbsolutelyPositioned()) {
      this.$innerDom.css({
        "display": "table-cell",
        "width": "100%",
        "padding": "4px 7px"
      });
    }
    else {
      this.$innerDom.css("padding", ".6em");
    }
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  Button.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Call Super update method
    Button.$super.prototype.update.apply(this, arguments);
    
    // Need to get parent when manipulating this in jQuery Mobile
    var dom = this.$dom,
        innerDom = this.$innerDom,
        textDom = this.$textDom,
        imgDom = this.$imgDom,
        textShowStateChanged = false,
        imgShowStateChanged = false;
    
    // Font
    if (hasUpdate("font")) {
      this.getFont().update(textDom);
    }
    
    // Text (textFormat is for LocalizedLabel)
    if (hasUpdate("text") || hasUpdate("textFormat")) {
      textShowStateChanged = updateText(this);
    }

    // Image
    if (hasUpdate("image")) {
      imgShowStateChanged = updateImage(this);
    }
                
    // Text To Icon Align and Text Icon Gap (or if the hidden state of the text or image elements have changed)
    if (hasUpdate("textToIconAlign") || hasUpdate("textIconGap") || textShowStateChanged || imgShowStateChanged) {
      updateImagePosition(this, innerDom);
    }
    
    // Horizontal alignment
    if (hasUpdate("halign")) {
      var halign = getHalign(this);
      dom.css("text-align", halign);
      imgDom.css("text-align", halign);
    }
    
    // Vertical alignment
    if (hasUpdate("valign")) {
      var valign = getValign(this);
      dom.css("vertical-align", valign);
      innerDom.css("vertical-align", valign);
      imgDom.css("vertical-align", valign);
    }
    
    // Enabled  
    if (hasUpdate("enabled")) {      
      this.$buttonDom.button(this.isEnabled() ? "enable" : "disable");
    }
    
    // Visibility (need to hide parent)
    if (hasUpdate("visible")) {
      if (this.getVisible()) {
        dom.show();
      }
      else {
        dom.hide();
      }
    }
    
    // Blink
    if (hasUpdate("blink")) {    
      if (this.getBlink()) {
        if (this.$blinkTicket.isExpired()) {  
          // Create the blink timer
          this.$blinkTicket = baja.clock.schedulePeriodically(function () {
            var visible = innerDom.css("visibility") === "visible";
            visible = !visible;
            visible = visible ? "visible" : "hidden";
            innerDom.css("visibility", visible);
          }, 1000);
        }
      }
      else {
        this.$blinkTicket.cancel();
        innerDom.css("visibility", "visible");
      }
    }
    
    // Background color
    if (hasUpdate("background")) {
      this.getBackground().update({ 
        dom: dom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    // Text color
    if (hasUpdate("foreground")) {
      this.getForeground().update({ 
        dom: dom, 
        colorSelector: "color" 
      });
    }
  };
    
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  var buttonHtml = "<button type='button' data-theme='a'></button>";
  Button.prototype.load = function(dom) {
  
    // Create the button DOM object
    var that = this,
        button = that.$dom = $(buttonHtml);
        
    // Set up click handler
    button.click(function () {
      that.handleEvent("click");
    });
       
    // Append to the DOM    
    button.appendTo(dom);
  };
  
  /**
   * @class Image Button.
   *
   * The Widget that represents Niagara's 'kitPx:ImageButton' Type.
   *
   * @name ToggleButton
   * @extends Widget
   */
  var ImageButton = function() {
    ImageButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:ImageButton");
      
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  ImageButton.prototype.update = function(hasUpdate) { 
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Call Super update method
    ImageButton.$super.prototype.update.apply(this, arguments);
    
    var textDom = this.$textDom,
        imgDom = this.$imgDom,
        normalStr,
        disabledStr,
        imageStr,
        isDisabled = false;
        
    // Normal or disabled
    if (hasUpdate("normal") || hasUpdate("disabled")) {
      disabledStr = this.getDisabled().toString();
      if (!(disabledStr === "" || disabledStr === "null") && !this.isEnabled()) {
        normalStr = disabledStr;
        isDisabled = true;
      }
      else {
        normalStr = this.getNormal().toString();
      }
      
      if (normalStr === "" || normalStr === "null") {
        // Image
        imageStr = this.getImage().toString();
        if (imageStr === "" || imageStr === "null") {
          imgDom.hide();
        }
        else {
          imgDom.show();
        }
        textDom.show();
      }
      else {
        // Cancel any blinking since this doesn't work in Workbench Px anyway
        this.$blinkTicket.cancel();
        
        if (isDisabled) {
          textDom.show();
        }
        else {
          textDom.hide();
        }

        setImageSrc(imgDom, "/ord?" + normalStr.toString());
        imgDom.show();
      }
    }
  };
  
  /**
   * @class Toggle Button.
   *
   * The Widget that represents Niagara's 'bajaui:ToggleButton' and 'bajaui:CheckBox' Type.
   *
   * @name ToggleButton
   * @extends Widget
   */
  var ToggleButton = function() {
    ToggleButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("bajaui:ToggleButton").registerType("bajaui:CheckBox");
    
  /**
   * Handle a UI Widget event.
   */
  ToggleButton.prototype.doHandleEvent = function(eventName) {
    // Toggle the selection
    this.setSelected(!this.getSelected());
    
    // Update the ToggleButton's UI
    this.update(function hasUpdate(propName) {
      return propName === "selected";
    });
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  ToggleButton.prototype.update = function(hasUpdate) {  
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Call Super update method
    ToggleButton.$super.prototype.update.apply(this, arguments);
    
    // Selection
    if (hasUpdate("selected")) {
      this.$dom.css("border", this.getSelected() ? "2px solid black" : "2px solid white");
    }
  };  
  
  var labelHtml = "<div class='widgetLabel'>" + 
                    "<div class='widgetInnerOuter'>" + 
                      "<div class='widgetLabelInner'>" + 
                        "<img class='labelImage' src='#' />" + 
                        "<span class='labelText'></span>" + 
                      "</div>" +
                    "</div>" + 
                  "</div>";
    
  /**
   * @class Label.
   *
   * The Widget that represents Niagara's 'bajaui:Label' Type.
   *
   * @name Label
   * @extends Widget
   */
  var Label = function() {
    Label.$super.apply(this, arguments);
    this.$blinkTicket = baja.clock.expiredTicket;
    this.$imgDom = null;
    this.$textDom = null;
    this.$textDomShown = true;
    this.$imgDomShown = true;
    this.$innerDom = null;
  }.$extend(Widget).registerType("bajaui:Label");
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  Label.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    
    var that = this,
        dom = that.$dom,
        imgDom = that.$imgDom,
        textDom = that.$textDom,
        innerDom = that.$innerDom,
        textShowStateChanged = false,
        imgShowStateChanged = false;
      
    // Call Super update method
    Label.$super.prototype.update.apply(that, arguments);
        
    // Blink
    if (hasUpdate("visible") || hasUpdate("blink")) {
      if (that.getBlink() && that.getVisible()) { 
        if (that.$blinkTicket.isExpired()) {  
          // Create the blink timer
          that.$blinkTicket = baja.clock.schedulePeriodically(function () {
            var visible = dom.css("visibility") === "visible";
            visible = !visible;
            dom.css("visibility", visible ? "visible" : "hidden");
          }, 1000);
        }
      }
      else {
        // Cancel any outstanding ticket
        that.$blinkTicket.cancel();
        
        if (that.getVisible()) {
          // Show the element
          dom.css("visibility", "visible");
        }
      }
    }
    
    // Font
    if (hasUpdate("font")) {
      that.getFont().update(textDom);
    }
    
    // Text (textFormat is for LocalizedLabel)
    if (hasUpdate("text") || hasUpdate("textFormat")) {
      textShowStateChanged = updateText(that);
    }

    // Image
    if (hasUpdate("image")) {
      imgShowStateChanged = updateImage(that);
    }
                
    // Text To Icon Align and Text Icon Gap (or if the hidden state of the text or image elements have changed)
    if (hasUpdate("textToIconAlign") || hasUpdate("textIconGap") || textShowStateChanged || imgShowStateChanged) {
      updateImagePosition(that, innerDom);
    }
    
    // Horizontal alignment
    if (hasUpdate("halign")) {
      var halign = getHalign(that);
      dom.css("text-align", halign);
      innerDom.css("text-align", halign);
      imgDom.css("text-align", halign);
    }
    
    // Vertical alignment
    if (hasUpdate("valign")) {
      var valign = getValign(that);
      dom.css("vertical-align", valign);
      imgDom.css("vertical-align", valign);
      innerDom.css("vertical-align", valign);
      textDom.css("vertical-align", valign);
    }
        
    // Background color
    if (hasUpdate("background")) {
      that.getBackground().update({ 
        dom: dom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    // Text color
    if (hasUpdate("foreground")) {
      that.getForeground().update({
        dom: dom, 
        colorSelector: "color"
      });
    }
  };
            
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  Label.prototype.load = function(dom) {
    // Create the DOM object
    var that = this,
        label = that.$dom = $(labelHtml);
                             
    that.$textDom = label.find("span.labelText");
    that.$imgDom = label.find("img.labelImage");
    that.$innerDom = label.find("div.widgetLabelInner");
    
    if (!this.isAbsolutelyPositioned()) {
      label.css("padding", "7px 7px");
    }
    
    // Append to the DOM    
    label.appendTo(dom);
                
    // Set up click handler
    label.click(function () {
      that.handleEvent("click");
      
      // Return false to override default click behavior
      return false;
    });
  };
  
  /**
   * @class BoundLabel.
   *
   * The Widget that represents Niagara's 'kitPx:BoundLabel' Type.
   *
   * @name BoundLabel
   * @extends Widget
   */
  var BoundLabel = function() {
    BoundLabel.$super.apply(this, arguments);
  }.$extend(Label).registerType("kitPx:BoundLabel");
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  BoundLabel.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Call Super update method
    BoundLabel.$super.prototype.update.apply(this, arguments);

    // Padding
    if (hasUpdate("padding")) {
      this.getPadding().update(this.$dom, /*excess*/this.isAbsolutelyPositioned() ? 0 : 7);
    }
  };
   
  /**
   * @class HyperlinkLabel
   *
   * The Widget that represents Niagara's 'bajaui:HyperlinkLabel' Type.
   *
   * @name HyperlinkLabel
   * @extends Label
   */
  var HyperlinkLabel = function() {
    HyperlinkLabel.$super.apply(this, arguments);
  }.$extend(Label).registerType("bajaui:HyperlinkLabel");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  HyperlinkLabel.prototype.load = function(dom) {    
    HyperlinkLabel.$super.prototype.load.apply(this, arguments);
    
    if (this.getOrd() !== baja.Ord.DEFAULT) {
      this.addMouseOver();
    }
  };
      
  /**
   * Handle a UI Widget event.
   */
  HyperlinkLabel.prototype.doHandleEvent = function(eventName) {
    var hlink = this.getOrd();
    if (hlink !== baja.Ord.DEFAULT) {
      pxUtilMobile.hyperlink(this, this.getOrd());
      
      // Return true so the event doesn't filter down to the bindings
      return true;
    }
  };
      
  /**
   * @class Slider.
   *
   * The Widget that represents Niagara's 'bajaui:Slider' Type.
   *
   * @name Slider
   * @extends Widget
   */
  var Slider = function() {
    Slider.$super.apply(this, arguments);
    this.$input = null;
    this.$fromUpdate = false;
    this.$sliderId = null;
  }.$extend(Widget).registerType("bajaui:Slider");
  
  /**
   * Called after the Widget has loaded and any enhancements have 
   * been made to the DOM.
   */
  Slider.prototype.postLoad = function() {
    var that = this,
        input = that.$input = $("#" + that.$sliderId, this.$dom.parent());
                
    // Throttle this so it doesn't happen too often
    var throttledFunc = baja.throttle(function () {
      // Update the Widget with the new value
      that.setValue(Number.DEFAULT.decodeFromString(input.attr("value")));
      that.handleEvent("slide");
    }, 500);
    
    // Set up click handler
    input.bind("change", function (event, ui) {
      // Only fire this event if it hasn't been fired as a result of an update.
      if (!that.$fromUpdate) {
        throttledFunc();
      }
    });
    
    this.$dom.find("div.ui-slider").css("width", "90%");
  };
  
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  Slider.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Call Super update method
    Slider.$super.prototype.update.apply(this, arguments);
    
    var that = this,
        input = that.$input;
            
    if (hasUpdate("enabled")) {
      input.pxslider(this.isEnabled() ? "enable" : "disable");
    }
    
    if (hasUpdate("min")) {
      input.attr("min", this.getMin());
    }    
    
    if (hasUpdate("max")) {
      input.attr("max", this.getMax());
    } 
    
    if (hasUpdate("value")) {
      input.attr("value", this.getValue());
    } 
    
    if (hasUpdate("increment")) {
      input.attr("step", this.getIncrement());
    }
            
    try {
      // Signal the refresh is coming from the result of an update
      this.$fromUpdate = true;
      input.pxslider('refresh', this.getValue());
    }
    finally {
      this.$fromUpdate = false;
    }
  };
        
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @name Slider#load
   * @function
   *
   * @param {Object} dom DOM element to append content too.
   */  
  var sliderIdCounter = 0,
      sliderHtml = "<div class='mobileSlider' style='text-align: center' data-role='fieldcontain'>" +
                     "<label style='display: none;' for='{id}'></label>" +
                     "<input type='pxrange' name='slider' id='{id}' value='0' min='{min}' " +
                     "max='{max}' step='{increment}' style='display: none;' data-theme='a' />" +
                   "</div>";
  Slider.prototype.load = function(dom) {
    // Create a unique slider id
    this.$sliderId = "slider" + sliderIdCounter++;
  
    // Create the DOM object
    var slider = this.$dom = $(sliderHtml.patternReplace({
      id: this.$sliderId,
      min: this.getMin().toString(),
      max: this.getMax().toString(),
      increment: this.getIncrement().toString()      
    }));
    
    // Append to the DOM    
    slider.appendTo(dom);
  };
     
  /**
   * @class GenericFieldEditor
   *
   * The Widget that represents Niagara's 'kitPx:GenericFieldEditor' Type.
   *
   * @name GenericFieldEditor
   * @extends Widget
   */
  var GenericFieldEditor = function() {
    GenericFieldEditor.$super.apply(this, arguments);
    this.$fieldEditor = null;
  }.$extend(Widget).registerType("kitPx:GenericFieldEditor");
    
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */  
  GenericFieldEditor.prototype.load = function(dom) {
    this.$dom = $('<div class="widgetFieldEditor" />').appendTo(dom);
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  GenericFieldEditor.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
  
    if (hasUpdate("enabled") && this.$fieldEditor) {
      this.$fieldEditor.setReadonly(!this.isEnabled());
    }
  
    // Call Super update method
    GenericFieldEditor.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * Return true if the Widget (and all parent Widgets) are enabled.
   *
   * @return {Boolean}
   */
  GenericFieldEditor.prototype.isEnabled = function() {
    // Check whether we can read the given Slot
    var enabled = true;
    this.getSlots().is("workbench:WbFieldEditorBinding").eachValue(function(binding) {
      var target = binding.getTarget(),
          slot;
      
      if (target) {
        if (target.slot) {
          slot = target.slot;
        }
        if (slot && (slot.getFlags() & baja.Flags.READONLY) === baja.Flags.READONLY) {
          enabled = false;
        }
      }
      
      // Return true because we don't want to iterate through anything else
      return true;
    });
            
    // If enabled, call Super isEnabled method
    return enabled ? GenericFieldEditor.$super.prototype.isEnabled.apply(this, arguments) : enabled;
  };
  
  /**
   * Return whether the Widget is modified or not.
   *
   * @returns {Boolean}
   */  
  GenericFieldEditor.prototype.isModified = function() {
    return this.$fieldEditor && this.$fieldEditor.isModified();
  };
  
  /**
   * Clear the modified state of the Widget.
   */
  GenericFieldEditor.prototype.clearModified = function() {
    if (this.$fieldEditor) {
      this.$fieldEditor.setModified(false, {silent: true});
    }
  };
  
  /**
   * Save the Widget
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {Function} obj.ok the ok callback handler.
   * @param {Function} ok.fail the fail callback handler.
   * @param {baja.comm.Batch} [obj.batch]
   */  
  GenericFieldEditor.prototype.save = function(obj) {    
    var fe = this.$fieldEditor;
    if (fe && fe.isModified()) {
      fe.saveValue(obj);
    } else {
      obj.ok();
    }
  };
  
  /**
   * Validate the widget's field editor
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {Function} obj.ok the ok callback handler.
   * @param {Function} ok.fail the fail callback handler.
   * @param {baja.comm.Batch} [obj.batch]
   */
  GenericFieldEditor.prototype.validate = function (obj) {
    var fe = this.$fieldEditor;
    if (fe) {
      fe.validate(obj);
    } else {
      obj.ok();
    }
  };
    
  /**
   * @class SetPointFieldEditor
   *
   * The Widget that represents Niagara's 'kitPx:SetPointFieldEditor' Type.
   *
   * @name SetPointFieldEditor
   * @extends GenericFieldEditor
   */
  var SetPointFieldEditor = function() {
    SetPointFieldEditor.$super.apply(this, arguments);
    this.$loadingFe = false;
  }.$extend(GenericFieldEditor).registerType("kitPx:SetPointFieldEditor");
    
  var loadSetPointFieldEditorWorkflow = util.flow.sequential(
    function step1__makeFieldEditor(cx) {
      niagara.fieldEditors.makeFor({
        value: cx.value,
        facets: cx.facets,
        readonly: cx.readonly
      }, this);
    },
    function step2__buildAndLoad(cx, editor) {
      editor.buildAndLoad(cx.targetElement, this);
    }
  );
  
  /**
   * Load the field editor from the given binding.
   * 
   * @param binding.
   */
  SetPointFieldEditor.prototype.loadSetPoint = function(binding) {
    var fe = this.$fieldEditor;
    
    // don't reload if currently modified
    if (fe && fe.isModified()) {
      return;
    }
    
    // If we're in the middle of asynchonously loading the field editor then bail
    if (this.$loadingFe) {
      return;
    }
    
    // map binding to a value/facets           
    var value = binding.getTarget().getObject(),
        facets = util.slot.getFacets(value),
        that = this;
    
    if (value.getType().is("baja:IBoolean")) {
      value = Boolean.getBooleanFromIBoolean(value);
    }            
    else if (value.getType().is("baja:INumeric")) {
      value = Number.getNumberFromINumeric(value);
    }            
    else if (value.getType().is("baja:IEnum")) {
      value = baja.Enum.getEnumFromIEnum(value);
    }         
    else if (value.getType().is("baja:IStatusValue")) { 
      // assume string point 
      value = value.getValue();
    } 
    
    // Lazily create or load the field editor
    if (!fe) {
      
      // Signal we're loading the field editor
      this.$loadingFe = true;
      
      loadSetPointFieldEditorWorkflow.invoke({
        value: value,
        facets: facets,
        targetElement: this.$dom,
        readonly: !that.isEnabled()
      }, {
        ok: function (fe) {          
          that.$fieldEditor = fe;
          
          // Set the field editor to readonly if need be
          //fe.setReadonly(!that.isEnabled()); //done in constructor
                    
          // Signal that we've loaded the field editor
          that.$loadingFe = false;
           
          // Update binding since the field editor is loaded asynchronously and
          // the target may have changed. Therefore, we should reload the value.
          binding.update();
        },
        fail: function (err) {
          baja.error("Failed to create field editor: " + err);
        }
      });
    }
    else {
      fe.facets = facets;
      fe.loadValue(value);
    }
  };
  
  /**
   * Save the Widget
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {Function} obj.ok the ok callback handler.
   * @param {Function} ok.fail the fail callback handler.
   */  
  SetPointFieldEditor.prototype.save = function(obj) {  
    var fe = this.$fieldEditor, that = this;
    
    if (fe && fe.isModified()) {
      // Save the field editor
      fe.saveValue({
        ok: function (value) {
          var found = false;
          
          // The the SetPointBinding
          that.getSlots().properties().is("kitPx:SetPointBinding").each(function (slot) {
            // TODO: Ok, fail callbacks need to be passed through                                       
            found = true;
          
            // Call save on the SetPointBinding
            this.get(slot).saveSetPoint({
              value: value,
              ok: obj.ok,
              fail: obj.fail,
              batch: obj.batch
            });
            
            // Make sure this only happens once so return true
            return true;
          });
          
          if (!found) {
            obj.fail("Could not find valid SetPointBinding");
          }
        },
        fail: obj.fail
      });

    }
    else {
      obj.ok();
    }
  };
          
  /**
   * @class SaveButton
   *
   * The Widget that represents Niagara's 'kitPx:SaveButton' Type.
   *
   * @name SaveButton
   * @extends Button
   */
  var SaveButton = function() {
    SaveButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:SaveButton");
  
  function updateEnabled(propName) {
    return propName === "enabled";
  }
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  SaveButton.prototype.load = function(dom) { 
    SaveButton.$super.prototype.load.apply(this, arguments);
  
    var that = this,
        throttledFunc = baja.throttle(function () {
          that.update(updateEnabled);
        }, 1000),
        parentDiv;
        
    // Respond to editor change or saved events    
    this.getRootContainer().getDomElement().bind("editorchange editorsave", throttledFunc);
  };  

  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @name SaveButton#update
   * @function
   *
   * @param {Function} [hasUpdate]
   */
  SaveButton.prototype.update = function(hasUpdate) {   
    hasUpdate = hasUpdate || hasUpdateAll;
  
    var that = this;  
    if (hasUpdate("text")) {
      // The text always says save
      this.setText(baja.lex("mobile").get("save"));
    }
  
    if (hasUpdate("enabled")) {
      this.setEnabled(this.getRootContainer().isWidgetTreeModified());
    }
    
    // Call super update
    SaveButton.$super.prototype.update.apply(this, arguments);
  };
   
  /**
   * Handle a UI Widget event.
   */
  SaveButton.prototype.doHandleEvent = function(eventName) {
    // For each modifyable Widget, save the changes
    this.getRootContainer().saveWidgets();
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class RefreshButton
   *
   * The Widget that represents Niagara's 'kitPx:RefreshButton' Type.
   *
   * @name RefreshButton
   * @extends Button
   */
  var RefreshButton = function() {
    RefreshButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:RefreshButton");
  
 
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Function} [hasUpdate]
   */
  RefreshButton.prototype.update = function(hasUpdate) { 
    hasUpdate = hasUpdate || hasUpdateAll;
  
    if (hasUpdate("text")) {
      // The text always says save
      this.setText(baja.lex("mobile").get("refresh"));
    }
    
    // Call super update
    RefreshButton.$super.prototype.update.apply(this, arguments);
  };
    
  /**
   * Handle a UI Widget event.
   */
  RefreshButton.prototype.doHandleEvent = function(eventName) {
    // Refresh the page
    this.getRootContainer().refresh();
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class BackButton
   *
   * The Widget that represents Niagara's 'kitPx:BackButton' Type.
   *
   * @name BackButton
   * @extends Button
   */
  var BackButton = function() {
    BackButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:BackButton");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Function} [hasUpdate]
   */
  BackButton.prototype.update = function(hasUpdate) { 
    hasUpdate = hasUpdate || hasUpdateAll;
  
    if (hasUpdate("text")) {
      // The text always says back
      this.setText(baja.lex("mobile").get("back"));
    }
    
    // Call super update
    BackButton.$super.prototype.update.apply(this, arguments);
  };
    
  /**
   * Handle a UI Widget event.
   */
  BackButton.prototype.doHandleEvent = function(eventName) {
    pxUtilMobile.saveModifyQuestion(this, function yesNo() {
      window.history.back();
    });
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class ForwardButton
   *
   * The Widget that represents Niagara's 'kitPx:ForwardButton' Type.
   *
   * @name ForwardButton
   * @extends Button
   */
  var ForwardButton = function() {
    ForwardButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:ForwardButton");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Function} [hasUpdate]
   */
  ForwardButton.prototype.update = function(hasUpdate) {   
    hasUpdate = hasUpdate || hasUpdateAll;
  
    if (hasUpdate("text")) {
      // The text always says back
      this.setText(baja.lex("mobile").get("forward"));
    }
        
    // Call super update
    ForwardButton.$super.prototype.update.apply(this, arguments);
  };
    
  /**
   * Handle a UI Widget event.
   */
  ForwardButton.prototype.doHandleEvent = function(eventName) {
    pxUtilMobile.saveModifyQuestion(this, function yesNo() {
      window.history.forward();
    });
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class LogoffButton
   *
   * The Widget that represents Niagara's 'kitPx:LogoffButton' Type.
   *
   * @name LogoffButton
   * @extends Button
   */
  var LogoffButton = function() {
    LogoffButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:LogoffButton");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Function} [hasUpdate]
   */
  LogoffButton.prototype.update = function(hasUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
  
    if (hasUpdate("text")) {
      // The text always says back
      this.setText(baja.lex("mobile").get("logoff"));
    }
    
    // Call super update
    LogoffButton.$super.prototype.update.apply(this, arguments);
  };
    
  /**
   * Handle a UI Widget event.
   */
  LogoffButton.prototype.doHandleEvent = function(eventName) {
    pxUtilMobile.saveModifyQuestion(this, function yesNo() {
      window.location.assign("/logout");
    });
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class MobileJavaScriptButton
   *
   * The Widget that represents Niagara's 'mobile:BMobileJavaScriptButton' Type.
   *
   * @name MobileJavaScriptButton
   * @extends Button
   */
  var MobileJavaScriptButton = function() {
    MobileJavaScriptButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("mobile:MobileJavaScriptButton");
      
  /**
   * Handle a UI Widget event.
   */
  MobileJavaScriptButton.prototype.doHandleEvent = function(eventName) {
    var evil = eval; // Workaround for JsLint
    
    try {
      // Set this variable so it can be accessed by the JavaScript
      window.widget = this;
    
      // Evaluate the JavaScript
      evil("(function(){" + this.getJavaScript() + "\n}());");            
    }
    catch(err) {
      baja.error(err);
    }
    finally {
      window.widget = undefined;
    }
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class LocalizableLabel
   *
   * The Widget that represents Niagara's 'kitPx:LocalizableLabel' Type.
   *
   * @name LocalizableLabel
   * @extends HyperlinkLabel
   */
  var LocalizableLabel = function() {
    LocalizableLabel.$super.apply(this, arguments);
    this.$bkDom = null;
  }.$extend(Label).registerType("kitPx:LocalizableLabel");
  
  /**
   * Return the text for the label (this overrides the automatically
   * generated Property accessor for accessing the text Property).
   *
   * @return {String}
   */
  LocalizableLabel.prototype.getText = function() {  
    // If there's valid normal text then use that
    var text = this.get("text");
    if (text.length > 0) {
      return text;
    }
    
    // Do Format if we need too
    var textFormat = this.getTextFormat();
    if (textFormat !== baja.Format.DEFAULT) {
      return textFormat.format();
    }
    
    return "";
  };
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  LocalizableLabel.prototype.load = function(dom) { 
    this.$bkDom = $("<div></div>"); 
    this.$bkDom.appendTo(dom);
    LocalizableLabel.$super.prototype.load.apply(this, [this.$bkDom]);
  };
  
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  LocalizableLabel.prototype.update = function(hasUpdate) {  
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Background image
    if (hasUpdate("backgroundImage")) {    
      var bkImg = this.getBackgroundImage().encodeToString();
      bkImg = bkImg === "null" ? "" : bkImg;
      this.$bkDom.css("background", bkImg ? "url(/ord?" + bkImg + ") no-repeat center center" : "");
    }
    
    // Call Super update method
    LocalizableLabel.$super.prototype.update.apply(this, arguments);
  };
  
  /**
   * @class LocalizableButton
   *
   * The Widget that represents Niagara's 'kitPx:LocalizableButton' Type.
   *
   * @name LocalizableButton
   * @extends Button
   */
  var LocalizableButton = function() {
    LocalizableButton.$super.apply(this, arguments);
  }.$extend(Button).registerType("kitPx:LocalizableButton");
  
  // TODO: Support background image
  
  /**
   * Return the text for the label (this overrides the automatically
   * generated Property accessor for accessing the text Property).
   *
   * @function
   *
   * @return {String}
   */
  LocalizableButton.prototype.getText = LocalizableLabel.prototype.getText;
  
  /**
   * Handle a UI Widget event.
   */
  LocalizableButton.prototype.doHandleEvent = function(eventName) {
    var hlink = this.getHyperlink();
    if (hlink === baja.Ord.DEFAULT) {
      return false;
    }
  
    var that = this;
    function hyperlink() {
      pxUtilMobile.hyperlink(that, hlink);
    }
  
    if (this.getConfirmRequired()) {
      dialogs.yesNo({
        content: baja.lex("mobile").get({ key: "hyperlinkTo", args: hlink }),
        yes: function (cb) {
          hyperlink();
          cb.ok();
        }
      });
    }
    else {
      hyperlink();
    }
    
    // Return true so the event doesn't filter down to the bindings
    return true;
  };
  
  /**
   * @class Picture
   *
   * The Widget that represents Niagara's 'bajaui:Picture' Type.
   *
   * @name Picture
   * @extends Widget
   */
  var Picture = function() {
    Picture.$super.apply(this, arguments);
    this.$imgDom = null;
  }.$extend(Widget).registerType("bajaui:Picture");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   * By default, this method throws an error as it's designed to be overriden.
   * <p>
   * Anything that implements this method should assign the associated DOM element for the
   * Widget to the '$dom' attribute.
   *
   * @param {Object} DOM element to append content too.
   */
  var pictureHtml = "<div class='widgetPicture'><img class='pictureImage' src='#'></img></div>";
  Picture.prototype.load = function(dom) {
    var newDom = this.$dom = $(pictureHtml);
    this.$imgDom = newDom.find("img.pictureImage");
    newDom.appendTo(dom);
    
    // Set up click handler
    var that = this;
    newDom.click(function () {
      that.handleEvent("click");
      
      // Return false to override default click behavior
      return false;
    });
  };
  
  /**
   * Updates the src and layout of the Picture's <code>&lt;img&gt;</code> tag.
   * Called whenever there is an update to the Picture's <code>image</code> or
   * <code>scale</code> properties.
   * 
   * @private
   */
  function updatePictureImage(pic) {
    var imgSrc = getImageSrc(pic),
        imgDom = pic.$imgDom;
    
    if (!imgSrc) {
      /* 
       * no image to show. 
       */
      imgDom.hide();
    } else if (pic.isAbsolutelyPositioned()) {
      /* 
       * we are absolutely positioned, so layout according to size and scale. 
       */
      calculateImageDimensions(imgSrc, function (imgW, imgH) {
        var layout = pic.getLayout(),
            outerWidth = layout.getWidth(),
            outerHeight = layout.getHeight(),
            innerWidth = imgW,
            innerHeight = imgH,
            halign = pic.getHalign().getTag(),
            valign = pic.getValign().getTag(),
            scale = pic.getScale().getTag(),
            css = getScaledLayoutCSS(outerWidth, outerHeight,
                                    innerWidth, innerHeight,
                                    halign, valign, scale);

        setImageSrc(imgDom, imgSrc);
        imgDom.css(css).show();
      });
    } else {
      /*
       * we are in a fluid mobile layout, so no scaling.
       */
      setImageSrc(imgDom, imgSrc);
      imgDom.show();
    }
  }
  
  // TODO: Support animate (although animate will probably prove to be impossible)

  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   */
  Picture.prototype.update = function(hasUpdate) {  
    hasUpdate = hasUpdate || hasUpdateAll;
  
    // Call Super update method
    Picture.$super.prototype.update.apply(this, arguments);
    
    // Image
    if (hasUpdate("image") || hasUpdate("scale")) {
      updatePictureImage(this);
    }
    
    // Horizontal alignment
    if (hasUpdate("halign")) {
      var halign = getHalign(this);
      this.$dom.css("text-align", halign);
    }
    
    // Vertical alignment
    if (hasUpdate("valign")) {
      var valign = getValign(this);
      this.$dom.css("vertical-align", valign);
    }
  };
  
  /**
   * @class BoundTable
   *
   * The Widget that represents Niagara's 'bajaui:BoundTable' Type.
   *
   * @name BoundTable
   * @extends Widget
   */
  var BoundTable = function() {
    BoundTable.$super.apply(this, arguments);
    this.$tableDom = null;
  }.$extend(Widget).registerType("bajaui:BoundTable");
  
  /**
   * Append the Widget's DOM content onto the specified DOM element.
   * <p>
   * By default, this method throws an error as it's designed to be overriden.
   * <p>
   * Anything that implements this method should assign the associated DOM element for the
   * Widget to the '$dom' attribute.
   *
   * @param {Object} DOM element to append content too.
   */
  var tableHtml = "<div class='boundTable'></div>",
      footerHtml = 
        '<div class="boundTableFooter" data-role="footer" data-position="fixed">' +
          '<div data-role="navbar" class="tableNavBar">' +
            '<ul>' +
              '<li><a href="#" data-theme="b" class="prevTable" data-role="button" data-icon="arrow-l" data-iconpos="notext"></a></li>' +
              '<li><a href="#" data-theme="b" class="nextTable" data-role="button" data-icon="arrow-r" data-iconpos="notext"></a></li>' +
            '</ul>' +
          '</div>' +
        '</div>';
  
  BoundTable.prototype.load = function(dom) {
    var tableDom = $(tableHtml),
        footerDom = $(footerHtml);
    
    
    
    tableDom.appendTo(dom);
    footerDom.appendTo(tableDom).trigger('create');
    
    this.$dom = tableDom;
    this.$footer = footerDom;
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   * @param {Boolean} [firstUpdate] set to true on Widgets first update when the page is first loaded.
   */
  BoundTable.prototype.update = function(hasUpdate, firstUpdate) {
    // If the table has been already loaded then bail (for performance reasons)
    if (this.$tableLoaded) {
      return;
    }
    
    // For performance reasons, ensure a BoundTable only loads once  
    hasUpdate = hasUpdate || hasUpdateAll;
  
    BoundTable.$super.prototype.update.apply(this, arguments);    
    
    // TODO: Implement table with more styling, limit and offset (next and last buttons) - 
    // just like history App.
            
    // Get a list of all the bound TableBindings
    var bindings = [];
    this.getSlots(function (slot) {
      return slot.isProperty() && slot.getType().is("bajaui:TableBinding") && this.get(slot).isBound();
    }).each(function (slot) {
      bindings.push(this.get(slot));
    });
    
    // Bail if there are no bindings
    if (bindings.length === 0) {
      return;
    }
    
    // Mark the table as loaded (before any asynchronous network calls happen)
    this.$tableLoaded = true;
    
    var view = new niagara.mobile.table.DataTableView(),
        dom = this.$dom,
        tableContainer = $('<div class="tableContent"/>').prependTo(dom),
        footer = dom.find('div:jqmData(role=footer)'),
        domHeight = dom.height(),
        footerHeight = footer.height();
    
    view.initializeDOM(tableContainer, function () {
      touchScroll(tableContainer, { horiz: true, vert: true, showBlockade: true });
      tableContainer.height(domHeight - footerHeight);
      view.bindFooter(footer);
      view.loadValue(bindings[0].getOrd());
    });
  };
  
  /**
   * Load the child Widgets.
   * <p>
   * {@link baja.Widget#load} must be called before this method is invoked.
   *
   * @function
   */
  BoundTable.prototype.loadChildren = noop;
  
  /**
   * @class Bargraph
   *
   * The Widget that represents Niagara's 'kitPx:Bargraph' Type.
   *
   * @name Bargraph
   * @extends Widget
   */
  var Bargraph = function() {
    Bargraph.$super.apply(this, arguments);
    this.$dom = null;
    this.$bargraphDom = null;
    this.$barDom = null;
  }.$extend(Widget).registerType("kitPx:Bargraph");
  
  var bargraphHtml = "<div class='bargraph'>" +
                       "<div class='innerBargraph'>" + 
                         "<div class='bargraphBar'></div>" +
                       "</div'>" + 
                     "</div>",
      bargraphScaleHtml = "<div class='bargraphScale'></div>";
  
  Bargraph.prototype.load = function(dom) {
    var that = this,
        outerDom = $(bargraphHtml),
        bargraphDom = $(".innerBargraph", outerDom),
        barDom = $(".bargraphBar", outerDom);
    
    if (!that.isAbsolutelyPositioned()) {
      bargraphDom.css("padding", "14px 14px");
    }    
    else {
      bargraphDom.css({
        "width": "100%",
        "height": "100%"
      });
    }
    
    // Set up click handler
    outerDom.click(function () {
      that.handleEvent("click");
      
      // Return false to override default click behavior
      return false;
    });
        
    dom.append(outerDom);
        
    that.$dom = outerDom;
    that.$bargraphDom = bargraphDom;
    that.$barDom = barDom;
  };
    
  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} [hasUpdate] called to test whether a 
   *                               given Property can update the DOM.
   * @param {Boolean} [firstUpdate] set to true on Widgets first update when the page is first loaded.
   */
  Bargraph.prototype.update = function(hasUpdate, firstUpdate) {
    hasUpdate = hasUpdate || hasUpdateAll;
    
    var that = this,
        dom = that.$dom,
        bargraphDom = that.$bargraphDom,
        barDom = that.$barDom,
        isVertical = that.getOrientation().is("vertical"),
        min = that.getMin(),
        max = that.getMax(),
        value,
        scale,
        scaleDom,
        scaleNum,
        perc,
        i;
       
    Bargraph.$super.prototype.update.apply(that, arguments); 
        
    if (hasUpdate("background")) {
      that.getBackground().update({ 
        dom: bargraphDom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    if (hasUpdate("foreground")) {
      that.getForeground().update({ 
        dom: dom, 
        imageSelector: "border",
        colorSelector: "border",
        prefix: "1px solid "
      });
    }
    
    // Fill
    if (hasUpdate("fill")) {
      that.getFill().update({ 
        dom: barDom, 
        colorSelector: "background",
        imageSelector: "background"
      });
    }
    
    // Update bar chart value
    if (hasUpdate("value") ||
        hasUpdate("min") || 
        hasUpdate("max") ||
        hasUpdate("vertical")) {
        
      value = that.getValue();
      value = Math.max(((value - min) / (max - min)) * 100, 0);
      value = String(value) + "%";
                  
      if (isVertical) {
        barDom.css({
          "width": "100%",
          "height": value,
          "bottom": "0px"
        });
      }
      else {
        barDom.css({
          "width": value,
          "height": "100%",
          "top": "0px"
        });
      }
    }
    
    // Scale
    if (hasUpdate("scale") ||
        hasUpdate("scaleVisible") ||
        hasUpdate("orientation") ||
        hasUpdate("foreground")) {
       
      // Remove all existing scales from the DOM       
      $(".bargraphScale", bargraphDom).remove();
      
      if (that.getScaleVisible()) {
        scale = that.getScale();
        scale = scale <= 0 ? 10 : scale;
        scaleNum = (max - min) / scale;
        scaleNum = scaleNum <= 0 ? 10 : scaleNum;
                                
        for (i = 1; i < scaleNum; ++i) {
          scaleDom = $(bargraphScaleHtml);
          perc = ((100 / scaleNum) * i) + "%";
          
          if (isVertical) {
            scaleDom.css({
              "position": "absolute",
              "right": "0px",
              "width": "4px",
              "height": "1px",
              "bottom": perc
            });
          }
          else {
            scaleDom.css({
              "position": "absolute",
              "bottom": "0px",
              "height": "4px",
              "width": "1px",
              "left": perc
            });
          }
          
          // Color of the scale
          that.getForeground().update({ 
            dom: scaleDom,
            imageSelector: "background",
            colorSelector: "background"
          });
          
          bargraphDom.append(scaleDom);
        }
      } 
    }
  };
  
  /**
   * Load the child Widgets.
   * <p>
   * {@link baja.Widget#load} must be called before this method is invoked.
   *
   * @function
   */
  Bargraph.prototype.loadChildren = noop;
    
}(baja));


