//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

/**
 * @fileOverview Converters used in the Px App. 
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, BaseBajaObj, window, niagara*/ 
  
(function pxConverters(baja) {

  // Use ECMAScript 5 Strict Mode
  "use strict";
  
  niagara.util.require(
    "$.mobile",
    "niagara.util.px"
  );
  
  var pxUtil = niagara.util.px;
    
  /**
   * @class Converter.
   * <p>
   * A Component that represents a Niagara 'baja:Converter' Type.
   * <p>
   * Converters are used to convert a value from one baja:Object Type to another. 
   *
   * @name Converter
   * @extends baja.Struct
   */
  var Converter = function() {
    Converter.$super.apply(this, arguments);
  }.$extend(baja.Struct).registerType("baja:Converter");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  Converter.prototype.convert = function(from, to, target) {
    return null;
  };
  
  /**
   * @class Object To String Converter.
   * <p>
   * A Component that represents a Niagara 'converters:ObjectToString' Type.
   *
   * @name ObjectToString
   * @extends Converter
   */
  var ObjectToString = function() {
    ObjectToString.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:ObjectToString");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  ObjectToString.prototype.convert = function(from, to, target) {
    return this.getFormat().format({
      object: from, 
      target: target
    });
  };
  
  /**
   * @class Pass Through Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BPassThrough' Type.
   *
   * @name PassThrough
   * @extends Converter
   */
  var PassThrough = function() {
    PassThrough.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:PassThrough");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  PassThrough.prototype.convert = function(from, to, target) {
    if (from.getType().is(to.getType())) {
      return from;
    }
    return to;
  };
  
  /**
   * @class Fixed Simple Converter.
   * <p>
   * A Component that represents a Niagara 'converters:FixedSimple' Type.
   *
   * @name FixedSimple
   * @extends Converter
   */
  var FixedSimple = function() {
    FixedSimple.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:FixedSimple");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  FixedSimple.prototype.convert = function(from, to, target) {
    return this.getValue();
  };
  
  ////////////////////////////////////////////////////////////////
  // Numeric Converters
  ////////////////////////////////////////////////////////////////  
  
  /**
   * @class Numeric To Simple Map.
   * <p>
   * A Component that represents a Niagara 'converters:NumericToSimpleMap' Type.
   * <p>
   * Please note, this isn't a complete implementation but will do for the Px App.
   *
   * @name NumericToSimpleMap
   * @extends baja.Simple
   */
  var NumericToSimpleMap = function(items, def) {
    NumericToSimpleMap.$super.apply(this, arguments);
    this.$items = items;
    this.$def = def;
  }.$extend(baja.Simple);
  
  /**
   * Make a NumericToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {NumericToSimpleMap}
   */
  NumericToSimpleMap.make = function(obj) {
    obj = baja.objectify(obj, "str");
    
    if (obj.str === "null") {
      return NumericToSimpleMap.DEFAULT; 
    }
        
    // Get the prototype instance (used to decode everything else)
    var protoRes = /^([^ ]+) (.*)/.exec(obj.str);
    
    if (protoRes.length <= 1) {
      throw new Error("Cannot decode NumericToSimpleMap" + obj.str);
    }
    
    // Decode the prototype simple we're going to decode from
    var proto = baja.$(protoRes[1]);
    
    // Separate out the value pairs
    var valPairRegEx = /([^\;]*)=([^\;]*)/g;
    var valPairRes, defValue = null, value, numRegEx, numRes, items = [];
    
    valPairRes = valPairRegEx.exec(protoRes[2]);
    while (valPairRes) {
      // Decode the value from the String
      value = proto.decodeFromString(valPairRes[2]);
      if (valPairRes[1] === "default") {
        defValue = value;
      }
      else {
        // Decode the numeric range
        numRegEx = /([^:]*):([^:]*)/;
        numRes = numRegEx.exec(valPairRes[1]);
        
        if (!numRes || (numRes && numRes.length !== 3)) {
          throw new Error("Could not decode NumericToSimpleMap: " + valPairRes[1]);
        }
        
        // Add the item with the min and max range to our array
        items.push({
          min: baja.Double.DEFAULT.decodeFromString(numRes[1]).valueOf(),
          max: baja.Double.DEFAULT.decodeFromString(numRes[2]).valueOf(),
          value: value
        });
      }
      
      valPairRes = valPairRegEx.exec(protoRes[2]);
    }
    
    return new NumericToSimpleMap(items, defValue);
  };
  
  /**
   * Make a NumericToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {NumericToSimpleMap}
   */
  NumericToSimpleMap.prototype.make = function(obj) {
    return NumericToSimpleMap.make.apply(NumericToSimpleMap, arguments);
  };
  
  /**
   * Decode a NumericToSimpleMap
   *
   * @param {String} str
   * @return {NumericToSimpleMap}
   */   
  NumericToSimpleMap.prototype.decodeFromString = function(str) {
    return NumericToSimpleMap.make(str);
  };
  
  /**
   * Encode the NumericToSimpleMap to a String
   *
   * @return {String}
   */  
  NumericToSimpleMap.prototype.encodeToString = function() {
    // TODO: Although the PxApp should never need to encode this, we should
    // probably implement this at some point.
    throw new Error("Currently not supported");
  };
  
  /**
   * Default NumericToSimpleMap instance
   */
  NumericToSimpleMap.DEFAULT = new NumericToSimpleMap([], null);
  
  // Register Type
  NumericToSimpleMap.registerType("converters:NumericToSimpleMap");   
  
  /**
   * Equality test
   *
   * @param obj num
   * @return {Boolean}
   */  
  NumericToSimpleMap.prototype.equals = function(obj) {    
    var it, objIt, i;
    if (baja.hasType(obj) && obj.getType().equals(this.getType()) && obj.$items.length === this.$items.length) {
      
      // Compare items
      for (i = 0; i < this.$items.length; ++i) {
        it = this.$items[i];
        objIt = obj.$items[i];
        
        if (it.min !== objIt.min || it.max !== objIt.max || !it.value.equals(objIt.value)) {
          return false;
        }
      }
      
      // Compare default values
      if (this.$def === null) {
        return obj.$def === null;
      }
      else {
        if (obj.$def === null) {
          return false;
        }
        else {
          return obj.$def.equals(this.$def);
        }
      }
    }
    return false;
  };
  
  /**
   * Get a simple by its numeric value or return the default if no mapping.
   *
   * @param {Number} num
   * @return {baja.Simple} the value (or null if no default is specified).
   */  
  NumericToSimpleMap.prototype.get = function(num) {
    var it, i;
    for (i = 0; i < this.$items.length; ++i) {                     
      it = this.$items[i];                    
      if (it.min <= num && num <= it.max) {
        return it.value;
      }
    }                       
    return this.$def;
  };
  
  /**
   * @class INumeric To Simple Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BINumericToSimple' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BINumeric like their Java counterparts.
   *
   * @name INumericToSimple
   * @extends Converter
   */
  var INumericToSimple = function() {
    INumericToSimple.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:INumericToSimple");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  INumericToSimple.prototype.convert = function(from, to, target) {
    var val = Number.getNumberFromINumeric(from);
        
    // Get the Simple from the map
    var simple = this.getMap().get(val);
    if (simple === null) {
      // Default back if we can't find anything
      simple = to;
    }
    
    return simple;
  };
  
  /**
   * @class INumeric To Number Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BINumericToNumber' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BINumeric like their Java counterparts.
   *
   * @name INumericToNumber
   * @extends Converter
   */
  var INumericToNumber = function() {
    INumericToNumber.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:INumericToNumber");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  INumericToNumber.prototype.convert = function(from, to, target) {
    var val = Number.getNumberFromINumeric(from);    
    return to.make(val);
  };
  
  ////////////////////////////////////////////////////////////////
  // Enum Converters
  ////////////////////////////////////////////////////////////////  
  
  /**
   * @class Enum To Simple Map.
   * <p>
   * A Component that represents a Niagara 'converters:EnumToSimpleMap' Type.
   * <p>
   * Please note, this isn't a complete implementation but will do for the Px App.
   *
   * @name EnumToSimpleMap
   * @extends baja.Simple
   */
  var EnumToSimpleMap = function(items, def) {
    EnumToSimpleMap.$super.apply(this, arguments);
    this.$items = items;
    this.$def = def;
  }.$extend(baja.Simple);
  
  /**
   * Make a EnumToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {EnumToSimpleMap}
   */
  EnumToSimpleMap.make = function(obj) {
    obj = baja.objectify(obj, "str");
    
    if (obj.str === "null") {
      return EnumToSimpleMap.DEFAULT; 
    }
        
    // Get the prototype instance (used to decode everything else)
    var protoRes = /^([^ ]+) (.*)/.exec(obj.str);
    
    if (protoRes.length <= 1) {
      throw new Error("Cannot decode EnumToSimpleMap" + obj.str);
    }
    
    // Decode the prototype simple we're going to decode from
    var proto = baja.$(protoRes[1]);
    
    // Separate out the value pairs
    var valPairRegEx = /([^\;]*)=([^\;]*)/g;
    var valPairRes, defValue = null, value, items = [];
        
    valPairRes = valPairRegEx.exec(protoRes[2]);    
    while (valPairRes) {
      // Decode the value from the String
      value = proto.decodeFromString(valPairRes[2]);
      if (valPairRes[1] === "default") {
        defValue = value;
      }
      else {
        items.push({
          ordinal: parseInt(valPairRes[1], 10),
          value: value
        });
      }
      
      valPairRes = valPairRegEx.exec(protoRes[2]);  
    }
    
    return new EnumToSimpleMap(items, defValue);
  };
  
  /**
   * Make a EnumToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {EnumToSimpleMap}
   */
  EnumToSimpleMap.prototype.make = function(obj) {
    return EnumToSimpleMap.make.apply(EnumToSimpleMap, arguments);
  };
  
  /**
   * Decode a EnumToSimpleMap
   *
   * @param {String} str
   * @return {EnumToSimpleMap}
   */   
  EnumToSimpleMap.prototype.decodeFromString = function(str) {
    return EnumToSimpleMap.make(str);
  };
  
  /**
   * Encode the EnumToSimpleMap to a String
   *
   * @return {String}
   */  
  EnumToSimpleMap.prototype.encodeToString = function() {
    // TODO: Although the PxApp should never need to encode this, we should
    // probably implement this at some point.
    throw new Error("Currently not supported");
  };
  
  /**
   * Default EnumToSimpleMap instance
   */
  EnumToSimpleMap.DEFAULT = new EnumToSimpleMap([], null);
  
  // Register Type
  EnumToSimpleMap.registerType("converters:EnumToSimpleMap");   
  
  /**
   * Equality test
   *
   * @param obj num
   * @return {Boolean}
   */  
  EnumToSimpleMap.prototype.equals = function (obj) {    
    var it, objIt, i;
    if (baja.hasType(obj) && obj.getType().equals(this.getType()) && obj.$items.length === this.$items.length) {
      
      // Compare items
      for (i = 0; i < this.$items.length; ++i) {
        it = this.$items[i];
        objIt = obj.$items[i];
        
        if (it.ordinal !== objIt.ordinal || !it.value.equals(objIt.value)) {
          return false;
        }
      }
      
      // Compare default values
      if (this.$def === null) {
        return obj.$def === null;
      }
      else {
        if (obj.$def === null) {
          return false;
        }
        else {
          return obj.$def.equals(this.$def);
        }
      }
    }
    return false;
  };
  
  /**
   * Get a simple by its ordinal value or return the default if no mapping.
   *
   * @param {Number} ordinal
   * @return {baja.Simple} the value (or null if no default is specified).
   */  
  EnumToSimpleMap.prototype.get = function(ordinal) {
    var it, i;
    for (i = 0; i < this.$items.length; ++i) {                     
      if (this.$items[i].ordinal === ordinal) {
        return this.$items[i].value;
      }      
    }                       
    return this.$def;
  };
  
  /**
   * @class IEnum To Simple Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BIEnumToSimple' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BIEnum like their Java counterparts.
   *
   * @name IEnumToSimple
   * @extends Converter
   */
  var IEnumToSimple = function() {
    IEnumToSimple.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:IEnumToSimple");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  IEnumToSimple.prototype.convert = function(from, to, target) {
    // Find the enum
    var val = baja.Enum.getEnumFromIEnum(from);
    
    // Get the Simple from the map
    var simple = this.getMap().get(val.getOrdinal());
    if (simple === null) {
      // Default back if we can't find anything
      simple = to;
    }
    
    return simple;
  };
  
  /**
   * @class IEnum To Enum Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BIEnumToEnum' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BIEnum like their Java counterparts.
   *
   * @name IEnumToEnum
   * @extends Converter
   */
  var IEnumToEnum = function() {
    IEnumToEnum.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:IEnumToEnum");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  IEnumToEnum.prototype.convert = function(from, to, target) {
    var val = baja.Enum.getEnumFromIEnum(from);    
    return to.getRange().get(val.getOrdinal());
  };
  
  ////////////////////////////////////////////////////////////////
  // Boolean Converters
  ////////////////////////////////////////////////////////////////  
  
  /**
   * @class BIBoolean To Simple Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BIBooleanToSimple' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BIBoolean like their Java counterparts.
   *
   * @name IBooleanToSimple
   * @extends Converter
   */
  var IBooleanToSimple = function() {
    IBooleanToSimple.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:IBooleanToSimple");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  IBooleanToSimple.prototype.convert = function(from, to, target) {
    return Boolean.getBooleanFromIBoolean(from) ? this.getTrueValue() : this.getFalseValue();
  };
  
  /**
   * @class BIBoolean To Boolean Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BIBooleanToBoolean' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BIBoolean like their Java counterparts.
   *
   * @name IBooleanToBoolean
   * @extends Converter
   */
  var IBooleanToBoolean = function() {
    IBooleanToBoolean.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:IBooleanToBoolean");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  IBooleanToBoolean.prototype.convert = function(from, to, target) {
    return Boolean.getBooleanFromIBoolean(from);
  };
  
  ////////////////////////////////////////////////////////////////
  // Status Converters
  ////////////////////////////////////////////////////////////////  
  
  /**
   * @class BIStatus To Simple Converter.
   * <p>
   * A Component that represents a Niagara 'converters:BIStatusToSimple' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BIStatus like their Java counterparts.
   *
   * @name IStatusToSimple
   * @extends Converter
   */
  var IStatusToSimple = function() {
    IStatusToSimple.$super.apply(this, arguments);
  }.$extend(Converter).registerType("converters:IStatusToSimple");
  
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  IStatusToSimple.prototype.convert = function(from, to, target) {
    var status = baja.Status.getStatusFromIStatus(from);
    
    if (status.isDisabled()) {
      return this.getDisabled();
    }
    else if (status.isFault()) {
      return this.getFault();
    }
    else if (status.isDown()) {
      return this.getDown();
    }
    else if (status.isAlarm()) {
      return this.getAlarm();
    }
    else if (status.isStale()) {
      return this.getStale();
    }
    else if (status.isOverridden()) {
      return this.getOverridden();
    }
    else if (status.isUnackedAlarm()) {
      return this.getUnackedAlarm();
    }
    else if (status.isNull()) {
      return this.getNullStatus();
    }
    else {
      return to;
    }
  };
  
    /**
   * @class BIStatus To Brush Converter.
   * <p>
   * A Component that represents a Niagara 'kitPx:BIStatusToBrush' Type.
   * <p>
   * Please note, this doesn't work properly since BajaScript Components don't have
   * properly implement BIStatus like their Java counterparts.
   *
   * @name IStatusToBrush
   * @extends Converter
   */
  var IStatusToBrush = function() {
    IStatusToBrush.$super.apply(this, arguments);
  }.$extend(Converter).registerType("kitPx:IStatusToBrush");
    
  /**
   * Convert the first object to the second object and return the result.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @return {baja.Object} the converted object.
   */
  IStatusToBrush.prototype.convert = function(from, to, target) {
    var status = baja.Status.getStatusFromIStatus(from);

    var obj = {
      proto: baja.$("gx:Brush"),
      isForeground: this.getMode().is("foreground")
    };
      
    var brush = null;
    
    if (status.isDisabled()) {
      obj.statusName = "disabled";
      brush = pxUtil.getStatusBrush(obj);
    }
    else if (status.isFault()) {
      obj.statusName = "fault";
      brush = pxUtil.getStatusBrush(obj);
    }
    else if (status.isDown()) {
      obj.statusName = "down";
      brush = pxUtil.getStatusBrush(obj);
    }
    else if (status.isAlarm()) {
      obj.statusName = "alarm";
      brush = pxUtil.getStatusBrush(obj);
    }
    else if (status.isStale()) {
      obj.statusName = "stale";
      brush = pxUtil.getStatusBrush(obj);
    }
    else if (status.isOverridden()) {
      obj.statusName = "overridden";
      brush = pxUtil.getStatusBrush(obj);
    }
    
    if (brush === null) {
      brush = to;
    }
    
    return brush;
  };
    
}(baja));


