/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to property sheet objects.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, browser: true */
/*global $, baja, niagara */



(function propSheet() {
  "use strict";
  
  niagara.util.require(
    'jQuery',
    'niagara',
    'niagara.util',
    'niagara.util.flow',
    'niagara.util.mobile.ListPageView',
    'niagara.util.mobile.commands',
    'niagara.util.mobile.views',
    'niagara.util.mobile.dialogs',
    'niagara.fieldEditors',
    'niagara.fieldEditors.BaseFieldEditor'
  );
  
  //imports
  var util = niagara.util,
      statusGradients = niagara.statusGradients,
      mobileUtil = util.mobile,
      commands = mobileUtil.commands,
      dialogs = mobileUtil.dialogs,
      ListView = mobileUtil.ListView,
      PageView = mobileUtil.PageView,
      ListPageView = mobileUtil.ListPageView,
      
      encodePageId = mobileUtil.encodePageId,
      escapeHtml = mobileUtil.escapeHtml,
      callbackify = util.callbackify,
      
      //private vars
      mobileLex,
      validateAndSaveWorkflow,
      loadEditorWorkflow,
      footerBarHtml,
      navbarHtml,
      
      //exports
      PropertySheet,
      PropertySheetListView;

    
  footerBarHtml = 
    '<div data-role="footer" data-position="fixed" data-theme="b">\n' +
      '<div data-role="navbar"><ul>\n' +
        '<li><a data-icon="check" class="saveLink">{save}</a></li>\n' +     
        '<li><a data-icon="refresh" class="refreshLink">{refresh}</a></li>\n' +
        '<li><a data-icon="gear" class="actionsLink">{actions}</a></li>\n' +
      '</ul></div>\n' +
    '</div>';
  
  navbarHtml =
    '<div data-role="navbar">' +
      '<ul>' +
        '<li>' +
          '<a class="componentLink">{title}</a>' +
        '</li>' +
      '</ul>' +
    '</div>';
  
  function getMobileLex() {
    if (!mobileLex) {
      mobileLex = baja.lex("mobile");
    }
    return mobileLex;
  }
  
  validateAndSaveWorkflow = util.flow.sequentialParallel(
    function validateAndSaveWorkflow__step1__validate(editor) {
      editor.validate(this);
    },
    function validateAndSaveWorkflow__step2__saveValue(editor) {
      editor.saveValue(this);
    }
  );
  
  loadEditorWorkflow = util.flow.sequential(
    function loadEditorWorkflow__step1__makeFor(cx) {
      var paramDiv = $('<div class="property"/>');
      cx.targetElement.append(paramDiv);
      niagara.fieldEditors.makeFor(cx, this);
    },
    function loadEditorWorkflow__step2__buildAndLoad(cx, editor) {
      editor.buildAndLoad(cx.targetElement, this);
    }
  );

  /**
   * Starts up a page-loading spinner, and tells the given callback literal to
   * hide it once ok or fail is called.
   * 
   * @memberOf niagara.schedule.ui
   * @private
   * @param {Object} callbacks
   * @param {Number} [timeout] how long to wait until the spinner appears
   * (default 1 second)
   * @returns {Object} new callbacks
   */
  function spinUntilCallback(callbacks, timeout) {
    callbacks = callbackify(callbacks);
    
    if (timeout === undefined || timeout === null) {
      timeout = 1000;
    }
    var spinner = util.mobile.spinnerTicket(timeout);
    
    function stopSpin() {
      spinner.hide();
    }
    
    util.aop.before(callbacks, 'ok', stopSpin);
    util.aop.before(callbacks, 'fail', stopSpin);
    
    return callbacks;
  }
  
  /**
   * Creates an editable div for a component or a property of a component.
   * This div will be displayed when linking to an editable property of the
   * currently displayed component, like StringWritable.in2, for example.
   * The appropriate field editor will be retrieved and inserted into the
   * created div, which will then be inserted into the target div. 
   * 
   * <p>What you will have afterwards:
   * 
   * <code><pre>
   * &lt;div id="targetElement"&gt;
   *   &lt;div class="property"&gt;
   *     &lt;div class="editor"&gt; &lt;!-- this div created by your field editor --&gt;
   *       &lt;!-- field editor stuff from niagara.fieldEditors here --&gt;
   *     &lt;/div&gt;
   *   &lt;/div&gt;
   * &lt;/div&gt;
   * </pre></code>
   * 
   * @private
   * @memberOf niagara.propSheet
   * 
   * @param {Object} params an object with extra parameters
   * 
   * @params {baja.Value} params.value the Baja value for which to load a
   * field editor - if omitted, defaults to
   * <code>params.container.get(params.slot)</code>
   * 
   * @param {jQuery} params.targetElement the div we are inserting into
   * 
   * @params {baja.Facets} [params.facets] facets that may be passed to the
   * created field editor
   * 
   * @param {baja.Complex} [params.container] the component whose property we are
   * editing.
   * 
   * @param {baja.Slot} [params.slot] the slot the edited object lives in
   * 
   * @param {Object} callbacks an object containing ok/fail callbacks
   * 
   * @param {Function} callbacks.ok an OK callback to run after the editor has 
   * been instantiated and built its HTML. The editor itself will be passed as 
   * the first parameter to this callback.
   * 
   * @see niagara.fieldEditors.makeFor
   */
  function loadEditor(params, callbacks) {

    baja.strictArg(params, Object);
    
    var container = params.container,
        slot = params.slot,
        value = container.get(slot),
        range;

    params.facets = params.facets || {};

    loadEditorWorkflow.invoke(params, callbacks);
  }
  
  /**
   * Validates and then saves a collection of modified editors. If any of the
   * validation steps fail, none of the save steps will be performed. If an
   * editor has not been modified (isModified() === false) then it will not
   * be saved.
   * 
   * @memberOf niagara.propSheet
   * @private
   * @param {Object|Array} editors a collection of field editors
   * @param {Object} [callbacks] an object containing ok/fail callbacks
   */
  function validateAndSaveEditors(editors, callbacks) {
    var editorsToSave = [];
    
    baja.iterate(editors, function (editor, i) {
      if (editor.isModified()) {
        editorsToSave.push(editor);
      }
    });
    
    validateAndSaveWorkflow.invoke(editorsToSave, callbacks);
  }
  
  
////////////////////////////////////////////////////////////////
// PropertySheetListView
////////////////////////////////////////////////////////////////
  
  /**
   * Forms the content listview of a property sheet.
   * 
   * @class
   * @extends niagara.util.mobile.ListView
   * @memberOf niagara.propSheet
   */
  PropertySheetListView = ListView.subclass(function PropertySheetListView() {
    this.fieldEditorMap = {};
  }, 'subscribable');
  
  /**
   * Loads the field editor divs for the property about to be edited.
   * 
   * @name niagara.propSheet.PropertySheet#loadEditorParams
   * @function
   * @private
   * 
   * @param {Object} params an object literal
   * @param {baja.Complex} params.container the object containing the property 
   * we are to be editing - for example, if we are editing 
   * <code>myNumericWritable.in2</code>, then the container is 
   * <code>myNumericWritable</code>
   * @param {baja.Slot} params.slot the Slot corresponding to the property we 
   * will edit
   * @param {jQuery} params.targetElement the div into which the field editor div 
   * will be inserted
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @param {Function} callbacks.ok a function to be called when the editor
   * has been fully loaded - the editor instance will be passed as the first
   * parameter to this function
   */
  PropertySheetListView.prototype.loadEditorParams = function (params, callbacks) {
    baja.strictArg(params, Object);
    baja.strictArg(params.container, baja.Complex);
    baja.strictArg(params.slot, baja.Slot);
    baja.strictArg(params.targetElement, $);
    
    callbacks = spinUntilCallback(callbacks);
    
    var that = this;
    
    loadEditor(params, {
      ok: function (editor) {
        that.fieldEditorMap[params.slot] = editor;
        callbacks.ok(editor);
      },
      fail: callbacks.fail
    });
  };
  
  /**
   * Override point to show field editors when expanding an editable property.
   * 
   * @name niagara.propSheet.PropertySheet#populateExpandedDiv
   * @function
   * 
   * @param {baja.Property} prop the property being expanded
   * @param {jQuery} expandedDiv the div that is expanded to show property
   * details
   */
  PropertySheetListView.prototype.populateExpandedDiv = function (prop, expandedDiv) {
    var ul = this.elements.ul,
        pageId = encodePageId(prop);
    
    this.loadEditorParams({
      container: this.value,
      slot: prop,
      targetElement: expandedDiv
    }, {
      ok: function () {
        expandedDiv.bind('editorchange', function (event, editor) {
          ul.children('li#' + pageId).find('div.display').addClass('dirty');
        });
      }
    });
  };
  
  /**
   * Saves all outstanding changes (dirty field editors) on the property
   * sheet and commits the changes up to the station.
   * 
   * @name niagara.propSheet.PropertySheetListView#save
   * @function
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  PropertySheetListView.prototype.save = function (callbacks) {
    callbacks = spinUntilCallback(callbacks);
    
    var that = this;
      
    validateAndSaveEditors(that.fieldEditorMap, {
      ok: function () {
        that.elements.ul.find('div.display.dirty')
          .removeClass('dirty')
          .addClass('editable');
        that.setModified(false);
        callbacks.ok();
      },
      fail: callbacks.fail
    });
  };
  
  /**
   * Performs subscription on this component's child components. The default 
   * behavior when a child component changes is simply to update the display
   * value on this navigator with the child component's new display value.
   * 
   * @name niagara.propSheet.PropertySheet#subscribeChildren
   * @function
   * @private
   * 
   * @param {baja.comm.Batch} batch a batch to use to subscribe children (should
   * be the same batch used to subscribe the parent component)
   */
  PropertySheetListView.prototype.subscribeChildren = function (batch) {
    if (!this.value.getType().isComponent()) {
      return;
    }
    
    var childrenSub = this.childrenSubscriber,
        that = this,
        subscribableChildren = [];
    
    if (!childrenSub) {
      childrenSub = new baja.Subscriber();
      
      childrenSub.attach('subscribed changed', function (subprop, cx) {
        that.updateDisplay(this.getPropertyInParent());
      });
      
      this.childrenSubscriber = childrenSub;
    }
        
    this.value.getSlots(function (slot) {
      if (!slot.isProperty()) {
        return false;
      }
      var type = slot.getType();
      return type.isComponent() && (type.is("baja:IStatus") || type.is("baja:VirtualComponent"));
    }).each(function (slot) {
      var comp = this.get(slot);
      if (!comp.isSubscribed()) {
        subscribableChildren.push(this.get(slot));
      }
    });

    childrenSub.subscribe({
      comps: subscribableChildren,
      batch: batch
    });
  };
  
  /**
   * Unsubscribes from all this component's child components.
   * 
   * @name niagara.propSheet.PropertySheet#unsubscribeChildren
   * @function
   * @private
   * 
   * @param {baja.comm.Batch} batch a batch to use to unsubscribe children 
   * (should be the same batch used to unsubscribe the parent component)
   */
  PropertySheetListView.prototype.unsubscribeChildren = function (batch) {
    if (!this.childrenSubscriber || !this.value.getType().isComponent()) {
      return;
    }
    
    this.childrenSubscriber.unsubscribeAll({ batch: batch });
  };
  
  function setStatus(element, status) {
    statusGradients.applyStatusCSS(element, status);
  }
  
  /**
   * A PropertySheet when making a list item will add a CSS class
   * <code>editable</code> to the value display div if the property
   * can be edited.
   * 
   * @name niagara.propSheet.PropertySheetListView#makeListItem
   * @function
   */
  util.aop.after(PropertySheetListView.prototype, 'makeListItem', function (args, li) {
    var complex = args[0], 
        prop = args[1], 
        status,
        statusBg = $('<div class="statusBg"></div>').prependTo(li);
    
    if (util.slot.isEditable(prop)) {
      li.find('div.display').addClass('editable');
    }
    
    if (prop.getType().is('baja:IStatus')) {
      status = baja.Status.getStatusFromIStatus(complex.get(prop));
      setStatus(statusBg, status);
    }
  });
  
  util.aop.after(PropertySheetListView.prototype, 'updateDisplay', function (args) {
    var prop = args[0],
        li,
        status;
    
    if (prop.getType().is('baja:IStatus')) {
      li = this.elements.ul.find('li.property#' + encodePageId(prop));
      status = baja.Status.getStatusFromIStatus(this.value.get(prop));
      setStatus(li.find('div.statusBg'), status);
    }
  });
  
  util.aop.after(PropertySheetListView.prototype, 'refresh', function () {
    this.fieldEditorMap = {};
  });

  util.aop.before(PropertySheetListView.prototype, 'refresh', function (args) {
    var callbacks = callbackify(args[0]),
        win = $(window),
        scrollTop = win.scrollTop(),
        that = this;
    util.aop.before(callbacks, 'ok', function () {
      win.scrollTop(scrollTop);
      that.subscribeChildren();
    });
  });
  
  /**
   * When expanding a property sheet slot, if the field editor contains an
   * editable text field, it will automatically be focused/selected.
   * 
   * @name niagara.propSheet.PropertySheetListView#doExpand
   * @function
   */
  util.aop.before(PropertySheetListView.prototype, 'doExpand', function (args) {
    var callback = args[1];
    if (typeof callback === 'function') {
      callback = util.aop.before(function () {
        var input = $(this).find('input[type=text]:eq(0)');
        if (!input.attr('disabled') && !input.attr('readonly')) {
          input.focus().select();
        }
      }, callback);
    }
    args[1] = callback;
  });

  
  
  
////////////////////////////////////////////////////////////////
// PropertySheet
////////////////////////////////////////////////////////////////
  /**
   * A property sheet view consisting of a JQM page whose content is formed
   * by a <code>PropertySheetListView</code>.
   * 
   * @class
   * @extends niagara.util.mobile.PageView
   * @memberOf niagara.propSheet
   */
  PropertySheet = PageView.subclass('subscribable');
  
  PropertySheet.prototype.instantiateContentView = function () {
    return new PropertySheetListView();
  };
  
  /**
   * Returns the display name for this page view.
   * 
   * @name niagara.propSheet.PropertySheet#getName
   * @function
   * @returns {String} the component display name - if none is returned
   * "Station" will be displayed
   */
  util.aop.after(PropertySheet.prototype, 'getName', function afterGetName(args, name) {
    return name || getMobileLex().get('propsheet.station');
  });
  
  /**
   * Adds a <code>viewloadvalue</code> listener to this view's div. Once the
   * property sheet's value has been fully loaded, it will show/hide the commands
   * button depending on whether the property sheet is showing a Component or
   * just a Struct. It will also update the footer bar to 
   * disable the Save button, and either enable or disable the Actions button
   * depending on whether the component has any actions to be fired.
   * 
   * @name niagara.propSheet.PropertySheet#armHandlers
   * @function
   * @see niagara.util.mobile.PageView#armHandlers
   */
  util.aop.after(PropertySheet.prototype, 'armHandlers', function armHandlers() {
    var that = this;
    that.$dom.bind('viewloadvalue', function (event, view) {
      if (view === that) {
        var value = that.value,
            headerDiv = that.getHeaderDiv(),
            commandsButton = headerDiv.find('a.commandsButton');
        
        if (value.getType().isComponent()) {
          commandsButton.show();
        } else {
          commandsButton.hide();
        }
        that.updateBars();
      }
    });
  });
  
  /**
   * Appends a footer bar to the JQM page for this view, and a view button to
   * the page's header. The footer bar contains link buttons for Save, Refresh,
   * and Actions.
   * 
   * @name niagara.propSheet.PropertySheet#createPage
   * @function
   * @see niagara.util.mobile.PageView#createPage
   */
  util.aop.after(PropertySheet.prototype, 'createPage', function createPage(args, page) {
    var headerDiv = page.children(':jqmData(role=header)'),
        lex = getMobileLex(),
        navbar = $(navbarHtml.patternReplace({
          title: escapeHtml(lex.get('loading'))
        })),
        footer = $(footerBarHtml.patternReplace({
          save: escapeHtml(lex.get('save')),
          refresh: escapeHtml(lex.get('refresh')),
          actions: escapeHtml(lex.get('actions'))
        }));
    
    page.append(footer);
    mobileUtil.preventNavbarHighlight(footer);
    
    headerDiv.append(commands.getCommandsButton().removeClass(this.$ignoreProfileClasses).click(commands.showCommandsHandler));
    headerDiv.append(navbar);
    
    return page;
  });
  
  /**
   * Updates the footer bar each time the property sheet is marked modified
   * (user-entered change to a field editor).
   * 
   * @name niagara.propSheet.PropertySheet#setModified
   * @function
   * @see niagara.util.mobile.View#setModified
   */
  util.aop.after(PropertySheet.prototype, 'setModified', function setModified() {
    this.updateBars();
  });
  
  /**
   * When removing a property sheet from the DOM structure, also examine every
   * datebox input (<code>$(this).jqmData('datebox').pickPage</code>) and remove
   * it from the JQM page container.
   * 
   * @name niagara.propSheet.PropertySheet#stop
   * @function
   * @see niagara.util.mobile.mixins.subscribableMixin.stop
   */
  util.aop.before(PropertySheet.prototype, 'stop', function (args) {
    var callbacks = callbackify(args[0]),
        that = this;
    util.aop.before(callbacks, 'ok', function () {
      that.getContentDiv().find('input:jqmData(role=datebox)').each(function () {
        var datebox = $(this).jqmData('datebox'),
            pickPage = datebox && datebox.pickPage;
        if (pickPage) {
          pickPage.remove();
        }
      });
    });
    return [callbacks];
  });
  
  /**
   * Finds actions on this property sheet's component that are fireable (not
   * hidden).
   * 
   * @name niagara.propSheet.PropertySheet#hasFireableActions
   * @function
   * 
   * @returns {Boolean} true if this property sheet's component has an action
   * that can be fired
   */
  PropertySheet.prototype.hasFireableActions = function () {
    return !!this.value.getSlots(function filter(slot) { 
      return util.slot.isFireable(slot);
    }).next();
  };
  
  /**
   * Enables/disables the save/action buttons on the footer, depending on
   * whether the given sheet has unsaved changes / fireable actions. The 
   * enable/disable is performed simply by adding the <code>ui-disabled</code>
   * class to the <code>&lt;a&gt;</code> links contained in the footer buttons.
   * JQM automatically checks for the presence of this class and prevents
   * click handlers from executing if found.
   * <p>
   * This function also sets the menu button in the header bar to turn red
   * depending on whether the sheet has unsaved changes.
   * 
   * @name niagara.propSheet.PropertySheet#updateBars
   * @function
   * @private
   * 
   * @param {niagara.propSheet.PropertySheet} sheet the property sheet
   * whose state the footer bar should reflect
   */
  PropertySheet.prototype.updateBars = function updateBars() {
    var page = this.$dom,
        header = this.getHeaderDiv(),
        footer = this.getFooterDiv(),
        
        footerLinks = footer.find('a'),
        actionsLink = footerLinks.filter('.actionsLink'),
        commandsButton = header.find('.commandsButton'),
        saveLink = footerLinks.filter('.saveLink');
    
    actionsLink.toggleClass('ui-disabled', !this.hasFireableActions());
    saveLink.toggleClass('ui-disabled', !this.isModified());
    commandsButton.toggleClass('red', this.isModified());
  };
  
  /** 
   * @namespace
   * @name niagara.propSheet
   */
  util.api('niagara.propSheet', {
    'public': {
      PropertySheet: PropertySheet,
      PropertySheetListView: PropertySheetListView
    },
    'private': {
      loadEditor: loadEditor,
      validateAndSaveEditors: validateAndSaveEditors
    }
  });
}());

