//Copyright 2011, Tridium, Inc. All Rights Reserved.

/**
 * History app utilities
 *
 * @author Logan Byam
 * @date March 23 2012
 * @version 0.1.0.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint white: true, bitwise: true, regexp: true, sloppy: true */

/*global $, baja, niagara, decodeURIComponent, decodeURI*/

(function () {
  
      //imports
  var util = niagara.util,
      fe = niagara.fieldEditors,
      callbackify = util.callbackify,
      MILLIS_IN_DAY = baja.RelTime.MILLIS_IN_DAY,

      //local vars
      mobileLex = util.lazyLex('mobile'),
      
      //constants
      ABS_TIME_RANGE_TYPE = 'baja:AbsTimeRange',
      HISTORY_SCHEME = 'history',
      MTD_GET_HISTORY_RANGE = 'getHistoryRange',
      SSH_TYPE = 'mobile:HistoryServerSideHandler',
      START_AFTER_END_LEX_KEY = 'history.message.startAfterEnd',
      STRING_TYPE = 'baja:String',
      TIME_RANGE_TAG_NAME = 'timeRange',

      TIME_RANGE_REGEX = /start=([^;]*);end=([^;\|]*)/,
      PERIOD_REGEX = /period=([^;\|%]*)/,
      
      //exports
      HistoryQueryParamsEditor;

  
  /**
   * Attempts to parse a period ID from a history query params string using the
   * <code>period</code> param.
   * 
   * @memberOf niagara.util.mobile.history
   * @param {String} a history query params string
   * @returns {String} the period ID, or undefined if could not be found
   */
  function parsePeriodId(paramString) {
    var periodMatch = PERIOD_REGEX.exec(paramString);
    if (periodMatch) {
      return periodMatch[1];
    }
  }
  
  /**
   * Attempts to parse a <code>baja:AbsTimeRange</code> from a history query
   * params string using the <code>start</code> and <code>end</code> params.
   * 
   * @memberOf niagara.util.mobile.history

   * @param {String} a history query params string
   * @returns {baja.Struct} a <code>baja:AbsTimeRange</code>, or undefined if
   * could not be parsed
   */
  function parseStartEnd(paramString) {
    var rangeMatch = TIME_RANGE_REGEX.exec(decodeURI(paramString)),
        decodeFromString = baja.AbsTime.DEFAULT.decodeFromString;
    
    if (rangeMatch) {
      return baja.$(ABS_TIME_RANGE_TYPE, {
        startTime: decodeFromString(decodeURIComponent(rangeMatch[1])),
        endTime: decodeFromString(decodeURIComponent(rangeMatch[2]))
      });
    }
  }
  
  
  /**
   * Constructs a <code>BAbsTimeRange</code> using a tag from a 
   * <code>BDynamicTimeRangeType</code>, relative to the current system time.
   * 
   * @private
   * @memberOf niagara.util.mobile.history
   * @param {String} tag the selected tag from a 
   * <code>BDynamicTimeRangeType</code>
   * @returns {baja.Struct} a <code>baja:AbsTimeRange</code>
   */
  function getDynamicAbsTimeRange(tag) {
    var startDate, 
        endDate, 
        now = new Date(),
        year = now.getFullYear(),
        month = now.getMonth(),
        day = now.getDate(),
        dayOfWeek = now.getDay(),
        time = now.getTime(),
        // convert offset from minutes to millis, and JS returns a negative 
        // value for some reason
        offset = now.getTimezoneOffset() * -60000;
    
    switch (tag) {
    
    case 'today':
      startDate = new Date(year, month, day);
      endDate = new Date(year, month, day + 1);
      break;
      
    case 'last24Hours':
      startDate = new Date(time - MILLIS_IN_DAY);
      endDate = now;
      break;
      
    case 'yesterday':
      startDate = new Date(year, month, day - 1);
      endDate = new Date(year, month, day);
      break;
      
    case 'weekToDate':
      startDate = new Date(year, month, day - dayOfWeek);
      endDate = now;
      break;
      
    case 'lastWeek':
      startDate = new Date(year, month, day - dayOfWeek - 7);
      endDate = new Date(year,  month, day - dayOfWeek);
      break;
      
    case 'last7Days':
      startDate = new Date(year, month, day - 7);
      endDate = new Date(year, month, day);
      break;
      
    case 'monthToDate':
      startDate = new Date(year, month, 1);
      endDate = now;
      break;
      
    case 'lastMonth':
      startDate = new Date(year, month - 1, 1);
      endDate = new Date(year, month, 1);
      break;
      
    case 'yearToDate':
      startDate = new Date(year, 0, 1);
      endDate = now;
      break;
      
    case 'lastYear':
      startDate = new Date(year - 1, 0, 1);
      endDate = new Date(year, 0, 1);
      break;
      
    default:
      startDate = now; endDate = now;
    }
    
    return baja.$('baja:AbsTimeRange', {
      startTime: baja.AbsTime.make({ jsDate: startDate, offset: offset }),
      endTime: baja.AbsTime.make({ jsDate: endDate, offset: offset })
    });
  }
  
  /**
   * @memberOf niagara.util.mobile.history
   */
  function toTimeRange(paramString) {
    var timeRange = parseStartEnd(paramString),
        periodId;
    
    if (timeRange) {
      return timeRange;
    } else {
      periodId = parsePeriodId(paramString);
      if (periodId) {
        return getDynamicAbsTimeRange(periodId);
      }
    }
  }
  
  /**
   * Takes a value (either a <code> BAbsTimeRange</code> or a range type tag)
   * and, optionally, delta mode, and converts them into a usable history
   * query param string.
   * 
   * @memberOf niagara.util.mobile.history
   * @param {String|baja.Struct} value a <code>baja:AbsTimeRange</code> or
   * <code>BDynamicTimeRangeType</code> tag to convert to a history query param
   * string
   * @param {Boolean} isDelta true if delta mode is turned on
   */
  function toParamString(value, isDelta) {
    var paramString = '', 
        deltaString = isDelta ? 'delta=true' : '',
        start, end, startTime, endTime;
    
    if (value.getType().is(ABS_TIME_RANGE_TYPE)) {
      start = value.getStartTime();
      end = value.getEndTime();
      startTime = start.encodeToString();
      endTime = end.encodeToString();
      
      paramString = '?start=' + startTime + ';end=' + endTime;
    } else if (value !== 'none' && value !== 'timeRange') {
      paramString = '?period=' + value;
    }
    
    if (isDelta) {
      if (paramString) {
        paramString += ';' + deltaString;
      } else {
        paramString = '?' + deltaString;
      }
    }
    
    return paramString;
  }
  
  /**
   * Retrieves the default date range for the currently viewed history from the
   * server, using the <code>mobile:HistoryServerSideHandler</code>'s
   * <code>getHistoryRange</code> method.
   * 
   * @memberOf niagara.util.mobile.history
   * @private
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @returns {baja.Struct} the default <code>BAbsTimeRange</code> (to be passed
   * to the ok callback)
   */
  function getDefaultTimeRange(callbacks) {
    callbacks = callbackify(callbacks);
    
    //set our initial time query range
    util.ord.get(niagara.view.app.appOrd, function (component) {
      component.serverSideCall({
        typeSpec: SSH_TYPE,
        methodName: MTD_GET_HISTORY_RANGE,
        value: baja.Ord.make(niagara.view.history.query),
        fail: callbacks.fail,
        ok: callbacks.ok
      });
    });
  }
  
  /**
   * A field editor for a parameter string used by a history query. Includes
   * a dropdown for a <code>BDynamicTimeRangeType</code> and an editor for
   * a <code>BAbsTimeRange</code> (for when "Time Range" is selected).
   * <p>
   * Registered on <code>baja:String</code> and instantiated using key 
   * <code>"historyQueryParams"</code>.
   *  
   * @class
   * @name niagara.util.mobile.history.HistoryQueryParamsEditor
   * @extends niagara.fieldEditors.mobile.MobileFieldEditor
   */
  HistoryQueryParamsEditor = (function HistoryQueryParamsEditor() {

    /**
     * Parses out information from a history query params string. Once parsed,
     * <code>callbacks.ok</code> will be called with the following two
     * parameters:
     * 
     * <ul>
     * 
     * <li><code>timeRange</code>: the <code>BAbsTimeRange</code> represented
     * by these history params (either encoded directly into the string using
     * <code>start</code> and <code>end</code> parameters, or if the string
     * represents a period, a time range constructed via
     * <code>niagara.util.mobile.history.getDynamicAbsTimeRange</code>)</li>
     * 
     * <li><code>periodId</code>: if the value is a params string
     * representing a period (using a <code>BDynamicTimeRangeType</code>), this
     * will be the period ID string, e.g. <code>last24Hours</code> - otherwise 
     * undefined</li>
     * 
     * <li><code>isDelta</code>: true if delta mode is turned on</li>
     * 
     * </ul>
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor.parseHistoryParams
     * @function
     * @private
     * @inner
     * @param {String|baja.Struct} value either a history params string
     * (or an ORD containing a history params string), or a
     * <code>baja:AbsTimeRange</code> to convert to a history params string.
     */
    function parseHistoryParams(value, callbacks) {
      callbacks = callbackify(callbacks);
      
      var type = value.getType(),
          periodId,
          isDelta = false,
          timeRange;
      
      if (type.is(STRING_TYPE)) { //we have a params string!
        periodId = parsePeriodId(value);
        if (!periodId) { //it's not a period, so try and parse it from start/end params
          timeRange = parseStartEnd(value);
          //set period dropdown accordingly
          if (timeRange) {
            periodId = 'timeRange';
          }
        }
        // else { if it IS a period, we're going to pull down the default range 
        // from the station }
        
        isDelta = (value.indexOf('delta=true') >= 0);
      } else if (type.is(ABS_TIME_RANGE_TYPE)) {
        timeRange = value;
      }
      
      if (timeRange) {
        callbacks.ok(timeRange, periodId, isDelta);
      } else {
        // no time range, no start/end params - so pull down a
        // default time range from the station
        getDefaultTimeRange(function (timeRange) {
          callbacks.ok(timeRange, periodId, isDelta);
        });
      }
    }
    
    /**
     * Creates a JQM select dropdown containing the different values of a
     * <code>bql:DynamicTimeRangeType</code>. These tags will be appended
     * directly into the history query as <code>periodId</code>.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor.makeSelect
     * @function
     * @private
     * @inner
     */
    function makeSelect() {
      var select = $('<select data-role="selectmenu" data-theme="b"/>'),
          option = $('<option value="none"/>')
            .text(mobileLex.get('loading'))
            .appendTo(select);
      
      return select;
    }
    
    function populateSelect(select) {
      var rangeType = baja.$('bql:DynamicTimeRangeType'),
          range = rangeType.getRange();
      
      select.empty();
      $('<option/>')
        .attr('value', 'none')
        .text(mobileLex.get('history.selectTimeRange'))
        .appendTo(select);
      
      baja.iterate(range.getOrdinals(), function (ordinal) {
        $('<option/>')
          .attr('value', range.getTag(ordinal))
          .text(rangeType.make(ordinal).getDisplayTag())
          .appendTo(select);
      });
      
      select.trigger('change');
    }
    
    var deltaButtonHTML =
      '<div class="deltaButtonContainer">' +
        '<a data-theme="a" data-role="button" data-icon="delta" title="Delta" ' +
          'class="ui-checkbox-off" data-iconpos="notext"/>' +
      '</div>';
    /**
     * Appends the <code>BDynamicTimeRangeType</code> select, and (optionally)
     * a toggle button for delta mode, then instantiates
     * the default editor for a <code>BAbsTimeRange</code> below it.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#doInitializeDOM
     * @function
     * @see niagara.fieldEditors.BaseFieldEditor#doInitializeDOM
     */
    function doInitializeDOM(dom, callbacks) {
      var that = this,
          select = that.$select = makeSelect(),
          deltaButton = $(deltaButtonHTML);
      
      that.$deltaButton = deltaButton.children('a');
      
      dom.append(select);

      if (that.params.showDelta) {
        dom.append(deltaButton);
      }

      dom.addClass('historyQueryParams');
      
      fe.makeFor({
        value: baja.$(ABS_TIME_RANGE_TYPE),
        parent: this
      }, function (editor) {
        that.$editor = editor;
        editor.initializeDOM(dom, callbacks);
      });
    }
    
    /**
     * Loads up a history query params string. The <code>BAbsTimeRange</code>
     * parsed from the params string will be loaded into the 
     * <code>BAbsTimeRange</code> field editor.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#doLoadValue
     * @function
     * @param {String} value a history query params string
     * @see niagara.fieldEditors.BaseFieldEditor#doLoadValue
     * @see niagara.util.mobile.history.HistoryQueryParamsEditor.parseHistoryParams
     */
    function doLoadValue(value, callbacks) {
      var that = this;
      
      parseHistoryParams(value, function (timeRange, periodId, isDelta) {
        populateSelect(that.$select);
        
        if (periodId) {
          that.$select.val(periodId).selectmenu('refresh');
          that.updateReadonlyStatus();
        }
        
        that.setDelta(isDelta);
        
        that.$editor.loadValue(timeRange, {
          ok: function () {
            callbacks.ok();
          },
          fail: callbacks.fail
        });
      });
    }

    /**
     * If "Time Range" is selected, then the <code>BAbsTimeRange</code>'s
     * current value will be converted to a <code>start</code>/<code>end</code>
     * history query param string. Otherwise, the currently selected tag in the
     * dropdown will be converted into a <code>period</code> param string. 
     * 
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#getSaveData
     * @function
     * @see niagara.fieldEditors.BaseFieldEditor#getSaveData
     */
    function getSaveData(callbacks) {
      var that = this,
          selectedTag = that.$select.val(),
          delta = that.$isDelta;
      
      if (selectedTag === TIME_RANGE_TAG_NAME) {
        this.$editor.getSaveData(function (timeRange) {
          callbacks.ok(toParamString(timeRange, delta));
        });
      } else {
        callbacks.ok(toParamString(selectedTag, delta));
      }
    }
    
    /**
     * Whenever the user selects a new value from the
     * <code>BDynamicTimeRangeType</code> dropdown, calculate the proper
     * <code>BAbsTimeRange</code> (using <code>getTimeRange()</code>) and load
     * it into the <code>BAbsTimeRange</code> field editor. If the user selects
     * anything but "Time Range" (manually entering times), set the
     * <code>BAbsTimeRange</code> field editor to readonly.
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#armHandlers
     * @function
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#armHandlers
     */
    function armHandlers() {
      var that = this;
      
      that.$select.change(function () {
        that.updateReadonlyStatus();
      });
      
      that.$deltaButton.click(function () {
        that.setDelta(!that.$isDelta);
      });
    }
    
    /**
     * Check to see if the given time range is valid for a history query (that
     * the start time is before the end time).
     * 
     * @name niagara.util.mobile.history.HistoryQueryParamsEditor#validate
     * @function
     * @see niagara.fieldEditors.mobile.MobileFieldEditor#validate
     */
    function validate(paramString, callbacks) {
      var timeRange = parseStartEnd(paramString),
          startDate, startMillis, endDate, endMillis, show;

      if (!timeRange) {
        // we're a period instead of a start/end range - so can't be invalid
        return callbacks.ok();
      }
      
      startDate = timeRange.getStartTime();
      startMillis = startDate.getJsDate().getTime();
      endDate = timeRange.getEndTime();
      endMillis = endDate.getJsDate().getTime();
      
      if (startMillis >= endMillis) {
        show = baja.TimeFormat.SHOW_DATE | baja.TimeFormat.SHOW_TIME;
        callbacks.fail(mobileLex.get({
          key: START_AFTER_END_LEX_KEY,
          args: [ startDate.toString({ show: show }), endDate.toString({ show: show }) ]
        }));
      } else {
        callbacks.ok();
      }
    }
    
    function postCreate() {
      
      this.$isDelta = false;

      /**
       * Updates the <code>BAbsTimeRange</code> editor's readonly status -
       * <code>false</code> if "Time Range" is selected from the dropdown, and 
       * true otherwise.
       * 
       * @name niagara.util.mobile.history.HistoryQueryParamsEditor#updateReadonlyStatus
       * @function
       * @private
       */
      this.updateReadonlyStatus = function () {
        var selectedTag = this.$select.val(),
            timeRangeSelected = (selectedTag === TIME_RANGE_TAG_NAME);
        
        this.$editor.setReadonly(!timeRangeSelected);
      };
      
      /**
       * Sets/unsets delta mode. This automatically updates the highlighting for
       * the delta button as well.
       * 
       * @name niagara.util.mobile.history.HistoryQueryParamsEditor#setDelta
       * @function
       * @param {Boolean} delta
       * @private
       */
      this.setDelta = function (delta) {
        this.$isDelta = delta;

        this.$deltaButton
          .toggleClass('ui-checkbox-off', !delta)
          .toggleClass('ui-checkbox-on ui-btn-active', delta);
        
        this.setModified(true);
      };
    }
    
    return fe.defineEditor(fe.mobile.MobileFieldEditor, {
      doInitializeDOM: doInitializeDOM,
      doLoadValue: doLoadValue,
      getSaveData: getSaveData,
      armHandlers: armHandlers,
      postCreate: postCreate,
      validate: validate
    });
    
  }());
  
  fe.register(STRING_TYPE, HistoryQueryParamsEditor, 'historyQueryParams');
  
  
  /**
   * @namespace
   * @name niagara.util.mobile.history
   */
  util.api('niagara.util.mobile.history', {
    'public': {
      HistoryQueryParamsEditor: HistoryQueryParamsEditor,
      parsePeriodId: parsePeriodId,
      parseStartEnd: parseStartEnd,
      toParamString: toParamString,
      toTimeRange: toTimeRange
    },
    'private': {
      getDefaultTimeRange: getDefaultTimeRange
    }
  });
}());
