//Copyright 2011, Tridium, Inc. All Rights Reserved.

/**
 * Niagara Mobile History
 *
 * @author Jason Spangler
 * @date Jan 19, 2012
 * @version 0.1.0.0
 */

//JsLint options (see http://www.jslint.com)
/*jslint white: true, browser: true */
/*global niagara, baja, $ */

(function historyTable() {

  // Use ECMAScript 5 Strict Mode
  "use strict";  

  niagara.util.require(
      '$.mobile',
      'niagara.util',
      'niagara.util.View',
      'niagara.util.mobile.history',
      'niagara.util.mobile.touchscroll',
      'niagara.fieldEditors',
      'niagara.util.mobile.commands',
      'niagara.util.mobile.dialogs'
  );

      //imports
  var util = niagara.util,
      mobileUtil = util.mobile,
      historyUtil = mobileUtil.history,
      touchScroll = mobileUtil.touchscroll.touchScroll,
      PageView = mobileUtil.PageView,
      commands = mobileUtil.commands,
      dialogs = mobileUtil.dialogs,
      View = util.View,
      fieldEditors = niagara.fieldEditors,
      AsyncCommand = commands.AsyncCommand,
      
      
      //local vars
      tableView,
      mobileLex = util.lazyLex('mobile'),
      currentParamString = niagara.view.history.query,
      
      colMenuBtnHtml = '<a class="colMenuBtn" href="#" data-role="button" data-icon="check">{display}</a>',
      colMenuHtml = '<div class="colMenu"><div data-role="fieldcontain"><fieldset data-role="controlgroup"></fieldset></div>',
      colMenuCheckboxHtml = '<input type="checkbox" name="checkbox-{i}" id="checkbox-{i}" data-theme="a" />' +
        '<label for="checkbox-{i}" >{display}</label>',
        
      //constants
      TIME_RANGE_BTN_DISPLAY_KEY = 'history.timeRange.select',
      TABLE_VIEW_ID = 'mobile:MobileHistoryAppTableView',
      HISTORY_SCHEME = 'history',
      PAGE_ID = '#historyTable',
      CONTENT_DIV_ID = '#historyTableContent',
      PAGE_TITLE_ID = '#historyTableTitle',
      DEFAULT_LIMIT = 20,
      SSH_TYPE = 'mobile:HistoryServerSideHandler',
      MTD_GET_HISTORY_RANGE = 'getHistoryRange',
      COL_DISPLAY = mobileLex.get('history.columns.menu');

  /**
   * Default ok callback handler.
   *
   * @private
   */
  function defOk() {    
    $.mobile.hidePageLoadingMsg();
  }

  /**
   * Default fail callback handler.
   *
   * @private
   */
  function defFail(err) {
    dialogs.unrecoverableError(err);
  }

  /**
   * This method forces a redirect of the current page to the chart view
   * of the current view ORD with the configured time query parameters (if any).
   * 
   */
  function redirect(paramString) {
    var query, queryList, historyOrd,
        ord = niagara.view.history.query;

    //parse the view ord from the view
    queryList = baja.Ord.make(ord).parse();
    queryList.getCursor().each(function (value) {
      if (value.getSchemeName() === HISTORY_SCHEME) {
        historyOrd = value.toString();
        //remove trailing parameters if present
        if(historyOrd.indexOf('?') > 0) {
          historyOrd = historyOrd.substring(0,historyOrd.indexOf('?')); 
        }
        
        //add time query range if defined
        if (paramString) {
          historyOrd += paramString;
        }
        
        if (query !== undefined) { 
          query += "|" + historyOrd; 
        } else { 
          query = historyOrd;
        }
      }
    });

    query = baja.Ord.make(query, {
      viewQuery: new baja.ViewQuery(TABLE_VIEW_ID)
    });
    tableView.loadValue(query);
  }  



//Main


  function buildTimeQueryEditor(callbacks) {
    var showDelta = tableView && 
                    tableView.$recordType && 
                    tableView.$recordType.is('baja:Number');
    
    //open a dialog with our editor
    dialogs.fieldEditor({
      title: mobileLex.get(TIME_RANGE_BTN_DISPLAY_KEY),
      value: currentParamString,
      showDelta: showDelta, 
      key: 'historyQueryParams',
      ok: function (paramString, cb) {
        currentParamString = paramString;
        redirect(paramString);
        cb.ok();
      },
      fail: dialogs.error,
      callbacks: callbacks
    });
  }
  
  /**
   * Binds the Show Columns menu to the given DOM anchor.
   * 
   * @memberOf niagara.mobile.table.DataTableView#
   * 
   * @param anchor - DOM anchor to bind our menu display to.
   */
  function bindShowColumnsMenu(anchor) {
    var colMenu = $(colMenuHtml),
        anchorInner = anchor.children('.ui-btn-inner'),
        contents = colMenu.find('fieldset'),
        expandedClass = 'expanded',
        slideSpeed = 'medium';

    colMenu.appendTo($(anchor)).hide();
    
    anchorInner.click(function () {
      if (anchor.hasClass(expandedClass)) {
        colMenu.slideUp(slideSpeed, function () {
          anchor.removeClass(expandedClass);
        });
      } else {
        anchor.addClass(expandedClass);
        colMenu.slideDown(slideSpeed);
      }
    });
    
    //an arrow up button for when the columns list is expanded, will be
    //hidden when not expanded via css
    anchorInner.append($('<span class="ui-icon ui-icon-arrow-u"/>'));

    //append check boxes for each column to our column selector menu
    baja.iterate(tableView.columns, function (col) {

      //create a code-friendly version of the display name
      var id = col.getDisplayName().replace(/\s/g, "-"),
          colMenuCheckBox = $(colMenuCheckboxHtml.patternReplace({ 
            i: id, 
            display: col.getDisplayName() 
          }));

      //create the menu item checkbox with function and append
      colMenuCheckBox
        //include a click function for each checkbox item to show/hide
        //table columns
        .attr('checked', 'checked')
        .appendTo(contents)
        .click(function () {
          //if the checkbox is currently checked, our click event will
          //uncheck the checkbox, which means we really want to hide
          //the indicated table column. If our checkbox is not checked,
          //it means it is about to be checked, and we want to show
          //the selected column
          tableView.toggleColumnDisplay(id);
        });
    });
    colMenu.find('input').checkboxradio().addClass('ui-btn-icon-left');
  }
  
  /**
   * This method creates the show columns menu and appends the menu to
   * the specified target div DOM element.
   * 
   * @memberOf niagara.mobile.table.DataTableView#
   * 
   * @param targetDiv - div DOM element to append our show columns menu button.
   */
  function makeShowColumnsMenu(targetDiv) {
    var colMenuBtn = $(colMenuBtnHtml.patternReplace({ 
          display: COL_DISPLAY
        }));
    targetDiv.append(colMenuBtn);
    bindShowColumnsMenu(colMenuBtn).hide();
  }

  /**
   * this method sets the title for this page.
   */
  function setPageTitle () {
    var pageTitle = niagara.view.history.displayName;
    document.title = pageTitle;
    $(PAGE_TITLE_ID).text(pageTitle);
  }

  /**
   * This method initializes our table content by creating a new menu and
   * creating the footer navigation bars along with loading the table data.
   */
  function initTable() {    
    var ord = niagara.view.history.query,
        page = $(PAGE_ID),
        pageHeader = page.children(':jqmData(role=header)'),
        menuBtn = pageHeader.find('a.colMenuBtn'),
        footer = page.children(':jqmData(role=footer)'),
        contentDiv = $(CONTENT_DIV_ID);

    //create our table view content
    tableView = new niagara.mobile.table.DataTableView({ 
      limit: DEFAULT_LIMIT, 
      ord: ord 
    });

    tableView.onpageswap = function (contentDiv) {
      var tableContainer = contentDiv.find('.tableContainer');
      touchScroll(tableContainer, { horiz: true, showBlockade: true });
    };
    
    tableView.initializeDOM(contentDiv, function () {
      tableView.bindFooter(footer);
      
      tableView.loadValue(baja.Ord.make(ord), {
        ok: function () {

          //bind our page functionality
          bindShowColumnsMenu(menuBtn);
        },
        fail: defFail
      });
    });
  }

  /**
   * This method assumes the view is already initialized and simply loads the
   * data resolved from the current view ORD into the table model.
   */
  function loadTableData() {
    var ord = niagara.view.history.query;
    tableView.loadValue(baja.Ord.make(ord), {
      ok: baja.ok,
      fail: defFail
    });
  }

  /**
   * Initialize our view
   */
  function initAppView(){
    var page = $(PAGE_ID),
        pageHeader = page.children(':jqmData(role=header)'),
        queryCmd = new AsyncCommand(mobileLex.get(TIME_RANGE_BTN_DISPLAY_KEY), buildTimeQueryEditor),
        cmds = [];
      
    if (niagara.view.profile.showLogoutCmd) {
      cmds.push(commands.getLogoutCmd());
    }
      
    if (niagara.view.profile.showHome) {
      cmds.splice(0, 0, commands.getHomeCmd());
    }
    
    commands.setDefaultCommands(cmds);
    commands.prependDefaultCommand(queryCmd);

    $("#commandsButton").click(commands.showCommandsHandler);
    
    setPageTitle();

    //set our page header CSS
    pageHeader.css({
      '-webkit-transform': 'none',
      'overflow': 'visible'
    });

    if (tableView === undefined) {
      initTable();
    } else {
      loadTableData();
    }
  }

  baja.started(initAppView);

}());