//Copyright 2011, Tridium, Inc. All Rights Reserved.

/**
 * Niagara Mobile History
 *
 * @author Jason Spangler
 * @version 0.1.0.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint white: true, browser: true, bitwise: true, regexp: true */
/*global niagara, baja, $ */

(function historyChart() {
  
  // Use ECMAScript 5 Strict Mode
  "use strict";  
  
  niagara.util.require(
    '$.mobile',
    'niagara.util.View',
    'niagara.util.mobile.commands',
    'niagara.util.mobile.dialogs',
    'niagara.util.mobile.history'
  );
  
      //imports
  var util = niagara.util,
      mobileUtil = util.mobile,
      historyUtil = mobileUtil.history,
      commands = mobileUtil.commands,
      dialogs = mobileUtil.dialogs,
      fieldEditors = niagara.fieldEditors,
      callbackify = util.callbackify,
      View = util.View,
      AsyncCommand = commands.AsyncCommand,
      app = niagara.view.app,
      
      //constants
      TIME_RANGE_BTN_DISPLAY_KEY = 'history.timeRange.select',
      DATA_ROLE_HEADER_ID = ':jqmData(role=header)',
      CHECKBOX_TOGGLE_CLASSES = 'ui-checkbox-off ui-checkbox-on ui-btn-active',
      CHECKBOX_OFF_CLASS = 'ui-checkbox-off',
      TS_COL = 'timestamp',
      VALUE_COL = 'value',
      PRECISION_FACET_NAME = 'precision',
      UNITS_FACET_NAME = 'units',
      ABS_TIME_TYPE = 'baja:AbsTime',
      CHART_VIEW_ID = 'mobile:MobileHistoryAppChartView',
      HISTORY_SCHEME = 'history',
      PAGE_ID = '#historyChart',
      CONTENT_DIV_ID = '#historyChartContent',
      PAGE_TITLE_ID = '#historyChartTitle',
      HIGHLIGHT_BTN_ID = '#chartHighlightBtn',
      RESET_BTN_ID = '#chartResetBtn',
      CHART_DIV_HTML = '<div id="{id}" class="historyChart" />',
      HIGHLIGHT_FORMAT_STRING = '<table class="jqplot-highlighter">' +
                                  '<tr>' +
                                    '<td>date:</td>' +
                                    '<td>{dateFormat}</td>' +
                                  '</tr>' +
                                  '<tr>' +
                                    '<td>value:</td>' +
                                    '<td>{valueFormat}</td>' +
                                  '</tr>' +
                                '</table>',
      RECORD_LIMIT = app.recordLimit || 1000,
      WARNING_THRESHOLD = app.warningThreshold || 1000,

      //local vars
      mobileLex = util.lazyLex('mobile'),
      warningDialogConfirmed = false,
      chartView,
      currentParamString = niagara.view.history.query,
      currentTimeRange = historyUtil.toTimeRange(currentParamString),
      
      //exports
      HistoryChartView;
  
  /**
   * Default ok callback handler.
   *
   * @memberOf niagara.mobile.charts.HistoryChartView#
   */
  function defOk() {    
    $.mobile.hidePageLoadingMsg();
  }

  /**
   * Default fail callback handler.
   *
   * @memberOf niagara.mobile.charts.HistoryChartView#
   */
  function defFail(err) {
    dialogs.unrecoverableError(err);
  }
  
  /**
   * This method forces a redirect of the current page to the chart view
   * of the current view ORD with the configured time query parameters (if any).
   * 
   * @memberOf niagara.mobile.charts
   * 
   * @param {String} paramString 
   */
  function redirect(paramString) {
    var query, queryList, queryParams, historyOrd,
        ord = niagara.view.history.query;
    
    
    //parse the view ord from the view
    queryList = baja.Ord.make(ord).parse();
    
    queryList.getCursor().each(function (value) {
      if (value.getSchemeName() === HISTORY_SCHEME) {
        historyOrd = value.toString();
        //remove trailing parameters if present
        if (historyOrd.indexOf('?') > 0) {
          historyOrd = historyOrd.substring(0,historyOrd.indexOf('?')); 
        }
        
        //add time query range if defined
        if (paramString) {
          historyOrd += paramString.replace(/\+/g, "%2B");
        }
        
        if (query !== undefined) {
          query += "|" + historyOrd;
        } else {
          query = historyOrd;
        }
      }
    });
    
    window.location.assign("/ord?" + query + "%7Cview:" + CHART_VIEW_ID);
  }
  
  function sizePageFullscreen() {
    var page = $(PAGE_ID),
        windowHeight = window.innerHeight || $(window).height();
    page.height(windowHeight + 1000);
    $.mobile.silentScroll(0);
    page.height(windowHeight);
    util.mobile.setContentHeight(page);
  }
  

  
////////////////////////////////////////////////////////////////
// History Chart View
////////////////////////////////////////////////////////////////
 
  /**
   * View for our history chart.
   * 
   * @memberOf niagara.mobile.charts.HistoryChartView
   */
  HistoryChartView = View.subclass(function HistoryChartView(chartId) {
    this.$chartId = 'chart-' + chartId;
    this.graph = undefined;
    this.highlightMode = false; //flag for touch pan and scan / disable zoom
  });

  /**
  * In the build HTML method, create an empty target DIV with height and
  * width set. This is the div we will use to set our chart.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  * 
  * @param {jQuery} targetElement Target HTML element that we will append our initizialed DOM to.
  * @param {Object} callbacks an object containing ok/fail callbacks
  */
  HistoryChartView.prototype.doInitializeDOM = function (targetElement, callbacks) {
    var div = $(CHART_DIV_HTML.patternReplace({ id: this.$chartId }));
    targetElement.html(div);
    callbacks.ok();
  };

  /**
  * Forces the history chart to redraw itself.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  */
  HistoryChartView.prototype.redraw = function redraw(force) {
    this.resetLeftMargin();
    if (this.graph && !force) {
      this.graph.replot({ clear: true, resetAxes: true });
    } else {
      this.cleanup();
      this.graph = $.jqplot(this.$chartId, [this.value], this.options);
    }
    this.updateLeftMargin();
  };
  
  HistoryChartView.prototype.resetLeftMargin = function resetLeftMargin() {
    var div = $('#' + this.$chartId);
    div.css('padding-left', 0);
  };
  
  HistoryChartView.prototype.updateLeftMargin = function updateLeftMargin() {
    var div = $('#' + this.$chartId),
        xleft = div.find('.jqplot-xaxis-tick').eq(0).position().left,
        yleft = div.find('.jqplot-yaxis').position().left,
        leftMost = Math.min(xleft, yleft);
    
    div.css('padding-left', -leftMost + 4);
  };

  /**
  * Method to reset our zoom status on the graph.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  */
  HistoryChartView.prototype.resetZoom = function resetZoom() {
    if (this.graph) {
      //seems that if you call resetZoom while zoom = false, subsequent zooms
      //will barf. let's force it to true and then set it back afterwards
      
      var cursor = this.graph.plugins.cursor,
          wasZoom = cursor.zoom;
      cursor.zoom = true;
      this.graph.resetZoom();
      cursor.zoom = wasZoom;
    }
  };

  /**
  * This method toggles the highlighter mode by toggling the zoom functionality.
  * With zoom disabled, the touch events for the highlight feature are 
  * recognized and processed.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  */
  HistoryChartView.prototype.toggleHighlightMode = function toggleHighlightMode() {
    this.highlightMode = !this.highlightMode;
    this.graph.plugins.cursor.zoom = !this.highlightMode;
  };

  /**
  * This method destroys the current graph to release memory from jqPlot.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  */
  HistoryChartView.prototype.cleanup = function cleanup () {
    if (this.graph) {
      this.graph.destroy();
    }
  };
  
  function makeAbsTimeFormat(showSeconds) {
    var TF = baja.TimeFormat,
        obj = {
          show: TF.SHOW_DATE | TF.SHOW_TIME | (showSeconds ? TF.SHOW_SECONDS : 0),
          textPattern: mobileLex.get('history.chart.dateFormat')
        };
    return obj;
  }
  
  /**
   * Create the toString format object for the X axis. If both the start and
   * end dates are the same as the current year, we can leave the year out and
   * save a little space.
   * @memberOf niagara.mobile.charts
   * @private
   */
  function makeXTickFormat() {
    var obj = makeAbsTimeFormat(false),
        currentYear,
        startYear,
        endYear;

    if (currentTimeRange) {
      currentYear = new Date().getFullYear();
      startYear = currentTimeRange.getStartTime().getDate().getYear();
      endYear = currentTimeRange.getEndTime().getDate().getYear();
      if (startYear === currentYear && endYear === currentYear) {
        obj.textPattern = obj.textPattern.replace(/\W*Y+[^\w\s]*/, '');
      }
    }
    
    return obj;
  }
  
  /**
   * @memberOf niagara.mobile.charts
   * @private
   */
  function makeXAxisTickOptions() {
    var fullFormat = makeAbsTimeFormat(true),
        xTickFormat = makeXTickFormat(),
        angle = -35;
    
    return {
      formatter: function (format,val) {
        if (isNaN(val)) {
          return " ";
        }
        var absTime = baja.$(ABS_TIME_TYPE).make({ jsDate: new Date(val) }),
            //this is totally hand-wavey magical - jqplot doesn't seem to bind 
            //the formatter function to a context when drawing a popup, but it 
            //does when drawing the X axis
            showingPopup = (this === undefined);
        
        if (showingPopup) {
          return absTime.toString(fullFormat);
        } else {
          return absTime.toString(xTickFormat);
        }
      },
      labelPosition: 'auto',
      angle: angle
    };
  }

  /**
  * Create a chart using the jqPlot framework.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  * 
  * @param {Array} data
  *        - Array of Objects containing time/value pairs to plot.
  * @param {String} [yAxisLabel]
  *        - Label for the Y axis.       
  * @param {String} [unitSymbol]
  *        - Units label for the Y axis. 
  */
  HistoryChartView.prototype.makeJqPlotGraph = function (data, yAxisLabel, unitSymbol) {
    var options, div, left;

    options = {
       axes: {
         xaxis: {
           tickRenderer: $.jqplot.CanvasAxisTickRenderer,
           autoscale: true,
           tickOptions: makeXAxisTickOptions()
         },
         yaxis: {
           label: yAxisLabel + " " + unitSymbol,
           labelRenderer: $.jqplot.CanvasAxisLabelRenderer,
           tickOptions: { formatter: this.$formatter }
         }
       },
       seriesDefaults: {
         showMarker: true,
         markerOptions: {
           size: 4.5
         }
       },
       cursor: { 
         show: true, 
         zoom: true,  
         showTooltip: false, 
         clickReset: false, 
         dblClickReset: true,
         looseZoom: true
       },
       highlighter: {
         show: true,
         tooltipAxes: 'xy',
         yvalues: 1,
         tooltipLocation: 'n',
         tooltipOffset: 30,
         formatString: this.$highlightFormat
       }
    };
    
    this.options = options;

    //for the very first plot, only plot the first and last data points - just
    //enough to get the chart to lay itself out so we can adjust for the axis
    //labels in updateLeftMargin(). no reason to plot hundreds or thousands of
    //points just to lay the chart out. the full amount of data will be plotted
    //when redraw() is called.
    this.graph = $.jqplot(this.$chartId, [[data[0], data[data.length - 1]]], options);

    $.mobile.hidePageLoadingMsg();
    
    this.updateLeftMargin();
    
    this.redraw(true);
  };
  
  /**
   * Shows a "no data found" error dialog when no data is found for the 
   * given time range.
   * @memberOf niagara.mobile.charts
   * @private
   */
  function showNoDataError() {
    var startTime, endTime, toStringObj, msg,
        TF = baja.TimeFormat;
    
    if (currentTimeRange) {
      startTime = currentTimeRange.getStartTime();
      endTime = currentTimeRange.getEndTime();
      toStringObj = { show: TF.SHOW_DATE | TF.SHOW_TIME };
      msg = mobileLex.get({
        key: 'history.message.noDataFoundInTimeRange',
        args: [ startTime.toString(toStringObj), endTime.toString(toStringObj) ]
      });
    } else {
      msg = mobileLex.get('history.message.noDataFound');
    }
    
    dialogs.error(msg);
  }
  
  
  /**
   * Sets the chart into boolean display mode.
   * @memberOf niagara.mobile.charts.HistoryChartView#
   * @private
   */
  HistoryChartView.prototype.setToBooleanMode = function setToBooleanMode(value) {
    var that = this,
        lastValue,
        bajaLex = baja.lex('baja'),
        trueText = bajaLex.get('true'),
        falseText = bajaLex.get('false');
    
    that.$pushData = function (data, timestamp, value) {
      if (lastValue !== undefined && value !== lastValue) {
        // replicate the last value so we get 90 degree corners.
        // shave off a tenth of a millisecond because older android phones seem
        // to run into floating point problems and start drawing 45 degree lines
        // instead of right angles. the time displayed is only accurate to the
        // minute anyway
        data.push([timestamp, lastValue]);
        data.push([timestamp + 0.1, value]);
      } else {
        data.push([timestamp, value]);
      }
      lastValue = value;
    };
    
    that.$highlightFormat = HIGHLIGHT_FORMAT_STRING.patternReplace({
      dateFormat: '%s',
      valueFormat: '%s'
    });
    
    that.$formatter = function (format, val) {
      val = Math.round(val * 100);
      
      if (val === 100) {
        return trueText;
      } else if (val === 0) {
        return falseText;
      } else {
        return '';
      }
    };
  };
  
  /**
   * Sets the chart into enum display mode.
   * @memberOf niagara.mobile.charts.HistoryChartView#
   * @private
   */
  HistoryChartView.prototype.setToEnumMode = function setToEnumMode(value) {
    var that = this,
        lastValue,
        frozenType = value.getFrozenType(),
        frozenInstance,
        getTag;
    
    that.$pushData = function (data, timestamp, value) {
      var ordinal = value.getOrdinal();
      if (lastValue !== undefined && ordinal !== lastValue) {
        // replicate the last value so we get 90 degree corners.
        // shave off a tenth of a millisecond because older android phones seem
        // to run into floating point problems and start drawing 45 degree lines
        // instead of right angles. the time displayed is only accurate to the
        // minute anyway
        data.push([timestamp, lastValue]);
        data.push([timestamp + 0.1, ordinal]);
      } else {
        data.push([timestamp, ordinal]);
      }
      
      lastValue = ordinal;
    };
    
    that.$highlightFormat = HIGHLIGHT_FORMAT_STRING.patternReplace({
      dateFormat: '%s',
      valueFormat: '%s'
    });
    
    if (frozenType) { //we are frozen and therefore have display tags
      frozenInstance = baja.$(frozenType);
      getTag = function (val) {
        try {
          return frozenInstance.make(val).getDisplayTag();
        } catch (e) {
          return '';
        }
      };
    } else {
      getTag = function (val) {
        return value.getTag(val);
      };
    }
    
    that.$formatter = function (format, val) {
      var tag = getTag(val);
      if (tag === String(val)) {
        //doesn't exist
        return '';
      } else {
        return tag;
      }
    };
  };
  
  /**
   * Sets the chart into numeric display mode.
   * @memberOf niagara.mobile.charts.HistoryChartView#
   * @private
   */
  HistoryChartView.prototype.setToNumericMode = function setToNumericMode(value) {
    var that = this;
    
    that.$pushData = function (data, timestamp, value) {
      data.push([timestamp, value]);
    };
    
    that.$highlightFormat = HIGHLIGHT_FORMAT_STRING.patternReplace({
      dateFormat: '%s',
      valueFormat: '%.{precision}f {unitSymbol}'.patternReplace({
        precision: String(that.precision.valueOf()),
        unitSymbol: that.unitSymbol
      })
    });
  
    that.$formatter = $.jqplot.DefaultTickFormatter;
  };
  
  /**
   * Sets this history chart view into a mode suitable for displaying a
   * boolean, enum, or numeric chart, depending on the type of the
   * input object.
   * 
   * @memberOf niagara.mobile.charts.HistoryChartView#
   * @private
   * @param {Boolean|Number|baja.Enum} value 
   */
  HistoryChartView.prototype.setValueType = function setValueType(value) {
    var that = this,
        type = value.getType();
    
    that.$recordType = type;
    
    if (type.is('baja:Boolean')) {
      that.setToBooleanMode(value);
    } else if (type.is('baja:EnumRange')) { 
      that.setToEnumMode(value);
    } else {
      that.setToNumericMode(value);
    }
  };
  
  /**
   * Sets up the history chart given the value column of its backing collection.
   * The column's Type (and, if present, facets) will be used to set the chart's
   * precision, axis label, units, and mode (Boolean, Numeric, or Enum). This
   * will be called from <code>niagara.mobile.charts.buildArray()</code>.
   * 
   * @memberOf niagara.mobile.charts.HistoryChartView#
   * @private
   * @param {Object} valueCol the value column retrieved from a BoxTable or
   * BoxCollection's <code>getCol</code> method.
   */
  HistoryChartView.prototype.setupFromValueColumn = function (valueCol) {
    // Get data to plot
    var that = this,
        facets = valueCol.getFacets(),
        units = facets && facets.get(UNITS_FACET_NAME);
        
    // declare instance variables
    that.precision = (facets && facets.get(PRECISION_FACET_NAME)) || 0;
    that.axisLabel = valueCol.getDisplayName();
    that.unitSymbol = '';

    //define our unit symbol if present
    if (units) {
      units = units.encodeToString().split(";");
      that.unitSymbol = units[1];
    }

    if (facets && facets.get('range')) {
      that.setValueType(facets.get('range'));
    } else {
      that.setValueType(baja.$(valueCol.getType()));
    }
  };
  
  /**
   * Given a BoxTable/BoxCollection, will iterate over its cursor and assemble
   * the data it contains into a two-dimensional array. Each item in this array 
   * will itself be an array of the form <code>[timestamp, value]</code> where
   * <code>timestamp</code> is a <code>Number</code> (milliseconds) and 
   * <code>value</code> is the value to be displayed (<code>Boolean</code>,
   * <code>Number</code>, or <code>baja.Enum</code>). This array can then be
   * passed into <code>HistoryChartView#loadValue</code>.
   *  
   * @memberOf niagara.mobile.charts
   * @private
   * @param {baja.Component} collection a BoxTable/BoxCollection to be cursored
   * through and assembled into an array
   * @param {Number} offset the current offset (starts at 0 and will be
   * increased as <code>buildArray</code> calls itself recursively)
   * @param {Number} limit the number of records to retrieve from the server
   * each time
   * @param {Object} callbacks an object containing ok/fail callbacks (the 
   * finished/built data array will be passed as an argument to the callbacks'
   * ok handler)
   */
  HistoryChartView.prototype.$buildArray = function(collection, offset, limit, callbacks, 
      /* only given by buildArray itself */ dataArray) {
    callbacks = callbackify(callbacks);
    dataArray = dataArray || [];
    
    var that = this;
    
    function doCursor() {
      var valueCol = collection.getCol(VALUE_COL),
          lastIndex = 0;
      
      //sanity check
      if (!valueCol) {
        return callbacks.fail();
      }
      
      that.setupFromValueColumn(valueCol);
      
      collection.cursor({
        ok: function (cursor) {
          cursor.after(function () {
            if (lastIndex >= limit) {
              //there is more
              that.$buildArray(collection, offset + limit, limit, callbacks, dataArray);
            } else {
              //all done
              callbacks.ok(dataArray);
            }
          });
          
          cursor.each(function (obj, index) {
            lastIndex = index;

            if (index >= limit) {
              return;
            }
            var absTime = this.get(TS_COL),
                ms = absTime.getJsDate().getTime(),
                val = this.get(VALUE_COL).valueOf();
            
            that.$pushData(dataArray, ms, val);
          });
        },
        fail: function (err) {
          callbacks.fail(err);
        },
        offset: offset,
        // limit + 1 - pull down an extra record so if we actually get 
        // limit + 1 records back, then we know there are more records past 
        // the limit and we need to retrieve another batch. if we only get 
        // limit or less, then we're done.
        limit: limit + 1
      });
    }
    
    if (offset >= WARNING_THRESHOLD && !warningDialogConfirmed) {
      dialogs.okCancel({
        content: mobileLex.get({
          key: 'history.message.slowChartWarningDialog',
          args: [ Math.min(dataArray.length, WARNING_THRESHOLD) ]
        }),
        title: mobileLex.get('history.chart.tooManyRecords'),
        ok: function (cb) {
          warningDialogConfirmed = true;
          cb.ok();
          $.mobile.showPageLoadingMsg();
          doCursor();
        },
        cancel: function (cb) {
          if (dataArray.length > WARNING_THRESHOLD) {
            dataArray = dataArray.slice(0, WARNING_THRESHOLD);
          }
          cb.ok();
          callbacks.ok(dataArray);
        }
      });
    } else {
      doCursor();
    }
  };
  
  HistoryChartView.prototype.$collectionToDataArray = function(collection, callbacks) {
    this.$buildArray(collection, 0, RECORD_LIMIT, callbacks);
  };
  
  HistoryChartView.prototype.$ordToDataArray = function(ord, callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    
    baja.Ord.make(ord).get({
      fail: defFail,
      ok: function (value) {
        var type = value.getType();
      
        // If this resolves to a history extension then find its history id and then resolve that as an ORD
        if (type.is("history:HistoryExt")) {
          // Lease the History Config so we can find its history id
          value.getHistoryConfig().lease({
            ok: function () {
              baja.Ord.make("history:" + this.getId()).get({
                ok: function (value) {
                  that.$collectionToDataArray(value, callbacks);
                },
                fail: defFail
              });
            },
            fail: defFail
          });
        }
        else if (type.is("baja:ICollection")) {
          that.$collectionToDataArray(value, callbacks);
        }
      },
      lease: true
    });
  };

  /**
  * this method loads a record collection array into our chart. It is 
  * expected that the record collection will include both a timestamp column
  * and a value column. If these columns are not present, the fail method of
  * the callbacks object is called.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  * 
  * @param {Array} value an array of data to be plotted (the output of
  * <code>niagara.mobile.charts.buildArray</code>)
  * @param {Object} callbacks an object containing ok/fail callbacks
  */
  HistoryChartView.prototype.doLoadValue = function (data, callbacks) {
    // Get data to plot
    var that = this;

    that.$dom.find('div.historyChart').empty();
    that.cleanup();

    
    function plotDataArray(data) {
      if (data && data.length) {
        that.value = data;
        that.makeJqPlotGraph(data, that.axisLabel, that.unitSymbol);
        callbacks.ok(); 
      } else {
        showNoDataError();
        callbacks.fail();
      } 
    }
    
    if ($.isArray(data)) {
      plotDataArray(data);
    } else {
      that.$ordToDataArray(data, plotDataArray);
    }
  };

  /**
  * When view is deactivated, destroy graph.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  */
  HistoryChartView.prototype.deactivated = function() {
    this.cleanup();
  };

  /**
  * If our view is activated, check to see if we already have our data values
  * initialized. If so, we must rebuild our graph as the loadValue function
  * is not called once a page already exists.
  * 
  * @memberOf niagara.mobile.charts.HistoryChartView#
  */
  HistoryChartView.prototype.activated = function() {
    if (this.data) {
      this.makeJqPlotGraph(this.data, this.axisLabel, this.unitSymbol);
    }
  };
  
  /**
   * Arms event handlers to properly redraw the chart when the window is resized,
   * and handle clicks on the reset zoom / highlight buttons.
   * 
   * @memberOf niagara.mobile.charts.HistoryChartView#
   */
  HistoryChartView.prototype.armHandlers = function () {
    var that = this,
        changeOrientation = util.debounce(function () {
          sizePageFullscreen();
          that.redraw();
        }, 500),
        supportsOrientation = 'onorientationchange' in window;
    
    //perform setup for our chart div
    //force redraw on page resize
    $(window).bind(supportsOrientation ? 'orientationchange' : 'resize', 
        changeOrientation);
    
    //set click function for our reset button
    $(RESET_BTN_ID).click(function () {
      that.resetZoom();
    });

    $(HIGHLIGHT_BTN_ID).click(function () {
      that.toggleHighlightMode();
      $(this).toggleClass(CHECKBOX_TOGGLE_CLASSES);
    });
      
    $(HIGHLIGHT_BTN_ID).addClass(CHECKBOX_OFF_CLASS);
  };

////////////////////////////////////////////////////////////////
// Main
////////////////////////////////////////////////////////////////
  
  /**
   * This method initializes our history chart by setting up the chart
   * time query editor and initializing the DOM.
   * 
   * @memberOf niagara.mobile.charts
   */
  function initChart(ord) {
    var page = $(PAGE_ID),
        contentDiv = $(CONTENT_DIV_ID);
    
    sizePageFullscreen();
      
    // create our view chartView
    chartView = new HistoryChartView(page.attr('id'));
    chartView.initializeDOM(contentDiv, function () {
      chartView.loadValue(ord);
    });
  }
 
  /**
   * This method sets the title for our page by parsing out the history
   * ID from our ORD.
   * 
   * @memberOf niagara.mobile.charts
   */
  function setPageTitle() {
    var pageTitle = niagara.view.history.displayName;
    document.title = pageTitle;
    $(PAGE_TITLE_ID).text(pageTitle);
  }

  /**
   * This method initializes and displays a time query editor that allows the
   * end user to set a time query for the current history displayed in the
   * history chart.
   * 
   * @memberOf niagara.mobile.charts
   */
  function buildTimeQueryEditor(callbacks) {

    //open a dialog with our editor
    dialogs.fieldEditor({
      title: mobileLex.get(TIME_RANGE_BTN_DISPLAY_KEY),
      value: currentParamString,
      showDelta: chartView && (chartView.$recordType.is('baja:Number')),
      key: 'historyQueryParams',
      ok: function (paramString, cb) {
        currentParamString = paramString;
        currentTimeRange = historyUtil.toTimeRange(paramString);
        redirect(paramString);
        cb.ok();
      },
      callbacks: callbacks
    });
  }
   
  /**
   * Initialize our view
   */
  function initAppView() {
    $.mobile.showPageLoadingMsg();
    
    var ord = niagara.view.history.query,
        page = $(PAGE_ID),
        pageHeader = page.children(DATA_ROLE_HEADER_ID),
        queryCmd = new AsyncCommand(mobileLex.get(TIME_RANGE_BTN_DISPLAY_KEY), buildTimeQueryEditor),
        cmds = [];
    
    if (niagara.view.profile.showLogoutCmd) {
      cmds.push(commands.getLogoutCmd());
    }
    
    if (niagara.view.profile.showHome) {
      cmds.splice(0, 0, commands.getHomeCmd());
    }
    commands.setDefaultCommands(cmds);
    commands.prependDefaultCommand(queryCmd);
  
    $("#commandsButton").click(commands.showCommandsHandler);
  
    setPageTitle();
  
    //setup our page header
    pageHeader.css({
      '-webkit-transform': 'none',
      'overflow': 'visible'
    });

    if (chartView === undefined) { 
      initChart(ord); 
    } else {
      chartView.loadValue(ord);
    }
  }
 
  /**
   * Declare our namespace in the views space.
   */
  util.api('niagara.mobile.charts', {
    HistoryChartView: HistoryChartView,
    initAppView: initAppView
  });
  
}());