/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview The built-in field editors available to all Niagara Mobile
 * apps. These field editors make use of jQuery Mobile with the Datebox plugin.
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, sloppy: true, browser: true */
/*global $, baja, BaseBajaObj, niagara */

(function () {
  
  niagara.util.require(
    'niagara.fieldEditors',
    'niagara.util.mobile.dialogs',
    'niagara.util.flow',
    'jQuery.mobile',
    'jQuery.mobile.datebox'
  );
      //imports
  var util = niagara.util,
      callbackify = util.callbackify,
      dialogs = util.mobile.dialogs,
      fe = niagara.fieldEditors,
      composite = fe.composite,
      defineEditor = fe.defineEditor,
      register = fe.register,
      makeFor = fe.makeFor,
      toSaveDataComponent = fe.toSaveDataComponent,
      BaseFieldEditor = niagara.fieldEditors.BaseFieldEditor,

      //local vars
      mobileLex = null,
      dimensionRegex = /\(\S*\)/,
      quantityDb,

      //exports
      simple = {},
      special = {},
      status = {},
      ord = {},
      facets = {},
      units = {},
      MobileFieldEditor;
  

  /**
   * Retrieves the unit database from the station using 
   * <code>ComponentServerSideHandler#getUnits</code>. Stored in a private
   * var so will be retrieved at most once per page load.
   * 
   * @memberOf niagara.fieldEditors.mobile
   * @private
   */
  function getQuantityDb(callbacks) {
    callbacks = callbackify(callbacks);
    
    if (quantityDb) {
      return callbacks.ok(quantityDb);
    }
    
    baja.Ord.make(niagara.view.ord).get(function (comp) {
      comp.serverSideCall({
        typeSpec: 'mobile:ComponentServerSideHandler',
        methodName: 'getUnits',
        ok: function (units) {
          quantityDb = JSON.parse(units);
          
          //give each unit a reference back to its parent quantity
          baja.iterate(quantityDb, function (quantity) {
            var units = quantity.u;
            baja.iterate(units, function (unit) {
              unit.q = quantity;
            });
          });
          
          //return an array of units given an (unescaped) quantity name
          quantityDb.getUnits = function (quantityName) {
            var q = this[util.indexOf(this, quantityName, function (q, name) {
              return q.q === name;
            })];
            return q && q.u;
          };
          
          //change mathematical symbols like 'squared' and 'cubed' suffixes to
          //their string-encoded-unit equivalents. quantity names should be
          //escaped when 1) setting a value on an html element, or 2) comparing
          //a string-encoded Unit.
          quantityDb.escapeQuantityName = function (quantityName) {
            return quantityName
              .replace(/\u00b7/g, ')(')
              .replace(/\u00b2/g, '2')
              .replace(/\u00b3/g, '3');
          };
          
          //given a unit name (like 'milliampere') and unit dimension (like
          //'(A)' find the actual unit reference from the db, so that we may
          //get its information like whether it is prefixed, offset, scale,
          //etc.
          quantityDb.getUnitFromNameAndDimension = function (unitSymbol, unitDimension) {
            return baja.iterate(this, function (quantity) {
              var q = quantityDb.escapeQuantityName(quantity.q),
                  dimensionMatch = q.match(dimensionRegex),
                  dimension = dimensionMatch && dimensionMatch[0];
              
              //we found a potential quantity match, since its dimension matches
              //the unit's dimension. but there can be multiple quantities with
              //the same dimension, so we have to search this quantity to see
              //if it contains the actual unit.
              if (dimension === unitDimension) {
                return baja.iterate(quantity.u, function (unit) {
                  if (unit.n === unitSymbol) {
                    return unit;
                  }
                });
              }
            });
          };
          
          callbacks.ok(quantityDb);
        },
        fail: callbacks.fail
      });
    });
  }
  
  function getMobileLex() {
    if (!mobileLex) {
      mobileLex = baja.lex('mobile');
    }
    return mobileLex;
  }
  
  /**
   * This is the code that the datebox plugin typically runs on pagecreate -
   * since we load pages dynamically we have to instantiate the datebox
   * ourselves. Almost a straight copy-paste from jquery.mobile.datebox.js
   * 
   * @private
   * @memberOf niagara.fieldEditors.mobile
   * 
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  function handleDateBox(callbacks) {
    var div = this.$dom;
    
    $(":jqmData(role='datebox')", div).each(function () {
      var rep = $("<div/>").html($(this).clone()).html()
        .replace(/\s+type=["']date['"]?/, " type=\"text\" ");
      $(this).replaceWith(rep);
    });
    div.find('input[type="text"]').textinput();
    $(":jqmData(role='datebox')", div).datebox();

    callbacks.ok();
  }

  
  
  /**
   * A field editor subclass that makes use of JQM widgets for Niagara Mobile
   * apps.
   * 
   * @class
   * @extends niagara.fieldEditors.BaseFieldEditor
   */
  MobileFieldEditor = BaseFieldEditor.subclass(function MobileFieldEditor(value, container, slot, params) {
    BaseFieldEditor.call(this, value, container, slot, params);
  });
  
  /**
   * After setting the field editor value, this method will be called to refresh
   * any widgets to reflect the updated value. On a mobile field editor, 
   * <code>refresh</code> methods will be called on any JQM widgets container
   * in the field editor's DOM.
   * 
   * @name niagara.fieldEditors.MobileFieldEditor#refreshWidgets
   * @function
   */
  MobileFieldEditor.prototype.refreshWidgets = function refreshWidgets() {
    var dom = this.$dom,
        select = dom.find('select'),
        input = dom.find('input'),
        switches = select.filter('.ui-slider-switch'),
        dropdowns = select.filter('.ui-select select'),
        checkboxes = input.filter('.ui-checkbox input'),
        dateboxes = input.filter('.ui-input-datebox input');
        
    if (switches.length) {
      switches.slider('refresh');
    }
    if (dropdowns.length) {
      dropdowns.selectmenu('refresh');
    }
    if(checkboxes.length) {
      checkboxes.checkboxradio('refresh');
    }
    if (dateboxes.length) {
      dateboxes.datebox('refresh');
    }
    
    dom.removeClass('ui-br');
  };
  
  /**
   * Sets a mobile field editor's readonly status by calling 
   * <code>disable</code> or <code>enable</code> methods on any editable JQM
   * widgets contained in the field editor's DOM.
   * 
   * @name niagara.fieldEditors.MobileFieldEditor#setReadonly
   * @function
   * @param {Boolean} readonly
   */
  MobileFieldEditor.prototype.setReadonly = function setReadonly(readonly) {
    this.$readonly = readonly;
    
    var select = this.$dom.find('select'),
        input = this.$dom.find('input'),
        textarea = this.$dom.find('textarea'),
        action = readonly ? 'disable' : 'enable';
    
    select.filter(':jqmData(role=slider)').slider(action);
    select.filter(':jqmData(role!=slider)').selectmenu(action);
    input.filter('[type="checkbox"]').checkboxradio(action);
    input.filter(':jqmData(role="datebox")').datebox(action);
    input.filter('[type="text"]').textinput(action);
    textarea.textinput(action);
  };
  
  /**
   * Initializes the HTML created in <code>makeFor</code>, triggering the
   * creation of any JQM widgets, setting the value, and arming change handlers.
   * 
   * @name niagara.fieldEditors.MobileFieldEditor#postInitializeDOM
   * @function
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  MobileFieldEditor.prototype.postInitializeDOM = function postInitializeDOM(callbacks) {
    if (this.isReadonly()) {
      this.$dom.find('input,textarea,select').attr('disabled', 'disabled');
    }
    this.$dom.trigger('create');

    callbacks.ok();
  };
  


  

  

  
  ////////////////////////////////////////////////////////////////
  // Simple field editors for primitive Baja objects
  ////////////////////////////////////////////////////////////////
  
  (function simpleFunctions() {
    var absTimeLoadValueWorkflow;
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function getFacets(editor) {
      var facets = editor.facets,
          container = editor.container;
      
      if (facets && facets !== baja.Facets.NULL) {
        return facets;
      } else if (container) {
        return util.slot.getFacets(container, editor.slot);
      } else {
        return baja.Facets.DEFAULT;
      }
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function makeTimeInput(slotName) {
      var timeInput = $('<input type="date" data-role="datebox" name="' + slotName + '_time"/>'),
          lex = getMobileLex();
      
      timeInput.attr('data-options', JSON.stringify({
        mode: 'timebox',
        useDialogForceFalse: true,
        setTimeButtonLabel: lex.get({
          key: 'propsheet.datebox.setTimeButtonLabel',
          def: 'Set Time'
        })
      }));
      
      return timeInput;
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function getDateboxDefaultOptions() {
      var lex = getMobileLex();
      
      return {
        daysOfWeek: lex.get({
          key: 'propsheet.datebox.daysOfWeek',
          def: 'Sunday Monday Tuesday Wednesday Thursday Friday Saturday'
        }).split(' '),
        daysOfWeekShort: lex.get({
          key: 'propsheet.datebox.daysOfWeekShort',
          def: 'Su Mo Tu We Th Fr Sa'
        }).split(' '),
        monthsOfYear: lex.get({
          key: 'propsheet.datebox.monthsOfYear',
          def: 'January February March April May June July August September October November December'
        }).split(' '),
        monthsOfYearShort: lex.get({
          key: 'propsheet.datebox.monthsOfYearShort',
          def: 'Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec'
        }).split(' '),
        setDateButtonLabel: lex.get({
          key: 'propsheet.datebox.setDateButtonLabel',
          def: 'Set Date'
        })
      };
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function makeDateInput(slotName) {
      var dateInput = $('<input type="date" data-role="datebox" name="' + slotName + '_date" />'),
          options = getDateboxDefaultOptions();
      
      options.mode = 'calbox';
      options.useDialogForceFalse = true;
      options.calHighToday = false;
      
      dateInput.attr('data-options', JSON.stringify(options));
      
      return dateInput;
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function getBajaDateFromDatebox(editor) {
      var dateInput = editor.$dom.find('input[name$=date]'),
          date = dateInput.jqmData('datebox').theDate;
      return baja.Date.make({
        year: date.getFullYear(),
        month: date.getMonth(),
        day: date.getDate()
      });
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function getBajaTimeFromDatebox(editor, accuracy) {
      var timeInput = editor.$dom.find('input[name$=time]'),
          theDate = timeInput.jqmData('datebox').theDate,
          ms = 0,
          index = util.indexOf(['hour', 'minute', 'second', 'millisecond'], accuracy || 'millisecond');
      
      if (index < 0) {
        throw new Error("invalid accuracy argument " + accuracy);
      }
      if (index >= 0) {
        ms += theDate.getHours() * util.time.MILLIS_IN_HOUR;
      }
      if (index >= 1) {
        ms += theDate.getMinutes() * util.time.MILLIS_IN_MINUTE;
      }
      if (index >= 2) {
        ms += theDate.getSeconds() * util.time.MILLIS_IN_SECOND; 
      }
      if (index >= 3) {
        ms += theDate.getMilliseconds();
      }
      
      return baja.Time.make(ms);
    }
    
    /**
     * Keep the actual date object up to date when typing manually. Otherwise
     * if you bind 'editorchange' to this editor, getSaveData() will return
     * out of date information.
     * 
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function dateboxArmHandlers() {
      var timeInput = this.$dom.find('input:jqmData(role=datebox)');
      timeInput.bind('keyup', function () {
        var $this = $(this),
            val = $this.val();
        $this.trigger('datebox', { method: 'set', value: val });
      });
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @private
     */
    function minMaxCheck(min, max, value, callbacks) {
      var err;
      
      if ((min !== undefined && min !== null && value < min) ||
          (max !== undefined && max !== null && value > max)) {
        
        err = "Cannot save property \"" + this.slot + "\". ";
        if (value < min) {
          err += value + " < " + min;
        } else {
          err += value + " > " + max;
        }
        err += " [" + min + " - " + max + "]";
        callbacks.fail(err);
      } else {
        callbacks.ok();
      }
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function stringValidate(value, callbacks) {
      var facets = getFacets(this),
        min, max;
      
      if (facets) {
        min = facets.get("min");
        max = facets.get("max");
        minMaxCheck.call(this, min, max, value.length, callbacks);
      } else {
        callbacks.ok();
      }
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function stringLoadValue(value, callbacks) {
      
      if (value === null || value === undefined) {
        return callbacks.ok();
      }
      
      var stringValue;
      
      if (typeof value.encodeToString === 'function') {
        stringValue = value.encodeToString();
      } else {
        stringValue = JSON.stringify(baja.bson.encodeValue(value));
//        stringValue = getMobileLex().get({
//          key: 'fieldeditors.notRegistered',
//          args: String(value.getType())
//        });
        this.setReadonly(true);
      }
      
      this.$dom.find('input,textarea')
        .val(stringValue);
      
      callbacks.ok();
    }

    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function stringHtmlBuilder(targetElement, callbacks) {
      var input,
          facets = getFacets(this),
          fieldWidth,
          cols;

      if (facets && facets.get("multiLine")) {
        fieldWidth = facets.get("fieldWidth");
        input = $('<textarea cols="40" rows="10"></textarea>');
        if (fieldWidth) {
          input.attr('cols', fieldWidth);
        }
        
      } else {
        input = $('<input type="text"></input>');  
      }

      input
        .attr('name', this.name)
        .textinput();
      
      input.appendTo(targetElement);

      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function stringGetSaveData(obj) {
      obj.ok(this.$dom.find('input,textarea').val());
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function numericHtmlBuilder(targetElement, callbacks) {
      if (this.value === null || this.value === undefined) {
        this.value = '0';
      }
      
      var wrapper = $('<div class="numericWrapper ui-input-text ui-body-c ui-corner-all ui-shadow-inset"/>'),
          label = $('<label class="unitsDisplay"/>').appendTo(wrapper),
          span = $('<span class="inputWrapper"/>').appendTo(wrapper),
          input = $('<input type="text" data-theme="c"/>')
            .attr('name', this.name)
            .appendTo(span);
      
      wrapper.appendTo(targetElement);
      
      callbacks.ok();
    }
    

    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function numericValidate(value, callbacks) {
      var facets = getFacets(this),
        min, max;
      
      if (facets) {
        min = facets.get("min");
        max = facets.get("max");
        minMaxCheck.call(this, min, max, value, callbacks);
      } else {
        callbacks.ok();
      }
    }
    
    function isNumber(num) {
      if (num === undefined || num === null) {
        return false;
      }
      return typeof num === 'number' || num.getType().is('baja:Number');
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function numericLoadValue(value, callbacks) {
      var that = this,
          dom = that.$dom,
          number = value.valueOf(),
          facets = getFacets(that),
          precision = facets && facets.get('precision'),
          units = facets && facets.get('units'),
          unitSplit, 
          unitSymbol,
          unitDisplay, 
          label = dom.find('.unitsDisplay'),
          input = dom.find('input');
      
      function doStringLoadValue() {
        stringLoadValue.call(that, number, callbacks);
      }
      
      if (isNumber(number) && isNumber(precision)) {
        number = number.toFixed(Math.min(precision, 20));
      }
      
      if (units) {
        unitSplit = String(units).split(';');
        unitSymbol = unitSplit[1];


        //if the unit is prefixed, we want to move the symbol over to the left
        getQuantityDb(function (db) {
          var unitName = unitSplit[0],
              unitDimension = unitSplit[2],
              unit = db.getUnitFromNameAndDimension(unitName, unitDimension),
              paddingLeft, 
              paddingRight;
          if (unit) {
            unitSymbol = unit.s;
          }

          if (unit && unit.p) { //is prefixed
            paddingRight = label.css('padding-right');
            paddingLeft = label.css('padding-left');
            
            //move the label over to the left, and flip the padding settings
            label.css({
              'float': 'left',
              'padding-left': paddingRight,
              'padding-right': paddingLeft
            });
            
            input.css('text-align', 'left');
          }

          label.text(unitSymbol);
          label.show();
          doStringLoadValue();
        });
      } else {
        //no units, so nix the units label altogether
        label.hide();
        doStringLoadValue();
      }
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function numericGetSaveData(obj) {
      var oldType = this.value.getType();
      
      stringGetSaveData.call(this, {
        ok: function (val) {
          var number = Number(val);

          if (oldType.is('baja:Integer')) {
            switch (val.toLowerCase()) {
            case "min":
              return obj.ok(baja.Integer.MIN_VALUE);
            case "max": 
              return obj.ok(baja.Integer.MAX_VALUE);
            default:
              if (isNaN(number)) {
                return obj.fail(val + " is not a valid Integer value");
              }
              return obj.ok(baja.Integer.make(number));
            }
          } else if (oldType.is('baja:Long')) {
            switch (val.toLowerCase()) {
            case "min":
              return obj.ok(baja.Long.MIN_VALUE);
            case "max": 
              return obj.ok(baja.Long.MAX_VALUE);
            default:
              if (isNaN(number)) {
                return obj.fail(val + " is not a valid Long value");
              }
              return obj.ok(baja.Long.make(number));
            }
          } else if (oldType.is('baja:Float')) {
            switch (val.toLowerCase()) {
            case "nan": 
              return obj.ok(baja.Float.NAN);
            case "-inf":
              return obj.ok(baja.Float.NEGATIVE_INFINITY);
            case "+inf": 
              return obj.ok(baja.Float.POSITIVE_INFINITY);
            default:
              if (isNaN(number)) {
                return obj.fail(val + " is not a valid Float value");
              }
              return obj.ok(baja.Float.make(number));
            }
          } else if (oldType.is('baja:Double')) {
            switch (val.toLowerCase()) {
            case "nan": 
              return obj.ok(baja.Double["NaN"]);
            case "-inf":
              return obj.ok(baja.Double.NEGATIVE_INFINITY);
            case "+inf": 
              return obj.ok(baja.Double.POSITIVE_INFINITY);
            default:
              if (isNaN(number)) {
                return obj.fail(val + " is not a valid Double value");
              }
              return obj.ok(baja.Double.make(number));
            }
          } else {
            if (isNaN(number)) {
              return obj.fail(val + " is not a valid numeric value");
            } else {
              return obj.ok(number);
            }
          }
        },
        fail: obj.fail,
        batch: obj.batch
      });
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function booleanLoadValue(value, callbacks) {
      
      if (value === undefined || value === null) {
        return callbacks.ok();
      }

      this.$dom
        .find('select')
        .val(value ? 'true' : 'false');
      
      callbacks.ok();
    }
    
    function getTrueFalseText(editor) {
      var facets = editor.facets,
          result,
          bajaLex,
          trueText,
          falseText;
      if (!(facets && facets.get('trueText') && facets.get('falseText'))) {
        facets = getFacets(editor);
      }
      
      if (facets) {
        trueText = facets.get('trueText');
        falseText = facets.get('falseText');
      }
      
      if (!trueText || !falseText) {
        bajaLex = baja.lex('baja');
        trueText = bajaLex.get('true');
        falseText = bajaLex.get('false');
      }
      
      return { 
        trueText: baja.Format.format(trueText), 
        falseText: baja.Format.format(falseText) 
      };
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function booleanHtmlBuilder(targetElement, callbacks) {
      var name = this.name,
          select = $('<select data-role="slider" />')
            .attr('name', name),
//            .attr('id', name),
          text = getTrueFalseText(this),
          optionTrue = $('<option/>').val('true').text(text.trueText),
          optionFalse = $('<option/>').val('false').text(text.falseText);
      
      select
        .append(optionFalse)
        .append(optionTrue)
        .appendTo(targetElement);

      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function booleanCheckboxHtmlBuilder(targetElement, callbacks) {
      var l = this.label,
          escaped = util.mobile.encodePageId(l),
          fieldset = $('<fieldset data-role="controlgroup"/>'),
          label = $('<label/>')
            .attr('for', escaped)
            .text(l),
          checkbox = $('<input type="checkbox" data-theme="b"/>')
            .attr('name', escaped)
            .attr('id', escaped);
      
      fieldset
        .append(checkbox)
        .append(label);
      targetElement
        .append(fieldset);
      
      
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function booleanCheckboxLoadValue(value, callbacks) {
      this.$dom.find('input')
        .attr('checked', value)
        .checkboxradio('refresh');
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function booleanCheckboxGetSaveData(callbacks) {
      callbacks.ok(this.$dom.find('input').is(':checked'));
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function booleanGetSaveData(obj) {
      var select = this.$dom.find('select');
      obj.ok(select.val() === 'true');
    }
    
    function getEnumRangeDisplay(ordinal, range) {
      var lexicon = range.getOptions().get('lexicon'),
          got = range.get(ordinal),
          tag = got.getTag();
      if (lexicon) {
        return baja.lex(lexicon).get({
          key: tag,
          def: tag
        });
      } else {
        return baja.SlotPath.unescape(got.toString());
      }
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     * @param {Number} ordinal
     * @param {baja.EnumRange} range
     * @returns {jQuery}
     */
    function makeEnumOption(ordinal, range) {

      var display = getEnumRangeDisplay(ordinal, range),
          option = $('<option/>')
                    .val(ordinal)
                    .text(baja.SlotPath.unescape(display));
      
      return option;
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function enumLoadValue(value, callbacks) {
      var input = this.$dom.find('select').empty(),
          range = this.facets.get('range') || value.getRange(), 
          ordinals = range.getOrdinals();
      
      baja.iterate(ordinals, function (ordinal) {
        input.append(makeEnumOption(ordinal, range));
      });
      
      input.val(String(value.getOrdinal()));
      
      this.range = range;
      
      callbacks.ok();
    }

    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function enumHtmlBuilder(targetElement, callbacks) {
      var input = $('<select data-theme="a" />')
        .attr('name', this.name);
//        .attr('id', this.name);
      
      $('<option/>')
        .attr('value', 'n/a')
        .val(getMobileLex().get('loading'))
        .appendTo(input);
      
      input.appendTo(targetElement);
      
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function dynamicEnumGetSaveData(obj) {
      var select = this.$dom.find('select'),
          range = this.range; //set in loadValue
      
      obj.ok(baja.DynamicEnum.make({
        ordinal: Number(select.val()),
        range: range
      }));
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function frozenEnumGetSaveData(obj) {
      var select = this.$dom.find('select'),
          frozenEnum = this.value;

      obj.ok(frozenEnum.get(Number(select.val())));
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function absTimeHtmlBuilder(targetElement, callbacks) {
      var name = this.name;

      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function absTimeGetSaveData(callbacks) {
      var that = this;
      that.$dateEditor.getSaveData(function (date) {
        that.$timeEditor.getSaveData(function (time) {
          callbacks.ok(baja.AbsTime.make({
            date: date,
            time: time,
            offset: that.$offset
          }));
        });
      });
    }
    
    // cx properties: editor, absTime
    absTimeLoadValueWorkflow = util.flow.sequential(
      function absTimeLoadValueWorkflow__step1__makeDateEditor(cx) {
        cx.editor.$dom.empty();
        cx.editor.$offset = cx.absTime.$offset || 0;
        
        fe.makeFor({
          value: cx.absTime.getDate(),
          parent: cx.editor,
          readonly: cx.editor.isReadonly()
        }, this);
      },
      function absTimeLoadValueWorkflow__step2__loadDateEditor(cx, dateEditor) {
        cx.editor.$dateEditor = dateEditor;
        dateEditor.buildAndLoad(cx.editor.$dom, this);
      },
      function absTimeLoadValueWorkflow__step3__makeTimeEditor(cx) {
        fe.makeFor({
          value: cx.absTime.getTime(),
          parent: cx.editor,
          readonly: cx.editor.isReadonly()
        }, this);
      },
      function absTimeLoadValueWorkflow__step4__loadTimeEditor(cx, timeEditor) {
        cx.editor.$timeEditor = timeEditor;
        timeEditor.buildAndLoad(cx.editor.$dom, this);
      },
      function absTimeLoadValueWorkflow__step5__armEditorChangeListeners(cx) {
        function setEditorModified() {
          cx.editor.setModified(true);
          return false;
        }
        //upon a change to the date or time sub-editor, set the
        //parent AbsTime editor to modified and let it bubble up its own
        //editorchange event - otherwise any editorchange listeners on
        //the AbsTime editor will receive the event from the wrong editor
        cx.editor.$dateEditor.$dom.bind('editorchange', setEditorModified);
        cx.editor.$timeEditor.$dom.bind('editorchange', setEditorModified);
        this.ok();
      }
    );
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function absTimeLoadValue(value, callbacks) {
      absTimeLoadValueWorkflow.invoke({
        editor: this,
        absTime: value
      }, callbacks);
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function absTimeSetReadonly(readonly) {
      var that = this;
      setTimeout(function () {
        if (that.$dateEditor) {
          that.$dateEditor.setReadonly(readonly);
        }
        if (that.$timeEditor) {
          that.$timeEditor.setReadonly(readonly);
        }
      }, 0);
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function dateHtmlBuilder(targetElement, callbacks) {
      targetElement.append(makeDateInput(this.name));
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function dateGetSaveData(callbacks) {
      callbacks.ok(getBajaDateFromDatebox(this));
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function dateLoadValue(value, callbacks) {
      var dateInput = this.$dom.find('input:jqmData(role=datebox)');
          
      dateInput.trigger('datebox', {
        method: 'set',
        value: value.toString({ textPattern: 'YYYY-MM-DD' })
      });
      
      callbacks.ok();
    }

    function getRelTimeMinMax(editor) {
      var facets = getFacets(editor),
        max = facets.get('max'),
        min = facets.get('min'),
        maxMillis = max && max.getType().is('baja:RelTime') && max.getMillis(),
        minMillis = min && min.getType().is('baja:RelTime') && min.getMillis(),
        str;

      if (minMillis > maxMillis) {
        minMillis = null;
        maxMillis = null;
      }

      if (maxMillis !== null || minMillis !== null) {
        str = '[' +
          (minMillis === baja.Long.MIN_VALUE ? '-inf' : baja.RelTime.make(minMillis).encodeToString() + 'ms') +
          ' - ' +
          (maxMillis === baja.Long.MAX_VALUE ? '+inf' : baja.RelTime.make(maxMillis).encodeToString() + 'ms') +
          ']';
      }

      return {
        min: minMillis,
        max: maxMillis,
        str: str,
        normalize: function (millis) {
          if (maxMillis !== null && millis > maxMillis) {
            return maxMillis;
          } else if (minMillis !== null && millis < minMillis) {
            return minMillis;
          }
          return millis;
        }
      };
    }

    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function relTimeLoadValue(value, callbacks) {
      if (!value) {
        return callbacks.ok();
      }
      
      var that = this,
          dom = that.$dom,
          input = dom.find('input:jqmData(role="datebox")'),
          d = value.getDaysPart(),
          h = value.getHoursPart(),
          m = value.getMinutesPart(),
          s = value.getSecondsPart(),
          newValue = "",
          minMax = getRelTimeMinMax(that),
          rangeDiv = that.rangeDiv;


      newValue += d + " Day" + (d > 1 ? 's' : '') + ', ';
      newValue += util.time.toTimeString(h, m, s);

      input.trigger('datebox', {
        method: 'set',
        value: newValue
      });

      if (rangeDiv) {
        rangeDiv.text(minMax.str);
      }

      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function relTimeHtmlBuilder(targetElement, callbacks) {
      var that = this,
          name = that.name,
          params = that.params,
          input = $('<input type="date" data-role="datebox" />')
            .attr('name', name),
          lex = getMobileLex(),
          options,
          minMax = getRelTimeMinMax(that),
          rangeDiv = that.rangeDiv = $('<div class="relTimeRange"></div>');
      
      options = {
        mode: 'durationbox',
        useDialogForceFalse: true,
        durationLabel: lex.get({
          key: 'propsheet.datebox.durationLabel',
          def: 'Days Hours Minutes Seconds'
        }).split(' '),
        setDurationButtonLabel: lex.get({
          key: 'propsheet.datebox.setDurationButtonLabel',
          def: 'Set Duration'
        }),
        durationFormat: params.durationFormat,
        durationOrder: params.durationOrder
      };
      
      
      //TODO: document use of 
      input.attr('data-options', JSON.stringify(options));
      
      input.appendTo(targetElement);

      targetElement.delegate('input:jqmData(role="datebox")', 'datebox', function (e, passed) {
        var $this = $(this);

        if (passed.method === 'open') {
          var dateboxControls = $('.ui-datebox-container .ui-datebox-controls');
          if (!dateboxControls.prev().hasClass('relTimeRange')) {
            rangeDiv.insertBefore(dateboxControls);
          }
        } else if (passed.method === 'close') {
          rangeDiv.remove();
        } else if (passed.method === 'refresh') {
          // refresh is triggered on initial load and on every spinner click.
          var db = $this.jqmData('datebox'),
              oldTheDate = db.theDate,
              newVal = oldTheDate - db.initDate,
              format,
              mod = newVal % 1000,
              seconds = newVal - mod + (mod ? 1000 : 0),
              norm;

          // normalize to be within valid min and max range
          norm = minMax.normalize(seconds);

          if (norm !== seconds) {
            // we've spun to an invalid duration, so reset the inner state to
            // a duration that is valid and trigger a refresh.
            // revisit if datebox api changes...
            db.theDate = new Date(db.initDate.getTime() + norm);

            $this.trigger('datebox', {
              method: 'dorefresh'
            });
          }
        }
      });

      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function relTimeGetSaveData(obj) {
      var input = this.$dom.find('input:jqmData(role="datebox")'),
          initDate = input.jqmData('datebox').initDate,
          theDate = input.jqmData('datebox').theDate,
          ms = theDate.getTime() - initDate.getTime();
      ms = Math.ceil(ms / 1000) * 1000;
      obj.ok(baja.RelTime.make(ms));
    }

    function relTimeValidate(value, callbacks) {
      var minMax = getRelTimeMinMax(this);

      minMaxCheck.call(this, minMax.min, minMax.max, value.getMillis(), callbacks);
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function timeLoadValue(value, callbacks) {
      if (value === null || value === undefined) {
        return callbacks.ok();
      }
      
      this.$dom.find('input:jqmData(role="datebox")')
        .trigger('datebox', {
          method: 'set',
          value: value.toString({textPattern: 'HH:mm'})
        });
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function timeHtmlBuilder(targetElement, callbacks) {
      targetElement.append(makeTimeInput(this.name));

      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function timeGetSaveData(obj) {
      obj.ok(getBajaTimeFromDatebox(this, 'minute'));
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function nonEditableHtmlBuilder(targetElement, callbacks) {
      var span = $('<span class="nonEditable" />'),
          msg;
      
      msg = getMobileLex().get({
        key: 'propsheet.message.nonEditable',
        def: 'Not Editable In Mobile Property Sheet'
      });
      
      span.text(this.value.getType() + " " + msg);
      span.appendTo(targetElement);
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function nonEditableLoadValue(value, callbacks) {
      callbacks.ok();
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.simple
     */
    function nonEditableGetSaveData(obj) {
      obj.ok(this.value);
    }
    
    
    /**
     * A namespace containing functions for editing BSimples. Note that functions
     * contained within this namespace will not work properly when called on
     * their own - they should be used to define a new field editor class using
     * <code>niagara.fieldEditors.defineEditor</code>.
     * @namespace
     * @name niagara.fieldEditors.mobile.simple
     */
    util.api(simple, {
      'public': {
        absTimeHtmlBuilder: absTimeHtmlBuilder,
        absTimeLoadValue: absTimeLoadValue,
        absTimeGetSaveData: absTimeGetSaveData,
        absTimeSetReadonly: absTimeSetReadonly,
        booleanHtmlBuilder: booleanHtmlBuilder,
        booleanLoadValue: booleanLoadValue,
        booleanGetSaveData: booleanGetSaveData,
        booleanCheckboxHtmlBuilder: booleanCheckboxHtmlBuilder,
        booleanCheckboxLoadValue: booleanCheckboxLoadValue,
        booleanCheckboxGetSaveData: booleanCheckboxGetSaveData,
        dateboxArmHandlers: dateboxArmHandlers,
        dateGetSaveData: dateGetSaveData,
        dateHtmlBuilder: dateHtmlBuilder,
        dateLoadValue: dateLoadValue,
        dynamicEnumGetSaveData: dynamicEnumGetSaveData,
        enumHtmlBuilder: enumHtmlBuilder,
        enumLoadValue: enumLoadValue,
        frozenEnumGetSaveData: frozenEnumGetSaveData,
        getDateboxDefaultOptions: getDateboxDefaultOptions,
        getEnumRangeDisplay: getEnumRangeDisplay,
        makeEnumOption: makeEnumOption,
        nonEditableHtmlBuilder: nonEditableHtmlBuilder,
        nonEditableLoadValue: nonEditableLoadValue,
        nonEditableGetSaveData: nonEditableGetSaveData,
        numericHtmlBuilder: numericHtmlBuilder,
        numericLoadValue: numericLoadValue,
        numericGetSaveData: numericGetSaveData,
        numericValidate: numericValidate,
        relTimeHtmlBuilder: relTimeHtmlBuilder,
        relTimeLoadValue: relTimeLoadValue,
        relTimeGetSaveData: relTimeGetSaveData,
        relTimeValidate: relTimeValidate,
        stringHtmlBuilder: stringHtmlBuilder,
        stringLoadValue: stringLoadValue,
        stringGetSaveData: stringGetSaveData,
        stringValidate: stringValidate,
        timeHtmlBuilder: timeHtmlBuilder,
        timeLoadValue: timeLoadValue,
        timeGetSaveData: timeGetSaveData
      }
    });
  }());
  
  (function specialFunctions() {
    var DurationSelectEditor,
        RelTimeEditor,
        OverrideRelTimeEditor,
        durationOptions = [
          [ "override.perm",    0 ],
          [ "override.min1",    60000 ],
          [ "override.min15",   15 * 60000 ],
          [ "override.min30",   30 * 60000 ],
          [ "override.hour1",   60 * 60000 ],
          [ "override.hour2",   120 * 60000 ],
          [ "override.hour3",   180 * 60000 ],
          [ "override.custom",  -1 ]
        ];
    
    /**
     * Creates a select dropdown with predetermined choices for selecting
     * an override duration (permanent, 1 minute, 15 minutes, etc.)
     * 
     * @private
     * @memberOf niagara.fieldEditors.mobile.special
     * 
     * @param {Number} duration the preset duration, in milliseconds - if this
     * matches one of the predetermined choices, that one will be preselected
     * in the dropdown.
     * 
     * @returns {jQuery} a <code>&lt;select&gt;</code> element with override
     * duration choices
     */
    function durationSelectHtmlBuilder(targetElement, callbacks) {
      var select = $('<select class="durationChoice" data-theme="a" />'),
          lex = baja.lex('control'),
          max = this.facets.get('max'),
          i,
          name,
          millis;

      // for a RelTime, only support "max" facets of type RelTime,
      // c.f. WritableSupport#getMaxOverrideDuration()
      if (max !== null && !max.getType().is('baja:RelTime')) {
        max = null;
      }

      function addOption(name, millis) {
        var option = $('<option value="' + millis + '"/>');
        option.text(lex.get(name));
        option.appendTo(select);
      }

      for (i = 0; i < durationOptions.length; i++) {
        name = durationOptions[i][0];
        millis = durationOptions[i][1];

        // add to dropdown if there is no max set, if it's a custom duration,
        // or the duration is less than the max
        if (max === null || millis === -1 || (millis > 0 && millis <= max.getMillis())) {
          addOption(name, millis);
        }
      }

      targetElement.append(select);
      callbacks.ok();
    }
    
    /**
     * Sets the value of the duration select dropdown depending on the value
     * in millis of the input RelTime.
     * 
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    function durationSelectLoadValue(value, callbacks) {
      var millis = value ? value.getMillis() : '-1',
          select = this.$dom.find('select'),
          option = select.find('option[value="' + millis + '"]');
      if (option.length) {
        select.val(String(millis));
      } else {
        select.val('-1');
      }
      select.selectmenu('refresh');
      callbacks.ok();
    }
    
    /**
     * Gets a RelTime depending on the currently selected duration. If
     * Custom is selected, returns -1 (since we have no RelTime.NULL).
     * 
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    function durationSelectGetSaveData(callbacks) {
      var select = this.$dom.find('select'),
          val = Number(select.val());
      callbacks.ok(val === -1 ? -1 : baja.RelTime.make(Number(val)));
    }
    
    /**
     * Builds a composite editor for a RelTime - one editor for the duration
     * select dropdown, and one editor for the actual datebox.
     * 
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    function overrideRelTimeHtmlBuilder(targetElement, callbacks) {
      var that = this,
          durationEd = new DurationSelectEditor(baja.RelTime.make(0),
            that.value,
            'duration',
            { parent: that }),
          relTimeEd = new RelTimeEditor(baja.RelTime.make(0),
            that.value,
            'duration',
            { parent: that }),
          maxOverride = that.facets.get('maxOverrideDuration');

      function buildEditors(maxOverride) {
        //propagate maxOverride facet down as max facet of the relTime editor
        //and duration dropdown
        if (maxOverride !== null) {
          relTimeEd.facets = baja.Facets.make(relTimeEd.facets, {
            max: maxOverride,
            min: baja.RelTime.make(1000)
          });
          // max override is set, so don't allow permanent etc.
          durationEd.facets = baja.Facets.make(durationEd.facets, {
            max: maxOverride,
            min: baja.RelTime.make(1000)
          });
        }

        durationEd.initializeDOM(targetElement, function () {
          that.$durationSelectEditor = durationEd;
          relTimeEd.initializeDOM(targetElement, function () {
            that.$relTimeEditor = relTimeEd;
            callbacks.ok();
          });
        });
      }

      if (maxOverride !== null) {
        buildEditors(maxOverride);
      } else {
        baja.Ord.make('station:|slot:/').get({
          lease: 100,
          ok: function (station) {
            var facets = station.get('sysInfo');
            buildEditors(facets.get('maxOverrideDuration'));
          },
          fail: callbacks.fail
        });
      }
    }
    
    /**
     * Sets the values of both the duration select dropdown and the datebox
     * to the given RelTime. If the given RelTime does not match an existing
     * duration in the dropdown, then the dropdown will be set to Custom
     * and the datebox set to read-enabled.
     * 
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    function overrideRelTimeLoadValue(relTime, callbacks) {
      var that = this,
          durationEd = that.$durationSelectEditor,
          relTimeEd = that.$relTimeEditor;
      
      durationEd.loadValue(relTime, function () {
        relTimeEd.loadValue(relTime, function () {
          durationEd.getSaveData(function (selectedRelTime) {
            relTimeEd.setReadonly(that.isReadonly() || selectedRelTime !== -1);
            callbacks.ok();
          });
        });
      });
    }
    
    /**
     * Arms a handler on the duration select dropdown. Whenever the user changes
     * the selection, the selected RelTime value will be loaded into the
     * datebox. The datebox will be set to read-enabled if Custom is selected,
     * otherwise read-only.
     * 
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    function overrideRelTimeArmHandlers() {
      var that = this,
          relTimeEd = that.$relTimeEditor,
          select = that.$dom.find('select');
      select.change(function () {
        var val = $(this).val();
        if (val === '-1') {
          relTimeEd.setReadonly(false);
          relTimeEd.loadValue(baja.RelTime.make(0));
        } else {
          relTimeEd.setReadonly(true);
          relTimeEd.loadValue(baja.RelTime.make(Number(val)));
        }
        that.setModified(true);
      });
    }
    
    /**
     * Just retrieves the save data from the RelTime datebox.
     * 
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    function overrideRelTimeGetSaveData(callbacks) {
      var ed = this.$relTimeEditor;
      ed.validate({
        ok: function () {
          ed.getSaveData(callbacks);
        },
        fail: callbacks.fail
      });
    }
    
    DurationSelectEditor = defineEditor(MobileFieldEditor, {
      doInitializeDOM: durationSelectHtmlBuilder,
      getSaveData: durationSelectGetSaveData,
      doLoadValue: durationSelectLoadValue
    });
    
    RelTimeEditor = defineEditor(MobileFieldEditor, {
      doInitializeDOM: simple.relTimeHtmlBuilder,
      getSaveData: simple.relTimeGetSaveData,
      doLoadValue: simple.relTimeLoadValue,
      validate: simple.relTimeValidate,
      postInitializeDOM: handleDateBox
    });
    
    /**
     * @memberOf niagara.fieldEditors.mobile.special
     * @private
     */
    OverrideRelTimeEditor = defineEditor(MobileFieldEditor, {
      doInitializeDOM: overrideRelTimeHtmlBuilder,
      doLoadValue: overrideRelTimeLoadValue,
      getSaveData: overrideRelTimeGetSaveData,
      armHandlers: overrideRelTimeArmHandlers
    });
    
    /**
     * A namespace containing miscellaneous functions for creating specialty
     * field editors.
     * 
     * @namespace
     * @name niagara.fieldEditors.mobile.special
     */
    util.api(special, {
      'public': {
        OverrideRelTimeEditor: OverrideRelTimeEditor
      },
      'private': {
        durationSelectGetSaveData: durationSelectGetSaveData,
        durationSelectHtmlBuilder: durationSelectHtmlBuilder,
        durationSelectLoadValue: durationSelectLoadValue,
        overrideRelTimeArmHandlers: overrideRelTimeArmHandlers,
        overrideRelTimeGetSaveData: overrideRelTimeGetSaveData,
        overrideRelTimeHtmlBuilder: overrideRelTimeHtmlBuilder,
        overrideRelTimeLoadValue: overrideRelTimeLoadValue
      }
    });
  }());
  
  
  ////////////////////////////////////////////////////////////////
  // field editors for StatusString, StatusBoolean, etc
  ////////////////////////////////////////////////////////////////
  
  (function statusFunctions() {
    
    /**
     * Adds the null/default checkbox to a primitive editor to create a
     * Status field editor.
     * 
     * @private
     * @memberOf niagara.fieldEditors.mobile.status
     * @param {jQuery} div the field editor div to append the checkbox to
     * @param {String} name the slot name for this field editor div
     * @param {baja.Status} status the initial status value for the edited
     * status object
     */
    function appendNullDefaultCheckbox(div, name) {
      var fieldset = $('<fieldset data-role="controlgroup" class="nulldefault"/>'),
          checkbox = $('<input type="checkbox" data-theme="a" name="nulldefault"/>')
            .attr('id', 'nulldefault')
            .addClass('hidden'),
          label = $('<label/>')
            .attr('for','nulldefault')
            .text(getMobileLex().get('propsheet.nullDefault'));
      
      fieldset.append(checkbox).append(label);
      div.append(fieldset);
    }
    
    /**
     * Given a wrapped editor for a Status type, retrieves the appropriate
     * Status to set depending on whether the Null/Default box is checked.
     * 
     * @private
     * @memberOf niagara.fieldEditors.mobile.status
     * @param {jQuery} div the wrapped field editor
     * @param {String} name the slot name of the edited object
     * @returns {baja.Status} NULL if the check box is checked, ok if not
     */
    function getStatus(div, name) {
      var nullDefault = div.find('input[name=nulldefault]').is(':checked'),
          status = nullDefault ? baja.Status.nullStatus : baja.Status.ok;
      return status;
    }
    
    /**
     * Sets the input element on a Status editor to readonly or not, depending
     * on whether the Null/Default checkbox is checked.
     * 
     * @private
     * @memberOf niagara.fieldEditors.mobile.status
     * @param {jQuery} div the editor div
     * @param {String} name the slot name of the property we are editing
     */
    function readOnlyInput(div, name) {
      var input = div.find('input[type=text], textarea'),
          select = div.find('select'),
          checkbox = div.find('input[type=checkbox]');
      if (checkbox.is(':checked')) {
        input.attr('disabled', 'disabled');
        select.filter(':jqmData(role="slider")').slider('disable');
        select.filter(':jqmData(role!="slider")').selectmenu('disable');
      } else {
        input.removeAttr('disabled');
        select.filter(':jqmData(role="slider")').slider('enable');
        select.filter(':jqmData(role!="slider")').selectmenu('enable');
      }
    }
    
    /**
     * Creates a Status field editor by wrapping a primitive editor in some
     * extra processing. For instance, it will take an editor for baja:String
     * and return an editor for baja:StatusString.
     * 
     * @private
     * @memberOf niagara.fieldEditors.mobile.status
     * @param {Function} Editor a primitive field editor constructor
     * @returns {Function} a Status field editor constructor
     */
    function wrapWithStatus(SimpleEditor) {

      return function StatusEditor(container, editedObject, slot, params) {
        var editor = new SimpleEditor(container, editedObject, slot, params),
            validateSteps = editor.$validateSteps,
            aop = util.aop,
            feUtil = util.fieldEditors;
        
        
        //after building the primitive HTML, append the null/default checkbox
        util.aop.before(editor, 'doInitializeDOM', function (args) {
          var targetElement = args[0],
              callbacks = args[1],
              name = this.name;
          
          util.aop.before(callbacks, 'ok', function () {
            appendNullDefaultCheckbox(targetElement, name);
          });
        });
        
        //when validating, we want to validate the Simple value
        baja.iterate(validateSteps, function (validateStep, i) {
          validateSteps[i] = function (saveData, callbacks) {
            validateStep.call(this, saveData.getValue(), callbacks);
          };
        });
        
        //add a handler to enable/disable the input when the checkbox is
        //checked or unchecked
        aop.after(editor, 'armHandlers', function (args, value) {
          var checkbox = this.$dom.find('input[type="checkbox"]'),
              that = this;
          checkbox.change(function () {
            if (!that.isReadonly()) {
              readOnlyInput(that.$dom, that.name);
            }
          });
        });
        
        aop.after(editor, 'setReadonly', function (args, value) {
          if (!this.isReadonly()) {
            readOnlyInput(this.$dom, this.name);
          }
        });
        
        //after setting the primitive value, set the null/default checkbox also
        editor.doLoadValue = function (value, callbacks) {
          var statusValue = value;
          if (!statusValue) {
            return callbacks.ok();
          }
          
          SimpleEditor.prototype.doLoadValue.call(this, statusValue.getValue(), {
            ok: function () {
              var status = statusValue.getStatus(),
                  checkbox = editor.$dom.find('input[type="checkbox"]');
              
              if (status.isNull()) {
                checkbox.attr('checked', 'checked');
              } else {
                checkbox.removeAttr('checked');
              }
              callbacks.ok();
            },
            fail: callbacks.fail
          });
        };
        
        //retrieve the primitive value, then construct a Status object and
        //set its status based on the checkbox's value
        editor.getSaveData = function (obj) {
          SimpleEditor.prototype.getSaveData.call(this, {
            ok: function (simpleValue) {
              var status = getStatus(editor.$dom, editor.name),
                  saveData = toSaveDataComponent(editor.value);
                  
              saveData.setValue(simpleValue);
              saveData.setStatus(status);
              obj.ok(saveData);
            },
            fail: obj.fail
          });
        };
        
        //enable/disable input based on checkbox status
        aop.after(editor, 'refreshWidgets', function (args, value) {
          if (!this.isReadonly()) {
            readOnlyInput(this.$dom, this.name);
          }
        });
        
        return editor;
      };
    }
    
    /**
     * A namespace containing field editor functions for working with 
     * <code>BStatusValue</code>s.
     * @namespace
     * @name niagara.fieldEditors.mobile.status
     */
    util.api(status, {
      'public': {
        wrapWithStatus: wrapWithStatus
      },
      'private': {
        appendNullDefaultCheckbox: appendNullDefaultCheckbox,
        getStatus: getStatus,
        readOnlyInput: readOnlyInput
      }
    });
  }());
  

  
  (function ordFunctions() {
    /**
     * @memberOf niagara.fieldEditors.mobile.ord
     * @private
     */
    function validateSelector(selector) {
      selector = String(selector);
      
      if (selector.indexOf('slotPathOrd') < 0 || selector.indexOf('displayName') < 0) {
        throw "Must select slotPathOrd and displayName";
      }
    }
    
    /**
     * Builds an empty select dropdown (to be populated in 
     * <code>doLoadValue</code>)
     * @memberOf niagara.fieldEditors.mobile.ord
     */
    function ordComponentSelectorHtmlBuilder(targetElement, callbacks) {
      //initial html: empty. building up of dropdown will occur in loadValue
      var select = $('<select data-role="selectmenu" data-theme="a" />')
//                     .attr('id', this.name)
                     .attr('name', this.name)
                     .appendTo(targetElement),
          loading = $('<option/>')
                      .text(getMobileLex().get('loading'))
                      .appendTo(select);
      callbacks.ok();
    }

    /**
     * Populates a select dropdown with the results of a BQL query.
     * 
     * @param {baja.Ord} value the ORD with which we want to retrieve a list of
     * available components - this ORD should be in the form of a BQL query and
     * return at least the fields <code>slotPathOrd</code> and 
     * <code>displayName</code>.
     * <p>
     * Example: <code>station:|slot:/|bql: select displayName, slotPathOrd 
     * from schedule:CalendarSchedule</code>
     * 
     * @memberOf niagara.fieldEditors.mobile.ord
     */
    function ordComponentSelectorLoadValue(value, callbacks) {
      var editorDiv = this.$dom,
          select = editorDiv.find('select'),
          myOrd = this.value;
      
      select.html($('<option/>').text(getMobileLex().get('loading')));

      try {
        validateSelector(value);
      } catch (e) {
        this.$dom.text(e);
      }
      
      baja.Ord.make(String(value)).get({
        cursor: {
          before: function () {
            select.empty();
          },
          each: function () {
            var slotPathOrd = this.get('slotPathOrd'),
                displayName = this.get('displayName'),
                option = $('<option/>')
                  .val(slotPathOrd)
                  .text(displayName);
            
            if (util.ord.equivalent(myOrd, slotPathOrd)) {
              option.attr('selected', 'selected');
            }
            option.appendTo(select);
          },
          after: function () {
            select.selectmenu('refresh');
            callbacks.ok();
          },
          fail: callbacks.fail,
          limit: 100
        },
        fail: callbacks.fail
      });
    }
    
    /**
     * @memberOf niagara.fieldEditors.mobile.ord
     */
    function ordComponentSelectorGetSaveData(obj) {
      obj.ok(baja.Ord.make(this.$dom.find('select').val()));
    }
    
    /**
     * A namespace containing field editor functions for working with 
     * <code>BOrds</code>s.
     * @namespace
     * @name niagara.fieldEditors.mobile.ord
     */
    util.api(ord, {
      ordComponentSelectorHtmlBuilder: ordComponentSelectorHtmlBuilder,
      ordComponentSelectorLoadValue: ordComponentSelectorLoadValue,
      ordComponentSelectorGetSaveData: ordComponentSelectorGetSaveData
    });
  }());
  
  (function facetsFunctions() {
    
    /**
     * Called when first loading the facets editor or when selecting a 
     * different facet via the dropdown; loads a field editor so the user
     * can edit that facet.
     * 
     * @memberOf niagara.fieldEditors.mobile.facets
     * @private
     * @param {jQuery} subEditorDiv the div in which to load the sub-editor
     * @param {String} key the key for the edited facet
     */
    function loadFacetEditor(subEditorDiv, key) {
      var that = this,
          editedValue = that.facetValues[key];
      
      if (editedValue === undefined) {
        return;
      }
      
      makeFor({
        value: editedValue,
        parent: that,
        readonly: that.isReadonly()
      }, function (editor) {
        delete that.currentEditor;
        delete that.currentKey;
        
        editor.buildAndLoad(subEditorDiv.empty(), {
          ok: function () {
            that.$dom.trigger('create');
            that.$dom.trigger('updatelayout');
            that.currentEditor = editor;
            that.currentKey = key;
          }
        });
      });
      
    }
    
    /**
     * Retrieves the value from the current facet editor and saves that value
     * into <code>this.facetValues</code>. Called when saving the overall
     * <code>BFacets</code> editor, or when selecting a different facet
     * from the dropdown.
     * 
     * @memberOf niagara.fieldEditors.mobile.facets
     * @private
     * @param {niagara.fieldEditors.BaseFieldEditor} facetsEditor the editor
     * for the enclosing <code>BFacets</code> object
     * @param {niagara.fieldEditors.BaseFieldEditor} subFacetEditor the editor
     * for the currently editor sub-facet
     * @param {Function} ok a function to execute once the current facet value
     * is saved
     */
    function saveFacetEditor(facetsEditor, subFacetEditor, callbacks) {
      callbacks = callbackify(callbacks);
      if (subFacetEditor && subFacetEditor.isModified()) {
        subFacetEditor.getSaveData({
          ok: function (saveData) {
            facetsEditor.facetValues[facetsEditor.currentKey] = saveData;
            callbacks.ok();
          },
          fail: callbacks.fail
        });
      } else {
        callbacks.ok();
      }
    }
    
    /**
     * For editing a <code>baja.Facets</code> object, creates a select dropdown
     * containing available facets on that object and a div to contain the 
     * sub-editor for an individual facet.
     * @memberOf niagara.fieldEditors.mobile.facets
     */
    function facetsHtmlBuilder(targetElement, callbacks) {
      var facetSelect = $('<select data-role="selectmenu" data-theme="a"><option value="-1">loading</option></select>'),
          subEditorDiv = $('<div class="subEditor"/>');
      
      facetSelect.appendTo(targetElement);
      subEditorDiv.appendTo(targetElement);
      
      callbacks.ok();
    }
    
    /**
     * Populates the select dropdown with an entry for each facet on the edited
     * <code>baja.Facets</code> object. Also, initializes 
     * <code>this.facetValues</code> with the initial values of each facet.
     * The sub-field editors will set the values on 
     * <code>this.facetValues</code>; this object will be assembled into a new
     * <code>baja.Facets</code> object in <code>getSaveData()</code>.
     * @memberOf niagara.fieldEditors.mobile.facets
     */
    function facetsLoadValue(value, callbacks) {
      var that = this,
          facetSelect = that.$dom.find('select'),
          facetValues = that.facetValues = {},
          facets = value,
          firstKey;
      
      facetSelect.empty();
      baja.iterate(facets.getKeys(), function (key) {
        var option = $('<option/>').attr('value', key).text(key);
        facetSelect.append(option);
        facetValues[key] = facets.get(key);
        
        firstKey = firstKey || key;
      });      
      
      facetSelect.val(firstKey).trigger('change');
      that.$lastKey = firstKey;
      
      callbacks.ok();
    }
    

    /**
     * Arms a change listener on the select dropdown so that whenever a new
     * facet is selected, a sub-field-editor will be loaded for that facet.
     * Also undelegates the usual user value change detection - selecting a new
     * facet from the dropdown should not in itself trigger 
     * <code>editorchange</code>.
     * @memberOf niagara.fieldEditors.mobile.facets
     */
    function facetsArmHandlers() {
      var that = this,
          facetSelect = that.$dom.find('select'),
          subEditorDiv = that.$dom.find('div.subEditor');
      
      that.$dom.undelegate();
      
      facetSelect.bind('change', function () {
        var key = $(this).val();
        saveFacetEditor(that, that.currentEditor, {
          ok: function () {
            loadFacetEditor.call(that, subEditorDiv, key);
            that.$lastKey = key;
          },
          fail: function (err) {
            dialogs.error(err);
            facetSelect.val(that.$lastKey).selectmenu('refresh');
          }
        });
      });
    }
    
    /**
     * Assembles the values in <code>this.facetValues</code> into a new instance
     * of <code>baja.Facets</code>. This will likely be a combination of the
     * original values from the edited <code>baja.Facets</code> instance (those
     * which were not edited by the user) and new, user-edited values. If the
     * current facet editor is dirty, will save that editor first before
     * retrieving the value.
     * @memberOf niagara.fieldEditors.mobile.facets
     */
    function facetsGetSaveData(obj) {
      var that = this;
      saveFacetEditor(that, that.currentEditor, {
        ok: function () {
          var newKeys = [], newValues = [];
          baja.iterate(that.facetValues, function (value, key) {
            newKeys.push(key);
            newValues.push(value);
          });
          obj.ok(baja.Facets.make(newKeys, newValues));
        },
        fail: obj.fail
      });
    }
    
    /**
     * @namespace
     * @name niagara.fieldEditors.mobile.facets
     */
    util.api(facets, {
      'public': {
        facetsArmHandlers: facetsArmHandlers,
        facetsHtmlBuilder: facetsHtmlBuilder,
        facetsLoadValue: facetsLoadValue,
        facetsGetSaveData: facetsGetSaveData
      },
      'private': {
        loadFacetEditor: loadFacetEditor,
        saveFacetEditor: saveFacetEditor
      }
    });
  }());
  
  (function unitsFunctions() {


    /**
     * @memberOf niagara.fieldEditors.mobile.units
     */
    function unitsHtmlBuilder(targetElement, callbacks) {
      var quantitySelect = $('<select data-role="selectmenu" data-theme="a" name="quantity" />')
            .append('<option>loading</option>'),
          unitSelect = $('<select data-role="selectmenu" data-theme="a" name="units" />')
            .append('<option>loading</option>');
      targetElement.append(quantitySelect);
      targetElement.append(unitSelect);
      
      getQuantityDb({
        ok: function (quantityDb) {
          quantitySelect.empty();
          
          //populate quantity select...
          baja.iterate(quantityDb, function (quantity) {
            var quantityName = quantity.q;
            $('<option/>')
              .attr('value', quantityDb.escapeQuantityName(quantityName || ''))
              .text(quantityName)
              .appendTo(quantitySelect);
          });
          
          //upon selecting a new quantity, grab the units from the quantity db
          //and populate the units select
          quantitySelect.bind('change', function () {
            var quantityName = $(this).find('option:selected').text(),
                units = quantityDb.getUnits(quantityName);
                
            unitSelect.empty();
            baja.iterate(units, function (unit) { 
              var option = $('<option/>')
                    .val(unit.n)
                    .text(unit.n + ' (' + unit.s + ')');
              option.jqmData('unit', unit);
              option.jqmData('dimension', unit.d);
              
              option.appendTo(unitSelect);
            });
            
            unitSelect.selectmenu('refresh').trigger('updatelayout');
          });
          
          quantitySelect.appendTo(targetElement);
          unitSelect.appendTo(targetElement);
          callbacks.ok();
        },
        fail: callbacks.fail
      });
    }

    /**
     * @memberOf niagara.fieldEditors.mobile.units
     */
    function unitsLoadValue(unit, callbacks) {
      
      var quantitySelect = this.$dom.find('select[name=quantity]'),
          unitSelect = this.$dom.find('select[name=units]'),
          unitString = unit.encodeToString(),
          unitArray = unit.encodeToString().split(';'),
          unitName = unitArray[0],
          unitDimension = unitArray[2] || '()';
      
      getQuantityDb(function (db) {
        var unit = db.getUnitFromNameAndDimension(unitName, unitDimension),
            quantity = unit && unit.q,
            quantityName = quantity && quantity.q;
        if (quantityName) {
          quantitySelect.val(quantityDb.escapeQuantityName(quantityName)).trigger('change');
          unitSelect.val(unitName).trigger('change');
        }
      });

      callbacks.ok();
    }

    /**
     * @memberOf niagara.fieldEditors.mobile.units
     */
    function unitsGetSaveData(callbacks) {
      var unitSelect = this.$dom.find('select[name=units]'),
          option = unitSelect.find('option:selected'),
          unit = option.jqmData('unit'),
          arr = [];
          
      arr.push(unit.n);
      arr.push(unit.s === unit.n ? '' : unit.s);
      arr.push(unit.d);
      arr.push(
        (unit.sc === 1 ? '' : '*' + unit.sc) +
        (unit.o ? '+' + unit.o : ''));
      
      //return the string encoding that the station wants - the field editor
      //will automatically make it into an object that bajascript likes to
      //encode. see niagara.fieldEditors#toFakeBajaObject
      callbacks.ok(arr.join(';') + ';');
    }

    /**
     * @namespace
     * @name niagara.fieldEditors.mobile.units
     */
    util.api(units, {
      getQuantityDb: getQuantityDb,
      unitsHtmlBuilder: unitsHtmlBuilder,
      unitsLoadValue: unitsLoadValue,
      unitsGetSaveData: unitsGetSaveData
    });
  }());
  
  (function registerDefaultTypes() {
    function registerAll(obj) {
      baja.iterate(obj, function (Editor, typeSpec) {
        register(typeSpec, Editor);
      });
    }
    
    
    var simpleFEs, statusFEs, compositeFEs, nonEditableFEs, nonEditableFE;
    
    nonEditableFE = defineEditor(MobileFieldEditor, {
      doInitializeDOM: simple.nonEditableHtmlBuilder,
      getSaveData: simple.nonEditableGetSaveData,
      doLoadValue: simple.nonEditableLoadValue
    });
    
    simpleFEs = {
      'baja:AbsTime': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.absTimeHtmlBuilder,
        getSaveData: simple.absTimeGetSaveData,
        doLoadValue: simple.absTimeLoadValue,
        setReadonly: simple.absTimeSetReadonly,
        postInitializeDOM: handleDateBox
      }),
      'baja:Boolean': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.booleanHtmlBuilder,
        getSaveData: simple.booleanGetSaveData,
        doLoadValue: simple.booleanLoadValue
      }),
      'baja:Date': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.dateHtmlBuilder,
        getSaveData: simple.dateGetSaveData,
        doLoadValue: simple.dateLoadValue,
        postInitializeDOM: handleDateBox,
        armHandlers: simple.dateboxArmHandlers
      }),
      'baja:DynamicEnum': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.enumHtmlBuilder,
        getSaveData: simple.dynamicEnumGetSaveData,
        doLoadValue: simple.enumLoadValue
      }),
      'baja:FrozenEnum': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.enumHtmlBuilder,
        getSaveData: simple.frozenEnumGetSaveData,
        doLoadValue: simple.enumLoadValue
      }),
      'baja:Number': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.numericHtmlBuilder,
        getSaveData: simple.numericGetSaveData,
        doLoadValue: simple.numericLoadValue,
        validate: simple.numericValidate
      }),
      'baja:RelTime': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.relTimeHtmlBuilder,
        getSaveData: simple.relTimeGetSaveData,
        doLoadValue: simple.relTimeLoadValue,
        validate: simple.relTimeValidate,
        postInitializeDOM: handleDateBox
      }),
      'baja:String': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.stringHtmlBuilder,
        getSaveData: simple.stringGetSaveData,
        doLoadValue: simple.stringLoadValue,
        validate: simple.stringValidate
      }),
      'baja:Time': defineEditor(MobileFieldEditor, {
        doInitializeDOM: simple.timeHtmlBuilder,
        getSaveData: simple.timeGetSaveData,
        doLoadValue: simple.timeLoadValue,
        postInitializeDOM: handleDateBox,
        armHandlers: simple.dateboxArmHandlers
      }),
      'baja:Facets': defineEditor(MobileFieldEditor, {
        armHandlers: facets.facetsArmHandlers,
        doInitializeDOM: facets.facetsHtmlBuilder,
        getSaveData: facets.facetsGetSaveData,
        doLoadValue: facets.facetsLoadValue,
        setReadonly: facets.facetsSetReadonly
      }),
      'baja:Unit': defineEditor(MobileFieldEditor, {
        doInitializeDOM: units.unitsHtmlBuilder,
        getSaveData: units.unitsGetSaveData,
        doLoadValue: units.unitsLoadValue
      })
    };
    
    statusFEs = {
      "baja:StatusBoolean": status.wrapWithStatus(simpleFEs["baja:Boolean"]),
      "baja:StatusEnum": status.wrapWithStatus(simpleFEs["baja:DynamicEnum"]),
      "baja:StatusNumeric": status.wrapWithStatus(simpleFEs["baja:Number"]),
      "baja:StatusString": status.wrapWithStatus(simpleFEs["baja:String"])
    };
    
    compositeFEs = {
      "control:BooleanOverride": composite.makeComposite([
        { slot: 'duration', key: 'override' },
        'value'
      ]),
      "control:EnumOverride": composite.makeComposite([
        { slot: 'duration', key: 'override' },
        'value'
      ]),
      "control:NumericOverride": composite.makeComposite([
        { slot: 'duration', key: 'override' },
        'value'
      ]),
      "control:Override": composite.makeComposite([
        { slot: 'duration', key: 'override' }
      ]),
      "control:StringOverride": composite.makeComposite([
        { slot: 'duration', key: 'override' },
        'value'
      ]),
      "baja:AbsTimeRange": composite.makeComposite(['startTime', 'endTime']),
      "baja:TimeRange": composite.makeComposite(['startTime', 'endTime'])
    };
    
    nonEditableFEs = {
      "baja:EnumRange": nonEditableFE,
      "baja:Password": nonEditableFE,
      "baja:TimeZone": nonEditableFE
//      "baja:TypeSpec": nonEditableFE
    };
    
    registerAll(simpleFEs);
    registerAll(statusFEs);
    registerAll(compositeFEs);
    registerAll(nonEditableFEs);
    
    register('baja:Boolean', fe.defineEditor(MobileFieldEditor, {
      doInitializeDOM: simple.booleanCheckboxHtmlBuilder,
      doLoadValue: simple.booleanCheckboxLoadValue,
      getSaveData: simple.booleanCheckboxGetSaveData
    }), 'checkbox');
  }());
  
  (function registerSpecialTypes() {
    register('baja:RelTime', special.OverrideRelTimeEditor, 'override');
    
    register('baja:Ord',
      defineEditor(MobileFieldEditor, {
        doInitializeDOM: ord.ordComponentSelectorHtmlBuilder,
        getSaveData: ord.ordComponentSelectorGetSaveData,
        doLoadValue: ord.ordComponentSelectorLoadValue
      }), 'componentSelector');
  }());
  

  /**
   * @namespace
   * @name niagara.fieldEditors.mobile
   */
  util.api('niagara.fieldEditors.mobile', {
    'public': {
      MobileFieldEditor: MobileFieldEditor,
      
      facets: facets,
      ord: ord,
      simple: simple,
      special: special,
      status: status,
      units: units
    }
  });
}());
