/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Functions relating to the BaseFieldEditor object that all
 * field editors in Niagara mobile apps extend from. Also, functions relating
 * to registering field editors on Types.
 * 
 * @author Logan Byam
 * @version 0.0.1
 */

/*jslint white: true, sloppy: true */
/*global niagara, $, baja, LazyLoad */

(function () {

  niagara.util.require(
    'niagara.util.View',
    'niagara.util.flow',
    'LazyLoad.js'
  );
  
  //imports
  var util = niagara.util,
      callbackify = util.callbackify,
      View = util.View,

      //private vars
      fieldEditorMap = {},
      dataTypeMap,
      
      buildAndLoadWorkflow,
      lazyLoadWorkflow,
      saveWorkflow,
      validateWorkflow,
      
      //constants
      MOBILE_FIELD_EDITOR_TYPE = 'mobile:MobileFieldEditor',
      WORKBENCH_FIELD_EDITOR_TYPE = 'workbench:WbFieldEditor',
      FIELD_EDITOR_FACET = 'fieldEditor',
      
      //exports
      BaseFieldEditor,
      composite = {};
  
  //copy-pasted from SimpleIntrospector.java
  dataTypeMap = {
    "baja.Boolean": 'b',
    "baja:Integer": 'i',
    "baja:Long": 'l',
    "baja:Float": 'f',
    "baja:Double": 'd',
    "baja:String": 's',
    "baja:DynamicEnum": 'e',
    "baja:EnumRange": 'E',
    "baja:AbsTime": 'a',
    "baja:RelTime": 'r',
    "baja:TimeZone": 'z',
    "baja:Unit": 'u'
  };
  
  
////////////////////////////////////////////////////////////////
//Async workflows
////////////////////////////////////////////////////////////////
  
  /**
   * Sequential async workflow that simply performs <code>initializeDOM</code>
   * and <code>loadValue</code> on a field editor in one step.
   * @memberOf niagara.fieldEditors
   * @private
   * @field
   */
  buildAndLoadWorkflow = util.flow.sequential(
    function (cx) {
      cx.editor.initializeDOM(cx.targetElement, this);
    },
    function (cx) {
      cx.editor.loadValue(cx.editor.value, this);
    }
  );
  
  /**
   * Parallel async workflow that downloads JS/CSS resources using LazyLoad.js.
   * @memberOf niagara.fieldEditors
   * @private
   * @field
   */
  lazyLoadWorkflow = util.flow.parallel(
    function lazyLoadWorkflow__step1__loadJS(cx) {
      if (cx.js.length) {
        LazyLoad.js(cx.js, this.ok);
      } else {
        this.ok();
      }
    },
    function lazyLoadWorkflow__step2__loadCSS(cx) {
      if (cx.css.length) {
        LazyLoad.css(cx.css, this.ok);
      } else {
        this.ok();
      }
    }
  );
  
  /**
   * Sequential async workflow that retrieves save data from a field editor,
   * performs the save (performing a server-side update, if we are editing
   * a mounted component), triggers <code>editorsave</code> on the field
   * editor's DOM element, and resets the modified flag to false.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @field
   */
  saveWorkflow = util.flow.sequential(
    function saveWorkflow__step1__getSaveData(cx) {
      cx.editor.getSaveData(this);
    },
    function saveWorkflow__step2__doSaveValue(cx, saveData) {
      cx.editor.doSaveValue(cx.editor.value, saveData, this);
    },
    function saveWorkflow__step3__semanticsCheckAndSave(cx, savedValue) {
      var editor = cx.editor, 
          container = editor.container, 
          slot = editor.slot,
          savedType = savedValue.getType(),
          editorValue = editor.value;
      
      if (savedType.isComponent() &&
          savedType.is(editorValue.getType()) &&
          savedValue !== editorValue) {
        return this.fail("Component editors must support edit-by-ref semantics");
      }
      
      cx.savedValue = savedValue;
     
      if (!savedType.isComponent() && container && slot) {
        //since we're not editing a Component, the batch could not have been 
        //used in doSaveValue - so it's good to be used here instead
        container.set($.extend(this, { slot: slot, value: savedValue }));
      } else {
        //no need to save - components get edited in-place
        this.ok();
      }
    },
    function saveWorkflow__step4__setModifiedAndValue(cx) {
      var editor = cx.editor;
      editor.$dom.trigger('editorsave', [editor]);
      editor.setModified(false);
      this.ok(editor.value = cx.savedValue);
    }
  );
  
  /**
   * Sequential async workflow that retrieves save data from a field editor and
   * performs validation on it.
   * @memberOf niagara.fieldEditors
   * @private
   * @field
   */
  validateWorkflow = util.flow.sequential(
    function validateWorkflow__step1__getSaveData(cx) {
      var editor = cx.editor,
          validateSteps = editor.$validateSteps;
      if (validateSteps.length > 0) {
        editor.getSaveData(this);
      } else {
        this.exit();
      }
    },
    function validateWorkflow__step2__doValidate(cx, saveData) {
      var editor = cx.editor,
          validateSteps = editor.$validateSteps,
          validateInvocations;

      if (validateSteps.length > 0) {
        //we have one or more validation steps to perform - do them all
        //sequentially
        validateInvocations = [];
        baja.iterate(validateSteps, function (validateStep) {
          validateInvocations.push(function (callbacks) {
            validateStep.call(editor, saveData, callbacks);
          });
        });
        util.flow.runSequential(validateInvocations, this);
      } else {
        //no validation steps to perform
        this.ok(saveData);
      }
    }
  );
  
  
////////////////////////////////////////////////////////////////
//Utility Functions
////////////////////////////////////////////////////////////////
  
  /**
   * Wraps a field editor in a new div with a label attached.
   * 
   * @memberOf niagara.fieldEditors
   */
  function toLabeledEditorContainer(editor, targetElement) {
    var containerDiv = $('<div class="labeledEditor"/>').appendTo(targetElement),
        label = $('<label class="editorLabel" />'),
        editorContainer = $('<div class="editorContainer"/>');
      
    if (editor.label !== undefined) {
      label.text(editor.label);
    }
    
    containerDiv.append(label).append(editorContainer);
    return editorContainer;
  }
  
  
////////////////////////////////////////////////////////////////
// BaseFieldEditor
////////////////////////////////////////////////////////////////
  
  /**
   * The base field editor. This editor will come with all functionality
   * built in except hooks for building the HTML, setting a value, and
   * retrieving the set value. This constructor should never be invoked
   * directly - use <code>niagara.fieldEditors.makeFor</code> method 
   * instead.
   * 
   * @private
   * @class
   * @extends niagara.util.View
   * @memberOf niagara.fieldEditors
   * @see niagara.fieldEditors.makeFor
   */
  BaseFieldEditor = View.subclass(function BaseFieldEditor(value, container, slot, params) {
    if (value === undefined || value === null) {
      //for subclassing / prototype creation
      return this;
    }
    
    var modified = false,
        name = String(slot);
    
    params = baja.objectify(params);
    
    if (value.getType().isComplex() && !value.getType().isComponent()) {
      value = value.newCopy();
    }
    
    this.value = value;
    this.container = container || (value.getParent && value.getParent());
    this.slot = slot || (value.getName && value.getName());
    this.name = String(this.slot);
    
    this.params = params;
    this.facets = params.facets || baja.Facets.DEFAULT;
    this.label = params.label ||
                 (container && container.getDisplayName && slot && container.getDisplayName(slot)) ||
                 (value.getName && value.getName()) ||
                 (slot && String(slot));
    
    this.parent = params.parent;
    
    this.isModified = function () {
      return modified;
    };
    
    this.setModified = function (dirty, params) {
      if (this.$squelchChanges) {
        return;
      }
      
      params = baja.objectify(params);
      var that = this,
          parent = that.parent,
          div = that.$dom;
      
      modified = dirty;
      
      if (dirty) {
        if (parent) {
          parent.setModified(true, $.extend({ silent: true }, params));
        }
        if (div && !params.silent) {
          div.trigger('editorchange', [that, params]);
        }
      }
    };
    
    this.$validateSteps = [];
    
    this.$readonly = params.readonly || (params.parent && params.parent.isReadonly()) || false;
    
    this.postCreate();
  });
  
  /**
   * A field editor subclass may, optionally, define some extra initialization
   * code here to set up the state of the editor after the constructor
   * completes. The default behavior is to do nothing.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#postCreate
   * @function
   */
  BaseFieldEditor.prototype.postCreate = function () {
    
  };
  
  /**
   * After setting the field editor value, this method will be called to refresh
   * any widgets to reflect the updated value. By default, does nothing.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#refreshWidgets
   * @function
   */
  BaseFieldEditor.prototype.refreshWidgets = function refreshWidgets() {
    
  };
  
  /**
   * Sets a field editor's readonly status. Default behavior is simply to set
   * the <code>disabled</code> attribute on any <code>input</code> or
   * <code>select</code> elements contained in the field editor's DOM.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#setReadonly
   * @function
   * @param {Boolean} readonly
   */
  BaseFieldEditor.prototype.setReadonly = function setReadonly(readonly) {
    var inputs = this.$dom.find('select,input');
    
    this.$readonly = readonly;
    
    if (readonly) {
      inputs.attr('disabled', 'disabled');
    } else {
      inputs.removeAttr('disabled');
    }
  };
  
  /**
   * @name niagara.fieldEditors.BaseFieldEditor#isReadonly
   * @function
   * @return {Boolean}
   */
  BaseFieldEditor.prototype.isReadonly = function isReadonly() {
    return this.$readonly;
  };
  
  function isCheckable(element) {
    return element.is('input[type="checkbox"]') || 
      element.is('input[type="radio"]');
  }
  
  BaseFieldEditor.$detectChangeEvent = function (event) {
    var element = $(this),
        editorDiv = element.closest('div.editor'),
        editor = editorDiv.data('editor'),
        oldValue = element.data('oldValue'),
        value;
    
    if (isCheckable(element)) {
      //for JQM checkboxes "val()" always seems to return "on"
      value = element.is(':checked');
    } else {
      value = element.val();
    }
    
    if (!editor.$squelchChanges && oldValue !== undefined && (oldValue !== value)) {
      editor.setModified(true, {
        oldValue: oldValue,
        newValue: value,
        element: element
      });
    }
    
    element.data('oldValue', value);
  };
  
  /**
   * Arms change handlers on this field editor - will set this editor's isModified
   * flag if data is typed into a text input, say, or if a Null/Default checkbox
   * is checked or unchecked.
   * @name niagara.fieldEditors.BaseFieldEditor#armChangeHandlers
   * @function
   */
  BaseFieldEditor.prototype.armHandlers = function armHandlers() {
    var that = this,
        editorDiv = this.$dom;
    
    editorDiv.delegate('input,select,textarea',
        'change keyup', BaseFieldEditor.$detectChangeEvent);
  };
  
  /**
   * A quick way of performing <pre>initializeDOM</pre> and <pre>loadValue</pre> 
   * in one step. The value passed to <pre>loadValue</pre> will be the same
   * passed into the constructor as <pre>params.value</pre>.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#buildAndLoad
   * @function
   * @param {jQuery} targetElement the div in which the editor should build ids
   * html
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  BaseFieldEditor.prototype.buildAndLoad = function buildAndLoad(targetElement, callbacks) {
    buildAndLoadWorkflow.invoke({
      editor: this,
      targetElement: targetElement
    }, callbacks);
  };
  


  /**
   * Prevents <pre>setModified</pre> from being called until 
   * <pre>loadValue</pre> has completed - this allows any widgets to initialize,
   * perhaps triggering change events, before we start listening for
   * user-entered changes.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#loadValue
   * @function
   * @see niagara.util.mobile.View#loadValue
   */
  util.aop.before(BaseFieldEditor.prototype, 'loadValue', function (args) {
    var value = args[0], 
        callbacks = args[1],
        that = this;
    
    callbacks = callbackify(callbacks);
    
    that.$squelchChanges = true;
    
    util.aop.before(callbacks, 'ok', function () {
      that.$squelchChanges = false;
    });
    util.aop.before(callbacks, 'fail', function () {
      that.$squelchChanges = false;
    });
    
    return [value, callbacks];
  });
  
  /**
   * Adds CSS classes to a div that correspond to the given Type and all of
   * its supertypes.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {jQuery} div the div on which to add CSS classes
   * @param {Type} type
   */
  function addSuperTypeClasses(div, type) {
    baja.iterate(type, function (type) {
      div.addClass(String(type.getTypeSpec()).replace(':', '-'));
    }, function (type) {
      return type.getSuperType();
    });
  }
  
  /**
   * Adds CSS classes reflecting the whole supertype chain - so that your
   * CSS can target divs with a CSS class of <pre>baja-StatusValue</pre>,
   * for instance.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#postLoadValue
   * @function
   * @see niagara.util.mobile.View#postLoadValue
   */
  BaseFieldEditor.prototype.postLoadValue = function postLoadValue(callbacks) {
    var value = this.value;
    if (value !== null && value !== undefined) {
      addSuperTypeClasses(this.$dom, value.getType());
    }
    
    this.$dom.find('input,select,textarea').each(function () {
      var $this = $(this);
      if (isCheckable($this)) {
        $this.data('oldValue', $this.is(':checked'));
      } else {
        $this.data('oldValue', $this.val());
      }
    });

    callbacks.ok();
  };

  
  /**
   * A BaseFieldEditor will wrap its content in a div with
   * <pre>class="editor"</pre>. It will
   * also give this div <pre>data('editor', this)</pre>.
   * 
   * @name niagara.fieldEditor.BaseFieldEditor#initializeDOM
   * @function
   * @see niagara.util.mobile.View#initializeDOM
   */
  util.aop.before(BaseFieldEditor.prototype, 'initializeDOM', function beforeInitializeDOM(args) {
    var targetElement = args[0], callbacks = args[1];
    targetElement = $('<div data-role="fieldcontain" class="editor"/>')
      .data('editor', this)
      .appendTo(targetElement);
    return [targetElement, callbacks];
  });
  
  /**
   * @memberOf niagara.fieldEditors
   * @private
   */
  function toFakeBajaObject(stringEncoded, type) {
    //getDataTypeSymbol is required in order to encode a simple as part of a
    //Facets object
    
    var obj = baja.$(type).decodeFromString(stringEncoded),
        symbol = dataTypeMap[type.toString()];
    
    obj.getDataTypeSymbol = function () {
      return symbol;
    };
    
    return obj;
  }
  /**
   * Pulls any user-entered data from the editor div's input/select elements,
   * and assembles them into an actual value object.
   * @name niagara.fieldEditors.BaseFieldEditor#retrieveValue
   * @function
   * @param {Object} obj an object with ok/fail callbacks and other parameters
   * @param {Function} obj.ok a callback to execute once the save data has been
   * retrieved. This save data object will be passed as the first argument to
   * obj.ok.
   * @param {Function} [obj.fail] a callback to execute if there is a problem
   * retrieving the save data
   * @returns {baja.Value} the value constructed from user-entered data 
   */
  BaseFieldEditor.prototype.getSaveData = function getSaveData(callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this,
        myType = that.value.getType();
    
    try {
      that.doGetSaveData({
        ok: function (saveData) {
          if (myType.is('baja:Simple') &&
              !myType.is('baja:String') &&
              saveData.getType().is('baja:String')) {
            callbacks.ok(toFakeBajaObject(saveData, myType));
          } else {
            callbacks.ok(saveData);
          }
        },
        fail: callbacks.fail
      });
    } catch (e) {
      callbacks.fail(e);
    }
  };
  

  /**
   * Retrieves the currently entered value, and saves it. 
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#saveValue
   * @function
   * @returns {niagara.fieldEditors.BaseFieldEditor} this
   */
  BaseFieldEditor.prototype.saveValue = function saveValue(callbacks) {
    callbacks = callbackify(callbacks);
    saveWorkflow.invoke({
      editor: this,
      batch: callbacks.batch
    }, {
      ok: function (savedValue) {
        callbacks.ok(savedValue);
      },
      fail: function (err) {
        callbacks.fail(err);
      }
    });
  };
  
  /**
   * Adds a validation step to this field editor. A validation step consists
   * of a function that takes in two parameters; the first parameter is the
   * value to be validated (this is typically the direct output of
   * <code>getSaveData</code>) and the second is an object consisting of
   * ok/fail callbacks.
   * <p>
   * <code>addValidateStep</code> may also take in an array of these functions
   * - they will all be added.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#addValidateStep
   * @function
   * @param {Array|Function} step a validate step function, or an array of them
   */
  BaseFieldEditor.prototype.addValidateStep = function addValidateStep(step) {
    if ($.isArray(step)) {
      this.$validateSteps = this.$validateSteps.concat(step);
    } else {
      this.$validateSteps.push(step);
    }
  };
  
  /**
   * Validates the currently entered value before saving. The default behavior
   * is to accept all input as valid, but custom validation code can be
   * provided by passing a validate function into
   * <code>niagara.fieldEditors.defineEditor</code>.
   * 
   * @name niagara.fieldEditors.BaseFieldEditor#validate
   * @function
   * @param value the value to validate
   * @param {Object} callbacks an object with ok/fail callbacks (plus an
   * optional batch)
   * @throws {Error} if the currently entered value failed to validate
   */
  BaseFieldEditor.prototype.validate = function validate(callbacks) {
    callbacks = callbackify(callbacks);
    
    var that = this;
    
    validateWorkflow.invoke({
      editor: that,
      batch: callbacks.batch
    }, {
      ok: function (validated) {
        callbacks.ok(validated || that.value);
      },
      fail: callbacks.fail
    });
  };
  

  
////////////////////////////////////////////////////////////////
//Field Editor Registry
////////////////////////////////////////////////////////////////
  
  
  
  /**
   * @memberOf niagara.fieldEditors
   * @param component
   */
  function toSaveDataComponent(component) {
    var copy = component.newCopy(true);
    copy.editedSlots = {};
    
    copy.getSlots().properties().isComplex().each(function (slot) {
      var subComponent = copy.get(slot);
      copy.set({
        slot: slot,
        value: toSaveDataComponent(subComponent)
      });
    });
    
    util.aop.after(copy, 'set', function (args, value) {
      var obj = args[0];
      this.editedSlots[String(obj.slot)] = obj.value; 
      return value;
    });

    return copy;
  }
  
  /**
   * The default <code>doSaveValue</code> behavior used by a field editor when
   * a custom implementation of <code>doSaveValue</code> is not passed in to 
   * <code>defineEditor</code>. It handles a specific type of data passed out
   * of <code>getSaveData</code> - if you choose to allow the default
   * <code>doSaveValue</code> behavior, your implementation of 
   * <code>getSaveData</code> must return data in this format. This format will
   * be described below.
   * 
   * <ul>
   * <li>If editing a <code>baja:Simple</code>, simply return a new instance of
   * the edited type. For instance, if editing a <code>baja:Double</code>,
   * return <code>baja.Double.make(1.23)</code>.</li>
   * <li>If editing a <code>baja:Complex</code>, you have two choices.
   *   <ul>
   *     <li>Return an object literal where keys correspond to slot names and
   *         values are either <code>baja:Simples</code> (if the slot type
   *         is <code>baja:Simple</code>) or further object literals following
   *         the same format (if the slot type is <code>baja:Complex</code>).
   *         For example: <code>{ title: 'aString', subComponent: {
   *           title: 'aString' } }</code></li>
   *     <li>Or, pass your edited object into 
   *         <code>niagara.fieldEditors.toSaveDataComponent</code>, set its 
   *         slots directly, and return that. The advantage of this approach is
   *         that it functions as a sort of "preview" of the final saved object
   *         - <code>validate()</code> can examine it as if it were the actual
   *         edited object, already saved. You can also pass the output of 
   *         <code>getSaveData()</code> into any server side calls that expect
   *         a <code>BValue</code> - useful for performing server-side
   *         validation.</li>
   *   </ul>
   * </li>
   * </ul>
   * @memberOf niagara.fieldEditors
   * @private
   * @param {baja.Value} valueToSave the value being saved
   * @param {baja.Complex|Object} saveData the save data output from 
   * <code>getSaveData</code>
   * @param {Object} obj an object with ok/fail callbacks and parameters
   */
  function defaultDoSaveValue(valueToSave, saveData, obj) {
    var contexts = [],
        saveChangesToValueWorkflow;
    
    saveChangesToValueWorkflow = util.flow.parallel(
      function saveChangesToValueWorkflow(cx) {
        var doOK = this.ok,
            slot = valueToSave.getSlot(cx.slotName),
            slotValue = cx.slotValue;
//        if (slot.getType().isComponent()) {        
//          //we edited a slot of type BComponent - recurse down 
//          defaultDoSaveValue(valueToSave.get(slot), slotValue, this);
//        } else {
          //we edited a slot of type BSimple - just set it directly
          valueToSave.set({
            slot: slot,
            value: slotValue.newCopy(true),
            ok: function () {
              doOK(saveData);
            },
            fail: this.fail,
            batch: obj.batch
          });
//        }
      }
    );
    
    //saving a BSimple
    if (valueToSave.getType().isSimple()) {
      obj.ok(saveData);
      
    } else { //saving a BComplex
      if (saveData instanceof baja.Component && !saveData.editedSlots) {
        return obj.fail("provided raw instance of Component (type " +
            saveData.getType() + ") to defaultDoSaveValue - does your field " +
            "editor use niagara.fieldEditors.toSaveDataComponent()?");
      }
      
      //if using toSaveDataComponent, the saveData will have an editedSlots
      //property containing only those slots that were actually changed - 
      //otherwise, just treat saveData as an object literal
      baja.iterate(saveData.editedSlots || saveData, function (slotValue, slotName) {
        contexts.push({
          slotName: slotName,
          slotValue: slotValue
        });
      });
      
      saveChangesToValueWorkflow.invoke(contexts, {
        ok: function () {
          obj.ok(valueToSave);
        },
        fail: obj.fail
      });
    }
  }

  /**
   * The main method to define a field editor. This method takes in three
   * methods implementing the three primary behaviors of a field editor.
   * 
   * @memberOf niagara.fieldEditors
   * 
   * @param {Function} FieldEditor the field editor constructor we wish to
   * subclass, e.g. <code>niagara.fieldEditors.BaseFieldEditor</code> or
   * <code>niagara.fieldEditors.mobile.MobileFieldEditor</code>
   * 
   * @param {Object} params an object holding parameters
   * @param {Function} params.doInitializeDOM a function that will create the 
   * necessary HTML (returned as a jQuery div object), including such elements 
   * as text inputs, select menus, and checkboxes, that define the behavior of 
   * this field editor. This function takes two parameters:
   * <ul><li>A baja.Value (representing the object to be edited)</li>
   * <li>An object literal with <code>ok()</code> and <code>fail(err)</code>
   * callback functions (the object is guaranteed to have both of these
   * properties defined when it is passed to your function). It is up to your 
   * implementation of <code>doInitializeDOM</code> to call these functions as 
   * appropriate.</li></ul>
   * 
   * @param {Function} params.doLoadValue a method that will populate the HTML 
   * elements of the div created in the previous step with the given value. For 
   * instance, if this were a String editor, this function would take in a 
   * baja:String and simply set the value of its text input element. This 
   * function takes two parameters:
   * <ul><li>A baja.Value (representing the value to be loaded).</li>
   * <li>An object literal with <code>ok()</code> and <code>fail(err)</code> 
   * callback functions (the object is guaranteed to have both of these 
   * properties defined when it is passed to your function). It is up to your 
   * implementation of <code>doLoadValue</code> to call these ok/fail callbacks
   * as appropriate.</li></ul>
   * 
   * @param {Function} params.getSaveData a function that will retrieve the 
   * current values of the input elements of the editor div and assemble them 
   * into an object. The output of this function will be passed into both
   * <code>validate()</code> (if provided) and <code>doSaveValue</code>.
   * TODO: convert this to async
   * 
   * @param {Function} [params.doSaveValue] a function that will commit
   * any changes made to this field editor. If editing a <code>baja:Simple</code>,
   * it will simply build a new instance of the edited type using user-entered
   * data (edit-by-value semantics). If editing a <code>baja:Complex</code>, 
   * then user-entered data in the field editor will be applied directly to
   * the object itself (edit-by-reference semantics). If editing a 
   * subscribed/mounted component, an asynchronous call to the server must be
   * made to commit those changes as well.
   * 
   * <p>This function can be omitted entirely from your field editor definition.
   * If omitted, <code>defaultDoSaveValue</code> will be used, which should be
   * fine for most purposes. Please see the documentation for that function for
   * information about the expected output from <code>getSaveData()</code>.
   * 
   * <p>This function takes three parameters:
   * <ul><li>A baja.Value (representing the value being edited)</li>
   * <li>An object (of any type) - this is the output from 
   * <code>getSaveData</code></li>
   * <li>An object literal with <code>ok()</code> and <code>fail(err)</code> 
   * callback functions (the object is guaranteed to have both of these 
   * properties defined when it is passed to your function). It is up to your 
   * implementation of <code>doSaveValue</code> to call these ok/fail callbacks
   * as appropriate.</li></ul>
   * 
   * @param {Function} [params.postCreate] an optional function that will 
   * execute after the field editor's constructor completes. This function may
   * perform more specialized initialization on the field editor.
   * 
   * @param {Function} [params.validate] an optional function that performs
   * validation logic on this field editor. This function will <i>not</i> be
   * called automatically - if you wish to perform validation on a field editor
   * then <code>validate()</code> must be called before <code>saveValue</code>.
   * 
   * @param {Function} [params.armHandlers] an optional function that
   * will run after the editor builds up its HTML in <code>doInitializeDOM</code>.
   * This is the time to register any specialized event handlers. This function
   * accepts no parameters, but at the time it is executed <code>this.$dom</code>
   * will refer to a fully-constructed jQuery object.
   * <p>Please see <code>niagara.fieldEditors.BaseFieldEditor#armHandlers</code>
   * for information about what event handlers come pre-baked - you will not
   * need to replicate this functionality in your own implementation.
   * 
   * @see niagara.fieldEditors.BaseFieldEditor
   * @see niagara.fieldEditors.defaultDoSaveValue
   * 
   * @returns {Function} a constructor function for this field editor type. The
   * function returned from <code>defineEditor</code> must be instantiated 
   * using "new" to obtain an actual field editor object.
   */
  function defineEditor(FieldEditor, params) {
    params = baja.objectify(params);
    baja.strictAllArgs(
        [params.doInitializeDOM, params.doLoadValue, params.getSaveData], 
        [Function, Function, Function]);
    
    var aop = util.aop,
        feUtil = util.fieldEditors;
    
    function ctor() {
      return FieldEditor.apply(this, arguments);
    }
    
    ctor.prototype = $.extend({}, FieldEditor.prototype);
    
    ctor.prototype.doInitializeDOM = params.doInitializeDOM;
    ctor.prototype.doLoadValue = params.doLoadValue;
    ctor.prototype.doGetSaveData = params.getSaveData;
    ctor.prototype.doSaveValue = params.doSaveValue || defaultDoSaveValue;
    
    if (params.armHandlers) {
      aop.after(ctor.prototype, 'armHandlers', params.armHandlers);
    }
    
    if (params.validate) {
      aop.after(ctor.prototype, 'postCreate', function (args) {
        this.addValidateStep(params.validate);
      });
    }
    
    if (params.postCreate) {
      aop.after(ctor.prototype, 'postCreate', function (args) {
        params.postCreate.call(this);
      });
    }
    
    if (params.setReadonly) {
      aop.after(ctor.prototype, 'setReadonly', function (args) {
        params.setReadonly.apply(this, args);
      });
    }
    
    if (params.postInitializeDOM) {
      aop.before(ctor.prototype, 'postInitializeDOM', function (args) {
        var callbacks = args[0],
            that = this,
            oldOK = callbacks.ok;
       callbacks.ok = function () {
         params.postInitializeDOM.call(that, {
           ok: oldOK,
           fail: callbacks.fail
         });
       };
      });
    }
    
    if (params.postLoadValue) {
      aop.before(ctor.prototype, 'postLoadValue', function (args) {
        var callbacks = args[0],
            that = this,
            oldOK = callbacks.ok;
       callbacks.ok = function () {
         params.postLoadValue.call(that, {
           ok: oldOK,
           fail: callbacks.fail
         });
       };
      });
    }

    return ctor;
  }
  
  /**
   * Returns true if the given type is either a mobile or workbench field editor
   * type (extends <code>mobile:MobileFieldEditor</code> or 
   * <code>workbench:WbFieldEditor</code>).
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {String|Type} type
   */
  function isFieldEditorType(type) {
    if (!type) {
      return false;
    }
    
    try {
      type = baja.lt(type);
      if (type.is(MOBILE_FIELD_EDITOR_TYPE)) {
        return true;
      } else if (type.is(WORKBENCH_FIELD_EDITOR_TYPE)) {
        return true;
      } else {
        baja.error("fieldEditor facet " + type + " does not reference a " +
            "sub-Type of MobileFieldEditor.");
      }
    } catch (e) {
      //does have a fieldEditor facet, but the Type is unknown.
      //possible fat-finger
      baja.error("fieldEditor facet " + type + " does not reference a " +
          "known Type.");
    }
    
    return false;
  }
  
  
  /**
   * Checks to see if a field editor/retriever is registered for this particular
   * Baja value.
   * 
   * @memberOf niagara.fieldEditors
   * @param {baja.Value|Type} value the value to check for a registered editor.
   * If this is a <code>Type</code>, will only check for editors that have been 
   * explicitly registered through <code>niagara.fieldEditors.register</code> 
   * or that have a <code>mobile:MobileFieldEditor</code> declared as an agent 
   * on it. If this is a <code>baja.Value</code>, will first check using the
   * value's <code>Type</code>, and failing that, will check for a
   * <code>fieldEditor</code> facet declared on it.
   * 
   * @returns {Boolean} true if this type has been registered
   */
  function isRegistered(value) {
    var type = value.getType ? value.getType() : value,
        facets,
        feFacet;
    
    while (type) {
      //i've explicitly registered an editor using fieldEditors.register
      if (fieldEditorMap[String(type)]) {
        return true;
      }
      //i have a MobileFieldEditor declared as an agent on this type
      if (niagara.view.customFieldEditors[String(type)]) {
        return true;
      }
      type = type.getSuperType();
    }
    
    //no field editors registered - let's see if the value has a
    //fieldEditor facet on it
    facets = value.getFacets && value.getFacets();
    feFacet = facets && facets.get(FIELD_EDITOR_FACET);
    return isFieldEditorType(feFacet);
  }
  
  /**
   * Creates a URL to the given JS file ORD. This should only be a path to
   * a MobileFieldEditor JS file, not just any arbitrary file.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {String} js the direct path to the JS file
   * @param {String|Type} type the type of the 
   * <code>mobile:MobileFieldEditor</code> whose JS resources we're loading
   * @param {Array} an array of the type specs of all the types this
   * <code>mobile:MobileFieldEditor</code> is registered as an agent on.
   * @see niagara.fieldEditors.doPreloadFromSpec
   */
  function createJsPath(js, type, agentOn) {
    var path = '/ord/' + js;
    path += '|view:mobile:MobileFieldEditorView%3FfeTypeSpec=' + type + 
        ';typeSpecs=' + agentOn.join(',');
    return path;
  }
  
  /**
   * Performs the preload of all JS/CSS resources declared by the given
   * MobileFieldEditor type. These resources are declared in
   * <code>niagara.view.fieldEditorResources</code> which is itself populated
   * when the page is generated in 
   * <code>com.tridium.mobile.BDefaultMobileWebProfile</code>.
   * <p>
   * The requests for the JS files will be routed through the
   * <code>mobile:MobileFieldEditorView</code>.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {Type|String} type the <code>mobile:MobileFieldEditor</code> type
   * whose JS/CSS resources we need to load
   * @param {Object} callbacks an object containing ok/fail callbacks
   */
  function doPreloadFromSpec(type, callbacks) {
    type = String(type);
    
    var fer = niagara.view.fieldEditorResources,
        cfe = niagara.view.customFieldEditors,
        resources = fer && fer[type],
        jsResources = [],
        cssResources = [];
    
    //type is not a MobileFieldEditor, or it's been loaded already
    if (!resources) {
      return callbacks.ok();
    }

    baja.iterate(resources.js, function (js) {
      jsResources.push(createJsPath(js, type, resources.agentOn));
    });
    
    baja.iterate(resources.css, function (css) {
      cssResources.push('/ord?' + css);
    });

    lazyLoadWorkflow.invoke({
      js: jsResources,
      css: cssResources
    }, {
      ok: function () {
        //all loaded, we'll never have to load these again
        delete fer[type];
        delete cfe[type];
        callbacks.ok();
      },
      fail: callbacks.fail
    });
  }
  
  /**
   * Any particular type may have a MobileFieldEditor declared as an agent on
   * it. If this is the case, we should load the JS for that MobileFieldEditor
   * and use that JS to instantiate the field editor for that type. This
   * function will look at all the MobileFieldEditors declared as agents on the
   * given type, and perform the preloads (in parallel) for those 
   * MobileFieldEditors.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {Type|String} type the type of the object we're trying to 
   * instantiate a field editor for
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @see niagara.fieldEditors.doPreloadFromSpec
   */
  function doPreloadFromAgents(type, callbacks) {
    type = String(type);
    
    var cfe = niagara.view.customFieldEditors,
        fer = niagara.view.fieldEditorResources,
        customFETypes = cfe && cfe[type],
        invocations = [];

    //no agents declared on this type
    if (!customFETypes) {
      return callbacks.ok();
    }
    
    baja.iterate(customFETypes, function (feType) {
      invocations.push(function (callbacks) {
        doPreloadFromSpec(feType, callbacks);
      });
    });
    
    //preload all the field editors declared as agents on this type
    util.flow.runParallel(invocations, {
      ok: function () {
        //all agents loaded, no need to ever check this type again
        delete cfe[type];
        callbacks.ok();
      },
      fail: callbacks.fail
    });
  }
  
  /**
   * Performs the actual checking and loading of custom field editor code on the
   * given Type.
   * <p>
   * If the type has an entry in <code>niagara.view.customFieldEditors</code>,
   * then for each corresponding <code>BMobileFieldEditor</code> type spec in
   * <code>niagara.view.fieldEditorResources</code>, it will dynamically load
   * all declared Javascript code.
   * <p>
   * Once all the JS resources have been downloaded and executed, the entries
   * will be deleted from <code>niagara.view.customFieldEditors</code> and
   * <code>niagara.view.fieldEditorResources</code>, to prevent having to check
   * them a second time, and <code>callbacks.ok</code> will be called.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {Type} type the type to check for custom field editor code
   * @param {Object} callbacks an object with ok/fail callbacks
   */
  function doPreloadCustomFieldEditor(type, callbacks) {
    
    var cfe = niagara.view.customFieldEditors,
        fer = niagara.view.fieldEditorResources;
    
    if (type.is(MOBILE_FIELD_EDITOR_TYPE)) {
      doPreloadFromSpec(type, callbacks);
    } else {
      doPreloadFromAgents(type, callbacks);
    }
  }
  
  /**
   * Checks for any custom field editor code registered on the given Type via
   * <code>BMobileFieldEditor</code> instances. It does this by checking
   * <code>niagara.view.customFieldEditors</code>, which is built up server-side
   * in <code>BDefaultMobileWebProfile</code>.
   * <p>
   * Recursively checks the given type and all its super types, passing each
   * one in turn to <code>doPreloadCustomFieldEditor</code>.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * 
   * @param {Type} type the type for check for custom field editor code
   * @param {Object} callbacks an object containing ok/fail callbacks
   * @see niagara.fieldEditors.doPreloadCustomFieldEditor
   */
  function preloadCustomFieldEditor(type, callbacks) {
    if (type) {
      doPreloadCustomFieldEditor(type, {
        ok: function () {
          preloadCustomFieldEditor(type.getSuperType(), callbacks);
        },
        fail: callbacks.fail
      });
    } else {
      callbacks.ok();
    }
  }
  
  /**
   * Given a type, and optional key, returns the field editor constructor 
   * registered for that type. The key allows multiple different field editors
   * to be registered on a particular Type - e.g., the RelTime editor used when
   * selecting an override duration has a select dropdown appended and is
   * registered using the key 'override'.
   * 
   * <p>If there is no field editor registered for the type as given, this
   * method walks up the supertype chain until one is found. If none is found,
   * it defaults to a string-based field editor.
   * 
   * @private
   * @memberOf niagara.fieldEditors
   * 
   * @param {Type} type The type for which to find an editor/retriever
   * @param {String} [key] The String key under which the editor is registered.
   * If no key is given, <code>'default'</code> is used.
   * @returns {Function} a field editor constructor registered for this type
   */
  function doGet(type, key, callbacks) {
    var got;
    
    if (!type) {
      callbacks.fail("cannot get field editor for null type");
    }
    
    if (typeof type === 'string') {
      type = baja.lt(type);
    }
    
    key = key || 'default';

    preloadCustomFieldEditor(type, {
      ok: function () {
        while (!got && type) {
          got = fieldEditorMap[String(type)];
          if (got && got[key]) {
            return callbacks.ok(got[key]);
          } else {
            type = type.getSuperType();
          }
        }
        
        doGet('baja:String', 'default', callbacks);
      },
      fail: callbacks.fail
    });
  }
  
  /**
   * Register a field editor and retriever for a given type.
   * 
   * @memberOf niagara.fieldEditors
   * @param {Type|String} type The type to register
   * @param {Function} editor The field editor constructor for this type
   * @param {String} [key] An optional key for registering a special type of
   * editor for this Type. If no key is given, <code>'default'</code> is used.
   */
  function register(type, editor, key) {
    if (!type) {
      throw new Error("cannot register field editor for null type");
    }
    
    if (typeof editor !== 'function') {
      throw new Error("editor is required");
    }
    
    type = String(type);
    key = key || 'default';
    
    var obj = fieldEditorMap[type] || {};
    obj[key] = editor;
    fieldEditorMap[type] = obj;
    return editor;
  }
  
  function computeFacets(editedObject, container, slot, inpFacets) {
    var facets = baja.Facets.DEFAULT,
        myFacets;
    
    if (container && slot) {
      myFacets = util.slot.getFacets(container, slot);
    } else {
      myFacets = util.slot.getFacets(editedObject);
    }
    
    if (myFacets) {
      facets = baja.Facets.make(facets, myFacets);
    }
    if (inpFacets) {
      facets = baja.Facets.make(facets, inpFacets);
    }
    
    return facets;
  }
  
  /**
   * Instantiates a field editor for a Baja value.
   * 
   * <p>The <code>params</code> input object must have at least one of the
   * following: a <code>value</code> property, or <code>container</code> and
   * <code>slot</code> properties. If <code>value</code> is present, a
   * field editor will always be constructed for that value. If 
   * <code>value</code> is omitted, <code>container</code> and <code>slot</code>
   * <i>must</i> be present - <code>value</code> will then default to
   * <code>container.get(slot)</code>.
   * 
   * <p>If a container and slot are specified as properties of the
   * <code>params</code> object, then calling <code>saveValue</code> on this 
   * field editor will cause the edited object to be committed/saved onto the 
   * container. This is most useful when editing a <code>BSimple</code> 
   * property of a <code>BComplex</code>. If we omitted container and slot, we
   * would have to perform <code>component.set({...})</code> ourselves - by 
   * including container and slot, the field editor will do the work for us.
   * 
   * @memberOf niagara.fieldEditors
   * 
   * @param {Object} params
   * 
   * @param {baja.Value} params.value the object we are editing. If omitted,
   * will default to <code>params.container.get(params.slot)</code>.
   * 
   * @param {baja.Component} [params.container] the component containing the 
   * property we are editing. If omitted, and if <code>params.value</code> is a 
   * mounted component, will default to <code>editedObject.getParent()</code>.
   * 
   * @param {baja.Slot} [params.slot] the slot defining where the edited object 
   * lives - only used if a container is used as well. If omitted, and if
   * <code>params.value</code> is a mounted component, will default to
   * <code>params.value.getPropertyInParent()</code>.

   * @param {String} [params.key] a key defining a special type of field editor
   * we want to retrieve for this object - must have already been registered
   * using <code>niagara.fieldEditors.register()</code>
   * 
   * @param {niagara.fieldEditors.BaseFieldEditor} [params.parent] defines
   * this field editor as being a child editor of a parent editor. In this case,
   * calling <code>setModified()</code> on this editor will cause the
   * parent editor to be marked as modified as well.
   * 
   * @param {String|Type} [params.type] specifies the type spec we want a field
   * editor for. Only to be used in very special circumstances where we are
   * editing an object of one type, using a field editor ordinarily used for
   * another type. If omitted (and in most cases should be), will default to
   * <code>editedObject.getType()</code>. 
   * 
   * @param {baja.Facets|Object} [params.facets] a Facets object that can be
   * applied to the created field editor. Input Facets always take priority
   * over facets that are retrieved inside the field editor from 
   * <code>this.slot</code> etc, and can be used in any of your field editor's
   * functions by referencing <code>this.facets</code>. This can also be
   * an object literal.
   * 
   * @param {Boolean} [params.readonly] set to true if the field editor should
   * be created in readonly mode.
   * 
   * @param {Object} callbacks an object containing ok/fail callbacks
   * 
   * @param {Function} callbacks.ok a callback to execute once the field editor
   * has been fully instantiated. The field editor instance will be passed
   * as the first argument to this callback.
   * 
   * @returns {niagara.fieldEditors.BaseFieldEditor} an instance of a field 
   * editor for this object (to be passed to the ok callback)
   */
  function makeFor(params, callbacks) {

    params = baja.objectify(params, 'value');
    callbacks = callbackify(callbacks);
    
    if (params.value === undefined) {
      try {
        baja.strictArg(params.container, baja.Complex);
      } catch (e) {
        callbacks.fail("container is required when value is undefined");
      }
      params.value = params.container.get(params.slot);
    }

    var editedObject = params.value,
        type = params.type || editedObject.getType(),
        feType,
        FieldEditor, 
        container, 
        slot,
        key = params.key;
    
    //TODO: throw expected errors for undefined
    container = params.container || 
      (editedObject.getParent && editedObject.getParent());
    slot = params.slot ||
      (editedObject.getPropertyInParent && editedObject.getPropertyInParent());
    
    params.facets = computeFacets(editedObject, container, slot, params.facets);
    
    //check to see if the component has a fieldEditor facet expressly registered
    feType = params.facets.get(FIELD_EDITOR_FACET);
    if (isFieldEditorType(feType) && isRegistered(baja.lt(feType))) {
      //hey, we do. load this kind of field editor instead. it will be
      //registered as an editor on the actual MobileFieldEditor subtype,
      //so use this type to retrieve the field editor instance but still load 
      //the old value into it
      type = baja.lt(feType);
    }
    
    doGet(type, key, {
      ok: function (FieldEditor) {
        callbacks.ok(new FieldEditor(editedObject, container, slot, params));
      },
      fail: callbacks.fail || baja.error
    });
  }
  
  
  ////////////////////////////////////////////////////////////////
  // Composite field editors
  ////////////////////////////////////////////////////////////////
  
  (function compositeFunctions() {
    var compositeInitializeDOMWorkflow,
        compositeLoadValueWorkflow,
        compositeGetSaveDataWorkflow;
    
    compositeInitializeDOMWorkflow = util.flow.sequentialParallel(
      function compositeInitializeDOMWorkflow__step1__makeFor(cx) {
        //cx includes container, slot, key, and parent params
        makeFor(cx, this);
      },
      function compositeInitializeDOMWorkflow__step2__initializeDOM(cx, editor) {
        cx.editor = editor;
        cx.editor.initializeDOM(toLabeledEditorContainer(cx.editor, cx.targetElement), this);
      },
      function compositeInitializeDOMWorkflow__step3__addToSubEditorMap(cx) {
        cx.parent.subEditorMap[cx.slot] = cx.editor;
        this.ok();
      }
    );
    
    compositeLoadValueWorkflow = util.flow.parallel(
      function compositeLoadValueWorkflow(cx) {
        cx.editor.loadValue(cx.value, this);
      }
    );
    
    compositeGetSaveDataWorkflow = util.flow.sequentialParallel(
      function compositeGetSaveDataWorkflow__step1__getSaveData(cx) {
        cx.editor.getSaveData(this);
      },
      function compositeGetSaveDataWorkflow__step2__setValue(cx, saveData) {
        cx.compositeSaveData.set($.extend(this, { slot: cx.slot, value: saveData }));
      }
    );
    
    function allSlots(editedObject) {
      var obj = {};
      editedObject.getSlots().properties().each(function (slot) {
        if (util.slot.isEditable(slot)) {
          obj[String(slot)] = 'default';
        }
      });
      return obj;
    }
    
    /**
     * Creates a method to build a composite editor div by aggregating one or 
     * more editor divs into one.
     * 
     * @private
     * @memberOf niagara.fieldEditors.composite
     * @see niagara.fieldEditors.BaseFieldEditor#initializeDOM
     * 
     * @param {Object} slots a mapping of slot names to field editor keys,
     * defining what types of field editors we want for our properties
     * @returns {Function} a function to build the composite div
     */
    function compositeInitializeDOM(slots) {
      return function (targetElement, callbacks) {
        var that = this,
            container = that.value,
            contexts = [];
        
        baja.iterate(slots || allSlots(container), function (feObj, slotName) {
          contexts.push({
            key: feObj.key,
            value: container.get(slotName),
            slot: container.getSlot(slotName),
            container: container,
            parent: that,
            facets: that.facets,
            targetElement: $('<div class="subEditor"/>').appendTo(targetElement)
          });
        });
        
        compositeInitializeDOMWorkflow.invoke(contexts, callbacks);
      };
    }
    
    /**
     * Sets the values in all the individual field editors in our composite
     * editor.
     * 
     * @private
     * @memberOf niagara.fieldEditors.composite
     * @see niagara.fieldEditors.BaseFieldEditor#loadValue
     * 
     * @param {Object} slots a mapping of slot names to field editor keys,
     * defining what types of field editors we want for our properties
     * @returns {Function} a setter function that takes in a baja.Complex, pulls
     * out that Complex's properties using the given slot names, and calls
     * loadValue on each of our sub-field editors.
     */
    function compositeLoadValue(slots) {
      return function (value, callbacks) {
        var that = this,
            contexts = [],
            subEditorMap = that.subEditorMap;
        
        if (value === null || value === undefined) {
          return callbacks.ok();
        }
        
        baja.iterate(slots || allSlots(value), function (feObj, slotName) {
          contexts.push({
            editor: subEditorMap[slotName],
            value: feObj.defaultValue || value.get(slotName)
          });
        });
        
        compositeLoadValueWorkflow.invoke(contexts, callbacks);
      };
    }
    
    /**
     * Retrieves the values from all of our individual field editors and
     * assembles them into an instance of our desired object type.
     * 
     * @private
     * @memberOf niagara.fieldEditors.composite
     * @see niagara.fieldEditors.BaseFieldEditor#getSaveData
     * 
     * @param {Object} obj a mapping of slot names to field editor keys,
     * defining what types of field editors we want for our properties
     * @returns {Function} a retriever function that will assemble our
     * individual field editor values into our desired composite type.
     */
    function compositeGetSaveData(slots) {
      return function (callbacks) {
        var subEditorMap = this.subEditorMap,
            contexts = [],
            saveData = toSaveDataComponent(this.value);

        baja.iterate(slots || allSlots(this.value), function (feObj, slotName) {
          var editor = subEditorMap[slotName];
          //if (editor.isModified()) {
            contexts.push({
              editor: editor,
              slot: slotName,
              compositeSaveData: saveData
            });
          //}
        });
        
        compositeGetSaveDataWorkflow.invoke(contexts, {
          ok: function () {
            callbacks.ok(saveData);
          },
          fail: callbacks.fail
        });
      };
    }
    
    /**
     * Arms change listeners on a composite field editor by individually
     * arming change listeners on all sub-field editors.
     * 
     * @private
     * @memberOf niagara.fieldEditors.composite
     * @see niagara.fieldEditors.BaseFieldEditor#armHandlers
     * 
     * @param {Object} obj a mapping of slot names to field editor keys,
     * defining what types of field editors we want for our properties
     * @returns {Function} a function that will arm change handlers on all
     * our sub-field editors.
     */
    function compositeArmHandlers(obj) {
      return function () {
        var subEditorMap = this.subEditorMap;

        baja.iterate(obj || allSlots(this.value), function (value, slotName) {
          var editor = subEditorMap[slotName];
          editor.armHandlers();
        });
      };
    }
    
    /**
     * Sets the readonly status of all sub-field editors.
     * 
     * @private
     * @memberOf niagara.fieldEditors.composite
     * @see niagara.fieldEditors.BaseFieldEditor#setReadonly
     * 
     * @param {Object} obj a mapping of slot names to field editor keys,
     * defining what types of field editors we want for our properties
     * @returns {Function} a function that will set readonly status on all
     * our sub-field editors.
     */
    function compositeSetReadonly(obj) {
      return function (readonly) {
        var subEditorMap = this.subEditorMap;
        baja.iterate(obj || allSlots(this.value), function (value, slotName) {
          var editor = subEditorMap[slotName];
          editor.setReadonly(readonly);
        });
      };
    }
    
    /**
     * Validates a composite field editor by sequentially calling the
     * validate() method of all of its sub-field editors. Returns a validate
     * step to be passed directly to <code>addValidateStep</code> during
     * the construction of the composite field editor.
     * 
     * @private
     * @memberOf niagara.fieldEditors.composite
     * @see niagara.fieldEditors.BaseFieldEditor#validate
     * 
     * @param {Object} obj a mapping of slot names to field editor keys,
     * defining what types of field editors we want for our properties
     * @returns {Function} a function that will set validate all of our
     * sub-field editors.
     */
    function compositeValidateStep(obj) {
      return function (saveData, callbacks) {
        var validateSteps = [],
            subEditorMap = this.subEditorMap;
        
        baja.iterate(obj || allSlots(this.value), function (value, slotName) {
          var editor = subEditorMap[slotName];
          validateSteps.push(function (callbacks) {
            editor.validate(callbacks);
          });
        });
        
        util.flow.runSequential(validateSteps, callbacks);
      };
    }
    
    /**
     * Creates a composite field editor. This field editor essentially breaks
     * our edited object down into individual properties, creates a field
     * editor for each one (choosing automatically depending on each property's
     * type), and assembles them into one big field editor.
     * 
     * <p>For instance, the composite editor for control:EnumOverride is
     * simply defined by two slot names: duration and value. Since these two
     * properties of an EnumOverride are Numeric and Enum respectively, the
     * field editor created will contain two editor divs, one for the numeric
     * and one for the enum, and upon saving will assemble those editor's values
     * into an EnumOverride object.
     * 
     * @memberOf niagara.fieldEditors.composite
     * 
     * @param {Array} names the slot names of the properties we wish to show
     * for this editor
     * 
     * @param {Function} [FieldEditor] the field editor constructor to subclass
     * - if omitted, defaults to BaseFieldEditor
     * 
     * @returns {Function} a field editor constructor
     */
    function makeComposite(names, FieldEditor, params) {
      names = names || [];
      baja.strictArg(names, Array);
      var slots;
      
      if (names.length) {
        slots = {};
        baja.iterate(names, function (val) {
          if (typeof val === 'string') {
            slots[val] = {
              key: 'default'
            };
          } else {
            slots[val.slot] = {
              key: val.key || 'default',
              defaultValue: val.defaultValue
            };
          }
        });
      }
      
      return defineEditor(FieldEditor || BaseFieldEditor, {
        doInitializeDOM: compositeInitializeDOM(slots), 
        doLoadValue: compositeLoadValue(slots), 
        getSaveData: compositeGetSaveData(slots),
//        armHandlers: compositeArmHandlers(slots),
        setReadonly: compositeSetReadonly(slots),
        validate: compositeValidateStep(slots),
        postCreate: function () {
          this.subEditorMap = {};
          if (params && params.validate) {
            this.addValidateStep(params.validate);
          }
        }
      });
    }
    
    /**
     * Returns a field editor to simply perform default editing of all
     * property slots on a BComplex.
     * 
     * @name niagara.fieldEditors.composite.allSlots
     * @function
     * @returns {Function} a new field editor constructor
     */
    function defineAllSlots() {
      var AllSlots = makeComposite();
      AllSlots.prototype.postInitializeDOM = function (callbacks) {
        this.$dom.addClass('allSlots');
        callbacks.ok();
      };
      return AllSlots;
    }
    
    /**
     * A namespace containing functions for creating composite field editors.
     * 
     * @namespace
     * @name niagara.fieldEditors.composite
     */
    util.api(composite, {
      'public': {
        allSlots: defineAllSlots,
        makeComposite: makeComposite
      },
      'private': {
        compositeArmHandlers: compositeArmHandlers,
        compositeInitializeDOM: compositeInitializeDOM,
        compositeGetSaveData: compositeGetSaveData,
        compositeLoadValue: compositeLoadValue
      }
    });
  }());
  
  /**
   * @namespace
   * @name niagara.fieldEditors
   */
  util.api('niagara.fieldEditors', {
    'public': {
      BaseFieldEditor: BaseFieldEditor,
      
      composite: composite,
      
      defineEditor: defineEditor,
      isRegistered: isRegistered,
      makeFor: makeFor,
      register: register,
      toLabeledEditorContainer: toLabeledEditorContainer,
      toSaveDataComponent: toSaveDataComponent
    },
    'private': {
      createJsPath: createJsPath,
      doPreloadCustomFieldEditor: doPreloadCustomFieldEditor,
      doPreloadFromAgents: doPreloadFromAgents,
      doPreloadFromSpec: doPreloadFromSpec,
      preloadCustomFieldEditor: preloadCustomFieldEditor,
      toFakeBajaObject: toFakeBajaObject
    }
  });
}());