/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Basic Web App Container View.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

/*jslint white: true, browser: true */
/*global $, baja, niagara */ 

(function desktop(pages) {
  // Use ECMAScript 5 Strict Mode
  "use strict";
  
  niagara.util.require(
    '$.mobile',
    'niagara.util.mobile.ListView',
    'niagara.util.mobile.ListPageView',
    'niagara.util.mobile.PageViewManager',
    'niagara.util.mobile.commands',
    'niagara.util.mobile.views'
  );

  var util = niagara.util,
      commands = util.mobile.commands,
      views = util.mobile.views,
      
      ListView = util.mobile.ListView,
      PageView = util.mobile.PageView,
      PageViewManager = util.mobile.PageViewManager,

      AppsView,
      AppsListView,
      AppsViewManager,
      
      appsMgr;
        
////////////////////////////////////////////////////////////////
//Util
////////////////////////////////////////////////////////////////

  /**
   * Default ok callback handler.
   *
   * @private
   */
  function defOk() {    
    $.mobile.hidePageLoadingMsg();
  }

  /**
   * Default fail callback handler.
   *
   * @private
   */
  function defFail(err) {
    baja.error(err);

    // Hide any loading title      
    $.mobile.hidePageLoadingMsg();

    // Load error page     
    $("<a href='#error' data-rel='dialog' data-transition='flip' />").click();  
  }
  
  /**
   * A component view to display apps and their corresponding shiny buttons on
   * a user's desktop screen.
   * 
   * @class
   * @memberOf niagara.desktop
   * @extends niagara.util.mobile.ListView
   * @private
   */
  AppsListView = ListView.subclass('subscribable');
  
  AppsListView.prototype.highlight = util.noop;
  
  util.aop.advisePrototype(AppsListView, {
    after: {
      /**
       * After building a list item for an app, check to see if the view type 
       * spec for the app matches the current view 
       * (<code>MobileDesktopView</code>). If not, set 
       * <code>data-ajax="false"</code> on the link to that app to trigger
       * a full page refresh to the app.
       * 
       * @name niagara.desktop.AppsView#makeListItem
       * @function
       */
      makeListItem: function (args, li) {
        var component = args[0],
            slot = args[1],
            appDisplayInfo = component.get(slot), 
            a = li.children('a');

        a.children('img').removeClass('ui-li-icon');
        a.jqmData('ord', baja.Ord.make({
          base: "station:",
          child: appDisplayInfo.get("appSlotPath")
        }));

        return li;
      },
      
      /**
       * When subscribing a <code>NiagaraMobileWebApps</code> component, listen
       * for the <code>appsModified</code> topic to be fired. When it is fired,
       * refresh the view to display any added/removed apps.
       * 
       * @name niagara.desktop.AppsView#makeSubscriber
       * @function
       */
      makeSubscriber: function (args, sub) {
        sub.attach("topicFired", function (topic) {
          // If appModified event is fired then simply refresh the view
          if (topic.getName() === "appsModified") {
            appsMgr.refreshAll();
          }
        });
        return sub;
      }
    }
  });
  
  /**
   * A list item for an app should have the app display name and icon set.
   * App icons are always linkable and never expandable.
   * 
   * @name niagara.desktop.AppsView#makeListItemParams
   * @function
   */
  AppsListView.prototype.makeListItemParams = function (component, prop) {
    var appDisplayInfo = component.get(prop);
    return {
      title: '',
      display: appDisplayInfo.get("appDisplayName"),
      icon: appDisplayInfo.get("appDisplayIcon"),
      expandable: false,
      linkable: true
    };
  };
  
  /**
   * <code>AppsView</code> has a specialized implementation of
   * <code>makeListView</code> that performs a server side call to retrieve app
   * display information (<code>DesktopServerSideCallHandlers.getAppDisplayInfo()</code>).
   * The output of this server side call - not the 
   * <code>NiagaraMobileWebApps</code> component itself - will be passed into
   * <code>makeListItem</code> to build up the display.
   * 
   * @name niagara.desktop.AppsView#makeListView
   * @function
   */
  AppsListView.prototype.makeListView = function (component, callbacks) {
    var ul = $('<ul />'), //no data-role="listview" - JQM should not enhance
        that = this;
    
    // Make a Server Side Call to get the information implemented by BIAppDisplay
    // (tried to do this with BQL but it became too complex so a Server Side Call is easiest)
    component.serverSideCall({
      typeSpec: "mobile:DesktopServerSideCallHandler",
      methodName: "getAppDisplayInfo",
      ok: function (result) {
        result.getSlots().each(function (slot) {
          var li = that.makeListItem(result, slot);
          ul.append(li);
        });
        callbacks.ok(ul);
      },
      fail: callbacks.fail
    }); 
  };
  
  /**
   * A component view to display apps and their corresponding shiny buttons on
   * a user's desktop screen.
   * 
   * @class
   * @memberOf niagara.desktop
   * @extends niagara.util.mobile.ComponentView
   * @private
   */
  AppsView = PageView.subclass('subscribable');
  AppsView.prototype.instantiateContentView = function () {
    return new AppsListView();
  };
  
  util.aop.advisePrototype(AppsView, {
    after: {      
      /**
       * Appends a view button to the page's header.
       * 
       * @name niagara.desktop.AppsView#createPage
       * @function
       */
      createPage: function (args, page) {
        var headerDiv = page.children(':jqmData(role=header)');
        headerDiv.append(commands.getCommandsButton().click(commands.showCommandsHandler));
        return page;
      }
    }
  });
  
  AppsViewManager = function() {
    PageViewManager.call(this);
  };
  AppsViewManager.prototype = new PageViewManager();
  
  AppsViewManager.prototype.instantiateView = function instantiateView(component, pageData, callbacks) {
    callbacks.ok(new AppsView());
  };

  /**
   * Registers Pages handlers and instigates a page change to display the
   * current <code>AppView</code>. Will be called upon <code>baja.start</code>.
   * @name niagara.desktop.initializeUI
   * @function
   * @private
   */
  function initializeUI() {
    appsMgr = new AppsViewManager();
    appsMgr.registerPages();
    
    function resetToViewOrd() {
      util.mobile.linkToOrdInternal(niagara.view.ord, { 
        transition: "none", 
        changeHash: false 
      });
    }

    pages.register('splash', {
      pageshow: resetToViewOrd
    });
    
    $("a.commandsButton").live("onShowCommands", function () {
      views.getViewsCommand().setOrd(appsMgr.selectedView.value.getNavOrd());
    });
  }
  
  initializeUI();

  /**
   * @namespace
   * @name niagara.desktop
   * @private
   */
  util.api('niagara.desktop', {
    'private': {
      AppsView: AppsView,
      initializeUI: initializeUI
    }
  });
}(niagara.util.mobile.pages));
