/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Niagara Mobile Alarm Console App.
 *
 * @author Gareth Johnson
 * @version 0.0.1.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, bitwise: true, regexp: false, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, baja, location, niagara, window, setTimeout, console*/ 

(function alarmApp(pages) {
  // Use ECMAScript 5 Strict Mode
  "use strict";
  
  niagara.util.require(
    '$.mobile',
    'niagara.util.mobile.dialogs',
    'niagara.util.mobile.views'
  );
        
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  var util = niagara.util,
      views = util.mobile.views,
      dialogs = util.mobile.dialogs,
      commands = util.mobile.commands,
      escapeHtml = util.mobile.escapeHtml,
      consoleDisplayName,
      model; // The remote Alarm Console Model the App uses for modelling the Console.

////////////////////////////////////////////////////////////////
// DOM Elements
////////////////////////////////////////////////////////////////  

  var checkboxHtml = 
    '<div data-role="fieldcontain">' +
      '<fieldset data-role="controlgroup" data-type="horizontal">' +
        '<input type="checkbox" id="{checkboxId}"/>' +
        '<label for="{checkboxId}"><span class="ui-btn-icon-notext"><span class="ui-icon"></span></span></label>' +
      '</fieldset>' +
    '</div>';
  
  var domRowPrototype = 
    '<li class="alarmRow">' +
      '<a class="alarmSelectLink">' +
        '<div class="checkboxContainer">' +
          '{checkboxHtml}' +
        '</div>' +
        '<div class="detailsContainer">' +
          '<table class="alarmFields">' +
          '</table>' +   
          '<div class="ackStateSummary"></div>' + 
        '</div>' +
      '</a>' +
      '<a href="#" data-transition="slide" data-split-icon="right" class="alarmLink"></a>' + 
    '</li>';
      
////////////////////////////////////////////////////////////////
// Util
////////////////////////////////////////////////////////////////

  /**
   * Default ok callback handler.
   *
   * @private
   */
  function defOk() {    
    $.mobile.hidePageLoadingMsg();
  }
 
  /**
   * Default fail callback handler.
   *
   * @private
   */
  function defFail(err) {
    baja.error(err);

    // Hide any loading title      
    $.mobile.hidePageLoadingMsg();
     
    // Load error page     
    $("<a href='#error' data-rel='dialog' data-transition='flip' />").click();  
  }
  
  function updateInsetStatus(ul) {
    if ($(window).width() > 320) {
      util.mobile.applyListviewInset(ul);
    } else {
      util.mobile.removeListviewInset(ul);
    }
  }
      
////////////////////////////////////////////////////////////////
// Notes
//////////////////////////////////////////////////////////////// 
  
  function addNotes(alarmInfo, notesString) {
    $.mobile.showPageLoadingMsg();
          
    // Add the user's notes
    alarmInfo.add({
      slot: "notes", 
      value: notesString
    });
        
    // Make the network call to add the notes
    var batch = new baja.comm.Batch();
    
    // Add the Notes to the alarms
    model.getParent().serverSideCall({
      typeSpec: "mobile:MobileAlarmServerSideCallHandler",
      methodName: "addNotes",
      value: alarmInfo, 
      fail: defFail, 
      batch: batch
    });
        
    // Perform a poll
    baja.station.sync({
      ok: function () {
        defOk();
      },   
      fail: function (err) {
        defFail(err);
      },   
      batch: batch
    });
    
    batch.commit();     
  }
  
  function showNotesDialog(alarmInfo) {
    var lex = baja.lex('mobile'),
        p = $('<p/>').text(lex.get('alarm.askAddNotes')),
        textarea = $('<textarea cols="40" rows="8" name="textarea" id="alarmConsoleNotes"/>'),
        fieldDiv = $('<div data-role="fieldcontain"/>').append(textarea);
    
    dialogs.okCancel({
      title: lex.get('alarm.addNotes'),
      content: function (elem, callbacks) {
        elem.append(p).append(fieldDiv);
        textarea.textinput();
        callbacks.ok();
      },
      ok: function (cb) {
        var notesString = textarea.val();
        
        if (notesString) {
          //ensure newline cos station doesn't
          addNotes(alarmInfo, notesString.replace(/\n*$/, '\n\n')); 
        }
        
        cb.ok();
      }
    });
  }

      
////////////////////////////////////////////////////////////////
// Alarm Field Selection
//////////////////////////////////////////////////////////////// 
  
  /**
   * @class Page for choosing fields to show in the App.
   *
   * @name niagara.alarm.console.mobile.ChooseFields
   * @inner
   */ 
  pages.register("chooseFields", (function pageChooseFields() {
    var defaultFields = ["timestamp", "value", "source", "alarmData.alarmValue"],
        fieldsArray = null;
    
    /**
     * Return an array of alarm fields.
     *
     * @name niagara.alarm.console.mobile.ChooseFields#getFields
     * @function
     */
    function getFields() {
      if (fieldsArray) {
        return fieldsArray;
      }
      
      // Lazily create alarm fields array
      fieldsArray = [];
     
      // If available, attempt to load columns from web storage
      if (window && window.localStorage) {
        try {
          var storedFields = window.localStorage.getItem(baja.getUserName() + "-NiagaraAlarmConsoleFields");
          if (storedFields) {
            defaultFields = JSON.parse(storedFields);
          }
        }
        catch (ignore) {}
      }
      
      var rec = baja.$("alarm:AlarmRecord");
    
      // Scan all Properties from alarm record...
      rec.getSlots().properties().each(function (prop) {
        fieldsArray.push({
          fieldName: prop.getName(),
          fieldDisplayName: this.getDisplayName(prop),
          selected: defaultFields.contains(prop.getName())       
        });
      });    
      
      // Scan for any alarm data fields...
      var i;
      for (i = 0; i < defaultFields.length; ++i) {
        var alarmDataIndex = defaultFields[i].indexOf("alarmData.");
        if (alarmDataIndex === 0) {
          fieldsArray.push({
            fieldName: defaultFields[i],
            fieldDisplayName: defaultFields[i],
            selected: true       
          });
        }
      }
      
      return fieldsArray;      
    }
    
    /**
     * Return a new field input for field selection.
     *
     * @name niagara.alarm.console.mobile.ChooseFields-createFieldInput
     * @function
     * @private
     *
     * @param {Object} field the field object to create the UI from.
     */    
    function createFieldInput(field) {
      // Add a check box and a label for the field
      var inp = $('<input type="checkbox" name="fieldRow" data-theme="c">');
      inp.attr('id', field.fieldName).append('</input>');
         
      if (field.selected) {
        inp.attr('checked', 'checked');
      }
                                         
      var label = $('<label class="chooseFieldRow">').attr('for', field.fieldName)
                              .text(field.fieldDisplayName)
                              .append('</label>');
                                                                                                                                                                    
      // Append to the Control Group
      $("#chooseFieldGroup").append(inp, label);
      
      // Associate the field as data onto the DOM element
      $(label).data("fieldData", field);
    }
    
    /**
     * Called once the user has made their selection and the UI needs to update
     * with the new field selection.     
     *
     * @name niagara.alarm.console.mobile.ChooseFields-updateFields
     * @function
     * @private
     */     
    function updateFields() { 
      var f = getFields();
      
      // Update the fields array for each checkbox that's selected
      $("#chooseFieldGroup .chooseFieldRow").each(function () {
        var i;
        for (i = 0; i < f.length; ++i) {
          if (f[i].fieldName === $(this).data("fieldData").fieldName) {
            f[i].selected = $(this).hasClass("ui-checkbox-on");
          }
        }
      });
          
      // Save settings to local storage
      if (window && window.localStorage) {
        var storedFields = [], i;
        for (i = 0; i < f.length; ++i) {
          if (f[i].selected) {
            storedFields.push(f[i].fieldName);
          }
        }
      
        try {
          window.localStorage.setItem(baja.getUserName() + "-NiagaraAlarmConsoleFields", JSON.stringify(storedFields));
        }
        catch (ignore) {}
      }
                            
      // Go back
      $("<a href='#' data-rel='back'/>").click();
    }
    
    /**
     * Called to add a new alarm data field to the choose columns Page.    
     *
     * @name niagara.alarm.console.mobile.ChooseFields-addAlarmDataField
     * @function
     * @private
     */
    function addAlarmDataField() {  
      var f = getFields();
      // Add an alarm data field
     
      // Get the name of the alarm data field
      var data = $("#addAlarmDataInput").attr("value");
      if (data === "") {
        return;
      }
     
      data = "alarmData." + data;
      // See if the alarm data field is already present
      var i;
      for (i = 0; i < f.length; ++i) {
        if (f[i].fieldName === data) {
          return;
        }
      }
     
      // Add the alarm data column to the fields
      var newField = {
        fieldName: data,
        fieldDisplayName: data,
        selected: false       
      };
      
      f.push(newField);
      createFieldInput(newField);
     
      // Update the user interface
      $(".ui-page").trigger("create");
    }
    
    /**
     * pagebeforecreate Pages event handler.
     *
     * @name niagara.alarm.console.mobile.ChooseFields#pagebeforecreate
     * @function
     */     
    function pagebeforecreate() {
      $("#chooseFieldsAddFieldButton").click(addAlarmDataField);
      $("#chooseFieldsOkButton").click(updateFields); 
    }
    
    /**
     * pagebeforeshow Pages event handler.
     *
     * @name niagara.alarm.console.mobile.ChooseFields#pagebeforeshow
     * @function
     */    
    function pagebeforeshow() {
      var f = getFields(),
          i;
    
      $("#chooseFieldGroup .ui-checkbox").remove();
      for (i = 0; i < f.length; ++i) {
        createFieldInput(f[i]);
      } 
      
      $(".ui-page").trigger("create");
    }
     
    // Functions exported as public     
    return {
      pagebeforecreate: pagebeforecreate,
      pagebeforeshow: pagebeforeshow,
      getFields: getFields
    };
  }())); 
  
////////////////////////////////////////////////////////////////
// Alarm Acknowledgement
////////////////////////////////////////////////////////////////   
  
  /**
   * Acknowledge the selected alarms.
   *
   * @name niagara.alarm.console.mobile.ackSelectedAlarms
   * @function
   */ 
  function ackSelectedAlarms() {
    $.mobile.showPageLoadingMsg();
        
    if (!pages.getCurrentHandler().hasSelected()) {
      $.mobile.hidePageLoadingMsg();
      dialogs.error(baja.lex('mobile').get('alarm.selectAlarms'));
      return;
    }    
   
    // Request the alarm information from the currently selected page   
    var info = pages.getCurrentHandler().getSelectedAlarmInfo(),
        batch = new baja.comm.Batch();
                
    // Ack the alarms with the selected alarms sources
    model.getParent().serverSideCall({
      typeSpec: "mobile:MobileAlarmServerSideCallHandler",
      methodName: "ackAlarms",
      value: info, 
      fail: defFail, 
      batch: batch
    });
    
    // Perform a poll
    baja.station.sync({
      ok: function () {
        defOk();
        pages.fireAll("alarmsAcked");
      },   
      fail: defFail,   
      batch: batch
    });
    
    batch.commit();
  } 
  
////////////////////////////////////////////////////////////////
// Force Clear
////////////////////////////////////////////////////////////////   
  
  /**
   * Force clear the selected alarms.
   *
   * @name niagara.alarm.console.mobile.forceClear
   * @function
   */ 
  function forceClear() {    
    // Bail if nothing is selected
    if (!pages.getCurrentHandler().hasSelected()) {
      dialogs.error(baja.lex('mobile').get('alarm.selectAlarms'));
      return;
    }

    // Confirm with the user that alarms should be force cleared
    dialogs.yesNo({
      title: baja.lex("mobile").get("alarm.forceClear"),
      content: baja.lex("mobile").get("alarm.forceClearWarning"),
      yes: function (callbacks) {       
        $.mobile.showPageLoadingMsg();
       
        // Request the alarm information from the currently selected page   
        var info = pages.getCurrentHandler().getSelectedAlarmInfo(),
            batch = new baja.comm.Batch();
        
        // Force clear the alarms
        model.getParent().serverSideCall({
          typeSpec: "mobile:MobileAlarmServerSideCallHandler",
          methodName: "forceClear",
          value: info, 
          fail: defFail, 
          batch: batch
        });
        
        // Once the alarms have been cleared, signal this has happened
        batch.addCallback(function () {
          pages.fireAll("alarmsCleared");
        });
        
        batch.commit();
        callbacks.ok();
      }
    });
  }   
    
////////////////////////////////////////////////////////////////
// Row Formatting
////////////////////////////////////////////////////////////////

  /**
   * Create a string value for a given field to be shown in an alarm row.
   *
   * @name niagara.alarm.console.mobile-createFieldString
   * @function
   * @private
   */
  function createFieldString(alarm, fieldName, fieldData) {
    if (fieldName === "alarmData.notes" || fieldName === "alarmData.instructions") {
      // Ensure the alarm notes and instructions appear with line breaks
      fieldData = fieldData.replace(/[\n]/g, "<br />");
    }
    else if (fieldName === "source") {
      // If the source is being asked for then return the source name if present within the data facets
      fieldData = alarm.get("alarmData").get("sourceName", fieldData);
    }
     
    if (fieldData.getType().isNumber()) {
      // Ensure that any numbers are to two decimal places
      fieldData = fieldData.valueOf().toFixed(2);
    }
    else if (fieldData.getType().is("baja:AbsTime")) {
      // Ensure date and time formatting looks ok
      fieldData = fieldData.getDate().toString() + " - " + fieldData.getTime().toString();
    }
    else if (fieldData.getType().is("baja:String")) {
      // Pass any string data through a Format
      fieldData = escapeHtml(baja.Format.format({
        object: alarm,
        pattern: fieldData
      }));
    }
    else if (fieldData.getType().is("baja:Facets")) {
      // Ensure the alarm data facets appears nicely
      var dataTable = "<table>",
          keys = fieldData.getKeys(),
          i;
      for (i = 0; i < keys.length; ++i) {
        dataTable += "<tr><td>" + keys[i] + "</td><td>" + createFieldString(alarm, fieldName, fieldData.get(keys[i])) + "</td></tr>";
      }
      dataTable += "</table>";
      
      fieldData = dataTable;
    }
        
    return fieldData.toString();
  }
  
  /**
   * Return the name of the alarm image to use for a given alarm.
   *
   * @name niagara.alarm.console.mobile-getAlarmImg
   * @function
   * @private
   *
   * @param {Object} alarm the alarm record.
   * @return {String} the name of the alarm image to use.
   */
  function getAlarmImg(alarm) {
    var sourceState = alarm.get("sourceState"),
        ackState = alarm.get("ackState");
  
    // Figure out which alarm icon to use
    var img = "alarm.png";
    if (sourceState.is("alert") && !ackState.is("acked")) {
      img = "alarmOrange.png";
    }
    else if (sourceState.is("alert") && ackState.is("acked")) {
      img = "alarmWhite.png";
    }
    else if (ackState.is("acked") && !sourceState.is("normal")) {
      img = "alarm.png";
    }
    else if (sourceState.is("normal") && !ackState.is("acked")) {
      img = "alarmGreen.png";
    }
    else if (!ackState.is("acked") && !sourceState.is("normal")) {
      img = "alarmRed.png";
    }
    else if (ackState.is("acked") && sourceState.is("normal")) {
      img = "alarmWhite.png";   
    } 

    return img;    
  }
  
  var alarmRecordPrototype = null;
  
  /**
   * Insert an alarm row into the DOM.
   *
   * @name niagara.alarm.console.mobile-insertRow
   * @function
   * @private
   *
   * @param {Object} alarm the alarm record.
   * @param {Object} elem the DOM element to insert the row into.
   * @param {Boolean} [showAll] show all alarm fields regardless of selection.
   */
  function insertRow(alarm, elem, showAll) {

    // Lazily create this alarm record prototype  
    if (alarmRecordPrototype === null) {
      alarmRecordPrototype = baja.$("alarm:AlarmRecord");
    }
  
    // Get a list of all the fields the row will display
    var fields = pages.getHandler("chooseFields").getFields();
            
    // Add all the selected fields to the row
    var alarmFieldsElem = $(".alarmFields", elem);

    // Util function used for making a row a hyperlink
    function addHyperlink(hyperlinkOrd, str) {
      if (hyperlinkOrd !== null) {        
        str = "<a rel='external' href='" + baja.Ord.make(hyperlinkOrd).toUri() + "'>" + str + "</a>";
      }
      return str;
    }    
    
    // Add the data for each field
    var i, fieldData, fieldName, alarmDataIndex, hyperlinkOrd, fieldDom;
    
    for (i = 0; i < fields.length; ++i) {
      // Only add a field if it's selected or we're showing all the fields
      if (!fields[i].selected && !showAll) {
        continue;
      }
      fieldData = null;
      fieldName = fields[i].fieldName;
      
        
      alarmDataIndex = fieldName.indexOf("alarmData.");
      if (alarmDataIndex > -1) {
        // If we're showing data from an alarm data from then remove the 'alarmData.' part from the name
        fieldName = fields[i].fieldName.substring(10, fields[i].fieldName.length);
        fieldData = alarm.get("alarmData").get(fieldName, null);
      }
      else {
        // Get value from alarm record property
        if (fieldName === "alarmData") {
          fieldData = alarm.get(fieldName); 
        }
        else {
          // If we're not showing the alarmData Facets or some information from it then try to use
          // the display string from the alarm slot. This will be localized to the Server and gets around
          // quite a few localization problems
          fieldData = alarm.getDisplay(fieldName);        
        }
        // Attempt to get the display name for the field from a BAlarmRecord. This will change depending
        // on the locale
        fieldName = alarmRecordPrototype.getDisplayName(fieldName);      
      }
                     
      // If there's no data then skip to the next field        
      if (fieldData === null) {
        continue;
      }
        
      hyperlinkOrd = alarm.get("alarmData").get("hyperlinkOrd");
      
      // Create the row and add it to the table      
      if (i === 0) {
        fieldDom = "<tr><td class='alarmFieldDetails' colspan='2'>";       
        var imgDom = "";
        
        // If the user has defined their own alarm icon then display it
        if (alarm.get("alarmData").get("icon") !== null) {
          imgDom += "<img src='/ord/" + alarm.get("alarmData").get("icon") + "'/>";
        }
        
        // If there's a hyperlink then show the relevant icon
        if (hyperlinkOrd !== null) {
          imgDom += "<img src='/ord/module://icons/x16/link.png'/>";
        }
        
        // Add the icon for the alarm to the first row of the table
        imgDom += "<img class='alarmImg' src='/ord/module://alarm/com/tridium/alarm/icons/" + getAlarmImg(alarm) +  "'/>";
        
        fieldDom += addHyperlink(hyperlinkOrd, imgDom);

        // For the first row, don't both showing the field name. It'll probably be the timestamp anyway        
        fieldName = "";
      } else {
        fieldDom = "<tr><td class='alarmFieldName'>";       
        fieldDom += "<strong>" + addHyperlink(hyperlinkOrd, fieldName) + "</strong></td><td class='alarmFieldDetails'>";
      }
      fieldDom += addHyperlink(hyperlinkOrd, createFieldString(alarm, fields[i].fieldName, fieldData)); 
      fieldDom += "</td></tr>";
            
      $(alarmFieldsElem).append($(fieldDom));
    }
  }
  
////////////////////////////////////////////////////////////////
// Alarm Details
//////////////////////////////////////////////////////////////// 
      
  /**
   * @class Page for showing all the details from a particular alarm.
   *
   * @name niagara.alarm.console.mobile.AlarmDetails
   * @inner
   */   
  pages.register("alarmDetails", (function dialogAlarmDetails() {
    var alarm, // The current alarm being displayed
        index; // The current array index of the alarm
  
    function show(alarmRecord, alarmRecordIndex) {
      alarm = alarmRecord;
      index = alarmRecordIndex;
            
      $.mobile.changePage("#alarmDetails", {
        transition: "slideup"
      }); 
    }
    
    function pagebeforeshow() {
      // If there's no alarm present then hyperlink back to the console view
      if (!alarm) {
        $.mobile.changePage("#alarmConsole", { transition: "none" });
        return;
      }
    
      // Update summary text for page
      var alarmSourceHandler = pages.getHandler("alarmSource"),
          offset = alarmSourceHandler.getOffset(),
          rowsReturned = alarmSourceHandler.getRowsReturned();
      
      $("#alarmDetailsTitle").text(baja.lex("mobile").get({
        key: "alarm.alarmDetails",
        args: [index + 1 + offset, offset + 1, offset + rowsReturned]
      }));
      
      // Disable previous button if we can't go back
      $("#alarmDetailsPrev").toggleClass('ui-disabled', offset === 0 && index === 0);
    
      // Insert table data
      var elem = $("#alarmDetailsContent");
      $(".alarmFields", elem).children().remove();
      insertRow(alarm, elem, /*showAll*/true); 
    }
    
    function fetchAlarm(newIndex, reverse) {
      pages.getHandler("alarmSource").fetchAlarm(newIndex, function (alarmRecord, alarmRecordIndex) {
        alarm = alarmRecord;
        index = alarmRecordIndex;
        
        $.mobile.changePage("#alarmDetails", {
          transition: "slide",
          changeHash: false,
          allowSamePageTransition: true,
          reverse: reverse
        }); 
      });
    }
    
    function pagebeforecreate() {
      $("#alarmDetailsPrev").click(function () {
        fetchAlarm(--index, /*reverse*/true);
      });
      
      $("#alarmDetailsNext").click(function () {
        fetchAlarm(++index, /*reverse*/false);
      });
    }
    
    return {
      show: show,
      pagebeforeshow: pagebeforeshow,
      pagebeforecreate: pagebeforecreate
    }; 
  }()));
    
////////////////////////////////////////////////////////////////
// Open Alarm Source
//////////////////////////////////////////////////////////////// 
  
  /**
   * @class Page for showing all the alarms for a particular alarm source.
   *
   * @name niagara.alarm.console.mobile.AlarmSource
   * @inner
   */   
  pages.register("alarmSource", (function pageAlarmSource() {
    // The current alarm source being viewed
    var alarmSource = null,
        offset = 0, // how many records we jump forwards
        increment = 20, //how many records to display at once
        rowsReturned = 0, //number of records actually returned, not always = increment,
        alarmRecords = [], // the alarm records
        alarmsReadonly; // Indicates whether the alarms are readonly or not
    
    function eachRow(func) {
      $("#alarmSource li.alarmRow").each(function () {
        func.call(this, $(this).data("alarmRowData"));
      });
    }
  
    function eachSelectedRow(func) {
      $("#alarmSource li.alarmRow:has(input:checked)").each(function () {
        func.call(this, $(this).data("alarmRowData"));
      });
    }
    
    function hasSelected() {
      var res = false;
      eachSelectedRow(function () {
        res = true;
      });
      return res;
    }
    
    function updateButtons() {    
      $('#alarmSourceAckAlarms, #alarmSourceAddNotes, #alarmSourceClear')
        .toggleClass('ui-disabled', !hasSelected());
    }
    
    /**
     * Return the selected alarm information for the current Page.
     * <p>
     * This will return a data structure that can be used in a network call
     * for managing alarms.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#getSelectedAlarmInfo
     * @function
     *
     * @param {Object} alarm information
     */ 
    function getSelectedAlarmInfo() {
      var info = new baja.Component(),
          alarms = new baja.Component();
      
      eachSelectedRow(function (rowData) {
        alarms.add({value: rowData.getAlarm().get("uuid")});
      });
            
      info.add({
        slot: "alarms", 
        value: alarms
      });
      
      info.add({slot: "modelName", value: model.getName()});
      
      return info;
    }
    
    /**
     * Show all AlarmSource Page for the given alarm source.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#show
     * @function
     *
     * @param {Object} alarm source
     * @param {Boolean} isReadonly indiciates whether the alarm source is
     *                             readonly (and cannot be written too).
     */
    function show(source, isReadonly) {
      // Reset the offset
      offset = 0;
        
      $.mobile.showPageLoadingMsg();
      
      // Set the alarm source we're going to show
      alarmSource = source;
      
      alarmsReadonly = isReadonly;
              
      $.mobile.changePage("#alarmSource", {
        transition: "slide"
      });
    }
    
    function retrieveOpenAlarms(callbacks) {
      baja.Ord.make("local:|alarm:|bql:select * from openAlarms where source = '" + 
          baja.SlotPath.escape(alarmSource.toString()) + 
          "' order by timestamp desc").get({
        cursor: {
          ok: function (cursor) {
            callbacks.ok(cursor);  
          },
          fail: defFail,
          offset: offset,
          limit: increment
        },
        fail: defFail
      });
    }
    
    /**
     * Refresh the alarm source information.
     * <p>
     * This will make a query to the Server to refresh the alarm information
     * currently being shown to the user.
     *
     * @name niagara.alarm.console.mobile.AlarmSource-refresh
     * @function
     * @private
     *
     * @param {Function} callback
     */    
    function refresh(callback) {  
      callback = callback || baja.noop;    

      $.mobile.silentScroll(0);
      $.mobile.showPageLoadingMsg();
                  
      // Note down all selected alarms
      var selected = [];
      eachSelectedRow(function (rowData) {
        selected.push(rowData.getAlarm().get("uuid").toString());
      });
            
      // Remove all of the current list items
      $("#alarmSource .alarmRow").remove();
          
      // Resolve the alarm query
      retrieveOpenAlarms({
        ok: function (openAlarms) {
          var ul = $('<ul data-role="listview" data-theme="c" />'),
              chkboxHtml = alarmsReadonly ? '' : checkboxHtml.patternReplace({ checkboxId: 'alarmSourceSelectAll' }),
              listHeader = $('<li class="listHeader" data-role="list-divider"/>')
                .append($('<div class="checkboxContainer"/>')
                  .append(chkboxHtml));
          
          // Enable or disable the previous button
          $('#alarmSourcePrev').toggleClass('ui-disabled', offset <= 0);

          alarmRecords = [];
          rowsReturned = 0;
          // Iterate through the alarm records and create the list        
          openAlarms.each(function () {
            var alarm = this.get();
            alarmRecords.push(alarm);
            
            var rowChkBoxHtml = alarmsReadonly ? '' : checkboxHtml.patternReplace({ checkboxId: 'alarmSourceCheckbox-' + alarm.get('uuid') }),
                elem = $(domRowPrototype.patternReplace({
                  checkboxHtml: rowChkBoxHtml
                }));
            
            if (!alarmsReadonly) {
              elem.find('input[type=checkbox]').bind('change', updateButtons);
            }
            
            insertRow(alarm, elem);
            
            $(elem).data("alarmRowData", { 
              getAlarm: function () {
                return alarm; 
              }
            });
            
            (function () {
              var index = rowsReturned;
              // Update hyperlink link    
              $(".alarmLink", elem).click(function () {
                pages.getHandler("alarmDetails").show(alarm, index);
              });
            }());
                      
            ul.append(elem);
            
            rowsReturned++;
          });
          
          // Update summary text
          var summaryText = " (" + (offset + 1) + " - " + (offset + rowsReturned) + ")";
          listHeader.append('<div class="detailsContainer">' + 
              baja.SlotPath.unescape(alarmSource.toString()) + summaryText +"</span>");               
          
          // Set the table title to the component's display name
          ul.prepend(listHeader);
                                        
          updateInsetStatus(ul);
                    
          $('#alarmSourceContent .alarmList').html(ul).trigger('create').trigger('updatelayout');
          
          // For each table row see if it needs to be reselected
          eachRow(function (rowData) {
            if (selected.contains(rowData.getAlarm().get("uuid").toString())) {
              $("input[type='checkbox']:first", this).attr("checked", true).checkboxradio("refresh");
            }
          });
          
          updateButtons();
          
          $.mobile.hidePageLoadingMsg();
          
          callback();
        },
        fail: function (err) {
          // Just output to console as this can be called when 
          // the user attempts to navigate away from a page.
          console.error(err);
          
          callback();
        }
      });     
    }
    
    function next(callback) {
      // If the number of rows returned less than the limit then
      // don't increment but just refresh
      if (rowsReturned >= increment) {
        offset += increment;
      }
      refresh(callback);
    }
    
    function prev(callback) {
      if (offset <= 0) {
        offset = 0;
        callback();
      }
      else {
        offset -= increment;
        refresh(callback);
      }
    }
          
    /**
     * pagebeforecreate Page event handler.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#pagebeforecreate
     * @function
     */ 
    function pagebeforecreate(obj) {            
      // Set up click handlers
      $("#alarmSourceAckAlarms").click(ackSelectedAlarms);
      $("#alarmSourceClear").click(function () {
        forceClear();
      });
      $("#alarmSourceAddNotes").click(function () {
        // Create selected alarm information from the previously shown page      
        var alarmInfo = getSelectedAlarmInfo();
        showNotesDialog(alarmInfo);
      });
      $("#alarmSourcePrev").click(function () { prev(); });
      $("#alarmSourceNext").click(function () { next(); });
    }
        
    function fetchAlarm(alarmRecordIndex, callback) {
      if (alarmRecordIndex < 0) {
        prev(function () {
          callback(alarmRecords[alarmRecords.length - 1], alarmRecords.length - 1);
        });
      }
      else if (alarmRecordIndex >= alarmRecords.length) {
        next(function () {
          callback(alarmRecords[0], alarmRecordIndex < increment && rowsReturned < increment ? alarmRecords.length - 1 : 0);
        });
      }
      else {
        callback(alarmRecords[alarmRecordIndex], alarmRecordIndex);
      }
    }
    
    /**
     * Return the Commands for the Page.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#getCommands
     * @function
     */
    function getCommands(obj) {
      var cmds = obj.commands;
      
      // Add refresh Command
      cmds.splice(0, 0, new commands.Command("%lexicon(mobile:refresh)%", function() { 
        refresh();
      }));
      
      return cmds;
    }
    
    /**
     * pagebeforeshow Page event handler.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#pagebeforeshow
     * @function
     */
    function pagebeforeshow(prevName) {
      // Remove all of the current list items if the last page was the alarm console
      if (prevName === "alarmConsole") {
        $("#alarmSource .alarmRow").remove();
      }
    }
    
    /**
     * pageshow Page event handler.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#pageshow
     * @function
     */
    function pageshow() { 
      // Hyperlink back to the original alarm console if there's nothing selected here    
      if (alarmSource === null) {
        $.mobile.changePage("#alarmConsole", { transition: "fade" });
        return;
      }
             
      refresh();
    }
        
    /**
     * Custom Page event handler called whenever alarms are acknowledged or cleared in the App.
     *
     * @name niagara.alarm.console.mobile.AlarmSource#alarmsUpdated
     * @function
     */    
    function alarmsUpdated() {
      // If the alarms are acked if this page is being displayed then refresh it
      if (pages.getCurrentName() === "alarmSource") {
        refresh();
      }
    }
        
    // Functions exported as public 
    return {
      // Page Event Handlers
      pagebeforecreate: pagebeforecreate,
      pagebeforeshow: pagebeforeshow,
      pageshow: pageshow,
      
      // Commands
      getCommands: getCommands,
      
      // Custom Page Event Handlers
      alarmsAcked: alarmsUpdated,
      alarmsCleared: alarmsUpdated,
      
      // Access
      show: show,
      getSelectedAlarmInfo: getSelectedAlarmInfo,
      hasSelected: hasSelected,
      fetchAlarm: fetchAlarm,
      getOffset: function () { return offset; },
      getRowsReturned: function () { return rowsReturned; }
    };
  }()));  
                    
////////////////////////////////////////////////////////////////
// Alarm Console
////////////////////////////////////////////////////////////////
  
  /**
   * @class Main Alarm Console Page.
   *
   * @name niagara.alarm.console.mobile.AlarmConsole
   * @inner
   */   
  pages.register("alarmConsole", (function pageAlarmConsole() {
    
    function eachRow(func) {
      $("#alarmConsole li.alarmRow").each(function () {
        func.call(this, $(this).data("alarmRowData"));
      });
    }
  
    function eachSelectedRow(func) {
      $("#alarmConsole li.alarmRow:has(input:checked)").each(function () {
        func.call(this, $(this).data("alarmRowData"));
      });
    }
    
    function hasSelected() {
      var res = false;
      eachSelectedRow(function () {
        res = true;
      });
      return res;
    }
    
    function updateButtons() {
      $('#alarmConsoleAckAlarms, #alarmConsoleAddNotes, #alarmConsoleClear')
        .toggleClass('ui-disabled', !hasSelected());
    }
     
    /**
     * Add a row to the Alarm Console.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole-addRow
     * @private
     * @function
     *
     * @param {Object} rowData data used to create the console row.
     */      
    function addRow(ul, rowData) {
      // Create an alarm row
      var id = rowData.getAlarm().getUuid(),
          isReadonly = (rowData.getParent().getFlags(rowData.getName()) & baja.Flags.READONLY) === baja.Flags.READONLY,
          chkboxHtml = isReadonly ? '' : checkboxHtml.patternReplace({ checkboxId: 'alarmConsoleCheckbox-' + id }),
          html = domRowPrototype.patternReplace({
            checkboxHtml: chkboxHtml
          }),
          elem = $(html),
          alarm = rowData.getAlarm(),
          ackStateSummary = rowData.getAckState();  
          
      // Update the elements of the row
      insertRow(alarm, elem);
                
      // Update hyperlink link    
      $(".alarmLink", elem).click(function () {        
        pages.getHandler("alarmSource").show(alarm.get("source"), isReadonly);
      });
      
      // Update the alarm Ack Summary text
      $(".ackStateSummary", elem).each(function () {
        $(this).text(ackStateSummary);
      });
      
      // If the row is marked as readonly then hide the check box  
      if (!isReadonly) {                  
        elem.find('input[type=checkbox]').bind('change', updateButtons);
      } 
      
      // Append the row 
      ul.append(elem);
      
      // Attach the row data to the DOM element for later reference
      $(elem).data("alarmRowData", rowData); 
    }
    
    /**
     * Update the entire Alarm Console.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole-update
     * @private
     * @function
     */    
    function update() { 
      // If the model hasn't been loaded yet then bail    
      if (!model) {
        return;
      }
      
      // Find all the currently selected rows
      var selected = [];
      eachSelectedRow(function (rowData) {
        selected.push(rowData.getAlarm().getSource().toString());
      });

      // Remove all the current alarm rows
      $("#alarmConsole div[class^=alarmRow]").remove();      
       
      var ul = $('<ul data-role="listview" data-theme="c" />');
      ul.append($('<li data-role="list-divider"/>').text(consoleDisplayName));
      
      // Iterate through all of the alarm console rows
      model.getSlots(function (slot) { 
        return slot.isProperty() && model.get(slot).getType().is("alarm:AlarmConsoleRow");
      }).each(function (slot) {
        addRow(ul, this.get(slot));
      });
            
      updateInsetStatus(ul);
      
      $('#alarmConsole .alarmList').html(ul).trigger('create');
             
      // For each table row see if it needs to be reselected
      eachRow(function (rowData) {
        if (selected.contains(rowData.getAlarm().getSource().toString())) {
          $("input[type='checkbox']:first", this).attr("checked", true).checkboxradio("refresh");
        }
      });
      
      updateButtons();
    }
    
    /**
     * Refresh the remote AlarmConsoleModel
     *
     * @name niagara.alarm.console.mobile.AlarmConsole-refresh
     * @private
     * @function
     */ 
    function refresh() {
      $.mobile.showPageLoadingMsg();
      
      model.getParent().serverSideCall({
        typeSpec: "mobile:MobileAlarmServerSideCallHandler",
        methodName: "refresh",
        value: model.getName(),
        ok: defOk,   
        fail: defFail  
      });
    }
            
    /**
     * Return the selected alarm information for the current Page.
     * <p>
     * This will return a data structure that can be used in a network call
     * for managing alarms.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole#getSelectedAlarmInfo
     * @function
     *
     * @param {Object} alarm information
     */ 
    function getSelectedAlarmInfo() {
      var info = new baja.Component(),
          alarmSources = new baja.Component();
      
      eachSelectedRow(function (rowData) {
        alarmSources.add({value: rowData.getAlarm().getSource()});
      });
            
      info.add({
        slot: "alarmSources", 
        value: alarmSources
      });
      
      info.add({slot: "modelName", value: model.getName()});
      
      return info;
    }
    
    /**
     * Custom Page event handler called whenever alarms are cleared in the App.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole#alarmsCleared
     * @function
     */    
    function alarmsCleared() {
      // If the alarms are force cleared on this page then refresh it
      if (pages.getCurrentName() === "alarmConsole") {
        refresh();
      }
    }
    
    /**
     * Return the Commands for the Page.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole#getCommands
     * @function
     */
    function getCommands(obj) {
      var cmds = obj.commands;
      
      // Add refresh Command
      cmds.splice(0, 0, new commands.Command("%lexicon(mobile:refresh)%", function() { 
        refresh();
      }));
      
      return cmds;
    }
  
    /**
     * pagebeforecreate Page event handler.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole#pagebeforecreate
     * @function
     */
    function pagebeforecreate(obj) {
      // Set up click handlers
      $("#alarmConsoleAckAlarms").click(ackSelectedAlarms);
      $("#alarmConsoleClear").click(forceClear);
      $("#alarmConsoleAddNotes").click(function () {
        // Create selected alarm information from the previously shown page      
        var alarmInfo = getSelectedAlarmInfo();
        showNotesDialog(alarmInfo);
      });
    }
    
    /**
     * pageshow Page event handler.
     *
     * @name niagara.alarm.console.mobile.AlarmConsole#pageshow
     * @function
     */
    function pageshow() {
      update();
    }
  
    // Functions exported as public 
    return {
      // Page Events Handlers
      pagebeforecreate: pagebeforecreate,
      pageshow: pageshow,
      
      // Commands
      getCommands: getCommands,
      
      // Custom Page Event Handlers
      alarmsCleared: alarmsCleared,
            
      // Access
      getSelectedAlarmInfo: getSelectedAlarmInfo,
      hasSelected: hasSelected,
      update: update
    };
  }()));
  
  function initializeUI() {
    $.mobile.showPageLoadingMsg();
    
    $('.alarmList')
      /*
       * tap a whole row and it will have the same effect as tapping the 
       * checkbox itself
       */
      .delegate('.alarmRow .alarmSelectLink', 'click', function () {
        var checkbox = $(this).find('input[type=checkbox]'),
            checked = checkbox.is(':checked');
        checkbox.attr('checked', !checked).trigger('change').checkboxradio('refresh');
      })
      
      /*
       * unfortunately this also means if you tap the checkbox itself it also
       * triggers the whole-row tap - so checkbox checks itself and row 
       * unchecks it right back again. undo this manually for now
       */
      .delegate('.alarmRow .ui-btn', 'click', function () {
        var checkbox = $(this).siblings('input[type=checkbox]'),
            checked = checkbox.is(':checked');
        checkbox.attr('checked', !checked).checkboxradio('refresh');
      })
      
      /*
       * arm the "select all" checkbox in the header row
       */
      .delegate('.listHeader input[type=checkbox]', 'change', function () {
        var $this = $(this),
            $checked = $this.is(':checked'),
            list = $this.closest('ul');
        list.find('.alarmRow input[type=checkbox]')
          .attr('checked', $checked)
          .trigger('change')
          .checkboxradio('refresh');
      });
    
    util.mobile.preventNavbarHighlight($('#alarmSource-footer, #alarmConsole-footer'));
    
    // Attach commands button handler
    $("#commandsButtonConsole, #commandsButtonSource").click(commands.showCommandsHandler);
    
    /*
     * if we rotate the phone or such, expand the alarm list to fullscreen
     * if we need to, otherwise keep pretty rounded inset
     */
    $(window).bind('resize', baja.throttle(function () {
      updateInsetStatus($('#alarmConsole ul:jqmData(role=listview)'));
      updateInsetStatus($('#alarmSource ul:jqmData(role=listview)'));
    }, 500));
    
    /*
     * no reason to let the header/footer hide themselves since basically
     * the whole page is links
     */
    $.mobile.fixedToolbars.setTouchToggleEnabled(false);
    
    // Resolve the ConsoleReceipient we're currently viewing
    var consoleOrd = baja.Ord.make(niagara.view.ord);
            
    // Resolve the alarm console
    consoleOrd.get({
      ok: function (alarmConsole) {
        // Set the table title to the component's display name
        consoleDisplayName = alarmConsole.getDisplayName();
                
        var batch = new baja.comm.Batch(),
            modelNavOrd;
                    
        // Add an Alarm Console Model to the Console Receipient in the Station...
        alarmConsole.serverSideCall({
          typeSpec: "mobile:MobileAlarmServerSideCallHandler",
          methodName: "createAlarmModel",
          // Called once the model has been added
          ok: function (ord) {                    
            modelNavOrd = ord;
          },
          fail: defFail,
          batch: batch
        });
        
        // Ensure sync buffers are flushed
        baja.station.sync({batch: batch});
        
        batch.addCallback(
          function ok() {
            modelNavOrd.get({
              ok: function (alarmModel) {
                model = alarmModel;
              
                // Use a ticket for the update to coalease lots of Component events
                var ticket = baja.clock.expiredTicket;
                model.attach("changed added removed", function () {
                  if (ticket.isExpired()) {
                    ticket = baja.clock.schedule(function () {
                      pages.getHandler('alarmConsole').update();
                    }, 20);
                  }
                });
                
                // Do first update
                pages.getHandler('alarmConsole').update();
                                  
                // Hide the loading title
                $.mobile.hidePageLoadingMsg();
                                    
                // Ensure nav footer bar appears
                $.mobile.fixedToolbars.show();
              },
              fail: defFail,
              subscriber: new baja.Subscriber()
            }); 
          },
          defFail
        );
        
        batch.commit();
      }, 
      fail: defFail,
      lease: true
    });
  }
  
  function tearDown() {
    if (model) {
      // Invoke the expire Action synchronously on shutdown. If we do this asynchronously
      // when the page unloads, it doesn't seem to get called.
      var batch = new baja.comm.Batch();
      
      model.getParent().serverSideCall({
        typeSpec: "mobile:MobileAlarmServerSideCallHandler",
        methodName: "expire",
        value: model.getName(),
        fail: function () {},
        batch: batch
      });

      batch.commitSync();
    }
  }
  
  baja.started(initializeUI);
  baja.preStop(tearDown); 
  
}(niagara.util.mobile.pages));