/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Simple Utility Dialogs
 *
 * @author Gareth Johnson
 */
 
//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true, 
  bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

/*global $, Image*/ 

/**
 * @namespace Simple Dialogs API for BajaScript Tutorials
 */
var dialogs = (function simpleDialogs() {
  // Use ECMAScript 5 Strict Mode
  "use strict";
  
  var dialogHtml = "<div id='simpleDialog'>" +
                     "<h2 id='dialogTitle'>{title}</h2>" + 
                     "<div id='dialogContent'></div>" + 
                     "<div id='dialogButtonPanel'></div>" +
                   "</div>",          
      buttonHtml = "<button class='dialogButton' id='{id}'>{text}</button>";
  
  /**
   * This is a conveniance method used for working with functions that take 
   * an Object Literal as an argument.
   * <p>
   * This method always ensures an Object is returned so its properties can be
   * further validated.
   *
   * @inner
   * @private
   */
  function objectify(obj, propName) {
    if (!(obj === undefined || obj === null)) {
      if (obj.constructor === Object) {
        return obj;
      }
      else if (typeof propName === "string") {
        var o = {};
        o[propName] = obj;
        return o;
      }
    }
    return {};
  }
  
  /**
   * A no-operation function.
   *
   * @inner
   * @private
   */
  function noop() {}
  
  /**
   * Define a default value for possibly undefined variables.
   *
   * @inner
   * @private
   *
   * @param val the value to be tested.
   * @param defVal the default value to be returned if the value is undefined.
   * @returns the default value if the value is undefined.
   */
  function def (val, defVal) {
    return val === undefined ? defVal : val;
  }
  
  /**
   * Replace patterned items in a string from an Object Map.
   *
   * @inner
   * @private
   *
   * @param {String} str
   * @param {Object} obj key pairs to replace in the text
   * @returns {String}
   */
  function patternReplace(str, obj) {
    return str.replace(/\{[a-zA-Z0-9]*\}/g, function (match) {
      match = match.substring(1, match.length - 1);
      return typeof obj[match] === "string" ? obj[match] : match;
    });
  }
    
  /**
   * Close any currently opened dialog boxes.
   *
   * @function
   * @name dialogs.close
   */   
  function close() {
    $.modal.close();
  }
  
  /**
   * Add a button handler to the dialog.
   *
   * @inner
   * @private
   *
   * @param dialogDom Dialog DOM element.
   * @param {String} id the id of the button.
   * @param {String} text the button text.
   * @param {Function} the handler for when the button is clicked.
   */
  function addButton(dialogDom, id, text, handler) {
    var buttonDom = $(patternReplace(buttonHtml, {
        id: id,
        text: text
      }));
      
    // When this button is clicked, call the handler
    buttonDom.click(handler);
  
    $("#dialogButtonPanel", dialogDom).append(buttonDom);
  }
  
  /**
   * A global function launch a dialog box.
   * <p>
   * By default, this dialog won't show any buttons.
   * <pre>
   *   dialogs.show({
   *     title: "BajaScript",
   *     content: "BajaScript Rocks!"
   *   });
   * </pre>
   * For other parameters, please see the author's documentation.
   * {@link http://www.ericmmartin.com/projects/simplemodal}.
   * <p>
   * Please note, any custom buttons handlers must call {@link dialogs.close}.
   *
   * @see http://www.ericmmartin.com/projects/simplemodal
   *
   * @function 
   * @name dialogs.show
   * 
   * @param {Object} params parameters for launching a dialog.
   * @param {String} [params.title] the dialog's title.
   * @param {String} [params.content] the dialog's content. Please note, this can be HTML.
   * @param {Boolean} [params.fadeIn] by default dialog's fade when shown.
   */
  function show(params) {
    var onOpen,
        dialogDom,
        buttonDom,
        buttonNames;
  
    params = objectify(params, "content");
    
    // Set up default dialog parameters
    params.title = def(params.title, "BajaScript Tutorials");
    params.content = def(params.content, "");
    params.fadeIn = def(params.fadeIn, true);
    buttonNames = params.buttonNames || {};
      
    // Create the DOM
    dialogDom = $(patternReplace(dialogHtml, {
                  title: params.title
                }));
    $('#dialogContent', dialogDom).html(params.content);
    
    // Attach buttons  
    if (params.ok || params.cancel || params.yes || params.no) {
      if (params.ok) {
        addButton(dialogDom, "ok", buttonNames.ok || "OK", params.ok);
      }
      if (params.yes) {
        addButton(dialogDom, "yes", buttonNames.yes || "Yes", params.yes);
      }
      if (params.no) {
        addButton(dialogDom, "no", buttonNames.no || "No", params.no);
      }
      if (params.cancel) {
        addButton(dialogDom, "cancel", buttonNames.cancel || "Cancel", params.cancel);
      }
    }
    else {
      $("#dialogButtonPanel", dialogDom).css("display", "none");
    }
    
    // If there's no content then completely hide the DIV for it
    if (!params.content) {
      $("#dialogContent", dialogDom).css("display", "none");
    }
                
    // Fade dialog in and invoke user onOpen
    onOpen = params.onOpen || noop;
    params.onOpen = function (dialog) {    
      // Show dialog
      dialog.overlay.show();
      
      if (params.fadeIn) {
        dialog.container.fadeIn("fast");
        dialog.data.fadeIn("fast");
      }
      else {
        dialog.container.show();
        dialog.data.show();
      }
    
      onOpen();
    };
      
    // Launch a dialog box using the Simple Modal Dialog jQuery Plugin
    $.modal(dialogDom, params);
  }
    
  /**
   * Show a dialog with an OK button.
   * <p>
   * Please note, any custom buttons handlers must call {@link dialogs.close}.
   *
   * @param {Object} params arguments for the ok dialog box.
   * @param {Function} [params.ok] called when the ok button is clicked.
   */
  function ok(params) {
    params = objectify(params, "content");
    params.ok = params.ok || close;
    show(params);
  }
  
  /**
   * Show a dialog with OK and Cancel buttons.
   * <p>
   * Please note, any custom buttons handlers must call {@link dialogs.close}.
   *
   * @param {Object} params arguments for the ok dialog box.
   * @param {Function} [params.ok] called when the ok button is clicked.
   * @param {Function} [params.cancel] called when the cancel button is clicked.
   */
  function okCancel(params) {
    params = objectify(params, "content");
    params.ok = params.ok || close;
    params.cancel = params.cancel || close;
    show(params);
  }
  
  /**
   * Show a dialog a Cancel button.
   * <p>
   * Please note, any custom buttons handlers must call {@link dialogs.close}.
   *
   * @param {Object} params arguments for the ok dialog box.
   * @param {Function} [params.cancel] called when the cancel button is clicked.
   */
  function cancel(params) {
    params = objectify(params, "content");
    params.cancel = params.cancel || close;
    show(params);
  }
  
  /**
   * Show a dialog with Yes and No buttons.
   * <p>
   * Please note, any custom buttons handlers must call {@link dialogs.close}.
   *
   * @param {Object} params arguments for the ok dialog box.
   * @param {Function} [params.yes] called when the yes button is clicked.
   * @param {Function} [params.no] called when the no button is clicked.
   */
  function yesNo(params) {
    params = objectify(params, "content");
    params.yes = params.yes || close;
    params.no = params.no || close;
    show(params);
  }
  
  /**
   * Show a dialog with Yes, No and Cancel buttons.
   * <p>
   * Please note, any custom buttons handlers must call {@link dialogs.close}.
   *
   * @param {Object} params arguments for the ok dialog box.
   * @param {Function} [params.yes] called when the yes button is clicked.
   * @param {Function} [params.no] called when the no button is clicked.
   * @param {Function} [params.cancel] called when the cancel button is clicked.
   */
  function yesNoCancel(params) {
    params = objectify(params, "content");
    params.yes = params.yes || close;
    params.no = params.no || close;
    params.cancel = params.cancel || close;
    show(params);
  }
  
  var loadImgSrc = "/module/testBajaScript/com/tridium/bajascript/tutorials/rc/images/loader.gif";
  
  /**
   * Show a loading dialog box.
   */
  function loading() {
    // Ensure the image is loaded before we display the loading dialog
    var img = new Image();
  
    img.onload = function () {
      show({
        title: "<div style='text-align: center;'>Loading</div>",
        content: "<img src='" + loadImgSrc + "'/>",
        fadeIn: false,
        onClose: function (dialog) {
          dialog.overlay.fadeOut("slow");
          dialog.container.fadeOut("slow");
          dialog.data.fadeOut("slow", function () {
            dialogs.close();
          });
        }
      });
    };
    
    img.src = loadImgSrc;
  }
  
  // Exported public functions
  return {
    show: show,
    ok: ok,
    cancel: cancel,
    okCancel: okCancel,
    yesNo: yesNo,
    yesNoCancel: yesNoCancel,
    loading: loading,
    close: close
  };
}());

  
