function startPresentation() {
  
  (function initPresentation() {
    /*
     * If the user doesn't realize he can press left/right to move between
     * slides, show a navigation hint. It will disappear if left or right is
     * pressed
     */
    var slideLeft = false,
        msPerSlide = 10000;
    
    function stepleave() {
      slideLeft = true;
      $('#navigationHint').css('opacity', 0);
      $(document).unbind('impress:stepleave', stepleave);
    }
    $(document).bind('impress:stepleave', stepleave);
    
    setTimeout(function () {
      if (!slideLeft) {
        $('#navigationHint').css('opacity', 0.8);
      }
    }, msPerSlide);
  }());
  
  (function initLiveUpdatesSlide() {
    var ords = [
          'station:|slot:/BajaScriptTutorials/impress/liveUpdate/Ramp',
          'station:|slot:/BajaScriptTutorials/impress/liveUpdate/SineWave',
          'station:|slot:/BajaScriptTutorials/impress/liveUpdate/Random',
        ],
        sub = new baja.Subscriber();
    
    /*
     * If the 'out' slot changes, update the corresponding label and slider
     * with the new value.
     */
    sub.attach('changed', function (prop, cx) {
      if (prop.getName() === 'out') {
        var name = this.getName(),
            value = this.get(prop).getValue(),
            label = $('#live' + name + 'Slider-label'),
            slider = $('#live' + name + 'Slider');
        
        label.text(name + " Value: " + Math.round(value));
        slider.val(value).slider('refresh');
      }
    });
    
    /*
     * Use a baja.BatchResolve to resolve an entire array of ORDs in one
     * network call. The given subscriber will be attached to all resolved
     * components.
     */
    new baja.BatchResolve(ords).resolve({
      subscriber: sub
    });
    
    /*
     * Initialize our JQM sliders. There is a bug where the slider() function
     * won't hide the non-enhanced sliders, so we'll just hide them with 
     * the hide() function so only the enhanced widgets will show. 
     */
    $('#liveRampSlider, #liveSineWaveSlider, #liveRandomSlider').slider().hide();
  }());
  
  (function initHistorySlide() {
    var chartView = new niagara.mobile.charts.HistoryChartView(),
        ord = 'station:|slot:/BajaScriptTutorials/impress/tutorialsHistorySineWave/NumericInterval',
        ticket,
        onHistorySlide;
    
    function redrawChart() {
      if (onHistorySlide) { //only draw if we're actually viewing the history slide
        /*
         * Every Niagara Mobile View has a loadValue function. This loads a
         * value into an already-bound view (that is, loadValue won't behave
         * correctly until after initializeDOM completes). Here, the
         * HistoryChartView expects either an array of data to plot, or an
         * ORD. The chart view will resolve the history ORD and build up its
         * own array of data to plot.
         * 
         * loadValue will have different behavior different View types.
         */
        chartView.loadValue(ord);
      }
    }
    
    function stopRefreshing() {
      clearInterval(ticket);
    }
    
    function startRefreshing() {
      stopRefreshing();
      
      ticket = setInterval(redrawChart, 3000);
    }

    /*
     * impress.js fires the impress:stepenter event whenever the slide
     * changes. If and only if we're viewing the history slide, start 
     * auto-refreshing the history chart.
     */
    document.addEventListener('impress:stepenter', function (event) {
      onHistorySlide = event.target.id === 'historySlide';
      if (onHistorySlide) {
        startRefreshing();
      } else {
        stopRefreshing();
      }
    });
    
    

    /*
     * Every Niagara Mobile View has an initializeDOM function that will
     * bind that view to a particular DOM element. Here, we want our history
     * chart to go into the #historyChart div. 
     */
    chartView.initializeDOM($('#historyChart'), function () {
      chartView.loadValue(ord);
    });
    
    /*
     * If we select a different component from the dropdown, immediately
     * resolve that ORD and start drawing the new history data.
     */
    $('#historySelect').change(function () {
      ord = decodeURIComponent($(this).val());
      chartView.loadValue(ord);
      startRefreshing();
    }).selectmenu();
  }());
  
  

  (function initFieldEditorsSlide() {
    
    var folderOrd = 'station:|slot:/BajaScriptTutorials/impress/fieldEditors',
        folder,
        folderEditor;
    
    /*
     * Update the value display.
     */
    function updatePropDisplay(component, prop) {
      var name = prop.getName(),
          value = component.get(prop),
          outputDiv = $('#' + name + 'Output'),
          nameSpan = $('<span/>').text(name + ' value: '),
          valueSpan = $('<span class="pointValue" />').text(String(value));
      
      outputDiv.html(nameSpan.add(valueSpan));
    }
    
    /*
     * When the component is first subscribed, populate the display for all
     * the slot values. Then when an individual slot changes, only update the
     * display for that slot.
     */
    function makeSubscriber() {
      var sub = new baja.Subscriber();

      sub.attach({
        subscribed: function () {
          this.getSlots().properties().each(function (prop) {
            updatePropDisplay(this, prop);
          });
        },
        changed: function (prop, cx) {
          updatePropDisplay(this, prop);
          folderEditor.loadValue(folderEditor.value);
        }
      });
      
      return sub;
    }
    
    /*
     * Here we have a long sequence of asynchronous calls. To avoid descending
     * into callback spaghetti we'll use the niagara.util.flow namespace to
     * simply define a sequence of functions and let the framework manage the
     * asynchronous callbacks for us.
     */
    var initFieldEditorsWorkflow = niagara.util.flow.sequential(
      /*
       * First resolve the ORD. We pass in this.ok and this.fail because in
       * util.flow, 'this' is bound to a special callback object the framework
       * will use to manage the async calls. It's our responsibility to pass
       * 'this' or this.ok/this.fail to the async functions as needed, or else 
       * call this.ok/this.fail ourselves.
       */
      function (cx) {
        baja.Ord.make(folderOrd).get({
          ok: this.ok,
          fail: this.fail,
          subscriber: makeSubscriber()
        });
      },
      
      /*
       * Next, asynchronously instantiate a field editor. Anything passed to
       * this.ok in the previous function will get passed to this one as
       * additional parameters.
       */
      function (cx, folder) {
        niagara.fieldEditors.makeFor(folder, this);
      },
      
      /*
       * Now, bind the field editor to a DOM element and load in a value. A
       * field editor has a convenience function called buildAndLoad() that
       * does this for us - it's basically just shorthand for 
       * calling initializeDOM() and then loadValue() without having to
       * manage a double set of callbacks.
       */
      function (cx, editor) {
        folderEditor = editor;
        editor.buildAndLoad($('#fieldEditors'), this);
      },
      
      /*
       * Finally, arm an event handler on the save button (now that we have
       * an actual field editor to save). Note how we call this.ok() at the end
       * - this is required even though we didn't actually call any
       * asynchronous methods here.
       */
      function (cx, editor) {
        var button = $('#saveButton');
        
        button.click(function () {
          editor.saveValue(function () {
            button.button('disable');
          });
        });
        
        this.ok();
      }
    );
  
    var button = $('#saveButton').button().button('disable');
  
    /*
     * A field editor may trigger an editorchange event whenever the entered
     * value is changed (key pressed, selection made, slider flipped, etc.).
     */
    $('#fieldEditors').bind('editorchange', function () {
      button.button('enable');
    });
    
    /*
     * Register a composite field editor on the baja:Folder type (since our
     * ORD resolves to a Folder). A composite field editor just iterates over
     * the slots on a component and loads an individual field editor for
     * each slot.
     */
    niagara.fieldEditors.register('baja:Folder', niagara.fieldEditors.composite.makeComposite());

    /*
     * Invoke the workflow that will resolve our ORD and load up our field
     * editors. The invoke() method takes in a context object that each
     * function in the workflow will receive in turn. This workflow doesn't 
     * need anything in the context, so I'll just pass in an empty object.
     */
    initFieldEditorsWorkflow.invoke({});
  }());
  
  
  
  
  
  (function initActionsSlide() {
    var overrideButton = $('#overrideButton').button(),
        invokeButton = $('#invokeButton').button(),
        overrideContainer = $('#overrideEditorContainer'),
        overrideEditor = $('#overrideEditor'),
        invokeContainer = $('#invokeButtonContainer'),
        writableOrd = 'station:|slot:/BajaScriptTutorials/impress/actionInvoke/EnumWritable';
    
    /*
     * Another async workflow to save the override field editor and use the
     * saved override object to invoke the action. This is a short enough 
     * series of events that I could have just done straight callbacks rather
     * than bring in util.flow - I just like to keep things looking nice. The
     * straight-up-callback method would look something like:
     * 
     * editor.saveValue({
     *   ok: function (override) {
     *     writable.invoke({
     *       slot: 'override',
     *       value: override,
     *       ok: function () {
     *         //something to do when the action has been invoked
     *       },
     *       fail: function (err) {
     *         //handle invocation error
     *       }
     *     });
     *   },
     *   fail: function (err) {
     *     //handle editor save error
     *   }
     * });
     */
    var invokeOverrideWorkflow = niagara.util.flow.sequential(
      /*
       * First save the editor.
       */
      function (cx) {
        cx.editor.saveValue(this);
      },
      
      /*
       * We get the override object as a parameter because saveValue() passes
       * it to this.ok(). Now we just invoke the action. And here is where
       * I use the writable property that got set way back in step 2 of
       * loadOverrideEditorWorkflow!
       */
      function (cx, override) {
        cx.writable.invoke($.extend(this, {
          slot: 'override',
          value: override
        }));
      }
    );
    
    /*
     * Whenever the out value changes, update the "The lights are..." display
     * to show its current value.
     */
    function makeSubscriber() {
      var sub = new baja.Subscriber();
      sub.attach('subscribed changed', function () {
        var value = this.get('out');
        $('#actionStatus span.pointValue').text(value.getDisplay());
      });
      return sub;
    }
    
    /*
     * Async workflow to load up the override editor.
     */
    var loadOverrideEditorWorkflow = niagara.util.flow.sequential(
        
      /*
       * First resolve our ORD...
       */
      function (cx) {
        baja.Ord.make(writableOrd).get({
          ok: this.ok,
          fail: this.fail,
          subscriber: makeSubscriber()
        });
      },
      
      /*
       * Now instantiate the field editor using the makeFor() function. Note
       * that I'm setting the writable property on the cx object - a single
       * context object will get passed from one function in the async workflow
       * to the next, so that they can share state.
       */
      function (cx, writable) {
        cx.writable = writable;
        
        niagara.fieldEditors.makeFor({
          value: baja.$('control:EnumOverride'),
          facets: cx.writable.get('facets')
        }, this);
      },
      
      /*
       * Now bind the editor to our DOM element and load up the value. 
       */
      function (cx, editor) {
        cx.editor = editor;
        editor.buildAndLoad($('#overrideEditor').empty(), this);
      },
      
      /*
       * At this point my context object has editor and writable properties -
       * and it just so happens that that's what invokeOverrideWorkflow wants,
       * so I can just pass the whole shebang right into it. Again, note the
       * explicit call to this.ok() even though everything was synchronous in
       * this step.
       */
      function (cx) {
        invokeButton.bind('click', function () {
          invokeOverrideWorkflow.invoke(cx);
        });
        this.ok();
      }
    );
    
    /*
     * This is just a bit of magic necessary to make the datebox time picker
     * work - it doesn't really know how to show its time picker popup inside 
     * of a 3d-scaled environment like impress.js, so we has to manually
     * position it.
     */
    $('.ui-input-datebox a.ui-btn').live('click', function (event) {
      $('#overrideEditor > .ui-datebox-container').css({
        left: event.pageX - overrideEditor.offset().left - 100,
        top: event.pageY - overrideEditor.offset().top - 100
      });
    });
    
    loadOverrideEditorWorkflow.invoke({});
  }());
  
  
  
  
  (function initAddRemove() {
    var controlsDiv = $('#addRemoveControls'),
        contentsDiv = $('#addRemoveContents'),
        typeSpecSelect = $('#addRemoveTypeSpec'),
        addButton = $('#addRemoveAddButton'),
        removeButton = $('#addRemoveRemoveButton'),
        ImpressRadioButtonView = niagara.util.mobile.RadioButtonView.subclass(),
        radioView,
        slotNameEditor,
        defaultValueEditor,
        addRemoveFolder,
        addRemoveFolderOrd = 'station:|slot:/BajaScriptTutorials/impress/addRemove';

    /*
     * Here I use niagara.util.aop to enhance my RadioButtonView subclass.
     * After the base class creates the label, I just add a hook in that will
     * add another bit to the label's HTML that will display the slot Type
     * as well as its name.
     */
    niagara.util.aop.after(ImpressRadioButtonView.prototype, 'makeLabel', function (args, label) {
      var slot = args[1],
          span = label.children('span').eq(0),
          text = span.text();
      span.text(text + ": " + slot.getType());
    });
    
    /*
     * Whenever a value changes or a slot is added/removed, we want to reload
     * the whole radio button list. We'll pass the function into baja.throttle
     * just to prevent it being called too often.
     */
    function makeSubscriber() {
      var sub = new baja.Subscriber();
      sub.attach('added removed changed', baja.throttle(function () {
        radioView.loadValue(this);
      }, 2500));
      return sub;
    }
    
    /*
     * First resolve our ORD and get our folder back. 
     */
    baja.Ord.make(addRemoveFolderOrd).get({
      ok: function () {
        //maintain a reference to our mounted folder
        addRemoveFolder = this;
        
        //instantiate our radio buttons into #addRemoveContents
        radioView = new ImpressRadioButtonView();
        radioView.initializeDOM($('#addRemoveContents'), function () {
          radioView.loadValue(addRemoveFolder);
        }); 
        
        /*
         * instantiate a String field editor (hence the "") and load it into
         * #slotNameEditor. Note that fieldEditors.makeFor and
         * radio.initializeDOM are running in parallel since one does not
         * depend on the other's completion.
         */
        niagara.fieldEditors.makeFor("", function (editor) {
          slotNameEditor = editor;
          editor.buildAndLoad($('#slotNameEditor'));
        });
      },
      subscriber: makeSubscriber()
    });
    

    $('#addRemoveControls').trigger('create');
    
    /*
     * Whenever the user types into the slot name field editor, enable/disable 
     * the add button depending on whether an actual slot name is entered.
     */
    $('#slotNameEditor').bind('editorchange', function () {
      slotNameEditor.getSaveData(function (slotName) {
        addButton.button(slotName.length ? 'enable' : 'disable');
      });
    });
    
    /*
     * Whenever the user selects a new *Writable type spec from the select 
     * dropdown, get the value of its out slot (StatusBoolean, StatusNumeric,
     * etc) and load up the field editor for that Type. 
     */
    typeSpecSelect.change(function () {
      var typeSpec = decodeURIComponent($(this).val()),
          writable = baja.$(typeSpec),
          out = writable.getOut();
      
      niagara.fieldEditors.makeFor(out, function (editor) {
        defaultValueEditor = editor;
        editor.buildAndLoad($('#defaultValueEditor').empty());
      });
    }).trigger('change'); //trigger 'change' so it does it once on page load
    
    
    /*
     * When the add button is clicked, get the slot name and fallback values
     * from the corresponding field editors, then add a slot to our folder.
     */
    addButton.click(function () {
      var typeSpec = decodeURIComponent(typeSpecSelect.val());

      slotNameEditor.saveValue(function (slotName) {
        if (slotName) {
          defaultValueEditor.saveValue(function (fallback) {
            addRemoveFolder.add({
              slot: slotName + '?', //the ? means it will append a number if the slot already exists
              value: baja.$(typeSpec, {
                fallback: fallback.newCopy()
              })
            });
          });
        }
      });
    });

    /*
     * When a radio button is selected, enable the remove button.
     */
    contentsDiv.bind('change', function () {
      removeButton.button('enable');
    });
    
    /*
     * When the remove button is clicked, remove the slot currently selected
     * in the radio button view.
     */
    removeButton.click(function () {
      var slotName = radioView.getSelectedSlot();
      if (slotName) {
        addRemoveFolder.remove({ slot: slotName });
        removeButton.button('disable');
      }
    });
  }());
  
  
  /*
   * Set up auto-advancing of slides.
   */
  (function initPlayPause() {
    var ticket,
        playPause = $('#playPause');
    
    function pause() {
      clearInterval(ticket);
      ticket = null;
      playPause.text('play');
    }
    
    function play() {
      ticket = setInterval(function () {
        impress().next();
      }, 4000);
      playPause.text('pause');
    }
    
    playPause.click(function () {
      if (ticket) {
        pause();
      } else {
        play();
      }
    });
    
    pause();
    
    /*
     * Pause the slideshow if the user presses left/right.
     */
    $(window).bind('keyup', function (event) {
      if (event.keyCode === 37 || event.keyCode === 39) {
        pause();
      }
    });
  }());
  
}