//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

//JsLint options (see http://www.jslint.com )
/*jslint white: true, vars: true, browser: true */

// Globals for JsLint to ignore 
/*global baja, testFolder, BaseBajaObj,
         callbackify, runAndWait, errTest, verify, verifyEq, failTest, 
         describe, xdescribe, it, xit, 
         expect, beforeEach, runs, waits, waitsFor, spyOn,
         console, bajaJsPrint,
         typeMismatchError, undefinedError */

var sysTest = { version: 1.0 };

describe("sys.js", function sysTest() {
  "use strict";
  
  function deleteDummyTypes() {
    delete baja.registry.$types['testBajaScript:TestDummyTypeA'];
    delete baja.registry.$types['testBajaScript:TestDummyTypeB'];
    delete baja.registry.$types['testBajaScript:TestDummyTypeC'];
    delete baja.registry.$types['testBajaScript:TestDummyTypeD'];
    delete baja.registry.$types['testBajaScript:TestDummyTypeE'];
  }
  
  describe("baja.Cursor", function Cursor() {
    describe("constructor", function Cursor_constructor() {
      it("does nothing", function doesNothing() {
        expect(new baja.Cursor()).toBeDefined();
      });
    });
    
    describe("get()", function Cursor_get() {
      it("is not implemented", function isNotImplemented() {
        errTest(function () {
          new baja.Cursor().get();
        }, "Not implemented");
      });
    });
    
    describe("each()", function Cursor_each() {
      it("is not implemented", function isNotImplemented() {
        errTest(function () {
          new baja.Cursor().each();
        }, "Not implemented");
      });
    });
  });
  
  describe("baja.AsyncCursor", function AsyncCursor() {
    describe("constructor", function AsyncCursor_constructor() {
      it("does nothing", function doesNothing() {
        expect(new baja.AsyncCursor()).toBeDefined();
      });
      
      it("extends baja.Cursor", function extendsCursor() {
        expect(new baja.AsyncCursor().constructor.$super).toBe(baja.Cursor);
      });
    });
  });
  
  describe("baja.SyncCursor", function SyncCursor() {
    describe("constructor", function SyncCursor_constructor() {
      it("does nothing", function doesNothing() {
        expect(new baja.SyncCursor()).toBeDefined();
      });
      
      it("extends baja.Cursor", function extendsCursor() {
        expect(new baja.SyncCursor().constructor.$super).toBe(baja.Cursor);
      });
    });
    
    describe("next()", function SyncCursor_next() {
      it("is not implemented", function isNotImplemented() {
        errTest(function () {
          new baja.SyncCursor().next();
        }, "Not implemented");
      });
    });
  });
  
  describe("baja.FilterCursor", function FilterCursor() {
    function toMap(keys, values) {
      var i ,
          map = new baja.OrderedMap();
      for (i = 0; i < keys.length; i++) {
        map.put(keys[i], values[i]);
      }
      return map;
    }
    
    /*
     * Creates a FilterCursor with the specified number of elements
     * (a = 0, b = 1, c = 2, etc)
     */
    function createCursor(numberOfElements, context) {
      var keys = 'abcdefghijklmnopqrstuvwxyz'
                   .substring(0, numberOfElements).split(''),
          values = [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25]
                     .slice(0, numberOfElements);
      return new baja.FilterCursor(context || {}, toMap(keys, values));
    }
    
    function isOdd(num) { 
      return num % 2 === 1; 
    }
    
    function isMultipleOf5(num) {
       return num % 5 === 0;
    }
    
    function isFifty(num) {
      return num === 50;
    }
    
    function isPrime(num) { 
      switch (num) {
      case 1: case 2: case 3: case 5: case 7: case 11: case 13: case 17: case 19: case 23:
        return true;
      default:
        return false;
      }
    }
    
    describe("constructor", function FilterCursor_constructor() {
      it("accepts a context and ordered map", function acceptsContextOrderedMap() {
        var context = {},
            orderedMap = new baja.OrderedMap(),
            cursor = new baja.FilterCursor(context, orderedMap);
        expect(cursor).toBeDefined();
        expect(cursor.$context).toBe(context);
        expect(cursor.$orderedMap).toBe(orderedMap);
      });
      
      it("extends SyncCursor", function extendsSyncCursor() {
        expect(new baja.FilterCursor().constructor.$super).toBe(baja.SyncCursor);
      });
    });
    
    describe("next()", function FilterCursor_next() {
      it("returns true if another element was found (without filter)", function moreElementsNoFilter() {
        var cursor = createCursor(2);
        verify(cursor.next());
        verify(cursor.next());
      });
      
      it("returns false if no more elements (without filter)", function noMoreElementsNoFilter() {
        var cursor = createCursor(2);
        cursor.next();
        cursor.next();
        verify(!cursor.next());
      });
      
      it("returns true if another element was found (with filter)", function moreElementsWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        verify(cursor.next());
        verify(cursor.next());
      });
      
      it("returns false if no more elements (with filter)", function noMoreElementsWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        cursor.next();
        cursor.next();
        verify(!cursor.next());
      });
      
      it("returns false for empty cursor", function emptyCursor() {
        var cursor = createCursor(0);
        verify(!cursor.next());
      });
    });
    
    describe("get()", function FilterCursor_get() {
      it("returns the current value (without filter)", function returnsValueNoFilter() {
        var cursor = createCursor(2);
        cursor.next();
        verifyEq(cursor.get(), 0);
        cursor.next();
        verifyEq(cursor.get(), 1);
      });
      
      it("returns the current value (with filter)", function returnsValueWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        cursor.next();
        verifyEq(cursor.get(), 1);
        cursor.next();
        verifyEq(cursor.get(), 3);
      });
      
      it("returns null if next() not called", function nullIfNextNotCalled() {
        var cursor = createCursor(2);
        expect(cursor.get()).toBeNull();
      });
      
      it("returns null if past end of cursor (next() returns false)", function nullPastEnd() {
        var cursor = createCursor(2);
        cursor.next();
        cursor.next();
        cursor.next();
        expect(cursor.get()).toBeNull();
      });
    });
    
    describe("getKey()", function FilterCursor_getKey() {
      it("returns current key (without filter)", function returnsKeyNoFilter() {
        var cursor = createCursor(2);
        cursor.next();
        verifyEq(cursor.getKey(), 'a');
        cursor.next();
        verifyEq(cursor.getKey(), 'b');
      });
      
      it("returns current key (with filter)", function returnsKeyWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        cursor.next();
        verifyEq(cursor.getKey(), 'b');
        cursor.next();
        verifyEq(cursor.getKey(), 'd');
      });
      
      it("returns null if next() not called", function nullIfNextNotCalled() {
        var cursor = createCursor(2);
        expect(cursor.getKey()).toBeNull();
      });
      
      it("returns null if past end of cursor (next() returns false)", function nullPastEnd() {
        var cursor = createCursor(2);
        cursor.next();
        cursor.next();
        cursor.next();
        expect(cursor.getKey()).toBeNull();
      });
    });
    
    describe("getIndex()", function FilterCursor_getIndex() {
      it("returns the current index (without filter)", function returnsIndexNoFilter() {
        var cursor = createCursor(4);
        cursor.next(); verifyEq(cursor.getIndex(), 0);
        cursor.next(); verifyEq(cursor.getIndex(), 1);
        cursor.next(); verifyEq(cursor.getIndex(), 2);
        cursor.next(); verifyEq(cursor.getIndex(), 3);
      });
      
      it("returns the current index (with filter)", function returnsIndexWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        cursor.next(); verifyEq(cursor.getIndex(), 1);
        cursor.next(); verifyEq(cursor.getIndex(), 3);
      });
      
      it("returns -1 if next() not called", function nullIfNextNotCalled() {
        var cursor = createCursor(2);
        verifyEq(cursor.getIndex(), -1);
      });
      
      it("stops incrementing after no more elements", function stopsIncrementing() {
        var cursor = createCursor(2);
        cursor.next();
        cursor.next();
        cursor.next();
        verifyEq(cursor.getIndex(), 2);
        cursor.next();
        verifyEq(cursor.getIndex(), 2);
      });
    });
    
    describe("each()", function FilterCursor_each() {
      it("calls a function on each element", function callsOnEach() {
        var context = {},
            cursor = createCursor(4, context),
            s = "";
        
        cursor.each(function (value) {
          s += value;
          expect(this).toBe(context);
        });
        
        verifyEq(s, "0123");
      });
      
      it("stops iterating if a truthy value is returned", function stopsOnTruthy() {
        var context = {},
            cursor = createCursor(4, context),
            s = "";
        
        cursor.each(function (value) {
          s += value;
          expect(this).toBe(context);
          return value === 1;
        });
        
        verifyEq(s, "01");
      });
      
      it("dies on non-Function input", function diesOnNonFunction() {
        errTest(function () {
          new baja.FilterCursor().each(1);
        }, typeMismatchError(Number, Function));
      });
    });
    
    describe("isEmpty()", function FilterCursor_isEmpty() {
      it("returns true if no elements (without filter)", function trueIfNoElementsNoFilter() {
        verify(createCursor(0).isEmpty());
      });
      
      it("returns false if has elements (without filter)", function falseIfElementsNoFilter() {
        var cursor = createCursor(2);
        verify(!cursor.isEmpty());
      });
      
      it("returns true if no elements (with filter)", function trueIfNoElementsWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isFifty);
        verify(cursor.isEmpty());
      });
      
      it("returns false if has elements (with filter)", function falseIfElementsNoFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        verify(!cursor.isEmpty());
      });
    });
    
    describe("toArray()", function FilterCursor_toArray() {
      it("returns array of values (without filter)", function returnsArrayWithoutFilter() {
        var cursor = createCursor(4);
        verifyEq(cursor.toArray(), [0, 1, 2, 3]);
      });
      
      it("returns array of values (with filter)", function returnsArrayWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        verifyEq(cursor.toArray(), [1, 3]);
      });
    });
    
    //TODO: FilterCursor#toMap has Slot-specific functionality - refactor out to SlotCursor
    xdescribe("toMap()", function FilterCursor_toMap() {
      it("returns map of values (without filter)", function returnsMapNoFilter() {
        var cursor = createCursor(4);
        verifyEq(cursor.toMap(), { a: 0, b: 1, c: 2, d: 3 });
      });
      
      it("returns map of values (with filter)", function returnsMapWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        verifyEq(cursor.toMap(), { b: 1, d: 3 });
      });
    });
    
    describe("getSize()", function FilterCursor_getSize() {
      it("returns number of elements (without filter)", function returnsSizeNoFilter() {
        var cursor = createCursor(4);
        verifyEq(cursor.getSize(), 4);
      });
      
      it("returns number of elements (with filter)", function returnsSizeWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isOdd);
        verifyEq(cursor.getSize(), 2);
      });

      it("returns 0 if cursor is empty (without filter)", function returns0IfEmptyNoFilter() {
        verifyEq(createCursor(0).getSize(), 0);
      });
      
      it("returns 0 if cursor is empty (with filter)", function returns0IfEmptyWithFilter() {
        var cursor = createCursor(4);
        cursor.filter(isFifty);
        verifyEq(cursor.getSize(), 0);
      });
    });
    
    describe("filter()", function FilterCursor_filter() {
      it("adds a single filter", function addsFilter() {
        var cursor = createCursor(20);
        cursor.filter(isMultipleOf5);
        verifyEq(cursor.toArray(), [0,5,10,15]);
      });
      
      it("appends another filter", function appendsFilter() {
        var cursor = createCursor(20);
        cursor.filter(isOdd);
        cursor.filter(isPrime);
        verifyEq(cursor.toArray(), [1,3,5,7,11,13,17,19]);
      });
      
      it("chains (returns this)", function chains() {
        var cursor = createCursor(20);
        expect(cursor.filter(isOdd)).toBe(cursor);
      });
    });
    
    describe("first()", function FilterCursor_first() {
      it("returns the first element (without filter)", function returnsFirstNoFilter() {
        var cursor = createCursor(20);
        verifyEq(cursor.first(), 0);
      });
      
      it("returns the first element (with filter)", function returnsFirstWithFilter() {
        var cursor = createCursor(20);
        cursor.filter(isOdd);
        verifyEq(cursor.first(), 1);
      });
      
      it("returns the first element even if iterated past it", function returnsFirstEvenIfIterated() {
        var cursor = createCursor(5);
        verifyEq(cursor.first(), 0);
        cursor.next(); cursor.next(); cursor.next();
        verifyEq(cursor.first(), 0);
      });
      
      it("returns null if has no elements (without filter)", function returnsNullIfNoElementsNoFilter() {
        var cursor = createCursor(0);
        expect(cursor.first()).toBeNull();
      });
      
      it("returns null if has no elements (with filter)", function returnsNullIfNoElementsWithFilter() {
        var cursor = createCursor(10);
        cursor.filter(isFifty);
        expect(cursor.first()).toBeNull();
      });
    });
    
    describe("last()", function FilterCursor_last() {
      it("returns the last element (without filter)", function returnsLastNoFilter() {
        var cursor = createCursor(5);
        verifyEq(cursor.last(), 4);
      });
      
      it("returns the last element (with filter)", function returnsLastWithFither() {
        var cursor = createCursor(20);
        cursor.filter(isMultipleOf5);
        verifyEq(cursor.last(), 15);
      });
      
      it("returns the last element even if iterated past it", function returnsLastEvenIfIterated() {
        var cursor = createCursor(2);
        cursor.next(); cursor.next(); cursor.next();
        verifyEq(cursor.last(), 1);
      });
      
      it("returns null if has no elements (without filter)", function returnsNullIfNoElementsNoFilter() {
        var cursor = createCursor(0);
        expect(cursor.last()).toBeNull();
      });
      
      it("returns null if has no elements (with filter)", function returnsNullIfNoElementsWithFilter() {
        var cursor = createCursor(10);
        cursor.filter(isFifty);
        expect(cursor.last()).toBeNull();
      });
    });
  });
  
  describe("baja.OrderedMap", function OrderedMap() {
    describe("constructor", function OrderedMap_constructor() {
      it("creates map and array properties", function createsMapAndArray() {
        var map = new baja.OrderedMap();
        expect(map.$array).toBeDefined();
        expect(map.$map).toBeDefined();
      });
    });
    
    describe("put()", function OrderedMap_put() {
      it("pushes onto the array if map is empty", function pushesIfEmpty() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        verifyEq(map.getKeys(), [ 'key1' ]);
        verifyEq(map.$map, { key1: 'val1' });
      });
      
      it("pushes onto the array if property does not exist", function pushesIfDoesNotExist() {
        var map = new baja.OrderedMap();
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        verifyEq(map.getKeys(), [ 'key2', 'key3' ]);
        verifyEq(map.$map, { key2: 'val2', key3: 'val3' });
      });
      
      it("replaces in the array if property already exists", function replacesIfExists() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        verifyEq(map.getKeys(), [ 'key1', 'key2', 'key3' ]);
        map.put('key2', 'newVal2');
        verifyEq(map.getKeys(), [ 'key1', 'key2', 'key3' ]);
        verifyEq(map.$map, { key1: 'val1', key2: 'newVal2', key3: 'val3' });
      });
    });
    
    describe("remove()", function OrderedMap_remove() {
      it("removes the specified key", function removesKey() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        map.remove('key2');
        verifyEq(map.getKeys(), [ 'key1', 'key3' ]);
        verifyEq(map.$map, { key1: 'val1', key3: 'val3' });
      });
      
      it("returns the element removed", function returnsRemovedElement() {
        var map = new baja.OrderedMap(),
            removed;
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        removed = map.remove('key1');
        verifyEq(removed, 'val1');
      });
      
      it("returns null if map is empty", function returnsNullIfEmpty() {
        expect(new baja.OrderedMap().remove('key1')).toBeNull();
      });
      
      it("returns null if key not found", function returnsNullIfNotFound() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        expect(map.remove('key3')).toBeNull();
      });
      
      it("makes no change if key not found", function makesNoChangeIfKeyNotFound() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.remove('key3');
        verifyEq(map.getKeys(), [ 'key1', 'key2' ]);
        verifyEq(map.$map, { key1: 'val1', key2: 'val2' });
      });
      
      it("dies on non-String input", function diesOnNonString() {
        errTest(function () {
          new baja.OrderedMap().remove(123);
        }, typeMismatchError(Number, String));
      });
    });
    
    describe("contains()", function OrderedMap_contains() {
      it("returns true if map contains key", function trueIfContains() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        verify(map.contains('key1'));
      });
      
      it("returns false if map does not contain key", function falseIfNotFound() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        verify(!map.contains('key2'));
      });
      
      it("returns false if map is empty", function falseIfEmpty() {
        var map = new baja.OrderedMap();
        verify(!map.contains('key1'));
      });
      
      it("dies on non-String input", function diesOnNonString() {
        errTest(function () {
          new baja.OrderedMap().contains(123);
        }, typeMismatchError(Number, String));
      });
    });
    
    describe("get()", function OrderedMap_get() {
      it("returns the value referenced by the key", function returnsValue() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        verifyEq(map.get('key1'), 'val1');
      });
      
      it("returns null if not found", function nullIfNotFound() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        expect(map.get('key3')).toBeNull();
      });
      
      it("returns null if map is empty", function nullIfEmpty() {
        expect(new baja.OrderedMap().get('key1')).toBeNull();
      });
    });
    
    describe("rename()", function OrderedMap_rename() {
      it("renames one key to a new one", function renames() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.rename('key1', 'key2');
        verifyEq(map.get('key2'), 'val1');
      });
      
      it("deletes the old key", function deletesOld() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.rename('key1', 'key2');
        expect(map.get('key1')).toBeNull();
      });
      
      it("maintains key order", function maintainsOrder() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        map.rename('key2', 'key4');
        verifyEq(map.getKeys(), [ 'key1', 'key4', 'key3' ]);
      });
      
      it("returns true if rename was successful", function trueIfSuccessful() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        verify(map.rename('key1', 'key2'));
      });
      
      it("returns false if old name does not exist", function falseIfOldDoesNotExist() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        verify(!map.rename('key2', 'key3'));
      });
      
      it("returns false if new name already exists", function falseIfNewExists() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        verify(!map.rename('key1', 'key2'));
      });
      
      it("dies on non-String old name", function diesOnNonStringOldName() {
        errTest(function () {
          new baja.OrderedMap().rename(123, '123');
        }, typeMismatchError(Number, String));
      });
      
      it("dies on non-String new name", function diesOnNonStringNewName() {
        errTest(function () {
          new baja.OrderedMap().rename('123', 123);
        }, typeMismatchError(Number, String));
      });
    });
    
    describe("getIndex()", function OrderedMap_getIndex() {
      it("returns the numeric index of the given key", function returnsIndexOfKey() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        verifyEq(map.getIndex('key1'), 0);
        verifyEq(map.getIndex('key2'), 1);
        verifyEq(map.getIndex('key3'), 2);
      });
      
      it("returns -1 if key not found", function returnsMinusOneIfNotFound() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        verifyEq(map.getIndex('key2'), -1);
      });
      
      it("returns -1 if map is empty", function returnsMinusOneIfEmpty() {
        verifyEq(new baja.OrderedMap().getIndex('key1'), -1);
      });
      
      it("dies on non-String input", function diesOnNonString() {
        errTest(function () {
          new baja.OrderedMap().getIndex(123);
        }, typeMismatchError(Number, String));
      });
    });
    
    describe("getFromIndex()", function OrderedMap_getFromIndex() {
      it("returns the value at the given index", function returnsValue() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        verifyEq(map.getFromIndex(0), 'val1');
        verifyEq(map.getFromIndex(1), 'val2');
        verifyEq(map.getFromIndex(2), 'val3');
      });
      
      it("returns null if not found", function nullIfNotFound() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        expect(map.getFromIndex(1)).toBeNull();
      });
      
      it("returns null if map is empty", function nullIfEmpty() {
        expect(new baja.OrderedMap().getFromIndex(0)).toBeNull();
      });
      
      it("dies on non-Number input", function diesOnNonNumber() {
        errTest(function () {
          new baja.OrderedMap().getFromIndex('1');
        }, typeMismatchError(String, Number));
      });
    });
    
    describe("getKeys()", function OrderedMap_getKeys() {
      it("returns an array of keys", function returnsArrayOfKeys() {
        var map = new baja.OrderedMap();
        map.put('key3', 'val3');
        map.put('key2', 'val2');
        map.put('key1', 'val1');
        verifyEq(map.getKeys(), [ 'key3', 'key2', 'key1' ]);
      });
      
      it("returns an empty array if map is empty", function emptyIfEmpty() {
        verifyEq(new baja.OrderedMap().getKeys(), []);
      });
    });
    
    describe("getSize()", function OrderedMap_getSize() {
      it("returns number of keys", function returnsNumberOfKeys() {
        var map = new baja.OrderedMap();
        map.put('key1', 'val1');
        map.put('key2', 'val2');
        map.put('key3', 'val3');
        map.put('key4', 'val4');
        verifyEq(map.getSize(), 4);
      });
      
      it("returns 0 for an empty map", function returnsZeroIfEmpty() {
        verifyEq(new baja.OrderedMap().getSize(), 0);
      });
    });
    
    describe("sort()", function OrderedMap_sort() {
      it("sorts using the given sort function", function sortsWithGivenFunction() {
        function sortByLastCharacter(str1, str2) {
          return str1.charCodeAt(str1.length - 1) - str2.charCodeAt(str2.length - 1);
        }
        
        var map = new baja.OrderedMap();
        map.put('keyh', 'valh');
        map.put('keyy', 'valy');
        map.put('keya', 'vala');
        map.sort(sortByLastCharacter);
        verifyEq(map.getKeys(), [ 'keya', 'keyh', 'keyy' ]);
      });
      
      it("dies on non-Function input", function diesOnNonFunction() {
        errTest(function () {
          new baja.OrderedMap().sort('hi');
        }, typeMismatchError(String, Function));
      });
    });
    
    describe("getCursor()", function OrderedMap_getCursor() {
      it("returns a FilterCursor by default", function returnsFilterCursor() {
        var map = new baja.OrderedMap(),
            context = {},
            cursor = map.getCursor(context);
        expect(cursor).toBeDefined();
        expect(cursor.$context).toBe(context);
        expect(cursor.$orderedMap).toBe(map);
        expect(cursor.constructor).toBe(baja.FilterCursor);
      });
    });
  });
  
  describe("Type (inner class)", function Type_test() {
    // this is going to involve some witchcraft. for those playing along at 
    // home, please don't instantiate Types like this. this is pretty bad.
    // you're going to see me go through a lot of unnecessary pain just to
    // get an instance of a Type. in the real world, use baja.lt() or 
    // baja.importTypes() instead.
    var Type = baja.lt('baja:Value').constructor;
    
    function makeType(typeSpec, args) {
      args = args || {};
      var superType = args.superType || null,
          isAbstract = args.isAbstract || false,
          isInterface = args.isInterface || false,
          interfaces = args.interfaces || [],
          contract = args.contract || null,
          trans = args.trans || false,
          iconStr = args.iconStr || null,
          isValue = args.isValue || false,
          isSimple = args.isSimple || false,
          isSingleton = args.isSingleton || false,
          isNumber = args.isNumber || false,
          isComplex = args.isComplex || false,
          isComponent = args.isComponent || false,
          isLink = args.isLink || false,
          isAction = args.isAction || false,
          isTopic = args.isTopic || false,
          isFrozenEnum = args.isFrozenEnum || false,
          isOrdScheme = args.isOrdScheme || false;
      return new Type(typeSpec, superType, isAbstract, isInterface, interfaces, 
          contract, trans, iconStr, isValue, isSimple, isSingleton, isNumber, 
          isComplex, isComponent, isLink, isAction, isTopic, isFrozenEnum, 
          isOrdScheme);
    }
    
    function TestSimple() {}
    TestSimple.registerType('testBajaScript:TestSimple');
    TestSimple.DEFAULT = new TestSimple();
    TestSimple.prototype.make = function (arg) {
      var ts = new TestSimple();
      ts.arg = arg;
      return ts;
    };
    
    function TestSingleton() {}
    TestSingleton.registerType('testBajaScript:TestSingleton');
    TestSingleton.DEFAULT = new TestSingleton();
    TestSingleton.prototype.make = function (arg) {
      var ts = new TestSingleton();
      ts.arg = arg;
      return ts;
    };
    
    describe("equals()", function Type_equals() {
      it("returns false if the other type is null/undefined", function falseIfNull() {
        var t = makeType('baja:String');
        verify(!t.equals(null));
        verify(!t.equals(undefined));
      });
      
      it("returns false if other object not a Type", function falseIfNotType() {
        var t = makeType('baja:String');
        verify(!t.equals({}));
      });
      
      it("returns false if other type spec is different", function falseIfDifferentSpec() {
        var t1 = makeType('baja:String'),
            t2 = makeType('baja:Boolean');
        verify(!t1.equals(t2));
      });
      
      it("returns true if type specs are the same", function trueIfSameSpec() {
        var t1 = makeType('baja:String'),
            t2 = makeType('baja:String');
        verify(t1.equals(t2));
      });
    });
    
    describe("getModuleName()", function Type_getModuleName() {
      it("returns the module name", function returnsModuleName() {
        verifyEq(makeType('someModule:AThing').getModuleName(), 'someModule');
      });
    });
    
    describe("getTypeName()", function Type_getTypeName() {
      it("returns the type name", function returnsTypeName() {
        verifyEq(makeType('someModule:AThing').getTypeName(), 'AThing');
      });
    });
    
    describe("getTypeSpec()", function Type_getTypeSpec() {
      it("returns the full type spec", function returnsTypeSpec() {
        verifyEq(makeType('someModule:AThing').getTypeSpec(), 'someModule:AThing');
      });
    });
    
    describe("getInstance()", function Type_getInstance() {
      it("dies if type is an interface", function diesIfInterface() {
        errTest(function () {
          makeType('baja:IAgent', { isInterface: true }).getInstance();
        }, "Cannot call 'getInstance' on an Interface Type: baja:IAgent");
      });
      
      it("dies if type is abstract", function diesIfAbstract() {
        errTest(function () {
          makeType('baja:Simple', { isAbstract: true }).getInstance();
        }, "Cannot call 'getInstance' on an Abstract Type: baja:Simple");
      });
      
      it("returns DEFAULT field of registered constructor, if registered and Simple", function returnsDefaultRegisteredSimple() {
        var type = makeType('testBajaScript:TestSimple', { isSimple: true }),
            instance = type.getInstance();
        expect(instance).toBe(TestSimple.DEFAULT);
      });
      
      it("returns output of DEFAULT.make() with params, if registered and Simple", function returnsMakeRegisteredSimple() {
        var type = makeType('testBajaScript:TestSimple', { isSimple: true }),
            arg = {},
            instance = type.getInstance(arg);
        expect(instance.arg).toBe(arg);
      });
      
      it("returns DEFAULT field of registered constructor, if registered and Singleton", function returnsDefaultRegisteredSingleton() {
        var type = makeType('testBajaScript:TestSingleton', { isSingleton: true }),
            instance = type.getInstance();
        expect(instance).toBe(TestSingleton.DEFAULT);
      });
      
      it("returns output of DEFAULT.make() with params, if registered and Singleton", function returnsMakeRegisteredSingleton() {
        var type = makeType('testBajaScript:TestSingleton', { isSingleton: true }),
            arg = {},
            instance = type.getInstance(arg);
        expect(instance.arg).toBe(arg);
      });
      
      it("caches a version of a Simple with no registered constructor", function cachesUnregisteredSimple() {
        var type = makeType('testBajaScript:TestUnregisteredSimple', {
              isSimple: true,
              superType: baja.lt('baja:Simple')
            }),
            instance1 = type.getInstance(),
            instance2 = type.getInstance();
        expect(instance1).toBe(instance2);
      });
      
      it("returns a DefaultSimple with params, if unregistered Simple", function returnsMakeUnregisteredSimple() {
        var type = makeType('testBajaScript:TestUnregisteredSimple', {
              isSimple: true,
              superType: baja.lt('baja:Simple')
            }),
            arg = 'hi there hello',
            instance = type.getInstance(arg);
        
        verify(instance instanceof baja.DefaultSimple);
        verifyEq(instance.encodeToString(), arg);
      });
      
      it("won't cache if not a Simple", function wontCacheNonSimple() {
        var type = makeType('testBajaScript:TestComponent', {
              isComponent: true,
              superType: baja.lt('baja:Component')
            }),
            instance1 = type.getInstance(),
            instance2 = type.getInstance();
        expect(instance1).not.toBe(instance2);
      });
      
      it("dies if Type does not have a valid supertype", function diesIfNoSuperType() {
        errTest(function () {
          var type = makeType('testBajaScript:TestComponent', {
            isComponent: true
          });
          type.getInstance();
        }, 'Could not find JS Constructor for Type: testBajaScript:TestComponent');
      });
      
      it("decodes the Contract, if Complex", function decodesComplexContract() {
        spyOn(baja.bson, 'decodeComplexContract');
        
        var type = makeType('testBajaScript:TestStruct', {
              isStruct: true,
              isComplex: true,
              superType: baja.lt('baja:Struct')
            }),
            instance = type.getInstance();
        
        expect(baja.bson.decodeComplexContract).toHaveBeenCalledWith(type, instance);
      });
      
      it("sets the default ordinal, if FrozenEnum", function setsFrozenEnumOrdinal() {
        var type = baja.lt('testBajaScript:TestFrozenEnum'),
            instance = type.getInstance();
        verifyEq(instance.$ordinal, 0);
      });
    });
    
    describe("is()", function Type_is() {
      it("returns true if type spec is equal", function trueIfTypeSpecEqual() {
        var type = makeType('testBajaScript:TestComponent');
        verify(type.is('testBajaScript:TestComponent'));
      });
      
      it("returns true if type is an interface and in my set of interfaces", function trueIfTypeInInterfaces() {
        var type = makeType('testBajaScript:TestComponent', {
              interfaces: [ 
                makeType('foo:baz', { isInterface: true }) 
              ]
            }),
            other = makeType('foo:baz', { isInterface: true });
        verify(type.is(other));
      });
      
      it("returns true if type is an interface but not in my set of interfaces", function trueIfTypeInInterfaces() {
        var type = makeType('foo:bar', {
              interfaces: [ 
                makeType('foo:baz', { isInterface: true }) 
              ]
            }),
            other = makeType('foo:buzz', { isInterface: true });
        verify(!type.is(other));
      });
      
      it("short circuits to false if other is Simple and I am not", function shortCircuitsOnSimple() {
        var type1 = makeType('foo:bar', { 
              isSimple: false, 
              superType: baja.lt('baja:Struct') 
            }),
            type2 = makeType('foo:baz', { 
              isSimple: true, 
              superType: baja.lt('baja:Simple') 
            });
        
        spyOn(type1.$superType, 'is');
        verify(!type1.is(type2));
        
        //no need to check supertype
        expect(type1.$superType.is).not.toHaveBeenCalled();
      });
      
      it("short circuits to false if other is Component and I am not", function shortCircuitsOnComponent() {
        var type1 = makeType('foo:bar', {
              isComponent: false,
              superType: baja.lt('baja:Simple')
            }),
            type2 = makeType('foo:baz', {
              isComponent: true,
              superType: baja.lt('baja:Component')
            });
        
        spyOn(type1.$superType, 'is');
        verify(!type1.is(type2));
        
        //no need to check supertype
        expect(type1.$superType.is).not.toHaveBeenCalled();
      });
      
      it("short circuits to false if other is Complex and I am not", function shortCircuitsOnComplex() {
        var type1 = makeType('foo:bar', {
              isComplex: false,
              superType: baja.lt('baja:Simple')
            }),
            type2 = makeType('foo:baz', {
              isComplex: true,
              superType: baja.lt('baja:Complex')
            });
        
        spyOn(type1.$superType, 'is');
        verify(!type1.is(type2));
        
        //no need to check supertype
        expect(type1.$superType.is).not.toHaveBeenCalled();
      });
      
      it("kicks up to my supertype is() if still not sure", function callsSuperTypeIs() {
        var type1 = makeType('foo:bar', {
              superType: baja.lt('baja:Component')
            }),
            type2 = makeType('foo:baz', {
              superType: baja.lt('baja:Component')
            });

        spyOn(type1.$superType, 'is');
        verify(!type1.is(type2));
        
        //need to check supertype
        expect(type1.$superType.is).toHaveBeenCalledWith(type2);
      });
      
      it("returns false if still not sure and I have no supertype", function falseIfNoSupertype() {
        var type1 = makeType('foo:bar'),
            type2 = makeType('foo:baz');
        
        verify(!type1.is(type2));
      });
    });

    describe("toString()", function Type_toString() {
      it("returns my type spec", function returnsTypeSpec() {
        verifyEq(makeType('i:AmAType').toString(), 'i:AmAType');
      });
    });
    
    describe("getIcon()", function Type_getIcon() {
      it("creates an Icon instance from string", function createsIcon() {
        var type = makeType('foo:bar', { iconStr: 'path/to/icon' }),
            icon = type.getIcon();
        verifyEq(icon.encodeToString(), 'path/to/icon');
      });
      
      it("returns Icon default instance if no icon string", function returnsDefault() {
        var type = makeType('foo:bar'),
            icon = type.getIcon();
        expect(icon).toBe(baja.Icon.getStdObjectIcon());
      });
      
      it("caches icon created from string", function cachesIcon() {
        var type = makeType('foo:bar', { iconStr: 'new/path/to/icon' }),
            icon = type.getIcon();
        expect(type.getIcon()).toBe(icon);
      });
    });
    
    describe("hasConstructor()", function Type_hasConstructor() {
      it("returns true if type has a registered JS constructor", function trueIfRegistered() {
        function Foo() {}
        Foo.registerType('foo:hasConstructorTest1');
        var type = makeType('foo:hasConstructorTest1');
        verify(type.hasConstructor());
      });
      
      it("returns false if type has no registered JS constructor", function falseIfNotRegistered() {
        var type = makeType('foo:hasConstructorTest2');
        verify(!type.hasConstructor());
      });
    });
  });
  
  describe("BaseBajaObj", function BaseBajaObjtest() {
    describe("equals()", function BaseBajaObj_equals() {
      it("returns true if the same object", function trueIfSame() {
        var obj1 = new BaseBajaObj();
        verify(obj1.equals(obj1));
      });
      
      it("returns false if other is different object", function falseIfDifferent() {
        var obj1 = new BaseBajaObj(),
            obj2 = new BaseBajaObj();
        verify(!obj1.equals(obj2));
      });
      
      it("returns false if other is undefined/null", function falseIfNull() {
        var obj1 = new BaseBajaObj();
        verify(!obj1.equals(undefined));
        verify(!obj1.equals(null));
      });
      
      it("returns false if other is another type", function falseIfOtherType() {
        var obj1 = new BaseBajaObj();
        verify(!obj1.equals(true));
        verify(!obj1.equals(1));
        verify(!obj1.equals(""));
      });
    });
  });
  
  describe("baja.$()", function $() {
    it("creates an instance of the given type spec", function createsInstance() {
      var comp = baja.$('baja:StatusBoolean');
      verify(comp);
      verifyEq(comp.getType().getTypeSpec(), 'baja:StatusBoolean');
    });
    
    it("creates dynamic slots using properties of second arg", function createsDynamic() {
      var comp = baja.$('control:NumericWritable', { foo: 'bar' });
      verify(comp.getSlot('foo'));
      verifyEq(comp.get('foo'), 'bar');
    });
    
    it("sets frozen slots using properties of second arg", function setsFrozen() {
      var comp = baja.$('control:BooleanWritable', { 
        setMinActiveTimeOnStart: true
      });
      expect(comp.get('setMinActiveTimeOnStart')).toBe(true);
    });
    
    it("dies if given type spec was not found", function diesIfNotFound() {
      errTest(function () {
        baja.$('baja:NotARealType');
      }, "Type 'baja:NotARealType' not found");
    });
  });
  
  describe("baja.def()", function def() {
    it("returns second arg if first is undefined", function secondIfUndef() {
      var arg = {};
      expect(baja.def(undefined, arg)).toBe(arg);
    });
    
    it("returns first arg if not undefined", function firstIfDef() {
      var arg1 = {}, arg2 = {};
      expect(baja.def(arg1, arg2)).toBe(arg1);
    });
  });
  
  describe("baja.hasType()", function hasType() {
    it("returns false if object is null/undefined", function falseIfNull() {
      verify(!baja.hasType(null));
      verify(!baja.hasType(undefined));
    });
    
    it("returns false if object is a Slot", function falseIfSlot() {
      var slot = baja.$('control:NumericWritable').getSlot('in2');
      verify(!baja.hasType(slot));
    });
    
    it("returns true if object has a getType() function and no Type given", function trueIfGetType() {
      verify(baja.hasType({ getType: function () {}}));
    });
    
    it("returns true if object has getType() and matches given Type", function trueIfMatchesType() {
      verify(baja.hasType(baja.$('baja:StatusBoolean'), 'baja:StatusBoolean'));
    });
    
    it("returns false if object has getType() and does not match given Type", function falseIfDoesntMatchType() {
      verify(!baja.hasType(baja.$('baja:StatusBoolean'), 'baja:StatusNumeric'));
    });
    
    it("returns false if object has no getType() function", function falseIfNoGetType() {
      verify(!baja.hasType({}));
    });
  });
  
  describe("baja.importTypes()", function importTypes() {
    it("pulls down Types synchronously", function pullsSynchronously() {
      deleteDummyTypes();
      
      var typeSpecs = [ 
            'testBajaScript:TestDummyTypeA'
          ],
          types = baja.importTypes(typeSpecs);
      verifyEq(types.length, 1);
      verifyEq(types[0].getTypeSpec(), typeSpecs[0]);
    });
    
    it("pulls down Types asynchronously", function pullsAsynchronously() {
      deleteDummyTypes();
      
      var typeSpecs = [ 
            'testBajaScript:TestDummyTypeB'
          ],
          types;
      
      runs(function () {
        baja.importTypes({
          typeSpecs: typeSpecs,
          ok: function (returnedTypes) {
            types = returnedTypes;
          }
        });
      });
      waitsFor(function () { return types; });
      runs(function () {
        verifyEq(types.length, 1);
        verifyEq(types[0].getTypeSpec(), typeSpecs[0]);
      });
    });
  });
  
  describe("baja.iterate()", function iterate() {
    it("iterates over an array", function array() {
      var a = [], i;
      for (i = 0; i < 10; i++) {
        a.push({});
      }
      baja.iterate(a, function (obj, i) {
        obj.foo = "bar";
        obj.index = i;
      });
      for (i = 0; i < 10; i++) {
        expect(a[i].foo).toEqual("bar");
        expect(a[i].index).toEqual(i);
      }
    });
    
    it("iterates over an array with start", function arrayWithStart() {
      var a = ['a', 'b', 'c', 'd', 'e'],
          s = '';
      baja.iterate(a, 2, function (str) {
        s += str;
      });
      expect(s).toEqual('cde');
    });
    
    it("iterates over an array with range", function arrayWithRange() {
      var a = ['a', 'b', 'c', 'd', 'e'],
          s = '';
      baja.iterate(a, 1, 4, function (str) {
        s += str;
      });
      expect(s).toEqual('bcd');
    });
    
    it("iterates over an array until a truthy value is returned", function arrayUntil() {
      var a = [], i, result;
      for (i = 0; i < 10; i++) {
        a.push({val: i});
      }
      result = baja.iterate(a, function (obj) {
        if (obj.val === 5) {
          return obj;
        }
      });
      expect(result).toEqual(a[5]);
    });
    
    it("iterates over an object", function object() {
      var o = {}, i;
      for (i = 0; i < 10; i++) {
        o["prop" + i] = {};
      }
      baja.iterate(o, function (obj, name) {
        obj.foo = "bar";
        obj.name = name;
      });
      for (i = 0; i < 10; i++) {
        expect(o["prop" + i].foo).toEqual("bar");
        expect(o["prop" + i].name).toEqual("prop" + i);
      }
    });
    
    it("iterates over an object until a truthy value is returned", function objectUntil() {
      var o = {}, i, result;
      for (i = 0; i < 10; i++) {
        o["prop" + i] = {val: i};
      }
      result = baja.iterate(o, function (obj) {
        if (obj.val === 4) {
          return obj;
        }
      });
      expect(result).toEqual(o.prop4);
    });
    
    it("iterates over baja props", function bajaProps() {
      var o = baja.$('control:NumericOverride'), a = [];
      baja.iterate(o.getSlots().properties(), function (obj) {
        a.push(obj);
      });
      expect(a[0].getName()).toEqual("duration");
      expect(a[1].getName()).toEqual("maxOverrideDuration");
    });
    
    it("iterates over baja props until a truthy value is returned", function bajaPropsUntil() {
      var o = baja.$('control:NumericOverride'), result;
      result = baja.iterate(o.getSlots().properties(), function (obj) {
        if (obj.getName() === 'duration') {
          return obj;
        }
      });
      expect(result).toBeDefined();
      expect(result.getName()).toEqual('duration');
    });
      
    it("iterates over a number", function number() {
      var s = '';
      baja.iterate(10, function (i) {
        s += i;
      });
      expect(s).toEqual('0123456789');
    });
    
    it("iterates over a range", function range() {
      var s = '';
      baja.iterate(4, 9, function (i) {
        s += i;
      });
      expect(s).toEqual('45678');
    });
  });
  
  describe("baja.lt()", function lt() {
    it("returns the Type direct from the registry, if it exists", function returnsStraightFromRegistry() {
      var typeSpec = 'testBajaScript:NeverGoingToExist',
          obj = {};
      baja.registry.$types[typeSpec] = obj;
      expect(baja.lt(typeSpec)).toBe(obj);
      delete baja.registry.$types[typeSpec];
    });
    
    it("makes a synchronous call if the Type has not already been loaded", function makesSyncCall() {
      deleteDummyTypes();
      
      var typeSpec = 'testBajaScript:TestDummyTypeC',
          type = baja.lt(typeSpec);
      verify(type);
      verifyEq(type.getTypeSpec(), typeSpec);
    });
    
    it("does not load contracts if encodeContracts is false", function doesNotLoadContracts() {
      deleteDummyTypes();
      
      var typeSpec = 'testBajaScript:TestDummyTypeD',
          type = baja.lt(typeSpec, false);
      expect(type.getContract()).toBeNull();
    });
    
    it("loads contracts if encodeContracts is true", function loadsContracts() {
      deleteDummyTypes();
      
      var typeSpec = 'testBajaScript:TestDummyTypeE',
          type = baja.lt(typeSpec, true);
      expect(type.getContract()).not.toBeNull();
    });
  });
  
  describe("baja.strictArg()", function strictArg() {
    it("dies for undefined", function diesForUndefined() {
      errTest(function () {
        baja.strictArg(undefined);
      }, undefinedError());
    });
    
    it("dies with error message for undefined + constructor", function diesForUndefinedConstructor() {
      errTest(function () {
        baja.strictArg(undefined, Number);
      }, undefinedError(Number));
    });
    
    it("dies with specified error message for undefined", function diesForUndefinedErrorMessage() {
      errTest(function () {
        baja.strictArg(undefined, Number, "A specified error message");
      }, "A specified error message");
    });
    
    it("is always ok with null", function isOkWithNull() {
      expect(baja.strictArg(null)).toBeNull();
    });
    
    it("is ok with any defined value if no constructor", function isOkDefinedNoConstructor() {
      expect(baja.strictArg("hello")).toEqual("hello");
    });
    
    it("is ok if value's constructor is given constructor", function isOkGivenConstructor() {
      function foo() {}
      var arg = { constructor: foo };
      expect(baja.strictArg(arg, foo)).toBe(arg);
    });
    
    it("is ok if value instanceof constructor", function isOkInstanceof() {
      function Foo() {}
      var arg = new Foo();
      expect(baja.strictArg(arg, Foo)).toBe(arg);
    });
    
    it("is ok if value is Type", function isOkIsType() {
      var arg = baja.$('baja:StatusBoolean');
      expect(baja.strictArg(arg, baja.lt('baja:StatusBoolean'))).toBe(arg);
    });
    
    it("is ok if value is type spec", function isOkIsTypeSpec() {
      var arg = baja.$('baja:StatusNumeric');
      expect(baja.strictArg(arg, 'baja:StatusNumeric')).toBe(arg);
    });
    
    it("dies if value is not Type", function diesIfNotType() {
      var arg = baja.$('baja:StatusEnum');
      errTest(function () {
        baja.strictArg(arg, baja.lt('baja:StatusString'));
      });
    });
    
    it("dies if value is not type spec", function diesIsNotTypeSpec() {
      var arg = baja.$('baja:StatusNumeric');
      errTest(function () {
        baja.strictArg(arg, 'baja:StatusBoolean');
      });
    });
    
    it("dies if invalid type spec", function diesIfInvalidTypeSpec() {
      errTest(function () {
        baja.strictArg('hello', 'baja:TotallyNotLegitTypeSpec');
      });
    });
  });
  
  describe("baja.objectify()", function objectify() {
    it("returns a new object if obj is null/undefined", function returnsNewIfNull() {
      expect(baja.objectify(null)).toEqual({});
      expect(baja.objectify(undefined)).toEqual({});
    });
    
    it("returns obj directly if it is an Object", function returnsIfObject() {
      var arg = {},
          objectified = baja.objectify(arg, 'prop');
      expect(objectified).toBe(arg);
      expect(objectified).toEqual({});
    });
    
    it("converts obj to a property if prop name is given", function convertsToProperty() {
      var arg = 123,
          objectified = baja.objectify(arg, 'prop');
      verifyEq(objectified.prop, arg);
    });
  });
  
  describe("baja.strictAllArgs()", function strictAllArgs() {
    it("is ok if all args match", function allArgsMatch() {
      var args = [ "", 0, true, baja.$('control:BooleanWritable') ],
          ctors = [ String, Number, Boolean, baja.lt('control:BooleanWritable') ];
      baja.strictAllArgs(args, ctors);
    });
    
    it("is ok if both arrays are empty for some reason", function arraysEmpty() {
      baja.strictAllArgs([], []);
    });
    
    it("dies if any one arg does not match", function diesIfArgDoesntMatch() {
      var args = [ "", 0, true, baja.$('control:BooleanWritable') ],
          ctors = [ String, Number, Boolean, baja.lt('control:NumericWritable') ];
      errTest(function () {
        baja.strictAllArgs(args, ctors);
      });
    });
    
    it("dies if different number of args and constructors", function diesLengthMismatch() {
      var args = [ "", 0, true ],
          ctors = [ String, Number ];
      errTest(function () {
        baja.strictAllArgs(args, ctors);
      });
    });
  });
  
  //skip this for now - works fine in the browser but baja.clock doesn't play
  //nice with rhino yet
  xdescribe("baja.throttle()", function throttle() {
    it("throttles back a function", function throttles() {
      var invocs = 0,
          ticket;
      
      function increment() {
        invocs++;
      }
      
      runs(function () {
        //run every 10 secs
        ticket = setInterval(increment, 10);
      });
      
      waits(500);
      
      runs(function () {
        //ABOUT 50 times... fudge factor because javascript
        expect(invocs).toBeGreaterThan(40);
        expect(invocs).toBeLessThan(60);
        clearInterval(ticket);
        invocs = 0;
        //still run every 10ms, but throttle back to every 100ms
        setInterval(baja.throttle(increment, 100), 10);
      });
      
      waits(500);
      
      runs(function () {
        //ABOUT 5 times...
        expect(invocs).toBeGreaterThan(2);
        expect(invocs).toBeLessThan(8);
        clearInterval(ticket);
      });
    });
  });
});

(function testSys() { 
  "use strict";
        
  describe("system", function testSystem() {
  
    describe("core", function testCoreSystem() {
    
      it("array contains", function testArrayContains() { 
        var a = [];
        expect(!a.contains("test")).toBeTruthy();
        a.push("test");
        expect(a.contains("test")).toBeTruthy();
        a.push("that");
        expect(a.contains("test")).toBeTruthy();
        expect(a.contains("that")).toBeTruthy();
        expect(!a.contains("this")).toBeTruthy();
        delete a[0];
        expect(!a.contains("test")).toBeTruthy();
      }); 
      
      it("extend", function testExtend() {
        function TestInherits1() {}
        TestInherits1.prototype.foo = function () {};
        function TestInherits2() {}
        TestInherits2.$extend(TestInherits1);
        var t = new TestInherits2();
        expect(typeof t.foo === "function").toBeTruthy();
      });
    });
    
    describe("registry and types", function registeryAndTypes() {
    
      it("getInstance", function testRegistryGetInstance() {
        // Check we can create a Component
        var comp = baja.$("baja:Component");
        verify(typeof comp === "object");
        verify(comp instanceof baja.Component);
        
        // Check we cna create a Struct
        var comp2 = baja.$("baja:Struct");
        verify(typeof comp2 === "object");
        verify(comp2 instanceof baja.Struct);
        
        runs(function () {
          // Check we can create a NumericWritable
          var point = baja.$("control:NumericWritable");
          verify(point.has("in1"));
          verify(point.has("in2"));
          verify(point.has("in3"));
          verify(point.has("in4"));
          verify(point.has("in5"));
          verify(point.has("in6"));
          verify(point.has("in7"));
          verify(point.has("in8"));
          verify(point.has("in9"));
          verify(point.has("in10"));
          verify(point.has("in11"));
          verify(point.has("in12"));
          verify(point.has("in13"));
          verify(point.has("in14"));
          verify(point.has("in16"));
          verify(point.has("out"));
          verify(point.has("fallback"));
          
          // Create a point with some values preset via Object Literal notation
          var point2 = baja.$("control:NumericWritable", {
            in4: baja.$("baja:StatusNumeric", {
              value: 12.34
            }),
            in10: baja.$("baja:StatusNumeric", {
              value: 567.89,
              status: baja.Status.fault
            })
          });
          
          verifyEq(point2.getIn4().getValue(), 12.34);
          verifyEq(point2.getIn10().getValue(), 567.89);
          verifyEq(point2.getIn10().getStatus(), baja.Status.fault);
          
          // Create a Component with some values added
          var comp3 = baja.$("baja:Component", {
            test1: "some text",
            test2: "some more text",
            test3: {
              test4: true,
              test5: baja.$("baja:Weekday").get("monday")
            }
          });
          
          verify(comp3.has("test1"));
          verifyEq(comp3.get("test1"), "some text");
          
          verify(comp3.has("test2"));
          verifyEq(comp3.get("test2"), "some more text");
          
          verify(comp3.has("test3"));
          verify(comp3.get("test3").getType().isComponent());
          
          var test3 = comp3.get("test3");
          
          verify(test3.has("test4"));
          verify(test3.get("test4"));
          
          verify(test3.has("test5"));
          verify(test3.get("test5").equals(baja.$("baja:Weekday").get("monday")));
         
          verifyEq(baja.$("baja:Double", 12.34), 12.34);
          verifyEq(baja.$("baja:String", "test me"), "test me");
          verifyEq(baja.$("baja:Boolean", true), true);
          
          verify(baja.$("baja:Weekday", "tuesday"),baja.$("baja:Weekday").get("tuesday")); 
        });
        
        // Error testing (make sure we can't create these Types)
        runs(function () {
          bajaJsPrint("\n*** Error Test: please ignore the following error messages ***\n");
                    
          errTest(function () {
            baja.$("foo:bar");
          });
  
          errTest(function () {
            baja.$("baja:Complex");
          });
  
          errTest(function () {
            baja.$("baja:Interface");
          });
  
          errTest(function () {
            baja.$("baja:Value");
          });
  
          errTest(function () {
            baja.$("baja:Object");
          });
        });
        
        runs(function () {
          bajaJsPrint("\n*** End of Error Test ***");
        }); 
      });


      
      it("is", function testIs() {        
        // Test Object 'is'  
        verify(baja.lt("baja:Object").is(baja.lt("baja:Object")));
        verify(!baja.lt("baja:Object").is(baja.lt("baja:Value")));
        
        // Test Value 'is'  
        verify(baja.lt("baja:Value").is(baja.lt("baja:Object")));
        verify(baja.lt("baja:Value").is(baja.lt("baja:Value")));
        verify(!baja.lt("baja:Value").is(baja.lt("baja:Simple")));
        
        // Test Simple 'is'
        verify(baja.lt("baja:Simple").is(baja.lt("baja:Simple")));
        verify(baja.lt("baja:Simple").is(baja.lt("baja:Value")));
        verify(baja.lt("baja:Simple").is(baja.lt("baja:Object")));
        verify(baja.lt("baja:Simple").is(baja.lt("baja:IEncodable")));
        verify(!baja.lt("baja:Simple").is(baja.lt("baja:Component")));
        
        // Test Complex 'is'  
        verify(baja.lt("baja:Complex").is(baja.lt("baja:Complex")));
        verify(baja.lt("baja:Complex").is(baja.lt("baja:Value")));
        verify(baja.lt("baja:Complex").is(baja.lt("baja:Object")));
        verify(!baja.lt("baja:Complex").is(baja.lt("baja:Component")));
        
        // Test Struct 'is'  
        verify(baja.lt("baja:Struct").is(baja.lt("baja:Complex")));
        verify(baja.lt("baja:Struct").is(baja.lt("baja:Value")));
        verify(baja.lt("baja:Struct").is(baja.lt("baja:Object")));
        verify(!baja.lt("baja:Struct").is(baja.lt("baja:Component")));
        
        // Test Component 'is'
        verify(baja.lt("baja:Component").is(baja.lt("baja:Complex")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:Value")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:Object")));
        verify(!baja.lt("baja:Component").is(baja.lt("baja:Struct")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:ISpaceNode")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:IProtected")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:IPropertyContainer")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:INavNode")));
        verify(baja.lt("baja:Component").is(baja.lt("baja:ICategorizable")));
        verify(!baja.lt("baja:IPropertyContainer").is(baja.lt("baja:Component")));
        verify(!baja.lt("baja:Component").is(baja.lt("baja:IEncodable")));
        
        // Test Component is with alternate String typeSpec argument
        verify(baja.lt("baja:Component").is("baja:Complex"));
        verify(baja.lt("baja:Component").is("baja:Value"));
        verify(baja.lt("baja:Component").is("baja:Object"));
        verify(!baja.lt("baja:Component").is("baja:Struct"));
        verify(baja.lt("baja:Component").is("baja:ISpaceNode"));
        verify(baja.lt("baja:Component").is("baja:IProtected"));
        verify(baja.lt("baja:Component").is("baja:IPropertyContainer"));
        verify(baja.lt("baja:Component").is("baja:INavNode"));
        verify(baja.lt("baja:Component").is("baja:ICategorizable"));
        verify(!baja.lt("baja:IPropertyContainer").is("baja:Component"));
        verify(!baja.lt("baja:Component").is("baja:IEncodable"));      
      });
      
      it("is accessors", function testIsAccessors() {
        var num = 12;
        var bool = true;
        var str = "testme";
        var comp = new baja.Component();
        var struct = new baja.Struct();
        
        verify(str.getType().isValue());
        verify(num.getType().isValue());
        verify(bool.getType().isValue());
        verify(comp.getType().isValue());
        verify(struct.getType().isValue());
        
        verify(str.getType().isSimple());
        verify(num.getType().isSimple());
        verify(bool.getType().isSimple());
        verify(!comp.getType().isSimple());
        verify(!struct.getType().isSimple());

        verify(!str.getType().isNumber());
        verify(num.getType().isNumber());
        verify(!bool.getType().isNumber());
        verify(!comp.getType().isNumber());
        verify(!struct.getType().isNumber());
        
        verify(!str.getType().isComplex());
        verify(!num.getType().isComplex());
        verify(!bool.getType().isComplex());
        verify(comp.getType().isComplex());
        verify(struct.getType().isComplex());
        
        verify(!str.getType().isStruct());
        verify(!num.getType().isStruct());
        verify(!bool.getType().isStruct());
        verify(!comp.getType().isStruct());
        verify(struct.getType().isStruct());
        
        verify(!str.getType().isComponent());
        verify(!num.getType().isComponent());
        verify(!bool.getType().isComponent());
        verify(comp.getType().isComponent());
        verify(!struct.getType().isComponent());
      });


      it("get type", function testGetType() {
        verifyEq(baja.$("baja:Component").getType().getTypeSpec(), "baja:Component");
        verifyEq(baja.$("baja:Struct").getType().getTypeSpec(), "baja:Struct");
        
        // Error testing
        errTest(function () {
          baja.$("baja:Value");
        });
      });
      
      it("ord scheme", function testOrdScheme() { 
        verifyEq(baja.registry.getOrdScheme("slot").getTypeSpec(), "baja:SlotScheme");
      });
    });
  });
    
  // TODO: write tests for timers
        
  /*
  sysTest.testTimers = function () {
    var i = 0;
    var t = baja.clock.schedulePeriodically(function () {
      print("Timer: " + (i++));
      print("Is Expired: " + this.isExpired());
      if (i === 5) {
        this.cancel();
        print("Is Expired after cancel: " + this.isExpired());
      }
    }, 2000);
  };
  */ 
}());