//
// Copyright 2010, Tridium, Inc. All Rights Reserved.
//

//JsLint options (see http://www.jslint.com )
/*jslint white: true, vars: true */

// Globals for JsLint to ignore 
/*global baja, testFolder,
         callbackify, runAndWait, errTest, verify, verifyEq, failTest, 
         describe, it, xit, expect, beforeEach, runs, spyOn,
         console,
         typeMismatchError, undefinedError */

var objTest = { version: 1.0 };

describe("obj.js", function objTest() {
  "use strict";
  
  function createEqualOk(str) {
    return function ok(res) {
      expect(res).toEqual(str);
    };
  }
  
  describe("baja.Facets", function Facets() {
    
    function verifyFacets(facets, obj) {
      var i = 0;
      baja.iterate(obj, function (value, key) {
        verifyEq(facets.get(key).valueOf(), value.valueOf());
        i++;
      });
      verifyEq(i, facets.getKeys().length);
    }


    function FakeBajaObject(stringEncoded, dataTypeSymbol, type) {
      var Ctor = function () {
        this.getDataTypeSymbol = function () {
          return dataTypeSymbol;
        };
        this.encodeToString = function () {
          return stringEncoded;
        };
        this.decodeFromString = function (str) {
          return type.getInstance().decodeFromString(str);
        };
        this.valueOf = function () {
          return stringEncoded;
        };
      }.$extend(baja.Simple).registerType(String(type));
      return new Ctor();
    }
    
    describe("constructor", function Facets_constructor() {
      it("accepts arrays of keys and values", function acceptsKeysValues() {
        var facets = new baja.Facets([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc']);
        verifyFacets(facets, { a: 'va', b: 'vb', c: 'vc' });
      });
      
      it("accepts empty arrays", function acceptsEmptyArrays() {
        var facets = new baja.Facets([], []);
        verifyEq(facets.getKeys().length, 0);
      });
      
      it("dies if keys missing", function diesOnMissingKeys() {
        errTest(function () {
          var facets = new baja.Facets();
        }, "Keys array required");
      });
      
      it("dies if values missing", function diesOnMissingValues() {
        errTest(function () {
          var facets = new baja.Facets([]);
        }, "Values array required");
      });
      
      it("dies if unequal number of keys and values", function diesOnUnequal() {
        errTest(function () {
          var facets = new baja.Facets([ 'a' ], [ 'va', 'vb' ]);
        }, 'baja.Facets Constructor must have an equal number of keys and values');
      });
      
      it("dies if passed a non-Baja value", function diesOnNonBajaValue() {
        errTest(function () {
          var facets = new baja.Facets([ 'a' ], [ {} ]);
        }, 'Facet value has no Baja Type associated with it');
      });
      
      it("dies if passed a non-BIDataValue", function diesOnNonDataValue() {
        errTest(function () {
          var facets = new baja.Facets([ 'a' ], [ baja.$('baja:TypeSpec') ]);
        }, 'Can only add baja:IDataValue Types to BFacets: baja:TypeSpec');
      });
      
      it("dies on a non-String key", function diesOnNonStringKey() {
        errTest(function () {
          var facets = new baja.Facets([ 1 ], [ 'a' ]);
        }, 'Facets keys must be a String');
      });
      
      it("dies on invalid key names", function diesOnInvalidKeys() {
        var illegalSlot = "Illegal name for Slot: ";
        errTest(function () {
          var facets = new baja.Facets(
              ["test 1", "test2", "test3"], 
              ["this1", "this2", "this3"]);
        }, illegalSlot + "test 1");
        
        errTest(function () {
          var facets = new baja.Facets(
              [" test1", "test2", "test3"], 
              ["this1", "this2", "this3"]);
        }, illegalSlot + " test1");
        
        errTest(function () {
          var facets = new baja.Facets(
              ["test1", "2test2", "test3"], 
              ["this1", "this2", "this3"]);
        }, illegalSlot + "2test2");
        
        errTest(function () {
          var facets = new baja.Facets(
              ["test1", "test2", "3"], 
              ["this1", "this2", "this3"]);
        }, illegalSlot + "3");
      });
    });
  
    describe("make() (static)", function Facets_make() {
      it("returns baja.Facets.DEFAULT for no arg", function noArg() {
        expect(baja.Facets.make()).toBe(baja.Facets.DEFAULT);
      });

      it("returns baja.Facets.DEFAULT for empty keys", function emptyKeys() {
        expect(baja.Facets.make([], [])).toBe(baja.Facets.DEFAULT);
      });
      
      it("accepts arrays of keys and values", function acceptsKeysValues() {
        var facets = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc']);
        verifyFacets(facets, { a: 'va', b: 'vb', c: 'vc' });
      });
      
      it("extends a Facets with another", function extendsFacetsFacets() {
        var f1 = baja.Facets.make([ 'a', 'b' ], [ 'va', 'vb' ]),
            f2 = baja.Facets.make([ 'c' ], [ 'vc' ]),
            f3 = baja.Facets.make(f1, f2);
        verifyFacets(f3, { a: 'va', b: 'vb', c: 'vc' });
      });
      
      it("extends a Facets with an object", function extendsFacetsObject() {
        var f1 = baja.Facets.make([ 'a' ], [ 'va' ]),
            f2 = baja.Facets.make(f1, { b: 'vb', c: 'vc' });
        verifyFacets(f2, { a: 'va', b: 'vb', c: 'vc' });
      });
      
      it("creates from object literal", function createsFromObject() {
        var facets = baja.Facets.make({ a: 'va', b: 'vb', c: 'vc' });
        verifyFacets(facets, { a: 'va', b: 'vb', c: 'vc' });
      });
      
      it("overwrites properties on the first facet with those on the second facet", function overwritesFacetsFacets() {
        var f1 = baja.Facets.make(['key1', 'key2', 'key3'],
                                  ['f1v1', 'f1v2', 'f1v3']),
            f2 = baja.Facets.make(['key1', 'key3', 'key4'],
                                  ['f2v1', 'f2v3', 'f2v4']),
            extend1 = baja.Facets.make(f1, f2),
            extend2 = baja.Facets.make(f2, f1);

        verifyEq(extend1.get('key1'), 'f2v1');
        verifyEq(extend1.get('key2'), 'f1v2');
        verifyEq(extend1.get('key3'), 'f2v3');
        verifyEq(extend1.get('key4'), 'f2v4');
        verifyEq(extend2.get('key1'), 'f1v1');
        verifyEq(extend2.get('key2'), 'f1v2');
        verifyEq(extend2.get('key3'), 'f1v3');
        verifyEq(extend2.get('key4'), 'f2v4');
      });
      
      it("overwrites properties on the first facet with those on the second object literal", function overwritesFacetObject() {
        var f1 = baja.Facets.make(['key1', 'key2', 'key3'],
                                  ['f1v1', 'f1v2', 'f1v3']),
            f2 = {
              key1: 'f2v1',
              key3: 'f2v3',
              key4: 'f2v4'
            },
        
            extend1 = baja.Facets.make(f1, f2);
        
        verifyEq(extend1.get('key1'), 'f2v1');
        verifyEq(extend1.get('key2'), 'f1v2');
        verifyEq(extend1.get('key3'), 'f2v3');
        verifyEq(extend1.get('key4'), 'f2v4');
      });
      
      it("overwrites properties on the first object literal with those on the second object literal", function overwritesObjectObject() {
        var f1 = {
              key1: 'f1v1',
              key2: 'f1v2', 
              key3: 'f1v3'
            },

            f2 = {
              key1: 'f2v1',
              key3: 'f2v3',
              key4: 'f2v4'
            },
        
            extend1 = baja.Facets.make(f1, f2);
        
        verifyEq(extend1.get('key1'), 'f2v1');
        verifyEq(extend1.get('key2'), 'f1v2');
        verifyEq(extend1.get('key3'), 'f2v3');
        verifyEq(extend1.get('key4'), 'f2v4');
      });
      
      it("dies when no keys given", function diesIfNoKeys() {
        errTest(function () {
          baja.Facets.make(undefined);
        }, 'Keys required');
      });
      
      it("dies on missing keys", function diesOnMissingKeys() {
        errTest(function () {
          baja.Facets.make(undefined, [ 'a' ]);
        }, "Keys required");
      });
      
      it("dies if keys array given but no values array", function diesOnKeysNoValues() {
        errTest(function () {
          baja.Facets.make([ 'a' ]);
        }, "Values array required");
      });
    });
    
    describe("encodeToString()", function Facets_encodeToString() {
      it("encodes DEFAULT to empty string", function encodesDEFAULT() {
        verifyEq(baja.Facets.DEFAULT.encodeToString(), "");
      });
      
      it("encodes empty keys to empty string", function encodesEmptyKeys() {
        verifyEq(baja.Facets.make([], []).encodeToString(), "");
      });
      
      it("encodes string values", function encodesStrings() {
        var facets = baja.Facets.make(["test1", "test2", "test3"], 
                                      ["this1", "this2", "this3"]),
            expected = "test1=s:this1|test2=s:this2|test3=s:this3";
        verifyEq(facets.encodeToString(), expected);
      });
      
      it("encodes integer/boolean values", function encodesIntegerBools() {
        var facets = baja.Facets.make(
              ["test1", "test2", "test3"], 
              [baja.Integer.make(1),  baja.Integer.make(2), true]
            ),
            expected = "test1=i:1|test2=i:2|test3=b:true";
        verifyEq(facets.encodeToString(), expected);
      });
      
      it("encodes a BSimple Bajascript doesn't know about", function encodesUnknownSimple() {
        var str = "Europe/London;0;3600000;01:00:00.000,utc,march,0,undefined,sunday,last;01:00:00.000,utc,october,0,undefined,sunday,last",
            fakeTimeZone = new FakeBajaObject(str, 'z', baja.lt('baja:TimeZone')),
            facets = new baja.Facets([ 'TimeZone' ], [ fakeTimeZone ]),
            expected = 'TimeZone=z:' + str;
        verifyEq(facets.encodeToString(), expected);
      });
      
      it("encodes a blank string", function encodesBlankString() {
        var facets = new baja.Facets([ 'escalated' ], [ '' ]);
        verifyEq(facets.encodeToString(), "escalated=s:");
      });
      
      it("encodes a BIG facets object", function encodesBig() {
        var timeZoneStr = "Europe/London;0;3600000;01:00:00.000,utc,march,0," +
                          "undefined,sunday,last;01:00:00.000,utc,october,0," +
                          "undefined,sunday,last",
            expected = "fromState=s:highLimit|Count=i:6640" +
                       "|highLimit=s:$380$2e0|status=s:$7bok$7d" +
                       "|presentValue=s:$367$2e0|alarmValue=d:82.1" +
                       "|sourceName=s:OutsideAirTemp|toState=s:normal" +
                       "|lowLimit=s:$320$2e0|offnormalValue=s:$382$2e1" +
                       "|deadband=s:$30$2e0|msgText=s:" +
                       "|TimeZone=z:" + timeZoneStr + "|escalated=s:",
            facets = baja.Facets.make({
              fromState: "highLimit",
              Count: baja.Integer.make(6640),
              highLimit: "80.0",
              status: "{ok}",
              presentValue: "67.0",
              alarmValue: baja.Double.make(82.1),
              sourceName: "OutsideAirTemp",
              toState: "normal",
              lowLimit: "20.0",
              offnormalValue: "82.1",
              deadband: "0.0",
              msgText: "",
              TimeZone: new FakeBajaObject(timeZoneStr, "z", baja.lt('baja:TimeZone')),
              escalated: ""
            });
        
        verifyEq(facets.encodeToString(), expected);
      });
    });
    
    describe("decodeFromString()", function Facets_decodeFromString() {
      //context irrelevant
      var decodeFromString = baja.Facets.prototype.decodeFromString;
      
      it("decodes empty string to DEFAULT", function decodesEmptyString() {
        expect(decodeFromString("")).toBe(baja.Facets.DEFAULT);
      });
      
      it("decodes string values", function decodesStringValues() {
        var str = "test1=s:this1|test2=s:this2|test3=s:this3",
            facets = decodeFromString(str);
        verifyFacets(facets, { test1: "this1", test2: "this2", test3: "this3" });
      });
      
      it("decodes integer/boolean values", function decodesIntegerBoolValues() {
        var str = "test1=i:1|test2=i:2|test3=b:true",
            facets = decodeFromString(str);
        verifyFacets(facets, { test1: 1, test2: 2, test3: true });
      });
      
      it("decodes a BSimple Bajascript doesn't know about", function decodesUnknownBSimple() {
        var str = "Europe/London;0;3600000;01:00:00.000,utc,march,0,undefined,sunday,last;01:00:00.000,utc,october,0,undefined,sunday,last",
            encoded = 'TimeZone=z:' + str,
            decoded = decodeFromString(encoded),
            timeZone = decoded.get('TimeZone');
        
        verify(timeZone);
        verifyEq(timeZone.encodeToString(), str);
      });
      
      it("decodes a blank string", function decodesBlankString() {
        var decoded = decodeFromString("escalated=s:");
        
        verifyEq(decoded.get('escalated'), '');
      });
      
      it("decodes a BIG string", function decodesBig() {
        var timeZoneStr = "Europe/London;0;3600000;01:00:00.000,utc,march,0," +
                          "undefined,sunday,last;01:00:00.000,utc,october,0," +
                          "undefined,sunday,last",
            encoded = "fromState=s:highLimit|Count=i:6640" +
                      "|highLimit=s:$380$2e0|status=s:$7bok$7d" +
                      "|presentValue=s:$367$2e0|alarmValue=d:82.1" +
                      "|sourceName=s:OutsideAirTemp|toState=s:normal" +
                      "|lowLimit=s:$320$2e0|offnormalValue=s:$382$2e1" +
                      "|deadband=s:$30$2e0|msgText=s:" +
                      "|TimeZone=z:" + timeZoneStr + "|escalated=s:",
             facets = decodeFromString(encoded);
        
        verifyFacets(facets, {
          fromState: "highLimit",
          Count: baja.Integer.make(6640),
          highLimit: "80.0",
          status: "{ok}",
          presentValue: "67.0",
          alarmValue: baja.Double.make(82.1),
          sourceName: "OutsideAirTemp",
          toState: "normal",
          lowLimit: "20.0",
          offnormalValue: "82.1",
          deadband: "0.0",
          msgText: "",
          TimeZone: new FakeBajaObject(timeZoneStr, "z", baja.lt('baja:TimeZone')),
          escalated: ""
        });
      });
      
      it("dies on invalid name in string", function diesOnInvalidName() {
        errTest(function () {
          decodeFromString('test 1=s:this1');
        }, 'Illegal name for Slot: test 1');
      });
      
      // TODO: encoding and decoding of numbers and the handling of NaN in JavaScript is not 
      // very clear cut. There's also the performance override by any extra decoding checks
      // on such a fundamental type. Therefore, just for now, this type of silliness is allowed...
      xit("dies on invalid value in string", function diesOnInvalidValue() {
        errTest(function () {
          var facets = decodeFromString('test1=i:notAnInteger');
        });
      });
    });
    
    describe("equals()", function Facets_equals() {
      it("equals itself", function equalsItself() {
        verify(baja.Facets.DEFAULT.equals(baja.Facets.DEFAULT));
      });
      
      it("equals if keys/values are the same", function equalsKeysValues() {
        var f1 = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc ']),
            f2 = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc ']);
        verify(f1.equals(f2));
        verify(f2.equals(f1));
      });
      
      it("does not equal if any key is different", function keyDifferent() {
        var f1 = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc ']),
            f2 = baja.Facets.make([ 'a', 'b', 'X' ], [ 'va', 'vb', 'vc ']);
        verify(!f1.equals(f2));
        verify(!f2.equals(f1));
      });
      
      it("does not equal if any value is different", function valueDifferent() {
        var f1 = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vB', 'vc ']),
            f2 = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc ']);
        verify(!f1.equals(f2));
        verify(!f2.equals(f1));
      });
      
      it("does not equal null", function doesNotEqualNull() {
        verify(!baja.Facets.DEFAULT.equals(null));
      });
      
      it("does not equal undefined", function doesNotEqualUndefined() {
        verify(!baja.Facets.DEFAULT.equals(undefined));
      });
      
      it("does not equal a different type", function doesNotEqualDiffType() {
        var f1 = baja.Facets.make([ 'a', 'b', 'c' ], [ 'va', 'vb', 'vc' ]),
            str = f1.encodeToString();
        verify(!f1.equals(str));
        verify(!f1.equals({}));
      });
    });
    
    describe("newCopy()", function Facets_newCopy() {
      it("returns the same instance", function returnsSameInstance() {
        var facets = baja.Facets.make({ a: 'va', b: 'vb' }),
            copy = facets.newCopy();
        expect(facets).toBe(copy);
      });
    });
    
    describe("toString()", function Facets_toString() {
      it("returns a readable string", function returnsReadableString() {
        var facets = baja.Facets.make(
              [ 'key1', 'key2', 'key3' ],
              [ 'value1', baja.Integer.make(2), baja.Double.make(3.0) ]
            ),
            expected = "key1=value1,key2=2,key3=3";
        verifyEq(facets.toString(), expected);
      });
    });
    
    describe("getKeys()", function Facets_getKeys() {
      it("returns an array of keys in order given to constructor", function returnsArray() {
        var keys = [ 'key1', 'key2', 'key3' ],
            values = [ 'value1', 'value2', 'value3' ],
            facets = new baja.Facets(keys, values),
            gotKeys = facets.getKeys();
        verifyEq(gotKeys[0], keys[0]);
        verifyEq(gotKeys[1], keys[1]);
        verifyEq(gotKeys[2], keys[2]);
        verifyEq(gotKeys.length, 3);
      });
    });
    
    describe("get()", function Facets_get() {
      it("returns value for the given key", function returnsValue() {
        var keys = [ 'key1', 'key2', 'key3' ],
            values = [ 'value1', 2, false ],
            facets = new baja.Facets(keys, values);
        expect(facets.get('key1')).toBe('value1');
        expect(facets.get('key2')).toBe(2);
        expect(facets.get('key3')).toBe(false);
      });
      
      it("returns null if not found", function returnsNullIfNotFound() {
        expect(new baja.Facets([ 'a' ], [ 'va' ]).get('b')).toBeNull();
      });
    });
  });
  
  describe("baja.String", function bajaString() {
    describe("constructor", function String_constructor() {
      it("registers Baja Type on Javascript constructor", function registersType() {
        verifyEq("civet".getType().getTypeSpec(), "baja:String");
        verify(baja.$("baja:String") === "");
      });
    });
    
    describe("make() (static)", function String_make() {
      it("returns the given string", function returnsString() {
        verifyEq(String.make("groundhog"), "groundhog");
      });
      
      it("returns DEFAULT if no argument given", function returnsDEFAULT() {
        verifyEq(String.make(), String.DEFAULT);
        verifyEq(String.make(), "");
      });
      
      it("dies if given a non-string", function diesIfNonString() {
        errTest(function () {
          String.make(1);
        }, 'Must supply a String when making BString instances: 1');
      });
    });
    
    describe("equals()", function String_equals() {
      it("is true if the strings are the same", function sameStrings() {
        verify("".equals(""));
        verify("platypus".equals("platypus"));
        verify(!"skink".equals("cryptosporidium"));
      });
      
      it("is false for null", function falseForNull() {
        verify(!"".equals(null));
      });
      
      it("is false for undefined", function falseForUndefined() {
        verify(!"".equals(undefined));
      });
      
      it("is false for a different type", function falseForDifferentType() {
        verify(!"10".equals(10));
      });
    });
    
    describe("newCopy()", function String_newCopy() {
      it("returns the same string", function returnsSame() {
        var s = "whiptail lizard";
        verify(s.newCopy() === s);
      });
    });

    describe("encodeToString()", function String_encodeToString() {
      it("returns self", function returnsSelf() {
        var s = "badger";
        verifyEq(s.encodeToString(), s);
      });
    });
    
    describe("decodeFromString()", function String_decodeFromString() {
      it("returns the input string", function returnsInputString() {
        var s = "marmoset";
        verifyEq(s.decodeFromString("giraffe"), "giraffe");
      });
    });
    
    describe("getDataTypeSymbol()", function String_getDataTypeSymbol() {
      it("returns 's'", function returnsS() {
        verifyEq("hippopotamus".getDataTypeSymbol(), "s");
      });
    });

    describe("capitalizeFirstLetter()", function String_capitalizeFirstLetter() {
      it("capitalizes the first letter", function capsFirst() {
        verifyEq("velociraptor".capitalizeFirstLetter(), "Velociraptor");
        verifyEq("Baboon".capitalizeFirstLetter(), "Baboon");
      });
      
      it("does not change empty string", function doesNothingForEmpty() {
        verifyEq("".capitalizeFirstLetter(), "");
      });
    });
  });
  
  describe("baja.Boolean", function bajaBoolean() {
    // Workaround for JsLint
    var B = Boolean;
    
    describe("constructor", function Boolean_constructor() {
      it("creates new Boolean object", function returnsInput() {
        verifyEq(new B(true).valueOf(), true);
        verifyEq(new B(false).valueOf(), false);
      });
      
      it("registers baja:Boolean type", function registersType() {
        verifyEq(baja.$('baja:Boolean'), false);
        verifyEq(String(true.getType()), 'baja:Boolean');
      });
    });
    
    describe("getBooleanFromIBoolean (static)", function Boolean_getBooleanFromIBoolean() {
      it("returns out value from a BooleanWritable", function returnsOutFromBooleanWritable() {
        var bool = Boolean.make(true),
            writable = baja.$('control:BooleanWritable', {
              out: baja.$('baja:StatusBoolean', {
                value: bool
              })
            }),
            foundBool = Boolean.getBooleanFromIBoolean(writable);
        
        expect(foundBool).toBe(bool);
      });
      
      it("returns value from StatusBoolean", function returnsValueFromStatusBoolean() {
        var bool = Boolean.make(true),
            statusBoolean = baja.$('baja:StatusBoolean', {
              value: bool
            }),
            foundBool = Boolean.getBooleanFromIBoolean(statusBoolean);
        
        expect(foundBool).toBe(bool);
      });
      
      it("returns boolean from Boolean", function returnsBoolean() {
        var bool = Boolean.make(true),
            foundBool = Boolean.getBooleanFromIBoolean(bool);
        
        expect(foundBool).toBe(bool);
      });
      
      it("calls getBoolean, if it exists", function callsGetBoolean() {
        var bool = Boolean.make(true),
            comp = new baja.Component(),
            foundBool;
        
        comp.getBoolean = function () { return bool; };
        
        foundBool = Boolean.getBooleanFromIBoolean(comp);
        
        expect(foundBool).toBe(bool);
      });
      
      it("dies if passing in a non-Baja object", function diesIfNonBaja() {
        errTest(function () {
          Boolean.getBooleanFromIBoolean({});
        });
      });
      
      it("returns dynamic 'out' slot from baja:Component", function dynamicOut() {
        var comp = baja.$('baja:Component', {
          out: baja.$('baja:StatusBoolean')
        });
        
        comp.get("out").setValue(true);
        expect(Boolean.getBooleanFromIBoolean(comp)).toBeTruthy();
      });
    });
    
    
    describe("make() (static)", function Boolean_make() {
      it("returns input value", function returnsInput() {
        verifyEq(Boolean.make(true), true);
        verifyEq(Boolean.make(false), false);
      });
      
      it("returns DEFAULT for undefined", function returnsDEFAULT() {
        verifyEq(Boolean.make(), Boolean.DEFAULT);
      });
      
      it("dies for non-boolean", function diesForNonBoolean() {
        errTest(function () {
          Boolean.make('crocodile');
        }, 'Must supply a Boolean when making BBoolean instances: crocodile');
      });
    });

    describe("enum functions", function Boolean_enumFunctions() {
      describe("isActive()", function Boolean_isActive() {
        it("returns true if this boolean is true", function returnsThisBoolean() {
          verify(true.isActive());
          verify(!false.isActive());
        });
      });
      
      describe("getOrdinal()", function Boolean_getOrdinal() {
        it("returns 1 if this boolean is true", function returns1() {
          verifyEq(true.getOrdinal(), 1);
        });
        
        it("returns 0 if this boolean is false", function returns0() {
          verifyEq(false.getOrdinal(), 0);
        });
      });
      
      describe("getTag()", function Boolean_getTag() {
        it("returns 'true' if this boolean is true", function returnsTrue() {
          verifyEq(true.getTag(), 'true');
        });
        
        it("returns 'false' if this boolean is false", function returnsFalse() {
          verifyEq(false.getTag(), 'false');
        });
      });
      
      describe("getDisplayTag()", function Boolean_getDisplayTag() {
        it("pulls trueText/falseText from input object literal", function fromLiteral() {
          var obj = { trueText: "OH YES", falseText: "OH NO" };
          verifyEq(true.getDisplayTag(obj), "OH YES");
          verifyEq(false.getDisplayTag(obj), "OH NO");
        });
        
        it("pulls trueText/falseText from input facets", function fromFacets() {
          var facets = baja.Facets.make({ trueText: "OH YES", falseText: "OH NO" });
          verifyEq(true.getDisplayTag(facets), "OH YES");
          verifyEq(false.getDisplayTag(facets), "OH NO");
        });
        
        it("falls back to baja lexicon if not found", function fromLex() {
          var lexData = baja.lex('baja').$data,
              oldTrue = lexData['true'],
              oldFalse = lexData['false'];
          
          try {
            //sneaky!
            lexData['true'] = "new true!";
            lexData['false'] = "new false!";
            
            verifyEq(true.getDisplayTag(), 'new true!');
            verifyEq(false.getDisplayTag(), 'new false!');
          } finally {
            //tidy up
            lexData['true'] = oldTrue;
            lexData['false'] = oldFalse;
          }
        });
        
        it("passes through baja.Format", function passesThroughFormat() {
          var obj = { 
                trueText: '%lexicon(baja:january)%',
                falseText: '%lexicon(baja:february)%'
              },
              lex = baja.lex('baja');
          
          verifyEq(true.getDisplayTag(obj), lex.get('january'));
          verifyEq(false.getDisplayTag(obj), lex.get('february'));
        });
      });
      
      describe("getOrdinal()", function Boolean_getOrdinal() {
        it("returns 0 if false", function returns0() {
          verifyEq(false.getOrdinal(), 0);
        });
        
        it("returns 1 if true", function returns1() {
          verifyEq(true.getOrdinal(), 1);
        });
      });
      
      describe("getRange()", function Boolean_getRange() {
        it("returns EnumRange.BOOLEAN_RANGE", function returnsBooleanRange() {
          expect(true.getRange()).toBe(baja.EnumRange.BOOLEAN_RANGE);
        });
        
        it("returns 'false' for tag 0", function tag0() {
          verifyEq(true.getRange().getTag(0), 'false');
        });
        
        it("returns 'true' for tag 1", function tag1() {
          verifyEq(true.getRange().getTag(1), 'true');
        });
      });
    });
    
    describe("equals()", function Boolean_equals() {
      it("returns true if two primitives are the same", function twoPrims() {
        verify(true.equals(true));
        verify(!true.equals(false));
        verify(!false.equals(true));
        verify(false.equals(false));
      });
      
      it("returns true if a primitive matches an object", function primObject() {
        verify(true.equals(new B(true)));
        verify(!true.equals(new B(false)));
        verify(!false.equals(new B(true)));
        verify(false.equals(new B(false)));
      });
      
      it("returns true if an object matches a primitive", function objectPrim() {
        verify(new B(true).equals(true));
        verify(!new B(true).equals(false));
        verify(!new B(false).equals(true));
        verify(new B(false).equals(false));
      });
      
      it("returns true if an object matches an object", function objectObject() {
        verify(new B(true).equals(new B(true)));
        verify(!new B(true).equals(new B(false)));
        verify(!new B(false).equals(new B(true)));
        verify(new B(false).equals(new B(false)));
      });
      
      it("returns false for undefined", function falseForUndefined() {
        verify(!true.equals(undefined));
        verify(!false.equals(undefined));
      });
      
      it("returns false for null", function falseForNull() {
        verify(!true.equals(null));
        verify(!false.equals(null));
      });
      
      it("returns false for non-boolean", function falseForNonBoolean() {
        verify(!true.equals('true'));
        verify(!false.equals('false'));
      });
    });
    
    describe("newCopy()", function Boolean_newCopy() {
      it("returns self", function returnsSelf() {
        verifyEq(true.newCopy(), true);
        verifyEq(false.newCopy(), false);
      });
    });
    
    describe("toString()", function Boolean_toString() {
      it("returns simple string for no arguments", function returnsString() {
        verifyEq(true.toString(), 'true');
        verifyEq(false.toString(), 'false');
      });
      
      it("returns getDisplayTag() if arguments given", function returnsTag() {
        verifyEq(true.toString({ trueText: 'i am true' }), 'i am true');
      });
    });
    
    describe("getDataTypeSymbol()", function Boolean_getDataTypeSymbol() {
      it("returns 'b'", function returnsB() {
        verifyEq(true.getDataTypeSymbol(), 'b');
      });
    });
  });
  
  describe("baja.Number", function bajaNumber() {
    it("cannot be instantiated", function cannotBeInstantiated() {
      errTest(function () {
        baja.$('baja:Number');
      }, "Cannot call 'getInstance' on an Abstract Type: baja:Number");
    });
    
    describe("getNumberFromINumeric (static)", function Number_getNumberFromINumeric() {
      it("returns out value from NumericWritable", function returnsOutFromNumericWritable() {
        var num = 745,
            writable = baja.$('control:NumericWritable', {
              out: baja.$('baja:StatusNumeric', {
                value: num
              })
            }),
            foundNumber = Number.getNumberFromINumeric(writable);
        
        expect(foundNumber).toBe(num);
      });
      
      it("returns value from StatusNumeric", function returnsValueFromStatusNumeric() {
        var num = 54943,
            statusNumeric = baja.$('baja:StatusNumeric', {
              value: num
            }),
            foundNumber = Number.getNumberFromINumeric(statusNumeric);
        
        expect(foundNumber).toBe(num);
      });
      
      it("returns number value from Number", function returnsNumberValueFromNumber() {
        var num = 4363,
            dbl = baja.Double.make(num),
            foundNumber = Number.getNumberFromINumeric(dbl);
        
        expect(foundNumber).toBe(num);
      });
      
      it("calls getNumber, if it exists", function callsGetNumber() {
        var num = 438752,
            comp = new baja.Component(),
            foundNumber;
        
        comp.getNumber = function () { return num; };
        
        foundNumber = Number.getNumberFromINumeric(comp);
        
        expect(foundNumber).toBe(num);
      });
      
      it("dies if passing in a non-Baja object", function diesIfNonBaja() {
        errTest(function () {
          Number.getNumberFromINumeric({});
        });
      });
      
      it("returns value from dynamic 'out' Property on baja:Component", function returnsNumberFromDynOut() {
        var comp = baja.$('baja:Component', {
          out: baja.$('baja:StatusNumeric')
        });
        
        comp.get("out").setValue(23);
        expect(Number.getNumberFromINumeric(comp)).toEqual(23);
      });
    });
    
    describe("getNumber()", function Number_getNumber() {
      it("returns itself", function returnsItself() {
        var num = 765,
            dbl = baja.Double.make(num);
        
        expect(dbl.getNumber()).toBe(num);
      });
    });
  
    
    describe("baja.Double", function Double() {
      it("is Javascript Number", function isNumber() {
        expect(baja.Double).toBe(Number);
      });
      
      it("is registered on Type baja:Double", function isRegistered() {
        verifyEq(String(baja.Double.make(3).getType()), 'baja:Double');
      });
      
      describe("constructor", function Double_constructor() {
        it("creates a Javascript Number", function createsNumber() {
          verifyEq(new baja.Double(123.4), 123.4);
        });
      });
      
      describe("make() (static)", function Double_make() {
        it("just calls constructor", function callsConstructor() {
          verifyEq(baja.Double.make(123.4), 123.4);
        });
      });
      
      describe("getNumberFromINumeric() (static)", function Double_getNumberFromINumeric() {
        var getNumberFromINumeric = baja.Double.getNumberFromINumeric;
        
        it("returns direct value from baja:Number", function returnsFromNumber() {
          verifyEq(getNumberFromINumeric(baja.Double.make(45.6)), 45.6);
        });
        
        it("returns value slot from a StatusNumeric", function returnsStatusNum() {
          var statusNum = baja.$('baja:StatusNumeric', {
            value: 33.476
          });
          verifyEq(getNumberFromINumeric(statusNum), 33.476);
        });
        
        it("returns value from StatusNumeric out slot", function returnsStatusNumOutSlot() {
          var comp = baja.$('baja:Component', {
            out: baja.$('baja:StatusNumeric', {
              value: -3652.3
            })
          });
          
          comp.getOut = function () {
            return this.get('out');
          };
          
          verifyEq(getNumberFromINumeric(comp), -3652.3);
        });
      });

      
      describe("decodeFromString()", function Double_decodeFromString() {
        //context insensitive
        var decodeFromString = baja.Double.prototype.decodeFromString;
        
        it("decodes a numeric string", function decodesNumeric() {
          verifyEq(decodeFromString("12.546"), 12.546);
          verifyEq(decodeFromString("5"), 5);
          verifyEq(decodeFromString("3.402823669209385e+38"), 3.402823669209385e+38);
          verifyEq(decodeFromString("0xabcd"), 43981);
        });
        
        it("decodes +inf", function decodesPlusInf() {
          verifyEq(decodeFromString("+inf"), baja.Double.POSITIVE_INFINITY);
        });
        
        it("decodes -inf", function decodesMinusInf() {
          verifyEq(decodeFromString("-inf"), baja.Double.NEGATIVE_INFINITY);
        });
        
        it("decodes nan", function decodesNan() {
          verify(isNaN(decodeFromString("nan")));
        });
        
        it("decodes non-numeric to NaN", function decodesNonNumeric() {
          verify(isNaN(decodeFromString("WHARRGARBL")));
        });
      });
      
      describe("encodeToString()", function Double_encodeToString() {
        it("encodes numeric value", function encodesNumeric() {
          verifyEq(baja.Double.make(12.546).encodeToString(), '12.546');
          verifyEq(baja.Double.make(5).encodeToString(), '5');
          verifyEq(baja.Double.make(3.402823669209385e+38).encodeToString(),
              '3.402823669209385e+38');
          verifyEq(baja.Double.make(0xabcd).encodeToString(), '43981');
        });
        
        it("encodes +inf", function encodesPlusInf() {
          verifyEq(baja.Double.make(baja.Double.POSITIVE_INFINITY).encodeToString(),
              '+inf');
        });
        
        it("encodes -inf", function encodesMinusInf() {
          verifyEq(baja.Double.make(baja.Double.NEGATIVE_INFINITY).encodeToString(),
              '-inf');
        });
        
        it("encodes nan", function encodesNan() {
          verifyEq(baja.Double.make(baja.Double['NaN']).encodeToString(), 'nan');
        });
      });
      
      describe("getDataTypeSymbol()", function Double_getDataTypeSymbol() {
        it("returns 'd'", function returnsD() {
          verifyEq(baja.Double.DEFAULT.getDataTypeSymbol(), 'd');
        });
      });
      
      describe("equals()", function Double_equals() {
        it("returns true if same number", function sameNumber() {
          verify(baja.Double.make(12.546).equals(12.546));
          verify(baja.Double.make(5).equals(5));
          verify(baja.Double.make(3.402823669209385e+38)
              .equals(3.402823669209385e+38));
          verify(baja.Double.make(0xabcd).equals(43981));
        });
        
        it("returns false if different number", function differentNumber() {
          verify(!baja.Double.make(12.546).equals(12.547));
          verify(!baja.Double.make(5).equals(-5));
          verify(!baja.Double.make(3.402823669209385e+38)
                          .equals(3.402823669208385e+38));
          verify(!baja.Double.make(0xabcd).equals(43982));
        });
        
        it("returns true if both +inf", function bothPlusInf() {
          verify(baja.Double.POSITIVE_INFINITY
              .equals(baja.Double.POSITIVE_INFINITY));
        });
        
        it("returns true if both -inf", function bothPlusInf() {
          verify(baja.Double.NEGATIVE_INFINITY
              .equals(baja.Double.NEGATIVE_INFINITY));
        });
        
        it("returns false if different infs", function differentInfs() {
          verify(!baja.Double.POSITIVE_INFINITY
              .equals(baja.Double.NEGATIVE_INFINITY));
          verify(!baja.Double.NEGATIVE_INFINITY
              .equals(baja.Double.POSITIVE_INFINITY));
        });
        
        it("returns false if both nan", function bothNan() {
          verify(!baja.Double['NaN'].equals(baja.Double['NaN']));
        });

        it("returns false for different type", function differentType() {
          verify(!baja.Double.make(123).equals('123'));
        });
        
        it("returns false for undefined", function returnsFalseForUndefined() {
          verify(!baja.Double.make(123).equals(undefined));
        });
        
        it("returns false for null", function returnsFalseForNull() {
          verify(!baja.Double.make(123).equals(null));
        });
      });
      
      describe("newCopy()", function Double_newCopy() {
        it("returns self", function returnsSelf() {
          verifyEq(baja.Double.make(123).newCopy(), 123);
        });
      });
      
      describe("valueOf()", function Double_valueOf() {
        it("returns numeric value", function returnsNumeric() {
          verifyEq(baja.Double.make(124.5).valueOf(), 124.5);
          verifyEq(baja.Double.POSITIVE_INFINITY.valueOf(), 
              baja.Double.POSITIVE_INFINITY);
          verifyEq(baja.Double.NEGATIVE_INFINITY.valueOf(), 
              baja.Double.NEGATIVE_INFINITY);
          verify(isNaN(baja.Double['NaN'].valueOf()));
        });
      });
      
      describe("toString()", function Double_toString() {
        it("returns numeric string", function returnsNumeric() {
          verifyEq(baja.Double.make(765.4).toString(), '765.4');
          verifyEq(baja.Double.make(3.402823669209385e+38).toString(),
              '3.402823669209385e+38');
        });
        
        it("accepts object literal with precision", function takesObject() {
          var d = baja.Double.make(321.123456);
          verifyEq(d.toString({ precision: 1 }), '321.1');
          verifyEq(d.toString({ precision: 2 }), '321.12');
          verifyEq(d.toString({ precision: 3 }), '321.123');
          //now check for length instead of actual string because FP rounding
          verifyEq(d.toString({ precision: 4 }).length, 8);
          verifyEq(d.toString({ precision: 5 }).length, 9);
        });
        
        it("accepts Facets with precision", function takesFacets() {
          var d = baja.Double.make(9876.54321);
          
          verifyEq(d.toString(baja.Facets.make({precision: 1})), "9876.5");
          verifyEq(d.toString(baja.Facets.make({precision: 2})), "9876.54");
          verifyEq(d.toString(baja.Facets.make({precision: 3})), "9876.543");
          //now check for length instead of actual string because FP rounding
          verifyEq(d.toString(baja.Facets.make({precision: 4})).length, 9);
          verifyEq(d.toString(baja.Facets.make({precision: 5})).length, 10);
        });
      });
    });
    
    describe("baja.Float", function Float() {
      it("is registered on Type baja:Float", function isRegistered() {
        verifyEq(String(baja.Float.make(3).getType()), 'baja:Float');
      });
      
      it("is NOT Javascript Number", function isNotNumber() {
        expect(baja.Float).not.toBe(Number);
      });
      
      describe("constructor", function Float_constructor() {
        it("creates a Float", function createsNumber() {
          var f = new baja.Float(123.4);
          verifyEq(f.valueOf(), 123.4);
        });
      });
      
      describe("make() (static)", function Float_make() {
        it("just calls constructor", function callsConstructor() {
          verifyEq(baja.Float.make(123.4).valueOf(), 123.4);
        });
        
        it("dies for non-numeric input", function diesForNonNumeric() {
          errTest(function () {
            baja.Float('asdf');
          }, typeMismatchError(String, Number));
        });
      });
      
      
      describe("decodeFromString()", function Float_decodeFromString() {
        //context insensitive
        var decodeFromString = baja.Float.prototype.decodeFromString;
        
        it("decodes a numeric string", function decodesNumeric() {
          verifyEq(decodeFromString("12.546").valueOf(), 12.546);
          verifyEq(decodeFromString("5").valueOf(), 5);
        });
        
        it("decodes with less precision than Double", function lessPrecision() {
          verifyEq(decodeFromString("3.402823669209385e+38").valueOf(), 
              3.4028235e+38);
        });
        
        it("decodes +inf", function decodesPlusInf() {
          verifyEq(decodeFromString("+inf"), baja.Float.POSITIVE_INFINITY);
        });
        
        it("decodes -inf", function decodesMinusInf() {
          verifyEq(decodeFromString("-inf"), baja.Float.NEGATIVE_INFINITY);
        });
        
        it("decodes nan", function decodesNan() {
          expect(decodeFromString("nan")).toBe(baja.Float.NAN);
        });
        
        it("decodes non-numeric to NaN", function decodesNonNumeric() {
          expect(decodeFromString("NOT A FLOAT")).toBe(baja.Float.NAN);
        });
      });
      
      describe("encodeToString()", function Float_encodeToString() {
        it("encodes numeric value", function encodesNumeric() {
          verifyEq(baja.Float.make(12.546).encodeToString(), '12.546');
          verifyEq(baja.Float.make(5).encodeToString(), '5');
          verifyEq(baja.Float.make(3.402823669209385e+38).encodeToString(),
              baja.Float.MAX_VALUE.encodeToString());
          verifyEq(baja.Float.make(0xabcd).encodeToString(), '43981');
        });
        
        it("encodes +inf", function encodesPlusInf() {
          verifyEq(baja.Float.POSITIVE_INFINITY.encodeToString(),
              '+inf');
        });
        
        it("encodes -inf", function encodesMinusInf() {
          verifyEq(baja.Float.NEGATIVE_INFINITY.encodeToString(),
              '-inf');
        });
        
        it("encodes nan", function encodesNan() {
          verifyEq(baja.Float.NAN.encodeToString(), 'nan');
        });
      });
      
      describe("getDataTypeSymbol()", function Float_getDataTypeSymbol() {
        it("returns 'f'", function returnsF() {
          verifyEq(baja.Float.DEFAULT.getDataTypeSymbol(), 'f');
        });
      });
      
      describe("equals()", function Float_equals() {
        it("returns true if same number", function sameNumber() {
          verify(baja.Float.make(12.546).equals(baja.Float.make(12.546)));
          verify(baja.Float.make(5).equals(baja.Float.make(5)));
          verify(baja.Float.make(3.402823669209385e+38)
              .equals(baja.Float.make(3.402823669209385e+38)));
          verify(baja.Float.make(0xabcd).equals(baja.Float.make(43981)));
        });
        
        it("returns false if different number", function differentNumber() {
          verify(!baja.Float.make(12.546).equals(baja.Float.make(12.547)));
          verify(!baja.Float.make(5).equals(baja.Float.make(-5)));
          verify(!baja.Float.make(3.40e+38).equals(baja.Float.make(3.41e+38)));
          verify(!baja.Float.make(0xabcd).equals(baja.Float.make(43982)));
        });
        
        it("returns true if both +inf", function bothPlusInf() {
          verify(baja.Float.POSITIVE_INFINITY
              .equals(baja.Float.POSITIVE_INFINITY));
        });
        
        it("returns true if both -inf", function bothPlusInf() {
          verify(baja.Float.NEGATIVE_INFINITY
              .equals(baja.Float.NEGATIVE_INFINITY));
        });
        
        it("returns false if different infs", function differentInfs() {
          verify(!baja.Float.POSITIVE_INFINITY
              .equals(baja.Float.NEGATIVE_INFINITY));
          verify(!baja.Float.NEGATIVE_INFINITY
              .equals(baja.Float.POSITIVE_INFINITY));
        });
        
        it("returns false if both nan", function bothNan() {
          verify(!baja.Float.NAN.equals(baja.Float.NAN));
        });

        it("returns false for different type", function differentType() {
          verify(!baja.Float.make(123).equals('123'));
        });
        
        it("returns false for undefined", function returnsFalseForUndefined() {
          verify(!baja.Float.make(123).equals(undefined));
        });
        
        it("returns false for null", function returnsFalseForNull() {
          verify(!baja.Float.make(123).equals(null));
        });
      });
      
      describe("newCopy()", function Float_newCopy() {
        it("returns self", function returnsSelf() {
          var f = baja.Float.make(123);
          verify(f.newCopy() === f);
        });
      });
      
      describe("valueOf()", function Float_valueOf() {
        it("returns numeric value", function returnsNumeric() {
          verifyEq(baja.Float.make(124.5).valueOf(), 124.5);
          verifyEq(baja.Float.POSITIVE_INFINITY.valueOf(), 
              Number.POSITIVE_INFINITY);
          verifyEq(baja.Float.NEGATIVE_INFINITY.valueOf(), 
              Number.NEGATIVE_INFINITY);
          verify(isNaN(baja.Float.NAN.valueOf()));
        });
      });
      
      describe("toString()", function Float_toString() {
        it("returns numeric string", function returnsNumeric() {
          verifyEq(baja.Float.make(765.4).toString(), '765.4');
        });
        
        it("has less precision than Double", function lessPrecision() {
          verifyEq(baja.Float.make(3.402823669209385e+38).toString(),
            '3.4028235e+38');
        });
        
        it("accepts object literal with precision", function takesObject() {
          var d = baja.Float.make(321.123456);
          verifyEq(d.toString({ precision: 1 }), '321.1');
          verifyEq(d.toString({ precision: 2 }), '321.12');
          verifyEq(d.toString({ precision: 3 }), '321.123');
          //now check for length instead of actual string because FP rounding
          verifyEq(d.toString({ precision: 4 }).length, 8);
          verifyEq(d.toString({ precision: 5 }).length, 9);
        });
        
        it("accepts Facets with precision", function takesFacets() {
          var d = baja.Float.make(9876.54321);
          
          verifyEq(d.toString(baja.Facets.make({precision: 1})), "9876.5");
          verifyEq(d.toString(baja.Facets.make({precision: 2})), "9876.54");
          verifyEq(d.toString(baja.Facets.make({precision: 3})), "9876.543");
          //now check for length instead of actual string because FP rounding
          verifyEq(d.toString(baja.Facets.make({precision: 4})).length, 9);
          verifyEq(d.toString(baja.Facets.make({precision: 5})).length, 10);
        });
      });
    });

    
    describe("baja.Integer", function Integer() {
      it("is registered on Type baja:Integer", function isRegistered() {
        verifyEq(String(baja.Integer.make(3).getType()), 'baja:Integer');
      });
      
      it("is NOT Javascript Number", function isNotNumber() {
        expect(baja.Integer).not.toBe(Number);
      });
      
      describe("constructor", function Integer_constructor() {
        it("creates an Integer", function createsNumber() {
          verifyEq(new baja.Integer(123).valueOf(), 123);
        });
        
        it("DOES keep decimal points on constructor input", function keepsDec() {
          verifyEq(new baja.Integer(123.4).valueOf(), 123.4);
        });
      });
      
      describe("make() (static)", function Integer_make() {
        it("calls constructor", function callsConstructor() {
          verifyEq(baja.Integer.make(123).valueOf(), 123);
        });
        
        it("returns DEFAULT for 0", function returnsDefaultFor0() {
          expect(baja.Integer.make(0)).toBe(baja.Integer.DEFAULT);
        });
        
        it("returns DEFAULT for NaN", function returnsDefaultForNaN() {
          expect(baja.Integer.make(NaN)).toBe(baja.Integer.DEFAULT);
        });
        
        it("returns MAX_VALUE for max value", function returnsMax() {
          expect(baja.Integer.make(baja.Integer.MAX_VALUE.valueOf() - 1))
                  .not.toBe(baja.Integer.MAX_VALUE);
          expect(baja.Integer.make(baja.Integer.MAX_VALUE.valueOf()))
                  .toBe(baja.Integer.MAX_VALUE);
          expect(baja.Integer.make(baja.Integer.MAX_VALUE.valueOf() + 1))
                  .toBe(baja.Integer.MAX_VALUE);
        });
        
        it("returns MIN_VALUE for min value", function returnsMin() {
          expect(baja.Integer.make(baja.Integer.MIN_VALUE.valueOf() + 1))
                  .not.toBe(baja.Integer.MIN_VALUE);
          expect(baja.Integer.make(baja.Integer.MIN_VALUE.valueOf()))
                  .toBe(baja.Integer.MIN_VALUE);
          expect(baja.Integer.make(baja.Integer.MIN_VALUE.valueOf()))
                  .toBe(baja.Integer.MIN_VALUE);
        });
        
        it("truncates decimal points from input", function truncateDecimal() {
          verifyEq(baja.Integer.make(123.1).valueOf(), 123);
          verifyEq(baja.Integer.make(123.9).valueOf(), 123);
        });
        
        it("dies for non-numeric input", function diesForNonNumeric() {
          errTest(function () {
            baja.Integer('asdf');
          }, typeMismatchError(String, Number));
        });
      });
      
      
      describe("decodeFromString()", function Integer_decodeFromString() {
        //context insensitive
        var decodeFromString = baja.Integer.prototype.decodeFromString;
        
        it("decodes a numeric string", function decodesNumeric() {
          verifyEq(decodeFromString("5").valueOf(), 5);
        });
        
        it("truncates decimal points from numeric input", function truncateDec() {
          verifyEq(decodeFromString("12.546").valueOf(), 12);
        });
        
        it("decodes max", function decodesMax() {
          verifyEq(decodeFromString("max"), baja.Integer.MAX_VALUE);
        });
        
        it("decodes min", function decodesMin() {
          verifyEq(decodeFromString("min"), baja.Integer.MIN_VALUE);
        });
        
        it("decodes non-numeric to DEFAULT", function decodesNonNumeric() {
          expect(decodeFromString("NOT an Integer")).toBe(baja.Integer.DEFAULT);
        });
      });
      
      describe("encodeToString()", function Integer_encodeToString() {
        it("encodes numeric value", function encodesNumeric() {
          verifyEq(baja.Integer.make(5).encodeToString(), '5');
        });
        
        it("truncates decimal points from output", function truncateDec() {
          verifyEq(new baja.Integer(12.546).encodeToString(), '12');
        });
        
        it("encodes max", function encodesMax() {
          verifyEq(baja.Integer.MAX_VALUE.encodeToString(), 'max');
        });
        
        it("encodes min", function encodesMin() {
          verifyEq(baja.Integer.MIN_VALUE.encodeToString(), 'min');
        });

        it("encodes non-numeric to 0", function encodesNaN() {
          verifyEq(baja.Integer.make('NaN').encodeToString(), '0');
        });
      });
      
      describe("getDataTypeSymbol()", function Integer_getDataTypeSymbol() {
        it("returns 'i'", function returnsI() {
          verifyEq(baja.Integer.DEFAULT.getDataTypeSymbol(), 'i');
        });
      });
      
      describe("equals()", function Integer_equals() {
        it("returns true if same number", function sameNumber() {
          verify(baja.Integer.make(12.546).equals(baja.Integer.make(12.546)));
          verify(baja.Integer.make(5).equals(baja.Integer.make(5)));
          verify(baja.Integer.make(3.402823669209385e+38)
              .equals(baja.Integer.make(3.402823669209385e+38)));
          verify(baja.Integer.make(0xabcd).equals(baja.Integer.make(43981)));
        });
        
        it("returns false if different number", function differentNumber() {
          verify(!baja.Integer.make(5).equals(-5));
          verify(!baja.Integer.make(0xabcd).equals(43982));
        });
        
        it("returns true if both max", function bothMax() {
          verify(baja.Integer.MAX_VALUE.equals(baja.Integer.MAX_VALUE));
        });
        
        it("returns true if both min", function bothPlusInf() {
          verify(baja.Integer.MIN_VALUE.equals(baja.Integer.MIN_VALUE));
        });
        
        it("returns false if one max one mine", function differentInfs() {
          verify(!baja.Integer.MAX_VALUE
              .equals(baja.Integer.MIN_VALUE));
          verify(!baja.Integer.MIN_VALUE
              .equals(baja.Integer.MAX_VALUE));
        });
        
        it("returns false for different type", function differentType() {
          verify(!baja.Integer.make(123).equals('123'));
        });
        
        it("returns false for undefined", function returnsFalseForUndefined() {
          verify(!baja.Integer.make(123).equals(undefined));
        });
        
        it("returns false for null", function returnsFalseForNull() {
          verify(!baja.Integer.make(123).equals(null));
        });
      });
      
      describe("newCopy()", function Integer_newCopy() {
        it("returns self", function returnsSelf() {
          var i = baja.Integer.make(123);
          verify(i.newCopy() === i);
        });
      });
      
      describe("valueOf()", function Integer_valueOf() {
        it("returns numeric value", function returnsNumeric() {
          verifyEq(baja.Integer.make(124).valueOf(), 124);
          verifyEq(baja.Integer.MAX_VALUE.valueOf(), 2147483647);
          verifyEq(baja.Integer.MIN_VALUE.valueOf(), -2147483648);
        });
      });
      
      describe("toString()", function Integer_toString() {
        it("returns numeric string", function returnsNumeric() {
          verifyEq(baja.Integer.make(765).toString(), '765');
        });
      });
    });
    
    describe("baja.Long", function Long() {
      it("is registered on Type baja:Long", function isRegistered() {
        verifyEq(String(baja.Long.make(3).getType()), 'baja:Long');
      });
      
      it("is NOT Javascript Number", function isNotNumber() {
        expect(baja.Long).not.toBe(Number);
      });
      
      describe("constructor", function Long_constructor() {
        it("creates an Long", function createsNumber() {
          verifyEq(new baja.Long(123).valueOf(), 123);
        });
        
        it("DOES keep decimal points on constructor input", function keepsDec() {
          verifyEq(new baja.Long(123.4).valueOf(), 123.4);
        });
      });
      
      describe("make() (static)", function Long_make() {
        it("calls constructor", function callsConstructor() {
          verifyEq(baja.Long.make(123).valueOf(), 123);
        });
        
        it("returns DEFAULT for 0", function returnsDefaultFor0() {
          expect(baja.Long.make(0)).toBe(baja.Long.DEFAULT);
        });
        
        it("returns DEFAULT for NaN", function returnsDefaultForNaN() {
          expect(baja.Long.make(NaN)).toBe(baja.Long.DEFAULT);
        });
        
        it("returns MAX_VALUE for max value", function returnsMax() {
          expect(baja.Long.make(baja.Long.MAX_VALUE.valueOf() - 10000))
                  .not.toBe(baja.Long.MAX_VALUE);
          expect(baja.Long.make(baja.Long.MAX_VALUE.valueOf()))
                  .toBe(baja.Long.MAX_VALUE);
        });
        
        it("returns MIN_VALUE for min value", function returnsMin() {
          expect(baja.Long.make(baja.Long.MIN_VALUE.valueOf() + 10000))
                  .not.toBe(baja.Long.MIN_VALUE);
          expect(baja.Long.make(baja.Long.MIN_VALUE.valueOf()))
                  .toBe(baja.Long.MIN_VALUE);
        });
        
        it("truncates decimal points from input", function truncateDecimal() {
          verifyEq(baja.Long.make(123.1).valueOf(), 123);
          verifyEq(baja.Long.make(123.9).valueOf(), 123);
        });
        
        it("dies for non-numeric input", function diesForNonNumeric() {
          errTest(function () {
            baja.Long('asdf');
          }, typeMismatchError(String, Number));
        });
      });
      
      
      describe("decodeFromString()", function Long_decodeFromString() {
        //context insensitive
        var decodeFromString = baja.Long.prototype.decodeFromString;
        
        it("decodes a numeric string", function decodesNumeric() {
          verifyEq(decodeFromString("5").valueOf(), 5);
        });
        
        it("truncates decimal points from numeric input", function truncateDec() {
          verifyEq(decodeFromString("12.546").valueOf(), 12);
        });
        
        it("decodes max", function decodesMax() {
          verifyEq(decodeFromString("max"), baja.Long.MAX_VALUE);
        });
        
        it("decodes min", function decodesMin() {
          verifyEq(decodeFromString("min"), baja.Long.MIN_VALUE);
        });
        
        it("decodes non-numeric to DEFAULT", function decodesNonNumeric() {
          expect(decodeFromString("NOT a Long")).toBe(baja.Long.DEFAULT);
        });
      });
      
      describe("encodeToString()", function Long_encodeToString() {
        it("encodes numeric value", function encodesNumeric() {
          verifyEq(baja.Long.make(5).encodeToString(), '5');
        });
        
        it("truncates decimal points from output", function truncateDec() {
          verifyEq(new baja.Long(12.546).encodeToString(), '12');
        });
        
        it("encodes max", function encodesMax() {
          verifyEq(baja.Long.MAX_VALUE.encodeToString(), 'max');
        });
        
        it("encodes min", function encodesMin() {
          verifyEq(baja.Long.MIN_VALUE.encodeToString(), 'min');
        });

        it("encodes non-numeric to 0", function encodesNaN() {
          verifyEq(baja.Long.make('NaN').encodeToString(), '0');
        });
      });
      
      describe("getDataTypeSymbol()", function Long_getDataTypeSymbol() {
        it("returns 'l'", function returnsL() {
          verifyEq(baja.Long.DEFAULT.getDataTypeSymbol(), 'l');
        });
      });
      
      describe("equals()", function Long_equals() {
        it("returns true if same number", function sameNumber() {
          verify(baja.Long.make(12.546).equals(baja.Long.make(12.546)));
          verify(baja.Long.make(5).equals(baja.Long.make(5)));
          verify(baja.Long.make(3.402823669209385e+38)
              .equals(baja.Long.make(3.402823669209385e+38)));
          verify(baja.Long.make(0xabcd).equals(baja.Long.make(43981)));
        });
        
        it("returns false if different number", function differentNumber() {
          verify(!baja.Long.make(5).equals(-5));
          verify(!baja.Long.make(0xabcd).equals(43982));
        });
        
        it("returns true if both max", function bothMax() {
          verify(baja.Long.MAX_VALUE.equals(baja.Long.MAX_VALUE));
        });
        
        it("returns true if both min", function bothPlusInf() {
          verify(baja.Long.MIN_VALUE.equals(baja.Long.MIN_VALUE));
        });
        
        it("returns false if one max one mine", function differentInfs() {
          verify(!baja.Long.MAX_VALUE
              .equals(baja.Long.MIN_VALUE));
          verify(!baja.Long.MIN_VALUE
              .equals(baja.Long.MAX_VALUE));
        });
        
        it("returns false for different type", function differentType() {
          verify(!baja.Long.make(123).equals('123'));
        });
        
        it("returns false for undefined", function returnsFalseForUndefined() {
          verify(!baja.Long.make(123).equals(undefined));
        });
        
        it("returns false for null", function returnsFalseForNull() {
          verify(!baja.Long.make(123).equals(null));
        });
      });
      
      describe("newCopy()", function Long_newCopy() {
        it("returns self", function returnsSelf() {
          var i = baja.Long.make(123);
          verify(i.newCopy() === i);
        });
      });
      
      describe("valueOf()", function Long_valueOf() {
        it("returns numeric value", function returnsNumeric() {
          verifyEq(baja.Long.make(124).valueOf(), 124);
        });
      });
      
      describe("toString()", function Long_toString() {
        it("returns numeric string", function returnsNumeric() {
          verifyEq(baja.Long.make(765).toString(), '765');
        });
      });
    });
  });

  describe("baja.DefaultSimple", function DefaultSimple() {
    describe("constructor", function DefaultSimple_constructor() {
      it("sets this.$val to input string", function sets$val() {
        verifyEq(new baja.DefaultSimple("panda").$val, "panda");
      });
      
      it("defaults to empty string for null/undefined", function defaultsToEmptyString() {
        verifyEq(new baja.DefaultSimple(undefined).$val, "");
        verifyEq(new baja.DefaultSimple(null).$val, "");
      });
      
      it("dies on non-string input", function diesOnNonString() {
        errTest(function () {
          var ds = new baja.DefaultSimple(123);
        });
      });
      
      it("is the default constructor for an unknown BSimple e.g. CategoryMask", function isDefaultSimpleConstructor() {
        // TODO: This won't work anyone once BCategoryMask has actually be defined as a Type :!))))))   
        verify(!baja.lt("baja:CategoryMask").hasConstructor());
        
        var val = baja.$("baja:CategoryMask");
        verifyEq(val.getType().getTypeSpec(), "baja:CategoryMask");
        verify(val.constructor === baja.DefaultSimple);
        
        val = baja.$("baja:CategoryMask").make("test");
        verifyEq(val.valueOf(), "test");
      });
    });
    
    describe("make()", function DefaultSimple_make() {
      it("returns result of this.decodeFromString()", function proxies() {
        var ds = new baja.DefaultSimple("wombat"),
            newDs;
        spyOn(ds, 'decodeFromString').andCallThrough();
        newDs = ds.make("echidna");
        expect(ds.decodeFromString).toHaveBeenCalledWith("echidna");
        verifyEq(newDs.$val, "echidna");
      });
    });
    
    describe("decodeFromString()", function DefaultSimple_decodeFromString() {
      it("returns a new instance", function returnsNewInstance() {
        var ds = new baja.DefaultSimple("gopher"),
            newDs = ds.decodeFromString("banana spider");
        expect(newDs).not.toBe(ds);
        verifyEq(newDs.$val, "banana spider");
      });
      
      it("sets getType on new instance to this.getType", function setsGetType() {
        var ds = new baja.DefaultSimple("canary"),
            newDs,
            getType = function () { baja.outln("hi there"); };
        
        ds.getType = getType;
        newDs = ds.decodeFromString("dingo");
        expect(newDs.getType).toBe(getType);
      });
    });
    
    describe("encodeToString()", function DefaultSimple_encodeToString() {
      it("returns my string representation", function returnsString() {
        verifyEq(new baja.DefaultSimple("millipede").encodeToString(), "millipede");
      });
    });
    
    describe("valueOf()", function DefaultSimple_valueOf() {
      it("returns my string representation", function returnsString() {
        verifyEq(new baja.DefaultSimple("penguin").valueOf(), "penguin");
      });
    });
    
    describe("toString()", function DefaultSimple_toString() {
      it("returns my string representation", function returnsString() {
        verifyEq(new baja.DefaultSimple("rhinoceros").toString(), "rhinoceros");
      });
    });
    
    describe("equals()", function DefaultSimple_equals() {
      it("returns true if both string representations are the same", function sameString() {
        verify(new baja.DefaultSimple("gorilla").equals(
            new baja.DefaultSimple("gorilla")));
        verify(!new baja.DefaultSimple("hamster").equals(
            new baja.DefaultSimple("vampire bat")));
      });
      
      it("returns false for a different Type", function differentType() {
        verify(!new baja.DefaultSimple("lemur").equals(
            baja.Double.make(123)));
      });
      
      it("returns false for undefined/null", function falseForUndefinedNull() {
        verify(!new baja.DefaultSimple("orangutan").equals(undefined));
        verify(!new baja.DefaultSimple("sumatran tiger").equals(null));
      });
    });
  });
  
  describe("baja.EnumRange", function EnumRange() {
    describe("make() (static)", function EnumRange_make() {
      it("accepts array of ordinals and array of tags", function ordinalsTags() {
        var er = baja.EnumRange.make({
          ordinals: [2, 4, 6],
          tags: [ 'tag2', 'tag4', 'tag6']
        });
        
        verifyEq(er.getOrdinals().length, 3);
        verifyEq(er.getTag(2), 'tag2');
        verifyEq(er.getTag(4), 'tag4');
        verifyEq(er.getTag(6), 'tag6');
      });
      
      it("limits length using count param", function countParam() {
        var er = baja.EnumRange.make({
          ordinals: [1, 3, 5],
          tags: [ 'tag1', 'tag3', 'tag5' ],
          count: 2
        });
        
        verifyEq(er.getOrdinals().length, 2);
        verifyEq(er.getTag(1), 'tag1');
        verifyEq(er.getTag(3), 'tag3');
        verifyEq(er.getTag(5), '5');
      });
      
      it("accepts a frozen type spec", function acceptsFrozen() {
        var er = baja.EnumRange.make('baja:Weekday');
        verifyEq(er.getOrdinals().length, 7);
        verifyEq(er.getTag(0), 'sunday');
        verifyEq(er.getTag(6), 'saturday');
      });
      
      it("allows a frozen type spec plus additional ordinals/tags", function acceptsFrozenPlus() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 7, 8 ],
          tags: [ 'schmaturday', 'schmoozeday' ]
        });
        
        verifyEq(er.getOrdinals().length, 9);
        verifyEq(er.getTag(0), 'sunday');
        verifyEq(er.getTag(6), 'saturday');
        verifyEq(er.getTag(7), 'schmaturday');
        verifyEq(er.getTag(8), 'schmoozeday');
      });
      
      it("won't allow overwriting of a frozen tag", function wontOverwriteFrozen() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 0 ],
          tags: [ 'schmunday' ]
        });
        verifyEq(er.getOrdinals().length, 7);
        verifyEq(er.getTag(0), 'sunday');
      });
      
      it("dies on duplicate tag", function dupTag() {
        errTest(function () {
          baja.EnumRange.make({
            ordinals: [ 0, 1 ],
            tags: [ 'foo', 'foo' ]
          });
        }, 'Duplicate tag: foo=1');
      });
      
      it("dies on duplicate ordinal", function dupOrdinal() {
        errTest(function () {
          baja.EnumRange.make({
            ordinals: [ 6, 6 ],
            tags: [ 'yes', 'no' ]
          });
        }, 'Duplicate ordinal: no=6');
      });
      
      it("dies if adding duplicate tag to frozen", function dupTagToFrozen() {
        errTest(function () {
          baja.EnumRange.make({
            frozen: 'baja:Weekday',
            ordinals: [ 7 ],
            tags: [ 'wednesday' ]
          });
        }, 'Duplicate tag: wednesday=7');
      });
      
      it("returns DEFAULT when no params", function noParamsDefault() {
        expect(baja.EnumRange.make()).toBe(baja.EnumRange.DEFAULT);
      });
      
      it("returns DEFAULT when ordinals/tags empty", function emptyDefault() {
        expect(baja.EnumRange.make({ ordinals: [], tags: [] }))
          .toBe(baja.EnumRange.DEFAULT);
      });
      
      it("dies on more tags than ordinals", function moreTagsThanOrdinals() {
        errTest(function () {
          baja.EnumRange.make({
            ordinals: [ 1, 2 ],
            tags: [ 'tag1', 'tag2', 'tag3' ]
          });
        }, "Ordinals and tags arrays must match in length");
      });
      
      it("dies on more ordinals than tags", function moreOrdinalsThanTags() {
        errTest(function () {
          baja.EnumRange.make({
            ordinals: [ 1, 2, 3 ],
            tags: [ 'tag1', 'tag2' ]
          });
        }, "Ordinals and tags arrays must match in length");
      });
      
      it("dies on invalid tag name", function invalidTagName() {
        errTest(function () {
          baja.EnumRange.make({
            ordinals: [ 0 ],
            tags: [ 'gots spaces' ]
          });
        }, 'Illegal name for Slot: gots spaces');
      });
    });
    
    describe("decodeFromString()", function EnumRange_decodeFromString() {
      var er = baja.EnumRange.DEFAULT;
      
      it("decodes '{}' to DEFAULT", function decodesDefault() {
        expect(er.decodeFromString('{}')).toBe(baja.EnumRange.DEFAULT);
      });
      
      it("decodes a simple dynamic range", function decodesSimpleDynamic() {
        var decoded = er.decodeFromString("{x=0,y=1,z=2}");
        verifyEq(decoded.getOrdinals().length, 3);
        verifyEq(decoded.getTag(0), 'x');
        verifyEq(decoded.getTag(1), 'y');
        verifyEq(decoded.getTag(2), 'z');
      });
      
      it("decodes a frozen enum type spec", function decodesFrozenTypeSpec() {
        var decoded = er.decodeFromString("baja:Weekday");
        verifyEq(decoded.getOrdinals().length, 7);
        verifyEq(decoded.getTag(0), 'sunday');
        verifyEq(decoded.getTag(6), 'saturday');
      });
      
      it("decodes frozen + dynamic", function decodesFrozenDynamic() {
        var decoded = er.decodeFromString("baja:Weekday+{bumblebee=7}");
        verifyEq(decoded.getOrdinals().length, 8);
        verifyEq(decoded.getTag(6), 'saturday');
        verifyEq(decoded.getTag(7), 'bumblebee');
      });
      
      it("decodes dynamic + frozen", function decodesDynamicFrozen() {
        var decoded = er.decodeFromString("{mongoose=7}+baja:Weekday");
        verifyEq(decoded.getOrdinals().length, 8);
        verifyEq(decoded.getTag(0), 'sunday');
        verifyEq(decoded.getTag(6), 'saturday');
        verifyEq(decoded.getTag(7), 'mongoose');
      });
      
      it("decodes options facets", function decodesWithOptions() {
        var decoded = er.decodeFromString("{x=0,y=1}?facet1=s:this1|facet2=i:5"),
            options = decoded.getOptions();
        verifyEq(decoded.getOrdinals().length, 2);
        verifyEq(decoded.getTag(0), 'x');
        verifyEq(decoded.getTag(1), 'y');
        verifyEq(options.get('facet1'), 'this1');
        verifyEq(options.get('facet2').valueOf(), 5);
      });
      
      it("dies on invalid tag", function diesOnInvalidTag() {
        errTest(function () {
          er.decodeFromString("{x=0,bad tag=1}");
        }, 'Illegal name for Slot: bad tag');
      });
      
      it("dies on missing {", function diesOnMissingLeftBrace() {
        errTest(function () {
          er.decodeFromString("baja:Weekday+x=0,y=1}");
        }, 'Missing {');
      });
      
      it("dies on missing }", function diesOnMissingRightBrace() {
        errTest(function () {
          er.decodeFromString("{x=0,y=1");
        }, 'Missing }');
      });
      
      it("dies on unknown frozen type spec", function diesOnUnknownTypeSpec() {
        errTest(function () {
          er.decodeFromString("{x=0,y=1}+baja:Schweekday");
        }, 'Invalid frozen EnumRange spec: baja:Schweekday');
      });
      
      it("dies on invalid ordinal", function diesOnInvalidOrdinal() {
        errTest(function () {
          er.decodeFromString("{x=0,y=A}");
        }, 'Invalid ordinal: A');
      });
    });
    
    describe("encodeToString()", function EnumRange_encodeToString() {
      it("encodes a dynamic enum range to string", function encodesDynamic() {
        var er = baja.EnumRange.make({
              ordinals: [ 3, 4, 5 ],
              tags: [ 'tag3', 'tag4', 'tag5' ]
            }),
            expected = "{tag3=3,tag4=4,tag5=5}";
        verifyEq(er.encodeToString(), expected);
      });
      
      it("encodes a frozen enum to string", function encodesFrozen() {
        var er = baja.EnumRange.make('baja:Weekday'),
            expected = 'baja:Weekday';
        verifyEq(er.encodeToString(), expected);
      });
      
      it("encodes a dynamic + frozen enum", function encodesDynamicFrozen() {
        var er = baja.EnumRange.make({
              frozen: 'baja:Weekday',
              ordinals: [ 7, 8 ],
              tags: [ 'tag7', 'tag8' ]
            }),
            expected = "baja:Weekday+{tag7=7,tag8=8}";
        verifyEq(er.encodeToString(), expected);
      });
      
      it("encodes with options facets", function encodesOptions() {
        var options = baja.Facets.make({
              option1: 'option1',
              option2: 'option2'
            }),
            facets = baja.EnumRange.make({
              ordinals: [ 0, 1 ],
              tags: [ 'tag0', 'tag1' ],
              options: options
            }),
            expected = "{tag0=0,tag1=1}?option1=s:option1|option2=s:option2";
        verifyEq(facets.encodeToString(), expected);
      });
    });
    
    describe("get()", function EnumRange_get() {
      var frozenRange = baja.EnumRange.make('baja:Weekday'),
          dynamicRange = baja.EnumRange.make({
            ordinals: [ 0, 1, 2 ],
            tags: [ 'tag0', 'tag1', 'tag2' ]
          });
      
      it("returns enum from a frozen range for string", function frozenString() {
        var sunday = frozenRange.get('sunday'),
            saturday = frozenRange.get('saturday');
        verifyEq(String(sunday.getType()), 'baja:Weekday');
        verifyEq(sunday.getTag(), 'sunday');
        verifyEq(sunday.getOrdinal(), 0);
        verifyEq(saturday.getTag(), 'saturday');
        verifyEq(saturday.getOrdinal(), 6);
      });
      
      it("returns enum from a dynamic range for string", function dynamicString() {
        var enum0 = dynamicRange.get('tag0'),
            enum1 = dynamicRange.get('tag1');
        
        verifyEq(String(enum0.getType()), 'baja:DynamicEnum');
        verifyEq(enum0.getTag(), 'tag0');
        verifyEq(enum0.getOrdinal(), 0);
        verifyEq(enum1.getTag(), 'tag1');
        verifyEq(enum1.getOrdinal(), 1);
      });
      
      it("returns enum from a frozen range for ordinal", function frozenOrdinal() {
        var monday = frozenRange.get(1),
            tuesday = frozenRange.get(2);
        verifyEq(String(monday.getType()), 'baja:Weekday');
        verifyEq(monday.getTag(), 'monday');
        verifyEq(monday.getOrdinal(), 1);
        verifyEq(tuesday.getTag(), 'tuesday');
        verifyEq(tuesday.getOrdinal(), 2);
      });
      
      it("returns enum from a dynamic range for ordinal", function dynamicOrdinal() {
        var enum1 = dynamicRange.get(1),
            enum2 = dynamicRange.get(2);
        verifyEq(String(enum1.getType()), 'baja:DynamicEnum');
        verifyEq(enum1.getTag(), 'tag1');
        verifyEq(enum1.getOrdinal(), 1);
        verifyEq(enum2.getTag(), 'tag2');
        verifyEq(enum2.getOrdinal(), 2);
      });
      
      it("dies for unknown string on frozen range", function diesStringFrozen() {
        errTest(function () {
          frozenRange.get('schmunday');
        }, 'Unable to access enum');
      });
      
      it("dies for unknown string on dynamic range", function diesStringDynamic() {
        errTest(function () {
          dynamicRange.get('tag3');
        }, 'Unable to access enum');
      });
      
      it("dies for unknown ordinal on frozen range", function diesOrdinalFrozen() {
        errTest(function () {
          frozenRange.get(7);
        }, 'Unable to access enum');
      });
      
      it("dies for unknown ordinal on dynamic range", function diesOrdinalDynamic() {
        errTest(function () {
          dynamicRange.get(3);
        }, 'Unable to access enum');
      });
    });
    
    describe("getDataTypeSymbol()", function EnumRange_getDataTypeSymbol() {
      it("returns 'E'", function returnsE() {
        verifyEq(baja.EnumRange.DEFAULT.getDataTypeSymbol(), 'E');
      });
    });
    
    describe("getOrdinals()", function EnumRange_getOrdinals() {
      it("returns ordinals from a frozen range", function frozenOrdinals() {
        var er = baja.EnumRange.make('baja:Weekday');
        verifyEq(er.getOrdinals().join(), '0,1,2,3,4,5,6');
      });
      
      it("returns ordinals from a dynamic range", function dynamicOrdinals() {
        var er = baja.EnumRange.make({
          ordinals: [ 0, 2, 4, 6 ],
          tags: [ 'a', 'b', 'c', 'd' ]
        });
        verifyEq(er.getOrdinals().join(), '0,2,4,6');
      });
      
      it("merges frozen and dynamic ordinals", function frozenDynamic() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 4, 6, 8, 10 ],
          tags: [ 'a', 'b', 'c', 'd' ]
        });
        verifyEq(er.getOrdinals().join(), '0,1,2,3,4,5,6,8,10');
      });
    });
    
    describe("isOrdinal()", function EnumRange_isOrdinal() {
      it("returns true if ordinal exists in frozen range", function frozenOrdinal() {
        var er = baja.EnumRange.make('baja:Weekday');
        verify(er.isOrdinal(0));
        verify(er.isOrdinal(6));
        verify(!er.isOrdinal(-1));
        verify(!er.isOrdinal(7));
      });
      
      it("returns true if ordinal exists in dynamic range", function dynamicOrdinal() {
        var er = baja.EnumRange.make({
          ordinals: [ 3, 4, 5, 6 ],
          tags: [ 'a', 'b', 'c', 'd' ]
        });
        verify(er.isOrdinal(3));
        verify(er.isOrdinal(6));
        verify(!er.isOrdinal(2));
        verify(!er.isOrdinal(7));
      });
      
      it("returns true if ordinal exists in frozen/dynamic combo range", function frozenDynamicOrdinal() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 8, 10 ],
          tags: [ 'a', 'b' ]
        });
        verify(er.isOrdinal(0));
        verify(er.isOrdinal(6));
        verify(!er.isOrdinal(7));
        verify(er.isOrdinal(8));
        verify(!er.isOrdinal(9));
        verify(er.isOrdinal(10));
        verify(!er.isOrdinal(11));
      });
    });
    
    describe("isFrozenOrdinal()", function EnumRange_isFrozenOrdinal() {
      it("returns true if ordinal exists in frozen enum range", function ordinalExistsInFrozen() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 8, 10 ],
          tags: [ 'tag8', 'tag10' ]
        });
        verify(er.isFrozenOrdinal(0));
        verify(er.isFrozenOrdinal(6));
        verify(!er.isFrozenOrdinal(7));
        verify(!er.isFrozenOrdinal(8));
        verify(!er.isFrozenOrdinal(10));
      });
      
      it("dies on non-numeric input", function diesOnNonNumeric() {
        errTest(function () {
          baja.EnumRange.DEFAULT.isFrozenOrdinal("hi");
        }, typeMismatchError(String, Number));
      });
    });
    
    describe("isDynamicOrdinal()", function EnumRange_isDynamicOrdinal() {
      it("returns true if ordinal exists in dynamic enum range", function ordinalExistsInDynamic() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 8, 10 ],
          tags: [ 'tag8', 'tag10' ]
        });
        verify(!er.isDynamicOrdinal(0));
        verify(!er.isDynamicOrdinal(6));
        verify(!er.isDynamicOrdinal(7));
        verify(er.isDynamicOrdinal(8));
        verify(er.isDynamicOrdinal(10));
      });
      
      it("dies on non-numeric input", function diesOnNonNumeric() {
        errTest(function () {
          baja.EnumRange.DEFAULT.isFrozenOrdinal("hi");
        }, typeMismatchError(String, Number));
      });
    });
    
    describe("getFrozenType()", function EnumRange_getFrozenType() {
      it("returns type spec of the frozen enum range", function returnsFrozenType() {
        var er = baja.EnumRange.make({
              frozen: 'baja:Weekday',
              ordinals: [ 8, 10 ],
              tags: [ 'tag8', 'tag10' ]
            }),
            frozenType = er.getFrozenType();
        
        verifyEq(frozenType.getTypeSpec(), 'baja:Weekday');
      });
      
      it("returns null if no frozen range", function nullIfNone() {
        var er = baja.EnumRange.make({
          ordinals: [ 8, 10 ],
          tags: [ 'tag8', 'tag10' ]
        });
        expect(er.getFrozenType()).toBeNull();
      });
    });
    
    describe("getOptions()", function EnumRange_getOptions() {
      it("returns the options facets", function returnsOptions() {
        var facets = baja.Facets.make([ 'foo' ], [ 'bar' ]),
            er = baja.EnumRange.make({ frozen: 'baja:Weekday', options: facets});
        expect(er.getOptions()).toBe(facets);
      });
      
      it("returns baja.Facets.DEFAULT if no options provided", function returnsNullIfNone() {
        var er = baja.EnumRange.make('baja:Weekday');
        expect(er.getOptions()).toBe(baja.Facets.DEFAULT);
      });
    });
    
    describe("getTag()", function EnumRange_getTag() {
      it("returns a frozen enum tag for a given ordinal", function frozenTag() {
        var er = baja.EnumRange.make('baja:Weekday');
        verifyEq(er.getTag(0), 'sunday');
        verifyEq(er.getTag(6), 'saturday');
      });
      
      it("returns a dynamic enum tag for a given ordinal", function dynamicTag() {
        var er = baja.EnumRange.make({
          ordinals: [ 0, 1, 2 ],
          tags: [ 'a', 'b', 'c' ]
        });
        verifyEq(er.getTag(0), 'a');
        verifyEq(er.getTag(2), 'c');
      });
      
      it("returns a tag for a frozen/dynamic combo enum", function frozenDynamicTag() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 8, 10 ],
          tags: [ 'a', 'b' ]
        });
        verifyEq(er.getTag(0), 'sunday');
        verifyEq(er.getTag(6), 'saturday');
        verifyEq(er.getTag(10), 'b');
      });
      
      it("returns string representation of the ordinal if not found", function notFound() {
        var er = baja.EnumRange.make('baja:Weekday');
        verifyEq(er.getTag(7), '7');
        verifyEq(er.getTag(-1), '-1');
      });
    });
    
    describe("isTag()", function EnumRange_isTag() {
      it("returns true if the tag exists in a frozen enum", function frozenTag() {
        var er = baja.EnumRange.make('baja:Weekday');
        verify(er.isTag('sunday'));
        verify(er.isTag('wednesday'));
        verify(er.isTag('saturday'));
        verify(!er.isTag('schmendesday'));
      });
      
      it("returns true if the tag exists in a dynamic enum", function dynamicTag() {
        var er = baja.EnumRange.make({
          ordinals: [ 0, 1, 2 ],
          tags: [ 'tag0', 'tag1', 'tag2' ]
        });
        verify(er.isTag('tag0'));
        verify(er.isTag('tag2'));
        verify(!er.isTag('tag3'));
      });
      
      it("returns true if the tag exists in a frozen/dynamic combo enum", function frozenDynamicTag() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 8, 10 ],
          tags: [ 'tag8', 'tag10' ]
        });
        verify(er.isTag('sunday'));
        verify(er.isTag('saturday'));
        verify(er.isTag('tag10'));
        verify(!er.isTag('tag9'));
        verify(!er.isTag('january'));
      });
      
      it("dies on non-string input", function diesOnNonString() {
        errTest(function () {
          baja.EnumRange.DEFAULT.isTag(5);
        }, typeMismatchError(Number, String));
      });
    });
    
    describe("tagToOrdinal()", function EnumRange_tagToOrdinal() {
      it("converts a frozen tag to an ordinal", function frozenToOrdinal() {
        var er = baja.EnumRange.make('baja:Weekday');
        verifyEq(er.tagToOrdinal('sunday'), 0);
        verifyEq(er.tagToOrdinal('wednesday'), 3);
        verifyEq(er.tagToOrdinal('saturday'), 6);
      });
      
      it("converts a dynamic tag to an ordinal", function dynamicToOrdinal() {
        var er = baja.EnumRange.make({
          ordinals: [ 10, 20, 30 ],
          tags: [ 'tag10', 'tag20', 'tag30' ]
        });
        verifyEq(er.tagToOrdinal('tag10'), 10);
        verifyEq(er.tagToOrdinal('tag20'), 20);
        verifyEq(er.tagToOrdinal('tag30'), 30);
      });
      
      it("converts a frozen/dynamic combo tag to ordinal", function comboToOrdinal() {
        var er = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 5, 10, 15 ],
          tags: [ 'tag5', 'tag10', 'tag15' ]
        });
        verifyEq(er.tagToOrdinal('sunday'), 0);
        verifyEq(er.tagToOrdinal('friday'), 5);
        verifyEq(er.tagToOrdinal('tag10'), 10);
      });
      
      it("dies on non-string input", function diesOnNonString() {
        errTest(function () {
          baja.EnumRange.DEFAULT.tagToOrdinal(5);
        }, typeMismatchError(Number, String));
      });
      
      it("dies if not found", function diesOnNotFound() {
        errTest(function () {
          baja.EnumRange.make('baja:Weekday').tagToOrdinal('schmursday');
        }, 'Invalid tag: schmursday');
      });
    });
    
    describe("newCopy()", function EnumRange_newCopy() {
      it("returns self", function returnsSelf() {
        var er = baja.EnumRange.make({ ordinals: [ 0 ], tags: [ 'a' ]});
        expect(er.newCopy()).toBe(er);
      });
    });
    
    describe("equals()", function EnumRange_equals() {
      it("returns true if both ranges encode to the same string", function encodeToSameString() {
        var er1 = baja.EnumRange.make({
              frozen: 'baja:Weekday',
              ordinals: [ 10, 11 ],
              tags: [ 'tag10', 'tag11' ]
            }),
            er2 = baja.EnumRange.make({
              frozen: 'baja:Weekday',
              ordinals: [ 10, 11 ],
              tags: [ 'tag10', 'tag11' ]
            });
        verify(er1.equals(er2));
      });
      
      it("returns false if they encode to different strings", function encodeToDifferentStrings() {
        var er1 = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 10, 11 ],
          tags: [ 'tag10', 'tag11' ]
        }),
        er2 = baja.EnumRange.make({
          frozen: 'baja:Weekday',
          ordinals: [ 10, 11 ],
          tags: [ 'tag10', 'tag12' ]
        });
        verify(!er1.equals(er2));
      });
    });
    
    describe("baja.EnumRange.BOOLEAN_RANGE", function BooleanRange() {
      var br = baja.EnumRange.BOOLEAN_RANGE;
      
      describe("encodeToString()", function BooleanRange_encodeToString() {
        it("encodes boolean range to string", function encodesToString() {
          verifyEq(br.encodeToString(), "{false=0,true=1}");
        });
      });
      
      describe("get()", function BooleanRange_get() {
        it("returns boolean value for string", function returnsForString() {
          verifyEq(br.get('false'), false);
          verifyEq(br.get('true'), true);
        });
        
        it("returns boolean value for integer", function returnsForInteger() {
          verifyEq(br.get(0), false);
          verifyEq(br.get(1), true);
        });
      });
    });
  });
  
  describe("baja.EnumSet", function EnumSet() {
    function verifyEnumSet(enumSet, ordinals, range) {
      verifyEq(enumSet.getOrdinals(), ordinals);
      verify(enumSet.getRange().equals(range));
    }
    
    describe("constructor", function EnumSet_constructor() {
      it("accepts an array of ordinals and a range", function acceptsOrdinalsAndRange() {
        var ordinals = [ 0, 1, 2 ],
            range = baja.EnumRange.make({ ordinals: [ 0 ], tags: [ 'tag0' ] }),
            enumSet = new baja.EnumSet(ordinals, range);
        expect(enumSet.getOrdinals()).toBe(ordinals);
        expect(enumSet.getRange()).toBe(range);
      });
      
      it("is registered on baja:EnumSet Type", function isRegistered() {
        expect(baja.$('baja:EnumSet')).toBe(baja.EnumSet.DEFAULT);
      });
    });
    
    describe("make() (static)", function EnumSet_make() {
      it("accepts ordinals and range properties", function acceptsOrdinalsAndRange() {
        var ordinals = [ 0, 1 ],
            range = baja.EnumRange.make({ ordinals: [ 0 ], tags: [ 'tag0' ] }),
            enumSet = baja.EnumSet.make({ ordinals: ordinals, range: range });
        expect(enumSet.getOrdinals()).toBe(ordinals);
        expect(enumSet.getRange()).toBe(range);
      });
      
      it("defaults ordinals to empty array", function defaultsOrdinals() {
        var range = baja.EnumRange.make({ ordinals: [ 0 ], tags: [ 'tag0' ] }),
            enumSet = baja.EnumSet.make({ range: range });
        expect(enumSet.getOrdinals()).toEqual([]);
        expect(enumSet.getRange()).toBe(range);
      });
      
      it("defaults range to DEFAULT", function defaultsRange() {
        var ordinals = [ 0, 1 ],
            enumSet = baja.EnumSet.make({ ordinals: ordinals });
        expect(enumSet.getOrdinals()).toBe(ordinals);
        expect(enumSet.getRange()).toBe(baja.EnumRange.DEFAULT);
      });
      
      it("returns DEFAULT if ordinals is empty and range is DEFAULT", function returnsDefaultNoOrdinalsNoRange() {
        var enumSet = baja.EnumSet.make({ ordinals: [], range: baja.EnumRange.DEFAULT });
        expect(enumSet).toBe(baja.EnumSet.DEFAULT);
      });
      
      it("returns DEFAULT for missing / empty argument", function returnsDefault() {
        expect(baja.EnumSet.make()).toBe(baja.EnumSet.DEFAULT);
        expect(baja.EnumSet.make({})).toBe(baja.EnumSet.DEFAULT);
      });
      
      it("dies on non-Array ordinals", function diesOnNonArrayOrdinals() {
        errTest(function () {
          baja.EnumSet.make({ ordinals: "hello" });
        }, typeMismatchError(String, Array));
      });
      
      it("dies on non-EnumRange range", function diesOnNonEnumRangeRange() {
        errTest(function () {
          baja.EnumSet.make({ range: true });
        }, typeMismatchError(Boolean, baja.EnumRange));
      });
    });
    
    describe("decodeFromString()", function EnumSet_decodeFromString() {
      function decode(str) {
        return baja.EnumSet.DEFAULT.decodeFromString(str);
      }
      
      it("decodes ordinals and range", function decodesOrdinalAndRange() {
        var ordinals = [ 2, 4, 6, 8 ],
            range = baja.EnumRange.make({ ordinals: [ 2, 3 ], tags: [ 'tag2', 'tag3' ]}),
            enumSet = decode(ordinals.join() + "@" + range.encodeToString());
        verifyEnumSet(enumSet, ordinals, range);
      });
      
      it("decodes just ordinals", function decodesOrdinals() {
        var ordinals = [ 2, 3, 4 ],
            enumSet = decode(ordinals.join());
        verifyEnumSet(enumSet, ordinals, baja.EnumRange.DEFAULT);
      });
      
      it("decodes just range", function decodesRange() {
        var range = baja.EnumRange.make({ ordinals: [ 2, 3 ], tags: [ 'tag2', 'tag3' ]}),
            enumSet = decode("@" + range.encodeToString());
        verifyEnumSet(enumSet, [], range);
      });
      
      it("decodes empty string", function decodesEmpty() {
        var enumSet = decode("");
        verifyEnumSet(enumSet, [], baja.EnumRange.DEFAULT);
      });
      
      it("dies on invalid ordinal", function diesOnInvalidOrdinal() {
        errTest(function () {
          decode("1,2,A@{}");
        }, 'Invalid ordinal: A');
      });
    });
    
    describe("encodeToString()", function EnumSet_encodeToString() {
      it("encodes ordinals@range", function encodesOrdinalsRange() {
        var ordinals = [ 14, 16, 18 ],
            range = baja.EnumRange.make({ ordinals: [ 1 ], tags: [ 'tag1' ] }),
            enumSet = baja.EnumSet.make({ ordinals: ordinals, range: range });
        verifyEq(enumSet.encodeToString(), "14,16,18@" + range.encodeToString());
      });
      
      it("leaves out range if DEFAULT", function encodesDefaultRange() {
        var ordinals = [ 20, 22, 24 ],
            enumSet = baja.EnumSet.make({ ordinals: ordinals });
        verifyEq(enumSet.encodeToString(), "20,22,24");
      });
      
      it("encodes empty string if no ordinals and no range", function encodesEmptyString() {
        verifyEq(baja.EnumSet.make().encodeToString(), "");
      });
    });
    
    describe("getDataTypeSymbol()", function EnumSet_getDataTypeSymbol() {
      it("returns 'E'", function returnsE() {
        verifyEq(baja.EnumSet.make().getDataTypeSymbol(), 'E');
      });
    });
    
    describe("getOrdinals()", function EnumSet_getOrdinals() {
      it("returns ordinals", function returnsOrdinals() {
        var ordinals = [ 26, 27, 28 ],
            enumSet = baja.EnumSet.make({ ordinals: ordinals });
        expect(enumSet.getOrdinals()).toBe(ordinals);
      });
    });
    
    describe("getRange()", function EnumSet_getRange() {
      it("returns range", function returnsRange() {
        var range = baja.EnumRange.make({ ordinals: [ 27, 28 ], tags: [ 'tag27', 'tag28' ] }),
            enumSet = baja.EnumSet.make({ range: range });
        expect(enumSet.getRange()).toBe(range);
      });
    });
  });
  
  describe("baja.Enum", function Enum() {
    describe("getEnumFromIEnum (static)", function Enum_getEnumFromIEnum() {
      it("returns out slot from an EnumWritable", function returnsOutFromEnumWritable() {
        var en = baja.DynamicEnum.make(3),
            enumWritable = baja.$('control:EnumWritable', {
              out: baja.$('baja:StatusEnum', {
                value: en
              })
            }),
            foundEnum = baja.Enum.getEnumFromIEnum(enumWritable);
        
        expect(foundEnum).toBe(en);
      });
      
      it("returns value slot from a StatusEnum", function returnsValueFromStatusEnum() {
        var en = baja.DynamicEnum.make(1),
            statusEnum = baja.$('baja:StatusEnum', {
              value: en
            }),
            foundEnum = baja.Enum.getEnumFromIEnum(statusEnum);
        expect(foundEnum).toBe(en);
      });
      
      it("calls getEnum if it exists", function callsGetEnum() {
        var en = baja.DynamicEnum.make(1),
            comp = new baja.Component(),
            foundEnum;
        
        comp.getEnum = function () { return en; };
        
        foundEnum = baja.Enum.getEnumFromIEnum(comp);
        
        expect(foundEnum).toBe(en);
      });
      
      it("returns DynamicEnum.DEFAULT if no other match", function returnsDefault() {
        expect(baja.Enum.getEnumFromIEnum("foo"))
          .toBe(baja.DynamicEnum.DEFAULT);
      });
      
      it("dies on a non-Baja object", function diesOnNonBaja() {
        errTest(function () {
          baja.Enum.getEnumFromIEnum({});
        });
      });
      
      it("returns value dynamic 'out' Property on baja:Component", function returnsValueFromDynOut() {
        var comp = baja.$('baja:Component', {
          out: baja.$('baja:StatusEnum')
        });
        
        comp.get("out").setValue(baja.$("baja:Weekday").get("tuesday"));
        expect(baja.Enum.getEnumFromIEnum(comp).is("tuesday")).toBeTruthy();
      });
    });
    
    describe("getEnum", function Enum_getEnum() {
      it("returns itself", function returnsItself() {
        var en = baja.DynamicEnum.make(15);
        expect(en.getEnum()).toBe(en);
      });
    });
  });
  
  
  describe("baja.DynamicEnum", function DynamicEnum() {
    describe("constructor", function DynamicEnum_constructor() {
      it("accepts ordinal and range", function acceptsOrdinalAndRange() {
        var ordinal = 2,
            range = baja.EnumRange.make('baja:Weekday'),
            de = new baja.DynamicEnum(ordinal, range);
        
        verifyEq(de.getOrdinal(), ordinal);
        expect(de.getRange()).toBe(range);
      });
      
      it("dies on non-numeric ordinal", function diesOnNonNumericOrdinal() {
        errTest(function () {
          baja.DynamicEnum('a', baja.EnumRange.DEFAULT);
        }, typeMismatchError(String, Number));
      });
      
      it("dies on missing ordinal", function diesOnMissingOrdinal() {
        errTest(function () {
          baja.DynamicEnum(undefined, baja.EnumRange.DEFAULT);
        }, undefinedError(Number));
      });
      
      it("dies on range not EnumRange", function diesOnRangeNotEnumRange() {
        errTest(function () {
          baja.DynamicEnum(2, 15);
        });
      });
      
      it("defaults range to DEFAULT", function defaultsRangeToDEFAULT() {
        var ordinal = 0,
            de = new baja.DynamicEnum(ordinal);
        expect(de.getRange()).toBe(baja.EnumRange.DEFAULT);
      });
    });
    
    describe("make() (static)", function DynamicEnum_make() {
      it("accepts a single ordinal and sets range to DEFAULT", function acceptsOrdinal() {
        var de = baja.DynamicEnum.make(2);
        verifyEq(de.getOrdinal(), 2);
        expect(de.getRange()).toBe(baja.EnumRange.DEFAULT);
        expect(de).not.toBe(baja.DynamicEnum.DEFAULT);
      });
      
      it("returns DEFAULT if ordinal is 0/undefined and range is DEFAULT/undefined", function returnsDEFAULT() {
        var def = baja.DynamicEnum.DEFAULT;
        expect(baja.DynamicEnum.make()).toBe(def);
        expect(baja.DynamicEnum.make(0)).toBe(def);
        expect(baja.DynamicEnum.make({ ordinal: 0, range: baja.EnumRange.DEFAULT }))
          .toBe(def);
      });
      
      it("defaults ordinal to 0 if omitted", function defaultsOrdinalTo0() {
        var range = baja.EnumRange.make('baja:Weekday'),
            de = baja.DynamicEnum.make({ range: range });
        
        expect(de.getOrdinal()).toBe(0);
        expect(de.getRange()).toBe(range);
      });
      
      it("defaults range to DEFAULT if omitted", function defaultsRangeToDefault() {
        var ordinal = 2,
            de = baja.DynamicEnum.make({ ordinal: 2 });
        verifyEq(de.getOrdinal(), 2);
        expect(de.getRange()).toBe(baja.EnumRange.DEFAULT);
      });
      
      it("accepts object literal with ordinal and range", function acceptsObject() {
        var ordinal = 5,
            range = baja.EnumRange.make({ ordinals: [ 2, 3 ], tags: [ 'a', 'b' ]}),
            de = baja.DynamicEnum.make({ ordinal: ordinal, range: range });
        verifyEq(de.getOrdinal(), ordinal);
        expect(de.getRange()).toBe(range);
      });
      
      it("directly returns an input DynamicEnum", function createsFromDynamic() {
        var ordinal = 5,
            range = baja.EnumRange.make({ ordinals: [ 2, 3 ], tags: [ 'a', 'b' ]}),
            sourceEnum = baja.DynamicEnum.make({ ordinal: ordinal, range: range }),
            de = baja.DynamicEnum.make({ en: sourceEnum });
        
        expect(de).toBe(sourceEnum);
      });
      
      it("creates from a frozen enum", function createsFromFrozen() {
        var frozenEnum = baja.$('baja:Weekday').get('tuesday'),
            de = baja.DynamicEnum.make({ en: frozenEnum });
        expect(de.getRange()).toBe(frozenEnum.getRange());
        verifyEq(de.getOrdinal(), frozenEnum.getOrdinal());
      });
      
      it("creates from a Boolean", function createsFromBoolean() {
        var de = baja.DynamicEnum.make({ en: true });
        expect(de.getRange()).toBe(baja.EnumRange.BOOLEAN_RANGE);
        verifyEq(de.getOrdinal(), 1);
      });
      
      it("creates from a FrozenEnum string typespec", function createsFromTypeSpec() {
        var de = baja.DynamicEnum.make({ en: 'baja:Weekday' });
        expect(de.getRange()).toBe(baja.$('baja:Weekday').getRange());
        verifyEq(de.getOrdinal(), 0);
      });
      
      it("dies on non-Enum en argument", function diesOnNonEnum() {
        errTest(function () {
          baja.DynamicEnum.make({ en: 99 });
        }, 'Argument must be an Enum');
      });
      
      it("dies on unknown type for en argument", function diesOnUnknownType() {
        errTest(function () {
          baja.DynamicEnum.make({ en: [] });
        }, 'Invalid Enum Argument');
      });
      
      it("dies on unknown frozen enum type spec", function diesOnUnknownSpec() {
        errTest(function () {
          baja.DynamicEnum.make({ en: 'baja:Whoogle' });
        });
      });
    });
    
    describe("decodeFromString()", function DynamicEnum_decodeFromString() {
      it("decodes a single ordinal", function decodesOrdinal() {
        var de = baja.DynamicEnum.DEFAULT.decodeFromString("3");
        verifyEq(de.getOrdinal(), 3);
        expect(de.getRange()).toBe(baja.EnumRange.DEFAULT);
      });
      
      it("decodes an ordinal plus dynamic range", function decodesOrdinalPlusDynamicRange() {
        var de = baja.DynamicEnum.DEFAULT.decodeFromString("3@{a=1,b=2,c=3}"),
            range = de.getRange();
        verifyEq(de.getOrdinal(), 3);
        verifyEq(range.getOrdinals().join(), '1,2,3');
        verifyEq(range.getTag(1), 'a');
        verifyEq(range.getTag(2), 'b');
        verifyEq(range.getTag(3), 'c');
      });
      
      it("decodes an ordinal plus frozen range", function decodesOrdinalPlusFrozenRange() {
        var de = baja.DynamicEnum.DEFAULT.decodeFromString("5@baja:Weekday");
        verifyEq(de.getOrdinal(), 5);
        verifyEq(de.getRange().encodeToString(), 'baja:Weekday');
      });
    });
    
    describe("encodeToString()", function DynamicEnum_encodeToString() {
      it("encodes just an ordinal", function encodesOrdinal() {
        var de = baja.DynamicEnum.make(2);
        verifyEq(de.encodeToString(), '2');
      });
      
      it("encodes an ordinal plus dynamic range", function encodesOrdinalPlusDynamicRange() {
        var range = baja.EnumRange.make({ ordinals: [ 2, 3 ], tags: [ 'a', 'b' ]}), 
            de = baja.DynamicEnum.make({ ordinal: 4, range: range});
        verifyEq(de.encodeToString(), "4@{a=2,b=3}");
      });
      
      it("encodes an ordinal plus frozen range", function encodeOrdinalPlusFrozenRange() {
        var range = baja.EnumRange.make('baja:Weekday'),
            de = baja.DynamicEnum.make({ ordinal: 6, range: range });
        verifyEq(de.encodeToString(), "6@baja:Weekday");
      });
    });
    
    describe("getDataTypeSymbol()", function DynamicEnum_getDataTypeSymbol() {
      it("returns 'e'", function returnsE() {
        verifyEq(baja.DynamicEnum.DEFAULT.getDataTypeSymbol(), 'e');
      });
    });
    
    describe("isActive()", function DynamicEnum_isActive() {
      it("returns true if ordinal is not 0", function returnsTrueIfNot0() {
        verify(baja.DynamicEnum.make(1).isActive());
        verify(!baja.DynamicEnum.make(0).isActive());
      });
    });
    
    describe("getOrdinal()", function DynamicEnum_getOrdinal() {
      it("returns the enum's ordinal", function returnsOrdinal() {
        verifyEq(baja.DynamicEnum.make({ ordinal: 7 }).getOrdinal(), 7);
        verifyEq(baja.DynamicEnum.make({ ordinal: -1 }).getOrdinal(), -1);
      });
    });
    
    describe("getRange()", function DynamicEnum_getRange() {
      it("returns the enum's range", function returnsRange() {
        var range = baja.EnumRange.make({ ordinals: [ 1 ], tags: [ 'a' ]}),
            de = baja.DynamicEnum.make({ ordinal: 4, range: range });
        expect(de.getRange()).toBe(range);
      });
    });
    
    describe("getTag()", function DynamicEnum_getTag() {
      it("returns the tag in the enum's range for its ordinal", function returnsTag() {
        var range = baja.EnumRange.make({
              ordinals: [ 0, 1, 2, 3 ],
              tags: [ 'tag0', 'tag1', 'tag2', 'tag3' ]
            });
        
        verifyEq(baja.DynamicEnum.make({ ordinal: 0, range: range }).getTag(), 
            'tag0');
        verifyEq(baja.DynamicEnum.make({ ordinal: 3, range: range }).getTag(), 
            'tag3');
      });
      
      it("returns a string representation of the ordinal if not found", function ordinalNotFound() {
        var range = baja.EnumRange.make({
              ordinals: [ 0 ],
              tags: [ 'tag0' ]
            });
        verifyEq(baja.DynamicEnum.make({ ordinal: 0, range: range }).getTag(), 
            'tag0');
        verifyEq(baja.DynamicEnum.make({ ordinal: 1, range: range }).getTag(), 
            '1');
      });
    });
    
    describe("toString()", function DynamicEnum_toString() {
      it("returns output of getTag()", function returnsGetTag() {
        var range = baja.EnumRange.make('baja:Weekday');
        verifyEq(baja.DynamicEnum.make({ ordinal: 0, range: range }).toString(), 
            'sunday');
        verifyEq(baja.DynamicEnum.make({ ordinal: 6, range: range }).toString(), 
            'saturday');
      });
    });
    
    describe("is()", function DynamicEnum_is() {
      it("returns true if input arg matches enum's ordinal", function matchesOrdinal() {
        var de = baja.DynamicEnum.make(3);
        verify(de.is(3));
        verify(!de.is(5));
        verify(de.is(de));
        verify(de.is(baja.DynamicEnum.make(3)));
      });
      
      it("returns true if input arg matches enum's tag", function matchesTag() {
        var range = baja.EnumRange.make('baja:Weekday'),
            de = baja.DynamicEnum.make({ ordinal: 3, range: range });
        verify(de.is('wednesday'));
        verify(!de.is('thursday'));
        verify(de.is(de));
      });
    });
  });
  
  describe("baja.FrozenEnum", function FrozenEnum() {
    describe("constructor", function FrozenEnum_constructor() {
      function verifyFrozenEnum(en, ordinals, tags) {
        var range = en.getRange();
        verifyEq(range.getOrdinals(), ordinals);
        baja.iterate(0, ordinals.length, function (i) {
          verifyEq(range.getTag(ordinals[i]), tags[i]);
        });
      }
      
      it("is registered on Baja FrozenEnum types", function isRegistered() {
        var jobState = baja.$('baja:JobState'),
            month = baja.$('baja:Month'),
            unitConversion = baja.$('baja:UnitConversion'),
            weekday = baja.$('baja:Weekday');
        
        verifyFrozenEnum(jobState,
            [ 0, 1, 2, 3, 4, 5 ],
            [ 'unknown', 'running', 'canceling', 
              'canceled', 'success', 'failed' ]);
        verifyFrozenEnum(month,
            [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 ],
            [ 'january', 'february', 'march', 'april', 'may', 'june', 'july',
              'august', 'september', 'october', 'november', 'december' ]);
        verifyFrozenEnum(unitConversion, 
            [ 0, 1, 2 ], 
            [ 'none', 'metric', 'english' ]);
        verifyFrozenEnum(weekday, 
            [ 0, 1, 2, 3, 4, 5, 6 ], 
            [ 'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 
              'saturday' ]);

      });
    });
    
    describe("make()", function FrozenEnum_make() {
      it("accepts tag and calls this.get()", function acceptsTag() {
        var en = baja.$('baja:Weekday'),
            made = en.make('tuesday');
        verifyEq(made.getRange().encodeToString(), 'baja:Weekday');
        verifyEq(made.getOrdinal(), 2);
        verifyEq(made.getTag(), 'tuesday');
      });
      
      it("accepts ordinal and calls this.get()", function acceptsOrdinal() {
        var en = baja.$('baja:Weekday'),
            made = en.make(1);
        verifyEq(made.getRange().encodeToString(), 'baja:Weekday');
        verifyEq(made.getOrdinal(), 1);
        verifyEq(made.getTag(), 'monday');
      });
      
      it("dies if tag not found", function diesIfTagNotFound() {
        errTest(function () {
          baja.$('baja:Weekday').make('schoozeday');
        }, 'Unable to access enum');
      });
      
      it("dies if ordinal not found", function diesIfOrdinalNotFound() {
        errTest(function () {
          baja.$('baja:Weekday').make('7');
        }, 'Unable to access enum');
      });
    });
    
    describe("decodeFromString()", function FrozenEnum_decodeFromString() {
      it("calls this.get()", function callsGet() {
        var en = baja.$('baja:Weekday'),
            decoded = en.decodeFromString('friday');
      });
      
      it("dies if non-string, non-numeric input given", function diesIfNonString() {
        errTest(function () {
          baja.$('baja:Weekday').decodeFromString({});
        });
      });
    });
    
    describe("encodeToString()", function FrozenEnum_encodeToString() {
      it("returns this.getTag()", function returnsGetTag() {
        var en = baja.$('baja:Weekday');
        verifyEq(en.get('monday').encodeToString(), 'monday');
        verifyEq(en.get('tuesday').encodeToString(), 'tuesday');
      });
    });
    
    describe("isActive()", function FrozenEnum_isActive() {
      it("returns true if ordinal is not 0", function returnsTrueOrdinalNot0() {
        var en = baja.$('baja:Weekday');
        verify(!en.get('sunday').isActive());
        verify(en.get('monday').isActive());
      });
    });
    
    describe("getOrdinal()", function FrozenEnum_getOrdinal() {
      it("returns the enum's ordinal", function returnsOrdinal() {
        var en = baja.$('baja:Weekday');
        verifyEq(en.get('sunday').getOrdinal(), 0);
        verifyEq(en.get('saturday').getOrdinal(), 6);
      });
    });
    
    describe("getRange()", function FrozenEnum_getRange() {
      it("returns the enum's range", function returnsRange() {
        var en = baja.$('baja:Weekday'),
            range = en.getRange();
        verifyEq(range.encodeToString(), 'baja:Weekday');
        verifyEq(range.getOrdinals().join(), '0,1,2,3,4,5,6');
      });
    });
    
    describe("getTag()", function FrozenEnum_getTag() {
      it("returns the enum's tag", function returnsTag() {
        var en = baja.$('baja:Weekday');
        verifyEq(en.get(0).getTag(), 'sunday');
        verifyEq(en.get(6).getTag(), 'saturday');
      });
    });
    
    describe("getDisplayTag()", function FrozenEnum_getDisplayTag() {
      it("returns a display friendly version of the tag", function returnsDisplay() {
        verifyEq(baja.$('baja:Weekday').get(3).getDisplayTag(), 'Wednesday');
        verifyEq(baja.$('baja:UnitConversion').get(2).getDisplayTag(), 'English');
      });
    });
    
    describe("toString()", function FrozenEnum_toString() {
      it("returns getDisplayTag()", function returnsDisplayTag() {
        verifyEq(baja.$('baja:JobState').get(2).toString(), 'Canceling');
      });
    });
    
    describe("get()", function FrozenEnum_get() {
      it("retrieves another enum from this enum's range", function callsGet() {
        var en = baja.$('baja:Weekday');
        expect(en.get('monday').get('tuesday')).toBe(en.get('tuesday'));
        expect(en.get('wednesday')).toBe(en.getRange().get('wednesday'));
      });
    });
    
    describe("is()", function FrozenEnum_is() {
      it("returns true if this enum's ordinal matches input", function matchesOrdinal() {
        var en = baja.$('baja:Weekday');
        verify(en.get('monday').is(1));
        verify(en.get('tuesday').is(2));
        verify(!en.get('wednesday').is(2));
        verify(en.is(en));
        verify(en.is(baja.$("baja:Weekday", "sunday")));
      });
      
      it("returns true if this enum's tag matches input", function matchesTag() {
        var en = baja.$('baja:Weekday');
        verify(en.get(2).is('tuesday'));
        verify(en.get(4).is('thursday'));
        verify(!en.get(4).is('friday'));
      });
    });
  });
  
  describe("baja.Status", function Status() {
    describe("make() (static)", function Status_make() {
      it("accepts a single integer bitmask", function acceptsInteger() {
        var s = baja.Status.make(6);
        verifyEq(s.getBits(), 6);
        expect(s.getFacets()).toBe(baja.Facets.DEFAULT);
      });
      
      it("accepts an object literal with bits property", function acceptsObjectBits() {
        var s = baja.Status.make({ bits: 7 });
        verifyEq(s.getBits(), 7);
        expect(s.getFacets()).toBe(baja.Facets.DEFAULT);
      });
      
      it("sets bits to 1 if state flag is true", function stateFlagTrue() {
        var origStatus = baja.Status.make(5), // 00101
            newStatus = baja.Status.make({ 
              orig: origStatus, 
              bits: 10, //01010
              state: true 
            });
        verifyEq(newStatus.getBits(), 15); //01111
      });
      
      it("sets bits to 0 if state flag is false", function stateFlagFalse() {
        var origStatus = baja.Status.make(29), // 11101
            newStatus = baja.Status.make({
              orig: origStatus,
              bits: 24, //11000
              state: false
            });
        verifyEq(newStatus.getBits(), 5); //00101
      });
      
      it("accepts facets property", function acceptsFacets() {
        var facets = baja.Facets.make(['a'], ['aa']),
            status = baja.Status.make({
              bits: 1,
              facets: facets
            });
        expect(status.getFacets()).toBe(facets);
      });
      
      it("accepts a baja.Status in place of bits", function acceptsStatus() {
        var inStatus = baja.Status.make(5),
            status = baja.Status.make({ bits: inStatus });
        verifyEq(status.getBits(), 5);
      });
      
      it("interns if bits <= 256", function interns() {
        expect(baja.Status.make(256)).toBe(baja.Status.make(256));
        expect(baja.Status.make(257)).not.toBe(baja.Status.make(257));
      });
      
      it("returns DEFAULT if orig+state works out to 0", function origState0Default() {
        var origStatus = baja.Status.make(5),
            status = baja.Status.make({
              bits: 5,
              orig: origStatus,
              state: false
            });
        expect(status).toBe(baja.Status.DEFAULT);
      });
      
      it("returns orig if orig+state results same bits as orig", function origSameBits() {
        var origStatus = baja.Status.make(10),
            status = baja.Status.make({
              bits: 0,
              orig: origStatus,
              state: true
            });
        expect(status).toBe(origStatus);
      });
      
      it("dies if bits not provided", function diesIfNoBits() {
        errTest(function () {
          baja.Status.make();
        }, undefinedError(Number));
        
        errTest(function () {
          baja.Status.make({ bits: undefined });
        }, undefinedError(Number));
      });
      
      it("dies if bits is neither Status nor Number", function nonNumericBits() {
        errTest(function () {
          baja.Status.make({ bits: 'bits' });
        }, typeMismatchError(String, Number));
      });
      
      it("dies if orig argument is not Status", function nonStatusOrig() {
        errTest(function () {
          baja.Status.make({ bits: 3, orig: "HELLO", state: true });
        });
      });
      
      it("dies if state argument is not boolean", function nonBooleanState() {
        errTest(function () {
          baja.Status.make({ bits: 3, orig: baja.Status.DEFAULT, state: "YO" });
        }, typeMismatchError(String, Boolean));
      });
    });
    
    //TODO: should this actually check for IStatus type instead of duck typing via slots?
    describe("getStatusFromIStatus() (static)", function Status_getStatusFromIStatus() {
      it("pulls status from an 'out' slot with BStatusValue type", function pullsFromOut() {
        var status = baja.Status.make(99),
            comp = baja.$('control:NumericWritable', {
              out: baja.$('baja:StatusNumeric', {
                status: status
              })
            }),
            foundStatus = baja.Status.getStatusFromIStatus(comp);
        expect(foundStatus).toBe(status);
      });
      
      it("pulls status from a BStatusValue", function pullsFromStatusValue() {
        var status = baja.Status.make(100),
            sv = baja.$('baja:StatusBoolean', {
              status: status
            }),
            foundStatus = baja.Status.getStatusFromIStatus(sv);
        expect(foundStatus).toBe(status);
      });
      
      it("returns baja.Status.DEFAULT if no status can be determined", 
          function returnsDefault() {
        expect(baja.Status.getStatusFromIStatus("foo"))
          .toBe(baja.Status.DEFAULT);
      });
      
      it("dies if passing in a non-Baja object", function diesIfNonBaja() {
        errTest(function () {
          baja.Status.getStatusFromIStatus({});
        });
      });
      
      it("pulls status from an 'out' slot, even if it is a dynamic slot", function pullsFromDynamicSlot() {
        var status = baja.Status.make(99),
            comp = baja.$('baja:Component', {
              out: baja.$('baja:StatusNumeric', {
                status: status
              })
            }),
            foundStatus = baja.Status.getStatusFromIStatus(comp);
        expect(foundStatus).toBe(status);
      });
    });
    
    describe("decodeFromString()", function Status_decodeFromString() {
      it("decodes a simple hex string", function decodesHex() {
        var status = baja.Status.ok.decodeFromString("FF");
        verifyEq(status.getBits(), 255);
        expect(status.getFacets()).toBe(baja.Facets.DEFAULT);
      });
      
      it("decodes a hex string + facets", function decodesHexPlusFacets() {
        var status = baja.Status.ok.decodeFromString(
            "80;facet1=s:value1|facet2=s:value2");
        verifyEq(status.getBits(), 128);
        verifyEq(status.getFacets().get('facet1'), 'value1');
        verifyEq(status.getFacets().get('facet2'), 'value2');
      });
    });
    
    describe("encodeToString()", function Status_encodeToString() {
      it("encodes a simple hex string", function encodesHex() {
        var encoded = baja.Status.make(160).encodeToString();
        verifyEq(encoded.toLowerCase(), "a0");
      });
      
      it("encodes a hex string + facets", function encodesHexPlusFacets() {
        var facets = baja.Facets.make([ 'a', 'b' ], [ 'va', 'vb' ]),
            status = baja.Status.make({ bits: 172, facets: facets }),
            encoded = status.encodeToString();
        verifyEq(encoded.toLowerCase(), "ac;a=s:va|b=s:vb");
      });
    });
    
    describe("equals()", function Status_equals() {
      it("returns true if bits and facets are the same", function sameBitsAndFacets() {
        var f1 = baja.Facets.make([ 'a', 'b' ], [ 'c', 'd' ]),
            f2 = baja.Facets.make([ 'a', 'b' ], [ 'c', 'd' ]),
            s1 = baja.Status.make({ bits: 20, facets: f1 }),
            s2 = baja.Status.make({ bits: 20, facets: f2 });
        verify(s1.equals(s2));
      });
      
      it("returns false if bits are different", function differentBits() {
        var f1 = baja.Facets.make([ 'a', 'b' ], [ 'c', 'd' ]),
            f2 = baja.Facets.make([ 'a', 'b' ], [ 'c', 'd' ]),
            s1 = baja.Status.make({ bits: 20, facets: f1 }),
            s2 = baja.Status.make({ bits: 21, facets: f2 });
        verify(!s1.equals(s2));
      });
      
      it("returns false if facets are different", function differentFacets() {
        var f1 = baja.Facets.make([ 'a', 'b' ], [ 'c', 'd' ]),
            f2 = baja.Facets.make([ 'a', 'b' ], [ 'c', 'e' ]),
            s1 = baja.Status.make({ bits: 20, facets: f1 }),
            s2 = baja.Status.make({ bits: 20, facets: f2 });
        verify(!s1.equals(s2));
      });
    });
    
    describe("getFacets()", function Status_getFacets() {
      it("returns my facets", function returnsFacets() {
        var facets = baja.Facets.make([ 'b' ], [ 'b' ]),
            status = baja.Status.make({ bits: 2, facets: facets});
        expect(status.getFacets()).toBe(facets);
      });
    });
    
    describe("get()", function Status_get() {
      it("returns a facet from my facets", function returnsFacet() {
        var facets = baja.Facets.make([ 'aFacet' ], [ 'aValue' ]),
            status = baja.Status.make({ bits: 2, facets: facets });
        verifyEq(status.get('aFacet', 'do not return this'), 'aValue');
      });
      
      it("returns null if facet not found", function facetNotFound() {
        var facets = baja.Facets.make([ 'aFacet' ], [ 'aValue' ]),
            status = baja.Status.make({ bits: 2, facets: facets });
        expect(status.get('bFacet')).toBeNull();
      });
      
      it("returns given default if facet not found", function givenDefaultFacetNotFound() {
        var facets = baja.Facets.make([ 'aFacet' ], [ 'aValue' ]),
            status = baja.Status.make({ bits: 2, facets: facets });
        verifyEq(status.get('notFound', 'iAmADefault'), 'iAmADefault');
      });
    });
    
    describe("getBits()", function Status_getBits() {
      it("returns my bits", function returnsBits() {
        verifyEq(baja.Status.make(35).getBits(), 35);
      });
    });
    
    describe("isValid()", function Status_isValid() {
      it("returns false if disabled bit is set", function falseIfDisabled() {
        verify(!baja.Status.make(baja.Status.DISABLED).isValid());
      });
      
      it("returns false if fault bit is set", function falseIfFault() {
        verify(!baja.Status.make(baja.Status.FAULT).isValid());
      });
      
      it("returns false if down bit is set", function falseIfDown() {
        verify(!baja.Status.make(baja.Status.DOWN).isValid());
      });
      
      it("returns false if stale bit is set", function falseIfStale() {
        verify(!baja.Status.make(baja.Status.STALE).isValid());
      });

      it("returns false if null bit is set", function falseIfNull() {
        verify(!baja.Status.make(baja.Status.NULL).isValid());
      });

      it("returns true if none of disabled/fault/down/stale/null bits are set", function allValid() {
        var bits = 0xFFFF
                   & ~baja.Status.DISABLED
                   & ~baja.Status.FAULT
                   & ~baja.Status.DOWN
                   & ~baja.Status.STALE
                   & ~baja.Status.NULL; //all bits set except those
        verify(baja.Status.make(bits).isValid());
      });
    });
    
    describe("isOk()", function Status_isOk() {
      it("returns true if bits = 0", function bits0() {
        verify(baja.Status.make(0).isOk());
        verify(!baja.Status.make(1).isOk());
      });
    });
    
    describe("isDisabled()", function Status_isDisabled() {
      it("returns true if disabled bit is set", function disabledBitSet() {
        verify(baja.Status.make(0xFF).isDisabled());
        verify(baja.Status.make(0x01).isDisabled());
        verify(!baja.Status.make(0xFE).isDisabled());
        verify(!baja.Status.make(0x00).isDisabled());
      });
    });
    
    describe("isFault()", function Status_isFault() {
      it("returns true if fault bit is set", function faultBitSet() {
        verify(baja.Status.make(0xFF).isFault());
        verify(baja.Status.make(0x02).isFault());
        verify(!baja.Status.make(0xFD).isFault());
        verify(!baja.Status.make(0x00).isFault());
      });
    });
    
    describe("isDown()", function Status_isDown() {
      it("returns true if down bit is set", function downBitSet() {
        verify(baja.Status.make(0xFF).isDown());
        verify(baja.Status.make(0x04).isDown());
        verify(!baja.Status.make(0xFB).isDown());
        verify(!baja.Status.make(0x00).isDown());
      });
    });
    
    describe("isAlarm()", function Status_isAlarm() {
      it("returns true if alarm bit is set", function alarmBitSet() {
        verify(baja.Status.make(0xFF).isAlarm());
        verify(baja.Status.make(0x08).isAlarm());
        verify(!baja.Status.make(0xF7).isAlarm());
        verify(!baja.Status.make(0x00).isAlarm());
      });
    });
    
    describe("isStale()", function Status_isStale() {
      it("returns true if stale bit is set", function staleBitSet() {
        verify(baja.Status.make(0xFF).isStale());
        verify(baja.Status.make(0x10).isStale());
        verify(!baja.Status.make(0xEF).isStale());
        verify(!baja.Status.make(0x00).isStale());
      });
    });
    
    describe("isOverridden()", function Status_isOverridden() {
      it("returns true if overridden bit is set", function overriddenBitSet() {
        verify(baja.Status.make(0xFF).isOverridden());
        verify(baja.Status.make(0x20).isOverridden());
        verify(!baja.Status.make(0xDF).isOverridden());
        verify(!baja.Status.make(0x00).isOverridden());
      });
    });

    describe("isNull()", function Status_isNull() {
      it("returns true if null bit is set", function nullBitSet() {
        verify(baja.Status.make(0xFF).isNull());
        verify(baja.Status.make(0x40).isNull());
        verify(!baja.Status.make(0xBF).isNull());
        verify(!baja.Status.make(0x00).isNull());
      });
    });
    
    describe("isUnackedAlarm()", function Status_isUnackedAlarm() {
      it("returns true if unacked alarm bit is set", function unackedBitSet() {
        verify(baja.Status.make(0xFF).isUnackedAlarm());
        verify(baja.Status.make(0x80).isUnackedAlarm());
        verify(!baja.Status.make(0x7F).isUnackedAlarm());
        verify(!baja.Status.make(0x00).isUnackedAlarm());
      });
    });
  });
  
  describe("baja.Time", function Time() {
    var millisInHour = 1000 * 60 * 60,
        millisInMinute = 1000 * 60;
    
    function toMillis(hour, min, sec, ms) {
      return hour * millisInHour +
             min * millisInMinute +
             sec * 1000 +
             ms;
    }
    function verifyTime(time, hour, min, sec, ms) {
      verify(time);
      verifyEq(time.getHour(), hour);
      verifyEq(time.getMinute(), min);
      verifyEq(time.getSecond(), sec);
      verifyEq(time.getMillisecond(), ms);
      verifyEq(time.getTimeOfDayMillis(), toMillis(hour, min, sec, ms));
    }
    
    describe("constructor", function Time_constructor() {
      it("accepts hour, minute, second, and ms", function acceptsArgs() {
        var hour = 1, minute = 2, second = 3, ms = 4,
            time = new baja.Time(hour, minute, second, ms);
        verifyEq(time.$hour, hour);
        verifyEq(time.$min, minute);
        verifyEq(time.$sec, second);
        verifyEq(time.$ms, ms);
      });
      
      it("is registered on baja:Time type", function isRegistered() {
        expect(baja.$('baja:Time')).toBe(baja.Time.DEFAULT);
      });
    });
    
    describe("make() (static)", function Time_make() {
      it("accepts an object literal with hour/min/sec/ms", function acceptsHourMinSecMs() {
        var hour = 1, min = 2, sec = 3, ms = 4,
            time = baja.Time.make({
              hour: hour,
              min: min,
              sec: sec,
              ms: ms
            });
        verifyTime(time, hour, min, sec, ms);
      });
      
      it("defaults hour/min/sec/ms to 0", function defaultsHourMinSecMsTo0() {
        var time = baja.Time.make({});
        verifyTime(time, 0, 0, 0, 0);
      });
      
      it("accepts an object literal with relTime property", function acceptsRelTime() {
        var hour = 10, min = 9, sec = 8, ms = 7,
            relTime = baja.RelTime.make({
              hours: hour,
              minutes: min,
              seconds: sec,
              ms: ms
            }),
            time = baja.Time.make({ relTime: relTime });
        
        verifyTime(time, hour, min, sec, ms);
      });
      
      it("ignores the days field on a relTime", function ignoresRelTimeDays() {
        var day = 7, hour = 23, min = 59, sec = 59, ms = 999,
            relTime = baja.RelTime.make({
              days: day,
              hours: hour,
              minutes: min,
              seconds: sec,
              ms: ms
            }),
            time = baja.Time.make({ relTime: relTime });
        
        verifyTime(time, hour, min, sec, ms);
      });
      
      it("accepts a number of milliseconds past midnight", function acceptsMillisPastMidnight() {
        var hour = 5, min = 4, sec = 3, ms = 2,
            milliseconds = toMillis(hour, min, sec, ms),
            time = baja.Time.make(milliseconds);
        
        verifyTime(time, hour, min, sec, ms);
      });
      
      it("accepts an object literal with milliseconds property", function acceptsMillisecondsProperty() {
        var hour = 23, min = 59, sec = 59, ms = 999,
            milliseconds = toMillis(hour, min, sec, ms),
            time = baja.Time.make({ milliseconds: milliseconds });
        
        verifyTime(time, hour, min, sec, ms);
      });
      
      it("returns baja.Time.DEFAULT if millis === 0", function returnsDefaultIf0() {
        expect(baja.Time.make(0)).toBe(baja.Time.DEFAULT);
        expect(baja.Time.make()).toBe(baja.Time.DEFAULT);
        expect(baja.Time.make({ milliseconds: 0 })).toBe(baja.Time.DEFAULT);
        expect(baja.Time.make({ hour: 0, min: 0, sec: 0, ms: 0 })).toBe(baja.Time.DEFAULT);
        expect(baja.Time.make(1)).not.toBe(baja.Time.DEFAULT);
      });
      
      it("dies if hours less than 0", function diesHoursLessThan0() {
        errTest(function () {
          baja.Time.make({ hour: -1, min: 0, sec: 0, ms: 0 });
        }, 'Invalid time: -1:0:0.0');
      });
      
      it("dies if hours greater than 23", function diesHoursGreaterThan23() {
        errTest(function () {
          baja.Time.make({ hour: 24, min: 0, sec: 0, ms: 0 });
        }, 'Invalid time: 24:0:0.0');
      });
      
      it("dies if minutes less than 0", function diesMinutesLessThan0() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: -1, sec: 0, ms: 0 });
        }, 'Invalid time: 0:-1:0.0');
      });
      
      it("dies if minutes greater than 59", function diesMinutesGreaterThan59() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 60, sec: 0, ms: 0 });
        }, 'Invalid time: 0:60:0.0');
      });
      
      it("dies if seconds less than 0", function diesSecondsLessThan0() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 0, sec: -1, ms: 0 });
        }, 'Invalid time: 0:0:-1.0');
      });
      
      it("dies if seconds greater than 59", function diesSecondsGreaterThan59() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 0, sec: 60, ms: 0 });
        }, 'Invalid time: 0:0:60.0');
      });
      
      it("dies if millis less than 0", function diesMillisLessThan0() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 0, sec: 0, ms: -1 });
        }, 'Invalid time: 0:0:0.-1');
      });
      
      it("dies if millis greater than 999", function diesMillisGreaterThan999() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 0, sec: 0, ms: 1000 });
        }, 'Invalid time: 0:0:0.1000');
      });
      
      it("dies if non-numeric hours given", function diesNonNumericHours() {
        errTest(function () {
          baja.Time.make({ hour: '0', min: 0, sec: 0, ms: 0 });
        }, typeMismatchError(String, Number));
      });
      
      it("dies if non-numeric minutes given", function diesNonNumericMinutes() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: '0', sec: 0, ms: 0 });
        }, typeMismatchError(String, Number));
      });
      
      it("dies if non-numeric seconds given", function diesNonNumericSeconds() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 0, sec: '0', ms: 0 });
        }, typeMismatchError(String, Number));
      });
      
      it("dies if non-numeric ms given", function diesNonNumericMs() {
        errTest(function () {
          baja.Time.make({ hour: 0, min: 0, sec: 0, ms: '0' });
        }, typeMismatchError(String, Number));
      });
    });
    
    describe("decodeFromString()", function Time_decodeFromString() {
      function verifyDecode(str, hour, min, sec, ms) {
        var time = baja.Time.DEFAULT.decodeFromString(str);
        verifyTime(time, hour, min, sec, ms);
      }
      
      it("decodes from HH:MM:SS.SSS format", function decodesWMillis() {
        verifyDecode('00:00:00.000', 0, 0, 0, 0);
        verifyDecode('01:02:03.456', 1, 2, 3, 456);
        verifyDecode('23:59:59.999', 23, 59, 59, 999);
      });
      
      it("dies on invalid time format", function diesOnInvalid() {
        errTest(function () {
          baja.Time.DEFAULT.decodeFromString('00:00:00');
        }, 'Failed to decode time: 00:00:00');
      });
    });
    
    describe("encodeToString()", function Time_encodeToString() {
      function verifyEncode(hour, min, sec, ms, str) {
        var time = baja.Time.make({ hour: hour, min: min, sec: sec, ms: ms });
        verifyEq(time.encodeToString(), str);
      }
      
      it("encodes to HH:MM:SS.SSS format", function encodes() {
        verifyEncode(0, 0, 0, 0, "00:00:00.000");
        verifyEncode(1, 1, 1, 1, "01:01:01.001");
        verifyEncode(23, 59, 59, 999, "23:59:59.999");
      });
    });
    
    describe("equals()", function Time_equals() {
      it("returns true if millis of day is the same", function sameMillis() {
        verify(baja.Time.DEFAULT.equals(baja.Time.DEFAULT));
        verify(baja.Time.make(1).equals(baja.Time.make(1)));
        verify(baja.Time.make({ hour: 1, min: 1, sec: 1, ms: 1 })
            .equals(baja.Time.make(3661001)));
      });
      
      it("returns false if millis of day is different", function differentMillis() {
        verify(!baja.Time.make(1).equals(baja.Time.make(2)));
      });
      
      it("returns false for different type", function differentType() {
        verify(!baja.Time.DEFAULT.equals("hello"));
        verify(!baja.Time.make(100).equals(baja.$('baja:Component')));
      });
      
      it("returns false for non-Baja object", function nonBajaObject() {
        verify(!baja.Time.DEFAULT.equals({}));
      });
      
      it("returns false for null/undefined", function nullUndefined() {
        verify(!baja.Time.DEFAULT.equals(null));
        verify(!baja.Time.DEFAULT.equals());
      });
    });
    
    describe("getHour()", function Time_getHour() {
      it("returns hours of day", function returnsHours() {
        verifyEq(baja.Time.DEFAULT.getHour(), 0);
        verifyEq(baja.Time.make({ hour: 3, min: 4, sec: 5, ms: 6 }).getHour(), 3);
        verifyEq(baja.Time.make(14400001).getHour(), 4);
        verifyEq(baja.Time.make(90000000).getHour(), 1);
      });
    });
    
    describe("getMinute()", function Time_getMinute() {
      it("returns minutes of hour", function returnsMinutes() {
        verifyEq(baja.Time.DEFAULT.getMinute(), 0);
        verifyEq(baja.Time.make({ hour: 3, min: 4, sec: 5, ms: 6 }).getMinute(), 4);
        verifyEq(baja.Time.make(300001).getMinute(), 5);
      });
    });
    
    describe("getSecond()", function Time_getSecond() {
      it("returns seconds of minute", function returnsSeconds() {
        verifyEq(baja.Time.DEFAULT.getSecond(), 0);
        verifyEq(baja.Time.make({ hour: 3, min: 4, sec: 5, ms: 6 }).getSecond(), 5);
        verifyEq(baja.Time.make(17001).getSecond(), 17);
      });
    });
    
    describe("getMillisecond()", function Time_getMillisecond() {
      it("returns milliseconds of second", function returnsMilliseconds() {
        verifyEq(baja.Time.DEFAULT.getMillisecond(), 0);
        verifyEq(baja.Time.make({ hour: 3, min: 4, sec: 5, ms: 6 }).getMillisecond(), 6);
        verifyEq(baja.Time.make(467347234).getMillisecond(), 234);
      });
    });
    
    describe("getTimeOfDayMillis()", function Time_getTimeOfDayMillis() {
      it("returns millis of day", function returnsMillis() {
        verifyEq(baja.Time.make(1234567).getTimeOfDayMillis(), 1234567);
        verifyEq(baja.Time.make({ hour: 3, min: 4, sec: 5, ms: 6})
            .getTimeOfDayMillis(), 11045006);
        verifyEq(baja.Time.make({ hour: 23, min: 59, sec: 59, ms: 999 })
            .getTimeOfDayMillis(), 86399999);
      });
      
      it("truncates to day", function truncates() {
        verifyEq(baja.Time.make(86400001).getTimeOfDayMillis(), 1);
      });
      
      it("caches calculated result", function cachesCalculatedResult() {
        var time = baja.Time.make(55555);
        verifyEq(time.getTimeOfDayMillis(), 55555);
        verifyEq(time.$timeOfDayMs, 55555);
      });
    });
    
    describe("add()", function Time_add() {
      it("adds a RelTime", function addsRelTime() {
        var time = baja.Time.make(5000),
            relTime = baja.RelTime.make(555),
            newTime = time.add(relTime);
        verifyEq(newTime.getTimeOfDayMillis(), 5555);
      });
      
      it("adds a RelTime and truncates to day", function addsRelTimeTruncates() {
        var time = baja.Time.make(86399999),
            relTime = baja.RelTime.make(2),
            newTime = time.add(relTime);
        verifyEq(newTime.getTimeOfDayMillis(), 1);
      });
      
      it("adds a number of milliseconds", function addsMillis() {
        var time = baja.Time.make(5000),
            newTime = time.add(333);
        verifyEq(newTime.getTimeOfDayMillis(), 5333);
      });
      
      it("adds a number of milliseconds and truncates to day", function addsMillisTruncates() {
        var time = baja.Time.make(86399999),
            newTime = time.add(3);
        verifyEq(newTime.getTimeOfDayMillis(), 2);
      });
      
      it("subtracts a number of milliseconds", function subtractsMillis() {
        var time = baja.Time.make(9999),
            newTime = time.add(-1999);
        verifyEq(newTime.getTimeOfDayMillis(), 8000);
      });
      
      it("subtracts a number of milliseconds and truncates to day", function subtractsMillisTruncates() {
        var time = baja.Time.make(3),
            newTime = time.add(-5);
        verifyEq(newTime.getTimeOfDayMillis(), 86399998);
      });
      
      it("adds a Time", function addsTime() {
        var time = baja.Time.make(65),
            newTime = time.add(baja.Time.make(12));
        verifyEq(newTime.getTimeOfDayMillis(), 77);
      });
      
      it("adds a Time and truncates to day", function addsTimeTruncates() {
        var time = baja.Time.make(86399997),
            newTime = time.add(baja.Time.make(10));
        verifyEq(newTime.getTimeOfDayMillis(), 7);
      });
      
      it("dies on undefined", function diesOnUndefined() {
        errTest(function () {
          baja.Time.DEFAULT.add(undefined);
        }, undefinedError());
      });
      
      it("dies on a non-time, non-numeric input", function diesOnNonTime() {
        errTest(function () {
          baja.Time.DEFAULT.add('hello');
        }, typeMismatchError(String, Number));
      });
    });
    
    describe("isBefore()", function Time_isBefore() {
      it("returns true if this time is before the input time", function trueIfBefore() {
        verify(baja.Time.make(0).isBefore(baja.Time.make(1)));
        verify(baja.Time.make(86399998).isBefore(baja.Time.make(86399999)));
        verify(baja.Time.make(0).isBefore(baja.Time.make(86399999)));
      });
      
      it("returns false if this time is not before the input time", function falseIfAfter() {
        verify(!baja.Time.make(1).isBefore(baja.Time.make(0)));
        verify(!baja.Time.make(86399999).isBefore(baja.Time.make(86399998)));
        verify(!baja.Time.make(86399999).isBefore(baja.Time.make(0)));
      });
      
      it("returns false if the times are the same", function falseIfSame() {
        verify(!baja.Time.make(1).isBefore(baja.Time.make(1)));
        verify(!baja.Time.make(86399999).isBefore(baja.Time.make(86399999)));
      });
      
      it("dies on a non-Time input", function diesOnNonTime() {
        errTest(function () {
          baja.Time.DEFAULT.isBefore(5);
        }, typeMismatchError(Number, baja.Time));
      });
    });
    
    describe("isAfter()", function Time_isAfter() {
      it("returns true if this time is after the input time", function trueIfAfter() {
        verify(baja.Time.make(1).isAfter(baja.Time.make(0)));
        verify(baja.Time.make(86399999).isAfter(baja.Time.make(86399998)));
        verify(baja.Time.make(86399999).isAfter(baja.Time.make(0)));
      });
      
      it("returns false if this time is not after the input time", function falseIfBefore() {
        verify(!baja.Time.make(0).isAfter(baja.Time.make(1)));
        verify(!baja.Time.make(86399998).isAfter(baja.Time.make(86399999)));
        verify(!baja.Time.make(0).isAfter(baja.Time.make(86399999)));
      });
      
      it("returns false if the times are the same", function falseIfSame() {
        verify(!baja.Time.make(1).isAfter(baja.Time.make(1)));
        verify(!baja.Time.make(86399999).isAfter(baja.Time.make(86399999)));
      });
      
      it("dies on a non-Time input", function diesOnNonTime() {
        errTest(function () {
          baja.Time.DEFAULT.isAfter(5);
        }, typeMismatchError(Number, baja.Time));
      });
    });
    
    describe("newCopy()", function Time_newCopy() {
      it("returns this", function returnsThis() {
        var time = baja.Time.make(8584385);
        expect(time.newCopy()).toBe(time);
      });
    });
    
    describe("toString()", function Time_toString() {
      it("formats to h:mm a by default", function formatshmmaDefault() {
        verifyEq(baja.Time.make({ hour: 1, min: 2 }).toString(), '1:02 AM');
        verifyEq(baja.Time.make({ hour: 23, min: 59 }).toString(), '11:59 PM');
      });
      
      it("formats h:mm", function formatshmm() {
        var obj = { textPattern: 'h:mm' };
        verifyEq(baja.Time.make({ hour: 1, min: 2 }).toString(obj), '1:02');
        verifyEq(baja.Time.make({ hour: 10, min: 20 }).toString(obj), '10:20');
      });
      it("formats hh:mm", function formatshhmm() {
        var obj = { textPattern: 'hh:mm' };
        verifyEq(baja.Time.make({ hour: 1, min: 2 }).toString(obj), '01:02');
        verifyEq(baja.Time.make({ hour: 23, min: 59 }).toString(obj), '11:59');
      });
      
      it("formats HH:mm", function formatsHHmm() {
        var obj = { textPattern: 'HH:mm' };
        verifyEq(baja.Time.make({ hour: 1, min: 2 }).toString(obj), '01:02');
        verifyEq(baja.Time.make({ hour: 23, min: 59 }).toString(obj), '23:59');
      });
      
      it("formats HH:mm:ss (no millis)", function formatsHHmmssNoMillis() {
        var obj = {
          textPattern: 'HH:mm:ss',
          show: baja.TimeFormat.SHOW_SECONDS
        };
        verifyEq(baja.Time.make({ hour: 23, min: 59, sec: 12 }).toString(obj), 
            '23:59:12');
      });
      
      it("formats HH:mm:ss (with millis)", function formatsHHmmssWithMillis() {
        var obj = {
          textPattern: 'HH:mm:ss',
          show: baja.TimeFormat.SHOW_SECONDS | baja.TimeFormat.SHOW_MILLIS
        };
        verifyEq(baja.Time.make({ hour: 13, min: 45, sec: 55, ms: 567 })
            .toString(obj), '13:45:55.567');
      });
      
      it("formats hh:mm a", function formatshhmma() {
        var obj = { textPattern: 'hh:mm a' };
        verifyEq(baja.Time.make({ hour: 5, min: 15 }).toString(obj), '05:15 AM');
        verifyEq(baja.Time.make({ hour: 20, min: 30 }).toString(obj), '08:30 PM');
      });
      
      it("formats HH:mm a", function formatsHHmma() {
        var obj = { textPattern: 'HH:mm a' };
        verifyEq(baja.Time.make({ hour: 5, min: 15 }).toString(obj), '05:15 AM');
        verifyEq(baja.Time.make({ hour: 20, min: 30 }).toString(obj), '20:30 PM');
      });
    });
  });
  
  describe("baja.Date", function bajaDate() {
    function verifyDate(date, year, month, day) {
      verifyEq(date.getYear(), year);
      verify(date.getMonth().equals(baja.$('baja:Month').get(month)));
      verifyEq(date.getDay(), day);
    }
    
    describe("constructor", function Date_constructor() {
      it("accepts year month and day", function acceptsYearMonthDay() {
        var year = 2012, month = 2, day = 6, 
            date = new baja.Date(year, month, day);
        verifyDate(date, year, month, day);
      });
      
      it("is registered on type baja:Date", function isRegistered() {
        expect(baja.$('baja:Date')).toBe(baja.Date.DEFAULT);
      });
    });
    
    describe("make() (static)", function Date_make() {
      it("accepts year month (numeric) and day", function acceptsYearMonthNumericDay() {
        var year = 2000, month = 2, day = 5, 
            date = baja.Date.make({ year: year, month: month, day: day });
        verifyDate(date, year, month, day);
      });
      
      it("accepts year month (baja:Month) and day", function acceptsYearMonthEnumAndDay() {
        var year = 2001, month = baja.$('baja:Month').get(3), day = 20,
            date = baja.Date.make({ year: year, month: month, day: day });
        verifyDate(date, year, 3, day);
      });
      
      it("accepts jsDate Javascript Date property", function acceptsJSDate() {
        var jsDate = new Date(2001, 4, 9, 0, 0, 0, 0),
            date = baja.Date.make({ jsDate: jsDate });
        verifyDate(date, 2001, 4, 9);
      });
      
      it("returns DEFAULT if all values match DEFAULT values", function returnsDefault() {
        var date = baja.Date.make({
              year: baja.Date.DEFAULT.getYear(),
              month: baja.Date.DEFAULT.getMonth(),
              day: baja.Date.DEFAULT.getDay()
            });
        expect(date).toBe(baja.Date.DEFAULT);
      });
      
      it("dies on missing year", function diesOnMissingYear() {
        errTest(function () {
          baja.Date.make({ month: 1, day: 1 });
        }, undefinedError(Number));
      });
      
      it("dies on non-numeric year", function diesOnNonNumericYear() {
        errTest(function () {
          baja.Date.make({ year: "2011", month: 1, day: 1 });
        }, typeMismatchError(String, Number));
      });
      
      it("dies on missing month", function diesOnMissingMonth() {
        errTest(function () {
          baja.Date.make({ year: 2000, day: 1 });
        }, undefinedError(Number));
      });
      
      it("dies on non-numeric/non-enum month", function diesOnNonNumericMonth() {
        errTest(function () {
          baja.Date.make({ year: 2011, month: "1", day: 1 });
        }, typeMismatchError(String, Number));
      });
      
      it("dies on missing day", function diesOnMissingDay() {
        errTest(function () {
          baja.Date.make({ year: 2000, month: 1 });
        }, undefinedError(Number));
      });
      
      it("dies on non-numeric day", function diesOnNonNumericDay() {
        errTest(function () {
          baja.Date.make({ year: 2011, month: 1, day: "1" });
        }, typeMismatchError(String, Number));
      });
      
      it("dies on year < 0", function diesOnYearLessThan0() {
        baja.Date.make({ year: 0, month: 1, day: 1 });
        errTest(function () {
          baja.Date.make({ year: -1, month: 1, day: 1 });
        }, 'Invalid date range');
      });
      
      it("dies on month < 0", function diesOnMonthLessThan0() {
        baja.Date.make({ year: 1, month: 0, day: 1 });
        errTest(function () {
          baja.Date.make({ year: 1, month: -1, day: 1 });
        }, 'Invalid date range');
      });
      
      it("dies on month > 11", function diesOnMonthGreaterThan11() {
        baja.Date.make({ year: 1, month: 11, day: 1 });
        errTest(function () {
          baja.Date.make({ year: 1, month: 12, day: 1 });
        }, 'Invalid date range');
      });
      
      it("dies on day < 1", function diesOnDayLessThan1() {
        baja.Date.make({ year: 1, month: 1, day: 1 });
        errTest(function () {
          baja.Date.make({ year: 1, month: 1, day: 0 });
        }, 'Invalid date range');
      });
      
      it("dies on day > 31", function diesOnDayGreaterThan31() {
        baja.Date.make({ year: 1, month: 1, day: 31 });
        errTest(function () {
          baja.Date.make({ year: 1, month: 1, day: 32 });
        }, 'Invalid date range');
      });
    });
    
    describe("decodeFromString()", function Date_decodeFromString() {
      function decode(str) {
        return baja.Date.DEFAULT.decodeFromString(str);
      }
      
      it("decodes from yyyy-mm-dd format", function decodesyymmdd() {
        verifyDate(decode('0000-01-01'), 0, 0, 1);
        verifyDate(decode('2008-11-24'), 2008, 10, 24);
        verifyDate(decode('1981-06-17'), 1981, 5, 17);
        verifyDate(decode('1979-05-28'), 1979, 4, 28);
        verifyDate(decode('9999-12-31'), 9999, 11, 31);
      });
      
      it("dies on invalid format", function diesOnInvalidFormat() {
        errTest(function () {
          decode('x999-99-99');
        }, 'Could not decode baja.Date: x999-99-99');
      });
    });
    
    describe("encodeToString()", function Date_encodeToString() {
      function verifyEncode(year, month, day, str) {
        verifyEq(baja.Date.make({ year: year, month: month, day: day })
            .encodeToString(), str);
      }
      
      it("encodes to yyyy-mm-dd format", function encodesyymmdd() {
        verifyEncode(0, 0, 1, '0000-01-01');
        verifyEncode(2008, 10, 24, '2008-11-24');
        verifyEncode(1981, 5, 17, '1981-06-17');
        verifyEncode(1979, 4, 28, '1979-05-28');
        verifyEncode(9999, 11, 31, '9999-12-31');
      });
    });
    
    describe("equals()", function Date_equals() {
      it("returns true if year/month/day are the same", function trueIfSameYMD() {
        verify(baja.Date.make({ year: 0, month: 0, day: 1 }).equals(
               baja.Date.make({ year: 0, month: 0, day: 1 })));
        verify(baja.Date.make({ year: 9999, month: 11, day: 31 }).equals(
               baja.Date.make({ year: 9999, month: 11, day: 31 })));
      });
      
      it("returns false if year/month/day are different", function falseIfDifferentYMD() {
        verify(!baja.Date.make({ year: 0, month: 0, day: 1 }).equals(
                baja.Date.make({ year: 0, month: 0, day: 2 })));
        verify(!baja.Date.make({ year: 9999, month: 10, day: 31 }).equals(
                baja.Date.make({ year: 9999, month: 11, day: 31 })));
        verify(!baja.Date.make({ year: 9998, month: 11, day: 31 }).equals(
                baja.Date.make({ year: 9999, month: 11, day: 31 })));
      });
      
      it("returns true if jsDate is the same day", function trueIfSameJsDate() {
        var jsDate1 = new Date(),
            jsDate2 = new Date(jsDate1.getTime());
        verify(baja.Date.make({ jsDate: jsDate1 }).equals(
               baja.Date.make({ jsDate: jsDate2 })));
      });
      
      it("returns false if jsDate is different day", function falseIfDifferentJsDate() {
        var jsDate1 = new Date(),
            jsDate2 = new Date(jsDate1.getTime() - baja.RelTime.MILLIS_IN_DAY);
        verify(!baja.Date.make({ jsDate: jsDate1 }).equals(
                baja.Date.make({ jsDate: jsDate2 })));
      });
      
      it("returns false for a non-Date object", function falseIfNonDate() {
        var date = baja.Date.make({ year: 2012, month: 2, day: 6 });
        verify(!date.equals('2012-03-06'));
        verify(!date.equals({}));
      });
      
      it("returns false for null/undefined()", function falseIfUndefined() {
        var date = baja.Date.make({ year: 2012, month: 2, day: 6 });
        verify(!date.equals(null));
        verify(!date.equals(undefined));
      });
    });
    
    describe("today()", function Date_today() {
      it("returns a date representing today", function returnsToday() {
        var jsDate = new Date(),
            today = baja.Date.today();
        
        //and now, JUST IN CASE we're running tests at the stroke of midnight...
        verify(Math.abs(today.getYear() - jsDate.getFullYear()) <= 1);
        verify(Math.abs(today.getMonth().getOrdinal() - jsDate.getMonth()) <= 1);
        verify(Math.abs(today.getDay() - jsDate.getDate()) <= 1);
      });
    });
    
    describe("getYear()", function Date_getYear() {
      it("returns the year", function returnsYear() {
        verifyEq(baja.Date.make({ year: 2011, month: 1, day: 1 }).getYear(), 2011);
      });
    });
    
    describe("getMonth()", function Date_getMonth() {
      it("returns the baja:Month", function returnsMonth() {
        var date = baja.Date.make({ year: 0, month: 6, day: 3 }),
            month = date.getMonth();
        verifyEq(month.getType().toString(), 'baja:Month');
        verifyEq(month.getOrdinal(), 6);
        verifyEq(month.getTag(), 'july');
      });
    });
    
    describe("getDay()", function Date_getDay() {
      it("returns the day", function returnsDay() {
        verifyEq(baja.Date.make({ year: 0, month: 0, day: 27 }).getDay(), 27);
      });
    });
    
    describe("getJsDate()", function Date_getJsDate() {
      it("returns a Javascript date", function returnsJsDate() {
        var date = baja.Date.make({ year: 2005, month: 4, day: 15 }),
            jsDate = date.getJsDate();
        
        verifyEq(jsDate.getFullYear(), 2005);
        verifyEq(jsDate.getMonth(), 4);
        verifyEq(jsDate.getDate(), 15);
        verifyEq(jsDate.getHours(), 0);
        verifyEq(jsDate.getMinutes(), 0);
        verifyEq(jsDate.getSeconds(), 0);
        verifyEq(jsDate.getMilliseconds(), 0);
      });
    });
    
    describe("getWeekday()", function Date_getWeekday() {
      it("returns the baja:Weekday", function returnsWeekday() {
        var date = baja.Date.make({ year: 2012, month: 2, day: 6 }),
            weekday = date.getWeekday();
        verifyEq(weekday.getType().toString(), 'baja:Weekday');
        verifyEq(weekday.getTag(), 'tuesday');
      });
    });
    
    describe("isBefore()", function Date_isBefore() {
      function before(year1, month1, day1, year2, month2, day2) {
        var date1 = baja.Date.make({ year: year1, month: month1, day: day1 }),
            date2 = baja.Date.make({ year: year2, month: month2, day: day2 });
        return date1.isBefore(date2);
      }
      it("returns true if this date is before the given date", function returnsTrueIfBefore() {
        verify(before(2012, 2, 6, 2013, 2, 6));
        verify(before(2012, 2, 6, 2012, 3, 6));
        verify(before(2012, 2, 6, 2012, 2, 7));
        verify(before(2012, 11, 31, 2013, 0, 1));
        verify(before(2012, 2, 31, 2012, 3, 1));
      });
      
      it("returns false if this date is after the given date", function returnsFalseIfAfter() {
        verify(!before(2013, 2, 6, 2012, 2, 6));
        verify(!before(2013, 3, 6, 2012, 2, 6));
        verify(!before(2012, 2, 7, 2012, 2, 6));
        verify(!before(2013, 0, 1, 2012, 11, 31));
        verify(!before(2013, 3, 1, 2012, 2, 31));
      });
      
      it("returns false if dates are the same", function returnsFalseIfSame() {
        verify(!before(2012, 2, 6, 2012, 2, 6));
      });
    });
    
    describe("isAfter()", function Date_isAfter() {
      function after(year1, month1, day1, year2, month2, day2) {
        var date1 = baja.Date.make({ year: year1, month: month1, day: day1 }),
            date2 = baja.Date.make({ year: year2, month: month2, day: day2 });
        return date1.isAfter(date2);
      }
      it("returns true if this date is after the given date", function returnsTrueIfAfter() {
        verify(after(2013, 2, 6, 2012, 2, 6));
        verify(after(2013, 3, 6, 2012, 2, 6));
        verify(after(2012, 2, 7, 2012, 2, 6));
        verify(after(2013, 0, 1, 2012, 11, 31));
        verify(after(2013, 3, 1, 2012, 2, 31));
      });
      
      it("returns false if this date is before the given date", function returnsFalseIfBefore() {
        verify(!after(2012, 2, 6, 2013, 2, 6));
        verify(!after(2012, 2, 6, 2012, 3, 6));
        verify(!after(2012, 2, 6, 2012, 2, 7));
        verify(!after(2012, 11, 31, 2013, 0, 1));
        verify(!after(2012, 2, 31, 2012, 3, 1));
      });
      
      it("returns false if dates are the same", function returnsFalseIfSame() {
        verify(!after(2012, 2, 6, 2012, 2, 6));
      });
    });
    
    describe("toString()", function Date_toString() {
      it("formats to DD-MMM-YY by default", function yymmmddByDefault() {
        verifyEq(baja.Date.make({ year: 2012, month: 2, day: 6 }).toString(),
            '06-Mar-12');
      });
      
      it("formats YY-MM-DD", function formatsyymmdd() {
        var obj = { textPattern: 'YY-MM-DD' },
            date = baja.Date.make({ year: 2012, month: 2, day: 6 });
        verifyEq(date.toString(obj), '12-03-06');
      });
      
      it("formats YYYY-MM-DD", function formatsyyyymmdd() {
        var obj = { textPattern: 'YYYY-MM-DD' },
            date = baja.Date.make({ year: 2012, month: 2, day: 6 });
        verifyEq(date.toString(obj), '2012-03-06');
      });
      
      it("formats YY-M-D", function formatsyyyymmdd() {
        var obj = { textPattern: 'YY-M-D' },
            date = baja.Date.make({ year: 2012, month: 2, day: 6 });
        verifyEq(date.toString(obj), '12-3-6');
      });
    });
    
    describe("valueOf()", function Date_valueOf() {
      it("returns this", function returnsThis() {
        var date = baja.Date.make({ year: 1999, month: 8, day: 4 });
        expect(date.valueOf()).toBe(date);
      });
    });
  });
  
  describe("baja.RelTime", function RelTime() {
    function verifyRelTime(rt, days, hours, minutes, seconds, ms) {
      var totalDays = days,
          totalHours = totalDays * 24 + hours,
          totalMinutes = totalHours * 60 + minutes,
          totalSeconds = totalMinutes * 60 + seconds,
          totalMillis = totalSeconds * 1000 + ms;
      verifyEq(rt.getDays(), totalDays);
      verifyEq(rt.getDaysPart(), days);
      verifyEq(rt.getHours(), totalHours);
      verifyEq(rt.getHoursPart(), hours);
      verifyEq(rt.getMinutes(), totalMinutes);
      verifyEq(rt.getMinutesPart(), minutes);
      verifyEq(rt.getSeconds(), totalSeconds);
      verifyEq(rt.getSecondsPart(), seconds);
      verifyEq(rt.getMillis(), totalMillis);
      verifyEq(rt.getMillisPart(), ms);
    }
    describe("constructor", function RelTime_constructor() {
      it("accepts a number of milliseconds", function acceptsMillis() {
        var rt = new baja.RelTime(888);
        verifyRelTime(rt, 0, 0, 0, 0, 888);
      });
      
      it("is registered on type baja:RelTime", function isRegistered() {
        expect(baja.$('baja:RelTime')).toBe(baja.RelTime.DEFAULT);
      });
    });
    
    describe("make() (static)", function RelTime_make() {
      it("accepts a number of milliseconds", function acceptsMillis() {
        var rt = baja.RelTime.make(93784005);
        verifyRelTime(rt, 1, 2, 3, 4, 5);
      });
      
      it("accepts a negative number of milliseconds", function acceptsNegMillis() {
        var rt = baja.RelTime.make(-93784005);
        verifyRelTime(rt, -1, -2, -3, -4, -5);
      });
      
      it("ignores decimal places", function ignoresDecimalPlaces() {
        verifyEq(baja.RelTime.make(12345.5).getMillis(), 12345);
        verifyEq(baja.RelTime.make(-12345.5).getMillis(), -12345);
      });
      
      it("accepts days/hours/minutes/seconds/ms", function acceptsFields() {
         var days = 1, hours = 2, minutes = 3, seconds = 4, ms = 5,
             rt = baja.RelTime.make({
               days: days,
               hours: hours,
               minutes: minutes,
               seconds: seconds,
               ms: ms
             });
         
         verifyRelTime(rt, days, hours, minutes, seconds, ms);
      });
      
      it("returns DEFAULT if millis = 0", function returnsDefault() {
        expect(baja.RelTime.make(0)).toBe(baja.RelTime.DEFAULT);
        expect(baja.RelTime.make({ days: 0, hours: 0, minutes: 0, seconds: 0, ms: 0 }))
          .toBe(baja.RelTime.DEFAULT);
      });
      
      it("dies on a non-numeric ms property", function diesOnNonNumericMs() {
        errTest(function () {
          baja.RelTime.make({ ms: '123' });
        }, typeMismatchError(String, Number));
      });
    });
    
    describe("decodeFromString()", function RelTime_decodeFromString() {
      function decode(str) {
        return baja.RelTime.DEFAULT.decodeFromString(str);
      }
      
      it("simply decodes a number of millis", function decodesMillis() {
        verifyRelTime(decode('90061001'), 1, 1, 1, 1, 1);
      });
      
      it("dies on non-numeric string", function diesOnNonNumeric() {
        errTest(function () {
          decode("asdf");
        }, "Unable to create RelTime: asdf");
      });
    });
    
    describe("encodeToString()", function RelTime_encodeToString() {
      it("returns a string of number of milliseconds", function returnsString() {
        verifyEq(baja.RelTime.make(667788).encodeToString(), '667788');
      });
    });
    
    describe("getDataTypeSymbol()", function RelTime_getDataTypeSymbol() {
      it("returns 'r'", function returnsR() {
        verifyEq(baja.RelTime.DEFAULT.getDataTypeSymbol(), 'r');
      });
    });
    
    describe("valueOf()", function RelTime_valueOf() {
      it("returns number of milliseconds", function returnsMillis() {
        verifyEq(baja.RelTime.make(883366).valueOf(), 883366);
      });
    });
    
    describe("toString()", function RelTime_toString() {
      it("returns a string of number of milliseconds", function returnsString() {
        verifyEq(baja.RelTime.make(992277).toString(), "992277");
      });
    });
    
    describe("getDays()", function RelTime_getDays() {
      it("returns total number of days rounded down (positive)", function returnsDaysPos() {
        var rt = baja.RelTime.make({ days: 1.9 });
        verifyEq(rt.getDays(), 1);
      });
      
      it("returns total number of days rounded up (negative)", function returnsDaysNeg() {
        var rt = baja.RelTime.make({ days: -1.9 });
        verifyEq(rt.getDays(), -1);
      });
    });
    
    describe("getDaysPart()", function RelTime_getDaysPart() {
      it("returns number of days (partial), rounded down (positive)", function returnsDaysPos() {
        var rt = baja.RelTime.make({ days: 1.9 });
        verifyEq(rt.getDaysPart(), 1);
      });
      
      it("returns number of days (partial), rounded up (negative)", function returnsDaysNeg() {
        var rt = baja.RelTime.make({ days: -1.9 });
        verifyEq(rt.getDaysPart(), -1);
      });
    });
    
    describe("getHours()", function RelTime_getHours() {
      it("returns total number of hours rounded down (positive)", function returnsHoursPos() {
        var rt = baja.RelTime.make({ hours: 28.5 });
        verifyEq(rt.getHours(), 28);
      });
      
      it("returns total number of hours rounded up (negative)", function returnsHoursNeg() {
        var rt = baja.RelTime.make({ hours: -28.5 });
        verifyEq(rt.getHours(), -28);
      });
    });
    
    describe("getHoursPart()", function RelTime_getHoursPart() {
      it("returns number of hours (partial), rounded down (positive)", function returnsHoursPos() {
        var rt = baja.RelTime.make({ hours: 28.5 });
        verifyEq(rt.getHoursPart(), 4);
      });
      
      it("returns number of hours (partial), rounded up (negative)", function returnsHoursNeg() {
        var rt = baja.RelTime.make({ hours: -28.5 });
        verifyEq(rt.getHoursPart(), -4);
      });
    });
    
    describe("getMinutes()", function RelTime_getMinutes() {
      it("returns total number of minutes rounded down (positive)", function returnsMinutesPos() {
        var rt = baja.RelTime.make({ days: 1, hours: 4, minutes: 10.5 });
        verifyEq(rt.getMinutes(), 1690);
      });
      
      it("returns total number of minutes rounded up (negative)", function returnsMinutesNeg() {
        var rt = baja.RelTime.make({ days: -1, hours: -4, minutes: -10.5 });
        verifyEq(rt.getMinutes(), -1690);
      });
    });
    
    describe("getMinutesPart()", function RelTime_getMinutesPart() {
      it("returns number of minutes (partial), rounded down (positive)", function returnsMinutesPos() {
        var rt = baja.RelTime.make({ days: 1, hours: 4, minutes: 10.5 });
        verifyEq(rt.getMinutesPart(), 10);
      });
      
      it("returns number of minutes (partial), rounded up (negative)", function returnsMinutesNeg() {
        var rt = baja.RelTime.make({ days: -1, hours: -4, minutes: -10.5 });
        verifyEq(rt.getMinutesPart(), -10);
      });
    });
    
    describe("getSeconds()", function RelTime_getSeconds() {
      it("returns total number of seconds rounded down (positive)", function returnsSecondsPos() {
        var rt = baja.RelTime.make({ days: 1, hours: 4, minutes: 10, seconds: 20.5 });
        verifyEq(rt.getSeconds(), 101420);
      });
      
      it("returns total number of seconds rounded up (negative)", function returnsSecondsNeg() {
        var rt = baja.RelTime.make({ days: -1, hours: -4, minutes: -10, seconds: -20.5 });
        verifyEq(rt.getSeconds(), -101420);
      });
    });
    
    describe("getSecondsPart()", function RelTime_getSecondsPart() {
      it("returns number of seconds (partial), rounded down (positive)", function returnsSecondsPos() {
        var rt = baja.RelTime.make({ days: 1, hours: 4, minutes: 10, seconds: 20.5 });
        verifyEq(rt.getSecondsPart(), 20);
      });
      
      it("returns number of seconds (partial), rounded up (negative)", function returnsSecondsNeg() {
        var rt = baja.RelTime.make({ days: -1, hours: -4, minutes: -10, seconds: -20.5 });
        verifyEq(rt.getSecondsPart(), -20);
      });
    });
    
    describe("getMillis()", function RelTime_getMillis() {
      it("returns total number of millis rounded down (positive)", function returnsMillisPos() {
        verifyEq(baja.RelTime.make(33557799.5).getMillis(), 33557799);
      });
      
      it("returns total number of millis rounded up (negative)", function returnsMillisNeg() {
        verifyEq(baja.RelTime.make(-33557799.5).getMillis(), -33557799);
      });
    });
    
    describe("getMillisPart()", function RelTime_getMillisPart() {
      it("returns number of millis (partial), rounded down (positive)", function returnsMillisPos() {
        verifyEq(baja.RelTime.make(33557799.5).getMillisPart(), 799);
      });
      
      it("returns number of millis (partial), rounded up (negative)", function returnsMillisNeg() {
        verifyEq(baja.RelTime.make(-33557799.5).getMillisPart(), -799);
      });
    });
    
    describe("newCopy()", function RelTime_newCopy() {
      it("returns this", function returnsThis() {
        var rt = baja.RelTime.make(876);
        expect(rt.newCopy()).toBe(rt);
      });
    });
  });
  
  describe("baja.AbsTime", function AbsTime() {
    function verifyAbsTime(absTime, date, time, offset) {
      verify(absTime.getDate().equals(date));
      verify(absTime.getTime().equals(time));
      verifyEq(absTime.getOffset(), offset);
    }
    describe("constructor", function AbsTime_constructor() {
      it("accepts date, time, and offset", function takesDateTimeOffset() {
        var date = baja.Date.today(),
            time = baja.Time.make(0),
            offset = 3600000,
            at = new baja.AbsTime(date, time, offset);
        expect(at.getDate()).toBe(date);
        expect(at.getTime()).toBe(time);
        expect(at.getOffset()).toBe(offset);
      });
      
      it("is registered on baja:AbsTime Type", function isRegistered() {
        expect(baja.$('baja:AbsTime')).toBe(baja.AbsTime.DEFAULT);
      });
    });
    
    describe("make() (static)", function AbsTime_make() {
      it("accepts date, time, and offset properties", function acceptsDateTimeOffset() {
        var date = baja.Date.today(),
            time = baja.Time.make(999),
            offset = 3600000,
            at = baja.AbsTime.make({ date: date, time: time, offset: offset });
        
        verifyAbsTime(at, date, time, offset);
      });
      
      it("defaults date to DEFAULT", function defaultsDate() {
        var time = baja.Time.make(999),
            offset = 3600000,
            at = baja.AbsTime.make({ time: time, offset: offset });
        verifyAbsTime(at, baja.Date.DEFAULT, time, offset);
      });
      
      it("defaults time to DEFAULT", function defaultsTime() {
        var date = baja.Date.today(),
            offset = 3600000,
            at = baja.AbsTime.make({ date: date, offset: offset });
        verifyAbsTime(at, date, baja.Time.DEFAULT, offset);
      });
      
      it("defaults offset to 0", function defaultsOffset() {
        var date = baja.Date.today(),
            time = baja.Time.make(999),
            at = baja.AbsTime.make({ date: date, time: time });
        verifyAbsTime(at, date, time, 0);
      });
      
      it("accepts jsDate property", function acceptsJsDate() {
        var year = 2000,
            month = 4,
            day = 25,
            hour = 17,
            minute = 29,
            second = 56,
            ms = 765,
            jsDate = new Date(year, month, day, hour, minute, second, ms),
            date = baja.Date.make({ year: year, month: month, day: day }),
            time = baja.Time.make({ hour: hour, min: minute, sec: second, ms: ms }),
            offset = 7200000,
            at = baja.AbsTime.make({ jsDate: jsDate, offset: offset });
        verifyAbsTime(at, date, time, offset);
      });
      
      it("uses time zone offset from jsDate if no offset specified", function usesJsDateOffset() {
        var year = 2000,
            month = 4,
            day = 25,
            hour = 17,
            minute = 29,
            second = 56,
            ms = 765,
            jsDate = new Date(year, month, day, hour, minute, second, ms),
            date = baja.Date.make({ year: year, month: month, day: day }),
            time = baja.Time.make({ hour: hour, min: minute, sec: second, ms: ms }),
            at = baja.AbsTime.make({ jsDate: jsDate });
        verifyAbsTime(at, date, time, jsDate.getTimezoneOffset() * -60000);
      });
      
      it("returns DEFAULT when no/empty argument", function returnsDefault() {
        expect(baja.AbsTime.make()).toBe(baja.AbsTime.DEFAULT);
        expect(baja.AbsTime.make({})).toBe(baja.AbsTime.DEFAULT);
      });
      
      it("returns DEFAULT if date/time are DEFAULT and offset is 0", function returnsDefaultWhenDefault() {
        var at = baja.AbsTime.make({
              date: baja.Date.DEFAULT,
              time: baja.Time.DEFAULT,
              offset: 0
            }),
            at2 = baja.AbsTime.make({
              date: baja.Date.make({ year: 1970, month: 0, day: 1 }),
              time: baja.Time.make({ hour: 0, min: 0, sec: 0, ms: 0 }),
              offset: 0
            });
        expect(at).toBe(baja.AbsTime.DEFAULT);
        expect(at2).toBe(baja.AbsTime.DEFAULT);
      });
      
      it("dies on non-Date jsDate", function diesOnNonDateJsDate() {
        errTest(function () {
          baja.AbsTime.make({ jsDate: 'hello' });
        }, 'jsDate must be a JavaScript Date');
      });
      
      it("dies on non-baja.Date date", function diesOnNonDateDate() {
        errTest(function () {
          baja.AbsTime.make({ date: 'hello' });
        }, typeMismatchError(String, baja.Date));
      });
      
      it("dies on non-baja.Time time", function diesOnNonTimeTime() {
        errTest(function () {
          baja.AbsTime.make({ time: 'hello' });
        }, typeMismatchError(String, baja.Time));
      });
      
      it("dies on non-numeric offset", function diesOnNonNumericOffset() {
        errTest(function () {
          baja.AbsTime.make({ offset: 'hello' });
        }, typeMismatchError(String, Number));
      });
    });

    describe("now() (static)", function AbsTime_now() {
      it("returns an AbsTime with the current system time", function returnsCurrent() {
        var currentDate = new Date(),
            now = baja.AbsTime.now(),
            nowJsDate = now.getJsDate();
        verify(Math.abs(currentDate.getTime() - nowJsDate.getTime()) < 10);
      });
    });

    describe("decodeFromString()", function AbsTime_decodeFromString() {
      function decode(str) {
        return baja.AbsTime.DEFAULT.decodeFromString(str);
      }
      
      it("decodes from YYYY-MM-DDTHH:mm:ss.sssZ format", function decodesZulu() {
        var at = decode("1234-05-06T12:34:56.789Z"),
            date = baja.Date.make({ year: 1234, month: 4, day: 6 }),
            time = baja.Time.make({ hour: 12, min: 34, sec: 56, ms: 789 }),
            offset = 0;
        verifyAbsTime(at, date, time, offset);
      });
      
      it("decodes from YYYY-MM-DDTHH:mm:ss.ssZ format", function decodesZulu() {
        var at = decode("1234-05-06T12:34:56.79Z"),
            date = baja.Date.make({ year: 1234, month: 4, day: 6 }),
            time = baja.Time.make({ hour: 12, min: 34, sec: 56, ms: 79 }),
            offset = 0;
        verifyAbsTime(at, date, time, offset);
      });
      
      it("decodes from YYYY-MM-DDTHH:mm:ss.sZ format", function decodesZulu() {
        var at = decode("1234-05-06T12:34:56.7Z"),
            date = baja.Date.make({ year: 1234, month: 4, day: 6 }),
            time = baja.Time.make({ hour: 12, min: 34, sec: 56, ms: 7 }),
            offset = 0;
        verifyAbsTime(at, date, time, offset);
      });
      
      it("decodes from YYYY-MM-DDTHH:mm:ss.sss+HH:mm format", function decodesPosOffset() {
        var at = decode("9999-12-31T23:59:59.999+23:59"),
            date = baja.Date.make({ year: 9999, month: 11, day: 31 }),
            time = baja.Time.make({ hour: 23, min: 59, sec: 59, ms: 999 }),
            offset = baja.RelTime.MILLIS_IN_HOUR * 23 + baja.RelTime.MILLIS_IN_MINUTE * 59;
        verifyAbsTime(at, date, time, offset);
      });
      
      it("decodes from YYYY-MM-DDTHH:mm:ss.sss-HH:mm format", function decodesNegOffset() {
        var at = decode("0001-02-03T04:05:06.007-12:34"),
            date = baja.Date.make({ year: 1, month: 1, day: 3 }),
            time = baja.Time.make({ hour: 4, min: 5, sec: 6, ms: 7 }),
            offset = baja.RelTime.MILLIS_IN_HOUR * -12 + baja.RelTime.MILLIS_IN_MINUTE * -34;
        verifyAbsTime(at, date, time, offset);
      });
      
      it("dies on bad date format", function diesOnBadDateFormat() {
        errTest(function () {
          decode("0001-02:03T04:05:06.007+05:00");
        }, 'Could not decode AbsTime: 0001-02:03T04:05:06.007+05:00');
      });
      
      it("dies on invalid date", function diesOnInvalidDate() {
        errTest(function () {
          decode("0001-02-32T04:05:06.007+05:00");
        }, 'Invalid date range');
      });
      
      it("dies on bad time separator", function diesOnBadSeparator() {
        errTest(function () {
          decode("0001-02-03U04:05:06.007+05:00");
        }, 'Could not decode AbsTime: 0001-02-03U04:05:06.007+05:00');
      });
      
      it("dies on invalid time", function diesOnInvalidTime() {
        errTest(function () {
          decode("0001-02-03T24:05:06.007+05:00");
        }, 'Invalid time: 24:5:6.7');
      });
      
      it("dies on bad time format", function diesOnBadTimeFormat() {
        errTest(function () {
          decode("0001-02-03T04-05:06.007+05:00");
        }, 'Could not decode AbsTime: 0001-02-03T04-05:06.007+05:00');
      });
      
      it("dies on bad zulu indicator", function diesOnBadZulu() {
        errTest(function () {
          decode("0001-02-03T04:05:06.007A");
        }, 'Could not decode AbsTime: 0001-02-03T04:05:06.007A');
      });
      
      it("dies on bad offset format", function diesOnBadOffsetFormat() {
        errTest(function () {
          decode("0001-02-03T04:05:06.007+05-00");
        }, 'Could not decode AbsTime: 0001-02-03T04:05:06.007+05-00');
      });
    });
    
    describe("encodeToString()", function AbsTime_encodeToString() {
      function verifyEncode(absTime, str) {
        verifyEq(absTime.encodeToString(), str);
      }
      
      it("encodes to YYYY-MM-DDTHH:mm:ss.sssZ format if no offset", function encodesZulu() {
        var absTime = baja.AbsTime.make({
              date: baja.Date.make({ year: 2222, month: 2, day: 4 }),
              time: baja.Time.make({ hour: 5, min: 6, sec: 7, ms: 8 }),
              offset: 0
            });
        verifyEncode(absTime, "2222-03-04T05:06:07.008Z");
      });
      
      it("encodes to YYYY-MM-DDTHH:mm:ss.sss+HH:mm format if offset is positive", function encodesPosOffset() {
        var absTime = baja.AbsTime.make({
              date: baja.Date.make({ year: 3333, month: 3, day: 5 }),
              time: baja.Time.make({ hour: 6, min: 7, sec: 8, ms: 9 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * 13 + baja.RelTime.MILLIS_IN_MINUTE * 46
            });
        verifyEncode(absTime, "3333-04-05T06:07:08.009+13:46");
      });

      it("encodes to YYYY-MM-DDTHH:mm:ss.sss-HH:mm format if offset is negative", function encodesNegOffset() {
        var absTime = baja.AbsTime.make({
              date: baja.Date.make({ year: 3333, month: 3, day: 5 }),
              time: baja.Time.make({ hour: 6, min: 7, sec: 8, ms: 9 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -13 + baja.RelTime.MILLIS_IN_MINUTE * -46
            });
        verifyEncode(absTime, "3333-04-05T06:07:08.009-13:46");
      });
    });
    
    describe("equals()", function AbsTime_equals() {
      it("returns true if date, time, and offset are the same", function trueIfSame() {
        var absTime1 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            }),
            absTime2 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            });
        verify(absTime1.equals(absTime2));
      });
      
      it("returns false if date is different", function falseIfDifferentDate() {
        var absTime1 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            }),
            absTime2 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 4 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            });
        verify(!absTime1.equals(absTime2));
      });
      
      it("returns false if time is different", function falseIfDifferentTime() {
        var absTime1 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            }),
            absTime2 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 889 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            });
        verify(!absTime1.equals(absTime2));
      });
      
      it("returns false if offset is different", function falseIfDifferentOffset() {
        var absTime1 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -30
            }),
            absTime2 = baja.AbsTime.make({
              date: baja.Date.make({ year: 7777, month: 6, day: 5 }),
              time: baja.Time.make({ hour: 23, min: 10, sec: 9, ms: 888 }),
              offset: baja.RelTime.MILLIS_IN_HOUR * -4 + baja.RelTime.MILLIS_IN_MINUTE * -31
            });
        verify(!absTime1.equals(absTime2));
      });
      
      it("returns false for a non-AbsTime", function falseIfNotAbsTime() {
        var absTime1 = baja.AbsTime.now();
        verify(!absTime1.equals(baja.Time.make(0)));
        verify(!absTime1.equals("hello"));
      });
      
      it("returns false for null/undefined", function falseIfNullUndefined() {
        var absTime1 = baja.AbsTime.now();
        verify(!absTime1.equals(null));
        verify(!absTime1.equals(undefined));
      });
    });
    
    describe("getDataTypeSymbol()", function AbsTime_getDataTypeSymbol() {
      it("returns 'a'", function returnsA() {
        verifyEq(baja.AbsTime.now().getDataTypeSymbol(), 'a');
      });
    });
    
    describe("getTime()", function AbsTime_getTime() {
      it("returns the time portion", function returnsTime() {
        var time = baja.Time.make(223344),
            absTime = baja.AbsTime.make({ time: time });
        expect(absTime.getTime()).toBe(time);
      });
    });
    
    describe("getDate()", function AbsTime_getDate() {
      it("returns the date portion", function returnsDate() {
        var date = baja.Date.today(),
            absTime = baja.AbsTime.make({ date: date });
        expect(absTime.getDate()).toBe(date);
      });
    });
    
    describe("getOffset()", function AbsTime_getOffset() {
      it("returns the offset", function returnsOffset() {
        var offset = 8675309,
            absTime = baja.AbsTime.make({ offset: offset });
        expect(absTime.getOffset()).toBe(offset);
      });
    });

    describe("getJsDate()", function AbsTime_getJsDate() {
      it("returns a Javascript date matching this AbsTime", function returnsJsDate() {
        var absTime = baja.AbsTime.make({
              date: baja.Date.make({ year: 1066, month: 9, day: 14 }),
              time: baja.Time.make({ hour: 9, min: 5, sec: 18, ms: 888 })
            }),
            jsDate = absTime.getJsDate();
        verifyEq(jsDate.getFullYear(), 1066);
        verifyEq(jsDate.getMonth(), 9);
        verifyEq(jsDate.getDate(), 14);
        verifyEq(jsDate.getHours(), 9);
        verifyEq(jsDate.getMinutes(), 5);
        verifyEq(jsDate.getSeconds(), 18);
        verifyEq(jsDate.getMilliseconds(), 888);
      });
    });
    
    describe("toString()", function AbsTime_toString() {
      function toAbs(year, month, day, hour, min, sec, ms) {
        return baja.AbsTime.make({
          date: baja.Date.make({ year: year, month: month, day: day }),
          time: baja.Time.make({ hour: hour, min: min, sec: sec, ms: ms })
        });
      }
      
      function verifyStr(year, month, day, hour, min, sec, ms, obj, str) {
        verifyEq(toAbs(year, month, day, hour, min, sec, ms).toString(obj), str);
      }

      it("formats to DD-MMM-YY h:mm:ss.sss a by default", function defaultFormat() {
        var obj;
        verifyStr(1066, 9, 14, 9, 5, 18, 888, obj, "14-Oct-66 9:05:18.888 AM");
      });
      
      it("formats to YYYY-MM-DD", function formatsYYYYMMDD() {
        var obj = { textPattern: "YYYY-MM-DD" };
        verifyStr(1066, 9, 14, 9, 5, 18, 888, obj, "1066-10-14");
      });

      it("formats to hh:mm:ss a", function formatshhmmsssss() {
        var obj = { textPattern: "hh:mm:ss a" };
        verifyStr(1066, 9, 14, 9, 5, 18, 888, obj, "09:05:18.888 AM");
        verifyStr(1066, 9, 14, 21, 5, 18, 888, obj, "09:05:18.888 PM");
      });

      it("formats to HH:mm:ss", function formatsHHmmsssss() {
        var obj = { textPattern: "HH:mm:ss" };
        verifyStr(1066, 9, 14, 9, 5, 18, 888, obj, "09:05:18.888");
        verifyStr(1066, 9, 14, 21, 5, 18, 888, obj, "21:05:18.888");
      });
      
      it("hides millis", function hidesMillis() {
        var obj = { 
              textPattern: "HH:mm:ss", 
              show: baja.TimeFormat.SHOW_TIME | baja.TimeFormat.SHOW_SECONDS  
            };
        verifyStr(1066, 9, 14, 9, 5, 18, 888, obj, "09:05:18");
      });
    });
    
    describe("valueOf()", function AbsTime_valueOf() {
      it("returns this", function returnsThis() {
        var absTime = baja.AbsTime.now();
        expect(absTime.valueOf()).toBe(absTime);
      });
    });
    
    describe("newCopy()", function AbsTime_newCopy() {
      it("returns this", function returnsThis() {
        var absTime = baja.AbsTime.now();
        expect(absTime.newCopy()).toBe(absTime);
      });
    });
  });

  describe("baja.Icon", function Icon() {
    describe("constructor", function Icon_constructor() {
      it("accepts array of ords", function acceptsOrds() {
        var ords = [ 'ord1', 'ord2' ],
            icon = new baja.Icon(ords);
        verifyEq(icon.getImageOrds(), ords);
      });
      
      it("is registered on baja.Icon Type", function isRegistered() {
        expect(baja.$('baja:Icon')).toBe(baja.Icon.DEFAULT);
      });
    });
    
    describe("make() (static)", function Icon_make() {
      it("accepts a single ord string", function acceptsString() {
        var ord = 'module://icons/x16/home.png',
            icon = baja.Icon.make(ord);
        verifyEq(icon.getImageOrds().join(), ord);
      });
      
      it("accepts an array of ords", function acceptsOrdArray() {
        var ords = [ 'module://icons/x16/home.png', 'module://icons/x16/cloud.png' ],
            icon = baja.Icon.make(ords);
        verifyEq(icon.getImageOrds().join(), ords.join());
      });
      
      it("converts string ords into baja.Ords", function convertsOrds() {
        var ords = [ 'module://icons/x16/convertme1.png', 'module://icons/x16/convertme2.png' ],
            icon = baja.Icon.make(ords),
            imageOrds = icon.getImageOrds();
        
        verifyEq(imageOrds[0].getType().toString(), 'baja:Ord');
        verifyEq(imageOrds[0].toString(), ords[0]);
        verifyEq(imageOrds[1].getType().toString(), 'baja:Ord');
        verifyEq(imageOrds[1].toString(), ords[1]);
      });
      
      it("interns for the same set of ords", function interns() {
        var ords1 = [ 'module://icons/x16/intern1.png', 'module://icons/x16/intern2.png' ],
            ords2 = [ 'module://icons/x16/intern3.png', 'module://icons/x16/intern4.png' ];
        expect(baja.Icon.make(ords1)).toBe(baja.Icon.make(ords1));
        expect(baja.Icon.make(ords2)).not.toBe(baja.Icon.make(ords1));
      });
      
      it("returns DEFAULT for no arguments", function returnsDefaultNoArgs() {
        expect(baja.Icon.make()).toBe(baja.Icon.DEFAULT);
      });
      
      it("returns DEFAULT for empty string", function returnsDefaultEmptyString() {
        expect(baja.Icon.make("")).toBe(baja.Icon.DEFAULT);
      });
      
      it("returns DEFAULT for empty array", function returnsDefaultEmptyArray() {
        expect(baja.Icon.make([])).toBe(baja.Icon.DEFAULT);
      });
      
      it("dies on a non-String/non-Array", function diesOnNonStringNonArray() {
        errTest(function () {
          baja.Icon.make(true);
        }, "Invalid argument for baja.Icon.make");
      });
    });
    
    describe("getStdObjectIcon() (static)", function Icon_getStdObjectIcon() {
      it("returns a standard object icon", function returnsStandardIcon() {
        var icon = baja.Icon.getStdObjectIcon();
        verifyEq(icon.getImageOrds().join(), 'module://icons/x16/object.png');
      });
    });
    
    describe("encodeToString()", function Icon_encodeToString() {
      it("encodes ord list separated by newlines", function encodesOrds() {
        var ords = [ 'ord1', 'ord2', 'ord3' ],
            icon = baja.Icon.make(ords);
        verifyEq(icon.encodeToString(), ords.join('\n'));
      });
    });
    
    describe("decodeFromString", function Icon_decodeFromString() {
      it("decodes ord list separated by newlines", function decodesOrds() {
        var ords = [ 'ord4', 'ord5', 'ord6' ],
            str = ords.join('\n'),
            icon = baja.Icon.DEFAULT.decodeFromString(str);
        verifyEq(icon.getImageOrds().join(), ords.join());
      });
    });
    
    describe("toString()", function Icon_toString() {
      it("returns ord list separated by newlines", function encodesOrds() {
        var ords = [ 'ord1', 'ord2', 'ord3' ],
            icon = baja.Icon.make(ords);
        verifyEq(icon.toString(), ords.join('\n'));
      });
    });
    
    describe("valueOf()", function Icon_valueOf() {
      it("returns ord list separated by newlines", function encodesOrds() {
        var ords = [ 'ord1', 'ord2', 'ord3' ],
            icon = baja.Icon.make(ords);
        verifyEq(icon.valueOf(), ords.join('\n'));
      });
    });
    
    describe("getImageOrds()", function Icon_getImageOrds() {
      it("returns array of baja.Ords", function returnsOrdArray() {
        var ords = [ 'ordMoe', 'ordLarry', 'ordCurly' ],
            icon = baja.Icon.make(ords),
            imageOrds = icon.getImageOrds(),
            i;
        
        verifyEq(imageOrds.length, 3);
        for (i = 0; i < imageOrds.length; i++) {
          verifyEq(imageOrds[i].getType().toString(), 'baja:Ord');
          verifyEq(imageOrds[i].toString(), ords[i]);
        }
      });
      
      it("maintains immutability", function maintainsImmutability() {
        var ords = [ 'ordShemp' ],
            icon = baja.Icon.make(ords);
        
        verifyEq(icon.getImageOrds().join(), ords.join());
        icon.getImageOrds().push(baja.Ord.make('iAmANewOrd'));
        verifyEq(icon.getImageOrds().join(), ords.join());
      });
    });
    
    describe("getImageUris()", function Icon_getImageUris() {
      it("changes starting 'module://' to '/module/'", function changesModule() {
        var ords = [ 'module://icons/icon1',
                     'module://alarm/icon2',
                     '/module/box/icon3',
                     'hello/module://' ],
            icon = baja.Icon.make(ords),
            imageUris = icon.getImageUris();
        verifyEq(imageUris[0], '/module/icons/icon1');
        verifyEq(imageUris[1], '/module/alarm/icon2');
        verifyEq(imageUris[2], '/module/box/icon3');
        verifyEq(imageUris[3], 'hello/module://');
      });
    });
  });
  
  describe("baja.Format", function Format() {
    describe("constructor", function Format_constructor() {
      it("accepts a string format", function acceptsString() {
        var fmt = new baja.Format("i'm a format");
        verifyEq(fmt.toString(), "i'm a format");
      });
      
      it("is registered on baja:Format Type", function isRegistered() {
        expect(baja.$('baja:Format')).toBe(baja.Format.DEFAULT);
      });
    });
    
    describe("make() (static)", function Format_make() {
      it("accepts a string format", function acceptsString() {
        var fmt = baja.Format.make("i'm a format too");
        verifyEq(fmt.toString(), "i'm a format too");
      });
      
      it("returns DEFAULT for empty string", function returnsDefaultForEmpty() {
        expect(baja.Format.make("")).toBe(baja.Format.DEFAULT);
      });
      
      it("returns DEFAULT for null/undefined", function returnsDefaultForNull() {
        expect(baja.Format.make(null)).toBe(baja.Format.DEFAULT);
        expect(baja.Format.make(undefined)).toBe(baja.Format.DEFAULT);
      });
      
      
      it("dies for non-String", function diesForNonString() {
        errTest(function () {
          baja.Format.make(45);
        }, typeMismatchError(Number, String));
      });
    });
    
    describe("decodeFromString", function Format_decodeFromString() {
      it("just passes the string to make()", function passesToMake() {
        var fmt = baja.Format.DEFAULT.decodeFromString("decode me");
        verifyEq(fmt.toString(), "decode me");
      });
    });
    
    describe("encodeToString()", function Format_encodeToString() {
      it("just returns the pattern", function returnsPattern() {
        var fmt = baja.Format.make("encode me");
        verifyEq(fmt.encodeToString(), "encode me");
      });
    });
    
    describe("toString()", function Format_toString() {
      it("just returns the pattern", function returnsPattern() {
        var fmt = baja.Format.make("toString me");
        verifyEq(fmt.toString(), "toString me");
      });
    });
    
    describe("valueOf()", function Format_valueOf() {
      it("just returns the pattern", function returnsPattern() {
        var fmt = baja.Format.make("valueOf me");
        verifyEq(fmt.valueOf(), "valueOf me");
      });
    });
        
    describe("format()", function Format_format() {
      describe("when formatting unmounted objects", function unmounted() {
        it("formats '%%' to '%'", function formatsDoublePercent() {
          verifyEq(baja.Format.make("%%").format(), "%");
          verifyEq(baja.Format.make("%%%%%%").format(), "%%%");
        });
        
        it("formats identity (%.%) on a simple object using toString", function formatsIdentityToString() {
          var object = {
                toString: function () {
                  return "valid";
                }
              },
              fmt = baja.Format.make("%.%");
          verifyEq(fmt.format({ object: object }), 'valid');
        });
        
        it("formats a value from a component using a slot name", function formatsComponentUsingSlotName() {
          var comp = baja.$('baja:Component', {
                test: 'testSlotValue'
              }),
              fmt = baja.Format.make("%test%");
          verifyEq(fmt.format({ object: comp, display: false }), 'testSlotValue');
        });
        
        it("formats a value from a component using dot-separated slot names", function formatsComponentUsingDotSeparatedSlots() {
          var comp = baja.$('baja:Component', {
                subComp: baja.$('baja:Component', {
                  sub2: baja.$('baja:Component', {
                    theValue: 'got the value'
                  })
                })
              }),
              fmt = baja.Format.make("%subComp.sub2.theValue%");
          verifyEq(fmt.format({ object: comp, display: false }), 'got the value');
        });
        
        it("formats a value from an object using getter functions", function formatsObjectUsingGetters() {
          var obj = {
                getFoo: function () {
                  return {
                    getBar: function () {
                      return {
                        getValue: function () {
                          return 88.75;
                        }
                      };
                    }
                  };
                }
              },
              fmt = baja.Format.make("%foo.bar.value%");
          verifyEq(fmt.format({ object: obj, display: false }), '88.75');
        });
        
        it("formats a value from an object using straight up function names", function formatsObjectUsingFunctionNames() {
          var obj = {
                gimme: function () {
                  return {
                    mah: function () {
                      return {
                        steel: function () {
                          return "reserve";
                        }
                      };
                    }
                  };
                }
              },
              fmt = baja.Format.make("%gimme.mah.steel%");
          verifyEq(fmt.format({ object: obj, display: false }), 'reserve');
        });
        
        it("formats using a combination of function names, getters and slots", function formatsUsingEverything() {
          var comp = baja.$('baja:Component', {
                compValue: 'i am a component'
              }),
              obj = {
                namedFunction: function () {
                  return {
                    getComp: function () {
                      return comp;
                    }
                  };
                }
              },
              fmt = baja.Format.make("%namedFunction.comp.compValue%");
          verifyEq(fmt.format({ object: obj, display: false }), 'i am a component');
        });
        
        it("formats identity (%.%) on an OrdTarget to a struct", function formatsIdentityOrdTargetStruct() {
          var comp = baja.$('baja:Component', {
                statusNum: baja.$('baja:StatusNumeric', {
                  value: 45
                })
              }),
              ord = baja.Ord.make('slot:statusNum/value'),
              resolveCallbacks = callbackify({
                base: comp,
                ok: function (target) {
                  var fmt = baja.Format.make('%.%');
                  verifyEq(fmt.format({ target: target, display: false }), '45');
                },
                fail: function (err) {
                  console.log('fail: ' + err);
                }
              });
          
          runAndWait(function () {
            ord.resolve(resolveCallbacks);
          }, resolveCallbacks);
        });
        
        it("formats a lexicon message", function formatsLexicon() {
          verifyEq(baja.Format.make("%lexicon(baja:nav.sysHome.description)%").format(),
                  baja.lex('baja').get('nav.sysHome.description'));
        });
        
//        it("formats a Component with a Facets Property", function formatsCompWithFacets() {
//          var c = baja.$("baja:Component"),
//              msgText = "This is some text",
//              cb;
//
//          c.add({
//            slot: "facets",
//            value: baja.Facets.make(["msgText"], [msgText])
//          });
//
//          cb = callbackify({
//            ok: createEqualOk(msgText),
//            pattern: "%facets.msgText%",
//            object: c
//          });
//
//          runAndWait(function () {
//            baja.lex(cb);
//          }, cb);
//        });
      });
      
      describe("when formatting mounted objects", function mounted() {
        var numWritableValue = 44632.34,
            statusNumValue = 77436.34;
        
        (function () {
          var addCallbacks = callbackify({
            slot: 'formatTestComponent',
            value: baja.$('baja:Component', {
              subComponent: baja.$('baja:Component', {
                numWritable: baja.$('control:NumericWritable', {
                  facets: baja.Facets.make({ precision: 2 }),
                  fallback: baja.$('baja:StatusNumeric', {
                    value: numWritableValue
                  })
                }),
                statusNum: baja.$('baja:StatusNumeric', {
                  value: statusNumValue
                })
              })
            })
          });
          
          runAndWait(function () {
            testFolder.add(addCallbacks);
          }, addCallbacks);
        }());
        
        it("formats identity (%.%) on an OrdTarget to a Simple", function formatsIdentityOnOrdTargetSimple() {
          var fmt = baja.Format.make("%.%"),
              resolveCallbacks = callbackify({
                base: testFolder,
                ok: function (target) {
                  verifyEq(fmt.format({ target: target }), String(statusNumValue));
                }
              }), 
              ord = baja.Ord.make("slot:formatTestComponent/subComponent/statusNum/value");
          
          runAndWait(function () {
            ord.resolve(resolveCallbacks);
          }, resolveCallbacks);
        });
        
        //TODO: doesn't work - wants either target && target.container, or object
        it("formats identity (%.%) on an OrdTarget to a component", function formatsIdentityOnOrdTargetComponent() {
          var fmt = baja.Format.make("%.%"),
              resolveCallbacks = callbackify({
                base: testFolder,
                ok: function (target) {
                  verifyEq(fmt.format({ target: target }), String(numWritableValue) + " {ok} @ def");
                }
              }), 
              ord = baja.Ord.make("slot:formatTestComponent/subComponent/numWritable");
          
          runAndWait(function () {
            ord.resolve(resolveCallbacks);
          }, resolveCallbacks);
        });
        
        it("formats identity (%.%) on a mounted component", function formatsIdentityOnMountedComponent() {
          var fmt = baja.Format.make("%.%"),
              numWritable = testFolder.get('formatTestComponent')
                                      .get('subComponent')
                                      .get('numWritable');
          verifyEq(fmt.format({ object: numWritable }), numWritable.getDisplay('out'));
        });
        
        it("formats a value from a component using dot-separated slot names", function formatsComponentDotSeparatedSlots() {
          var fmt = baja.Format.make("%formatTestComponent.subComponent.numWritable.out.value%");
          verifyEq(fmt.format({ object: testFolder }), String(numWritableValue));
        });
        
        it("formats a value from a component using getter functions", function formatsComponentWithGetters() {
          var fmt = baja.Format.make("%formatTestComponent.subComponent.numWritable.type%");
          verifyEq(fmt.format({ object: testFolder }), "control:NumericWritable");
        });
      });
    });
  });

  describe("baja.Permissions", function Permissions() {
  
    describe("make", function make() {
      it("make (static) with numeric mask", function makeStaticNum() {
        var perm = baja.Permissions.make(baja.Permissions.OPERATOR_READ);  
        expect(perm).toEquivalent(baja.Permissions.make("r"));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ);
        
        perm = baja.Permissions.make(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE);  
        expect(perm).toEquivalent(baja.Permissions.make("rI"));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE);
      });
      
      it("make (static) with String encoding", function makeStaticStr() {
        var perm = baja.Permissions.make("r");  
        expect(perm).toEquivalent(baja.Permissions.make(baja.Permissions.OPERATOR_READ));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ);
        
        perm = baja.Permissions.make("rI");  
        expect(perm).toEquivalent(baja.Permissions.make(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE);
      });
      
      it("make with numeric mask", function makeNum() {
        var perm = baja.Permissions.DEFAULT.make(baja.Permissions.OPERATOR_READ);  
        expect(perm).toEquivalent(baja.Permissions.DEFAULT.make("r"));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ);
        
        perm = baja.Permissions.DEFAULT.make(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE);  
        expect(perm).toEquivalent(baja.Permissions.DEFAULT.make("rI"));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE);
      });
      
      it("make with String encoding", function makeStr() {
        var perm = baja.Permissions.DEFAULT.make("r");  
        expect(perm).toEquivalent(baja.Permissions.DEFAULT.make(baja.Permissions.OPERATOR_READ));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ);
        
        perm = baja.Permissions.DEFAULT.make("rI");  
        expect(perm).toEquivalent(baja.Permissions.DEFAULT.make(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE));
        expect(perm.getMask()).toEqual(baja.Permissions.OPERATOR_READ | baja.Permissions.ADMIN_INVOKE);
      });
    });
    
    describe("constants", function constants() {
      it("all", function all() {
        expect(baja.Permissions.all.hasOperatorRead()).toBeTruthy();
        expect(baja.Permissions.all.hasOperatorWrite()).toBeTruthy();
        expect(baja.Permissions.all.hasOperatorInvoke()).toBeTruthy();
        expect(baja.Permissions.all.hasAdminRead()).toBeTruthy();
        expect(baja.Permissions.all.hasAdminWrite()).toBeTruthy();
        expect(baja.Permissions.all.hasAdminInvoke()).toBeTruthy();
        
        expect(baja.Permissions.make(baja.Permissions.OPERATOR_READ | baja.Permissions.OPERATOR_WRITE | baja.Permissions.OPERATOR_INVOKE |
                                     baja.Permissions.ADMIN_READ | baja.Permissions.ADMIN_WRITE | baja.Permissions.ADMIN_INVOKE)).toEqual(baja.Permissions.all);                                     
      });
      
      it("none", function none() {
        expect(baja.Permissions.none.hasOperatorRead()).not.toBeTruthy();
        expect(baja.Permissions.none.hasOperatorWrite()).not.toBeTruthy();
        expect(baja.Permissions.none.hasOperatorInvoke()).not.toBeTruthy();
        expect(baja.Permissions.none.hasAdminRead()).not.toBeTruthy();
        expect(baja.Permissions.none.hasAdminWrite()).not.toBeTruthy();
        expect(baja.Permissions.none.hasAdminInvoke()).not.toBeTruthy();
        
        expect(baja.Permissions.make(0)).toEqual(baja.Permissions.none);
      });
    });
    
    describe("has", function has() {
      it("operatorWrite and adminRead", function operatorWriteAdminRead() {
        var perms = baja.Permissions.make(baja.Permissions.OPERATOR_WRITE | baja.Permissions.ADMIN_READ);
        expect(perms.hasOperatorRead()).not.toBeTruthy();
        expect(perms.hasOperatorWrite()).toBeTruthy();
        expect(perms.hasOperatorInvoke()).not.toBeTruthy();
        expect(perms.hasAdminRead()).toBeTruthy();
        expect(perms.hasAdminWrite()).not.toBeTruthy();
        expect(perms.hasAdminInvoke()).not.toBeTruthy();  
        
        expect(perms.has(baja.Permissions.OPERATOR_WRITE)).toBeTruthy();       
        expect(perms.has(baja.Permissions.OPERATOR_READ)).not.toBeTruthy();  
        expect(perms.has(baja.Permissions.OPERATOR_WRITE | baja.Permissions.ADMIN_READ)).toBeTruthy();  
        expect(perms.has(baja.Permissions.OPERATOR_WRITE | baja.Permissions.ADMIN_READ | baja.Permissions.ADMIN_INVOKE)).not.toBeTruthy();  
        
        expect(perms.getMask()).toEqual(baja.Permissions.OPERATOR_WRITE | baja.Permissions.ADMIN_READ);
      });
      
      it("operatorInvoke and adminWrite", function operatorInvokeAdminWrite() {
        var perms = baja.Permissions.make(baja.Permissions.OPERATOR_INVOKE | baja.Permissions.ADMIN_WRITE);
        expect(perms.hasOperatorRead()).not.toBeTruthy();
        expect(perms.hasOperatorWrite()).not.toBeTruthy();
        expect(perms.hasOperatorInvoke()).toBeTruthy();
        expect(perms.hasAdminRead()).not.toBeTruthy();
        expect(perms.hasAdminWrite()).toBeTruthy();
        expect(perms.hasAdminInvoke()).not.toBeTruthy();  
        
        expect(perms.has(baja.Permissions.OPERATOR_INVOKE)).toBeTruthy();       
        expect(perms.has(baja.Permissions.OPERATOR_READ)).not.toBeTruthy();  
        expect(perms.has(baja.Permissions.OPERATOR_INVOKE | baja.Permissions.ADMIN_WRITE)).toBeTruthy();  
        expect(perms.has(baja.Permissions.OPERATOR_INVOKE | baja.Permissions.ADMIN_WRITE | baja.Permissions.ADMIN_INVOKE)).not.toBeTruthy();  
        
        expect(perms.getMask()).toEqual(baja.Permissions.OPERATOR_INVOKE | baja.Permissions.ADMIN_WRITE);
      });
    });
    
    describe("encodeToString and decodeFromString", function encodeDecode() {
      it("all", function all() {
        expect(baja.Permissions.all.encodeToString()).toEqual("irwIRW");
        expect(baja.Permissions.DEFAULT.decodeFromString("irwIRW")).toEqual(baja.Permissions.all);
      });
      
      it("none", function none() {       
        expect(baja.Permissions.none.encodeToString()).toEqual("");
        expect(baja.Permissions.DEFAULT.decodeFromString("")).toEqual(baja.Permissions.none);
      });
      
      it("operatorInvoke and adminWrite", function operatorInvokeAdminWrite() {
        var perms = baja.Permissions.make(baja.Permissions.OPERATOR_INVOKE | baja.Permissions.ADMIN_WRITE);
        
        expect(perms.encodeToString()).toEqual("iW");
        expect(baja.Permissions.DEFAULT.decodeFromString("iW")).toEquivalent(perms);
      });
    });
    
  });  
  
  describe("NameMap", function NameMap() {
  
    function verifyNameMap(map) {
      var s = map.encodeToString(),
          x = baja.NameMap.DEFAULT.decodeFromString(s);
          
      expect(x.equals(map));
      expect(x.encodeToString()).toEqual(s);
    }
    
    it("DEFAULT instance", function testDefault() {
      // Test default
      expect(baja.NameMap.make({})).toEqual(baja.NameMap.DEFAULT);
      expect(baja.NameMap.DEFAULT.decodeFromString("{}")).toEqual(baja.NameMap.DEFAULT);
    });
  
    it("make (static)", function testStaticMake() {
      // Test basic map creation
      var o = baja.NameMap.make({"keyA": "value a"});
      expect(o.get("keyA").toString()).toEqual("value a");  
      verifyNameMap(o);

      o = baja.NameMap.make({"keyA": baja.Format.make("value a")});
      expect(o.get("keyA").toString()).toEqual("value a");     
      verifyNameMap(o);
    });

    it("make (non-static)", function testMake() {
      // Test basic map creation
      var o = baja.NameMap.DEFAULT.make({"keyA": "value a"});
      verifyNameMap(o);
      expect(o.get("keyA").toString()).toEqual("value a");     
      expect(o.get("keyA") instanceof baja.Format).toBeTruthy();  
    });

    it("make with special characters", function makeSpecial() {
      var obj = {},
          key = "key\\~! @#$%^&*()-{}=;+[]",
          val = "value\\~! @#$%^&*()-{}=;+[]",
          o;

      obj[key] = val;
          
      o = baja.NameMap.make(obj);
      verifyNameMap(o);
      
      expect(o.get(key)).not.toBeNull();
      expect(o.get(key).toString()).toEqual(val);
    });

    it("get", function testGet() {
      // Test name map get
      var o = baja.NameMap.make({"keyA": "value a", "keyB": "value b", "keyC": "value c"});
      verifyNameMap(o);
      expect(o.get("keyA").toString()).toEqual("value a");     
      expect(o.get("keyB").toString()).toEqual("value b");     
      expect(o.get("keyC").toString()).toEqual("value c");     
      expect(o.get("keyD")).toBeNull();  
    });

    it("list", function testList() {
      // Test name map list
      var o = baja.NameMap.make({"keyA": "value a", "keyB": "value b", "keyC": "value c"}),
         list = o.list(),
         i,
         s;
         
      verifyNameMap(o);   
         
      expect(list.length).toEqual(3);

      for (i = 0; i < list.length; ++i) {
       s = list[i];
       expect(s === "keyA" || s === "keyB" || s === "keyC").toBeTruthy();
      }
    });
    
    it("simple encodeToString", function simpleEncode() {
      var map = baja.NameMap.make({"keyA": "value a", "keyB": "value b", "keyC": "value c"});
      verifyNameMap(map);
      expect(map.encodeToString()).toEqual("{keyA=value a;keyB=value b;keyC=value c;}");
    });
    
    it("simple decodeFromString", function simpleDecode() {
      var map = baja.NameMap.DEFAULT.decodeFromString("{keyA=value a;keyB=value b;keyC=value c;}"),
          list = map.list(),
          i,
          s;
          
      verifyNameMap(map);   
         
      expect(list.length).toEqual(3);

      for (i = 0; i < list.length; ++i) {
       s = list[i];
       expect(s === "keyA" || s === "keyB" || s === "keyC").toBeTruthy();
      }      
    });
    
    it("escaped encodeToString", function escapedEncode() {
      expect(baja.NameMap.make({"key=A": "value=A"}).encodeToString()).toEqual("{key\\=A=value\\=A;}");
      expect(baja.NameMap.make({"key{A": "value{A"}).encodeToString()).toEqual("{key\\{A=value\\{A;}");
      expect(baja.NameMap.make({"key}A": "value}A"}).encodeToString()).toEqual("{key\\}A=value\\}A;}");
      expect(baja.NameMap.make({"key\\A": "value\\A"}).encodeToString()).toEqual("{key\\\\A=value\\\\A;}");
      expect(baja.NameMap.make({"key;A": "value;A"}).encodeToString()).toEqual("{key\\;A=value\\;A;}");
      expect(baja.NameMap.make({"key;={}\\A;={}\\": "value;={}\\A;={}\\"}).encodeToString()).toEqual("{key\\;\\=\\{\\}\\\\A\\;\\=\\{\\}\\\\=value\\;\\=\\{\\}\\\\A\\;\\=\\{\\}\\\\;}");
    });
    
    it("escaped decodeFromString", function escapedDecode() {
      var map = baja.NameMap.DEFAULT.decodeFromString("{key\\=A=value\\=A;}");
      verifyNameMap(map);
      expect(map.list().length).toEqual(1);
      expect(map.get("key=A").toString()).toEqual("value=A");
      
      map = baja.NameMap.DEFAULT.decodeFromString("{key\\{A=value\\{A;}");
      verifyNameMap(map);
      expect(map.list().length).toEqual(1);
      expect(map.get("key{A").toString()).toEqual("value{A");
      
      map = baja.NameMap.DEFAULT.decodeFromString("{key\\}A=value\\}A;}");
      verifyNameMap(map);
      expect(map.list().length).toEqual(1);
      expect(map.get("key}A").toString()).toEqual("value}A");
      
      map = baja.NameMap.DEFAULT.decodeFromString("{key\\\\A=value\\\\A;}");
      verifyNameMap(map);
      expect(map.list().length).toEqual(1);
      expect(map.get("key\\A").toString()).toEqual("value\\A");
      
      map = baja.NameMap.DEFAULT.decodeFromString("{key\\;A=value\\;A;}");
      verifyNameMap(map);
      expect(map.list().length).toEqual(1);
      expect(map.get("key;A").toString()).toEqual("value;A");
      
      map = baja.NameMap.DEFAULT.decodeFromString("{key\\;\\=\\{\\}\\\\A\\;\\=\\{\\}\\\\=value\\;\\=\\{\\}\\\\A\\;\\=\\{\\}\\\\;}");
      verifyNameMap(map);
      expect(map.list().length).toEqual(1);
      expect(map.get("key;={}\\A;={}\\").toString()).toEqual("value;={}\\A;={}\\");
    });
  });
});
