//
// Copyright 2012, Tridium, Inc. All Rights Reserved.
//

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: false, eqeqeq: true,
bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

// Globals for JsLint to ignore 
/*global baja, XMLHttpRequest, errTest, verify, verifyEq, describe, it, xit, testFolder, callbackify, expect,
runAndWait, serverCall, verifyServerCall, lease, runs, waitsFor, bajaJsPrint, add, errCallbackify, console,
afterEach*/

var compTest = { version: 1.0 };

(function testComp() { 

  "use strict";
  
  describe("component", function testComponents() {
          
    describe("add", function testAdd() {
      
      describe("unmounted", function unmounted() {
        function fail(err) {
          throw new Error(err);
        }
        
        // Test Component
        it("add", function add() {
          var c = baja.$("baja:Component");
          
          // Add a Property and try out getter and setter
          c.add({
            slot: "foo",
            value: "the Property value!"
          });
          
          verifyEq(c.get("foo"), "the Property value!");
          c.set({
            slot: "foo",
            value: "Something else"
          });
          verifyEq(c.get("foo"), "Something else");
          
          // Add a Component to a Component
          var c2 = baja.$("baja:Component");
          c.add({
            slot: "comp", 
            value: c2
          });
          
          verify(c.get("comp") === c2);
          verify(c2.$parent === c);
          verify(c2.getParent() === c);
          verify(c2.$propInParent === c.getSlot("comp"));
        });
        
        it("add auto", function autoAdd() {
          // Test auto gen name
          var c = baja.$("baja:Component");
          
          verifyEq(c.add({
            value: "testMe"
          }).getName(), "String");
          
          verifyEq(c.add({
            slot: null,
            value: "testMe"
          }).getName(), "String1");
          
          verifyEq(c.add({
            value: "testMe"
          }).getName(), "String2");
          
          verifyEq(c.add({
            slot: "test?", 
            value: "testMe"
          }).getName(), "test");
          
          verifyEq(c.add({
            slot: "test?",
            value: "testMe"
          }).getName(), "test1");
          
          verifyEq(c.add({
            slot: "test?",
            value: "testMe"
          }).getName(), "test2");
                
          errTest(function () {
            verifyEq(c.add({
              slot: "test ",
              value: "testMe",
              fail: fail
            }));
          });
          
          errTest(function () {
            verifyEq(c.add({
              slot: "test @",
              value: "testMe",
              fail: fail
            }));
          });
          
          errTest(function () {
            verifyEq(c.add({
              slot: "",
              value: "testMe",
              fail: fail
            }));
          });
          
          errTest(function () {
            verifyEq(c.add({
              slot: "1234",
              value: "testMe",
              fail: fail
            }));
          });
          
          // Error testing
          errTest(function () {
            c.add({
              slot: "test", 
              value: "an error test",
              fail: fail
            }); // Should cause Duplicate Slot exception
          });
              
          errTest(function () {
            c.add({
              slot: "test1",
              value: "an error test",
              fail: fail
            }); // Should cause Duplicate Slot exception
          });

          errTest(function () {
            c.add({
              value: new baja.Object(),
              fail: fail
            }); // Can only add concrete BValues
          });
          
          errTest(function () {
            c.add({
              value: new baja.Complex(),
              fail: fail
            }); // Can only add concrete BValues
          });
          
          errTest(function () {
            var c2 = baja.$("baja:Component");
            var c3 = baja.$("baja:Component");
            c2.add({
              value: c3,
              fail: fail
            });      
            c.add({
              value: c3,
              fail: fail
            }); // Cause an already parented exception
          });
        });
        
      });
      
      describe("mounted", function mounted() {
              
        it("string", function addString() {
          var propName = "addString",
              value = "slot_value";
          
          // Argument for add
          var addCallbacks = callbackify({
            slot: propName,
            value: value,
            ok: function (prop) {
              
              // Check we've got back what we expect through BajaScript
              expect(prop.getName()).toEqual(propName);
              expect(this.get(prop)).toEqual(value);
              
              // Now check what's been added to the Server
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: this.getNavOrd(),
                propName: propName,
                value: value,
                facets: baja.Facets.DEFAULT,
                flags: 0
              }));
            }
          });
          
          // Run and wait for this async op to complete
          runAndWait(function () {
            testFolder.add(addCallbacks);
          }, addCallbacks);
        });
        
        it("control point", function addPoint() {
          runs(function () {
            var propName = "addControlPoint",
                point = baja.$("control:NumericWritable"),
                pointValue = 123.45,
                flags = baja.Flags.SUMMARY | baja.Flags.TRANSIENT,
                facets = baja.Facets.make({ test1: "value1" });
            
            point.getIn4().setValue(pointValue);
            
            // Argument for add
            var addCallbacks = callbackify({
              slot: propName,
              value: point,
              ok: function (prop) {
                var comp = this.get(prop);
                
                // Ensure this component is leased
                lease(comp);
                            
                // Check we've got back what we expect through BajaScript
                expect(comp.getName()).toEqual(propName);
                expect(this.get(prop).getIn4().getValue()).toEqual(pointValue);
                
                // Now check what's been added to the Server
                verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                  ord: this.getNavOrd(),
                  propName: propName,
                  facets: facets,
                  flags: flags,
                  typeSpec: "control:NumericWritable"
                }));
                
                // Verify dynamic Property Slot...                  
                expect(prop.getName()).toEqual(propName);
                expect(prop.isFrozen()).toBeFalsy();
                expect(prop.getFlags()).toEqual(flags);
                expect(prop.getFacets().equals(facets)).toBeTruthy();
                expect(prop.getDefaultFlags()).toEqual(flags);
                expect(prop.getDefaultValue()).toEqual(comp);
                expect(prop.isTopic()).toBeFalsy();
                expect(prop.isAction()).toBeFalsy();
                expect(prop.isProperty()).toBeTruthy();            
              },
              flags: flags,
              facets: facets
            });
            
            // Run and wait for this async op to complete
            runAndWait(function () {
              testFolder.add(addCallbacks);
            }, addCallbacks);
          });
        });
        
        describe("add name", function testAddName() {
                
          it("wild card", function wildcardName() {
            var propName = "addWildCard?",
                value = 12.34;
            
            add(propName, value, "addWildCard");
            add(propName, value, "addWildCard1");
            add(propName, value, "addWildCard2");
            add(propName, value, "addWildCard3");
            add(propName, value, "addWildCard4");
          });
          
          it("null", function nullName() {
            var propName = null,
                value = baja.$("schedule:BooleanSchedule");
            
            add(propName, value, "BooleanSchedule");
            add(propName, value, "BooleanSchedule1");
            add(propName, value, "BooleanSchedule2");
            add(propName, value, "BooleanSchedule3");
            add(propName, value, "BooleanSchedule4");
          });
          
          it("escaped", function escapedName() {
            var newPropName = "F$3123$20add$20$2d$20this$20is$20a$20name$21$40",
                value = baja.$("baja:Weekday").get("tuesday");
            
            add(newPropName, value, newPropName);
          });
        });
        
        it("batch", function testBatch() {
          var batch = new baja.comm.Batch(),
              count = 0,
              addTest1 = "addBatchTest1",
              addTest2 = "addBatchTest2";
          
          // Batch up and couple of add operations together
          runs(function () {
            testFolder.add({
              slot: addTest1,
              value: baja.$("baja:Folder"),
              batch: batch,
              ok: function (prop) {
                count++;
              }
            });
            
            // Check NOTHING has been added to the Server at this point
            expect(count).toEqual(0);
            
            testFolder.add({
              slot: addTest2,
              value: baja.$("baja:Folder"),
              batch: batch,
              ok: function (prop) {
                count++;
              }
            });
            
            expect(count).toEqual(0);
                    
            // Commit the batch asynchronously
            batch.commit();
          });
          
          // For for everything to be added
          waitsFor(function () {
            return count === 2;
          });
          
          // Check what was added
          runs(function () {
            // Now check what's been added to the Server
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: addTest1,
              typeSpec: "baja:Folder"
            }));
            
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: addTest2,
              typeSpec: "baja:Folder"
            }));
          });       
        });
        
        it("error", function error() {
          var ok = function () {
            verify(false);
          },
          fail = function () {
            verify(true); // Pass the test if this fails...
          };
          
          function errorAdd(param) {
            param.ok = ok;
            param.fail = fail;
            
            // Error test add
            var addCallbacks = callbackify(param);
            runAndWait(function () {
              testFolder.add(addCallbacks);
            }, addCallbacks);
          }
          
          runs(function () {
            bajaJsPrint("\n*** Error Test: please ignore the following error messages ***\n");
            
            // These should all throw errors....
            errorAdd({}); // No value defined
            
            var propName = "addTestError";
            add(propName, "a test", propName);
            errorAdd({slot: propName, value: "a test"}); // Duplicate slot name
            errorAdd({slot: " invalid name", value: "a test"}); // Invalid slot name
            errorAdd({slot: "addErrorInvalidValue", value: {}}); // Invalid value
            errorAdd({slot: "addErrorNoValue"}); // No value
          });
          
          runs(function () {
            bajaJsPrint("\n*** End of Error Test ***");
          });         
        });       
      });
    });
  
    it("unique name", function testUniqueName() {
      var c = baja.$("baja:Component");
      
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test");
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test1");
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test2");
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test3");
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test4");
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test5");
      
      c.remove("test2");
      
      verifyEq(c.add({slot: c.getUniqueName("test"), value: "a value"}).getName(), "test2");
      
      errTest(function () {
        c.getUniqueName();
      });
      
      errTest(function () {
        c.getUniqueName("");
      });
      
      errTest(function () {
        c.getUniqueName(32);
      });
      
      errTest(function () {
        c.getUniqueName("1test");
      });
      
      errTest(function () {
        c.getUniqueName(" test");
      });
    });
  
    describe("set", function testSet() {
      
      describe("unmounted", function unmounted() {
        function fail(err) {
          throw new Error(err);
        }
        
        it("dynamic", function setDynamic() {
          var c = baja.$("baja:Component");
          
          c.add({
            slot: "fooBar",
            value: 1
          });
          verifyEq(c.get("fooBar"), 1);
          
          c.set({
            slot: "fooBar",
            value: 2
          });
          verifyEq(c.get("fooBar"), 2);
          
          c.set({
            slot: "fooBar",
            value: 23
          });
          verifyEq(c.get("fooBar"), 23);
        });
                
        it("frozen", function setFrozen() {
          var point = baja.$("control:NumericWritable"),
              statVal = baja.$("baja:StatusNumeric"),
              val = 12.3;
          
          // Set entire Struct
          statVal.setValue(val);
          point.setIn4(statVal);
          
          verify(point.getIn4().equals(statVal));
          verify(point.getIn4().getStatus().equals(statVal.getStatus()));
          verify(point.getIn4().getValue().equals(val));
          verify(point.get("in4").equals(statVal));
          
          // Set struct property
          val = 34.57;
          point.getIn5().setValue(val);
          verify(point.getIn5().getValue().equals(val));
          verify(point.get("in5").getValue().equals(val));
        });
        
        it("error", function error() {
          var c = baja.$("baja:Component");
          
          errTest(function () {
            c.set({fail: fail});
          });
          
          errTest(function () {
            c.set({
              slot: "fooBar",
              fail: fail
            });
          });
          
          errTest(function () {
            c.set({
              value: 3,
              fail: fail
            });
          });
          
          errTest(function () {
            c.set({
              slot: "test",
              value: 3,
              fail: fail
            });
          });
          
          runs(function () {
            // Attempt to set an Action
            var point = baja.$("control:NumericWritable");
            point.set({
              slot: "set",
              value: 123123,
              ok: function () {
                verify(false);
              },
              fail: function (err) {
                verify(true);
              }
            });
          });
          
          runs(function () {
            // Attempt to set a Topic
            var cls = baja.$("alarm:AlarmClass");
            cls.set({
              slot: "alarm",
              value: 123123,
              ok: function () {
                verify(false);
              },
              fail: function (err) {
                verify(true);
              }
            });
          });
        });
      });
    
      describe("mounted", function mounted() {
        
        describe("point", function point() {
          
          it("status numeric value (auto-gen accessor)", function setPointStatusNumericValueAcc() {
            
            // Set the value property on a status numeric using the auto-generated accessor
            var propName = "setNwStatNumValueAcc";
            
            // Add a point and set a value on it. Ensure the value changes on the Server
            add(propName, baja.$("control:NumericWritable"));
            
            runs(function () {
              var point = lease(testFolder.get(propName)),
                  value = 123123,
                  setCallback = callbackify({
                    value: value
                  });
              
              // Run and wait for this async op to complete
              runAndWait(function () {
                point.getIn4().setValue(setCallback);
              }, setCallback);
              
              runs(function () {
                expect(point.getIn4().getValue()).toEqual(value);
              });
              
              // Check the value has been 'set' as expected in the Station
              runs(function () {
                verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                  ord: baja.Ord.make(point.getNavOrd().toString() + "/in4"),
                  propName: "value",
                  value: value
                }));
              });
            });
          });
          
          it("status numeric (auto-gen accessor)", function statusNumericAcc() {
            
            // Set a point's status numeric using the auto-generated accessor
            var propName = "setNwStatNumAcc";
            
            // Add a point and set a value on it. Ensure the value changes on the Server
            add(propName, baja.$("control:NumericWritable"));
            
            runs(function () {
              var point = lease(testFolder.get(propName)),
                  value = baja.$("baja:StatusNumeric", {
                    value: 2345.234
                  }),
                  setCallback = callbackify({
                    value: value
                  });
              
              // Run and wait for this async op to complete
              runAndWait(function () {
                point.setIn4(setCallback);
              }, setCallback);
              
              runs(function () {
                expect(point.getIn4()).toEquivalent(value);
              });
              
              // Check the value has been 'set' as expected in the Station
              runs(function () {
                verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                  ord: point.getNavOrd(),
                  propName: "in4",
                  value: value
                }));
              });
            });
          });
          
          it("status numeric value (set)", function statusNumericValueSet() {
            
            // Set the value property on a status numeric using the set method
            var propName = "setNwStatNumValueSet";
            
            // Add a point and set a value on it. Ensure the value changes on the Server
            add(propName, baja.$("control:NumericWritable"));
            
            runs(function () {
              var point = lease(testFolder.get(propName)),
                  value = 123123,
                  setCallback = callbackify({
                    slot: "value",
                    value: value
                  });
              
              // Run and wait for this async op to complete
              runAndWait(function () {
                point.getIn4().set(setCallback);
              }, setCallback);
              
              runs(function () {
                expect(point.getIn4().getValue()).toEqual(value);
              });
              
              // Check the value has been 'set' as expected in the Station
              runs(function () {
                verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                  ord: baja.Ord.make(point.getNavOrd().toString() + "/in4"),
                  propName: "value",
                  value: value
                }));
              });
            });
          });
          
          it("status numeric (set)", function statusNumericSet() {
            
            // Set a point's status numeric using the set method
            var propName = "setNwStatNumSet";
            
            // Add a point and set a value on it. Ensure the value changes on the Server
            add(propName, baja.$("control:NumericWritable"));
            
            runs(function () {
              var point = lease(testFolder.get(propName)),
                  value = baja.$("baja:StatusNumeric", {
                    value: 2345.234
                  }),
                  setCallback = callbackify({
                    slot: "in4",
                    value: value
                  });
              
              // Run and wait for this async op to complete
              runAndWait(function () {
                point.set(setCallback);
              }, setCallback);
              
              runs(function () {
                expect(point.getIn4()).toEquivalent(value);
              });
              
              // Check the value has been 'set' as expected in the Station
              runs(function () {
                verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                  ord: point.getNavOrd(),
                  propName: "in4",
                  value: value
                }));
              });
            });
          });
        });
                
        describe("component", function component() {
          
          it("dynamic Property", function dynamicProperty() {
            
             // Set a point's status numeric using the set method
            var propName = "setDynString";
            
            // Add a point and set a value on it. Ensure the value changes on the Server
            add(propName, "hey this is a Property!");
                        
            runs(function () {
              var value = "another String!",
                  setCallback = callbackify({
                    slot: propName,
                    value: value
                  });
              
              // Run and wait for this async op to complete
              runAndWait(function () {
                testFolder.set(setCallback);
              }, setCallback);
              
              runs(function () {
                expect(testFolder.get(propName)).toEquivalent(value);
              });
              
              // Check the value has been 'set' as expected in the Station
              runs(function () {
                verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                  ord: testFolder.getNavOrd(),
                  propName: propName,
                  value: value
                }));
              });
            });
          });
        });
        
        it("batch", function testBatch() {
          
          var batch = new baja.comm.Batch(),
              compName1 = "setBatch1",
              compName2 = "setBatch2",
              propName = "in4";
                              
          add(compName1, baja.$("control:BooleanWritable"));
          add(compName2, baja.$("control:BooleanWritable"));
          
          var setcallback1 = callbackify({
            slot: propName,
            value: baja.$("baja:StatusBoolean", {
              value: true,
              status: baja.Status.disabled
            }),
            batch: batch
          });
          
          var setcallback2 = callbackify({
            slot: propName,
            value: baja.$("baja:StatusBoolean", {
              value: false,
              status: baja.Status.fault
            }),
            batch: batch
          });
          
          // Perform first set (with batch this should make no network call)
          runs(function () {
            testFolder.get(compName1).set(setcallback1);
          });
          
          // Perform second set (with batch this should make no network call)
          runs(function () {
            testFolder.get(compName2).set(setcallback2);
          });
          
          // Ensure the first value hasn't been set on the Server
          runs(function () {
            verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.get(compName1).getNavOrd(),
              propName: propName,
              value: setcallback1.value.newCopy()
            })));
          });
          
          // Ensure the second value hasn't been set on the Server
          runs(function () {
            verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.get(compName2).getNavOrd(),
              propName: propName,
              value: setcallback2.value.newCopy()
            })));
          });
          
          // Add callback
          runs(function () {
            batch.addCallback(function ok() {
              verify(true);
            },
            function fail(err) {
              verify(false);
            });
          });
                    
          // Commit the batch
          runs(function () {
            batch.commit();
          });
          
          waitsFor(function () {
            return setcallback1.isCalled();
          });
          
          waitsFor(function () {
            return setcallback2.isCalled();
          });
           
          // Now everything has been committed then make sure the server values are what they're suppose to be
          runs(function () {
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.get(compName1).getNavOrd(),
              propName: propName,
              value: setcallback1.value.newCopy()
            }));
            
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.get(compName2).getNavOrd(),
              propName: propName,
              value: setcallback2.value.newCopy()
            }));
          });
        });
        
      });
    }); 
  
    describe("get", function testGet() {
      
      it("verify frozen Property Slot", function verifyFrozenPropertySlot() { 
        runs(function () {
          var property = baja.$("control:NumericWritable").getSlot("in4");
          
          expect(property.getName()).toEqual("in4");
          expect(property.isFrozen()).toBeTruthy();
          expect(property.getFlags()).toEqual(baja.Flags.TRANSIENT);
          expect(property.getFacets().equals(baja.Facets.DEFAULT)).toBeTruthy();
          expect(property.getDefaultFlags()).toEqual(baja.Flags.TRANSIENT);
          expect(property.getDefaultValue().getValue()).toEqual(0);
          expect(property.getDefaultValue().getStatus().equals(baja.Status.nullStatus)).toBeTruthy();
          expect(property.isTopic()).toBeFalsy();
          expect(property.isAction()).toBeFalsy();
          expect(property.isProperty()).toBeTruthy();
        });
      });

      it("dynamic", function dynamic() {
        var fl = baja.$("baja:Float").make(12),
            c = baja.$("baja:Component");
        
        c.add({
          slot: "num",
          value: fl
        });
        
        verify(c.get("num") === fl);
        verify(c.getValueOf("num") === 12);
        verify(c.getValueOf("num") !== fl);
        
        var c2 = baja.$("baja:Component");
        c.add({
          slot: "c2",
          value: c2
        });
        
        verify(c.get("c2") === c2);
        verify(c.getValueOf("c2") === c2);
      });
      
      it("frozen", function frozen() {
        var point = baja.$("control:NumericWritable"),
            value = 12.34;
        
        point.getIn4().setValue(value);
        
        expect(point.getIn4().getValue()).toEqual(value);
        expect(point.getIn4().get("value")).toEqual(value);
        expect(point.getIn4().getValueOf("value")).toEqual(value);
      });
    });
          
    describe("cursors", function testCursors() {
    
      it("dynamic Properties", function dynamicProperties() {
        runs(function () {
          var c = baja.$("baja:Component"),
              i,
              cur;
          
          verify(c.getSlots().isEmpty());
          verifyEq(c.getSlots().getSize(), 0);
          
          // Add a bunch of Properties
          i = 0;
          for (i = 0; i < 10; ++i) {
            c.add({
              value: "testMe" + i,
              cx: { display: "testDisplayName" + i}
            });
          }
          
          verify(!c.getSlots().isEmpty());
          verifyEq(c.getSlots().getSize(), 10);
          
          verifyEq(c.$map.getSize(), 10);
          
          // Verify we can read the Slots out again
          i = 0;
          cur = c.getSlots();
          while (cur.next()) {
            verifyEq(c.get(cur.get()), "testMe" + i);
            i++;
          }
          
          // Verify we can read the Slot display Strings out again
          i = 0;
          cur = c.getSlots();
          while (cur.next()) {
            verifyEq(cur.getDisplay(), "testDisplayName" + i);
            i++;
          }
          verifyEq(i, 10);
          
          // Verify that each works as expected
          i = 0;
          c.getSlots().each(function (slot) {
            verify(slot instanceof baja.Slot);
            verify(this === c);
            verifyEq(this.get(slot), "testMe" + i);
            i++;
          });
          verifyEq(i, 10);
          
          // Verify we can read the Properties out again
          cur = c.getSlots().properties();
          i = 0;
          while (cur.next()) {
            verifyEq(c.get(cur.get()), "testMe" + i);
            i++;
          }
          verifyEq(i, 10);
          
          var slots = cur.toArray();
          verifyEq(slots.length, 10);
          for (i = 0; i < slots.length; ++i) {
            verifyEq(c.get(slots[i]), "testMe" + i);
          }
          
          var values = cur.toValueArray();
          verifyEq(values.length, 10);
          for (i = 0; i < values.length; ++i) {
            verifyEq(values[i], "testMe" + i);
          }
          
          // Test each
          cur = c.getSlots().properties();
          i = 0;
          cur.each(function (slot, index) {
            verifyEq(i, index);
            verifyEq(this.get(cur.get()), "testMe" + i);
            i++;
          });
          verifyEq(i, 10);
          
          cur = c.getSlots().properties();
          i = 0;
          cur.eachValue(function (value, index) {
            verifyEq(i, index);
            verifyEq(value, "testMe" + i);
            i++;
          });
          verifyEq(i, 10);
         
          cur = c.getSlots().properties();
          i = 0;
          cur.eachDisplay(function (display, index) {
            verifyEq(i, index);
            verifyEq(display, "testDisplayName" + i);
            i++;
          });
          verifyEq(i, 10);
          
          c = baja.$("baja:Component");
          
          // Test first, firstValue, last, lastValue
          verify(c.getSlots().first() === null);
          verify(c.getSlots().firstValue() === null);
          verify(c.getSlots().firstDisplay() === null);
          verify(c.getSlots().last() === null);
          verify(c.getSlots().lastValue() === null);
          verify(c.getSlots().lastDisplay() === null);
          
          baja.iterate(4, function (i) {
            c.add({
              slot: "prop" + i,
              value: "test" + i,
              cx: { display: "display" + i }
            });
          });
              
          verifyEq(c.getSlots().getSize(), 4);
          verifyEq(c.getSlots().first().getName(), "prop0");
          verifyEq(c.getSlots().firstValue(), "test0");
          verifyEq(c.getSlots().firstDisplay(), "display0");
          verifyEq(c.getSlots().last().getName(), "prop3");
          verifyEq(c.getSlots().lastValue(), "test3");
          verifyEq(c.getSlots().lastDisplay(), "display3");
          
          // Test toValueArray
          verifyEq(c.getSlots().toValueArray().join(","), "test0,test1,test2,test3");
          
          // Test slot map
          var map = c.getSlots().toMap();
          
          verifyEq("prop0", map.prop0.getName());
          verifyEq("prop1", map.prop1.getName());
          verifyEq("prop2", map.prop2.getName());
          verifyEq("prop3", map.prop3.getName());
          
          // Test value map
          map = c.getSlots().toValueMap();
          verifyEq(map.prop0, "test0");
          verifyEq(map.prop1, "test1");
          verifyEq(map.prop2, "test2");
          verifyEq(map.prop3, "test3");
          
          // Test display map
          map = c.getSlots().toDisplayMap();
          verifyEq(map.prop0, "display0");
          verifyEq(map.prop1, "display1");
          verifyEq(map.prop2, "display2");
          verifyEq(map.prop3, "display3");
          
          // Test 'is'
          baja.iterate(4, 9, function (i) {
            c.add({
              slot: "num" + i,
              value: i,
              flags: baja.Flags.SUMMARY
            });
          });
          
          verifyEq(c.getSlots().getSize(), 9);
          verifyEq(c.getSlots().is("baja:Double").getSize(), 5);
          verifyEq(c.getSlots().is("baja:String").getSize(), 4);
          verifyEq(c.getSlots().is("baja:Boolean").getSize(), 0);
          verifyEq(c.getSlots().is("baja:Double", "baja:String").getSize(), 9);
          verifyEq(c.getSlots().isSimple().getSize(), 9);
          verifyEq(c.getSlots().isValue().getSize(), 9);
          verifyEq(c.getSlots().isComponent().getSize(), 0);
          verifyEq(c.getSlots().isNumber().getSize(), 5);
          
          // Equal Type
          verifyEq(c.getSlots().equalType("baja:Double").getSize(), 5);
          verifyEq(c.getSlots().equalType("baja:String").getSize(), 4);
          verifyEq(c.getSlots().equalType("baja:String", "baja:Double").getSize(), 9);
          verifyEq(c.getSlots().equalType("baja:Boolean").getSize(), 0);
          verifyEq(c.getSlots().equalValue("test3").first().getName(), "prop3");
          verifyEq(c.getSlots().equalValue("test2", "test3").toArray().join(","), "prop2,prop3");
          
          // Equivalent
          baja.iterate(3, function (i) {
            c.add({ 
              slot: "bool" + i,
              value: true,
              flags: baja.Flags.OPERATOR | baja.Flags.SUMMARY
            });
          });
          
          verifyEq(c.getSlots().equivalent("test3").first().getName(), "prop3");
          verifyEq(c.getSlots().equivalent(true).getSize(), 3);
          verify(!c.getSlots().equivalent(true).isEmpty());
          verifyEq(c.getSlots().equivalent(false).getSize(), 0);
          verify(c.getSlots().equivalent(false).isEmpty());
          verifyEq(c.getSlots().equivalent(true).toValueArray().join(","), "true,true,true");
          verifyEq(c.getSlots().equivalent("test2", "test3").toArray().join(","), "prop2,prop3");
          
          // slotName
          verifyEq(c.getSlots().slotName(/^prop/).getSize(), 4);
          verifyEq(c.getSlots().slotName(/^num/).getSize(), 5);
          verifyEq(c.getSlots().slotName(/^bool/).getSize(), 3);
          
          // Custom filter
          verifyEq(c.getSlots().filter(function (slot) {
            return this.getFlags(slot) & baja.Flags.OPERATOR;
          }).toValueArray().join(","), "true,true,true");
          
          // Properties, Actions and Topics
          verifyEq(c.getSlots().properties().getSize(), 12);
          verifyEq(c.getSlots().dynamic().properties().getSize(), 12);
          verifyEq(c.getSlots().frozen().properties().getSize(), 0);
          verifyEq(c.getSlots().frozen().actions().getSize(), 0);
          verifyEq(c.getSlots().frozen().topics().getSize(), 0);
          
          // Flags
          verifyEq(c.getSlots().flags(baja.Flags.SUMMARY).toValueArray().join(","), "4,5,6,7,8,true,true,true");
          
          // Add a child component
          c.add({
            value: baja.$("baja:Component")
          });
          
          verifyEq(c.getSlots().isComponent().getSize(), 1);
          verifyEq(c.getSlots().isComplex().getSize(), 1);
          verifyEq(c.getSlots().isStruct().getSize(), 0);
        });
      });
      
      it("frozen Properties", function frozenProperties() {
        runs(function () {
                  
          var p = baja.$("control:NumericWritable"),
              map, 
              array;
          
          // Test first, firstValue, last, lastValue
          expect(p.getSlots().first()).toEqual(p.getSlot("facets"));                   
          expect(p.getSlots().firstValue().getType()).toEqual(baja.Facets.DEFAULT.getType());
          expect(p.getSlots().firstDisplay()).toEqual("units=null,precision=1,min=-inf,max=+inf");
          
          expect(p.getSlots().last()).toEqual(p.getSlot("set"));
          expect(p.getSlots().lastValue()).toBeNull();
          expect(p.getSlots().lastDisplay()).toBeNull();
          
          // Add some dynamic Properties to this point
          baja.iterate(4, function (i) {
            p.add({
              slot: "prop" + i,
              value: "test" + i,
              cx: { display: "display" + i }
            });
          });
              
          expect(p.getSlots().getSize()).toEqual(31);
          
          expect(p.getSlots().first().getName()).toEqual("facets");
          expect(p.getSlots().last().getName()).toEqual("prop3");
          expect(p.getSlots().lastValue()).toEqual("test3");
          expect(p.getSlots().lastDisplay()).toEqual("display3");
                    
          expect(p.getSlots().toValueArray().length).toEqual(25);
          expect(p.getSlots().frozen().getSize()).toEqual(27);
          expect(p.getSlots().dynamic().getSize()).toEqual(4);
          
          // Test slot map
          map = p.getSlots().toMap();
          
          expect(map.in1.getName()).toEqual("in1");
          expect(map.set.getName()).toEqual("set");
          expect(map.set.isAction()).toBeTruthy();
          expect(map.prop2.getName()).toEqual("prop2");
          
          expect(p.getSlots().is("baja:StatusNumeric").getSize()).toEqual(18); 
        });
      });
        
      it("frozen Actions", function frozenActions() {
        
        runs(function () {
          var p = baja.$("control:NumericWritable"),
              map = p.getSlots().actions().toMap();
          
          // Test Action access
          expect(p.getSlots().actions().getSize()).toEqual(6);
          expect(map.set.getName()).toEqual("set");
          expect(map.emergencyOverride.getName()).toEqual("emergencyOverride");
          expect(map.emergencyAuto.getName()).toEqual("emergencyAuto");
          expect(map.override.getName()).toEqual("override");
          expect(map.auto.getName()).toEqual("auto");
          expect(map.execute.getName()).toEqual("execute");
        });
      });
      
      it("frozen Topics", function frozenTopics() {
        
        runs(function () {
          // Test Topic access
          var alarmCls = baja.$("alarm:AlarmClass"),
              array = alarmCls.getSlots().topics().toArray();
          
          expect(array.length).toEqual(4);
          expect(array[0]).toEqual(alarmCls.getSlot("alarm"));
          expect(array[1]).toEqual(alarmCls.getSlot("escalatedAlarm1"));
          expect(array[2]).toEqual(alarmCls.getSlot("escalatedAlarm2"));
          expect(array[3]).toEqual(alarmCls.getSlot("escalatedAlarm3"));
        }); 
      });
    });
    
    describe("remove", function testRemove() {
      
      describe("unmounted", function unmounted() {
      
        it("dynamic", function dynamic() {
                    
          var c = new baja.Component();
          c.add({
            slot: "test", 
            value: 12.2
          });
          
          c.add({
            slot: "test1", 
            value: true
          });
          
          c.add({
            slot: "test2",
            value: "This is a String"
          });
          
          c.add({
            slot: "test3", 
            value: 123
          });
          
          c.add({
            slot: "test4",
            value: new baja.Component()
          });
          
          expect(c.getSlots().getSize()).toEqual(5);
          
          c.remove("test");
          expect(c.getSlots().getSize()).toEqual(4);
          
          c.remove({
            slot: "test1"
          });
          expect(c.getSlots().getSize()).toEqual(3);
          
          c.remove(c.getSlot("test2"));
          expect(c.getSlots().getSize()).toEqual(2);
          
          c.remove("test3");
          expect(c.getSlots().getSize()).toEqual(1);
          
          c.remove(c.get("test4"));
          expect(c.getSlots().getSize()).toEqual(0);
          
          c.add(callbackify({
            slot: "test", 
            value: "this is a test"
          }));
                    
          expect(c.getSlots().getSize()).toEqual(1);
          
          c.remove("test");
          expect(c.getSlots().getSize()).toEqual(0);       
          
        });
        
        it("error", function error() {          
          var c = baja.$("baja:Component");
          
          c.remove(errCallbackify({
            slot: "doesNotExist"
          }));
          
          runs(function () {
            
            // Attempt to remove a frozen Slot!
            var point = baja.$("control:StringWritable");
            
            point.remove(errCallbackify({
              slot: "in3"
            }));
            
            expect(point.has("in3")).toBeTruthy();
            
          });
          
        });
      });
      
      describe("mounted", function mounted() {
        
        it("component", function component() {
          
          var propName = "removeEnumPoint";
            
          runs(function () {
            add(propName, baja.$("control:EnumWritable"));
          });
        
          runs(function () {
            var removeCallback = callbackify({
              slot: propName
            });
            
            // Remove and wait for the callback
            runAndWait(function () {
              testFolder.remove(removeCallback);
            }, removeCallback);
            
            runs(function () {            
              // Make sure Property no longer exists on the Server
              verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName
              })));
            });
          });         
        });
        
        it("batch", function testBatch() {
          
          var batch = new baja.comm.Batch(),
              propName1 = "removeBatchEnumPoint1",
              propName2 = "removeBatchEnumPoint2";
          
          add(propName1, baja.$("control:EnumWritable"));
          add(propName2, baja.$("control:EnumWritable"));
          
          var removecallback1 = callbackify({
            slot: propName1,
            batch: batch
          }); 
          
          runs(function () {
            testFolder.remove(removecallback1);
          });
     
          var removecallback2 = callbackify({
            slot: propName2,
            batch: batch
          }); 
          
          runs(function () {
            testFolder.remove(removecallback2);
          }, removecallback2);
          
          runs(function () {
            // Make sure Property still exists (we haven't committed the batch yet)
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName1
            }));
            
            // Make sure Property still exists (we haven't committed the batch yet)
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName2
            }));
          });  
          
          runs(function () {
            batch.addCallback(function ok() {
              verify(true);
            },
            function fail(err) {
              verify(false);
            });
          });
          
          runs(function() {
            batch.commit();
          });
          
          waitsFor(function () {
            return removecallback1.isCalled();
          });
          
          waitsFor(function () {
            return removecallback2.isCalled();
          });
          
          runs(function () { 
            // Make sure Property doesn't exist on the Server
            verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName1
            })));
            
            // Make sure Property doesn't exist on the Server
            verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName2
            })));
          });
        });  
      });
    });
  
    describe("rename", function testRename() {
      
      describe("unmounted", function unmounted() {
        
        it("dynamic", function dynamic() {
                    
          var c = new baja.Component();
          
          c.add(callbackify({
            slot: "boo",
            value: "baaa!!!!"
          }));
          
          c.add(callbackify({
            slot: "test", 
            value: "some data"
          }));
          
          c.set(callbackify({
            slot: "test", 
            value: "Something else!"
          }));
          
          expect(c.get("test")).toEqual("Something else!");
          expect(c.getSlot("test")).not.toBeNull();
          
          c.rename(callbackify({
            slot: "test", 
            newName: "foo"
          }));
          
          verify(c.getSlot("foo") !== null);
          verify(c.getSlot("test") === null);
          
          verifyEq(c.get("foo"), "Something else!");
                                        
          var slot = c.getSlot("foo");
          c.rename({
            slot: slot, 
            newName: "doo"
          });
          
          verify(c.getSlot("foo") === null);
          verify(c.getSlot("doo") !== null);
          
          c.remove(slot);
          verify(c.getSlot("doo") === null);    
        });
        
        it("error", function error() {
          
          function errOk() {
            verify(false);
          }
          
          function errFail() {
            verify(true);
          }
          
          var c = new baja.Component();
                   
          c.add(callbackify({
            slot: "test", 
            value: "some data"
          }));
          
          // Error test rename with invalid slot names
          c.rename({
            slot: "test", 
            newName: "fo o",
            fail: errFail,
            ok: errOk
          });
        
          c.rename({
            slot: "test", 
            newName: "7foo",
            fail: errFail,
            ok: errOk
          });
          
          c.rename({
            slot: "test", 
            newName: "",
            fail: errFail,
            ok: errOk
          });
          
          c.rename({
            slot: "test", 
            newName: "qwer@ty",
            fail: errFail,
            ok: errOk
          });
          
          // Try to rename a Slot that doesn't exist
          c.rename({
            slot: "doodoodoo", 
            newName: "boo",
            fail: errFail,
            ok: errOk
          });
          
          // Try to rename a frozen Slot
          runs(function () {
            var point = baja.$("control:NumericWritable");
            point.rename({
              slot: "in4",
              newName: "somethingElse",
              ok: errOk,
              fail: errFail
            });
          });
        });
                
      });
      
      describe("mounted", function mounted() {
        
        it("rename string Property", function renameStr() {
          
          // Set a point's status numeric using the set method
          var propName = "renameDynString";
          
          // Add a point and set a value on it. Ensure the value changes on the Server
          add(propName, "hey this is a Property!");
                      
          runs(function () {
            var newName = "renameDynStringRenamed",
                renameCallback = callbackify({
                  slot: propName,
                  newName: newName
                });
            
            // Run and wait for this async op to complete
            runAndWait(function () {
              testFolder.rename(renameCallback);
            }, renameCallback);
            
            runs(function () {
              expect(testFolder.has(newName)).toBeTruthy();
              expect(testFolder.has(propName)).not.toBeTruthy();
            });
            
            // Check the value has been 'set' as expected in the Station
            runs(function () {
              
              // Ensure the newly named Property exists in the Station
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: newName
              }));
              
              // Ensure the old Property doesn't exist in the Station
              verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName
              })));
            });
          });
        });
               
        it("batch", function testBatch() {
          
          var batch = new baja.comm.Batch(),
              propName1 = "renameBatchEnumPoint1",
              newName1 = "renameBatch1",
              propName2 = "renameBatchEnumPoint2",
              newName2 = "renameBatch2";
          
          add(propName1, baja.$("control:EnumWritable"));
          add(propName2, baja.$("control:EnumWritable"));
          
          var renamecallback1 = callbackify({
            slot: propName1,
            newName: newName1,
            batch: batch
          }); 
          
          runs(function () {
            testFolder.rename(renamecallback1);
          });
     
          var renamecallback2 = callbackify({
            slot: propName2,
            newName: newName2,
            batch: batch
          }); 
          
          runs(function () {
            testFolder.rename(renamecallback2);
          }, renamecallback2);
          
          runs(function () {
            // Make sure Property still exists (we haven't committed the batch yet)
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName1
            }));
            
            // Make sure Property still exists (we haven't committed the batch yet)
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName2
            }));
          });  
          
          runs(function () {
            batch.addCallback(function ok() {
              verify(true);
            },
            function fail(err) {
              verify(false);
            });
          });
          
          runs(function() {
            batch.commit();
          });
          
          waitsFor(function () {
            return renamecallback1.isCalled();
          });
          
          waitsFor(function () {
            return renamecallback2.isCalled();
          });
          
          runs(function () { 
            // Make sure Property doesn't exist on the Server
            verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName1
            })));
            
            // Make sure Property doesn't exist on the Server
            verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: propName2
            })));
            
            // Make sure Property doesn't exist on the Server
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: newName1
            }));
            
            // Make sure Property doesn't exist on the Server
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: testFolder.getNavOrd(),
              propName: newName2
            }));
          });
        });  
      });     
    });
  
  
    describe("reorder", function testReorder() {
      
      describe("unmounted", function unmounted() {
        
        function addProp(c, name, value) {
          return c.add(callbackify({
            slot: name,
            value: value
          }));
        }
        
        it("reorder", function reorder() {
  
          // Simple reorder test
          function simpleReorder(c) {
            c.reorder(callbackify({
              dynamicProperties: ["first", "second", "third", "fourth", "fifth"]
            }));
            var cursor = c.getSlots().dynamic().properties(),
                i = 0;
            
            while (cursor.next()) {
              verifyEq(++i, c.get(cursor.get()).valueOf());
            }
            
            verifyEq(i, 5);
          }
          
          // Test reordering dynamic Properties
          var c = new baja.Component();
          c.add(callbackify({
            slot: "first",
            value: 1
          }));
          
          c.add(callbackify({
            slot: "second", 
            value: 2
          }));
          
          c.add(callbackify({
            slot: "fifth", 
            value: 5
          }));
          
          c.add(callbackify({
            slot: "third", 
            value: 3
          }));
          
          c.add(callbackify({
            slot: "fourth", 
            value: 4
          }));
          
          simpleReorder(c);
          
          c = new baja.Component();
                              
          addProp(c, "fifth", 5);
          addProp(c, "third", 3);
          addProp(c, "second", 2);
          addProp(c, "fourth", 4);
          addProp(c, "first", 1);    
          simpleReorder(c);
          
          var cursor = c.getSlots().properties();
                             
          cursor.next();
          verifyEq(c.get(cursor.get()).valueOf(), 1);
          cursor.next();
          verifyEq(c.get(cursor.get()).valueOf(), 2);
          cursor.next();
          verifyEq(c.get(cursor.get()).valueOf(), 3);
          cursor.next();
          verifyEq(c.get(cursor.get()).valueOf(), 4);
          cursor.next();
          verifyEq(c.get(cursor.get()).valueOf(), 5);    
          
          
          runs(function () {
            var p = baja.$("control:NumericWritable");
            
            addProp(p, "fifth", 5);
            addProp(p, "third", 3);
            addProp(p, "second", 2);
            addProp(p, "fourth", 4);
            addProp(p, "first", 1);    
            simpleReorder(p);
          });
          
        });
        
        it("error", function error() {
                                        
          // Miss out reordering sixth argument
          var c = new baja.Component();
          addProp(c, "fifth", 5);
          addProp(c, "third", 3);
          addProp(c, "first", 1);
          addProp(c, "second", 2);
          addProp(c, "fourth", 4);
          addProp(c, "sixth", 6);
          c.reorder(errCallbackify({
            dynamicProperties: ["first", "second", "third", "fourth", "fifth"]
          }));
                    
          // Only reorder seventh and eight
          c = new baja.Component();
          addProp(c, "fifth", 5);
          addProp(c, "third", 3);
          addProp(c, "first", 1);
          addProp(c, "second", 2);
          addProp(c, "fourth", 4);
          addProp(c, "sixth", 6);
          c.reorder(errCallbackify({
            dynamicProperties: ["seventh", "eigth"]
          }));
          
          // Attempt to reorder Properties that don't exist
          c = new baja.Component();
          c.reorder(errCallbackify({
            dynamicProperties: ["first", "second", "third"]
          }));
          
          runs(function () {            
            var p = baja.$("control:NumericWritable");
            
            addProp(p, "fifth", 5);
            addProp(p, "third", 3);
            addProp(p, "second", 2);
            addProp(p, "fourth", 4);
            addProp(p, "first", 1);   
            
            // Attempt to reorder a frozen Property...
            p.reorder(errCallbackify({
              dynamicProperties: ["first", "second", "third", "fourth", "fifth", "in4"]
            }));
          });
        });       
      });     
    
      describe("mounted", function mounted() {
    
        it("reorder", function reorder() {
          runs(function () {
            var reorderPoint = "reorderPoint",
                point;
            
            add(reorderPoint, baja.$("control:BooleanWritable"));
            
            runs(function () {
              point = lease(testFolder.get(reorderPoint));
              
              function confirmOrder(order) {
                // Get the reorder Slots from the test folder
                var currentOrder = point.getSlots().dynamic().properties().filter(function (prop) {
                  return prop.getName().indexOf("reorder") === 0;
                }).toArray();
                
                expect(currentOrder.join(",")).toEqual(order.join(","));
              }
              
              var count = 0,
                  countOk = function () {
                    count++;
                  };
              
              // Add some Properties to this point...
              var batch = new baja.comm.Batch();
              point.add({
                slot: "reorder1",
                value: baja.$("control:EnumPoint"),
                batch: batch,
                ok: countOk
              });
              
              point.add({
                slot: "reorder2",
                value: "this is a test",
                batch: batch,
                ok: countOk
              });
              
              point.add({
                slot: "reorder3",
                value: 12.56,
                batch: batch,
                ok: countOk
              });
                            
              batch.commit();
              
              waitsFor(function () {
                return count === 3;
              });
              
              runs(function () {
                var order =  ["reorder1", "reorder2", "reorder3"],
                    newOrder =  ["reorder3", "reorder2", "reorder1"];
                
                // Confirm order of properties in client
                confirmOrder(order);
                
                // Check the order on the Server
                verifyServerCall("checkOrder", baja.$("baja:Component", {
                  ord: testFolder.getNavOrd(),
                  propName: reorderPoint,
                  order: order.join(",")
                }));
                
                var reorderCallback = callbackify({
                  dynamicProperties: newOrder
                });
                
                // Run and wait for this async op to complete
                runAndWait(function () {
                  point.reorder(reorderCallback);
                }, reorderCallback);
                
                runs(function () {
                  // Confirm the new order in the client
                  confirmOrder(newOrder);
                  
                  // Check the order on the Server
                  verifyServerCall("checkOrder", baja.$("baja:Component", {
                    ord: testFolder.getNavOrd(),
                    propName: reorderPoint,
                    order: newOrder.join(",")
                  }));
                });
                
              });                            
            });
          });           
        });
      
        it("batch", function batch() {
          runs(function () {
            var reorderPoint = "reorderPointBatch",
                point;
            
            add(reorderPoint, baja.$("control:BooleanWritable"));
            
            runs(function () {
              point = lease(testFolder.get(reorderPoint));
              
              function confirmOrder(order) {
                // Get the reorder Slots from the test folder
                var currentOrder = point.getSlots().dynamic().properties().filter(function (prop) {
                  return prop.getName().indexOf("reorder") === 0;
                }).toArray();
                
                expect(currentOrder.join(",")).toEqual(order.join(","));
              }
              
              // Add some Properties to this point...
              var batch = new baja.comm.Batch(),
                  count = 0,
                  countOk = function () {
                    count++;
                  };
              
              point.add({
                slot: "reorder1",
                value: baja.$("control:EnumPoint"),
                batch: batch,
                ok: countOk
              });
              
              point.add({
                slot: "reorder2",
                value: "this is a test",
                batch: batch,
                ok: countOk
              });
              
              point.add({
                slot: "reorder3",
                value: 12.56,
                batch: batch,
                ok: countOk
              });
                            
              batch.commit();
              
              waitsFor(function () {
                return count === 3;
              });
              
              runs(function () {
                var order =  ["reorder1", "reorder2", "reorder3"],
                    newOrder =  ["reorder3", "reorder2", "reorder1"];
                
                // Confirm order of properties in client
                confirmOrder(order);
                
                // Check the order on the Server
                verifyServerCall("checkOrder", baja.$("baja:Component", {
                  ord: testFolder.getNavOrd(),
                  propName: reorderPoint,
                  order: order.join(",")
                }));
                
                var reorderBatch = new baja.comm.Batch(),
                    reorderCallback = callbackify({
                  dynamicProperties: newOrder,
                  batch: reorderBatch
                });
                                
                // Run and wait for this async op to complete
                runs(function () {
                  point.reorder(reorderCallback);
                });
                
                runs(function () {
                  // Check the order hasn't changed yet!
                  verifyServerCall("checkOrder", baja.$("baja:Component", {
                    ord: testFolder.getNavOrd(),
                    propName: reorderPoint,
                    order: order.join(",")
                  }));
                });
                
                runAndWait(function () {
                  reorderBatch.commit();
                }, reorderCallback);
                                                
                runs(function () {
                  // Confirm the new order in the client
                  confirmOrder(newOrder);
                  
                  // Check the order on the Server
                  verifyServerCall("checkOrder", baja.$("baja:Component", {
                    ord: testFolder.getNavOrd(),
                    propName: reorderPoint,
                    order: newOrder.join(",")
                  }));
                });
                
              });                            
            });
          });           
        });
      });
    });
  
    describe("set flags", function testSetFlags() {
      
      describe("unmounted", function unmounted() {
        
        it("set flags", function setFlags() {
          var c = new baja.Component();
          
          c.add({
            slot: "test1", 
            value: 12, 
            flags: baja.Flags.READONLY
          });
          
          expect(baja.Flags.encodeToString(c.getFlags("test1"))).toEqual("r");
          
          c.setFlags({
            slot: "test1", 
            flags: baja.Flags.OPERATOR
          });
          
          expect(baja.Flags.encodeToString(c.getFlags("test1"))).toEqual("o");
          
          c.add({
            slot: "test2", 
            value: 23, 
            flags: baja.Flags.OPERATOR
          });
          
          c.setFlags({
            slot: "test2", 
            flags: c.getFlags("test2") | baja.Flags.SUMMARY
          });
          
          expect(baja.Flags.encodeToString(c.getFlags("test2"))).toEqual("so");
          
          c.setFlags(errCallbackify({
            slot: "dumbSlotName", 
            flags: baja.Flags.OPERATOR
          }));
        
          var c2 = new baja.Component();
          c.add({ slot: "kid", value: c2, flags: baja.Flags.SUMMARY | baja.Flags.READONLY});
          expect(c2.getFlags()).toEqual(baja.Flags.SUMMARY | baja.Flags.READONLY);
        });
      });       
   
      describe("mounted", function mounted() {
        
        it("set flags", function setFlags() {
          
          // Add a Property and set the flags on it
          var propName = "setFlagsValue",
              value = "Some text!";
          
          add(propName, value);
          
          runs(function () {
            var flags = baja.Flags.SUMMARY | baja.Flags.TRANSIENT,
                setFlagsCallback = callbackify({
                  slot: propName,
                  flags: flags
                });
            
            // Check the flags on the Server are the default
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                flags: 0,
                value: value
              }));
            });
            
            // Run and wait for this async op to complete
            runAndWait(function () {
              testFolder.setFlags(setFlagsCallback);
            }, setFlagsCallback);
            
            runs(function () {
              expect(testFolder.getFlags(propName)).toEqual(flags);
            });
            
            // Check the flags have been set as expected in the Station
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                flags: flags,
                value: value
              }));
            });
          });          
        });    
        
        it("batch", function testBatch() {
          
          // Add a Property and set the flags on it
          var propName = "setFlagsBatchValue",
              value = baja.$("baja:StatusBoolean", {
                value: true
              });
          
          add(propName, value);
          
          runs(function () {
            var flags = baja.Flags.SUMMARY | baja.Flags.TRANSIENT,
                batch = new baja.comm.Batch(),
                setFlagsCallback = callbackify({
                  slot: propName,
                  flags: flags,
                  batch: batch
                });
            
            // Check the flags on the Server are the default
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                flags: 0,
                value: value
              }));
            });
            
            // Run the setFlags
            runs(function () {
              testFolder.setFlags(setFlagsCallback);
            });
            
            // Ensure the flags on the client are still the same
            runs(function () {
              expect(testFolder.getFlags(propName)).toEqual(0);
            });
            
            // Ensure the flags are still the same on the Station (it should be batched up)
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                flags: 0,
                value: value.newCopy(true)
              }));
            });
            
            runs(function () {
              batch.addCallback(function ok() {
                verify(true);
              },
              function fail(err) {
                verify(false);
              });
            });
            
            // Commit the batch operation
            runAndWait(function () {
              batch.commit();
            }, setFlagsCallback);
                        
            runs(function () {
              expect(testFolder.getFlags(propName)).toEqual(flags);
            });
            
            // Check the flags have been set as expected in the Station
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                flags: flags,
                value: value.newCopy(true)
              }));
            });
          });          
        }); 
        
      });
    });  
  
    describe("set facets", function testSetFacets() {
      
      describe("unmounted", function unmounted() {
        
        it("set facets", function testSetFacets() {
          var c = new baja.Component(),
              facets = baja.Facets.make(["key1", "key2"], [23, "some text"]);
          
          c.add({
            slot: "test", 
            value: 34, 
            flags: baja.Flags.SUMMARY, 
            facets: facets
          });
          
          expect("key1=d:23|key2=s:some$20text").toEqual(c.getFacets("test").encodeToString());
          expect(baja.Facets.DEFAULT.decodeFromString("key1=d:23|key2=s:some$20text").get("key2")).toEqual("some text");
          
          var facets2 = baja.Facets.make(["keya", "keyb"], [10, "ILJ"]);
          
          c.setFacets({
            slot: "test", 
            facets: facets2
          });
          expect("keya=d:10|keyb=s:ILJ").toEqual(c.getFacets("test").encodeToString());
                
          // Try setting facets of Slot that doesn't exist
          c.setFacets(errCallbackify({
            slot: "sdfsdf", 
            facets: facets
          }));

          var c2 = new baja.Component();
          c.add({ slot: "kid", value: c2, facets: facets});
          expect(c2.getFacets().equals(facets)).toBeTruthy();
        });
      });      
    
      describe("mounted", function mounted() {
                  
        it("set facets", function setFacets() {
          
          // Add a Property and set the facets on it
          var propName = "setFacetsValue",
              value = "Some text!";
          
          add(propName, value);
          
          runs(function () {
            var facets = baja.Facets.make(["key1", "key2"], ["some other text", "some text"]),
                setFacetsCallback = callbackify({
                  slot: propName,
                  facets: facets
                });
            
            // Check the facets on the Server are the default
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                facets: baja.Facets.DEFAULT,
                value: value
              }));
            });
            
            // Run and wait for this async op to complete
            runAndWait(function () {
              testFolder.setFacets(setFacetsCallback);
            }, setFacetsCallback);
            
            runs(function () {
              expect(testFolder.getFacets(propName).encodeToString()).toEquivalent(facets.encodeToString());
            });
            
            // Check the facets have been set as expected in the Station
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                facets: facets,
                value: value
              }));
            });
          });          
        });    
        
        it("batch", function testBatch() {
          
          // Add a Property and set the flags on it
          var propName = "setFlagsBatchValusdfe",
              value = baja.$("baja:StatusString", {
                value: "I Love BajaScript!!!"
              });
          
          add(propName, value);
          
          runs(function () {
            var facets = baja.Facets.make(["key1", "key2"], ["some other text", "some text"]),
                batch = new baja.comm.Batch(),
                setFacetsCallback = callbackify({
                  slot: propName,
                  facets: facets,
                  batch: batch
                });
            
            // Check the facets on the Server are the default
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                facets: baja.Facets.DEFAULT,
                value: value
              }));
            });
            
            // Run the setFacets
            runs(function () {
              testFolder.setFacets(setFacetsCallback);
            });
            
            // Ensure the facets on the client are still the same
            runs(function () {
              expect(testFolder.getFacets(propName)).toEquivalent(baja.Facets.DEFAULT);
            });
            
            // Ensure the facets are still the same on the Station (it should be batched up)
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                facets: baja.Facets.DEFAULT,
                value: value.newCopy(true)
              }));
            });
            
            runs(function () {
              batch.addCallback(function ok() {
                verify(true);
              },
              function fail(err) {
                verify(false);
              });
            });
            
            // Commit the batch operation
            runAndWait(function () {
              batch.commit();
            }, setFacetsCallback);
                        
            runs(function () {
              expect(testFolder.getFacets(propName)).toEquivalent(facets);
            });
            
            // Check the facets have been set as expected in the Station
            runs(function () {
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: testFolder.getNavOrd(),
                propName: propName,
                facets: facets,
                value: value.newCopy(true)
              }));
            });
          });          
        });
      });
    });
    
    describe("loadSlots", function testLoadSlots() {
      
      describe("mounted", function mounted() {
      
        it("component", function component() {
       
          // Create the Components we need to resolve on the Station
          runs(function () {
            serverCall("createLoadSlotsComponents");
          });
          
          var syncCallback = callbackify({});
          runAndWait(function () {
            baja.station.sync(syncCallback);
          }, syncCallback);
          
          var comp,
              resolveCallback = callbackify({
            ok: function () {
              comp = this;
            }
          });
          
          runAndWait(function () {
            baja.Ord.make("station:|slot:/BajaScriptTestFolder/testLoadSlots/test/test/test").get(resolveCallback); 
          }, resolveCallback);
         
          var loadSlotsCallback = callbackify({
            ok: function () {
              // The Component should now be loaded, hence we can make these checks
              expect(comp.has("slot1")).toBeTruthy();
              expect(comp.has("slot2")).toBeTruthy(); 
              
              expect(comp.get("slot1")).toEqual("Some text!");
              expect(comp.get("slot2")).toEqual(baja.$("baja:Weekday").get("wednesday"));
            }
          });
          
          runAndWait(function () {
            comp.loadSlots(loadSlotsCallback);
          }, loadSlotsCallback);
        });
      });
    });
    
    describe("invoke", function testInvoke() {
      
      describe("mounted", function mounted() {
        
        it("verify frozen Action Slot", function verifyFrozenActionSlot() { 
          runs(function () {
            var action = baja.$("control:NumericWritable").getSlot("set");
            
            expect(action.getName()).toEqual("set");
            expect(action.isFrozen()).toBeTruthy();
            expect(action.getFlags()).toEqual(baja.Flags.OPERATOR);
            expect(action.getFacets().equals(baja.Facets.DEFAULT)).toBeTruthy();
            expect(action.getDefaultFlags()).toEqual(baja.Flags.OPERATOR);
            expect(action.getReturnType() === null).toBeTruthy();
            expect(action.getParamType().equals(baja.Double.DEFAULT.getType())).toBeTruthy();
            expect(action.getParamDefault()).toEqual(0);
            expect(action.isTopic()).toBeFalsy();
            expect(action.isAction()).toBeTruthy();
            expect(action.isProperty()).toBeFalsy();
          });
        });
        
        it("set on numeric writable (auto-gen)", function numericWritableAutoGen() {
          runs(function () {
            var pointName = "invokeNwAutoGen";
 
            var addCallback = callbackify({
              slot: pointName,
              value: baja.$("control:NumericWritable")
            });
              
            runAndWait(function () {
              testFolder.add(addCallback);
            }, addCallback);
            
            // Get the point we just added and ensure it's leased
            var point,
                valueChanged = false,
                setValue = 234;

            runs(function () {
              point = testFolder.get(pointName);
              lease(point);
              
              // Attach changed handler to wait for the output to change
              point.attach("changed", function (prop, cx) {
                if (prop.getName() === "out") {
                  if (this.get(prop).getValue() === setValue) {
                    valueChanged = true;
                  }
                }
              });  
            });
            
            // Call the 'set' Action on the newly added point
            runs(function () {
              point.set1(callbackify({
                value: setValue
              }));
            });
            
            waitsFor(function () {
              return valueChanged;
            }, "Waiting for point's out Property to change", 10000);
            
            runs(function () {
              // Double check on the Server this value has changed
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: baja.Ord.make(point.getNavOrd().toString() + "/out"), 
                propName: "value",
                value: setValue
              }));
            });
          });
        });
        
        it("one shot (invoke)", function oneShotInvoke() {
          runs(function () {
            var pointName = "invokeOneShotInvoke";
 
            var addCallback = callbackify({
              slot: pointName,
              value: baja.$("kitControl:OneShot", {
                time: baja.RelTime.make({ minutes: 1 })
              })
            });
              
            runAndWait(function () {
              testFolder.add(addCallback);
            }, addCallback);
            
            // Get the point we just added and ensure it's leased
            var point,
                valueChanged = false;

            runs(function () {
              point = testFolder.get(pointName);
              lease(point);
              
              // Attach changed handler to wait for the output to change
              point.attach("changed", function (prop, cx) {
                if (prop.getName() === "out") {
                  valueChanged = this.get(prop).getValue();
                }
              });  
            });
            
            // Call the 'set' Action on the newly added point using 'invoke'
            runs(function () {
              point.invoke(callbackify({
                slot: "fire"
              }));
            });
            
            waitsFor(function () {
              return valueChanged;
            }, "Waiting for point's out Property to change", 10000);
            
            runs(function () {
              // Double check on the Server this value has changed
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: baja.Ord.make(point.getNavOrd().toString() + "/out"), 
                propName: "value",
                value: true
              }));
            });
          });
        });
        
        // TODO: Need to try invoke on a dynamic Action.
               
        it("set on numeric writable (auto-gen, batch)", function numericWritableAutoGenBatch() {
          runs(function () {
            var pointName = "invokeNwAGBatch",
                batch = new baja.comm.Batch();
 
            var addCallback = callbackify({
              slot: pointName,
              value: baja.$("control:NumericWritable")
            });
              
            runAndWait(function () {
              testFolder.add(addCallback);
            }, addCallback);
            
            // Get the point we just added and ensure it's leased
            var point,
                valueChanged = false,
                setValue = 234;

            runs(function () {
              point = testFolder.get(pointName);
              lease(point);
              
              // Attach changed handler to wait for the output to change
              point.attach("changed", function (prop, cx) {
                if (prop.getName() === "out") {
                  if (this.get(prop).getValue() === setValue) {
                    valueChanged = true;
                  }
                }
              });  
            });
            
            // Call the 'set' Action on the newly added point
            runs(function () {
              point.set1(callbackify({
                value: setValue,
                batch: batch
              }));
            });
            
            runs(function () {
              // Make sure the value on the Server hasn't changed!
              verify(!serverCall("checkPropertyExists", baja.$("baja:Component", {
                ord: baja.Ord.make(point.getNavOrd().toString() + "/in4"), 
                propName: "value",
                value: setValue
              })));
            });
            
            runs(function () {
              batch.commit();
            });
            
            waitsFor(function () {
              return valueChanged;
            }, "Waiting for point's out Property to change", 10000);
            
            runs(function () {
              // Double check on the Server this value has changed
              verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
                ord: baja.Ord.make(point.getNavOrd().toString() + "/out"), 
                propName: "value",
                value: setValue
              }));
            });
          });
        });
      
        it("dynamic action", function dynamicAction() {
          var actionName = "invokeDynTestAction";
          
          runs(function () {
            add(actionName, baja.$("testBajaScript:TestAction"));          
          });
          
          runs(function () {
            var action = testFolder.getSlot(actionName);
            
            // Verify dynamic action slot...            
            expect(action.getName()).toEqual(actionName);
            expect(action.isFrozen()).toBeFalsy();
            expect(action.getFlags()).toEqual(0);
            expect(action.getFacets().equals(baja.Facets.DEFAULT)).toBeTruthy();
            expect(action.getDefaultFlags()).toEqual(0);
            expect(action.getReturnType().equals(baja.lt("baja:StatusNumeric"))).toBeTruthy();
            expect(action.getParamType().equals(baja.lt("baja:StatusNumeric"))).toBeTruthy();
            expect(action.getParamDefault()).toEquivalent(baja.$("baja:StatusNumeric", {
              value: 42
            }));
            expect(action.isTopic()).toBeFalsy();
            expect(action.isAction()).toBeTruthy();
            expect(action.isProperty()).toBeTruthy();
          });
          
          runs(function () {
            // Invoke the dynamic Action and make sure the return value has had its value incremented...
            var invokeCallback = callbackify({
              slot: actionName,
              value: baja.$("baja:StatusNumeric", {
                value: 56
              }),
              ok: function (retVal) {
                expect(retVal).toEquivalent(baja.$("baja:StatusNumeric", {
                  value: 57
                }));
              }
            });
            
            runAndWait(function () {
              testFolder.invoke(invokeCallback);
            }, invokeCallback);
          });
        });
      
        it("NCCB-372 - test box Contract infinite recursion", function () {
          runs(function () {
            var cb = callbackify({
              value: "test me",
              ok: function (folder) {
                expect(folder.getType().is("testBajaScript:TestFolder")).toBeTruthy();
              }
            });
          
            runAndWait(function () {
              // Test BOX infinite recursion bug by invoking special Action on TestFolder
              testFolder.testTypeScan(cb);
            }, cb);
          });
        }); 
      });      
    });
    
    describe("getActionParameterDefault", function testGetActionParameterDefault() {
      
      describe("mounted", function mounted() {
        
        it("Action Default of a point", function defaultFromPoint() {
          
          var pointName = "getActionParamDefNw",
              point,
              value = 43.43;
          
          runs(function () {
            add(pointName, baja.$("control:NumericWritable"));
          });
          
          runs(function () {
            point = lease(testFolder.get(pointName));
            
            var setCallback = callbackify({
              value: value
            });
            
            // Call the set Action on the point
            runAndWait(function () {
              point.set1(setCallback);
            }, setCallback);
            
            var actParamCallback = callbackify({
              slot: "set",
              ok: function (defValue) {
                expect(defValue).toEqual(value);
              }
            });
            
            // Ensure the default value for the Action parameter is the same as the value we just set
            runAndWait(function () {
              point.getActionParameterDefault(actParamCallback);
            }, actParamCallback);
          });          
        });
                
        // TODO: Batch test?
      });
      
      describe("unmounted", function unmounted() {
        
        it("Action Default of a point", function defaultFromPoint() {
          
          runs(function () {
            var point = baja.$("control:NumericWritable"), 
                value = 0;
            
            var actParamCallback = callbackify({
              slot: "set",
              ok: function (defValue) {
                expect(defValue).toEqual(value);
              }
            });
            
            // Ensure the default value for the Action parameter is the same as the value we just set
            runAndWait(function () {
              point.getActionParameterDefault(actParamCallback);
            }, actParamCallback);
          });
        });        
      
        it("error", function () {
          runs(function () {
            var point = baja.$("control:NumericWritable"), 
                value = 0;
            
            var actParamCallback = errCallbackify({
              slot: "in4"
            });
            
            // Ensure the default value for the Action parameter is the same as the value we just set
            runAndWait(function () {
              point.getActionParameterDefault(actParamCallback);
            }, actParamCallback);
          });
          
          runs(function () {
            var point = baja.$("control:NumericWritable"), 
                value = 0;
            
            var actParamCallback = errCallbackify({
              slot: "qweqwe"
            });
            
            // Ensure the default value for the Action parameter is the same as the value we just set
            runAndWait(function () {
              point.getActionParameterDefault(actParamCallback);
            }, actParamCallback);
          });
        });
      });      
    });
    
    describe("fire", function testFire() {
      
      describe("mounted", function mounted() {
        
        // TODO: Need to add Slot verification checks to all different Slot Types and not just Topics...
        
        it("verify frozen Topic Slot", function verifyFrozenTopicSlot() {
          var frozenTopic = testFolder.getSlot("testTopic");
          
          expect(frozenTopic.getName()).toEqual("testTopic");
          expect(frozenTopic.isFrozen()).toBeTruthy();
          expect(frozenTopic.getFlags()).toEqual(baja.Flags.SUMMARY);
          expect(frozenTopic.getFacets().equals(baja.Facets.make(["test"], ["this"]))).toBeTruthy();
          expect(frozenTopic.getDefaultFlags()).toEqual(baja.Flags.SUMMARY);
          expect(frozenTopic.getEventType().equals(baja.Double.DEFAULT.getType())).toBeTruthy();
          expect(frozenTopic.isTopic()).toBeTruthy();
          expect(frozenTopic.isAction()).toBeFalsy();
          expect(frozenTopic.isProperty()).toBeFalsy();
        });
        
        it("fire on test topic (auto-gen)", function fireTestTopicAutoGen() {
          var count = 20,
              firedCount = 0;
                    
          runs(function() {
            testFolder.attach("topicFired", function (topic, event, cx) {
              if (topic.getName() === "testTopic") {
                expect(event.getType().is("baja:Double")).toBeTruthy();
                firedCount++;
              }
            });
          });
          
          runs(function () {
            // Fire this topic X number of times
            baja.iterate(count, function (i) {
              testFolder.fireTestTopic(i);
            });
          });
          
          waitsFor(function() {
            // Wait for all of the topicFired events to be received
            return firedCount === count;
          }, "topic to fire", 10000);                    
        });
        
        it("fire on test topic (fire)", function fireTestTopic() {
          var count = 20,
              firedCount = 0;
                    
          runs(function() {
            testFolder.attach("topicFired", function (topic, event, cx) {
              if (topic.getName() === "testTopic") {
                expect(event.getType().is("baja:Double")).toBeTruthy();
                firedCount++;
              }
            });
          });
          
          runs(function () {
            // Fire this topic X number of times
            baja.iterate(count, function (i) {
              testFolder.fire({slot: "testTopic", value: i});
            });
          });
          
          waitsFor(function() {
            // Wait for all of the topicFired events to be received
            return firedCount === count;
          }, "topic to fire", 10000);      
          
          runs(function () {
            testFolder.detach("topicFired");
          });
        });
      
        it("fire on test topic (batch)", function fireTestTopicBatch() {
          var count = 20,
              firedCount = 0,
              batch = new baja.comm.Batch();
                    
          runs(function() {
            testFolder.attach("topicFired", function (topic, event, cx) {
              if (topic.getName() === "testTopic") {
                expect(event.getType().is("baja:Double")).toBeTruthy();
                firedCount++;
              }
            });
          });
          
          runs(function () {
            // Fire this topic X number of times
            baja.iterate(count, function (i) {
              testFolder.fireTestTopic({value: i, batch: batch});
            });
            
            batch.commit();
          });
          
          waitsFor(function() {
            // Wait for all of the topicFired events to be received
            return firedCount === count;
          }, "topic to fire (batch)", 10000); 
          
          runs(function () {
            testFolder.detach("topicFired");
          });
        });
      
        it("fire dynamic test topic", function fireDynamicTestTopic() {  
          var topicName = "fireDynamicTestTopic",
              fired = false;
          
          runs(function () {
            // Add a dynamic Property that's also a Topic that we can fire
            add(topicName, baja.$("testBajaScript:TestTopic"));
          });
          
          runs(function () { 
            var dynTopic = testFolder.getSlot(topicName),
                value = baja.$("baja:Weekday", "thursday").asDynamic();
            
            expect(dynTopic.getName()).toEqual(topicName);
            expect(dynTopic.isFrozen()).toBeFalsy();
            expect(dynTopic.getFlags()).toEqual(0);
            expect(dynTopic.getFacets().equals(baja.Facets.DEFAULT)).toBeTruthy();
            expect(dynTopic.getDefaultFlags()).toEqual(0);
            expect(dynTopic.getEventType().equals(baja.lt("baja:StatusEnum"))).toBeTruthy();
            expect(dynTopic.isTopic()).toBeTruthy();
            expect(dynTopic.isAction()).toBeFalsy();
            expect(dynTopic.isProperty()).toBeTruthy();
            
            testFolder.attach("topicFired", function(topic, event, cx) {
              if (topic.getName() === topicName) {
                expect(event.getValue().equals(value)).toBeTruthy();
                fired = true;
              }
            });
            
            // Fire a Topic
            testFolder.fire({
              slot: topicName, 
              value: baja.$("baja:StatusEnum", {
                value: value
              }) 
            });
          }); 
          
          waitsFor(function () {
            return fired;
          }, "Waiting for dynamic Topic to fire", 10000);
          
          runs(function () {
            testFolder.detach("topicFired");
          });
        });        
      });
    });
        
    describe("links", function testLinks() {
    
      // Please note: links are currently only implemented for mounted online Components.
      // Also note that BajaScript currently only has support for creating indirect links.
      
      describe("mounted", function mounted() {
        
        it("link two points (standard BLink)", function makeLinkStd() {
          
          var point1Name = "makeLinksNw1",
              point2Name = "makeLinksNw2",
              point1,
              point2,
              link;
          
          runs(function () {
            add(point1Name, baja.$("control:NumericWritable"));
            add(point2Name, baja.$("control:NumericWritable"));
          });
          
          runs(function() {
            point1 = lease(testFolder.get(point1Name));
            point2 = lease(testFolder.get(point2Name));
            
            var makeLinkCallback = callbackify({
              source: point1,
              sourceSlot: "out",
              targetSlot: "in4",
              ok: function(lk) {
                link = lk;
              }
            });
            
            runAndWait(function () {
              // Create a BLink instance (this doesn't add the link but just creates an instance)
              point2.makeLink(makeLinkCallback);
            }, makeLinkCallback);
          });
          
          runs(function () {
            expect(link.getType().equals(baja.lt("baja:Link"))).toBeTruthy();
          });
          
          runs(function () {
            var addCallback = callbackify({
              slot: "link",
              value: link
            });
            
            runAndWait(function () {
              // Add the link (hence creating an indirect link between the two Components)
              point2.add(addCallback);
            }, addCallback);                       
          });
          
          runs(function () {           
            // Make sure the link is added in the Server
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: point2.getNavOrd(),
              propName: "link",
              typeSpec: "baja:Link"
            }));
          });
          
          runs(function () {
            expect(point2.has("link")).toBeTruthy();
            
            var link = point2.get("link");
            expect(link.getType().equals(baja.lt("baja:Link"))).toBeTruthy();
                        
            expect(link.getEnabled()).toBeTruthy();
            expect(link.getSourceSlotName()).toEqual("out");
            expect(link.getTargetSlotName()).toEqual("in4");
            
            // Resolve the Source ORD and ensure we get back the same Source Component instance!
            var resolveSourceCallback = callbackify({
              base: baja.station,
              ok: function () {
                expect(this).toEqual(point1);
              }
            });
            
            runAndWait(function () {
              baja.Ord.make(link.getSourceOrd()).get(resolveSourceCallback);
            }, resolveSourceCallback);
            
            // Test getLinks (no arg)
            var links = point2.getLinks();
            expect(links.length).toEqual(1);
            expect(links[0]).toEqual(link);
            
            // Test getLinks (with slot name arg)
            links = point2.getLinks("in4");
            expect(links.length).toEqual(1);
            expect(links[0]).toEqual(link);
            expect(point2.getKnobs().length).toEqual(0);
            expect(point2.getKnobCount()).toEqual(0);
            
            // Expect an error to be thrown if Slot doesn't exist
            expect(function () {
              point2.getLinks("qweqwe");
            }).toThrow();
                        
            var knobs = point1.getKnobs(),
                knob = knobs[0];
            
            expect(knobs.length).toEqual(1);
            expect(point1.getKnobCount()).toEqual(1);
            expect(knob.getSourceSlotName()).toEqual("out");
            expect(knob.getTargetSlotName()).toEqual("in4");
                        
            // Resolve the Source ORD and ensure we get back the same Source Component instance!
            var resolveTargetCallback = callbackify({
              base: baja.station,
              ok: function () {
                expect(this).toEqual(point2);
              }
            });
            
            runAndWait(function () {
              baja.Ord.make(knob.getTargetOrd()).get(resolveTargetCallback);
            }, resolveTargetCallback);
            
            knobs = point1.getKnobs("out");
            expect(knobs.length).toEqual(1);
            expect(knobs[0]).toEqual(knob);
            
            // Expect the error to be throw in the Slot doesn't exist
            expect(function () {
              point1.getKnobs("qweqwe");
            }).toThrow();
          });
          
          // Make a change and ensure the link value propogates
          runs(function () {
            var setCallback = callbackify({
              value: 32
            });
            
            // Flag to indicate the value propogated via the link and update the other control point
            var valuePropogated = false;
            
            point2.attach("changed", function (prop, cx) {
              if (prop.getName() === "out") {
                if (this.getOut().getValue() === setCallback.value) {
                  valuePropogated = true;
                }
              }
            });
           
            // Make the 'set' Action invocation
            runAndWait(function () {
              point1.set1(setCallback);
            }, setCallback);
           
            // Wait for the second point's out slot to update
            waitsFor(function () {
              return valuePropogated;
            }, "waiting for value to propogate", 10000);
          });
        });
      
        it("link two points (BConversionLink)", function makeLinkConv() {
          
          var point1Name = "makeLinksConvNw1",
              point2Name = "makeLinksConvBw2",
              point1,
              point2,
              link;
          
          runs(function () {
            add(point1Name, baja.$("control:NumericWritable"));
            add(point2Name, baja.$("control:BooleanWritable"));
          });
          
          runs(function() {
            point1 = lease(testFolder.get(point1Name));
            point2 = lease(testFolder.get(point2Name));
            
            var makeLinkCallback = callbackify({
              source: point1,
              sourceSlot: "out",
              targetSlot: "in4",
              ok: function(lk) {
                link = lk;
              }
            });
            
            runAndWait(function () {
              // Create a BLink instance (this doesn't add the link but just creates an instance)
              point2.makeLink(makeLinkCallback);
            }, makeLinkCallback);
          });
          
          runs(function () {
            expect(link.getType().equals(baja.lt("baja:ConversionLink"))).toBeTruthy();
          });
          
          runs(function () {
            var addCallback = callbackify({
              slot: "link",
              value: link
            });
            
            runAndWait(function () {
              // Add the link (hence creating an indirect link between the two Components)
              point2.add(addCallback);
            }, addCallback);                       
          });
          
          runs(function () {           
            // Make sure the link is added in the Server
            verifyServerCall("checkPropertyExists", baja.$("baja:Component", {
              ord: point2.getNavOrd(),
              propName: "link",
              typeSpec: "baja:ConversionLink"
            }));
          });
          
          runs(function () {
            expect(point2.has("link")).toBeTruthy();
            
            var link = point2.get("link");
            expect(link.getType().equals(baja.lt("baja:ConversionLink"))).toBeTruthy();
                        
            expect(link.getEnabled()).toBeTruthy();
            expect(link.getSourceSlotName()).toEqual("out");
            expect(link.getTargetSlotName()).toEqual("in4");
            
            // Some extra checks for the ConversionLink...
            expect(link.getConverter().getType().is("converters:StatusNumericToStatusBoolean")).toBeTruthy();
            
            // Resolve the Source ORD and ensure we get back the same Source Component instance!
            var resolveSourceCallback = callbackify({
              base: baja.station,
              ok: function () {
                expect(this).toEqual(point1);
              }
            });
            
            runAndWait(function () {
              baja.Ord.make(link.getSourceOrd()).get(resolveSourceCallback);
            }, resolveSourceCallback);
            
            // Test getLinks (no arg)
            var links = point2.getLinks();
            expect(links.length).toEqual(1);
            expect(links[0]).toEqual(link);
            
            // Test getLinks (with slot name arg)
            links = point2.getLinks("in4");
            expect(links.length).toEqual(1);
            expect(links[0]).toEqual(link);
            expect(point2.getKnobs().length).toEqual(0);
            expect(point2.getKnobCount()).toEqual(0);
            
            // Expect an error to be thrown if Slot doesn't exist
            expect(function () {
              point2.getLinks("qweqwe");
            }).toThrow();
                        
            var knobs = point1.getKnobs(),
                knob = knobs[0];
            
            expect(knobs.length).toEqual(1);
            expect(point1.getKnobCount()).toEqual(1);
            expect(knob.getSourceSlotName()).toEqual("out");
            expect(knob.getTargetSlotName()).toEqual("in4");
                        
            // Resolve the Source ORD and ensure we get back the same Source Component instance!
            var resolveTargetCallback = callbackify({
              base: baja.station,
              ok: function () {
                expect(this).toEqual(point2);
              }
            });
            
            runAndWait(function () {
              baja.Ord.make(knob.getTargetOrd()).get(resolveTargetCallback);
            }, resolveTargetCallback);
            
            knobs = point1.getKnobs("out");
            expect(knobs.length).toEqual(1);
            expect(knobs[0]).toEqual(knob);
            
            // Expect the error to be throw in the Slot doesn't exist
            expect(function () {
              point1.getKnobs("qweqwe");
            }).toThrow();
          });
          
          // Make a change and ensure the link value propogates
          runs(function () {
            var setCallback = callbackify({
              value: 1
            });
            
            // Flag to indicate the value propogated via the link and update the other control point
            var valuePropogated = false;
            
            point2.attach("changed", function (prop, cx) {
              if (prop.getName() === "out") {
                if (this.getOut().getValue()) {
                  valuePropogated = true;
                }
              }
            });
           
            // Make the 'set' Action invocation
            runAndWait(function () {
              point1.set1(setCallback);
            }, setCallback);
           
            // Wait for the second point's out slot to update
            waitsFor(function () {
              return valuePropogated;
            }, "waiting for value to propogate", 10000);
          });
        });
      });      
    });
    
    describe("contracts", function testContracts() {
                
      it("auto generated property methods", function testPropMethodGenerate() {  
        
        runs(function () {
          var sn = baja.$("baja:StatusNumeric");    
              
          expect(typeof sn.getValue === "function").toBeTruthy();    
          expect(typeof sn.setValue === "function").toBeTruthy();
          expect(typeof sn.getValueDisplay === "function").toBeTruthy(); 
          
          expect(typeof sn.getStatus === "function").toBeTruthy();    
          expect(typeof sn.setStatus === "function").toBeTruthy(); 
          expect(typeof sn.getStatusDisplay === "function").toBeTruthy(); 
          
          sn.setValue(12.3);
          expect(sn.get("value")).toEqual(12.3); 
          
          sn.set({
            slot: "value",
            value: 120
          });   
          
          expect(sn.getValue()).toEqual(120);
        });
      });
      
      it("simple frozen prop access", function testFrozenPropAccess() {
      
        runs(function () {
          var sn = baja.$("baja:StatusNumeric");    
              
          verifyEq(sn.getSlots().frozen().properties().getSize(), 2);
          verifyEq(sn.getSlots().dynamic().properties().getSize(), 0);
        });
      });
      
      it("status numeric contract", function testStatusNumericContract() {  
        var sn = baja.$("baja:StatusNumeric"),
           i = 0,
           c = sn.getSlots(),
           s;
        
        while (c.next()) {
          s = c.get();
          
          if (i === 0) {
            verifyEq(s.getName(), "status");
          }
          else if (i === 1) {
            verifyEq(s.getName(), "value");
          }
          
          i++;
        }
        expect(i).toEqual(2);
        
        sn.setValue(34);
        expect(sn.getValue().valueOf()).toEqual(34);
            
        // Test equivalent on frozen Slots
        expect(sn).toEquivalent(sn);
        
        var sn2 = baja.$("baja:StatusNumeric");
        expect(sn).not.toEquivalent(sn2);
        expect(sn2).not.toEquivalent(sn);
        expect(sn).toEqual(sn);
        expect(sn2).toEqual(sn2);
        expect(sn).not.toEqual(sn2);
        expect(sn2).not.toEqual(sn);
        
        sn2.setValue(34);
        expect(sn).toEquivalent(sn2);
        expect(sn2).toEquivalent(sn);
        expect(sn).toEqual(sn);
        expect(sn2).toEqual(sn2);
        expect(sn).not.toEqual(sn2);
        expect(sn2).not.toEqual(sn);
        
        sn2.setValue(35);
        expect(sn).not.toEquivalent(sn2);
        expect(sn2).not.toEquivalent(sn);
        expect(sn).toEqual(sn);
        expect(sn2).toEqual(sn2);
        expect(sn).not.toEqual(sn2);
        expect(sn2).not.toEqual(sn);

        sn2.setValue(34);
        expect(sn).toEquivalent(sn2);
        expect(sn2).toEquivalent(sn);
        expect(sn).toEqual(sn);
        expect(sn2).toEqual(sn2);
        expect(sn).not.toEqual(sn2);
        expect(sn2).not.toEqual(sn);
        
        sn2.setStatus(baja.Status.down);
        expect(sn).not.toEquivalent(sn2);
        expect(sn2).not.toEquivalent(sn);
        expect(sn).toEqual(sn);
        expect(sn2).toEqual(sn2);
        expect(sn).not.toEqual(sn2);
        expect(sn2).not.toEqual(sn);
      });
    });
    
    describe("toPathString", function testToPathString() {
      
      describe("unmounted", function unmounted() {

        it("dynamic", function dynamic() {
          
          var c = new baja.Component();
          c.$handle = "1";
          
          var c1 = new baja.Component();
          c1.$handle = "2";
          
          var c2 = new baja.Component();
          c2.$handle = "3";
          
          var c3 = new baja.Component();
          c3.$handle = "4";
          
          var c4 = new baja.Component();
          c4.$handle = "5";
          
          c.add({
            slot: "first", 
            value: c1
          });
          
          c1.add({
            slot: "second", 
            value: c2
          });
          
          c2.add({
            slot: "third", 
            value: c3
          });
          
          c3.add({
            slot: "fourth", 
            value: c4
          });
          
          // Mount Components inside a Component Space
          var cs = new baja.ComponentSpace();
          cs.$fw("mount", c);
          
          expect(c.toPathString()).toEqual("/");
          expect(c1.toPathString()).toEqual("/first");
          expect(c2.toPathString()).toEqual("/first/second");
          expect(c3.toPathString()).toEqual("/first/second/third");
          expect(c4.toPathString()).toEqual("/first/second/third/fourth");

          var slotPath = c4.getSlotPath();
          expect(slotPath.isAbsolute()).toBeTruthy();
          var i;
          for (i = 0; i < slotPath.depth(); ++i) {
            if (i === 0) {
              verifyEq(slotPath.nameAt(i), "first");
            }
            else if (i === 1) {
              verifyEq(slotPath.nameAt(i), "second");
            }
            else if (i === 2) {
              verifyEq(slotPath.nameAt(i), "third");
            }
            else if (i === 3) {
              verifyEq(slotPath.nameAt(i), "fourth");
            }
            else if (i === 4) {
              verify(false);
            }
          }
          expect(i).toEqual(4);
          
          c2.rename({ 
            slot: "third", 
            newName: "fifth"
          });
          
          expect(c.toPathString()).toEqual("/");
          expect(c1.toPathString()).toEqual("/first");
          expect(c2.toPathString()).toEqual("/first/second");
          expect(c3.toPathString()).toEqual("/first/second/fifth");
          expect(c4.toPathString()).toEqual("/first/second/fifth/fourth");
        });        
      });      
    
      describe("mounted", function mounted() {
        
        it("toPathString for point", function toPathStringStrWtr() {            
          var pointName = "toPathStringStrWtr";
          
          runs(function () {
            add(pointName, baja.$("control:StringWritable"));
          });
          
          runs(function () { 
            expect(testFolder.get(pointName).toPathString()).toEqual("/BajaScriptTestFolder/" + pointName);          
          });
        });               
      });
    });
    
    describe("getSlotPath", function testGetSlotPath() {
      
      describe("mounted", function mounted() {
        
        it("getSlotPath of point", function slotPathOfPoint() {
                 
          var pointName = "getSlotPathStrWtr";
          
          runs(function () {
            add(pointName, baja.$("control:StringWritable"));
          });
          
          runs(function () { 
            expect(testFolder.get(pointName).getSlotPath().toString()).toEqual("slot:/BajaScriptTestFolder/" + pointName);          
          });
        });       
      });      
    });
    
    describe("BINavNode", function testNav() {
      
      describe("mounted", function mounted() {
      
        it("Nav methods of a point", function nav() {
          
          var pointName = "navStrWtr",
              point;
          
          runs(function () {
            add(pointName, baja.$("control:StringPoint"));
          });
          
          runs(function () { 
            point = lease(testFolder.get(pointName));
             
            expect(point.getName()).toEqual(pointName);
            expect(point.getDisplayName()).toEqual(pointName);
            expect(point.getNavName()).toEqual(pointName);
            expect(point.getNavDisplayName()).toEqual(pointName);
            
            expect(point.getNavOrd().toString()).toEqual("local:|station:|slot:/BajaScriptTestFolder/" + pointName);
            expect(point.getNavParent().getNavOrd().toString()).toEqual("local:|station:|slot:/BajaScriptTestFolder");
          });

          // TODO: Batch test for getNavChildren
          
          runs(function () {       
            var foundPoint = false,
                foundUserService = false, // We add the UserService into the Nav Children of testBajaScript:TestFolder
                getNavChildrenCallback = callbackify({
              ok: function (kids) {
                expect(kids instanceof Array).toBeTruthy();
                baja.iterate(kids, function (k) {
                  // Just find the Component we just added to show that we get something back
                  if (k.getType().isComponent()) {
                    if (k.getName() === pointName) {
                      foundPoint = true;
                    }
                    else if (k.getType().is("baja:UserService")) {
                      foundUserService = true;
                    }
                  }
                });
              }
            });
            
            runAndWait(function () {
              testFolder.getNavChildren(getNavChildrenCallback);
            }, getNavChildrenCallback);
            
            waitsFor(function () {
              return foundPoint && foundUserService;
            }, "Waiting to find correct children from getNavChildren", 10000);
          });
        });        
      });  
    });
                    
    it("isChildLegal", function testIsChildLegal() {
      var c = baja.$("baja:Component");
      c.isChildLegal = function (child) {
        return !child.getType().is("baja:Folder");
      };
      c.add({
        value: baja.$("baja:Component")
      }); 
      
      // Error test
      c.add(errCallbackify({
        value: baja.$("baja:Folder")
      }));
    });
    
    it("isParentLegal", function testIsParentLegal() {
      var c = baja.$("baja:Component");
      c.isParentLegal = function (p) {
        return !p.getType().is("baja:Folder");
      };  
      var folder = baja.$("baja:Folder");  
      folder.add({
        value: baja.$("baja:Component")
      }); 
      
      // Error test
      folder.add(errCallbackify({
        value: c
      }));
    });
                  
    it("flags", function testFlags() {
      var f = baja.Flags;
      
      // Simple tests
      var flags = f.ASYNC | f.SUMMARY | f.READONLY;
      expect(f.encodeToString(flags)).toEqual("rsa");
      expect(flags).toEqual(f.decodeFromString("rsa"));
      expect(f.encodeToString(0)).toEqual("");
      
      // Uber test
      flags = f.READONLY | f.TRANSIENT | f.HIDDEN | f.SUMMARY | f.ASYNC | f.NO_RUN | f.DEFAULT_ON_CLONE |
                  f.CONFIRM_REQUIRED | f.OPERATOR | f.EXECUTE_ON_CHANGE | f.FAN_IN | f.NO_AUDIT | f.COMPOSITE |
                  f.REMOVE_ON_CLONE | f.METADATA | f.LINK_TARGET | f.NON_CRITICAL | f.USER_DEFINED_1 
                  | f.USER_DEFINED_2 | f.USER_DEFINED_3 | f.USER_DEFINED_4;
                  
      expect(f.encodeToString(flags)).toEqual("rthsandcoxfApRmNL1234");
      expect(flags).toEqual(f.decodeFromString("rthsandcoxfApRmNL1234"));
    }); 
      
    describe("equivalent", function testEquivalent () {
    
      it("component", function component() {
        var c = baja.$("baja:Component");
        var c2 = baja.$("baja:Component");
        
        verify(c.equivalent(c));    
        verify(c.equivalent(c2));
        verify(c2.equivalent(c));
        verify(!c.equivalent(baja.$("baja:Vector")));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        c2.add({
          slot: "test",
          value: "some text"
        });
        
        verify(!c.equivalent(c2));
        verify(!c2.equivalent(c));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        c.add({
          slot: "test", 
          value: "some text"
        });
        
        verify(c.equivalent(c2));
        verify(c2.equivalent(c));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        c.remove("test");
        verify(!c.equivalent(c2));
        verify(!c2.equivalent(c));
        c.add({
          slot: "test",
          value: "some text"
        });
        
        function g(comp) {
          comp.add({
            slot: "foo", 
            value: 23
          });
          
          var comp2 = baja.$("baja:Component");
          comp2.add({
            slot: "date",
            value: baja.$("baja:AbsTime")
          });
          
          comp2.add({
            slot: "bool", 
            value: true
          });
          
          comp2.add({
            slot: "num", 
            value: 132.54245
          });
          
          comp2.add({
            slot: "vector", 
            value: baja.$("baja:Vector")
          });
          
          comp2.add({
            slot: "float", 
            value: baja.Float.make(1542.1)
          });
          
          comp.add({
            slot: "comp", 
            value: comp2
          });
          
          return comp;
        }
        
        verify(!c.equivalent(g(c2)));
        verify(!c2.equivalent(c));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        verify(g(c).equivalent(c2));
        verify(c.equivalent(c2));
        
        c2.get("comp").reorder({
          dynamicProperties: ["date", "bool", "num", "float", "vector"]
        });
        verify(!c.equivalent(c2));
        verify(!c2.equivalent(c));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        c2.get("comp").reorder(["date", "bool", "num", "vector", "float"]);
        verify(c.equivalent(c2));
        verify(c2.equivalent(c));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        c2.remove("comp");
        verify(!c.equivalent(c2));
        verify(!c2.equivalent(c));
        
        verify(c.equals(c));
        verify(c2.equals(c2));
        verify(!c.equals(c2));
        verify(!c2.equals(c));
        
        verify(!c.equivalent(null));
        verify(!c.equivalent(undefined));
        verify(!c.equivalent(""));
        verify(!c.equivalent(123));
        verify(!c.equivalent(true));
      });  
      
      it("numeric writable", function numericWritable() {        
        runs(function () {
          function create() {
            var nw = baja.$("control:NumericWritable", {
              in4: baja.$("baja:StatusNumeric", {
                value: 12.34
              }),
              in5: baja.$("baja:StatusNumeric", {
                value: 54.79 
              })
            }); 
            
            nw.add({slot: "test", value: "some test"});
            return nw;
          }
          
          var nw1 = create(),
              nw2 = create();
          
          expect(nw1).toEquivalent(nw2);
          expect(nw2).toEquivalent(nw1);
          expect(nw1).toEquivalent(nw1);
          expect(nw2).toEquivalent(nw2);
          
          // Make a change and ensure they're not equivalent
          nw2.add({ slot: "test2", value: 23.45});
          expect(nw1).not.toEquivalent(nw2);
          expect(nw2).not.toEquivalent(nw1);
          
          // Make a change and ensure they're both now equivalent again          
          nw1.add({ slot: "test2", value: 23.45});
          expect(nw1).toEquivalent(nw2);
          expect(nw2).toEquivalent(nw1);
        });        
      });
      
      it("status boolean", function statusBoolean() {
        runs(function () {
          var sb1 = baja.$("baja:StatusBoolean", {
            value: true
          });
          var sb2 = baja.$("baja:StatusBoolean", {
            value: false
          });
          
          expect(sb1).toEquivalent(sb1);
          expect(sb2).toEquivalent(sb2);
          
          expect(sb1).not.toEquivalent(sb2);
          expect(sb2).not.toEquivalent(sb1);
          
          // Make the change so they're now equivalent
          sb2.setValue(true);
          
          expect(sb1).toEquivalent(sb2);
          expect(sb2).toEquivalent(sb1);          
        });        
      });
    });
    
    it("newCopy", function testNewCopy() {
      var c = baja.$("baja:Component");
      c.add({
        slot: "test1", 
        value: ""
      });
      
      c.add({
        slot: "test2", 
        value: 12
      });
      
      c.add({
        slot: "test3", 
        value: true
      });
      
      c.add({
        slot: "test4", 
        value: baja.$("baja:AbsTime")
      });
      
      c.add({
        slot: "test5", 
        value: baja.$("baja:Component")
      });
      
      var copy = c.newCopy();
      verify(c !== copy);
      verify(c.equivalent(copy));
      
      c.set({
        slot: "test2",
        value: 15.2
      });
      
      copy = c.newCopy();
      verify(c !== copy);
      verify(c.equivalent(copy));
      
      // Dynamic Property should be removed when Component is cloned
      c.setFlags({
        slot: "test2",
        flags: baja.Flags.REMOVE_ON_CLONE
      });
      
      copy = c.newCopy();
      verify(copy.getSlot("test2") === null);
      verify(c !== copy);
      verify(!c.equivalent(copy));
      
      var sn = baja.$("baja:StatusNumeric");
      
      verify(sn.newCopy().equivalent(sn));
      sn.setValue(12.65);
      verify(sn.newCopy().equivalent(sn));
      
      // Total hack just for testing
      sn.getSlot("value").$setFlags(baja.Flags.DEFAULT_ON_CLONE);
      
      var sn2 = sn.newCopy();
      verify(!sn.equivalent(sn2));
      verify(!sn2.equivalent(sn));
      
      sn2 = sn.newCopy(true);
      verify(sn.equivalent(sn2));
      verify(sn2.equivalent(sn));
    });
    
    describe("permissions", function testPermissions() {
      it("unmounted", function testUnmounted() {
        runs(function () {
          var c = baja.$("control:NumericWritable"),
              perms = c.getPermissions();
          
          expect(perms).toEqual(baja.Permissions.all);
        });
      });
      
      it("mounted", function testMounted() {
        runs(function () {
          var ordCb = callbackify({
            ok: function () {
              expect(this.getPermissions().hasOperatorRead()).toBeTruthy();
            },
            lease: true
          });
          
          runAndWait(function () {
            baja.Ord.make("station:|service:box:BoxServlet").get(ordCb);
          }, ordCb);
        });
      });      
    });
    
    describe("attach", function testAttach() {
    
      describe("unmounted", function unmounted() {
              
        function makeSubComp() {
          var c = new baja.Component();
          // Hack the Component into subscription so the events work
          c.isSubscribed = function () {
            return true;
          };
          return c;
        }
        
        it("added", function testAttach() {
          var c = makeSubComp();
          var done = false;
          c.attach("added", function (prop, cx) {
            verifyEq(prop.getName(), "test");
            verifyEq(this.get(prop), "hi");
            done = true;
          });
          c.add({ slot: "test", value: "hi" });
          verify(done);
        });
        
        it("removed", function testRemoved() {
          var c = makeSubComp();
          c.add({ slot: "test", value: "hi" });
          var done = false;
          c.attach("removed", function (prop, val, cx) {
            verifyEq(prop.getName(), "test");
            verifyEq(val, "hi");
            done = true;
          });
          c.remove("test");
          verify(done);
        });
        
        it("changed", function testChanged() {
          var c = makeSubComp();
          c.add({ slot: "test", value: "hi" });
          var done = false;
          c.attach("changed", function (prop, cx) {
            verifyEq(prop.getName(), "test");
            verifyEq(this.get(prop), "bye");
            done = true;
          });
          c.set({
            slot: "test",
            value: "bye"
          });
          verify(done);
        });
        
        it("renamed", function testRenamed() {
          var c = makeSubComp();
          c.add({ slot: "test", value: "hi" });
          var done = false;
          c.attach("renamed", function (prop, oldName, cx) {
            verifyEq(prop.getName(), "best");
            verifyEq(oldName, "test");
            done = true;
          });
          c.rename({slot: "test", newName: "best"});
          verify(done);
        });
        
        it("reordered", function testReordered() {
          var c = makeSubComp();
          c.add({ slot: "test", value: "hi" });
          c.add({ slot: "test1", value: "hi2" });
          
          var done = false;
          c.attach("reordered", function (cx) {
            done = true;
          });
          c.reorder(["test1", "test"]);
          verify(done);
        });
        
        it("flags changed", function testFlagsChanged() {
          var c = makeSubComp();
          c.add({ slot: "test", value: "hi", flags: baja.Flags.SUMMARY | baja.Flags.READONLY });
          
          var done = false;
          c.attach("flagsChanged", function (slot, cx) {
            verifyEq(slot.getName(), "test");
            verifyEq(this.getFlags(slot), baja.Flags.SUMMARY);
            verifyEq(slot.getFlags(), baja.Flags.SUMMARY);
            done = true;
          });
          c.setFlags({slot: "test", flags: baja.Flags.SUMMARY});
          verify(done);
        });
        
        it("facets changed", function testFacetsChanged() {
          var c = makeSubComp();
          c.add({ 
            slot: "test", 
            value: "hi", 
            facets: baja.Facets.make(["key1", "key2"], ["value1", "value2"]) 
          });
          
          var newFacets = baja.Facets.make(["foo"], ["bar"]);
          
          var done = false;
          c.attach("facetsChanged", function (slot, cx) {
            verifyEq(slot.getName(), "test");
            verify(this.getFacets(slot).equals(newFacets));
            verify(slot.getFacets().equals(newFacets));
            done = true;
          });
          c.setFacets({slot: "test", facets: newFacets});
          verify(done);
        });
          
        it("component renamed", function testComponentRenamed() {
          var c = makeSubComp();
          var c2 = makeSubComp();
              
          c.add({ slot: "test", value: c2 });
          var done = false;
          
          c2.attach("componentRenamed", function (oldName, cx) {
            verifyEq(this.getName(), "best");
            verifyEq(oldName, "test");
            done = true;
          });
          
          c.rename({slot: "test", newName: "best"});
          verify(done);
        });
        
        it("component flags changed", function testComponentFlagsChanged() {
          var c = makeSubComp();
          var c2 = makeSubComp();
          c.add({ slot: "test", value: c2, flags: baja.Flags.SUMMARY | baja.Flags.READONLY });
          
          var done = false;
          c2.attach("componentFlagsChanged", function (cx) {
            verifyEq(this.getName(), "test");
            verifyEq(this.getFlags(), baja.Flags.SUMMARY);
            done = true;
          });
          c.setFlags({slot: "test", flags: baja.Flags.SUMMARY});
          verify(done);
        });
        
        it("component facets changed", function testComponentFacetsChanged() {
          var c = makeSubComp();
          var c2 = makeSubComp();
          c.add({ 
            slot: "test", 
            value: c2, 
            facets: baja.Facets.make(["key1", "key2"], ["value1", "value2"]) 
          });
          
          var newFacets = baja.Facets.make(["foo"], ["bar"]);
          
          var done = false;
          c2.attach("componentFacetsChanged", function (cx) {
            verifyEq(this.getName(), "test");
            verify(this.getFacets().equals(newFacets));
            done = true;
          });
          c.setFacets({slot: "test", facets: newFacets});
          verify(done);
        });
        
        it("component reordered", function testComponentReordered() {
          var c = makeSubComp();
          var kid1 = makeSubComp();
          var kid2 = makeSubComp();
          var kid3 = makeSubComp();
          var kid4 = makeSubComp();
          
          c.add({ slot: "kid1", value: kid1 });
          c.add({ slot: "kid2", value: kid2 });
          c.add({ slot: "kid3", value: kid3 });
          c.add({ slot: "kid4", value: kid4 });
          
          var count = 0;
          kid1.attach("componentReordered", function (cx) {
            count++;
          });
          kid2.attach("componentReordered", function (cx) {
            count++;
          });
          kid3.attach("componentReordered", function (cx) {
            count++;
          });
          kid4.attach("componentReordered", function (cx) {
            count++;
          });
          
          c.reorder(["kid4", "kid3", "kid2", "kid1"]);
          verifyEq(count, 4);
        });
        
        it("unmount", function testUnmount() {
          var c1 = makeSubComp();
          var c2 = makeSubComp();
          var c3 = makeSubComp();
          
          c1.add({slot: "c2", value: c2});
          c2.add({slot: "c3", value: c3});
          
          var count = 0;
          c1.attach("unmount", function (cx) {
            count++; // shouldn't get called - this is an error test
          });
          c2.attach("unmount", function (cx) {
            count++;
          });
          c3.attach("unmount", function (cx) {
            count++;
          });
          
          c1.remove("c2");
          verifyEq(count, 2);
        });
        
        it("attach detach", function testAttachDetach() {
          var c = new baja.Component();
          
          var changed = function () {
          };
          
          var changed2 = function () {
          };
          
          var added = function () {
          };
          
          var added2 = function () {
          };
          
          verify(!c.hasHandlers());
          
          // Test handler is added
          c.attach("changed", changed);
          verifyEq(c.getHandlers("changed").length, 1);
          verify(c.hasHandlers());
          
          // Test function handler can only be added once
          c.attach("changed", changed);
          c.attach("changed", changed);
          c.attach("changed", changed);
          verifyEq(c.getHandlers("changed").length, 1);
          verify(c.hasHandlers());
          verify(c.hasHandlers("changed"));
          verify(!c.hasHandlers("added"));
          verify(c.$handlers !== undefined);
          verify(c.hasHandlers("changed"));
          verify(!c.hasHandlers("added"));
          
          // Test detach
          c.detach("changed");
          verifyEq(c.getHandlers("changed").length, 0);
          verify(!c.hasHandlers());
          verify(c.$handlers === undefined);
          
          // Test attach with two handlers
          c.attach("changed", changed);
          c.attach("changed", changed2);
          verifyEq(c.getHandlers("changed").length, 2);
          
          // Test to a different event type and test handler sizes
          c.attach("added", added);
          c.attach("added", added2);
          verifyEq(c.getHandlers("changed").length, 2);
          verifyEq(c.getHandlers("added").length, 2);
          verifyEq(c.getHandlers("changed added").length, 4);
          verifyEq(c.getHandlers("added changed").length, 4);
          verifyEq(c.getHandlers("added changed topicFired").length, 4);
          verifyEq(c.getHandlers("topicFired").length, 0);
          verifyEq(c.getHandlers("added topicFired").length, 2);
          verify(c.hasHandlers("changed"));
          verify(c.hasHandlers("added"));
          verify(!c.hasHandlers("topicFired"));
          verify(c.hasHandlers("changed added"));
          verify(c.hasHandlers("added changed"));
          verify(!c.hasHandlers("added changed topicFired"));
          
          // Test detach all
          c.detach();
          verifyEq(c.getHandlers("changed").length, 0);
          verifyEq(c.getHandlers("added").length, 0);
          
          c.attach("changed", changed);
          c.attach("changed", changed2);
          c.attach("added", added);
          c.attach("added", added2);
          
          // Detach two handlers at a time
          c.detach("added changed");
          verifyEq(c.getHandlers("changed").length, 0);
          verifyEq(c.getHandlers("added").length, 0);
          verifyEq(c.getHandlers("changed added").length, 0);
          verify(!c.hasHandlers("changed"));
          verify(!c.hasHandlers("added"));
          verify(!c.hasHandlers("added changed"));
          verify(!c.hasHandlers("changed added"));
          
          // Attach via Object Literal
          c.attach({
            changed: function () {},
            added: function () {}
          });
          
          c.attach({
            changed: function () {},
            added: function () {}
          });
          
          verifyEq(c.getHandlers("changed").length, 2);
          verifyEq(c.getHandlers("added").length, 2);
        });
        
        it("fire handlers", function testFireHandlers() {
          var c = new baja.Component(),
              testCount = {
                count: 0
              },
              testErrorCount = 0,
              error = function() {
                testErrorCount++;
              },
              testFunc = function (obj, throwError) {
                if (throwError) {
                  throw new Error("Test error!");
                }
                obj.count++;
              };   
          
          c.attach("test", testFunc);
          
          // Test fire
          c.fireHandlers("test", error, c, testCount);
          verifyEq(testCount.count, 1);
          
          // Test error count
          c.fireHandlers("test", error, c, testCount, /*throwError*/true);
          verifyEq(testErrorCount, 1);
          
          // Test fire X number of times
          baja.iterate(10, function() {
            c.fireHandlers("test", error, c, testCount);
          });
          verifyEq(testCount.count, 11);
          
          // Test error count X number of times
          baja.iterate(10, function() {
            c.fireHandlers("test", error, c, testCount, /*throwError*/true);
          });
          verifyEq(testErrorCount, 11);
          
          c.fireHandlers("nothing", error, c, testCount);
          verifyEq(testCount.count, 11);
          verifyEq(testErrorCount, 11);
        });        
      });
    
      describe("mounted", function mounted() {
        var folderName = "testAttachFolder",
            testPointName = "testPoint",
            testPoint,
            folder,
            changedValue = 123,
            count = 0;
        
        // After each one of these tests, clean up by detaching all handlers        
        afterEach(function () {
          if (testFolder) {
            testFolder.detach();
          }
          if (testPoint) {
            testPoint.detach();
          }
          if (folder) {
            folder.detach();
          }
          count = 0;
        });
            
        it("added", function testAdded() {
          add(folderName, baja.$("baja:Folder"));
          
          runs(function () {
            folder = lease(testFolder.get(folderName));
            
            folder.attach("added", function (prop, cx) {
              expect(prop.getName()).toEqual(testPointName);
              expect(prop.getType().is("control:NumericWritable")).toBeTruthy();
              count++;
            });
            
            // Trigger added
            folder.add({
              slot: testPointName,
              value: baja.$("control:NumericWritable")
            });
          });
          
          // For for all the events we expect to fire
          waitsFor(function () {
            return count === 1;
          });
        });
        
        it("subscribed unsubscribed", function testSubscribed() {        
          runs(function () {
            testPoint = folder.get(testPointName);
          
            testPoint.attach("subscribed unsubscribed", function () {
              count++;
            });
          
            testPoint.lease(baja.RelTime.make({ seconds: 2 }));
          });
          
          waitsFor(function () {
            return count === 2;
          }, "attach subscribed unsubscribed", 60000);
        });
                
        it("changed", function testChanged() {
          runs(function () {
            lease(testPoint);
                        
            testPoint.attach("changed", function (prop, cx) {
              if (prop.getName() === "in4") {
                expect(prop.getName()).toEqual("in4");
                expect(this.get(prop).getValue()).toEquivalent(changedValue);
                count++;
              }
            });
            
            // Trigger set
            testPoint.getIn4().setValue(changedValue);
          });
                               
          waitsFor(function () {
            return count === 1;
          });
        });
                
        it("renamed and componentRenamed", function testRenamed() {        
          var newName = "testPointRenamed";
        
          runs(function () {          
            folder.attach("renamed", function (prop, oldName, cx) {
               expect(prop.getName()).toEqual(newName);
               expect(oldName).toEqual(testPointName);
               count++;
             });
             
            testPoint.attach("componentRenamed", function (oldName, cx) {
              expect(oldName).toEqual(testPointName);
              expect(this.getName()).toEqual(newName);
              count++;
            });
             
            folder.rename({
              slot: testPointName,
              newName: newName
            });
          });
          
          waitsFor(function () {
            return count === 2;
          });
             
          runs(function () {
            testPointName = newName;
          });
        });
        
        it("reordered and componentReordered", function testReordered() {
          runs(function () {
            var testPointName2 = "testPoint2",
                addCb = callbackify({
                  slot: testPointName2,
                  value: baja.$("control:EnumPoint")
                });
            
            runAndWait(function () {
              folder.add(addCb);
            }, addCb);
            
            // Verify existing order
            runs(function () {
              var slots = folder.getSlots().dynamic().toArray();
              expect(slots[0].getName()).toEqual(testPointName);
              expect(slots[1].getName()).toEqual(testPointName2);

              testPoint.attach("componentReordered", function (cx) {
                var newSlots = this.getParent().getSlots().dynamic().toArray();
                expect(newSlots[0].getName()).toEqual(testPointName2);
                expect(newSlots[1].getName()).toEqual(testPointName);
                count++;
              });
              
              folder.attach("reordered", function (cx) {
                var newSlots = this.getSlots().dynamic().toArray();
                expect(newSlots[0].getName()).toEqual(testPointName2);
                expect(newSlots[1].getName()).toEqual(testPointName);
                count++;
              });
            
              folder.reorder({
                dynamicProperties: [testPointName2, testPointName]
              });
            });
          });
          
          waitsFor(function () {
            return count === 2;
          });
        });
        
        it("topicFired", function () {
          runs(function () { 
            var value = baja.$("baja:Weekday", "friday").asDynamic();
                        
            testFolder.attach("topicFired", function(topic, event, cx) {
              if (topic.getName() === "testTopic") {
                expect(event.getValue().equals(value)).toBeTruthy();
                count++;
              }
            });
            
            // Fire a Topic
            testFolder.fire({
              slot: "testTopic", 
              value: baja.$("baja:StatusEnum", {
                value: value
              }) 
            });
          }); 
          
          waitsFor(function () {
            return count === 1;
          });
        });
        
        it("flagsChanged and componentFlagsChanged", function testFlagsChanged() {
          var newFlags = baja.Flags.SUMMARY | baja.Flags.OPERATOR;
        
          runs(function () {
            testPoint.attach("componentFlagsChanged", function (cx) {
              expect(this.getFlags()).toEqual(newFlags);
              count++;
            });
          
            folder.attach("flagsChanged", function (slot, cx) {
              expect(slot.getName()).toEqual(testPointName);
              expect(this.getFlags(slot)).toEqual(newFlags);
              count++;
            });
          
            folder.setFlags({ 
              slot: testPointName,
              flags: newFlags
            });
          });
                     
          waitsFor(function () {
            return count === 2;
          });
        });
        
        it("facetsChanged and componentFacetsChanged", function testFacetsChanged() {
          var newFacets = baja.Facets.make(["test1", "test2"], ["foo1", "foo2"]);
        
          runs(function () {
            testPoint.attach("componentFacetsChanged", function (cx) {
              expect(this.getFacets()).toEquivalent(newFacets);
              count++;
            });
          
            folder.attach("facetsChanged", function (slot, cx) {
              expect(slot.getName()).toEqual(testPointName);
              expect(this.getFacets(slot)).toEquivalent(newFacets);
              count++;
            });
          
            folder.setFacets({ 
              slot: testPointName,
              facets: newFacets
            });
          });
                     
          waitsFor(function () {
            return count === 2;
          });
        });
        
        it("removed and unmount", function testRemoved() {                    
          runs(function () {
             folder.attach("unmount", function (cx) {
               count++;
             });
             
             testPoint.attach("unmount", function (cx) {
               count++;
             });
          
             testFolder.attach("removed", function (prop, val, cx) {
               expect(prop.getName()).toEqual(folderName);
               expect(val.getType().is("baja:Folder")).toBeTruthy();
               count++;
             });
             
             testFolder.remove(folder);
          });
                    
          waitsFor(function () {
            return count === 3;
          });
        });
      });
    });    
  
    describe("getDisplayName", function getDisplayName() {
      it("Component - getDisplayName", function testGetDisplayName() {
        runs(function () {
          var slotName = "testDisplayName",
              displayName = "Hey This is a Display Name!",
              cb,
              nmCb,
              map = {};
              
          map[slotName] = displayName;
          
          cb = callbackify({
            slot: slotName,
            value: "fooA"
          });
          
          runAndWait(function () {
            testFolder.add(cb);
          }, cb);
          
          nmCb = callbackify({
            slot: "displayNames",
            value: baja.NameMap.make(map),
            flags: baja.Flags.HIDDEN | baja.Flags.READONLY
          });
          
          runAndWait(function () {
            testFolder.add(nmCb);
          }, nmCb);
          
          runs(function() {
            expect(testFolder.getDisplayName(slotName)).toEqual(displayName);
          });
        });
      });
    });
    
    describe("BOX Type Black List", function testBlackList() {
    
      describe("BPassword access", function testPasswordAccess() {
    
        it ("attempt to add a Password to the TestFolder", function testPasswordTestFolder() {
          runs(function () {
            var cb = callbackify({
              slot: "test",
              value: baja.$("baja:Password", "some password value"),
              ok: function () {
                expect(false).toBeTruthy();
              },
              fail: function () {
                expect(true).toBeTruthy();
              }
            });
            
            runAndWait(function () {
              testFolder.add(cb);
            }, cb);
          });
        });
        
        it ("access the admin's password (should always be blank)", function testUserPassword() {
          runs(function () {
            var cb = callbackify({
              ok: function () {
                expect(this.getPasswordDisplay()).toEqual("");
              },
              lease: true
            });
            
            baja.Ord.make("station:|slot:/Services/UserService/admin").get(cb);
          });
        });
      });
      
    });

    describe("Program Object classFile access", function testProgramAccess() {
    
      it ("add a Program Object, ensure we can invoke an Action", function testProgramObjectInvoke() {
        runs(function () {
          var sccb = callbackify({
            typeSpec: "testBajaScript:TestBajaScriptServerSideHandler",
            methodName: "addProgram"
          });
          
          // Run and wait for the Server Side Call to complete
          runAndWait(function () {
            testFolder.serverSideCall(sccb);
          }, sccb);
          
          // Now do a Component Sync for the proxy Component Space.
          var syncCb = callbackify({});         
          runAndWait(function () {
            baja.station.sync(syncCb);
          }, syncCb);
          
          // Test that we can't access the code in the Program Object
          runs(function () {
            var program = testFolder.get("program"),
                leaseCb = callbackify({});
            
            runAndWait(function () {
              program.lease(leaseCb);
            }, leaseCb);
            
            runs(function () {
              var invokeCb = callbackify({
                slot: "foo",
                ok: function (returnVal) {
                  expect(returnVal).toEqual("Foo called");
                }
              });
              
              // Make sure we can still invoke a dynamic Action on the Program object
              runAndWait(function () {
                program.invoke(invokeCb);
              }, invokeCb);
            });
          });
        });        
      });
      
      it("attempt to read the class file from a Program Object", function attemptToReadClassFile() {
        runs(function () {
          var program = testFolder.get("program"),
                leaseCb = callbackify({});
            
          runAndWait(function () {
            program.getCode().lease(leaseCb);
          }, leaseCb);
          
          runs(function () {
            expect(program.getCode().getClassFile().encodeToString()).toEqual("");
            expect(program.getCode().getClassFileDisplay()).toEqual("");
          });
        });
      });
      
      it("attempt to set the class file from a Program Object", function attemptToSetClassFile() {
        runs(function () {
          var program = testFolder.get("program"),
                leaseCb = callbackify({});
            
          runAndWait(function () {
            program.getCode().lease(leaseCb);
          }, leaseCb);
          
          runs(function () {
            var setCb = callbackify({
              slot: "classFile",
              value: baja.$("baja:Blob", "123"),
              ok: function () {
                expect(false).toBeTruthy();
              },
              fail: function () {
                expect(true).toBeTruthy();
              }
            });
            
            runAndWait(function () {
              program.getCode().set(setCb);
            }, setCb);
          });
        });
      });
    });
    
  });  
}());
