/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * Jasmine Unit Test Execution for Rhino
 *
 * @author Logan Byam
 * @version 1.0.0.0
 */

//JsLint options (see http://www.jslint.com )
/*jslint rhino: true, onevar: false, plusplus: true, white: true, undef: false, nomen: true, eqeqeq: true, bitwise: true, regexp: true, newcap: true, immed: true, strict: false, indent: 2, vars: true, continue: true */

// Globals for JsLint to ignore 
/*global test, baja,  window: true, location: true, jasmine, console, testFolder: true*/

function rhinoRunJasmine() { 
  "use strict";

  // Rhino Console Reporter
  var RhinoConsoleReporter = function() {
    //inspired by mhevery's jasmine-node reporter
    //https://github.com/mhevery/jasmine-node
    
    var print = console.print,
        runnerStartTime,
        suiteResults = [];
        
  ////////////////////////////////////////////////////////////////
  // Report Util
  ////////////////////////////////////////////////////////////////

    function newline() {
      print("\n\r");
    }

    function plural(str, count) {
      return count === 1 ? str : str + "s";
    }

    function repeat(thing, times) {
      var arr = [],
          i;
      for (i = 0; i < times; i++) {
        arr.push(thing);
      }
      return arr;
    }

    function indent(str, spaces) {
      var lines = (str || '').split("\n"),
          newArr = [],
          i;
          
      for (i = 0; i < lines.length; i++) {
        newArr.push(repeat(" ", spaces).join("") + lines[i]);
      }
      return newArr.join("\n");
    }
    
  ////////////////////////////////////////////////////////////////
  // Report Spec
  ////////////////////////////////////////////////////////////////
    
    function fullSuiteDescription(suite) {
      var fullDescription = suite.description;
      if (suite.parentSuite) {
        fullDescription = fullSuiteDescription(suite.parentSuite) + " - " + fullDescription;
      }
      return fullDescription;
    }

    this.reportRunnerStarting = function() {
      runnerStartTime =  new Date().getTime();
      
      newline();
      print("*** Starting Jasmine Unit Tests ***");
      newline();
    };

    this.reportSpecStarting = function() { /* do nothing */
    };

    this.reportSpecResults = function(spec) {
      var results = spec.results();
      if (results.skipped) {
        print("*");
      } 
      else if (results.passed()) {
        print(".");
      }
      else {
        print("F");
      }
    };

    this.reportSuiteResults = function(suite) {
      var suiteResult = {
        description: fullSuiteDescription(suite),
        failedSpecResults: []
      };

      suite.results().items_.forEach(function(spec) {
        if (spec.failedCount > 0 && spec.description) {
          suiteResult.failedSpecResults.push(spec);
        }
      });

      suiteResults.push(suiteResult);
    };
    
  ////////////////////////////////////////////////////////////////
  // Runner Results
  ////////////////////////////////////////////////////////////////
    
    function specFailureDetails(suiteDescription, specDescription, stackTraces) {                  
      newline();
      print("*** FAILED: " + suiteDescription + " - " + specDescription + " ***");
      newline();
      
      var i;
      for (i = 0; i < stackTraces.length; i++) {
        print(indent(stackTraces[i], 2));
        newline();
      }
    }

    function eachSpecFailure(callback) {
      var i,
          j,
          k,
          suiteResult,
          failedSpecResult,
          stackTraces,
          st,
          failure = false;
    
      for (i = 0; i < suiteResults.length; i++) {
        suiteResult = suiteResults[i];
        
        for (j = 0; j < suiteResult.failedSpecResults.length; j++) {
           
          failure = true;
           
          failedSpecResult = suiteResult.failedSpecResults[j];
          stackTraces = [];  
          
          for (k = 0; k < failedSpecResult.items_.length; k++) {    
            st = failedSpecResult.items_[k].message;

            if (failedSpecResult.items_[k].trace.stack) {
              st = "\n" + failedSpecResult.items_[k].trace.stack;
            }            
             
            stackTraces.push(st);
          }
          callback(suiteResult.description, failedSpecResult.description, stackTraces);
        }
      }
      
      // If there are no failures then pass this test as ok
      if (!failure) {
        test.passTest();
      }
    }

    this.reportRunnerResults = function(runner) {
      newline();  
      print("*** Reporting ***");
      newline();
      eachSpecFailure(function(suiteDescription, specDescription, stackTraces) {
        specFailureDetails(suiteDescription, specDescription, stackTraces);
      });
     
      // Finished
      newline();
      print("  Finished:      " + (new Date().getTime() - runnerStartTime) / 1000 + " seconds");
      newline();
      
      var results = runner.results();
         
      // Summary         
      print("  totalSpecs:    " + runner.specs().length);
      newline();
      print("  totalCount:    " + results.totalCount);
      newline();
      print("  totalPassed:   " + results.passedCount);
      newline();
      print("  totalFailures: " + results.failedCount);                 
      newline();
      newline();
      
      // Success message
      if (results.failedCount === 0) {
        print("*** All Jasmine tests passed!!!");
        newline();
      }
      
      print("*** Finished Jasmine Unit Tests ***");
      newline();
      newline();
    };
  };
  
  var jasmineEnv = jasmine.getEnv();
  jasmineEnv.addReporter(new RhinoConsoleReporter());
  jasmineEnv.execute();

}
