"use strict";
$(function () {
  let dateTimeComp = {
    template: '#datetime-panel-template',
    fp: null,
    data: function () {
      return {
        curDateTime: moment(),
        curTZ: '',
        tz_list: [],
        utcOffset: 0,
        utcOffsetMode: 'Configuration',
        // utcOffsetMode: 'OS',
        tzAbbr: '',

        error: null,
        spinner: null,
      };
    },
    computed: {
      utcOffsetStr: function () {
        return Number(this.utcOffset / 3600.0).toString() + ' hours';
      },
    },
    watch: {
      curTZ: function (newVal, oldVal) {
        if (!this.curDateTime) return;

        this.curDateTime.tz(newVal);
        if (this.fp)
          this.fp.setDate(this.curDateTime.format('YYYY-MM-DD HH:mm:ss'));

        this.utcOffset = this.curDateTime.utcOffset() * 60;
        this.tzAbbr = this.curDateTime.format('z');
      },
      curDateTime: function (newVal, oldVal) {
        this.utcOffset = newVal.utcOffset() * 60;
        this.tzAbbr = newVal.format('z');
      },
    },
    mounted: function () {
      this.getSpinner().spin(this.$el);
      this.loadConfig();
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },
      sysclockInNs: function () {
        return (
          this.curDateTime.unix() * 1000 -
          (this.curDateTime.second() + Date.UTC(2000, 0, 1, 0, 0, 0)) +
          '000000'
        );
      },
      loadConfig: function () {
        this.tz_list = moment.tz.names();

        let self = this;
        $.ajax({
          type: 'GET',
          url: '../../app/data_api.php?url=/app/objects/service/time',
          dataType: 'json',
          success: function (data) {
            if (data.redirect) window.location.href = data.redirect;
            else if (data.error) self.error = data.error.text;
            else self.handleResponse(data.response);
          },
          complete: function () {
            self.onConfigLoaded();
          },
        });
      },
      handleResponse: function (resp) {
        if (_.has(resp, 'resultCode') && resp.resultCode != 0) {
          this.error =
            "Can not access data time object, please make sure the datetime object's path(/service/time) is configured correctly";
          return;
        }
        if (!_.isArray(resp.data) || resp.data.length != 1) {
          this.error = 'Invalid response data';
          return;
        }

        let self = this;
        let tz = null;
        let secs = 0;
        _.each(resp.data[0].slots, function (slot) {
          if (slot.name == 'utcOffset') self.utcOffset = parseInt(slot.value);
          // always use "Configuration" UTC Offset mode
          //else if (slot.name == 'osUtcOffset')
          //  self.utcOffsetMode = slot.value == "true" ? "OS" : "Configuration";
          else if (slot.name == 'tz') tz = slot.value;
          else if (slot.name == 'nanos') {
            let nanos = slot.value;
            secs = nanos / 1000000000 + Date.UTC(2000, 0, 1, 0, 0, 0) / 1000;
            // self.curDateTime = new timezoneJS.Date(secs*1000, 'UTC')
          }
        });
        this.curTZ = tz;
        this.curDateTime = moment.tz(secs * 1000, tz);
      },
      onConfigLoaded: function () {
        this.getSpinner().stop();

        this.fp = $(this.$el)
          .find('#inputDateTime')
          .flatpickr({
            enableTime: true,
            defaultDate: this.curDateTime.format('YYYY-MM-DD HH:mm:ss'),
            dateFormat: 'Y-m-d H:i:S',
            time_24hr: true,
          });
        this.fp.config.onChange.push(this.onDateTimeChanged);
      },
      onDateTimeChanged: function (selectedDates, dateStr, instance) {
        let temp = moment.tz(
          dateStr + this.curDateTime.format('Z'),
          this.curTZ
        );
        //check if datetime has been corrected, if yes, then user must input an
        //invalid datetime, for example, America/New_York, 2020-03-08, during
        //DST forward period (2:00AM ~ 3:00AM), 2:30AM is an invalid clock time,
        //moment will always correct 2:30AM to 3:30AM;
        //another example is that use a wrong time offset, for the above timezone,
        //if set a datetime outside DST, but with DST offset, will be invalid:
        // 2020-12-01 01:00:00-04:00 should be 2020-12-01 01:00:00-05:00
        if (temp.format('YYYY-MM-DD HH:mm:ss') != dateStr)
          temp = moment.tz(dateStr, this.curTZ);

        //check if date time is valid, when user's input is invalid, the parsed
        //datetime will be under UTC timezone
        if (temp.tz() != this.curDateTime.tz() || !temp.isValid()) {
          instance.setDate(this.curDateTime.format('YYYY-MM-DD HH:mm:ss'));
          return;
        } else {
          this.curDateTime = temp;
          instance.setDate(this.curDateTime.format('YYYY-MM-DD HH:mm:ss'));
        }
      },

      postRequest: function (path, value, slotType, dataType, opts) {
        let self = this;
        $.ajax({
          type: 'POST',
          url: '../../app/data_api.php?url=/app/objects/service/time',
          timeout: 30000,
          data: {
            path: path,
            type: dataType,
            value: value,
            slotType: slotType,
          },
          dataType: 'json',
          success: function (data) {
            if (data.redirect) window.location.href = data.redirect;
            else if (data.error) self.error = data.error.text;
            else if (opts && _.isFunction(opts.success)) opts.success();
          },
          error: function (data, status) {
            self.error = 'Error: ' + status;
            if (opts && _.isFunction(opts.error)) opts.error();
          },
          complete: function () {
            if (opts && _.isFunction(opts.complete)) opts.complete();
          },
        });
      },
      saveConfig: function () {
        let self = this;
        this.error = null;

        let path = '/app/objects/service/time';
        let isOSUTCOffsetMode = this.utcOffsetMode == 'OS';
        this.getSpinner().spin(this.$el);
        this.postRequest(
          path + '.osUtcOffset',
          isOSUTCOffsetMode,
          'property',
          'bool',
          {
            success: function () {
              self.postRequest(path + '.tz', self.curTZ, 'property', 'Buf', {
                success: function () {
                  if (isOSUTCOffsetMode)
                    self.postRequest(
                      path + '.setSysClock',
                      self.sysclockInNs(),
                      'action',
                      'long',
                      {
                        complete: function () {
                          self.getSpinner().stop();
                        },
                      }
                    );
                  else
                    self.postRequest(
                      path + '.utcOffset',
                      self.utcOffset,
                      'property',
                      'int',
                      {
                        success: function () {
                          self.postRequest(
                            path + '.setSysClock',
                            self.sysclockInNs(),
                            'action',
                            'long',
                            {
                              complete: function () {
                                self.getSpinner().stop();
                              },
                            }
                          );
                        },
                        error: function () {
                          self.getSpinner().stop();
                        },
                      }
                    );
                },
                error: function () {
                  self.getSpinner().stop();
                },
              });
            },
            error: function () {
              self.getSpinner().stop();
            },
          }
        );
      },
    },
  };

  let ntpComp = {
    template: '#ntp-panel-template',
    data: function () {
      return {
        timeServer: 'time.windows.com',
        timeServerError: false,

        updateInterval: 1,
        updateIntervalError: false,

        enableTimeSync: '1',

        error: null,
        syncState: null,
        syncSuccess: false,

        spinner: null,

        timerId: null,
        iot_platform: window.iot_platform,
        fwSpecificUI: false,
      };
    },
    mounted: function () {
      this.loadConfig();
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },

      queryState: function () {
        let self = this;
        $.ajax({
          type: 'GET',
          url: 'datetime_config.php',
          data: {
            action: 'getState',
          },
          dataType: 'json',
          success: function (data, status, xhr) {
            if (data.redirect) window.location.href = data.redirect;
            else if (data.error) {
              self.syncSuccess = false;
              self.syncState = data.error.text;

              if (self.timerId) {
                clearTimeout(self.timerId);
              }
              //retry after 15 seconds
              self.timerId = setTimeout(_.bind(self.queryState, self), 15000);
            } else {
              if (_.has(data, 'state')) {
                let state = data.state;
                let stateCode = state.split(':', 1);
                self.syncSuccess = stateCode == '0';
                self.syncState = state.substring(stateCode.length + 1);
              } else {
                self.syncSuccess = false;
                self.syncState = null;
              }
              self.timerId = null;
            }
          },
          error: function (xhr, status, error) {
            self.error = error;
          },
        });
      },

      loadConfig: function () {
        let self = this;
        $.ajax({
          type: 'GET',
          url: 'datetime_config.php',
          data: {
            action: 'getTimeServer',
          },
          dataType: 'json',
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data, status, xhr) {
            if (data.redirect) window.location.href = data.redirect;
            else if (data.error) self.error = data.error.text;
            else {
              if (_.has(data, 'config')) {
                let config = data.config;
                if (_.has(config, 'timeServer')) {
                  //TODO: handle multiple time server config. now we only handle the first one
                  self.timeServer = _.first(config.timeServer.split(','));
                }
                if (_.has(config, 'updateInterval'))
                  self.updateInterval = parseInt(config.updateInterval);
                if (_.has(config, 'enableTimeSync'))
                  self.enableTimeSync = config.enableTimeSync;
              }
            }
          },
          error: function (xhr, status, error) {
            self.error = error;
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },
      validateInputs: function () {
        this.timeServerError = !/^[a-zA-Z0-9][-.a-zA-Z0-9]*[a-zA-Z0-9]$/.test(
          this.timeServer
        );
        this.updateIntervalError = this.updateInterval <= 0;
        return !this.timeServerError && !this.updateIntervalError;
      },
      saveConfig: function () {
        if (!this.validateInputs()) return;

        let self = this;
        $.ajax({
          type: 'POST',
          url: 'datetime_config.php',
          dataType: 'json',
          data: {
            action: 'updateTimeServer',
            enableTimeSync: this.enableTimeSync,
            timeServer: this.timeServer,
            updateInterval: this.updateInterval,
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data, status, xhr) {
            if (data.redirect) window.location.href = data.redirect;
            else if (data.error) self.error = data.error.text;
            else {
            }
          },
          error: function (xhr, status, error) {
            self.error = error;
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },

      syncDateTime: function () {
        if (this.enableTimeSync != '1') return;

        let self = this;
        $.ajax({
          type: 'POST',
          url: 'datetime_config.php',
          dataType: 'json',
          data: {
            action: 'syncDateTime',
          },
          beforeSend: function () {
            self.syncSuccess = false;
            self.syncState = null;
            self.getSpinner().spin(self.$el);
          },
          success: function (data, status, xhr) {
            if (data.redirect) window.location.href = data.redirect;
            else if (data.error) self.error = data.error.text;
            else {
              if (self.timerId) {
                clearTimeout(self.timerId);
              }
              self.timerId = setTimeout(_.bind(self.queryState, self), 10000);
            }
          },
          error: function (xhr, status, error) {
            self.error = error;
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },
    },
  };

  var app = new Vue({
    //{{{1
    el: '#datetime_config_content',
    data: function () {
      let sections = [
        { sec_id: 'datetime-panel', name: 'DateTime' },
        { sec_id: 'ntp-panel', name: 'NTP' },
      ];
      let anchor = window.location.hash.substring(1);
      let sec = sections.find(function (s) {
        return s.sec_id == anchor;
      });
      let active_sec_id = sec && sec.sec_id ? sec.sec_id : sections[0].sec_id;
      return {
        sections,
        active_sec_id,
      };
    },
    methods: {
      isSectionActive: function (sec) {
        return sec.sec_id == this.active_sec_id;
      },
    },
    components: {
      'datetime-panel': dateTimeComp,
      'ntp-panel': ntpComp,
    },
  });
});

// vim: foldmethod=marker ts=2 sw=2
