"use strict";
// vim: ts=2 sw=2 foldmethod=marker
$(function () {
  var bacnetOnly = false;

  // general functions  {{{1
  function uniqueId(prefix, existingIds) {
    var newId = '';
    var suffixNum = Date.now();
    do {
      newId = prefix + '_' + suffixNum.toString();
      ++suffixNum;
    } while (_.contains(existingIds, newId));
    return newId;
  }
  function uniqueIdEx(prefix) {
    return (
      prefix +
      new Date().getTime().toString() +
      '_' +
      _.random(0, 10000).toString()
    );
  }

  function uniqueName(prefix, existingNames) {
    var newName = '';
    var suffixNum = 0;
    do {
      newName = prefix + '_' + suffixNum.toString();
      ++suffixNum;
    } while (_.contains(existingNames, newName));
    return newName;
  }

  function mapObject(obj, processor) {
    var keys = _.keys(obj);
    var results = {},
      currentKey;
    for (var index = 0; index < keys.length; ++index) {
      currentKey = keys[index];
      results[currentKey] = processor(obj[currentKey], currentKey, obj);
    }
    return results;
  }

  function findPos(obj) {
    var curtop = 0;
    var curleft = 0;
    if (obj.offsetParent) {
      do {
        curleft += obj.offsetLeft;
        curtop += obj.offsetTop;
      } while ((obj = obj.offsetParent));
    }
    return [curleft, curtop];
  }

  function getShortTypeName(sedonaNodeData) {
    var type = null;
    if (_.has(sedonaNodeData, 'type')) type = sedonaNodeData.type;

    var shortName = null;
    if (type == 'easyioFGBacnet::BACnetClientBV') shortName = 'BV';
    else if (type == 'easyioFGBacnet::BACnetClientBI') shortName = 'BI';
    else if (type == 'easyioFGBacnet::BACnetClientBO') shortName = 'BO';
    else if (type == 'easyioFGBacnet::BACnetClientAV') shortName = 'AV';
    else if (type == 'easyioFGBacnet::BACnetClientAI') shortName = 'AI';
    else if (type == 'easyioFGBacnet::BACnetClientAO') shortName = 'AO';
    return shortName;
  }

  function isVarPath(path) {
    return path.startsWith('@Var');
  }

  function isSystemVarPath(path) {
    return path.startsWith('@Var/system/');
  }
  function isUserVarPath(path) {
    return path.startsWith('@Var/user/');
  }

  function cloneDeep(obj) {
    return JSON.parse(JSON.stringify(obj));
  }

  // Copies a string to the clipboard. Must be called from within an
  // event handler such as click. May return false if it failed, but
  // this is not always possible. Browser support for Chrome 43+,
  // Firefox 42+, Safari 10+, Edge and Internet Explorer 10+.
  // Internet Explorer: The clipboard feature may be disabled by
  // an administrator. By default a prompt is shown the first
  // time the clipboard is used (per session).
  function copyToClipboard(text) {
    if (window.clipboardData && window.clipboardData.setData) {
      // Internet Explorer-specific code path to prevent textarea being shown while dialog is visible.
      return window.clipboardData.setData('Text', text);
    } else if (
      document.queryCommandSupported &&
      document.queryCommandSupported('copy')
    ) {
      var textarea = document.createElement('textarea');
      textarea.textContent = text;
      textarea.style.position = 'fixed'; // Prevent scrolling to bottom of page in Microsoft Edge.
      document.body.appendChild(textarea);
      textarea.select();
      try {
        return document.execCommand('copy'); // Security exception may be thrown by some browsers.
      } catch (ex) {
        console.warn('Copy to clipboard failed.', ex);
        return prompt('Copy to clipboard: Ctrl+C, Enter', text);
      } finally {
        document.body.removeChild(textarea);
      }
    }
  }

  // payload jstree related functions
  // collect all data point paths starting from the topNode
  //  - jstree: the payload structure tree
  //  - topNode: payload node(must be antecedent node of property node)
  function collectAllDataBindings(jstree, topNode, compOnly = true) {
    let pathList = [];
    let nodeIdList = [topNode.id].concat(topNode.children_d);
    nodeIdList.forEach(function (cid) {
      let node = jstree.get_node(cid);
      let ntype = jstree.get_type(node);
      let isDataNode =
        (ntype == 'object' || ntype == 'topic') &&
        node.hasOwnProperty('original') &&
        node.original &&
        node.original.hasOwnProperty('objDataList') &&
        node.original['objDataList'];

      if (!isDataNode) return;
      let objPaths = node.original['objDataList'].map(function (objData) {
        return objData.path;
      });
      pathList.push(...objPaths);
    });
    if (compOnly === true)
      pathList = pathList.filter((path) => !isVarPath(path));
    return _.uniq(pathList);
  }

  // compute the most common data path based on data point path list
  function getCommonDataPath(bindingList) {
    if (!bindingList || bindingList.length == 0) return '';
    let bindingPartsList = bindingList.map(function (binding) {
      return binding.split('/');
    });
    let minLen = Math.min(
      ...bindingPartsList.map(function (parts) {
        return parts.length;
      })
    );
    if (minLen <= 0) return '';
    for (let i = minLen; i > 0; i--) {
      let initList = [];
      bindingPartsList
        .map(function (parts) {
          return parts.slice(0, i).join('/');
        })
        .reduce(function (dedupList, cur) {
          if (!dedupList.includes(cur)) dedupList.push(cur);
          return dedupList;
        }, initList);
      if (initList.length == 1) return initList[0];
    }
    return '';
  }

  function sendValidationReq(url, successCallback, errorCallback) {
    if (!url) return;

    let baseURL = '../../app/data_api.php';
    $.ajax({
      url: baseURL,
      method: 'GET',
      dataType: 'json',
      data: { url: '/app/objects' + url },
      success: successCallback,
      error: function (jq, status, error) {
        if (errorCallback) errorCallback();
        console.warn('got error: ' + error);
      },
    });
  }

  function fetchDataBindings(
    topNode,
    urls,
    startCallback,
    successCallback,
    failureCallback
  ) {
    if (!urls || urls.length == 0) return;

    let max_req_url = 1000;
    //group urls
    let reqURLs = urls.reduce(function (init, url) {
      let section = '';
      if (init.length == 0) init.push(section);
      else section = init[init.length - 1];

      if (section.length + url.length + 1 > max_req_url) init.push(url);
      else {
        if (section.length > 0) init[init.length - 1] = section + '~' + url;
        else init[init.length - 1] = url;
      }
      return init;
    }, []);

    let availableDataMapping = {};
    let errorDataList = [];
    let callback = function (data, status, jq) {
      if (data.redirect) {
        if (failureCallback) failureCallback();
        redirect(data.redirect);
        return;
      }
      if (!data.response) {
        console.warn('invalid response data format');
        if (failureCallback) failureCallback();
        return;
      }

      if (data.response.resultCode == 0) {
        data.response.data.reduce(function (init, data) {
          init[data.path] = data.slots.map(function (s) {
            return s.name;
          });
          return init;
        }, availableDataMapping);
      } else {
        data.response.errors.forEach(function (e) {
          //TODO: check only missing comp or slots after easyioCpt kit fixed
          //only handle component missing error
          if (e.errorCode != 2) return;

          errorDataList.push(e.path);
        });
      }

      if (reqURLs.length > 0)
        sendValidationReq(reqURLs.pop(), callback, failureCallback);
      else {
        // when all data fetched, mark invalid data bindings
        if (successCallback)
          successCallback(topNode, availableDataMapping, errorDataList);
      }
    };

    if (reqURLs.length > 0) {
      if (startCallback) startCallback();
      sendValidationReq(reqURLs.pop(), callback, failureCallback);
    }
  }

  const brokerTypeList = [
    'gcp-iot-core',
    'mqtt',
    'aws-iot-core',
    'azure-iot-hub',
  ];
  function brokerTypeLabel(type) {
    const typeLabelMapping = {
      'gcp-iot-core': 'Google Cloud',
      mqtt: 'General MQTT',
      'aws-iot-core': 'AWS IoT Core',
      'azure-iot-hub': 'Azure IoT Hub',
    };
    if (_.has(typeLabelMapping, type)) return typeLabelMapping[type];
    else return type;
  }

  function isNameValid(name) {
    return name && /^[-_a-zA-Z0-9]{1,32}$/.test(name);
  }

  function isTopicPathValid(path) {
    return path && /^[-_/a-zA-Z0-9]{1,256}$/.test(path);
  }

  //add original.objDataList
  function addDataBindingData(jstree, nodeJson) {
    if (!nodeJson) return;
    let node = jstree.get_node(nodeJson.id);
    if (node.original && node.original.objDataList) {
      nodeJson['objDataList'] = cloneDeep(node.original.objDataList);
    }

    if (_.has(nodeJson, 'children')) {
      nodeJson.children.forEach(function (childJson) {
        addDataBindingData(jstree, childJson);
      });
    }
  }

  function getTemplateJson(jstree, node, force) {
    let type = jstree.get_type(node);
    if (!force && type != 'topic' && type != 'object') return {};

    let options = { no_state: true, no_li_attr: true, no_a_attr: true };
    let json = jstree.get_json(node, options);
    addDataBindingData(jstree, json);
    return json;
  }

  function convertNodeUIDs(json) {
    let uid_mappings = {};

    //regenerate data binding uid
    let convertSlotUID = function (jsonNode) {
      if (jsonNode.objDataList) {
        jsonNode.objDataList.forEach(function (objData) {
          if (!objData.slots) return;
          objData.slots.forEach(function (slot) {
            let old = slot.uid;
            slot.uid = uniqueIdEx('#slot_');
            uid_mappings[old] = slot.uid;
          });

          if (objData.path && isVarPath(objData.path)) {
            let pos = objData.path.indexOf('$');
            let oldPath = objData.path;
            if (pos != -1) {
              let name = objData.path.substring(0, pos + 1);
              objData.path = uniqueIdEx(name);
            }
            if (objData.path != oldPath) {
              objData.slots.forEach(function (slot) {
                slot.path.replace(oldPath, objData.path);
              });
            }
          }
        });
      }

      if (_.has(jsonNode, 'children')) {
        jsonNode.children.forEach(function (child) {
          convertSlotUID(child);
        });
      }
    };
    convertSlotUID(json);

    //convert topic, object or property node id
    let convertNodeUID = function (jsonNode) {
      if (jsonNode.type == 'topic') {
        jsonNode.id = uniqueIdEx('#topic_');
      } else if (jsonNode.type == 'object') {
        jsonNode.id = uniqueIdEx('#object_');
      } else if (jsonNode.type == 'property') {
        let newId = uid_mappings[jsonNode.id];
        if (!newId) {
          //FIXME: handle Var type property
          jsonNode.id = uniqueIdEx('#slot_');
        } else {
          jsonNode.id = newId;
        }
      } else {
        console.warn('invalid node data for template');
        console.debug(jsonNode);
      }
      if (_.has(jsonNode, 'children')) {
        jsonNode.children.forEach(function (child) {
          convertNodeUID(child);
        });
      }
    };

    convertNodeUID(json);
    return json;
  }
  // }}}1

  Vue.component('service-status', {
    // {{{1
    template:
      '<div class="pull-right"><span class="label" :class="status_class" @click="label=window.service_version">{{ label }}</span></div>',
    data: function () {
      return {
        status: '',
        label: 'unavailable',
      };
    },
    computed: {
      status_class: function () {
        if (this.status.length > 0) return 'label-' + this.status;
        else return '';
      },
    },
    mounted: function () {
      (this.updateServiceStatus = _.debounce(
        _.bind(this.doUpdateServiceStatus, this),
        10000
      )),
        this.doUpdateServiceStatus();
    },
    methods: {
      updateServiceStatus: function () {},
      doUpdateServiceStatus: function () {
        var self = this;
        $.ajax({
          url: 'index.php?action=status',
          method: 'GET',
          dataType: 'json',
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            if (!resp.service_status) {
              self.status = '';
              self.label = 'unavailable';

              for (var j = 0; j < gConfig.brokerList.length; ++j) {
                Vue.set(gConfig.brokerList[j], 'state_code', -1);
                Vue.set(gConfig.brokerList[j], 'state_msg', '');
              }
              return;
            }

            var resp_status = resp.service_status;
            if (resp_status == 'started') {
              self.status = 'success';
              self.label = 'started';
            } else if (resp_status == 'stopped') {
              self.status = 'warning';
              self.label = 'stopped';
            } else if (resp_status == 'error') {
              self.status = 'important';
              self.label = 'error';
            } else {
              self.status = '';
              self.label = 'unavailable';
            }

            //reset states
            for (var j = 0; j < gConfig.brokerList.length; ++j) {
              Vue.set(gConfig.brokerList[j], 'state_code', -1);
              Vue.set(gConfig.brokerList[j], 'state_msg', '');
            }

            if (resp.broker_states) {
              var states = resp.broker_states;
              for (var i = 0; i < states.length; ++i) {
                var s = states[i];
                if (!s.name) continue;

                for (var j = 0; j < gConfig.brokerList.length; ++j) {
                  if (gConfig.brokerList[j].name != s.name) continue;

                  if (s.errno)
                    Vue.set(
                      gConfig.brokerList[j],
                      'state_code',
                      Number(s.errno)
                    );
                  else Vue.set(gConfig.brokerList[j], 'state_code', 0);

                  if (s.msg) Vue.set(gConfig.brokerList[j], 'state_msg', s.msg);
                  else Vue.set(gConfig.brokerList[j], 'state_msg', '');
                  break;
                }
              }
            }
          },
          error: function (data, status) {
            self.status = '';
            self.label = 'unavailable';
            console.warn('get error: ' + data);
          },
          complete: function () {
            self.updateServiceStatus();
          },
        });
      },
    },
  }); // }}}1

  Vue.component('modal-dialog', {
    //{{{1
    template: '#modal_dialog_template',
    props: {
      title: String,
      saveBtnLabel: {
        type: String,
        default: 'Save',
      },
      enterToSubmit: {
        type: Boolean,
        default: true,
      },
    },
    data: function () {
      return {};
    },
    mounted: function () {
      let self = this;
      this.$el.onshow = function () {
        $(self.$el).find('input').focus();
      };
    },
    methods: {
      onSubmit: function () {
        this.$emit('submitChange');
      },
    },
  }); //}}}1

  Vue.component('sortable-table-header', {
    //{{{1
    template: '#sortable_table_header_template',
    props: ['sortField', 'fieldName'],
    data: function () {
      return {
        ascendingSort: true,
      };
    },
    watch: {
      ascendingSort: function (newVal, oldVal) {
        this.$emit('sortBy', this.fieldName, newVal);
      },
    },
  }); //}}}1

  Vue.filter('capitalize', function (value) {
    //{{{1
    if (!value) return '';
    value = value.toString();
    return value.charAt(0).toUpperCase() + value.slice(1);
  }); //}}}1

  //define components
  let gcpProxiedDevicesManagerComp = {
    ///{{{1
    template: '#gcp_proxied_devices_manager_template',
    props: {
      devices: {
        type: Array,
        required: true,
      },
    },
    data: function () {
      return {
        newDevice: { device_id: '', uid: '' },
        newDeviceError: '',

        activeUId: null,
        editDevice: { device_id: '', uid: '' },
        editDeviceError: '',

        devicesSnapshot: cloneDeep(this.devices),
      };
    },
    watch: {
      devices: function (dlist) {
        this.devicesSnapshot.splice(0, this.devicesSnapshot.length);
        this.devicesSnapshot.push(...dlist);
      },
    },
    methods: {
      init: function () {
        this.devicesSnapshot.splice(0, this.devicesSnapshot.length);
        this.devicesSnapshot.push(...this.devices);
      },

      isDeviceActive: function (d) {
        return this.activeUId && this.activeUId == d.uid;
      },
      setActiveDevice: function (d) {
        this.activeUId = d.uid;
        this.editDeviceError = '';
        this.editDevice = Object.assign({}, this.editDevice, cloneDeep(d));
      },

      addDevice: function () {
        if (!this.newDevice || !this.newDevice.device_id) {
          this.newDeviceError = 'New Device ID can not be empty';
          return;
        }

        let index = this.devicesSnapshot.findIndex(
          (d) => d.device_id == this.newDevice.device_id
        );
        if (index >= 0) {
          this.newDeviceError = 'Device already exists';
          return;
        }

        this.newDevice.uid = uniqueIdEx('#device_');
        this.devicesSnapshot.unshift(cloneDeep(this.newDevice));
        this.newDevice.device_id = '';
        this.newDevice.uid = '';
        this.newDeviceError = '';
      },
      updateDevice: function (uid) {
        let index = this.devicesSnapshot.findIndex((d) => d.uid == uid);
        if (index < 0) {
          //this should not happen
          this.editDeviceError = 'invalid input';
          return;
        }

        if (this.editDevice.device_id.length <= 0) {
          this.editDeviceError = 'Device ID can not be empty';
          return;
        }
        let index2 = this.devicesSnapshot.findIndex(
          (d) => d.device_id == this.editDevice.device_id
        );

        if (index2 >= 0 && index2 != index) {
          this.editDeviceError = 'Device ID is invalid';
          return;
        }

        this.devicesSnapshot.splice(index, 1, cloneDeep(this.editDevice));

        this.editDeviceError = '';
        this.activeUId = null;
      },
      deleteDevice: function (uid) {
        let index = this.devicesSnapshot.findIndex((d) => d.uid == uid);
        if (index >= 0) this.devicesSnapshot.splice(index, 1);
      },
      cancelUpdate: function () {
        this.editDeviceError = '';
        this.activeUId = null;
      },

      save: function () {
        this.$emit('update:devices', cloneDeep(this.devicesSnapshot));
        return true;
      },
    },
  }; //}}}1

  let certGeneratorComp = {
    //{{{1
    template: '#cert_generator_template',
    props: {
      brokerType: {
        type: String,
        default: 'GoogleIoTCore',
        required: false,
      },
    },
    data: function () {
      return {
        name: '',
        cert_content: '',
        error: '',
        spinner: null,
      };
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },

      onGenerate: function () {
        this.cert_content = '';
        this.error = '';

        if (!/^[-_a-zA-z0-9]{1,32}$/.test(this.name)) {
          this.error =
            'invalid cert name, only -, _, and alphanumber allowed and 32 chars at most.';
          return;
        }

        let self = this;
        $.ajax({
          url: 'cert_generator.php',
          method: 'POST',
          dataType: 'json',
          data: {
            brokerType: this.brokerType,
            name: this.name,
          },
          beforeSend: function () {
            self.getSpinner().spin($(self.$el).find('.modal-body').get(0));
          },
          success: function (data) {
            if (data.redirect) redirect(data.redirect);
            if (data.error) self.error = data.error.text;
            if (data.cert) self.cert_content = data.cert;
          },
          error: function (data, status) {
            alert('Failed: ' + data);
            // msgbus.$emit("alert", "Error", "Failed to load config data: " + data, 'error');
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },

      copyCert: function () {
        copyToClipboard(this.cert_content);
      },
    },
  }; //}}}1

  let gcpIoTCoreComp = {
    //{{{1
    template: '#gcp_iot_core_template',
    props: ['brokerName', 'configData', 'nameError', 'errors'],
    data: function () {
      let _data = {
        name: this.brokerName,
        config: cloneDeep(this.configData),
        errors: this.errors,
      };
      if (!_.has(_data.config, 'data_precision'))
        _data.config['data_precision'] = 2;
      if (!_.has(_data.config, 'region'))
        _data.config['region'] = 'us-central1';
      if (!_.has(_data.config, 'gateway_enabled'))
        _data.config['gateway_enabled'] = false;
      if (!_.has(_data.config, 'proxied_devices'))
        _data.config['proxied_devices'] = [];
      return _data;
    },
    computed: {
      errors: function () {
        return _.filter([
          this.host_error,
          this.port_error,
          this.name_error,
          this.registry_id_error,
          this.project_id_error,
          this.device_id_error,
          this.data_precision_error,
          this.region_error,
          this.ca_file_error,
          this.cert_file_error,
          this.key_file_error,
          this.events_interval_error,
        ], function (error) { return error != null && error.length > 0; });
      },
      host_error: function () {
        return this.config.mqtt_host.length > 0 ? '' : 'Invalid MQTT Endpoint';
      },
      port_error: function () {
        return _.isNumber(this.config.mqtt_port) && this.config.mqtt_port > 0
          ? ''
          : 'Invalid MQTT Endpoint Port';
      },
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      registry_id_error: function () {
        return this.config.registry_id.length > 0 ? '' : 'Invalid Registry ID';
      },
      project_id_error: function () {
        return this.config.project_id.length > 0 ? '' : 'Invalid Project ID';
      },
      device_id_error: function () {
        return this.config.device_id.length > 0 ? '' : 'Invalid Device ID';
      },
      data_precision_error: function () {
        let b = false;
        let precision = this.config.data_precision;
        const pattern = /^[1-6]$/;
        if (precision != null && precision != "") {
          b = pattern.test(precision);
        }
        return b ? "" : "Invalid Precision : Only support [1-6]";
      },
      region_error: function () {
        if (_.has(this.config, 'region'))
          return this.config.region.length > 0 ? '' : 'Invalid Region';
        else return '';
      },
      ca_file_error: function () {
        return this.config.ca_file.length > 0 ? '' : 'Invalid CA File';
      },
      cert_file_error: function () {
        return this.config.cert_file.length > 0
          ? ''
          : 'Invalid Certificate File';
      },
      key_file_error: function () {
        return this.config.key_file.length > 0 ? '' : 'Invalid Key File';
      },
      events_interval_error: function () {
        return _.isNumber(this.config.events_interval) &&
          this.config.events_interval > 0
          ? ''
          : 'Invalid Update Interval';
      },
    },
    methods: {
      changeFile: function (title, oldval, callback) {
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst) return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function () {
        var self = this;
        this.changeFile('Choose CA File', self.config.ca_file, function (path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      changeClientCertFile: function () {
        var self = this;
        this.changeFile(
          'Choose Client Cert File',
          self.config.cert_file,
          function (path) {
            self.$set(self.config, 'cert_file', path);
          }
        );
      },
      changeKeyFile: function () {
        var self = this;
        this.changeFile(
          'Choose Key File',
          self.config.key_file,
          function (path) {
            self.$set(self.config, 'key_file', path);
          }
        );
      },

      manageProxiedDevices: function () {
        if (!this.config.gateway_enabled) return;

        this.$refs.proxied_devices_manager.init();
        $(this.$el).find('.proxied_device_dialog').modal('show');
      },
      onSaveProxiedDevices: function () {
        if (this.$refs.proxied_devices_manager.save())
          $(this.$el).find('.proxied_device_dialog').modal('hide');
      },

      generateCert: function () {
        $(this.$el).find('.cert-generator').modal('show');
      },

      isInputsValid: function () {
        return !(this.errors != null && this.errors.length > 0);
      },
      onSaveLocal: function () {
        if (!this.isInputsValid()) {
          msgbus.$emit(
            'alert',
            'Error',
            'Some inputs are invalid, please fix it and try again.',
            'error',
            0
          );
          return;
        }
        //notify topicComp to update its path
        if (this.config.device_id != this.configData.device_id) {
          let evtData = {
            brokerName: this.brokerName,
            oldDeviceId: this.configData.device_id,
            newDeviceId: this.config.device_id,
          };
          msgbus.$emit('broker:gcp:deviceIdChanged', evtData);
        }

        this.$emit('brokerUpdated', this.name, cloneDeep(this.config), cloneDeep(this.errors));
      },
      onReset: function () {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', cloneDeep(this.configData));
      },
    },
    components: {
      'gcp-proxied-devices-manager': gcpProxiedDevicesManagerComp,
      'cert-generator': certGeneratorComp,
    },
  }; // }}}1

  let mqttComp = {
    //{{{1
    template: '#mqtt-template',
    props: ['brokerName', 'configData', 'nameError', 'errors'],
    data: function () {
      let _data = {
        name: this.brokerName,
        config: cloneDeep(this.configData),
        errors: this.errors,
      };
      if (!_.has(_data.config, 'data_precision'))
        _data.config['data_precision'] = 2;
      return _data;
    },
    mounted: function () {
      if (_.has(this.config, 'no_tls')) this.tls_enabled = !this.config.no_tls;
    },
    computed: {
      errors: function () {
        return _.filter([
          this.host_error,
          this.port_error,
          this.name_error,
          this.client_id_error,
          this.user_error,
          this.passwd_error,
          this.data_precision_error,
          this.ca_file_error,
          this.cert_file_error,
          this.key_file_error,
          this.events_interval_error,
        ], function (error) { return error != null && error.length > 0; });
      },
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      host_error: function () {
        return this.config.mqtt_host.length > 0 ? '' : 'Invalid MQTT Host Name';
      },
      port_error: function () {
        return _.isNumber(this.config.mqtt_port) && this.config.mqtt_port > 0
          ? ''
          : 'Invalid MQTT Port';
      },
      client_id_error: function () {
        return '';
      },
      user_error: function () {
        return '';
        //return this.config.mqtt_user.length > 0 ? "" : "Invalid User Name";
      },
      passwd_error: function () {
        return '';
        //return this.config.mqtt_passwd.length > 0 ? "" : "Invalid Password";
      },
      data_precision_error: function () {
        let b = false;
        let precision = this.config.data_precision;
        const pattern = /^[1-6]$/;
        if (precision != null && precision != "") {
          b = pattern.test(precision);
        }
        return b ? "" : "Invalid Precision : Only support [1-6]";
      },

      ca_file_error: function () {
        return this.config.no_tls || this.config.ca_file.length > 0
          ? ''
          : 'Invalid CA File';
      },
      cert_file_error: function () {
        if (this.config.no_tls) return '';

        if (
          this.config.cert_file.length == 0 &&
          this.config.key_file.length > 0
        )
          return 'Invalid Certificate File';
        else return '';
      },
      key_file_error: function () {
        if (this.config.no_tls) return '';

        if (
          this.config.key_file.length == 0 &&
          this.config.cert_file.length > 0
        )
          return 'Invalid Key File';
        else return '';
      },

      events_interval_error: function () {
        return _.isNumber(this.config.events_interval) &&
          this.config.events_interval > 0
          ? ''
          : 'Invalid Update Interval';
      },

      tls_enabled: {
        get: function () {
          return !this.config.no_tls;
        },
        set: function (newVal) {
          this.$set(this.config, 'no_tls', !newVal);
        },
      },
    },
    methods: {
      changeFile: function (title, oldval, callback) {
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst) return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile('Choose CA File', self.config.ca_file, function (path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      changeClientCertFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile(
          'Choose Client Cert File',
          self.config.cert_file,
          function (path) {
            self.$set(self.config, 'cert_file', path);
          }
        );
      },
      changeKeyFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile(
          'Choose Key File',
          self.config.key_file,
          function (path) {
            self.$set(self.config, 'key_file', path);
          }
        );
      },

      isInputsValid: function () {
        return !(this.errors != null && this.errors.length > 0);
      },
      onSaveLocal: function () {
        if (!this.isInputsValid()) {
          msgbus.$emit(
            'alert',
            'Error',
            'Some inputs are invalid, please fix it and try again.',
            'error',
            0
          );
          return;
        }
        this.$emit('brokerUpdated', this.name, cloneDeep(this.config), cloneDeep(this.errors));
      },
      onReset: function () {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', cloneDeep(this.configData));
      },
    },
  }; //}}}1

  let awsIoTCoreComp = {
    //{{{1
    template: '#aws_iot_core_template',
    props: ['brokerName', 'configData', 'nameError', 'errors'],
    data: function () {
      let _data = {
        name: this.brokerName,
        config: cloneDeep(this.configData),
        errors: this.errors,
      };
      if (!_.has(_data.config, 'data_precision'))
      _data.config['data_precision'] = 2;
      return _data;
    },
    computed: {
      errors: function () {
        return _.filter([
          this.name_error,
          this.host_error,
          this.client_id_error,
          this.data_precision_error,
          this.ca_file_error,
          this.cert_file_error,
          this.key_file_error,
          this.events_interval_error,
        ], function (error) { return error != null && error.length > 0; });
      },
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      host_error: function () {
        return this.config.mqtt_host.length > 0 ? '' : 'Invalid AWS Endpoint';
      },
      client_id_error: function () {
        return '';
      },
      data_precision_error: function () {
        let b = false;
        let precision = this.config.data_precision;
        const pattern = /^[1-6]$/;
        if (precision != null && precision != "") {
          b = pattern.test(precision);
        }
        return b ? "" : "Invalid Precision : Only support [1-6]";
      },
      ca_file_error: function () {
        return this.config.ca_file.length > 0 ? '' : 'Invalid CA File';
      },
      cert_file_error: function () {
        if (
          this.config.cert_file.length == 0 &&
          this.config.key_file.length > 0
        )
          return 'Invalid Certificate File';
        else return '';
      },
      key_file_error: function () {
        if (
          this.config.key_file.length == 0 &&
          this.config.cert_file.length > 0
        )
          return 'Invalid Key File';
        else return '';
      },

      events_interval_error: function () {
        return _.isNumber(this.config.events_interval) &&
          this.config.events_interval > 0
          ? ''
          : 'Invalid Update Interval';
      },
    },
    methods: {
      changeFile: function (title, oldval, callback) {
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst) return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile('Choose CA File', self.config.ca_file, function (path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      changeClientCertFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile(
          'Choose Client Cert File',
          self.config.cert_file,
          function (path) {
            self.$set(self.config, 'cert_file', path);
          }
        );
      },
      changeKeyFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile(
          'Choose Key File',
          self.config.key_file,
          function (path) {
            self.$set(self.config, 'key_file', path);
          }
        );
      },

      isInputsValid: function () {
        return !(this.errors != null && this.errors.length > 0);
      },
      onSaveLocal: function () {
        if (!this.isInputsValid()) {
          msgbus.$emit(
            'alert',
            'Error',
            'Some inputs are invalid, please fix it and try again.',
            'error',
            0
          );
          return;
        }
        this.$emit('brokerUpdated', this.name, cloneDeep(this.config), cloneDeep(this.errors));
      },
      onReset: function () {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', cloneDeep(this.configData));
      },
    },
  }; //}}}1

  let azureIoTHubComp = {
    //{{{1
    template: '#azure_iot_hub_template',
    props: ['brokerName', 'configData', 'nameError', 'errors'],
    data: function () {
      let _data = {
        name: this.brokerName,
        config: cloneDeep(this.configData),
        errors: this.errors,
      };
      if (!_.has(_data.config, 'data_precision'))
        _data.config['data_precision'] = 2;
      return _data;
    },
    computed: {
      errors: function () {
        return _.filter([
          this.name_error,
          this.hub_error,
          this.device_id_error,
          this.sa_key_error,
          this.data_precision_error,
          this.ca_file_error,
          this.events_interval_error,
        ], function (error) { return error != null && error.length > 0; });
      },
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      hub_error: function () {
        return this.config.mqtt_host.length > 0 ? '' : 'Invalid IoT Hub';
      },
      device_id_error: function () {
        return this.config.mqtt_client_id.length > 0 ? '' : 'Invalid Device ID';
      },
      sa_key_error: function () {
        return this.config.sa_key.length > 0 ? '' : 'Invalid Shared Access Key';
      },
      data_precision_error: function () {
        let b = false;
        let precision = this.config.data_precision;
        const pattern = /^[1-6]$/;
        if (precision != null && precision != "") {
          b = pattern.test(precision);
        }
        return b ? "" : "Invalid Precision : Only support [1-6]";
      },

      hub: {
        get: function () {
          var length = this.config.mqtt_host.length;
          if (this.config.mqtt_host.endsWith('.azure-devices.net'))
            return this.config.mqtt_host.substring(
              0,
              length - '.azure-devices.net'.length
            );
          else return this.config.mqtt_host;
        },
        set: function (newVal) {
          if (newVal.endsWith('.azure-devices.net'))
            this.config.mqtt_host = newVal;
          else this.config.mqtt_host = newVal + '.azure-devices.net';
          this.config.mqtt_user =
            this.config.mqtt_host +
            '/' +
            this.config.mqtt_client_id +
            '/?api-version=2018-06-30';
        },
      },
      device_id: {
        get: function () {
          return this.config.mqtt_client_id;
        },
        set: function (newVal) {
          this.config.mqtt_client_id = newVal;
          this.config.mqtt_user =
            this.config.mqtt_host + '/' + newVal + '/?api-version=2018-06-30';
        },
      },

      ca_file_error: function () {
        return this.config.ca_file.length > 0 ? '' : 'Invalid CA File';
      },
      // cert_file_error: function() {
      //   if ( this.config.cert_file.length == 0 && this.config.key_file.length > 0 )
      //     return "Invalid Certificate File";
      //   else
      //     return "";
      // },
      // key_file_error: function() {
      //   if ( this.config.key_file.length == 0 && this.config.cert_file.length > 0 )
      //     return "Invalid Key File";
      //   else
      //     return "";
      // },

      events_interval_error: function () {
        return _.isNumber(this.config.events_interval) &&
          this.config.events_interval > 0
          ? ''
          : 'Invalid Update Interval';
      },
    },
    methods: {
      changeFile: function (title, oldval, callback) {
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst) return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function () {
        if (this.config.no_tls) return;

        var self = this;
        this.changeFile('Choose CA File', self.config.ca_file, function (path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      isInputsValid: function () {
        return !(this.errors != null && this.errors.length > 0);
      },
      onSaveLocal: function () {
        if (!this.isInputsValid()) {
          msgbus.$emit(
            'alert',
            'Error',
            'Some inputs are invalid, please fix it and try again.',
            'error',
            0
          );
          return;
        }
        this.config.mqtt_passwd = this.generateSAS();
        this.$emit('brokerUpdated', this.name, cloneDeep(this.config), cloneDeep(this.errors));
      },
      onReset: function () {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', cloneDeep(this.configData));
      },
      generateSAS: function () {
        var uri =
          this.config.mqtt_host + '/devices/' + this.config.mqtt_client_id;
        var encoded = encodeURIComponent(uri);
        var now = new Date();
        var year = 60 * 60 * 24 * 365;
        var ttl = Math.round(now.getTime() / 1000) + 5 * year;
        var signature = encoded + '\n' + ttl;
        var hash = CryptoJS.HmacSHA256(
          signature,
          CryptoJS.enc.Base64.parse(this.config.sa_key)
        );
        var hashInBase64 = CryptoJS.enc.Base64.stringify(hash);
        return (
          'SharedAccessSignature sr=' +
          encoded +
          '&sig=' +
          encodeURIComponent(hashInBase64) +
          '&se=' +
          ttl
        );
      },
    },
  }; //}}}1

  let propertyListPanelComp = {
    // {{{1
    template: '#property_list_panel_template',
    props: ['objData'],
    data: function () {
      return {
        objDataDomID: {},
        boxOpened: true,
      };
    },
    computed: {
      boxBtnClass: function () {
        if (this.boxOpened) return 'icon-folder-open';
        else return 'icon-folder-close';
      },
    },
    methods: {
      objDataUniqueDomID: function (objDataPath) {
        if (!_.has(this.objDataDomID, objDataPath))
          this.objDataDomID[objDataPath] = _.uniqueId('data_table_');
        return this.objDataDomID[objDataPath];
      },

      addProperty: function () {
        this.objData.slots.push({
          name: '',
          value: '',
          type: 'Str',
          slotType: 'property',
          enabled: true,
          visible: true,
        });
      },

      selectAll: function () {
        _.each(this.objData.slots, function (slot) {
          slot.enabled = true;
        });
      },
      inverseSelect: function () {
        _.each(this.objData.slots, function (slot) {
          slot.enabled = !slot.enabled;
        });
      },
      selectNone: function () {
        _.each(this.objData.slots, function (slot) {
          slot.enabled = false;
        });
      },
      deleteSelectedSlot: function () {
        this.objData.slots = this.objData.slots.filter(function (s) {
          return !s.enabled;
        });
      },
    },
  }; // }}}1

  let nameReplacementPreprocessorComp = {
    // {{{1
    template: '#name_replacement_preprocessor_template',
    props: ['payload', 'enabled'],
    data: function () {
      return {
        tooltip: "change slot's label based on its name and select it",
      };
    },
    created: function () {
      _.defaults(this.payload, { old_name: '', new_name: '' });
    },
    methods: {
      process: function (slot) {
        if (!slot) return;

        if (slot.name === this.payload.old_name) {
          slot.label = this.payload.new_name;
          slot.enabled = true;
        }
      },
    },
  }; //}}}1

  let preselectPreprocessorComp = {
    // {{{1
    template: '#preselect_preprocessor_template',
    props: ['payload', 'enabled'],
    data: function () {
      return {
        tooltip: 'select slot when its name match one of names specified here',
      };
    },
    created: function () {
      _.defaults(this.payload, { name_list: [] });
    },
    methods: {
      process: function (slot) {},
    },
  }; // }}}1

  let preprocessorListComp = {
    // {{{1
    template: '#preprocessor_list_template',
    data: function () {
      return {
        type_list: [
          {
            label: 'Name Replacement',
            icon: 'icon-font',
            component: 'name-replacement-preprocessor-comp',
          },
          // {label: 'Preselection', icon: 'icon-check', component: 'preselect-preprocessor-comp'},
        ],
        instance_list: [],
      };
    },
    methods: {
      addPreprocessorElem: function (type) {
        this.instance_list.push({
          enabled: true,
          component: type.component,
          payload: {},
        });
      },

      iconFromComp: function (component) {
        var type = _.find(this.type_list, function (type) {
          return type.component === component;
        });
        var icon = null;
        if (type) icon = type.icon;
        if (!icon) icon = 'icon-wrench';
        return icon;
      },

      initData: function (data) {
        if (data && _.has(data, 'preprocessor_list'))
          this.initPreprocessorList(data.preprocessor_list);
      },
      initPreprocessorList: function (inst_list) {
        var self = this;
        this.instance_list.splice(0, this.instance_list.length);
        inst_list.forEach(function (inst) {
          var isCompValid = _.any(self.type_list, function (type) {
            return type.component === inst.component;
          });
          if (isCompValid) {
            if (_.isString(inst['enabled']))
              inst['enabled'] = inst['enabled'] == 'true';

            self.instance_list.push(inst);
          }
        });
      },

      toJson: function () {
        return JSON.parse(JSON.stringify(this.instance_list));
      },

      process: function (slot) {
        if (!slot) return;

        _.each(this.$refs.preprocessor, function (preprocessor) {
          if (preprocessor.enabled) preprocessor.process(slot);
        });
      },
    },
    components: {
      'name-replacement-preprocessor-comp': nameReplacementPreprocessorComp,
      'preselect-preprocessor-comp': preselectPreprocessorComp,
    },
  }; //}}}1

  let deviceDataTreeComp = {
    //{{{1
    template: '#device_data_tree_comp_template',
    props: {
      listenMsgBusEvent: {
        type: Boolean,
        default: true,
      },
      showBtns: {
        type: Boolean,
        default: true,
      },
    },
    data: function () {
      return {};
    },
    mounted: function () {
      var self = this;
      $(this.$el)
        .find('.device-data-tree')
        .on('dblclick.jstree', function (event) {
          var tree = $(this).jstree();
          if (tree) self.addObjectToSchema();
        })
        .jstree({
          plugins: ['wholerow', 'sort', 'contextmenu'],
          core: {
            themes: {
              dots: true,
            },
            multiple: false,
            dblclick_toggle: false,
            data: function (node, cb) {
              var url;
              if (node.id === '#') {
                //GOTCHA: for GCP IoT, only expose BACnet data
                if (bacnetOnly)
                  cb.call(this, [
                    {
                      text: 'BACnet',
                      children: true,
                      path: '/EasyIO/BACnetS/BACnetC',
                      state: { selected: true, opened: true },
                    },
                  ]);
                else
                  cb.call(this, [
                    {
                      text: 'Controller',
                      children: true,
                      path: '/',
                      state: { selected: true, opened: true },
                    },
                  ]);
                return;
              } else {
                if (node.original && node.original.path)
                  url = node.original.path;
                else {
                  cb.call(this, false);
                  return;
                }
              }

              self.loadNodeData(url, cb);
            },
          },
          contextmenu: {
            items: {
              reload: {
                separator_before: false,
                separator_after: false,
                _disabled: false,
                icon: 'icon-refresh',
                label: 'Reload',
                action: function (data) {
                  self.reloadTree();
                },
              },
            },
          },
        });

      if (this.listenMsgBusEvent)
        msgbus.$on('focusSedonaObj', _.bind(this.focusObj, this));
    },
    methods: {
      jstree: function () {
        return $(this.$el).find('.device-data-tree').jstree(true);
      },
      loadNodeData: function (nodeURL, cb) {
        var self = this;
        var baseURL = '../../app/data_api.php';

        let url = '/app/objects';
        if (nodeURL == '/') url = url + nodeURL + '*';
        else url = url + nodeURL + '/*';
        $.ajax({
          url: baseURL,
          method: 'GET',
          dataType: 'json',
          data: {
            url: url,
          },
          success: function (data, status, jq) {
            if (data.redirect) redirect(data.redirect);

            if (!data.response || !data.response.data) {
              console.warn('invalid response data format');
              return;
            }

            if (
              !_.isArray(data.response.data) ||
              data.response.data.length <= 0
            ) {
              console.warn('invalid data format');
              return;
            } else {
              if (!_.isArray(data.response.data[0]))
                data.response.data = [data.response.data];
            }

            var treeData = self.convertData(data.response.data);
            if (treeData) cb.call(this, treeData);
          },
          error: function (jq, status, error) {
            alert('got some error: [' + status + '] ' + error);
          },
        });
      },
      convertData: function (data) {
        var treeData = [];
        for (var i = 0; i < data.length; ++i) {
          for (var j = 0; j < data[i].length; ++j) {
            var objData = data[i][j];
            var path = objData.path;
            var name = this.getFriendlyObjectName(objData);

            if (bacnetOnly) {
              //GOTCHA: add BACnet deviceId if possible
              //        this will be used to access data from BACnet shared memory
              //        this should be deleted when we don't get sedona data from
              //        shared memory
              var deviceId = this.getBACnetDeviceId(path);
              if (deviceId) objData['deviceId'] = deviceId;
            }

            var hasChildren =
              _.has(objData, 'childNum') && parseInt(objData.childNum) > 0;
            var icon = 'jstree-folder';
            if (!hasChildren) icon = 'jstree-file';
            var node = {
              id: path,
              path: path,
              text: name,
              children: hasChildren,
              icon: icon,
              raw: objData,
            };
            treeData.push(node);
          }
        }
        return treeData;
      },
      getParentPath: function (path) {
        if (!path) return null;

        var parts = path.split('/');
        parts.pop();
        return parts.join('/');
      },
      getFriendlyObjectName: function (sedonaNodeData) {
        //GOTCHA: since there is no object type info returned from easyioCpt
        //kit, we check the slots here to detect BACnet information
        var bacnetDeviceName = '';
        var bacnetDeviceID = null;
        var bacnetPointName = '';
        var bacnetPointID = null;
        for (var i = 0; i < sedonaNodeData.slots.length; ++i) {
          var slot = sedonaNodeData.slots[i];
          if (slot.name == 'deviceId') bacnetDeviceID = slot.value;
          else if (slot.name == 'deviceName') bacnetDeviceName = slot.value;
          else if (slot.name == 'objectId') bacnetPointID = slot.value;
          else if (slot.name == 'pointName') bacnetPointName = slot.value;
        }

        if (bacnetDeviceID && bacnetDeviceName.length > 0) {
          return bacnetDeviceName + ' [' + bacnetDeviceID + ']';
        } else if (bacnetPointID && bacnetPointName.length > 0) {
          var shortName = getShortTypeName(sedonaNodeData);
          if (shortName)
            return (
              bacnetPointName + ' [' + shortName + ':' + bacnetPointID + ']'
            );
          else return bacnetPointName + ' [' + bacnetPointID + ']';
        } else {
          var name = _.last(sedonaNodeData.path.split('/'));
          return name;
        }
      },
      getBACnetDeviceId: function (path) {
        var bacnetDeviceId = null;

        var jstree = this.jstree();
        var parentId = this.getParentPath(path);
        var parentNode = jstree.get_node(parentId);
        if (!parentNode) return bacnetDeviceId;

        if (
          _.has(parentNode.original, 'raw') &&
          parentNode.original.raw &&
          _.has(parentNode.original.raw, 'slots')
        ) {
          _.each(parentNode.original.raw.slots, function (slot) {
            if (slot.name == 'deviceId') bacnetDeviceId = slot.value;
          });
        }
        return bacnetDeviceId;
      },

      getSelectedObj: function () {
        let jstree = this.jstree();
        let objs = _.map(jstree.get_selected(true), function (node) {
          if (!node.original) return null;
          return node.original;
        });

        objs = _.compact(objs);
        return objs.length > 0 ? objs[0] : null;
      },
      addObjectToSchema: function () {
        let jstree = this.jstree();
        let objs = _.map(jstree.get_selected(true), function (node) {
          if (!node.original) return null;
          return node.original;
        });

        if (
          objs.length == 1 &&
          objs[0].raw &&
          objs[0].raw.type == 'sys::Folder'
        ) {
          jstree.toggle_node(jstree.get_selected());
        } else {
          objs = _.compact(objs);
          if (objs.length > 0) this.$emit('objectsToSchema', objs);
        }
      },
      reloadTree: function () {
        let jst = this.jstree();
        jst.get_selected(true).forEach(function (n) {
          jst.refresh_node(n);
        });
      },

      loadParents: function (path, callback) {
        let pathList = [];
        let index = 0;
        while (true) {
          index = path.indexOf('/', index + 1);
          if (index == -1) break;
          pathList.push(path.substring(0, index));
        }
        if (pathList.length > 0) {
          let jstree = this.jstree();
          jstree.load_node(pathList, callback);
        } else {
          if (callback) callback();
        }
      },
      focusObj: function (path) {
        let self = this;
        let jstree = this.jstree();
        if (jstree.get_node(path)) this.doFocusObj(path);
        else
          this.loadParents(path, function () {
            self.doFocusObj(path);
          });
      },
      doFocusObj: function (path) {
        let jstree = this.jstree();
        let node = jstree.get_node(path);

        if (node) {
          let curSelected = jstree.get_selected(false);
          jstree.deselect_node(curSelected);
          jstree.select_node(node);

          //scroll to the selected node, ref: https://github.com/vakata/jstree/issues/519
          let jstreeDom =
            document.getElementsByClassName('device-data-tree')[0];
          jstreeDom.scrollTop =
            findPos(document.getElementById(path))[1] -
            jstreeDom.offsetHeight / 2;
        }
      },
    },
  }; // }}}1

  let variableDataTreeComp = {
    //{{{1
    template: '#user_variable_tree_comp_template',
    data: function () {
      return {};
    },
    mounted: function () {
      var self = this;
      var elem = $(this.$el)
        .find('#userVariableTree')
        .on('dblclick.jstree', function (event) {
          var tree = $(this).jstree();
          if (tree) self.addObjectToSchema();
        })
        .jstree({
          plugins: ['wholerow', 'types'],
          core: {
            themes: {
              dots: true,
            },
            multiple: false,
            dblclick_toggle: false,
            data: [
              {
                text: 'Publish',
                type: 'category',
                state: { opened: true, selected: true },
                children: [
                  {
                    id: 'pub-label',
                    text: 'label',
                    type: 'user_object',
                    hint: 'user editable text value',
                    raw: {
                      slots: [
                        {
                          name: 'label',
                          slotType: 'property',
                          type: 'string',
                          value: '',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-number',
                    text: 'number',
                    type: 'user_object',
                    hint: 'user editable number value',
                    raw: {
                      slots: [
                        {
                          name: 'number',
                          slotType: 'property',
                          type: 'int',
                          value: 0,
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-boolean',
                    text: 'boolean',
                    type: 'user_object',
                    hint: 'user editable boolean value',
                    raw: {
                      slots: [
                        {
                          name: 'bool',
                          slotType: 'property',
                          type: 'bool',
                          value: false,
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-timestamp',
                    text: 'timestamp',
                    type: 'object',
                    raw: {
                      slots: [
                        {
                          name: 'timestamp',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-version',
                    text: 'version',
                    type: 'object',
                    hint: 'the version of UDMI',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'version',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-operational',
                    text: 'operational',
                    type: 'object',
                    hint: "the controller's operational state",
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'operational',
                          slotType: 'property',
                          type: 'bool',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-last_start',
                    text: 'last_start',
                    type: 'object',
                    hint: "the timestamp device was last turned on",
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'last_start',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-last_config',
                    text: 'last_config_timestamp',
                    type: 'object',
                    hint: 'last timestamp when config data received',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'last_config',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-make',
                    text: 'make',
                    type: 'object',
                    hint: 'the manufacturer of the device',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'make',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },
                  {
                    id: 'pub-state_etag',
                    text: 'state_etag',
                    type: 'object',
                    hint: 'state_etag string in last config data received',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'state_etag',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },
                ],
              },
              {
                text: 'Subscribe',
                type: 'category',
                state: { opened: true },
                children: [
                  {
                    id: 'sub-timestamp',
                    text: 'timestamp',
                    type: 'object',
                    hint: 'timestamp in received message',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'timestamp',
                          slotType: 'property',
                          type: 'timestamp',
                        },
                      ],
                    },
                  },

                  {
                    id: 'sub-version',
                    text: 'version',
                    type: 'object',
                    hint: 'UDMI version',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'version',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },

                  {
                    id: 'sub-sample_limit_sec',
                    text: 'sample_limit_sec',
                    type: 'object',
                    hint: 'min time in second between sample updates.',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'sample_limit_sec',
                          slotType: 'property',
                          type: 'number',
                        },
                      ],
                    },
                  },
                  {
                    id: 'sub-sample_rate_sec',
                    text: 'sample_rate_sec',
                    type: 'object',
                    hint: 'how often to publish events data to broker in second',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'sample_rate_sec',
                          slotType: 'property',
                          type: 'number',
                        },
                      ],
                    },
                  },

                  {
                    id: 'sub-state_etag',
                    text: 'state_etag',
                    type: 'object',
                    hint: 'config state_etag',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'state_etag',
                          slotType: 'property',
                          type: 'string',
                        },
                      ],
                    },
                  },

                  {
                    id: 'sub-min_loglevel',
                    text: 'min_loglevel',
                    type: 'object',
                    hint: 'The minimum loglevel for reporting log messages below which log entries should not be sent.',
                    supported_brokers: ['gcp-iot-core'],
                    raw: {
                      slots: [
                        {
                          name: 'min_loglevel',
                          slotType: 'property',
                          type: 'number',
                        },
                      ],
                    },
                  },
                ],
              },
            ],
          },
          types: {
            '#': {
              valid_children: ['category'],
            },
            category: {
              valid_children: ['category', 'object', 'user_object'],
              icon: 'jstree-folder',
            },
            // for MQTT publish, it means this object will be generated by backend and published to broker
            // for MQTT subscribe, it means this object might exist in subscribed msg
            object: { icon: './img/property.png' },
            user_object: { icon: './img/editable_property.png' },
          },
        });

      msgbus.$on(
        'editNodeForPub',
        _.bind(this.onDataNodeCategoryChanged, this)
      );
    },
    methods: {
      jstree: function () {
        return $(this.$el).find('#userVariableTree').jstree(true);
      },
      addObjectToSchema: function () {
        let jstree = this.jstree();
        var objs = _.map(jstree.get_selected(true), function (node) {
          if (node.type == 'category') {
            jstree.toggle_node(jstree.get_selected());
            return null;
          }
          if (!node.original) return null;

          var orig_data = cloneDeep(node.original);
          var path_prefix = '';
          if (orig_data.type == 'object') path_prefix = '@Var/system';
          else if (orig_data.type == 'user_object') path_prefix = '@Var/user';
          else {
            console.error('invalid node data type: ' + orig_data.type);
            return null;
          }

          //make sure object's path is unique
          orig_data['path'] =
            path_prefix + '/' + orig_data.id + uniqueIdEx('$');
          return orig_data;
        });
        objs = _.compact(objs);
        if (objs.length > 0) this.$emit('objectsToSchema', objs);
      },

      //events handler
      onDataNodeCategoryChanged: function (forPublish) {
        let tree = this.jstree();
        if (!tree) return;

        let root = tree.get_node('#');
        _.each(root.children_d, function (cid) {
          let c = tree.get_node(cid);
          if (c.type != 'object' && c.type != 'user_object') return;

          if (c.id.startsWith('pub-') == forPublish) tree.enable_node(c);
          else tree.disable_node(c);
        });
      },
    },
  }; // }}}1

  let deviceDataPanelComp = {
    //{{{1
    template: '#device_data_panel_template',
    data: function () {
      return {
        tabs: [
          {
            id: 'deviceDataTreeTab',
            label: 'Device Data',
            comp: 'device-tree',
          },
          {
            id: 'userVariableTreeTab',
            label: 'Variables',
            comp: 'variable-tree',
          },
        ],
        active_tab_id: 'deviceDataTreeTab',
      };
    },
    methods: {
      addObjectsToSchema: function (objs) {
        this.$emit('objectsToSchema', objs);
      },
    },

    components: {
      'device-tree': deviceDataTreeComp,
      'variable-tree': variableDataTreeComp,
    },
  }; //}}}1

  let variableSlotsEditComp = {
    //{{{1
    template: '#variable_slots_edit_template',
    inject: ['getGConfig'],
    props: ['nodeID', 'objDataList', 'isPublishTopic'],
    data: function () {
      return {
        errorSlotUID: null,
      };
    },
    mounted: function () {
      msgbus.$on('propRenamed', _.bind(this.onPropRenamed, this));
    },
    methods: {
      isUserVarPath: isUserVarPath,
      isSystemVarPath: isSystemVarPath,
      validateInputs: function () {
        var self = this;
        var isError = _.any(this.objDataList, function (objData) {
          return _.any(objData.slots, function (slot) {
            if (slot.label.length == 0) {
              self.errorSlotUID = slot.uid;
              return true;
            }
            //only validate user var for publish; user var for subscribe
            //doesn't need user to specify a value
            if (self.isPublishTopic && isUserVarPath(slot.path)) {
              if (slot.type == 'string' && slot.value.length == 0) {
                self.errorSlotUID = slot.uid;
                return true;
              }
            }
            return false;
          });
        });
        if (isError) return false;
        else {
          this.errorSlotUID = null;
          return true;
        }
      },
      save: function () {
        var self = this;
        _.each(this.objDataList, function (objData) {
          _.each(objData.slots, function (slot) {
            slot.enabled = true;
          });
        });
        if (this.validateInputs()) {
          this.$emit(
            'nodeSchemaUpdated',
            this.nodeID,
            cloneDeep(this.objDataList)
          );
        } else console.warn('invalid input');
      },

      /////////////// Event Handlers ///////////////
      onPropRenamed: function (slotUID, slotLabel) {
        if (this.selectedSlotUID != slotUID) return;

        _.any(this.objDataList, function (objData) {
          return _.any(objData.slots, function (slot) {
            if (slot.uid == slotUID) {
              slot.label = slotLabel;
              return true;
            } else return false;
          });
        });
      },
    },
  }; //}}}1

  let sedonaSlotsEditComp = {
    //{{{1
    template: '#sedona_slots_edit_template',
    inject: ['getGConfig'],
    props: ['nodeID', 'objDataList', 'initialSelectedSlotId'],
    data: function () {
      return {
        errorSlotUID: null,
        selectedSlotUID: this.initialSelectedSlotId,
      };
    },
    mounted: function () {
      msgbus.$on('propRenamed', _.bind(this.onPropRenamed, this));
    },
    computed: {
      commonBindingPath: function () {
        if (this.objDataList.length < 1) return '';

        var pathPartList = _.map(this.objDataList, function (objData) {
          return objData.path.split('/');
        });
        var commonLen = 0;
        _.any(_.range(pathPartList[0].length - 1), function (index) {
          var part = null;
          var allIncludesPart = _.all(pathPartList, function (partList) {
            if (_.isNull(part)) {
              part = partList[index];
              return true;
            } else return partList[index] == part;
          });
          if (allIncludesPart) commonLen = index + 1;
          return !allIncludesPart;
        });

        if (commonLen < 2) return '';
        else return _.first(pathPartList[0], commonLen).join('/');
      },
    },
    watch: {
      initialSelectedSlotId: function (newVal, oldVal) {
        this.selectedSlotUID = newVal;
      },
      selectedSlotUID: function (newVal, oldVal) {
        var self = this;
        var objData = _.find(this.objDataList, function (objData) {
          return _.any(objData.slots, function (slot) {
            return slot.uid == self.selectedSlotUID;
          });
        });
        if (objData) msgbus.$emit('focusSedonaObj', objData.path);
      },
    },
    methods: {
      bindingTooltip: function (slotData) {
        if (bacnetOnly)
          return slotData.path.replace('/EasyIO/BACnetS/BACnetC/', '/BACnet/');
        else return slotData.path;
      },
      shortBindingPath: function (slotData) {
        var result = slotData.path;
        var commonLen = this.commonBindingPath.split('/').length;
        if (commonLen > 0) {
          var parts = slotData.path.split('/');
          result = _.last(parts, parts.length - commonLen).join('/');
        }
        return result;
      },

      cloudNodeSiblingNames: function () {
        var names = [];
        var cloudTree = $('#dataSchemaTree').jstree(true);
        if (!cloudTree) return names;

        var objNode = null;
        if (cloudTree.get_type(this.nodeID) == 'property')
          objNode = cloudTree.get_node(cloudTree.get_parent(this.nodeID));
        else objNode = cloudTree.get_node(this.nodeID);

        var self = this;
        return _.compact(
          _.map(objNode.children, function (child) {
            if (child == self.nodeID) return '';
            else return cloudTree.get_text(child);
          })
        );
      },
      currentBrokerType: function (propNode) {
        var cloudTree = $('#dataSchemaTree').jstree(true);
        if (!cloudTree) return null;

        var node = cloudTree.get_node(this.nodeID);
        if (!node) return null;

        // 1. find topic name
        while (node.id != '#' && node.type != 'topic') {
          node = cloudTree.get_node(node.parent);
        }

        if (node.type != 'topic') return null;

        // 2. find topic object
        var topic = _.find(gConfig.topicList, function (topic) {
          return topic.name == node.text;
        });
        if (!topic) return null;

        // 3. find broker object
        var brokerName = topic.broker;
        var broker = _.find(gConfig.brokerList, function (broker) {
          return brokerName == broker.name;
        });
        if (!broker) return null;
        return broker.type;
      },
      validateInputs: function () {
        var self = this;
        var curBrokerType = this.currentBrokerType();
        var siblingNames = this.cloudNodeSiblingNames();
        var isValid = _.all(this.objDataList, function (objData) {
          return _.all(objData.slots, function (slot) {
            if (!slot.enabled) return true;

            if (_.contains(siblingNames, slot.label)) {
              self.errorSlotUID = slot.uid;
              return false;
            } else {
              if (slot.label.length == 0) {
                self.errorSlotUID = slot.uid;
                return false;
              } else return true;

              // if (curBrokerType == "gcp-iot-core") {
              //   // for gcp-iot-core, the property's name must be
              //   // 'present_value', more details:
              //   // https://github.com/grafnu/daq/blob/pubber/schemas/udmi/README.md
              //   if (slot.label != "present_value") {
              //     self.errorSlotUID = slot.uid;
              //     return false;
              //   } else
              //     return true;
              // } else
              //   return true;
            }
          });
        });

        if (isValid) this.errorSlotUID = null;
        return isValid;
      },
      save: function () {
        var self = this;
        _.each(this.objDataList, function (objData) {
          _.each(objData.slots, function (slot) {
            slot.enabled = slot.uid == self.selectedSlotUID;
          });
        });
        if (this.validateInputs()) {
          this.$emit(
            'nodeSchemaUpdated',
            this.nodeID,
            cloneDeep(this.objDataList)
          );
        } else console.warn('invalid input');
      },

      locateDataNodeByPath: function () {
        if (this.objDataList.length == 0) return;

        let path = this.objDataList[0].path;
        msgbus.$emit('focusSedonaObj', path);
      },

      /////////////// Event Handlers ///////////////
      onPropRenamed: function (slotUID, slotLabel) {
        if (this.selectedSlotUID != slotUID) return;

        _.any(this.objDataList, function (objData) {
          return _.any(objData.slots, function (slot) {
            if (slot.uid == slotUID) {
              slot.label = slotLabel;
              return true;
            } else return false;
          });
        });
      },
    },
  }; //}}}1

  let nodeSchemaPanelComp = {
    //{{{1
    template: '#node_schema_panel_template',
    inject: ['getGConfig'],
    data: function () {
      return {
        nodeID: null,
        objDataList: [],
        selectedSlotUID: null,
        editMode: false,
        isVarObject: false,
        isPublishTopic: true,
      };
    },
    watch: {
      nodeID: function (newID, oldID) {
        if (newID == oldID || newID == null) return;

        //update 'isPublishTopic'
        this.isPublishTopic = this.getTopicCategory() == 0;
        msgbus.$emit('editNodeForPub', this.isPublishTopic);
      },
    },
    methods: {
      scrollToBottom: function () {
        var elem = this.$el.querySelector('.scrollable-content');
        $(elem).animate({ scrollTop: elem.scrollHeight }, 300);
      },
      scrollToTop: function () {
        var elem = this.$el.querySelector('.scrollable-content');
        $(elem).animate({ scrollTop: 0 }, 300);
      },
      // scrollToSlot: function(slotId) {
      //   var self = this;
      //   _.any(this.$refs.slotPanel, function(slotPanel) {
      //     if (slotPanel.slotData.uid == slotId) {
      //       slotPanel.highlight();
      //       return true;
      //     } else
      //       return false;
      //   });
      // },

      addObjects: function (objs) {
        //clear old data
        this.selectedSlotUID = null;
        this.objDataList.splice(0, this.objDataList.length);

        for (var i = 0; i < objs.length; ++i) {
          var obj = objs[i];
          //objData is stored under 'raw'
          if (!obj.raw) continue;

          //prevent to add duplicate objData
          if (
            _.any(this.objDataList, function (objData) {
              return objData.path === obj.path;
            })
          ) {
            console.info('device data is already added: ' + obj.path);
            continue;
          }

          var objData = {};
          var objPath = obj.path;
          objData['path'] = objPath;
          this.isVarObject = isVarPath(objPath);

          var slots = obj.raw.slots;
          if (bacnetOnly && _.has(obj, 'raw')) {
            objData['bacnetPtType'] = getShortTypeName(obj.raw);
            //GOTCHA: copy over BACnet deviceId
            //        should delete this hack when don't get data from BACnet
            //        shared memory
            if (_.has(obj.raw, 'deviceId'))
              objData['deviceId'] = obj.raw.deviceId;
            _.any(slots, function (slot) {
              if (slot.name == 'objectId') {
                objData['objectId'] = slot.value;
                return true;
              } else return false;
            });
          }

          objData['slots'] = [];
          for (var j = 0; j < slots.length; ++j) {
            var slot = slots[j];

            if (slot.slotType != 'property') continue;

            //FIXME: now we only can fetch BACnet object's data from shared
            //memory, and from shared memory, only following slots' data can be
            //fetched, so only display these slots to user, otherwise user will
            //become confused. this filter need to be removed when we support
            //fetching data from the data layer later
            if (
              bacnetOnly &&
              !_.contains(['objectId', 'pointName', 'presentValue'], slot.name)
            )
              continue;

            var slotCopy = {};
            slotCopy['uid'] = uniqueIdEx('#slot_');
            slotCopy['name'] = slot.name; //keep the sedona original slot name
            slotCopy['label'] = slot.name; //used for display
            slotCopy['path'] = objPath + '.' + slot.name;
            slotCopy['type'] = slot.type;
            slotCopy['enabled'] = this.isVarObject;
            slotCopy['visible'] = true;
            if (this.isVarObject && _.has(slot, 'value'))
              slotCopy['value'] = slot.value;

            objData['slots'].push(slotCopy);
          }
          this.objDataList.push(objData);
          this.$emit('objectAdded', objData);
        }
        var self = this;
        _.any(this.objDataList, function (objData) {
          return _.any(objData.slots, function (slot) {
            if (slot.enabled) {
              self.selectedSlotUID = slot.uid;
              return true;
            } else return false;
          });
        });

        if (
          !this.selectedSlotUID &&
          this.objDataList.length > 0 &&
          this.objDataList[0].slots.length > 0
        )
          this.selectedSlotUID = this.objDataList[0].slots[0].uid;
      },

      saveNodeSchema: function () {
        if (!this.nodeID) return;

        if (this.isVarObject) this.$refs.varsEditor.save();
        else this.$refs.slotsEditor.save();
      },

      updateNodeSchema: function (nodeID, objDataList, slotUID) {
        var self = this;
        this.nodeID = nodeID;
        this.selectedSlotUID = slotUID;
        this.objDataList.splice(0, this.objDataList.length);
        this.editMode = !!slotUID;

        if (objDataList) {
          objDataList.forEach(function (objData) {
            self.isVarObject = isVarPath(objData.path);
            self.objDataList.push(objData);
          });
        }
      },

      onNodeSchemaUpdated: function (nodeID, objDataList) {
        this.$emit('nodeSchemaUpdated', nodeID, objDataList);
      },

      getTopicCategory: function () {
        var cloudTree = $('#dataSchemaTree').jstree(true);
        if (!cloudTree) return 0;

        var node = cloudTree.get_node(this.nodeID);
        if (!node) return 0;

        // 1. find topic name
        while (node.id != '#' && node.type != 'topic') {
          node = cloudTree.get_node(node.parent);
        }

        if (node.type != 'topic') return 0;

        // 2. find topic object
        var topic = _.find(this.getGConfig().topicList, function (topic) {
          return topic.name == node.text;
        });
        if (!topic) return 0;

        return topic.category;
      },
    },
    components: {
      'property-list-panel-comp': propertyListPanelComp,
      'sedona-slots-edit-comp': sedonaSlotsEditComp,
      'variable-slots-edit-comp': variableSlotsEditComp,
    },
  }; // }}}1

  let msgSnippetEditorComp = {
    // {{{1
    template: '#msg_snippet_editor_template',
    props: [
      'createMode',
      'categoryNameList',
      'initTemplateName',
      'initCategoryName',
      'templateNameError',
      'categoryNameError',
    ],
    data: function () {
      return {
        templateName: this.initTemplateName,
        categoryName: this.initCategoryName,
      };
    },
    methods: {
      doModal: function () {
        let self = this;
        $(this.$el).on('shown', function () {
          self.templateName = self.initTemplateName;
          self.categoryName = self.initCategoryName;
        });
        $(this.$el).modal('show');
      },
      onSubmit: function () {
        let result = true;
        if (this.createMode)
          result = this.$parent.createTemplate(
            this.categoryName,
            this.templateName
          );
        else
          result = this.$parent.editTemplate(
            this.categoryName,
            this.templateName
          );
        if (!result) return;

        $(this.$el).modal('hide');
      },
    },
  }; //}}}1

  let msgSnippetPanelComp = {
    // {{{1
    template: '#msg_snippet_panel_template',
    newTemplateJson: null,
    data: function () {
      return {
        editTemplateId: null,
        curTemplateName: 'Template',
        curCategoryName: 'Default',

        categoryNameList: [],

        categoryNameError: '',
        templateNameError: '',
      };
    },
    mounted: function () {
      let self = this;
      $(this.$el)
        .find('.tree')
        .jstree({
          core: {
            multiple: false,
            check_callback: function (op, node, parent, pos, more) {
              if (op == 'rename_node') return self.validateNewName(node, pos);
              else return true;
            },
          },
          plugins: ['wholerow', 'unique', 'types', 'contextmenu'],
          checkbox: {
            whole_node: true,
            keep_selected_style: true,
            three_state: false,
          },
          types: {
            '#': {
              valid_children: ['category', 'snippet'],
            },
            category: {
              valid_children: ['snippet'],
              icon: './img/category.png',
            },
            snippet: {
              valid_children: ['topic', 'object'],
              icon: './img/snippet.png',
            },
            topic: {
              valid_children: ['object', 'property'],
              icon: './img/topic.png',
            },
            object: {
              valid_children: ['object', 'property'],
            },
            property: {
              valid_children: [],
              icon: './img/property.png',
            },
          },
          dnd: {},
          contextmenu: {
            items: function (node, cb) {
              return {
                use: {
                  separator_before: false,
                  separator_after: false,
                  _disabled: function (data) {
                    let jstree = $.jstree.reference(data.reference);
                    let node = jstree.get_node(data.reference);
                    return node.type == 'category';
                  },
                  icon: false,
                  label: 'Use Template',
                  action: function (data) {
                    self.useTemplate();
                  },
                },
                edit: {
                  separator_before: false,
                  separator_after: false,
                  _disabled: function (data) {
                    let jstree = $.jstree.reference(data.reference);
                    let node = jstree.get_node(data.reference);
                    return (
                      (node.type != 'category' && node.type != 'snippet') ||
                      node.text.startsWith('@')
                    );
                  },
                  icon: false,
                  label: 'Edit',
                  action: function (data) {
                    self.editNode();
                  },
                },
                delete: {
                  separator_before: true,
                  separator_after: false,
                  _disabled: function (data) {
                    let jstree = $.jstree.reference(data.reference);
                    let node = jstree.get_node(data.reference);
                    return (
                      (node.type != 'category' && node.type != 'snippet') ||
                      node.text.startsWith('@')
                    );
                  },
                  icon: 'icon-trash',
                  label: function (data) {
                    let jstree = $.jstree.reference(data.reference);
                    let node = jstree.get_node(data.reference);
                    let disabled =
                      (node.type != 'category' && node.type != 'snippet') ||
                      node.text.startsWith('@');
                    if (disabled) return "<span class='muted'>Delete</span>";
                    else return "<span class='text-error'>Delete</span>";
                  },
                  action: function (data) {
                    self.removeNode();
                  },
                },
              };
            },
          },
        });
    },
    methods: {
      jstree: function () {
        return $(this.$el).find('.tree').jstree(true);
      },
      initData: function (json) {
        let jstree = this.jstree();
        if (json) {
          if (_.isArray(json))
            json.forEach(function (child) {
              jstree.create_node('#', child);
            });
          else jstree.create_node('#', json);
        } else {
          jstree.create_node('#', {
            text: 'Default',
            type: 'category',
          });
        }

        jstree.open_node('#');
      },
      validateNewName: function (node, name) {
        return this.isNameValid(name);
      },
      editNode: function () {
        let jstree = this.jstree();
        let node = jstree.get_selected();
        let type = jstree.get_type(node);
        let text = jstree.get_text(node);
        if (type != 'category' && type != 'snippet') return;
        if (text.startsWith('@')) return;

        if (type == 'category') {
          jstree.edit(node);
        } else {
          this.$newTemplateJson = null;

          this.templateNameError = '';
          this.categoryNameError = '';

          this.curTemplateName = jstree.get_text(node);
          this.curCategoryName = jstree.get_text(jstree.get_parent(node));
          this.editTemplateId = node;

          this.updateCategoryNameList();
          this.$refs.msgSnippetEditor.doModal();
        }
      },
      removeNode: function () {
        let jstree = this.jstree();
        let node = jstree.get_selected();
        let type = jstree.get_type(node);
        let text = jstree.get_text(node);
        if (type != 'category' && type != 'snippet') return;
        if (text.startsWith('@')) return;

        if (
          window.confirm(
            "Are you sure to delete '" + jstree.get_text(node) + "' ?"
          )
        )
          jstree.delete_node(node);
      },

      updateCategoryNameList: function () {
        let jstree = this.jstree();
        if (!jstree) return;

        let root = jstree.get_node('#');
        this.categoryNameList = root.children_d
          .filter(function (nid) {
            return (
              jstree.get_type(nid) == 'category' &&
              !jstree.get_text(nid).startsWith('@')
            );
          })
          .map(function (nid) {
            return jstree.get_text(nid);
          });
      },
      curCategoryId: function () {
        let jstree = this.jstree();
        let root = jstree.get_node('#');
        return root.children.find(function (nid) {
          return jstree.get_type(nid) == 'category' && jstree.is_selected(nid);
        });
      },
      categoryIdByName: function (name) {
        let jstree = this.jstree();
        let root = jstree.get_node('#');
        return root.children.find(function (nid) {
          return (
            jstree.get_type(nid) == 'category' && jstree.get_text(nid) == name
          );
        });
      },
      //template name should be unique within one category
      isTemplateNameUnique: function (categoryName, templateName) {
        let catId = this.categoryIdByName(categoryName);
        if (!catId)
          // category is new
          return true;

        let jstree = this.jstree();
        let catNode = jstree.get_node(catId);
        return (
          catNode.children
            .filter(function (child) {
              return jstree.get_type(child) == 'snippet';
            })
            .map(function (child) {
              return jstree.get_text(child);
            })
            .indexOf(templateName) == -1
        );
      },
      isNameValid: function (name) {
        return /^\w{1,24}$/.test(name);
      },
      validateNames: function (categoryName, templateName) {
        this.templateNameError = '';
        this.categoryNameError = '';

        if (!categoryName || categoryName.length == 0)
          this.categoryNameError = 'Category name can not be empty.';
        else if (!templateName || templateName.length == 0)
          this.templateNameError = 'Template name can not be empty.';
        else {
          if (!this.isNameValid(categoryName))
            this.categoryNameError =
              'Only alphanumeric and underscore characters(1~24) allowed.';
          else if (!this.isNameValid(templateName))
            this.templateNameError =
              'Only alphanumeric and underscore characters(1~24) allowed.';
          else {
            let needToCheckNameUnique = true;
            if (this.editTemplateId !== null) {
              let jstree = this.jstree();
              let node = jstree.get_node(this.editTemplateId);
              let catId = this.categoryIdByName(categoryName);

              if (jstree.get_parent(node) != catId)
                // if category changed
                needToCheckNameUnique = true;
              else
                needToCheckNameUnique =
                  templateName != jstree.get_text(this.editTemplateId);
            }
            if (
              needToCheckNameUnique &&
              !this.isTemplateNameUnique(categoryName, templateName)
            )
              this.templateNameError = 'Template name is already used.';
          }
        }
        return (
          this.templateNameError.length == 0 &&
          this.categoryNameError.length == 0
        );
      },
      newTemplate: function (json) {
        if (!json) return;

        this.$newTemplateJson = json;
        this.curTemplateName = 'Template';

        this.templateNameError = '';
        this.categoryNameError = '';

        this.editTemplateId = null;
        this.updateCategoryNameList();

        let curCatId = this.curCategoryId();
        if (curCatId) {
          let curCatName = this.jstree().get_text(curCatId);
          if (!curCatName.startsWith('@')) this.curCategoryName = curCatName;
        } else {
          if (
            this.curCategoryName.length == 0 &&
            this.categoryNameList.length > 0
          )
            this.curCategoryName = this.categoryNameList[0];
        }
        this.$refs.msgSnippetEditor.doModal();
      },
      createTemplate: function (categoryName, templateName) {
        if (!this.validateNames(categoryName, templateName)) return false;

        let jstree = this.jstree();
        let catId = this.categoryIdByName(categoryName);
        if (catId === undefined)
          catId = jstree.create_node('#', {
            text: categoryName,
            type: 'category',
          });
        let templateId = jstree.create_node(catId, {
          text: templateName,
          type: 'snippet',
        });
        convertNodeUIDs(this.$newTemplateJson);
        jstree.create_node(templateId, this.$newTemplateJson);

        this.$newTemplateJson = null;

        //select the new category node
        jstree.deselect_all(true);
        jstree.select_node(catId);
        jstree.open_node(catId);
        return true;
      },
      editTemplate: function (categoryName, templateName) {
        if (!this.validateNames(categoryName, templateName)) return false;

        let jstree = this.jstree();
        let catId = this.categoryIdByName(categoryName);
        if (catId === undefined)
          catId = jstree.create_node('#', {
            text: categoryName,
            type: 'category',
          });

        let node = jstree.get_node(this.editTemplateId);
        if (templateName != jstree.get_text(node))
          jstree.rename_node(node, templateName);

        let parentId = jstree.get_parent(this.editTemplateId);
        if (parentId != catId) {
          jstree.move_node(this.editTemplateId, catId, 'last');
          jstree.open_node(catId);
        }

        return true;
      },

      useTemplate: function () {
        let jstree = this.jstree();
        let node = jstree.get_node(jstree.get_selected());

        // if there are no or multiple selected nodes, give up here
        if (!node) return;

        if (node.type == 'category') return;

        if (node.type != 'snippet') {
          if (node.parents.length < 3) return;
          node = jstree.get_node(node.parents[node.parents.length - 3]);
        }

        if (node.type != 'snippet' || node.children.length != 1) return;

        //the template node is the first child of snippet node
        node = jstree.get_node(node.children[0]);
        this.$emit('useTemplate', getTemplateJson(jstree, node));
      },
      templateJson: function () {
        let jstree = this.jstree();

        let options = {
          no_state: true,
          no_li_attr: true,
          no_a_attr: true,
        };
        let json = jstree.get_json('#', options);

        if (_.isArray(json))
          json.forEach(function (snippetJson) {
            addDataBindingData(jstree, snippetJson);
          });
        else addDataBindingData(jstree, json);
        return json;
      },
    },
    components: {
      'msg-snippet-editor': msgSnippetEditorComp,
    },
  }; //}}}1

  let dataBindingEditorComp = {
    // {{[1
    template: '#data_binding_editor_template',
    data: function () {
      return {
        spinner: null,

        topNode: null,
        origBinding: '', // the original binding path
        srcBinding: '', // the selected binding path to be replaced
        dstBinding: '', // the new binding path to replace srcBinding

        error: '',
      };
    },
    computed: {
      possibleTopBindings: function () {
        let parts = this.origBinding.split('/');
        if (parts.length < 2) return [];

        let result = [];
        for (let i = 2; i <= parts.length; i++) {
          result.unshift(parts.slice(0, i).join('/'));
        }
        return result;
      },
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },
      hasCommonDataBinding: function (jstree, node) {
        if (jstree.get_type(node) == 'property')
          node = jstree.get_node(jstree.get_parent(node));

        let bindingList = collectAllDataBindings(jstree, node, true);
        return getCommonDataPath(bindingList).length > 0;
      },
      showEditor: function (jstree, node) {
        this.origBinding = '';
        this.srcBinding = '';
        this.topNode = null;

        if (jstree.get_type(node) == 'property')
          node = jstree.get_node(jstree.get_parent(node));

        this.topNode = node;
        let bindingList = collectAllDataBindings(jstree, node, true);
        this.origBinding = getCommonDataPath(bindingList);
        this.srcBinding = this.origBinding;
        if (this.dstBinding.length == 0) this.dstBinding = this.srcBinding;

        $(this.$el).modal('show');
        this.$refs.deviceDataTree.focusObj(this.dstBinding);
      },
      hideEditor: function () {
        $(this.$el).modal('hide');
      },

      validateNewBinding: function (
        jstree,
        topNode,
        srcBinding,
        dstBinding,
        successCallback
      ) {
        let self = this;
        let regex = new RegExp('^' + srcBinding + '\\b');
        let src2DstPath = function (path) {
          return path.replace(regex, dstBinding);
        };
        let newBindingUrls = collectAllDataBindings(jstree, topNode, true).map(
          src2DstPath
        );
        fetchDataBindings(
          topNode,
          newBindingUrls,
          function () {
            self.getSpinner().spin(self.$el);
          },
          function (topNode, availableDataMapping, errorDataList) {
            self.getSpinner().stop();

            self.doValidateNewBinding(
              jstree,
              topNode,
              src2DstPath,
              availableDataMapping,
              errorDataList
            );
            if (self.error.length == 0 && successCallback) successCallback();
          },
          function () {
            self.getSpinner().stop();
          }
        );
      },
      doValidateNewBinding: function (
        jstree,
        topNode,
        src2DstPathConverter,
        availableDataMapping,
        errorDataList
      ) {
        this.error = '';
        //validate is there is missing data point
        if (errorDataList && errorDataList.length > 0) {
          this.error = 'Missing data points: ' + errorDataList.join(', ');
          return;
        }

        // validate if there is missing slot
        let nodeIdList = [topNode.id].concat(topNode.children_d);
        let errors = [];
        nodeIdList.forEach(function (cid) {
          let cnode = jstree.get_node(cid);
          let isDataNode =
            cnode !== undefined &&
            cnode.hasOwnProperty('original') &&
            cnode.original.hasOwnProperty('objDataList');
          if (!isDataNode) return;

          cnode.original.objDataList.forEach(function (objData) {
            if (isVarPath(objData.path)) return;

            let newPath = src2DstPathConverter(objData.path);
            if (availableDataMapping.hasOwnProperty(newPath)) {
              objData.slots.forEach(function (s) {
                let sepIndex = s.path.lastIndexOf('.');
                if (sepIndex == -1) return;

                //GOTCHA: under some unknown case, s.name will be the same as
                //s.label, that will cause the data binding check fail. as a
                //workaround, we extract the slot name from path and use it
                //for check
                let slotName = s.path.substring(sepIndex + 1);
                let valid =
                  availableDataMapping[newPath].indexOf(slotName) != -1;
                if (!valid)
                  errors.push(
                    `Missing slot "${slotName}" on data point "${newPath}"`
                  );
              });
            } else {
              errors.push(`Missing data point: "${newPath}"`);
            }
          });
        });

        if (errors.length == 0) this.$emit('dataBingingOk');
        else this.error = errors.join('; ');
      },
      replaceDataBinding: function (objData, srcBinding, dstBinding) {
        if (!objData || !objData.path || isVarPath(objData.path)) return;

        let regex = new RegExp('^' + srcBinding + '\\b');
        objData.path = objData.path.replace(regex, dstBinding);

        objData.slots.forEach(function (s) {
          s.path = s.path.replace(regex, dstBinding);
        });
      },
      modifyDataBinding: function (jstree, topNode, srcBinding, dstBinding) {
        //get all node ids including all descendents
        let nodeIdList = [topNode.id].concat(topNode.children_d);
        let self = this;
        nodeIdList.forEach(function (cid) {
          let cnode = jstree.get_node(cid);
          let isDataNode =
            cnode !== undefined &&
            cnode.hasOwnProperty('original') &&
            cnode.original.hasOwnProperty('objDataList');
          if (!isDataNode) return;

          cnode.original.objDataList.forEach(function (objData) {
            if (isVarPath(objData.path)) return;
            self.replaceDataBinding(objData, srcBinding, dstBinding);
          });
        });
        return true;
      },

      onSubmit: function () {
        let dataObj = this.$refs.deviceDataTree.getSelectedObj();
        if (!dataObj || !dataObj.path) {
          this.error = 'Selected data point is invalid.';
          return;
        }

        this.dstBinding = dataObj.path;
        this.$emit(
          'dataBindingChanged',
          this.topNode,
          this.srcBinding,
          this.dstBinding
        );
      },
    },
    components: {
      'device-tree': deviceDataTreeComp,
    },
  }; // }}}1

  let dataSchemaPanelComp = {
    // {{{1
    template: '#data_schema_panel_template',
    inject: ['getTopicList', 'getGConfig'],
    data: function () {
      return {
        spinner: null,
        splitter: null,

        selectedTopicName: 'NoMQTTTopic',
        availableTopicNames: [],
        newTopicTreeMode: true,
      };
    },
    mounted: function () {
      let self = this;
      let sizes = localStorage.getItem('data-schema-split-sizes');
      if (sizes) sizes = JSON.parse(sizes);
      else sizes = [80, 20];
      self.splitter = window.Split(
        ['#dataSchemaTreeContainer', '#msgTmplTree'],
        {
          sizes: sizes,
          minSize: 0,
          direction: 'vertical',
          gutterSize: 6,
          snapOffset: 50,
          gutterAlign: 'start',
          onDragEnd: function () {
            let sizes = self.splitter.getSizes();
            localStorage.setItem(
              'data-schema-split-sizes',
              JSON.stringify(sizes)
            );
          },
        }
      );

      $(this.$el)
        .find('#dataSchemaTree')
        .on('select_node.jstree', function (event, data) {
          if (!data.node) return;

          self.onNodeSelected(data.node);
        })
        .on('rename_node.jstree', function (event, data) {
          var jstree = self.jstree();
          if (!data.node || jstree.get_type(data.node) != 'property') return;

          var parentNode = jstree.get_node(data.node.parent);
          _.any(parentNode.original['objDataList'], function (objData) {
            return _.any(objData.slots, function (slot) {
              if (slot.enabled && slot.uid === data.node.id) {
                slot.label = data.text;
                msgbus.$emit('propRenamed', slot.uid, slot.label);
                return true;
              } else return false;
            });
          });
        })
        .on('delete_node.jstree', function (event, data) {
          var jstree = self.jstree();
          if (!data.node) return;

          var parentNode = jstree.get_node(data.parent);
          if (jstree.get_type(data.node) != 'property') {
            jstree.select_node(parentNode);
            return;
          }

          parentNode.original.objDataList = _.reject(
            parentNode.original.objDataList,
            function (objData) {
              return _.any(objData.slots, function (slot) {
                return slot.enabled && slot.uid === data.node.id;
              });
            }
          );
          jstree.select_node(parentNode);
        })
        .jstree({
          core: {
            multiple: false,
            check_callback: function (op, node, parent, pos, more) {
              if (op == 'rename_node') return self.validateNewName(node, pos);
              if (op == 'create_node') {
                let jstree = self.jstree();
                let names = parent.children.map(function (child) {
                  return jstree.get_text(child);
                });
                if (names.indexOf(node.text) != -1)
                  node.text = uniqueName(node.text, names);
              } else return true;
            },
            // 'data' : [{'id': '#device', 'text': 'device', 'type': 'object', 'parent': '#', 'state': {'selected': true, 'opened': true} }],
          },
          plugins: ['wholerow', 'dnd', 'unique', 'types', 'contextmenu'],
          checkbox: {
            whole_node: true,
            keep_selected_style: true,
            three_state: false,
          },
          types: {
            '#': {
              valid_children: ['topic'],
            },
            topic: {
              valid_children: ['object', 'property'],
              icon: './img/topic.png',
            },
            object: {
              valid_children: ['object', 'property'],
            },
            property: {
              valid_children: [],
              icon: './img/property.png',
            },
          },
          dnd: {
            is_draggable: function (nodes, event) {
              return !_.any(nodes, function (node) {
                return node.type == 'property';
              });
            },
          },
          contextmenu: {
            items: {
              // new: {
              //   separator_before: false,
              //   separator_after: false,
              //   _disabled: false,
              //   icon: 'icon-plus', //TODO: fix icon display issue
              //   label: 'New Folder',
              //   action: function (data) {
              //     self.newNode();
              //   },
              // },
              edit: {
                separator_before: false,
                separator_after: false,
                _disabled: false,
                icon: 'icon-edit',
                label: 'Rename',
                action: function (data) {
                  self.editNode();
                },
              },
              expand: {
                separator_before: false,
                separator_after: false,
                _disabled: false,
                icon: 'icon-resize-vertical',
                label: function (data) {
                  let jstree = $.jstree.reference(data.reference);
                  let node = jstree.get_node(data.reference);
                  return jstree.is_open(node) ? 'Collapse' : 'Expand';
                },
                action: function (data) {
                  let jstree = $.jstree.reference(data.reference);
                  let node = jstree.get_node(data.reference);
                  self.toggleNode();
                },
              },
              changeDataBinding: {
                separator_before: true,
                separator_after: false,
                _disabled: function (data) {
                  let jstree = $.jstree.reference(data.reference);
                  let node = jstree.get_node(data.reference);
                  if (node.type == 'property') return true;
                  return !self.hasCommonDataBinding(node);
                },
                icon: 'icon-magnet',
                label: 'Change DataBinding',
                action: function (data) {
                  self.changeDataBinding();
                },
              },
              validate: {
                separator_before: false,
                separator_after: true,
                _disabled: false,
                icon: 'icon-eye-open',
                label: 'Validate DataBinding',
                action: function (data) {
                  self.validateSelectedMsg();
                },
              },
              clone: {
                separator_before: false,
                separator_after: false,
                _disabled: function (data) {
                  let jstree = $.jstree.reference(data.reference);
                  let node = jstree.get_node(data.reference);
                  return node.type != 'object';
                },
                icon: 'icon-file',
                label: 'Clone',
                action: function (data) {
                  self.cloneNode(1);
                },
              },
              cloneX: {
                separator_before: false,
                separator_after: false,
                _disabled: function (data) {
                  let jstree = $.jstree.reference(data.reference);
                  let node = jstree.get_node(data.reference);
                  return node.type != 'object';
                },
                icon: 'icon-file',
                label: 'CloneMulti',
                submenu: {
                  '2copy': {
                    separator_before: false,
                    separator_after: false,
                    _disabled: false,
                    icon: 'icon-file',
                    label: '2 Times',
                    action: function (data) {
                      self.cloneNode(2);
                    },
                  },
                  '4copy': {
                    separator_before: false,
                    separator_after: false,
                    _disabled: false,
                    icon: 'icon-file',
                    label: '4 Times',
                    action: function (data) {
                      self.cloneNode(4);
                    },
                  },
                  '8copy': {
                    separator_before: false,
                    separator_after: false,
                    _disabled: false,
                    icon: 'icon-file',
                    label: '8 Times',
                    action: function (data) {
                      self.cloneNode(8);
                    },
                  },
                  '12copy': {
                    separator_before: false,
                    separator_after: false,
                    _disabled: false,
                    icon: 'icon-file',
                    label: '12 Times',
                    action: function (data) {
                      self.cloneNode(12);
                    },
                  },
                },
              },
              save_template: {
                separator_before: true,
                separator_after: false,
                _disabled: function (data) {
                  let jstree = $.jstree.reference(data.reference);
                  let node = jstree.get_node(data.reference);
                  return node.type != 'topic' && node.type != 'object';
                },
                icon: 'icon-th-large',
                label: 'Save As Template',
                action: function (data) {
                  self.saveAsTemplate();
                },
              },
              delete: {
                separator_before: true,
                separator_after: false,
                _disabled: false,
                icon: 'icon-trash',
                label: "<span class='text-error'>Delete</span>",
                action: function (data) {
                  self.removeNode();
                },
              },
            },
          },
        });
      msgbus.$on('topicDeleted', _.bind(this.onTopicDeleted, this));
      msgbus.$on('topicRenamed', _.bind(this.onTopicRenamed, this));
      msgbus.$on(
        'topicCategoryChanged',
        _.bind(this.onTopicCategoryChanged, this)
      );
    },
    methods: {
      jstree: function () {
        return $(this.$el).find('#dataSchemaTree').jstree(true);
      },
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },
      initData: function (data) {
        var jstree = this.jstree();
        if (data) {
          if (!_.has(data, 'jstree_data')) {
            msgbus.$emit('alert', 'Error', 'Invalid tree data.', 'error');
            return;
          }

          var jstree_data = data.jstree_data;
          if (_.isArray(jstree_data))
            _.each(jstree_data, function (a_tree_data) {
              jstree.create_node('#', a_tree_data);
            });
          else jstree.create_node('#', jstree_data);
          this.onNodeSelected(_.last(jstree.get_selected(true)));
        } else {
          var init_root_data = {
            id: '#mqtt_topic_0',
            text: 'NoMQTTTopic',
            type: 'topic',
            parent: '#',
            state: { selected: false, opened: true },
          };
          jstree.create_node('#', init_root_data);
          var init_device_data = {
            id: '#device_1',
            text: 'device',
            type: 'object',
            parent: '#mqtt_topic_0',
            state: { opened: true },
          };
          var device_node = jstree.create_node(
            '#mqtt_topic_0',
            init_device_data
          );
          jstree.select_node(device_node);
        }
      },
      initTemplateData: function (data) {
        this.$refs.msgSnippetPanel.initData(data);
      },
      newNode: function () {
        var jstree = this.jstree();
        var curNode = _.last(jstree.get_selected());
        if (!curNode) return;

        //always select a object type node as parent
        if (jstree.get_type(curNode) == 'property')
          curNode = jstree.get_parent(curNode);

        var newId = jstree.create_node(curNode, { type: 'object' });
        jstree.deselect_all();
        jstree.select_node(newId);
        jstree.edit(newId, 'new_node');
      },
      pickTopic: function () {
        var jstree = this.jstree();
        var rootNode = jstree.get_node('#');
        var usedTopicList = _.map(rootNode.children, function (childId) {
          return jstree.get_text(childId);
        });

        this.availableTopicNames = _.difference(
          _.map(this.getTopicList(), function (topic) {
            return topic.name;
          }),
          usedTopicList
        );
        this.selectedTopicName = _.first(this.availableTopicNames);

        $(this.$el).find('#msg_topic_picker_dialog').modal('show');
      },
      newTopicTree: function () {
        this.newTopicTreeMode = true;
        this.pickTopic();
      },
      getTopicCategory: function (topicName) {
        var topic = _.find(this.getTopicList(), function (t) {
          return t.name == topicName;
        });
        if (topic && topic.category !== undefined) return topic.category;
        else return 0;
      },

      createTopicTreeNode: function () {
        $(this.$el).find('.modal').modal('hide');

        if (!this.selectedTopicName) return;

        var jstree = this.jstree();
        var rootNode = jstree.get_node('#');
        var newId = uniqueId('#mqtt_topic', rootNode.children);

        var iconPath =
          './img/topic_' +
          this.getTopicCategory(this.selectedTopicName) +
          '.png';
        var initRootData = {
          id: newId,
          text: this.selectedTopicName,
          type: 'topic',
          icon: iconPath,
          parent: '#',
          state: { selected: true, opened: true },
        };
        jstree.create_node('#', initRootData);
        jstree.deselect_all();
        jstree.select_node(newId);
      },
      changeTopicTreeNode: function () {
        $(this.$el).find('.modal').modal('hide');

        if (!this.selectedTopicName) return;

        var jstree = this.jstree();
        var selected = jstree.get_selected();
        if (selected.length > 0) selected = _.last(selected);

        if (jstree.get_type(selected) != 'topic') return;

        var iconPath =
          './img/topic_' +
          this.getTopicCategory(this.selectedTopicName) +
          '.png';
        jstree.set_icon(selected, iconPath);
        jstree.rename_node(selected, this.selectedTopicName);
      },
      editNode: function () {
        var jstree = this.jstree();
        var selected = jstree.get_selected();
        if (selected.length > 0) selected = _.last(selected);

        if (jstree.get_type(selected) == 'topic') {
          this.newTopicTreeMode = false;
          this.pickTopic();
        } else jstree.edit(selected);
      },
      removeNode: function () {
        var jstree = this.jstree();
        let node = jstree.get_selected();
        if (
          !node ||
          !window.confirm(
            "Are you sure to delete node '" + jstree.get_text(node) + "' ?"
          )
        )
          return;

        jstree.delete_node(node);
      },
      cloneNode: function (times) {
        let jstree = this.jstree();
        let node = jstree.get_node(jstree.get_selected());
        let type = jstree.get_type(node);
        if (type != 'object') {
          alert("only can clone 'folder' node");
          return;
        }

        let name = jstree.get_text(node);
        let parentNode = jstree.get_node(jstree.get_parent(node));
        let nameList = parentNode.children.map(function (childId) {
          return jstree.get_text(childId);
        });

        let json = getTemplateJson(jstree, node, true);
        for (var i = 0; i < times; ++i) {
          convertNodeUIDs(json);
          let newName = uniqueName(name, nameList);
          json.text = newName;
          let newNode = jstree.create_node(parentNode, json, 'last');
          jstree.open_node(newNode);
          nameList.push(newName);
        }
      },
      toggleNode: function () {
        let jstree = this.jstree();
        let node = jstree.get_node(jstree.get_selected());
        if (jstree.is_open(node)) jstree.close_node(node);
        else jstree.open_all(node);
      },

      //collectDataBindings: function (topNode) {
      //  let jstree = this.jstree();

      //  //collect data binding paths
      //  let nodeIdList = [topNode.id].concat(topNode.children_d);
      //  let dataPathMappings = nodeIdList.reduce(function (init, cid) {
      //    let cnode = jstree.get_node(cid);
      //    let isDataNode =
      //      cnode !== undefined &&
      //      cnode.hasOwnProperty('original') &&
      //      cnode.original.hasOwnProperty('objDataList');
      //    if (!isDataNode) return init;

      //    cnode.original.objDataList.forEach(function (objData) {
      //      if (isVarPath(objData.path)) return;

      //      if (!init.hasOwnProperty(objData.path)) init[objData.path] = [];

      //      objData.slots
      //        .filter(function (s) {
      //          return s.enabled;
      //        })
      //        .forEach(function (s) {
      //          if (init[objData.path].indexOf(s.name) != -1) return;
      //          init[objData.path].push(s.name);
      //        });
      //    });

      //    return init;
      //  }, {});

      //  return Object.keys(dataPathMappings).map(function (compPath) {
      //    let slots = dataPathMappings[compPath];
      //    // to make request url shorter, when there are too many(>6) slots, we request all slots
      //    // GOTCHA: because a bug in easyioCpt kit, when request
      //    // /Folder/Add2.in1+in2+out, and the slot are missing, the path in
      //    // error response will be '/Folder/Add2', '/Folder/Add2/Add2',
      //    // '/Folder/Add2/Add2'. So we will fetch slots by path '/Folder/Add2'
      //    return compPath;
      //    // if (slots.length > 6)
      //    //   return compPath;
      //    // else
      //    //   return compPath + '.' + slots.join('+');
      //  });
      //},
      markNode: function (nid, valid) {
        let jstree = this.jstree();
        let node = jstree.get_node(nid);
        jstree.set_icon(
          node,
          valid ? './img/property.png' : './img/invalid_property.png'
        );
        //expand invalid nodes
        if (!valid)
          node.parents.forEach(function (pid) {
            jstree.open_node(pid);
          });
      },
      pathStartsWith: function (src, prefix) {
        return prefix == src || src.startsWith(prefix + '/');
      },
      markInvalidDataBindings: function (
        topNode,
        availableDataMapping,
        errorDataList
      ) {
        let self = this;
        let jstree = this.jstree();
        let nodeIdList = [topNode.id].concat(topNode.children_d);
        nodeIdList
          .filter(function (cid) {
            let type = jstree.get_type(cid);
            return type == 'topic' || type == 'object';
          })
          .forEach(function (cid) {
            let node = jstree.get_node(cid);
            if (!node.original || !node.original.objDataList) return;

            node.original.objDataList.forEach(function (objData) {
              if (isVarPath(objData.path)) return;

              if (
                errorDataList.some((errPath) =>
                  self.pathStartsWith(objData.path, errPath)
                )
              ) {
                objData.slots.forEach(function (s) {
                  if (!s.enabled) return;

                  self.markNode(s.uid, false);
                });
              }

              // if errorDataList not empty, skip other checks
              if (errorDataList.length > 0) return;

              if (availableDataMapping.hasOwnProperty(objData.path)) {
                objData.slots.forEach(function (s) {
                  if (!s.enabled) return;

                  let sepIndex = s.path.lastIndexOf('.');
                  if (sepIndex == -1) return;

                  //GOTCHA: under some unknown case, s.name will be the same as
                  //s.label, that will cause the data binding check fail. as a
                  //workaround, we extract the slot name from path and use it
                  //for check
                  let slotName = s.path.substring(sepIndex + 1);
                  let valid =
                    availableDataMapping[objData.path].indexOf(slotName) != -1;
                  self.markNode(s.uid, valid);
                });
              } else {
                objData.slots.forEach(function (s) {
                  if (!s.enabled) return;

                  self.markNode(s.uid, false);
                });
              }
            });
          });
      },

      doValidateDataBindings: function (node, bindingUrls) {
        let self = this;
        fetchDataBindings(
          node,
          bindingUrls,
          function () {
            self.getSpinner().spin(self.$el);
          },
          function (topNode, availableDataMapping, errorDataList) {
            self.getSpinner().stop();
            self.markInvalidDataBindings(
              topNode,
              availableDataMapping,
              errorDataList
            );
          },
          function () {
            self.getSpinner().stop();
          }
        );
      },

      validateMsgs: function () {
        let jstree = this.jstree();
        let topNode = jstree.get_node('#');
        let bindingUrls = collectAllDataBindings(jstree, topNode, true);
        this.doValidateDataBindings(topNode, bindingUrls);
      },
      validateSingleMsg: function (node) {
        let jstree = this.jstree();
        while (node.id != '#' && jstree.get_type(node) != 'topic') {
          //get the 'topic' parent node
          node = jstree.get_node(node.parent);
        }

        if (jstree.get_type(node) != 'topic') return;

        let topNode = node;
        let bindingUrls = collectAllDataBindings(jstree, topNode, true);
        this.doValidateDataBindings(topNode, bindingUrls);
      },
      validateSelectedMsg: function () {
        let jstree = this.jstree();
        let selectedList = jstree.get_selected(true);
        let self = this;
        selectedList.forEach(function (selectedNode) {
          self.validateSingleMsg(selectedNode);
        });
      },

      brokerType: function (node) {
        var jstree = this.jstree();
        // 1. find topic name
        while (node.id != '#' && node.type != 'topic') {
          node = jstree.get_node(node.parent);
        }

        if (node.type != 'topic') return null;

        var gConfig = this.getGConfig();
        // 2. find topic object
        var topic = _.find(gConfig.topicList, function (topic) {
          return topic.name == node.text;
        });
        if (!topic) return null;

        // 3. find broker object
        var brokerName = topic.broker;
        var broker = _.find(gConfig.brokerList, function (broker) {
          return brokerName == broker.name;
        });
        if (!broker) return null;
        return broker.type;
      },

      validateNewName: function (node, new_name) {
        return /^[-_a-zA-Z0-9]{1,128}$/.test(new_name);

        // let user to make sure the name is valid
        //         var jstree = this.jstree();
        //         var n = jstree.get_node(node);
        //         if (this.brokerType(n) != 'gcp-iot-core') return true;

        //         if (n.type == 'property') return true;
        //         else if (n.type == 'object')
        //           return /^[a-z][a-z0-9]*(_[a-z0-9]+)*$/.test(new_name);
        //         else return true;
      },

      updateModelData: function (node) {
        var jstree = this.jstree();
        if (
          jstree.get_type(node) != 'object' &&
          jstree.get_type(node) != 'topic'
        )
          return;

        var self = this;
        if (_.has(node.original, 'objDataList')) {
          var propNodeIds = _.select(node.children, function (child) {
            return jstree.get_type(child) == 'property';
          });
          node.original.objDataList = _.reject(
            node.original.objDataList,
            function (objData) {
              return !_.any(objData.slots, function (slot) {
                return slot.enabled && _.includes(propNodeIds, slot.uid);
              });
            }
          );
        }

        _.each(
          _.select(node.children, function (child) {
            return jstree.get_type(child) == 'object';
          }),
          function (child) {
            self.updateModelData(jstree.get_node(child));
          }
        );
      },

      onNodeSelected: function (node) {
        if (!node) return;

        var jstree = this.jstree();

        var slotUID = null;
        var objDataList = [];
        if (jstree.get_type(node) == 'property') {
          slotUID = node.id;
          var objNode = jstree.get_node(node.parent);

          _.any(objNode.original['objDataList'], function (objData) {
            if (
              _.any(objData.slots, function (slot) {
                return slot.enabled && slot.uid == slotUID;
              })
            ) {
              objDataList.push(cloneDeep(objData));
              return true;
            } else return false;
          });
        }

        this.$emit('nodeChanged', node.id, objDataList, slotUID);
      },

      addProperty: function (nodeID, slotData) {
        var jstree = this.jstree();
        if (jstree.get_type(nodeID) == 'property')
          nodeID = jstree.get_parent(nodeID);

        var node = jstree.get_node(nodeID);
        if (!node) {
          console.warn('can not find node by ID: ' + nodeID);
          return false;
        }

        if (jstree.get_node(slotData.uid)) {
          console.info('node alreay exists');
          return false;
        }

        var propData = {
          id: slotData.uid,
          text: slotData.label,
          type: 'property',
          parent: nodeID,
          state: { selected: false, opened: false },
        };
        if (!jstree.create_node(node, propData)) {
          //if failed, then most likely the prop'name is not unique, let's
          //generate a unique name now
          var childrenNames = _.map(node.children, function (child) {
            return jstree.get_text(child);
          });
          slotData.label = uniqueName(slotData.label, childrenNames);
          propData.text = slotData.label;
          if (!jstree.create_node(node, propData)) {
            console.warn(
              'can not create property node even with a unique name'
            );
            return false;
          }
        }

        if (!jstree.is_open(node)) jstree.open_node(node);

        return true;
      },

      syncNodeSchema: function (nodeID, objDataList) {
        var self = this;
        var jstree = this.jstree();
        var node = jstree.get_node(nodeID);
        if (node.type == 'object' || node.type == 'topic') {
          if (!_.has(node.original, 'objDataList'))
            node.original['objDataList'] = [];

          _.each(objDataList, function (objData) {
            _.any(objData.slots, function (slot) {
              if (slot.enabled) {
                if (self.addProperty(nodeID, slot))
                  node.original['objDataList'].push(objData);
                return true;
              } else return false;
            });
          });
        } else if (node.type == 'property') {
          //update property node
          var objNode = jstree.get_node(node.parent);
          if (!_.has(objNode.original, 'objDataList'))
            objNode.original['objDataList'] = [];

          //remove old objData first
          objNode.original.objDataList = _.reject(
            objNode.original.objDataList,
            function (objData) {
              return _.any(objData.slots, function (slot) {
                return slot.enabled && slot.uid == node.id;
              });
            }
          );

          //update objData and jstree node
          _.each(objDataList, function (objData) {
            _.any(objData.slots, function (slot) {
              if (slot.enabled) {
                objNode.original.objDataList.push(objData);
                jstree.set_id(node, slot.uid);
                jstree.rename_node(slot.uid, slot.label);
                node = jstree.get_node(slot.uid);
                return true;
              } else return false;
            });
          });

          this.onNodeSelected(node);
        }
      },

      jstreeJson: function () {
        var self = this;
        var jstree = this.jstree();

        this.updateModelData(jstree.get_node('#'));

        var tree_json = jstree.get_json('#');
        //GOTCHA: must copy objDataList from original to node object, otherwise
        //        it will not be dumped to json later
        if (_.isArray(tree_json))
          _.each(tree_json, function (topic_json) {
            self.mergeSlotData(jstree, topic_json);
          });
        else this.mergeSlotData(jstree, tree_json);

        return tree_json;
      },
      mappingList: function (tree_json) {
        if (_.isNull(tree_json)) tree_json = this.jstreeJson();

        let list = [];
        if (!_.isArray(tree_json)) {
          console.warn('invalid tree json format');
          return list;
        }

        var self = this;
        _.each(tree_json, function (topic_json) {
          var topic = topic_json.text;
          list.push(
            _.extend({ topic: topic }, self.buildMappingData(topic_json))
          );
        });
        return list;
      },
      mappingJson: function (tree_json) {
        var json = { topics: [], mappings: {} };
        if (!_.isArray(tree_json)) {
          console.warn('invalid tree json format');
          return json;
        }

        var self = this;
        _.each(tree_json, function (topic_json) {
          var topic = topic_json.text;
          json.topics.push(topic);
          _.extend(json.mappings, self.buildMappingData(topic_json));
        });
        return json;
      },
      mergeSlotData: function (jstree, node_json) {
        if (!node_json) return;

        var node = jstree.get_node(node_json.id);
        if (_.has(node.original, 'objDataList'))
          node_json['objDataList'] = node.original['objDataList'];

        for (var i = 0; i < node_json.children.length; ++i) {
          this.mergeSlotData(jstree, node_json.children[i]);
        }
      },
      buildOneNodeMappingData: function (tree_node) {
        var node_data = {};
        if (!tree_node) return node_data;

        if (
          _.has(tree_node, 'objDataList') &&
          _.isArray(tree_node.objDataList)
        ) {
          for (var i = 0; i < tree_node.objDataList.length; ++i) {
            var objData = tree_node.objDataList[i];
            if (!_.has(objData, 'slots') || !_.isArray(objData.slots)) continue;

            var compType = _.has(objData, 'compType') ? objData.compType : null;
            var slots = objData.slots;

            if (bacnetOnly) {
              //GOTCHA: collect 'objectId' and 'deviceId' value in BACnet network
              //        this is used to locate the BACnet data point
              //        this should be deleted when we don't get sedona data from
              //        shared memory
              var bacnetData = {};
              if (_.has(objData, 'deviceId') && _.has(objData, 'objectId')) {
                bacnetData['bacnet'] =
                  objData.deviceId + ':' + objData.objectId;
              }

              if (_.has(objData, 'bacnetPtType'))
                bacnetData['bacnetPtType'] = objData.bacnetPtType;
            }

            for (var j = 0; j < slots.length; ++j) {
              var slot = slots[j];
              if (!slot.enabled) continue;

              if (compType == 'propertyList')
                node_data[slot.label] = slot.value;
              else {
                node_data[slot.label] = {
                  isLeaf: true,
                  path: slot.path,
                  type: slot.type,
                  // slotType: slot.slotType
                };

                //add more data for user variable slot
                if (isUserVarPath(slot.path)) {
                  node_data[slot.label]['value'] = slot.value;
                }

                if (bacnetOnly) {
                  //GOTCHA: BACnet hack
                  //        this should be deleted when we don't get sedona data from
                  //        shared memory
                  if (!_.isEmpty(bacnetData))
                    _.extend(node_data[slot.label], bacnetData);
                }
              }
            }
          }
        }

        if (_.has(tree_node, 'children') && _.isArray(tree_node.children)) {
          for (var k = 0; k < tree_node.children.length; ++k) {
            var child_node = tree_node.children[k];
            var child_data = this.buildOneNodeMappingData(child_node);
            if (!_.isEmpty(child_data)) node_data[child_node.text] = child_data;
          }
        }
        return node_data;
      },
      buildMappingData: function (tree_json) {
        var device_json = {};
        if (!tree_json) return device_json;

        var node_data = this.buildOneNodeMappingData(tree_json);
        if (!_.isEmpty(node_data)) {
          device_json[tree_json.text] = node_data;
        }
        return device_json;
      },

      mockPreviewValue: function (value, key) {
        if (_.isObject(value)) {
          if (_.has(value, 'isLeaf') && value.isLeaf) {
            if (_.has(value, 'value')) return value.value;

            //prepare mock data for preview
            if (_.contains(['float', 'double'], value.type)) return 42.0;
            else if (_.contains(['int', 'byte', 'short', 'long'], value.type))
              return 42;
            else if (value.type == 'bool') return true;
            else return ('value_of_' + key).toUpperCase();
          } else return mapObject(value, _.bind(this.mockPreviewValue, this));
        } else if (_.isArray(value))
          return _.map(value, _.bind(this.mockPreviewValue, this));
        else return value;
      },

      previewJsons: function () {
        var result = [];
        var tree_json = this.jstreeJson();
        if (!tree_json) return result;
        var mapping_json = this.mappingJson(tree_json);
        if (
          !mapping_json ||
          !_.has(mapping_json, 'mappings') ||
          !_.has(mapping_json, 'topics')
        )
          return result;

        var self = this;
        return _.compact(
          _.map(mapping_json.topics, function (topic) {
            if (!_.has(mapping_json.mappings, topic)) return null;
            var mapping = mapping_json.mappings[topic];
            return [
              topic,
              mapObject(mapping, _.bind(self.mockPreviewValue, self)),
            ];
          })
        );
      },

      topicNodes: function () {
        let jstree = this.jstree();
        let rootNode = jstree.get_node('#');
        return _.select(rootNode.children, function (childId) {
          return jstree.get_type(childId) == 'topic';
        });
      },
      topicNodeByName: function (name) {
        let jstree = this.jstree();
        let idList = this.topicNodes();
        for (let i = 0; i < idList.length; ++i) {
          //TODO: this may changed when we display not only the topic name in jstree node later
          if (jstree.get_text(idList[i]) == name)
            return jstree.get_node(idList[i]);
        }
        return null;
      },
      onTopicDeleted: function (name) {
        let jstree = this.jstree();
        let idList = this.topicNodes();
        for (let i = 0; i < idList.length; ++i) {
          if (jstree.get_text(idList[i]) != name) continue;

          let topicNamePlaceholder = uniqueName(
            'NoMQTTTopic',
            _.map(idList, function (id) {
              return jstree.get_text(id);
            })
          );
          jstree.set_text(idList[i], topicNamePlaceholder);
          return;
        }
      },
      onTopicRenamed: function (oldName, newName) {
        let jstree = this.jstree();
        let topicNode = this.topicNodeByName(oldName);
        if (topicNode) jstree.rename_node(topicNode, newName);
      },
      onTopicCategoryChanged: function (name, oldCategory, newCategory) {
        let jstree = this.jstree();
        let topicNode = this.topicNodeByName(name);
        if (topicNode)
          jstree.set_icon(topicNode, './img/topic_' + newCategory + '.png');
      },

      templateJson: function () {
        return this.$refs.msgSnippetPanel.templateJson();
      },
      saveAsTemplate: function () {
        let jstree = this.jstree();
        let node = jstree.get_node(jstree.get_selected());
        let type = jstree.get_type(node);
        if (type != 'topic' && type != 'object') {
          console.warn(
            "only can create template from 'topic' or 'object' ndoe"
          );
          return;
        }

        this.$refs.msgSnippetPanel.newTemplate(getTemplateJson(jstree, node));
      },
      onUseTemplate: function (json) {
        //      6. more action buttons in template sidebar
        //      7. select template in menu
        let jstree = this.jstree();
        let selected = jstree.get_node(jstree.get_selected());
        if (!selected) return;

        convertNodeUIDs(json);
        if (json.type == 'topic') {
          json.children.forEach(function (child) {
            jstree.create_node(selected, child);
          });

          if (json.objDataList) {
            if (selected.original.objDataList)
              json.objDataList.forEach(function (objData) {
                selected.original.objDataList.push(objData);
              });
            else selected.original['objDataList'] = json.objDataList;
          }
        } else {
          jstree.create_node(selected, json);
        }

        let bindingUrls = collectAllDataBindings(jstree, selected, true);
        this.doValidateDataBindings(selected, bindingUrls);

        jstree.open_all(selected);
      },

      hasCommonDataBinding: function (node) {
        let jstree = this.jstree();
        return this.$refs.dataBindingEditor.hasCommonDataBinding(jstree, node);
      },
      changeDataBinding: function () {
        let jstree = this.jstree();
        let node = jstree.get_node(jstree.get_selected());
        if (!node) return;

        return this.$refs.dataBindingEditor.showEditor(jstree, node);
      },
      onDataBindingChanged: function (topNode, srcBinding, dstBinding) {
        //the topNode maybe not the same as the current selected node, because
        //if the selected node is a property node, we will need to go to its
        //parent node
        let jstree = this.jstree();
        let editor = this.$refs.dataBindingEditor;
        editor.validateNewBinding(
          jstree,
          topNode,
          srcBinding,
          dstBinding,
          function () {
            let success = editor.modifyDataBinding(
              jstree,
              topNode,
              srcBinding,
              dstBinding
            );
            if (!success) {
              alert(
                'Failed to modify data binding, please make sure the selected data point is correct.'
              );
              return;
            }
            editor.hideEditor();
          }
        );
      },
    },
    components: {
      'msg-snippet-panel-comp': msgSnippetPanelComp,
      'data-binding-editor-comp': dataBindingEditorComp,
    },
  }; // }}}1

  let schemaPreviewEntryComp = {
    //{{{1
    template:
      '<div class="schema-preview-entry"><p><img src="./img/topic.png" alt="MQTT Topic"><strong>{{topic}}</strong></p></div>',
    props: ['topic', 'json', 'category'],
    data: function () {
      return {};
    },
    mounted: function () {
      if (!this.json) return;
      this.renderJson();
    },
    watch: {
      json: function (newVal, oldVal) {
        this.renderJson();
      },
    },
    methods: {
      renderJson: function () {
        $(this.$el).find('.renderjson').remove();
        var elem = renderjson.set_show_to_level('all')(this.json);
        this.$el.appendChild(elem);
      },
    },
  };
  let schemaPreviewPanelComp = {
    template: '#schema_preview_panel_template',
    props: ['schema_jsons'],
    data: function () {
      return {};
    },
    components: {
      'schema-preview-entry': schemaPreviewEntryComp,
    },
  };
  //}}}1

  let dataMappingComp = {
    //{{{1
    template: '#data_mapping_template',
    data: function () {
      return {
        spinner: null,
        splitter: null,
        active_tab_id: 'mapping_editor',
        tabs: [
          { id: 'mapping_editor', label: 'Message Editor' },
          { id: 'schema_preview', label: 'Preview' },
          { id: 'preprocessor', label: 'Preprocessor' },
        ],
        previewJsons: [],
      };
    },
    mounted: function () {
      let self = this;
      let sizes = localStorage.getItem('data-mapping-split-sizes');
      if (sizes) sizes = JSON.parse(sizes);
      else sizes = [33, 40, 27];
      this.splitter = window.Split(
        ['#dataSchemaPanel', '#nodeSchemaPanel', '#deviceDataPanel'],
        {
          sizes: sizes,
          minSize: 50,
          direction: 'horizontal',
          gutterSize: 6,
          onDragEnd: function () {
            let sizes = self.splitter.getSizes();
            localStorage.setItem(
              'data-mapping-split-sizes',
              JSON.stringify(sizes)
            );
          },
        }
      );
    },
    watch: {
      active_tab_id: function (newVal, oldVal) {
        if (newVal == 'schema_preview') this.updatePreviewJson();
      },
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },

      addObjectsToSchema: function (objs) {
        this.$refs.nodeSchemaPanel.addObjects(objs);
      },

      onNodeChanged: function (nodeID, objDataList, slotUID) {
        this.$refs.nodeSchemaPanel.updateNodeSchema(
          nodeID,
          objDataList,
          slotUID
        );
      },

      onNodeSchemaUpdated: function (nodeID, objDataList) {
        this.$refs.dataSchemaPanel.syncNodeSchema(nodeID, objDataList);
      },

      onObjectAdded: function (objData) {
        var preprocessorList = this.$refs.preprocessorListPanel;
        if (preprocessorList && objData && _.has(objData, 'slots')) {
          _.each(objData.slots, function (slot) {
            preprocessorList.process(slot);
          });
        }
      },

      onShowTab: function (event) {
        this.active_tab_id = $(event.target).data('tab-id');
      },

      updatePreviewJson: function () {
        if (!this.$refs.dataSchemaPanel) this.previewJsons = [];
        else this.previewJsons = this.$refs.dataSchemaPanel.previewJsons();
      },

      getMappingData: function () {
        var tree_json = this.$refs.dataSchemaPanel.jstreeJson();
        if (!tree_json) return;
        return this.$refs.dataSchemaPanel.mappingList(tree_json);
      },
      getConfigData: function () {
        let config_json = {};
        let tree_json = this.$refs.dataSchemaPanel.jstreeJson();
        if (!tree_json) return config_json;
        config_json['jstree_data'] = tree_json;
        config_json['preprocessor_list'] =
          this.$refs.preprocessorListPanel.toJson();
        return config_json;
      },
      getTemplateData: function () {
        return this.$refs.dataSchemaPanel.templateJson();
      },

      initData: function (data) {
        this.$refs.dataSchemaPanel.initData(data);
        this.$refs.preprocessorListPanel.initData(data);
      },
      initTemplateData: function (data) {
        this.$refs.dataSchemaPanel.initTemplateData(data);
      },
    },
    components: {
      'device-data-panel': deviceDataPanelComp,
      'node-schema-panel': nodeSchemaPanelComp,
      'data-schema-panel': dataSchemaPanelComp,
      'preprocess-list-panel': preprocessorListComp,
      'schema-preview-panel': schemaPreviewPanelComp,
    },
  }; //}}}1

  let brokerManager = {
    //{{{1
    template: '#broker_manager_template',
    inject: ['getGConfig'],
    data: function () {
      return {
        brokerTypeList: brokerTypeList,
        activeBroker: null,
        brokerList: this.getGConfig().brokerList,
        newBroker: {
          name: '',
          type: 'mqtt',
          config: {},
          error: '',
        },
        nameError: '',
        sortField: null,
        sortAscending: true,
      };
    },
    created: function () {
      if (this.brokerList.length > 0) this.activeBroker = this.brokerList[0];
    },
    watch: {
      brokerList: function (val) {
        if (this.brokerList.length == 0) return;

        if (_.isNull(this.activeBroker))
          this.activeBroker = _.first(this.brokerList);
      },
    },
    methods: {
      addBroker: function () {
        this.newBroker.name = uniqueName(
          'NewBroker',
          _.map(this.brokerList, function (broker) {
            return broker.name;
          })
        );
        this.newBroker.uniqueKey = uniqueId(
          'BrokerKey',
          _.map(this.brokerList, function (broker) {
            return broker.uniqueKey;
          })
        );
        this.newBroker.error = '';
        this.newBroker.config = {};
        $(this.$el).find('#new_broker_dlg').modal('show');
        //FIXME: only work for the first time
        $(this.$el).find('#inputBrokerName').focus();
      },
      createBroker: function () {
        if (!this.isNewNameValid(this.newBroker.name)) {
          return;
        }

        // init broker's config
        // common init
        this.newBroker.config['mqtt_host'] = '';
        this.newBroker.config['mqtt_port'] = 8883;
        this.newBroker.config['mqtt_user'] = '';
        this.newBroker.config['mqtt_passwd'] = '';
        this.newBroker.config['mqtt_client_id'] = '';
        this.newBroker.config['data_precision'] = 2;
        this.newBroker.config['mqtt_qos'] = '1';
        this.newBroker.config['keepalive'] = 60;

        //data publish interval
        this.newBroker.config['events_interval'] = 30;

        this.newBroker.config['ca_file'] = '';
        this.newBroker.config['cert_file'] = '';
        this.newBroker.config['key_file'] = '';

        if (this.newBroker.type == 'gcp-iot-core') {
          //GCP IoT Core config
          this.newBroker.config['project_id'] = '';
          this.newBroker.config['registry_id'] = '';
          this.newBroker.config['device_id'] = '';
          this.newBroker.config['region'] = 'us-central1';

          //comm config
          this.newBroker.config['mqtt_host'] = 'mqtt.googleapis.com';
          this.newBroker.config['mqtt_port'] = 8883;

          //tls config
          this.newBroker.config['no_tls'] = false;
        } else if (this.newBroker.type == 'aws-iot-core') {
          this.newBroker.config['mqtt_host'] = '';
          this.newBroker.config['mqtt_port'] = 8883;

          //tls config
          this.newBroker.config['no_tls'] = false;
        } else if (this.newBroker.type == 'azure-iot-hub') {
          //NOTE: for azure iot hub, the ca file is fixed, the ca file is
          // //shipped together with the plugin
          // this.newBroker.config['ca_file'] = '_DigiCertBaltimoreRoot.crt';
          this.newBroker.config['sa_key'] = '';
          this.newBroker.config['no_tls'] = false;
        } else {
          //for general mqtt broker
          this.newBroker.config['mqtt_host'] = 'test.mosquitto.org';
          this.newBroker.config['mqtt_port'] = 8883;

          //tls config
          this.newBroker.config['no_tls'] = false;
        }
        this.newBroker.config['enabled'] = true;

        $(this.$el).find('.modal').modal('hide');
        this.brokerList.push(cloneDeep(this.newBroker));
        //focus the new added broker
        this.activeBroker = _.last(this.brokerList);
      },

      onBrokerUpdated: function (name, config, errors) {
        let activeBroker = this.activeBroker;
        if (!activeBroker) return;

        //validate if the new name is unique
        let isNameUnique = _.all(this.brokerList, function (broker) {
          if (broker.uniqueKey == activeBroker.uniqueKey) return true;
          return broker.name != name;
        });
        if (!isNameUnique) {
          this.nameError = 'The new broker name is not unique';
          return;
        } else this.nameError = '';

        if (config.data_precision != null && config.data_precision != '') {
          config.data_precision = parseInt(config.data_precision);
        } else {
          config.data_precision = 2;
        }

        let oldName = this.activeBroker.name;
        this.$set(this.activeBroker, 'name', name);

        //keep a copy of old data for computing updates
        let oldProxiedDevices = this.activeBroker.config.proxied_devices
          ? cloneDeep(this.activeBroker.config.proxied_devices)
          : null;

        this.$set(this.activeBroker, 'config', cloneDeep(config));
        this.$set(this.activeBroker, 'errors', cloneDeep(errors));

        if (name != oldName) msgbus.$emit('brokerRenamed', oldName, name);

        //detect the update on proxied_devices and trigger events
        if (oldProxiedDevices) {
          //for deleted devices
          oldProxiedDevices
            .filter((d) => {
              if (!config.proxied_devices) return true;
              let notfound =
                config.proxied_devices.findIndex((d2) => d2.uid == d.uid) < 0;
              return notfound;
            })
            .forEach((d) => msgbus.$emit('brokerDeviceDeleted', name, d.id));

          //for renamed devices
          if (config.proxied_devices) {
            oldProxiedDevices.forEach((d) => {
              let dNew = config.proxied_devices.find(
                (d2) => d2.uid == d.uid && d2.id != d.id
              );
              if (dNew)
                msgbus.$emit('brokerDeviceRenamed', name, d.id, dNew.id);
            });
          }
        }
      },

      cloneBroker: function () {
        this.newBroker.name = uniqueName(
          'ClonedBroker',
          _.map(this.brokerList, function (broker) {
            return broker.name;
          })
        );
        this.newBroker.uniqueKey = uniqueId(
          'BrokerKey',
          _.map(this.brokerList, function (broker) {
            return broker.uniqueKey;
          })
        );
        this.newBroker.type = this.activeBroker.type;
        this.newBroker.error = '';
        this.newBroker.config = cloneDeep(this.activeBroker.config);

        //clear device cert and key files
        this.newBroker.config['cert_file'] = '';
        this.newBroker.config['key_file'] = '';

        if (this.newBroker.type == 'gcp-iot-core') {
          this.newBroker.config['device_id'] = '';
          this.newBroker.config['gateway_enabled'] = false;
          this.newBroker.config['proxied_devices'] = [];
        } else if (this.newBroker.type == 'aws-iot-core') {
        } else if (this.newBroker.type == 'azure-iot-hub') {
          this.newBroker.config['device_id'] = '';
          this.newBroker.config['mqtt_user'] = '';
          this.newBroker.config['mqtt_passwd'] = '';
          this.newBroker.config['sa_key'] = '';
        }

        this.brokerList.push(cloneDeep(this.newBroker));
      },

      deleteBroker: function () {
        if (
          !confirm(
            'Are you sure to delete this broker(all its topics will be deleted together) ?'
          )
        )
          return;

        let name = this.activeBroker.name;
        for (let i = 0; i < this.brokerList.length; ++i) {
          if (this.brokerList[i].name != this.activeBroker.name) continue;

          if (this.brokerList.length > 1)
            this.activeBroker =
              this.brokerList[(i + 1) % this.brokerList.length];
          else this.activeBroker = { name: null, type: null };
          this.brokerList.splice(i, 1);
        }
        msgbus.$emit('brokerDeleted', name);

        if (this.brokerList.length == 0) this.activeBroker = null;
      },

      isNewNameValid: function (newName) {
        if (!isNameValid(newName)) {
          this.newBroker.error =
            "Name can only contain alphanumber, '-' and '_', 32 chars at most.";
          return false;
        }

        if (
          !_.all(this.brokerList, function (broker) {
            return newName != broker.name;
          })
        )
          this.newBroker.error = 'Name already exists';
        else this.newBroker.error = '';
        return this.newBroker.error.length == 0;
      },

      sortBy: function (field, ascending) {
        this.sortField = field;
        this.sortAscending = ascending;
        if (ascending)
          this.brokerList.sort(function (a, b) {
            return a[field] > b[field] ? -1 : 1;
          });
        else
          this.brokerList.sort(function (a, b) {
            return a[field] < b[field] ? -1 : 1;
          });
      },
    },
    filters: {
      typeLabel: brokerTypeLabel,
    },
    components: {
      'gcp-iot-core': gcpIoTCoreComp,
      mqtt: mqttComp,
      'aws-iot-core': awsIoTCoreComp,
      'azure-iot-hub': azureIoTHubComp,
    },
  }; //}}}1

  let gcpIoTCoreTopicComp = {
    //{{{1
    template: '#gcp_iot_core_topic_template',
    props: {
      topicName: String,
      topicCategory: Number,
      configData: Object,
      nameError: String,
      proxiedDeviceError: String,
      broker: String,
      proxiedDeviceId: {
        type: String,
        default: '',
        required: false,
      },
    },
    inject: ['getGConfig'],
    data: function () {
      return {
        name: this.topicName,
        category: this.topicCategory,
        config: cloneDeep(this.configData),
        proxiedDeviceIdCopy: this.proxiedDeviceId,
        subPath: '',
      };
    },
    mounted: function () {
      if (_.isEmpty(this.config)) this.$set(this.config, 'path', '');
      this.path = this.config.path;
    },
    watch: {
      category: function (newVal, oldVal) {
        if (newVal == 1) this.subPath = '';
      },
      configData: function (newVal) {
        this.config = cloneDeep(newVal);
      },
    },
    computed: {
      proxied_devices: function () {
        let self = this;
        let gConfig = this.getGConfig();
        let brokerConf = _.find(gConfig.brokerList, function (broker) {
          return broker.name == self.broker;
        });
        if (
          !brokerConf ||
          !brokerConf.config ||
          !brokerConf.config.proxied_devices
        )
          return [];

        return brokerConf.config.proxied_devices;
      },
      pathPrefix: function () {
        let self = this;
        let gConfig = this.getGConfig();
        let brokerConf = _.find(gConfig.brokerList, function (broker) {
          return broker.name == self.broker;
        });
        if (!brokerConf) return '';

        let device_id =
          this.proxiedDeviceIdCopy && this.proxiedDeviceIdCopy.length > 0
            ? this.proxiedDeviceIdCopy
            : brokerConf.config['device_id'];
        return '/devices/' + device_id + '/';
      },
      path: {
        get: function () {
          if (this.category == 1) return this.pathPrefix + 'config';
          else return this.pathPrefix + this.subPath;
        },
        set: function (newVal) {
          let prefix = this.pathPrefix;
          if (newVal.startsWith(prefix))
            this.subPath = newVal.substring(prefix.length);
          else this.subPath = '';
        },
      },
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      path_error: function () {
        if (this.category == 0) {
          // for public, both state and events are ok
          if (this.subPath != 'state' && !this.subPath.startsWith('events'))
            return 'Invalid Path for Publish';
        } else if (this.category == 1) {
          // for subscribe, only one path
          return '';
        } else {
          return 'Invalid Category';
        }
        if (this.subPath.length <= 0) return 'Topic path can not be empty';

        return '';
      },
    },
    methods: {
      onSaveLocal: function () {
        if (this.name_error.length != 0 || this.path_error.length != 0) return;

        this.config.path = this.path;
        this.$emit(
          'topicUpdated',
          this.name,
          this.category,
          cloneDeep(this.config),
          this.proxiedDeviceIdCopy
        );
      },
      onReset: function () {
        this.$set(this, 'name', this.topicName);
        this.$set(this, 'category', this.topicCategory);
        this.$set(this, 'config', cloneDeep(this.configData));
        this.$set(this, 'proxiedDeviceIdCopy', this.proxiedDeviceId);
        this.path = this.config.path;
        this.$emit('topicReset');
      },
    },
  }; //}}}1

  let azureIoTHubTopicComp = {
    //{{{1
    template: '#azure_iot_hub_topic_template',
    props: ['topicName', 'topicCategory', 'configData', 'nameError', 'broker'],
    inject: ['getGConfig'],
    data: function () {
      return {
        name: this.topicName,
        category: this.topicCategory,
      };
    },
    computed: {
      path: function () {
        let self = this;
        let gConfig = this.getGConfig();
        let brokerConf = _.find(gConfig.brokerList, function (broker) {
          return broker.name == self.broker;
        });
        if (!brokerConf) return '';

        if (this.category == 0)
          return (
            'devices/' +
            brokerConf.config['mqtt_client_id'] +
            '/messages/events'
          );
        else
          return (
            'devices/' +
            brokerConf.config['mqtt_client_id'] +
            '/messages/devicebound/#'
          );
      },
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
    },
    methods: {
      onSaveLocal: function () {
        this.$emit('topicUpdated', this.name, this.category);
      },
      onReset: function () {
        this.$set(this, 'name', this.topicName);
      },
    },
  }; //}}}1

  let mqttTopicComp = {
    //{{{1
    template: '#mqtt_topic_template',
    props: ['topicName', 'topicCategory', 'configData', 'nameError'],
    data: function () {
      return {
        name: this.topicName,
        category: this.topicCategory,
        config: cloneDeep(this.configData),
      };
    },
    mounted: function () {
      if (_.isEmpty(this.config)) this.$set(this.config, 'path', '');
    },
    computed: {
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      path_error: function () {
        return this.config.path.length > 0 ? '' : 'Invalid Topic Path';
      },
    },
    methods: {
      onSaveLocal: function () {
        this.$emit(
          'topicUpdated',
          this.name,
          this.category,
          cloneDeep(this.config)
        );
      },
      onReset: function () {
        this.$set(this, 'name', this.topicName);
        this.$set(this, 'category', this.topicCategory);
        this.$set(this, 'config', cloneDeep(this.configData));
      },
    },
  }; //}}}1

  let awsIoTCoreTopicComp = {
    //{{{1
    template: '#aws_iot_core_topic_template',
    props: ['topicName', 'topicCategory', 'configData', 'nameError'],
    data: function () {
      return {
        name: this.topicName,
        category: this.topicCategory,
        config: cloneDeep(this.configData),
      };
    },
    mounted: function () {
      if (_.isEmpty(this.config)) this.$set(this.config, 'path', '');
    },
    computed: {
      name_error: function () {
        if (this.nameError.length > 0) return this.nameError;

        return isNameValid(this.name)
          ? ''
          : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      path_error: function () {
        return this.config.path.length > 0 ? '' : 'Invalid Topic Path';
      },
    },
    methods: {
      onSaveLocal: function () {
        this.$emit(
          'topicUpdated',
          this.name,
          this.category,
          cloneDeep(this.config)
        );
      },
      onReset: function () {
        this.$set(this, 'name', this.topicName);
        this.$set(this, 'category', this.topicCategory);
        this.$set(this, 'config', cloneDeep(this.configData));
      },
    },
  }; //}}}1

  let topicManager = {
    //{{{1
    template: '#topic_manager_template',
    inject: ['getGConfig'],
    data: function () {
      return {
        topicList: this.getGConfig().topicList,
        brokerList: this.getGConfig().brokerList,
        activeTopic: null,
        newTopic: {
          name: 'NewTopic',
          enabled: true,
          broker: null,
          proxied_device: null,
          category: 0, // 0 for publish topic; 1 for subscribe topic
          config: {},
          error: '',
        },
        nameError: '',
        proxiedDeviceError: '',

        sortField: null,
        sortAscending: true,
      };
    },
    created: function () {
      if (this.topicList.length > 0) this.activeTopic = this.topicList[0];
    },
    mounted: function () {
      msgbus.$on('brokerDeleted', _.bind(this.onBrokerDeleted, this));
      msgbus.$on('brokerRenamed', _.bind(this.onBrokerRenamed, this));
      msgbus.$on(
        'brokerDeviceDeleted',
        _.bind(this.onBrokerDeviceDeleted, this)
      );
      msgbus.$on(
        'brokerDeviceRenamed',
        _.bind(this.onBrokerDeviceRenamed, this)
      );

      msgbus.$on('broker:gcp:deviceIdChanged', (evtData) =>
        this.onGCPDeviceIdChanged(evtData)
      );
    },
    watch: {
      topicList: function (val) {
        if (this.topicList.length == 0) return;

        if (this.activeTopic == null)
          this.activeTopic = _.first(this.topicList);
        else this.activeTopic = _.last(this.topicList);
      },
    },
    computed: {
      activeTopicType: function () {
        return this.topicTypeName(this.activeTopic.broker);
      },
      proxied_devices: function () {
        let brokerNode = this.brokerNode(this.newTopic.broker);
        if (
          brokerNode &&
          brokerNode.config &&
          brokerNode.config.proxied_devices &&
          brokerNode.config.proxied_devices.length > 0
        )
          return brokerNode.config.proxied_devices;
        else return [];
      },
    },
    methods: {
      brokerLabel: function (topic) {
        if (topic.proxied_device)
          return topic.broker + ' / ' + topic.proxied_device;
        else return topic.broker;
      },

      isNewTopicValid: function (newTopic) {
        var newName = newTopic.name;
        if (!isNameValid(newName)) {
          this.newTopic.error =
            "Name can only contain alphanumber, '-' and '_', 32 chars at most.";
          return false;
        }

        if (!this.validateIotCloudTopics()) return false;

        //check if topic is unique
        if (
          !_.all(this.topicList, function (topic) {
            return newName != topic.name;
          })
        )
          this.newTopic.error = 'Name already exists';
        else this.newTopic.error = '';
        return this.newTopic.error.length == 0;
      },

      validateIotCloudTopics: function () {
        //for IoT Cloud(gcp), there should be only one publish topic
        let self = this;
        let brokerNode = this.brokerNode(this.newTopic.broker);
        if (brokerNode && brokerNode.type == 'gcp-iot-core') {
          if (
            _.any(this.topicList, function (topic) {
              return (
                topic.broker == self.newTopic.broker &&
                topic.category == self.newTopic.category &&
                topic.proxied_device == self.newTopic.proxied_device &&
                self.newTopic.category == 1
              );
            })
          ) {
            // this.$set(this.newTopic, 'error', "GCP IoT Core type broker can only include one topic.");
            this.newTopic.error =
              'GCP IoT Core type broker can only include one subscription topic.';
            return false;
          }
        }
        return true;
      },
      addTopic: function () {
        this.newTopic.name = uniqueName(
          'NewTopic',
          _.map(this.topicList, function (topic) {
            return topic.name;
          })
        );
        this.newTopic['uniqueKey'] = uniqueId(
          'TopicKey',
          _.map(this.topicList, function (topic) {
            return topic.uniqueKey;
          })
        );
        if (!this.newTopic.broker) {
          if (this.brokerList.length > 0)
            this.newTopic.broker = this.brokerList[0].name;
        } else {
          let prevBroker = this.newTopic.broker;
          let isPrevBrokerValid = _.any(this.brokerList, function (broker) {
            return broker.name == prevBroker;
          });
          if (!isPrevBrokerValid) this.newTopic.broker = null;
        }

        this.newTopic.enabled = true;
        this.newTopic.error = '';
        this.newTopic.config = {};
        this.newTopic.category = 0;
        $(this.$el).find('.modal').modal('show');
        //FIXME: only work for the first time
        $(this.$el).find('#inputNewTopicName').focus();
      },
      createTopic: function () {
        if (!this.isNewTopicValid(this.newTopic) || !this.newTopic.broker) {
          return;
        }

        let brokerNode = this.brokerNode(this.newTopic.broker);
        if (brokerNode.type == 'mqtt' || brokerNode.type == 'aws-iot-core')
          this.newTopic.config['path'] = '';

        $(this.$el).find('.modal').modal('hide');
        this.topicList.push(cloneDeep(this.newTopic));
      },
      deleteTopic: function () {
        if (!confirm('Are you sure to delete this topic ?')) return;

        for (let i = 0; i < this.topicList.length; ++i) {
          if (this.topicList[i].name != this.activeTopic.name) continue;

          if (this.topicList.length > 1)
            this.activeTopic = this.topicList[(i + 1) % this.topicList.length];
          else this.activeTopic = { name: null, broker: null };
          this.doDeleteTopic(i);
          break;
        }
      },

      doDeleteTopic: function (index) {
        if (index >= this.topicList.length) return;

        let name = this.topicList[index].name;
        this.topicList.splice(index, 1);
        msgbus.$emit('topicDeleted', name);
      },

      sortBy: function (field, ascending) {
        this.sortField = field;
        this.sortAscending = ascending;
        this.topicList.sort(function (a, b) {
          let aval = '';
          let bval = '';
          if (field.toLowerCase() == 'connection/device') {
            aval = a['broker'] + '/' + a['proxied_device'];
            bval = b['broker'] + '/' + b['proxied_device'];
          } else {
            aval = a[field];
            bval = b[field];
          }

          if (ascending) return aval.localeCompare(bval);
          else return bval.localeCompare(aval);
        });
      },

      onTopicUpdated: function (name, category, config, proxiedDeviceId) {
        let activeTopic = this.activeTopic;
        if (!activeTopic) return;

        let isNameUnique = _.all(this.topicList, function (topic) {
          if (topic.uniqueKey == activeTopic.uniqueKey) return true;
          return topic.name != name;
        });
        if (!isNameUnique) {
          this.nameError = 'The new topic name is not unique';
          return;
        } else this.nameError = '';

        // validate if topic unique
        let self = this;
        let brokerNode = this.brokerNode(this.activeTopic.broker);
        if (brokerNode && brokerNode.type == 'gcp-iot-core') {
          if (
            _.any(this.topicList, function (topic) {
              return (
                topic.uniqueKey != self.activeTopic.uniqueKey &&
                topic.broker == self.activeTopic.broker &&
                topic.category == category &&
                topic.config.path == config.path
              );
            })
          ) {
            this.proxiedDeviceError = 'Proxied device topic is duplicate.';
            return;
          } else this.proxiedDeviceError = '';
        }

        let oldName = this.activeTopic.name;
        this.$set(this.activeTopic, 'name', name);

        if (proxiedDeviceId)
          this.$set(this.activeTopic, 'proxied_device', proxiedDeviceId);
        else this.$set(this.activeTopic, 'proxied_device', null);

        if (config) this.$set(this.activeTopic, 'config', cloneDeep(config));

        if (oldName != name) msgbus.$emit('topicRenamed', oldName, name);

        if (category === undefined) category = 0;
        let oldCategory = this.activeTopic.category;
        if (category != oldCategory) {
          this.$set(this.activeTopic, 'category', category);
          msgbus.$emit('topicCategoryChanged', name, oldCategory, category);
        }
      },
      onTopicReset: function () {
        this.nameError = '';
        this.proxiedDeviceError = '';
      },

      onBrokerDeleted: function (name) {
        for (let i = this.topicList.length; i > 0; --i) {
          let index = i - 1;
          let topic = this.topicList[index];
          if (topic.broker != name) continue;

          this.doDeleteTopic(index);
        }
      },
      onBrokerRenamed: function (oldName, newName) {
        for (let i = this.topicList.length; i > 0; --i) {
          let index = i - 1;
          let topic = this.topicList[index];
          if (topic.broker != oldName) continue;

          topic.broker = newName;
        }
      },
      onBrokerDeviceDeleted: function (broker, id) {
        let index = this.topicList.findIndex(
          (t) => t.broker == broker && t.proxied_device == id
        );
        if (index < 0) return;
        this.doDeleteTopic(index);
      },
      onBrokerDeviceRenamed: function (broker, oldId, newId) {
        let index = this.topicList.findIndex(
          (t) => t.broker == broker && t.proxied_device == oldId
        );
        this.topicList[index].proxied_device = newId;

        if (this.topicList[index].config && this.topicList[index].config.path) {
          let regex = new RegExp('/' + oldId + '/');
          this.topicList[index].config.path = this.topicList[
            index
          ].config.path.replace(regex, '/' + newId + '/');
        }
      },
      onGCPDeviceIdChanged: function (evtData) {
        if (!evtData) return;
        let { brokerName, oldDeviceId, newDeviceId } = evtData;

        let gConfig = this.getGConfig();
        for (let i = 0; i < gConfig.topicList.length; ++i) {
          let t = gConfig.topicList[i];
          if (!t || !t.config || !t.config.path) continue;
          if (t.broker != brokerName) continue;
          t.config.path = t.config.path.replace(
            `/devices/${oldDeviceId}/`,
            `/devices/${newDeviceId}/`
          );
        }
      },

      brokerNode: function (brokerName) {
        return _.find(this.brokerList, function (broker) {
          return broker.name == brokerName;
        });
      },

      topicTypeName: function (brokerName) {
        let brokerType = 'mqtt';
        _.any(this.brokerList, function (broker) {
          if (broker.name == brokerName) {
            brokerType = broker.type;
            return true;
          } else return false;
        });
        return brokerType + '-topic';
      },
    },
    components: {
      'gcp-iot-core-topic': gcpIoTCoreTopicComp,
      'mqtt-topic': mqttTopicComp,
      'aws-iot-core-topic': awsIoTCoreTopicComp,
      'azure-iot-hub-topic': azureIoTHubTopicComp,
    },
  }; //}}}1

  let configUploaderComp = {
    //{{{1
    template: '#config_upload_template',
    inject: ['getGConfig'],
    data: function () {
      return {
        title: 'Upload Config',
        msg: '',
        dzElem: null,
      };
    },
    mounted: function () {
      this.dzElem = $(this.$el).find('#mqtt_config_uploader_panel');
      this.dzElem.dropzone({
        url: 'file_manager.php',
        paramName: 'file',
        maxFileSize: 1,
        clickable: true,
        createImageThumbnails: false,
        acceptedFiles: '.tgz',
        previewsContainer: '.dropzone-previews',
      });

      let self = this;
      let dz = Dropzone.forElement(this.dzElem.get(0));
      dz.on('queuecomplete', function () {
        self.msg = 'File is uploaded and applied, reloading page now...';
        setTimeout(function () {
          location.reload();
        }, 1500);
      });
      dz.on('addedfile', function () {
        self.msg = 'Start to upload file ...';
      });
      dz.on('error', function (unused, error) {
        self.msg = 'Error: ' + error;
      });

      $(this.$el).on('hide', function () {
        self.msg = '';
      });
    },
    methods: {
      showDialog: function () {
        $(this.$el).modal('show');
      },
      getDropzone: function () {
        return Dropzone.forElement(this.dzElem.get(0));
      },
    },
  }; //}}}1

  //the global config for the whole DataService. the config data structure is
  //flat in js side, but it will be merge into a tree structure before sending
  //to controller; when loading the config from controller, we need to
  //reorganize it into the flat structure for js to use
  let gConfig = {
    //{{{1
    brokerList: [],
    topicList: [],
  }; //}}}1

  //create&init root object
  window.app = new Vue({
    // {{{1
    el: '#data_service_config_content',
    data: function () {
      let sections = [
        { sec_id: 'broker-manager', name: 'Connection Manager' },
        { sec_id: 'topic-manager', name: 'Topic Manager' },
        { sec_id: 'data-mapping', name: 'Message Manager' },
      ];
      let anchor = window.location.hash.substring(1);
      let sec = sections.find(function (s) {
        return s.sec_id == anchor;
      });
      let active_sec_id = sec && sec.sec_id ? sec.sec_id : sections[0].sec_id;

      return {
        sections,
        active_sec_id,
        spinner: null,
      };
    },
    mounted: function () {
      this.loadConfig();
    },
    methods: {
      isSectionActive: function (sec) {
        return sec.sec_id == this.active_sec_id;
      },
      getSpinner() {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },
      getGConfig: function () {
        return gConfig;
      },

      applyDataMappingData: function (resp) {
        if (!_.has(resp, 'brokerList')) {
          console.debug('can not find brokerList, skip');
          return;
        }

        let config = this.getGConfig();
        _.each(resp.brokerList, function (broker, brokerIndex) {
          let brokerUniqueKey = 'BrokerKey_' + brokerIndex;
          broker = _.defaults(broker, { enabled: true });
          var brokerData = {
            uniqueKey: brokerUniqueKey,
            name: broker.name,
            enabled: broker.enabled,
            type: broker.type,
            config: cloneDeep(broker.config),
          };
          config.brokerList.push(brokerData);

          if (_.has(broker, 'topicList') && _.isArray(broker.topicList)) {
            _.each(broker.topicList, function (topic, topicIndex) {
              let topicUniqueKey = 'TopicKey_' + brokerIndex + '_' + topicIndex;
              topic = _.defaults(topic, { enabled: true });
              var topicData = {
                uniqueKey: topicUniqueKey,
                name: topic.name,
                enabled: topic.enabled,
                broker: broker.name,
                proxied_device: topic.proxied_device,
                category: topic.category,
                config: cloneDeep(topic.config),
              };
              config.topicList.push(topicData);
            });
          }
        });
      },
      applyDataMappingConfig: function (resp) {
        if (!_.has(this.$refs, 'data-mapping')) return;

        let dataMappingInst = this.$refs['data-mapping'][0];
        dataMappingInst.initData(resp);
      },
      applyTemplateData: function (resp) {
        if (!_.has(this.$refs, 'data-mapping')) return;

        let dataMappingInst = this.$refs['data-mapping'][0];
        dataMappingInst.initTemplateData(resp);
      },
      cleanupTmplCatNode: function (catNode) {
        delete catNode.id;
        delete catNode.li_attr;
        delete catNode.a_attr;
        if (catNode.children && catNode.children.length > 0) {
          //cleanup snippet nodes
          for (let i = 0; i < catNode.children.length; i++) {
            let snipNode = catNode.children[i];
            delete snipNode.id;
            delete snipNode.li_attr;
            delete snipNode.a_attr;

            //process topic nodes
            if (snipNode.children && snipNode.children.length > 0) {
              for (let j = 0; j < snipNode.children.length; j++) {
                let topicNode = snipNode.children[j];
                snipNode.children[j] = convertNodeUIDs(topicNode);
              }
            }
            catNode.children[i] = snipNode;
          }
        }
        return catNode;
      },
      loadConfig: function () {
        let self = this;
        $.ajax({
          url: 'index.php',
          method: 'GET',
          dataType: 'json',
          data: {
            action: 'load_config',
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data, status, jq) {
            if (data.redirect) redirect(data.redirect);

            var resp_data = _.has(data, 'data') ? data.data : null;
            if (resp_data) {
              if (_.has(resp_data, 'certs_dir')) {
                let config = self.getGConfig();
                self.$set(config, 'certs_dir', resp_data.certs_dir);
              }
              if (
                _.has(resp_data, 'config_data') &&
                resp_data.config_data.length > 0
              ) {
                self.applyDataMappingConfig(JSON.parse(resp_data.config_data));
              }

              if (
                _.has(resp_data, 'mapping_data') &&
                resp_data.mapping_data.length > 0
              ) {
                self.applyDataMappingData(JSON.parse(resp_data.mapping_data));
              }

              let default_tmpl =
                _.has(resp_data, 'default_template_data') &&
                resp_data.default_template_data.length > 0
                  ? JSON.parse(resp_data.default_template_data)
                  : [];

              default_tmpl = default_tmpl.map((tmpl) =>
                self.cleanupTmplCatNode(tmpl)
              );
              if (
                _.has(resp_data, 'template_data') &&
                resp_data.template_data.length > 0
              ) {
                //merge default_tmpl into user's template
                let user_tmpl = JSON.parse(resp_data.template_data);
                user_tmpl = user_tmpl.map((utmpl) =>
                  self.cleanupTmplCatNode(utmpl)
                );
                default_tmpl
                  .filter((tmpl) => tmpl.type == 'category')
                  .forEach((tmpl) => {
                    let index = user_tmpl.findIndex(
                      (utmpl) => utmpl.text == tmpl.text
                    );

                    if (index == -1) user_tmpl.unshift(tmpl);
                    else user_tmpl.splice(index, 1, tmpl);
                  });
                self.applyTemplateData(user_tmpl);
              } else {
                self.applyTemplateData(default_tmpl);
              }
            }
          },
          error: function (data, status) {
            alert('Failed to load config data: ' + data);
            // msgbus.$emit("alert", "Error", "Failed to load config data: " + data, 'error');
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },

      prepareMappingData: function () {
        let gConfig = this.getGConfig();
        let config = {
          version: 2,
          timestamp: new Date().getTime(),
          certs_dir: gConfig.certs_dir,
          brokerList: [],
        };

        let mappingList = null;
        if (_.has(this.$refs, 'data-mapping')) {
          let dataMappingInst = this.$refs['data-mapping'][0];
          mappingList = dataMappingInst.getMappingData();
        }

        _.each(gConfig.brokerList, function (broker) {
          let brokerData = {
            name: broker.name,
            enabled: broker.enabled,
            type: broker.type,
            config: cloneDeep(broker.config),
          };
          config['brokerList'].push(brokerData);

          if (_.has(gConfig, 'topicList')) {
            brokerData['topicList'] = [];
            _.each(gConfig.topicList, function (topic) {
              if (topic.broker != brokerData.name) return;

              let topicData = {
                name: topic.name,
                enabled: topic.enabled,
                broker: topic.broker,
                category:
                  topic.category === undefined ? 0 : Number(topic.category), // make sure category is always number
                config: cloneDeep(topic.config),
              };

              if (topic.proxied_device && topic.proxied_device.length > 0)
                topicData['proxied_device'] = topic.proxied_device;

              //process config data based on broker type
              if (brokerData.type == 'azure-iot-hub') {
                //refers to: https://docs.microsoft.com/en-us/azure/iot-hub/iot-hub-mqtt-support#receiving-cloud-to-device-messages
                //since for azure iot hub, the events and config topics are
                //fixed, so we can hardcoded it here, and the topic will be
                //updated when client_id is changed.
                if (topicData.category == 0)
                  topicData.config.path =
                    'devices/' +
                    brokerData.config['mqtt_client_id'] +
                    '/messages/events';
                else
                  topicData.config.path =
                    'devices/' +
                    brokerData.config['mqtt_client_id'] +
                    '/messages/devicebound/#';
              }

              if (!_.isNull(mappingList)) {
                _.each(mappingList, function (mapping) {
                  if (mapping.topic != topic.name) return;
                  topicData['dataMapping'] = mapping;
                  return;
                });
              }
              brokerData['topicList'].push(topicData);
            });
          }
        });

        return config;
      },
      prepareJsTreeConfig: function () {
        let configData = {};
        if (_.has(this.$refs, 'data-mapping')) {
          let dataMappingInst = this.$refs['data-mapping'][0];
          configData = dataMappingInst.getConfigData();
        }
        configData['version'] = 1;
        configData['timestamp'] = new Date().getTime();
        return configData;
      },
      prepareTemplateData: function () {
        let templateData = {};
        if (_.has(this.$refs, 'data-mapping')) {
          let dataMappingInst = this.$refs['data-mapping'][0];
          templateData = dataMappingInst.getTemplateData();
        }
        return templateData;
      },
      isInputsValid: function () {
        let gConfig = this.getGConfig();

        let allBrokerValid = _.all(gConfig.brokerList, function (broker) {
          if (!_.has(broker, 'errors')) return true;
          return _.all(_.keys(broker.errors), function (key) {
            let has_error =
              _.has(broker.errors, key) && broker.errors[key] == true;
            return !has_error;
          });
        });
        if (!allBrokerValid) return false;

        let allTopicValid = _.all(gConfig.topicList, function (topic) {
          if (!_.has(topic, 'errors')) return true;
          return _.all(_.keys(topic.errors), function (key) {
            let has_error =
              _.has(topic.errors, key) && topic.errors[key] == true;
            return !has_error;
          });
        });
        return allBrokerValid && allTopicValid;
      },
      validateGcpTopicPath: function (device_id, topic_path) {
        return topic_path.startsWith(`/devices/${device_id}/`);
      },
      validateConfig: function () {
        let gConfig = this.getGConfig();

        let errors = [];
        gConfig.brokerList.forEach((broker) => {
          if (!broker.enabled) return;

          // validate if gcp iot core config data is consistent or not
          if (broker.type == 'gcp-iot-core') {
            let config = broker.config;
            let topicList = gConfig.topicList;
            if (!topicList || topicList.length == 0) return;

            let device_id = config.device_id;
            let proxiedDeviceIds = [];
            if (config.proxied_devices)
              proxiedDeviceIds = config.proxied_devices.map((d) => d.device_id);

            topicList.forEach((t) => {
              if (t.broker != broker.name) return;

              if (!t.proxied_device) {
                // no proxied device, verify device_id in topic path
                if (!t.config || !t.config.path || t.config.path.length == 0) {
                  errors.push(
                    `connection(${broker.name}) topic(${t.name}) path can not be empty`
                  );
                  return;
                }
                if (this.validateGcpTopicPath(device_id, t.config.path)) return;
                errors.push(
                  `connection(${broker.name}) topic(${t.name}) path doesn't match its device(${device_id})`
                );
                return;
              }

              // verify proxied_device is created
              if (!proxiedDeviceIds.includes(t.proxied_device)) {
                errors.push(
                  `connection(${broker.name}) topic(${t.name}) use an invalid proxied device(${t.proxied_device})`
                );
                return;
              }

              // verify topic's proxied_device and path matched
              if (!t.config || !t.config.path || t.config.path.length == 0) {
                errors.push(
                  `connection(${broker.name}) topic(${t.name}) path can not be empty`
                );
                return;
              }

              if (this.validateGcpTopicPath(t.proxied_device, t.config.path))
                return;
              errors.push(
                `connection(${broker.name}) topic(${t.name}) path doesn't match its device(${t.proxied_device})`
              );
              return;
            });
          }
        });

        return errors;
      },
      saveConfig: function () {
        if (!this.isInputsValid() || $('#broker_detail .control-group.error').length > 0) {
          msgbus.$emit(
            'alert',
            'Error',
            'Some inputs are invalid, please fix it and try again.',
            'error',
            0
          );
          alert('Error: Some inputs are invalid, please fix it and try again.');
          return;
        } else {
          let errors = this.validateConfig();
          if (errors && errors.length > 0) {
            let msg = errors.join(' ; ');
            msgbus.$emit('alert', 'Error', msg, 'error', 0);
            return;
          }
          msgbus.$emit('alert:clear');
        }
        let mappingData = this.prepareMappingData();

        if (!mappingData) {
          console.warn('mapping data is invalid');
          return;
        }
        let self = this;
        $.ajax({
          url: 'index.php',
          method: 'POST',
          dataType: 'json',
          data: {
            action: 'save_config',
            mapping_data: JSON.stringify(mappingData),
            config_data: JSON.stringify(this.prepareJsTreeConfig()),
            template_data: JSON.stringify(this.prepareTemplateData()),
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data, status, jq) {
            if (data.redirect) redirect(data.redirect);
          },
          error: function (data, status) {
            alert('Failed to save config data: ' + data);
            // msgbus.$emit("alert", "Error", "Failed to load config data: " + data, 'error');
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },

      uploadConfig: function () {
        let config_uploader_inst = this.$refs['config_uploader_inst'];
        if (config_uploader_inst) config_uploader_inst.showDialog();
      },
    },
    watch: {
      active_sec_id: function (val) {
        msgbus.$emit('alert:clear');
      },
    },
    provide: function () {
      return {
        getGConfig: this.getGConfig,
        getTopicList: function () {
          if (!_.has(gConfig, 'topicList')) gConfig['topicList'] = [];
          return gConfig.topicList;
        },
      };
    },
    components: {
      'broker-manager': brokerManager,
      'topic-manager': topicManager,
      'data-mapping': dataMappingComp,
      alert: alertComp,
      'file-selector': fileSelectorComp,
      'config-uploader': configUploaderComp,
    },
  }); //}}}1
});
