<?php 
// vim: ts=2 sw=2

set_include_path(get_include_path() . PATH_SEPARATOR . '../../app');

include_once "mqtt_service_base.php";

class DataServiceConfig extends MQTTServiceBase {
  public function doGet() {
    if (!isset($_GET['action']))
      return;
    $action = $_GET['action'];

    if (!$this->isSystemEnabled()) {
      die(L("Permission Denied."));
    }

    if ($action == 'download_config')
      $this->downloadConfig();
    else
      die(sprintf(L("Action %s is invalid."), $action));
  }

  public function doAjaxGet() {
    $action = 'load_config';
    if (isset($_GET['action']))
      $action = $_GET['action'];

    $response = array();
    if ($action == "status")
      $this->handleServiceStatusQuery();
    else if ($action == 'load_config')
      $this->handleDataMappingQuery();
    else
      $this->renderAjaxError($response, "invalid action: " . $action);
  }

  public function doAjaxPost() {
    $response = array();
    $action = "save_config";
    if (isset($_POST['action']))
      $action = $_POST['action']; 
    
    if ($action == "save_config") {
      $curPath = dirname(__FILE__);

      $config_data = $this->getInputData('config_data', false);
      $filename = build_file_path($curPath, "data_mapping_config.json");
      file_put_contents($filename, $config_data);

      $template_data = $this->getInputData('template_data', false);
      $filename = build_file_path($curPath, "data_mapping_template.json");
      file_put_contents($filename, $template_data);

      $mapping_data = $this->getInputData('mapping_data', false);
      $filename = build_file_path($curPath, "data_mapping.json");
      if (file_put_contents($filename, $mapping_data)) {
        $error = $this->restartService($filename);
        if (is_null($error))
          $this->renderAjaxSuccess($response);
        else
          $this->renderAjaxError($response, $error);
      }
      else
        $this->renderAjaxError($response, "can not write config data to file: " . $filename);
    } else if ($action == "update_broker") {
      $broker_name = $this->getInputData('broker_name', false);
      $mqtt_host = $this->getInputData('mqtt_host', false);
      $mqtt_port = $this->getInputData('mqtt_port', false);
      $err = $this->updateBroker($broker_name, $mqtt_host, $mqtt_port);
      if (is_null($err)) 
          $this->renderAjaxSuccess($response);
      else
          $this->renderAjaxError2($response, $err);
    } else
      $this->renderAjaxError($response, "invalid action: " . $action);
  }

  protected function handleServiceStatusQuery() {
    $response = array();
    $pfName = platformName();
    if ($pfName == 'FS' || $pfName == 'FI')
      $cmd = "/bin/sh /etc/init.d/S91mqtt-service status";
    else if ($pfName == 'FW')
      $cmd = "/bin/sh -c 'source /etc/init.d/mqtt-service && service_status'";
    else
      $this->renderAjaxError($response, "platform is not supported");
        
    $output = array();
    $return = '';
    $last_line = exec($cmd, $output, $return);
    if ($return != '0')
      $response['service_status'] = 'unknown';
    else {
      if (preg_match('/mqtt-service dead but pid file exists/', $last_line)
        || preg_match('/mqtt-service is stopped/', $last_line))
        $response['service_status'] = 'stopped';
      else if (preg_match('/mqtt-service \(pid \d+\) is running.../', $last_line)) {
        $response['service_status'] = 'started';
        $this->collectBrokerStates($response);
      } else
        $response['service_status'] = 'error';
    }
    $this->renderAjaxSuccess2($response);
  }

  protected function collectBrokerStates(&$response) {
    # only send SIGUSR1 signal to children process, not parent process
    $cmd = "top -b -n 1 | awk '/\/usr\/local\/bin\/[m]qtt-service/ {if ($2 != 1) print $1}' | xargs kill -SIGUSR1";
    $output = array();
    $return = '';
    $last_line = exec($cmd, $output, $return);
    if ($return != '0')
      return;

    $response['broker_states'] = array();
    foreach(glob("/tmp/mqtt-service/*.state") as $state_file) {
      //ignore state file that is older than 30 seconds
      if (time()-filemtime($state_file) > 30)
        continue;

      $content = file_get_contents($state_file);
      $data = $this->parseStateFile($content);
      if (!empty($data))
        $response['broker_states'][] = $data;
    }
  }

  protected function parseStateFile($content) {
    $data = array();
    // example data:
    //  3;name:mosquitto,errno:0,msg:MQTT connected
    $parts = explode(";", $content, 2);
    if (count($parts) != 2)
      return $data;

    $sec_num = intval($parts[0]);
    if ($sec_num <= 0)
      return $data;

    $secs = explode(",", $parts[1], $sec_num);
    foreach($secs as $sec) {
      $pairs = explode(":", $sec, 2);
      if (count($pairs) != 2)
        continue;

      $data[$pairs[0]] = $pairs[1];
    }
    return $data;
  }

  protected function handleDataMappingQuery() {
    $response = array();
    $response['data'] = array();

    $response['data']['certs_dir'] = build_file_path(dirname(__FILE__), "uploads", "certs");
    $filename = "data_mapping.json";
    if (file_exists($filename))
      $response['data']['mapping_data'] = file_get_contents($filename); 

    $filename = "data_mapping_config.json";
    if (file_exists($filename))
      $response['data']['config_data'] = file_get_contents($filename); 
    
    $filename = "data_mapping_template.json";
    if (file_exists($filename)) 
      $response['data']['template_data'] = file_get_contents($filename); 

    $filename = "default_" . $filename;
    if (file_exists($filename))
        $response['data']['default_template_data'] = file_get_contents($filename); 

    $this->renderAjaxSuccess2($response);
  }

  protected function downloadConfig() {
    $file = $this->compressConfig();
    if (file_exists($file)) {
      header('Content-Description: File Transfer');
      header('Content-Type: application/octet-stream');
      header('Content-Disposition: attachment; filename="'.basename($file).'"');
      header('Expires: 0');
      header('Cache-Control: must-revalidate');
      header('Pragma: public');
      header('Content-Length: ' . filesize($file));
      readfile($file);
      unlink($file);
      exit;
    } else {
      die(L("Can not download backup: {$name}"));
    }
  }
  protected function compressConfig() {
    $dir = dirname(__FILE__);
    $certDir = build_file_path($dir, "uploads", "certs");

    //make file name unique
    $name = "mqtt-config_" . date("YmdHis") . "_" . getmypid() . ".tgz";
    $tarFilePath = build_file_path($dir, $name);
    $cmd = "tar -C {$dir} -cf - data_mapping.json data_mapping_config.json data_mapping_template.json uploads | gzip -c > " . escapeshellarg($tarFilePath);
    if(!execShellCmd($cmd)) {
      error_log("failed to build backup tgz");
      return;
    }
    return $tarFilePath;
  }

  public function getServiceVersion() {
    $output = shell_exec("/usr/local/bin/mqtt-service --version");  
    if (is_null($output))
      return null;
    $output = rtrim($output);
    $parts = explode(" ", $output);
    if (count($parts) <= 1)
      return null;

    return array_pop($parts);
  }

  protected function updateBroker($broker_name, $mqtt_host, $mqtt_port) {
    if (is_null($broker_name)) return "'broker_name' param missing";

    $curPath = dirname(__FILE__);
    $filename = build_file_path($curPath, "data_mapping.json");
    if (!file_exists($filename)) return "cannot find 'data_mapping.json' file";

    $content = file_get_contents($filename);
    if (!$content) return "cannot load content of 'data_mapping.json'";

    $json = json_decode($content, true);
    if (is_null($json)) return "cannot decode 'data_mapping.json'";

    if (!array_key_exists('brokerList', $json) || !is_array($json['brokerList']))
      return "invalid 'brokerList' data";
    
    $found = false;
    foreach($json['brokerList'] as &$broker) {
      if ($broker['name'] != $broker_name) continue;

      $found = true;
      $config = &$broker['config'];
      if (!$config) return "cannot find broker config data";

      if (!is_null($mqtt_host))
        $config['mqtt_host'] = $mqtt_host;

      if (!is_null($mqtt_port)) {
        if (!is_numeric($mqtt_port))
          return "'mqtt_port' must be a numeric value";
        $config['mqtt_port'] = intval($mqtt_port);
      }

      break;
    }

    if ($found) {
      $res = file_put_contents($filename, json_encode($json));
      if (!$res) return "failed to save 'data_mapping.json' file";
    } else
      return "cannot find broker with name '{$broker_name}'";
    return null;
  }  
}

$controller = new DataServiceConfig();
$controller->run();

?>
<!DOCTYPE html>
<html>
  <head>
    <meta charset="utf-8" />
    <title><?php $controller->echoPageTitle("Data Service Config") ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <link rel="stylesheet" href="<?php A('../../css/bootstrap.min.css') ?>" media="screen" />
    <link rel="stylesheet" href="<?php A('../../css/dropzone.css') ?>" media="screen" />
    <link rel="stylesheet" href="<?php A('../../css/common.css') ?>" media="screen" />
    <link rel="stylesheet" href="<?php A('../../js/jstree/themes/default/style.min.css') ?>" media="screen" />

    <link rel="stylesheet" href="<?php A('css/data_service_config.css') ?>" media="screen" />
    <?php $controller->faviconHTMLInfo() ?>
  </head>
  <body style="height:100vh;">
    <div class="container-fluid wide-page" id="data_service_config_content">
      <div class="navbar">
        <div class="navbar-inner">
          <a id="home_nav_btn" class="brand" href="<?php echo $controller->utilityUrl() ?>"><?php echo $controller->utilityLabel() ?></a>
          <ul class="nav">
            <li class="divider-vertical"></li>
            <li v-for="sec in sections" :class="{active:isSectionActive(sec)}"><a :href="'#' + sec.sec_id" @click="active_sec_id=sec.sec_id">{{ sec.name }}</a></li>
          </ul>

          <ul class="nav pull-right">
            <li>
              <a href="log_viewer.php" target="_blank">Log Viewer</a>
            </li>
            <li class="divider-vertical"></li>
            <service-status class="top-right-uilet" />
          </ul>
          <!-- <button class="btn btn&#45;primary pull&#45;right">New Profile</button> -->
        </div>
      </div>

      <!-- <h3 class='text&#45;center'>Data Service Config</h3>  -->
      <div class="row" style="max-height:calc(100vh - 110px);">
      <?php if ($controller->isMQTTServiceEnabled()) { ?>
        <alert></alert>
        <template v-for="sec in sections">
          <component :is="sec.sec_id" class="tab-pane" :class="{hide: active_sec_id != sec.sec_id}" :ref="sec.sec_id" style="height:100%"></component>
        </template>
        <button type="button" class="btn btn-primary btn-large pull-right" @click="saveConfig">Save</button>
        <div class="btn-group dropup pull-right" style="margin-right: 6px;">
          <a class="btn btn-large dropdown-toggle" data-toggle="dropdown" href="#">
            Utilities
            <span class="caret"></span>
          </a>
          <ul class="dropdown-menu">
            <li> 
              <a href="#" @click="uploadConfig">Upload Config...</a> 
            </li>
            <li> 
              <a href="index.php?action=download_config" target="_blank">Download Config</a> 
            </li>
          </ul>
        </div>
      <?php } else { ?>
        <div class="alert">
          MQTT Service is disabled, go to 'ServiceControl' plugin to enable it. 
        </div>
      <?php } ?>
      </div>
      <file-selector ref="file_selector_inst" server_url="file_manager.php" ></file-selector>
      <config-uploader ref="config_uploader_inst"></config-uploader>
      <div class="row">
        <p class="text-center muted">
          MQTT Protocol: <a href="http://docs.oasis-open.org/mqtt/mqtt/v3.1.1/os/mqtt-v3.1.1-os.html">v3.1.1</a> ; MQTT Driver: <?php echo $controller->getServiceVersion(); ?>
        </p>
      </div>

    </div>

    <!-- component templates -->
    <script type="text/x-template" id="alert-template">
      <div class="alert" :class="label_class" v-show="show">
        <button type="button" class="close" @click="show=false">&times;</button>
        <strong>{{ title }}</strong> {{ msg }}
      </div>
    </script>

    <script type="text/x-template" id="gcp_iot_core_template">
      <div>
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>GCP IoT Core</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputName">Connection Name:</label>
          <div class="controls">
            <input type="text" id="inputName" placeholder="Connection Name" v-model.trim="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>

        <div class="control-group" :class="{error: host_error.length > 0}">
          <label class="control-label" for="inputHost">Endpoint:</label>
          <div class="controls">
            <input type="text" id="inputHost" placeholder="MQTT Endpoint" v-model.trim="config.mqtt_host">
            <span class="help-inline" v-show="host_error.length > 0">{{ host_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: port_error.length > 0}">
          <label class="control-label" for="inputPort">Port:</label>
          <div class="controls">
            <input type="text" id="inputPort" placeholder="MQTT Endpoint Port" v-model.number="config.mqtt_port">
            <span class="help-inline" v-show="port_error.length > 0">{{ port_error }}</span>
          </div>
        </div>

        <div class="control-group" :class="{error: project_id_error.length > 0}">
          <label class="control-label" for="inputProjectID">Project ID:</label>
          <div class="controls">
            <input type="text" id="inputProjectID" placeholder="Project ID" v-model.trim="config.project_id">
            <span class="help-inline" v-show="project_id_error.length > 0">{{ project_id_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: registry_id_error.length > 0}">
          <label class="control-label" for="inputRegistryID">Registry ID:</label>
          <div class="controls">
            <input type="text" id="inputRegistryID" placeholder="Registry ID" v-model.trim="config.registry_id">
            <span class="help-inline" v-show="registry_id_error.length > 0">{{ registry_id_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: device_id_error.length > 0}">
          <label class="control-label" for="inputDeviceID">Device ID:</label>
          <div class="controls">
            <input type="text" id="inputDeviceID" placeholder="Device ID" v-model.trim="config.device_id">
            <span class="help-inline" v-show="device_id_error.length > 0">{{ device_id_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: data_precision_error.length > 0}">
          <label class="control-label" for="inputDataPrecision">Data Precision:</label>
          <div class="controls">
            <input id="inputDataPrecision" placeholder="Data Precision" type="text" v-model.trim="config.data_precision">
            <span class="help-inline" v-show="data_precision_error.length > 0">{{ data_precision_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: region_error.length > 0}">
          <label class="control-label" for="inputRegion">Region:</label>
          <div class="controls">
            <select id="inputRegion" v-model="config.region">
              <option value="us-central1">us-central1</option>
              <option value="europe-west1">europe-west1</option>
              <option value="asia-east1">asia-east1</option>
            </select>
            <span class="help-inline" v-show="region_error.length > 0">{{ region_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>SSL/TLS</strong></label>
          <div class="controls">
            <button class="btn btn-info" @click.prevent="generateCert">Generate Cert</button>
          </div>
        </div>

        <div class="control-group" :class="{error: ca_file_error.length > 0}">
          <label class="control-label" for="inputCAFile">CA File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCAFile" placeholder="CA File (click to change)" v-model="config.ca_file" @click="changeCAFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.ca_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="ca_file_error.length > 0">{{ ca_file_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: cert_file_error.length > 0}">
          <label class="control-label" for="inputCertFile">Certificate File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCertFile" placeholder="Client Certificate File (click to change)" v-model="config.cert_file" @click="changeClientCertFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.cert_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="cert_file_error.length > 0">{{ cert_file_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: key_file_error.length > 0}">
          <label class="control-label" for="inputKeyFile">Key File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputKeyFile" placeholder="Key File (click to change)" v-model="config.key_file" @click="changeKeyFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.key_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="key_file_error.length > 0">{{ key_file_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>Gateway</strong></label>
          <div class="controls">
            <label class="checkbox">
              <input type="checkbox" v-model="config.gateway_enabled"> Enable
            </label>
          </div>
          <div class="controls">
            <button class="btn btn-info" :class="config.gateway_enabled ? '' : 'disabled'" @click.prevent="manageProxiedDevices">Manage Proxied Devices</button>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>Publish Interval</strong></label>
        </div>
        <div class="control-group" :class="{error: events_interval_error.length > 0}">
          <label class="control-label" for="inputEventInterval">Publish Interval:</label>
          <div class="controls">
            <div class="input-append">
              <input type="number" id="inputEventInterval" class="span7" placeholder="How often to publish data" v-model.number="config.events_interval">
              <span class="add-on">second</span>
            </div>
            <span class="help-inline" v-show="events_interval_error.length > 0">{{ events_interval_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>

      <modal-dialog class="proxied_device_dialog" title="Proxied Device Manager" data-backdrop="static" 
          :enterToSubmit="false" @submitChange="onSaveProxiedDevices">
        <gcp-proxied-devices-manager ref="proxied_devices_manager" :devices.sync="config.proxied_devices"></gcp-proxied-devices-manager>
      </modal-dialog>

      <cert-generator />

      </div>
    </script>

    <script type="text/x-template" id="mqtt-template">
      <form class="form-horizontal" autocomplete="off">
        <div class="control-group">
          <label class="control-label"><strong>General MQTT</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputName">Connection Name:</label>
          <div class="controls">
            <input type="text" id="inputName" placeholder="Connection Name" v-model.trim="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: host_error.length > 0}">
          <label class="control-label" for="inputHost">Host:</label>
          <div class="controls">
            <input type="text" id="inputHost" placeholder="MQTT Broker Host" v-model.trim="config.mqtt_host">
            <span class="help-inline" v-show="host_error.length > 0">{{ host_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: port_error.length > 0}">
          <label class="control-label" for="inputPort">Port:</label>
          <div class="controls">
            <input type="text" id="inputPort" placeholder="MQTT Broker Port" v-model.number="config.mqtt_port">
            <span class="help-inline" v-show="port_error.length > 0">{{ port_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: client_id_error.length > 0}">
          <label class="control-label" for="inputClientId">Client ID:</label>
          <div class="controls">
            <input type="text" id="inputClientId" placeholder="Client ID" v-model.trim="config.mqtt_client_id">
            <span class="help-inline" v-show="client_id_error.length > 0">{{ client_id_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: user_error.length > 0}">
          <label class="control-label" for="inputUserName">User:</label>
          <div class="controls">
            <input type="text" id="inputUserName" placeholder="User" v-model.trim="config.mqtt_user" autocomplete="off">
            <span class="help-inline" v-show="user_error.length > 0">{{ user_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: passwd_error.length > 0}">
          <label class="control-label" for="inputPassword">Password:</label>
          <div class="controls">
            <input id="inputPassword" type="password" placeholder="Password" v-model.trim="config.mqtt_passwd" autocomplete="new-password">
            <span class="help-inline" v-show="passwd_error.length > 0">{{ passwd_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: data_precision_error.length > 0}">
          <label class="control-label" for="inputDataPrecision">Data Precision:</label>
          <div class="controls">
            <input id="inputDataPrecision" placeholder="Data Precision" type="text" v-model.trim="config.data_precision">
            <span class="help-inline" v-show="data_precision_error.length > 0">{{ data_precision_error }}</span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputQoS">QoS:</label>
          <div class="controls">
            <select id="inputQoS" v-model="config.mqtt_qos">
              <option value="0">QoS0</option>
              <option value="1">QoS1</option>
              <option value="2">QoS2</option>
            </select>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>SSL/TLS</strong></label>
        </div>

        <div class="control-group">
          <div class="controls">
            <label class="checkbox">
              <input type="checkbox" id="inputEnableTls" v-model="tls_enabled">Enable TLS
            </label>
          </div>
        </div>
        <div class="control-group" :class="{error: ca_file_error.length > 0}">
          <label class="control-label" for="inputCAFile">CA File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCAFile" placeholder="CA File (click to change)" v-model="config.ca_file" @click="changeCAFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.ca_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="ca_file_error.length">Invalid CA File</span>
          </div>
        </div>
        <div class="control-group" :class="{error: cert_file_error.length > 0}">
          <label class="control-label" for="inputCertFile">Certificate File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCertFile" placeholder="Client Certificate File (click to change)" v-model="config.cert_file" @click="changeClientCertFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.cert_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="cert_file_error.length > 0">{{ cert_file_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: key_file_error.length > 0}">
          <label class="control-label" for="inputKeyFile">Key File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputKeyFile" placeholder="Key File (click to change)" v-model="config.key_file" @click="changeKeyFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.key_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="key_file_error.length">{{ key_file_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>Publish Interval</strong></label>
        </div>
        <div class="control-group" :class="{error: events_interval_error.length > 0}">
          <label class="control-label" for="inputEventInterval">Publish Interval:</label>
          <div class="controls">
            <div class="input-append">
              <input type="number" id="inputEventInterval" class="span7" placeholder="How often to publish data" v-model.number="config.events_interval">
              <span class="add-on">second</span>
            </div>
            <span class="help-inline" v-show="events_interval_error.length">{{ events_interval_error }}</span>
          </div>
        </div>
        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="aws_iot_core_template">
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>AWS IoT Core</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputName">Connection Name:</label>
          <div class="controls">
            <input type="text" id="inputName" placeholder="Connection Name" v-model.trim="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>

        <div class="control-group" :class="{error: host_error.length > 0}">
          <label class="control-label" for="inputHost">Endpoint:</label>
          <div class="controls">
            <input type="text" id="inputHost" placeholder="MQTT Endpoint" v-model.trim="config.mqtt_host">
            <span class="help-inline" v-show="host_error.length > 0">{{ host_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: client_id_error.length > 0}">
          <label class="control-label" for="inputClientId">Client ID:</label>
          <div class="controls">
            <input type="text" id="inputClientId" placeholder="Client ID" v-model.trim="config.mqtt_client_id">
            <span class="help-inline" v-show="client_id_error.length > 0">{{ client_id_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: data_precision_error.length > 0}">
          <label class="control-label" for="inputDataPrecision">Data Precision:</label>
          <div class="controls">
            <input id="inputDataPrecision" placeholder="Data Precision" type="text" v-model.trim="config.data_precision">
            <span class="help-inline" v-show="data_precision_error.length > 0">{{ data_precision_error }}</span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputQoS">QoS:</label>
          <div class="controls">
            <select id="inputQoS" v-model="config.mqtt_qos">
              <option value="0">QoS0</option>
              <option value="1">QoS1</option>
            </select>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>SSL/TLS</strong></label>
        </div>

        <div class="control-group" :class="{error: ca_file_error.length > 0}">
          <label class="control-label" for="inputCAFile">CA File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCAFile" placeholder="CA File (click to change)" v-model="config.ca_file" @click="changeCAFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.ca_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="ca_file_error.length > 0">{{ ca_file_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: cert_file_error.length > 0}">
          <label class="control-label" for="inputCertFile">Certificate File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCertFile" placeholder="Client Certificate File (click to change)" v-model="config.cert_file" @click="changeClientCertFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.cert_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="cert_file_error.length > 0">{{ cert_file_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: key_file_error.length > 0}">
          <label class="control-label" for="inputKeyFile">Key File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputKeyFile" placeholder="Key File (click to change)" v-model="config.key_file" @click="changeKeyFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.key_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="key_file_error.length > 0">{{ key_file_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>Publish Interval</strong></label>
        </div>
        <div class="control-group" :class="{error: events_interval_error.length > 0}">
          <label class="control-label" for="inputEventInterval">Publish Interval:</label>
          <div class="controls">
            <div class="input-append">
              <input type="number" id="inputEventInterval" class="span7" placeholder="How often to publish data" v-model.number="config.events_interval">
              <span class="add-on">second</span>
            </div>
            <span class="help-inline" v-show="events_interval_error.length > 0">{{ events_interval_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="azure_iot_hub_template">
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>Azure IoT Hub</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputName">Connection Name:</label>
          <div class="controls">
            <input type="text" id="inputName" placeholder="Connection Name" v-model.trim="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>

        <div class="control-group" :class="{error: hub_error.length > 0}">
          <label class="control-label" for="inputHub">IoT Hub:</label>
          <div class="controls">
            <input type="text" id="inputHub" placeholder="IoT Hub" v-model.trim="hub">
            <span class="help-inline" v-show="hub_error.length > 0">{{ hub_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: device_id_error.length > 0}">
          <label class="control-label" for="inputDeviceId">Device ID:</label>
          <div class="controls">
            <input type="text" id="inputDeviceId" placeholder="Device ID" v-model.trim="device_id">
            <span class="help-inline" v-show="device_id_error.length > 0">{{ device_id_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: sa_key_error.length > 0}">
          <label class="control-label" for="inputSaKey">Shared Access Key:</label>
          <div class="controls">
            <input type="text" id="inputSaKey" placeholder="Shared Access Key" v-model.trim="config.sa_key">
            <span class="help-inline" v-show="sa_key_error.length > 0">{{ sa_key_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: data_precision_error.length > 0}">
          <label class="control-label" for="inputDataPrecision">Data Precision:</label>
          <div class="controls">
            <input id="inputDataPrecision" placeholder="Data Precision" type="text" v-model.trim="config.data_precision">
            <span class="help-inline" v-show="data_precision_error.length > 0">{{ data_precision_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>SSL/TLS</strong></label>
        </div>
        <div class="control-group" :class="{error: ca_file_error.length > 0}">
          <label class="control-label" for="inputCAFile">CA File:</label>
          <div class="controls">
            <div class="input-append">
              <input type="text" id="inputCAFile" placeholder="CA File (click to change)" v-model="config.ca_file" @click="changeCAFile" readonly class='default-cursor span11'>
              <button class="btn btn-link" type="button" @click="config.ca_file=''"><i class="icon-remove"></i></button>
            </div>
            <span class="help-inline" v-show="ca_file_error.length > 0">{{ ca_file_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <label class="control-label"><strong>Publish Interval</strong></label>
        </div>
        <div class="control-group" :class="{error: events_interval_error.length > 0}">
          <label class="control-label" for="inputEventInterval">Publish Interval:</label>
          <div class="controls">
            <div class="input-append">
              <input type="number" id="inputEventInterval" class="span7" placeholder="How often to publish data" v-model.number="config.events_interval">
              <span class="add-on">second</span>
            </div>
            <span class="help-inline" v-show="events_interval_error.length > 0">{{ events_interval_error }}</span>
          </div>
        </div>

        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="file_selector_template">
      <div id="file_selector_modal" class="modal hide fade" tabindex="-1" role="dialog" aria-labelledby="file_selector_modal_label" aria-hidden="true" >
        <div class="modal-header">
          <button type="button" class="close" data-dismiss="modal" aria-hidden="true">×</button>
          <h3 id="file_selector_modal_label">{{ title }}</h3>
        </div>

        <div class="modal-body">
          <div id="file_upload_panel" style="overflow:scroll;">
            <div class="dropzone needsclick dz-clickable">
              <div class="dz-message needsclick">
                Drag and Drop file here or click to upload.
              </div>
            </div>
            <div class="dropzone dropzone-previews">
							<div class="dz-message">{{ previewMsg }}</div>
							<div class="table table-striped files dz-clickable">
							</div>
            </div>
          </div>
        </div>

        <div class="modal-footer">
          <button class="btn btn-primary" @click="saveFilePath">Ok</button>
          <button class="btn" data-dismiss="modal" aria-hidden="true">Close</button>
        </div>
      </div>
    </script>
    
    <script type="text/x-template" id="file_preview_template">
			<div class="file-row">
				<!-- This is used as the file preview template -->
				<div>
						<span class="preview"><img data-dz-thumbnail /></span>
				</div>
				<div>
						<p class="name" data-dz-name></p>
						<strong class="error text-danger" data-dz-errormessage></strong>
				</div>
				<div>
						<p class="size" data-dz-size></p>
						<div class="progress progress-striped active" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0">
							<div class="bar bar-success" style="width:0%;" data-dz-uploadprogress></div>
						</div>
				</div>
				<div>
<!--
					<button class="btn btn-primary start">
							<i class="glyphicon glyphicon-upload"></i>
							<span>Start</span>
					</button>
					<button data-dz-remove class="btn btn-warning cancel">
							<i class="glyphicon glyphicon-ban-circle"></i>
							<span>Cancel</span>
					</button>
-->
					<button data-dz-remove class="btn btn-danger delete">
						<i class="glyphicon glyphicon-trash"></i>
						<span>Delete</span>
					</button>
				</div>
			</div>
    </script>

    <script type="text/x-template" id="device_data_panel_template">
      <div id="deviceDataPanel" class="tabbable my-well flex-v-container">
        <ul class="nav nav-tabs">
          <li v-for="tab in tabs" v-once :class="{ active : tab.id == active_tab_id }"><a :href="'#' + tab.id" :data-tab-id='tab.id' data-toggle='tab'>{{ tab.label }}</a></li>
        </ul>
        <div class="tab-content flex-grow">
          <component v-for="tab in tabs" :is="tab.comp" :key="tab.id" :id="tab.id" :class="{ active : tab.id == active_tab_id }" class="tab-pane" v-on:objectsToSchema="addObjectsToSchema"/>
        </div>
      </div>
    </script>
    <script type="text/x-template" id="device_data_tree_comp_template">
      <div>
        <div class="row scrollable-content device-data-tree"></div>
        <p  v-if="showBtns" class="tree-btns">
        <button class="btn btn-mini" type="button" data-toggle="tooltip" title="add data point to schema" @click="addObjectToSchema">
          <i class="icon-plus"></i> Add
        </button>
        <button class="btn btn-mini" type="button" data-toggle="tooltip" title="reload selected node and its children" @click="reloadTree">
          <i class="icon-refresh"></i> Reload
        </button>
        </p>
      </div>
    </script>
    <script type="text/x-template" id="user_variable_tree_comp_template">
      <div>
        <div id="userVariableTree" class="row scrollable-content"></div>
        <p class="tree-btns">
          <button class="btn btn-mini" type="button" @click="addObjectToSchema">Add</button>
        </p>
      </div>
    </script>

    <script type="text/x-template" id="msg_snippet_editor_template">
      <modal-dialog :title="createMode ? 'Create Template' : 'Edit Template'" @submitChange="onSubmit" data-backdrop="static">
        <div class="control-group" :class="{error: templateNameError.length >0}">
          <label class="control-label" for="inputTmplName">Name:</label>
          <div class="controls">
            <input type="text" id="inputTmplName" 
              placeholder="New Template Name" v-model.trim="templateName">
            <span class="help-block">{{ templateNameError }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: categoryNameError.length > 0}">
          <label class="control-label" for="inputCategoryName">Category:</label>
          <div class="controls">
            <input type="text" id="inputCategoryName" class="span2" v-model.trim="categoryName">
            <div class="btn-group">
              <button class="btn dropdown-toggle" data-toggle="dropdown">
                Select
                <span class="caret"></span>
              </button>
              <ul class="dropdown-menu">
                <li v-for="cat in categoryNameList">
                  <a href="#" @click="categoryName=cat">{{cat}}</a>
                </li>
              </ul>
            </div>
            <span class="help-block">{{ categoryNameError }}</span>
          </div>
        </div>
      </modal-dialog>
    </script>

    <script type="text/x-template" id="msg_snippet_panel_template">
      <div class="flex-v-container scrollable-content" style="position: relative; height:100%;">
        <p class="text-center">Templates</p>
        <div id="msg_snippet_tree" class="tree scrollable-content flex-grow"></div>
        <div class="tree-btns">
          <button class="btn btn-mini" type="button" @click="useTemplate">
            <i class="icon-plus"></i> Use
          </button>
        </div>
        <msg-snippet-editor id="msg_snippet_editor" :createMode="editTemplateId === null" 
          :initTemplateName="curTemplateName" :initCategoryName="curCategoryName"
          :categoryNameList="categoryNameList" :templateNameError="templateNameError"
          :categoryNameError="categoryNameError"
          ref="msgSnippetEditor">
        </msg-snippet-editor>
      </div>
    </script>
    
    <script type="text/x-template" id="data_schema_panel_template">
      <div class='my-well flex-v-container'>
        <h4 class="center">Cloud Data Tree</h4>
        <div class="row flex-grow">
          <div id="dataSchemaTreeContainer" class="flex-v-container" style="position: relative;">
            <div id="dataSchemaTree" class="scrollable-content flex-grow"></div>
            <div class="tree-btns">
              <button class="btn btn-mini" type="button" @click="newTopicTree"><i class="icon-plus"></i> Msg</button>
              <button class="btn btn-mini" type="button" @click="newNode"><i class="icon-plus"></i> Folder</button>

              <div class="btn-group dropup">
                <a class="btn btn-mini btn-info dropdown-toggle" data-toggle="dropdown" href="#">
                  Actions
                  <span class="caret"></span>
                </a>
                <ul class="dropdown-menu">
                  <li> 
                    <a href="#" @click="removeNode"><i class="icon-trash"></i> <span class="text-error">Delete</span></a> 
                  </li>

                  <li class="divider"></li>

                  <li> 
                    <a href="#" @click="saveAsTemplate"><i class="icon-th-large"></i> Save As Template</a> 
                  </li>

                  <li class="divider"></li>

                  <li class="dropdown-submenu"> 
                    <a tabindex="-1" href="#"><i class="icon-file"></i> Clone Multi</a>
                    <ul class="dropdown-menu">
                      <li><a href="#" @click="cloneNode(2)">2 Times</a></li>
                      <li><a href="#" @click="cloneNode(4)">4 Times</a></li>
                      <li><a href="#" @click="cloneNode(8)">8 Times</a></li>
                      <li><a href="#" @click="cloneNode(12)">12 Times</a></li>
                    </ul>
                  </li>

                  <li> 
                    <a href="#" @click="cloneNode(1)"><i class="icon-file"></i> Clone</a> 
                  </li>

                  <li class="divider"></li>

                  <li> 
                    <a href="#" @click="validateMsgs"><i class="icon-eye-open"></i> Validate DataBinding</a> 
                  </li>
                  <li> 
                    <a href="#" @click="changeDataBinding"><i class="icon-eye-open"></i> Change DataBinding</a> 
                  </li>
                  <li class="divider"></li>

                  <li> 
                    <a href="#" @click="toggleNode"><i class="icon-resize-vertical"></i> Toggle</a> 
                  </li>

                  <li> 
                    <a href="#" @click="editNode"><i class="icon-edit"></i> Rename</a> 
                  </li>
                </ul>
              </div>
            </div>
          </div>

          <msg-snippet-panel-comp id="msgTmplTree" ref="msgSnippetPanel" @useTemplate="onUseTemplate"></msg-snippet-panel-comp>
          <data-binding-editor-comp ref="dataBindingEditor" @dataBindingChanged="onDataBindingChanged"></data-binding-editor-comp>
        </div>


        <modal-dialog id="msg_topic_picker_dialog" :title="newTopicTreeMode ? 'New Topic Tree' : 'Edit Topic'" v-on:submitChange="newTopicTreeMode ? createTopicTreeNode() : changeTopicTreeNode()">
          <div v-if="availableTopicNames.length > 0" class="control-group">
            <label class="control-label">Topic:</label>
            <div class="controls">
              <select v-model="selectedTopicName">
                <option v-for="topicName in availableTopicNames" :value="topicName">{{topicName}}</option>
                <!-- <option value="NoMQTTTopic">NoMQTTTopic</option> -->
              </select>
            </div>
          </div>
          <div v-else class="alert alert-block">
            There is no topics available, please create more topics from TopicManager first.
          </div>
        </modal-dialog>
      </div>
    </script>

    <script type="text/x-template" id="property_list_panel_template">
      <div class="comp-data-box">
        <button type="button" class="btn btn-fullwidth" data-toggle="collapse" :data-target="'#' + objDataUniqueDomID(objData.path)" @click="boxOpened = !boxOpened"><i :class="boxBtnClass"></i> {{ objData.path }}</button>
        <div :id="objDataUniqueDomID(objData.path)" class="collapse in">
        <table class="table table-striped table-hover table-condensed">
          <thead>
            <tr>
              <th></th>
              <th>Label</th>
              <th>Value</th>
              <th class="operation-box">Operations</th>
            </tr>
          </thead>
          <tbody>
              <tr v-for='(slot, index) in objData.slots'>
                <td  @click='slot.enabled = !slot.enabled'><input type='checkbox' v-model='slot.enabled'></td>
                <td><input type='text' class='input-small' v-model='slot.label' :disabled="!slot.enabled"></td>
                <td><input type='text' class='input-small' v-model='slot.value' :disabled="!slot.enabled"></td>
                <td class="operation-box">
                  <button class="btn btn-mini btn-danger btn-link" type="button" @click="objData.slots.splice(index, 1)">Delete</button>
                </td>
              </tr>
          </tbody>
        </table>
        <div class="comp-btn-group">
          <button class="btn btn-mini btn-success" id="newPropertyBtn" type="button" @click="addProperty">New</button>
          <button class="btn btn-mini" type="button" @click="selectAll">Select All</button>
          <button class="btn btn-mini btn-inverse" type="button" @click="inverseSelect">Inverse Select</button>
          <button class="btn btn-mini btn-warning" type="button" @click="selectNone">Select None</button>
          <button class="btn btn-mini btn-danger pull-right" type="button" @click="deleteSelectedSlot">Del Selected</button>
          <button id="delCompBtn" class="btn btn-mini btn-danger pull-right" type="button" @click="$emit('remove')">Del Comp</button>
        </div>
        </div>
      </div>
    </script>

    <script type="text/x-template" id="sedona_slots_edit_template">
      <table class="table table-striped table-hover table-condensed">
        <caption v-if="objDataList.length > 0">Path: <a href="#" @click.prevent="locateDataNodeByPath"><strong>{{ objDataList[0].path }}</strong></a></caption>
        <thead>
          <tr>
            <th></th>
            <th>Label</th>
            <th>Binding</th>
          </tr>
        </thead>
        <tbody>
          <template v-for='objData in objDataList'>
          <tr v-for='(slotData, index) in objData.slots' v-show="slotData.visible">
            <td><input type='radio' :value="slotData.uid" v-model='selectedSlotUID'></td>
            <td><input type='text' class='input-small' v-model.trim='slotData.label' 
              :disabled="selectedSlotUID!=slotData.uid" :class="{error: errorSlotUID == slotData.uid}"></td>
            <td>{{ shortBindingPath(slotData) }}</td>
          </tr>
          </template>
        </tbody>
      </table>
    </script>

    <script type="text/x-template" id="variable_slots_edit_template">
      <table class="table table-striped table-hover table-condensed">
        <thead>
          <tr>
            <th>Label</th>
            <th>Value</th>
          </tr>
        </thead>
        <tbody>
          <template v-for='objData in objDataList'>
          <tr v-for='(slotData, index) in objData.slots' v-show="slotData.visible">
            <td>
              <input type='text' class='input-small' v-model.trim='slotData.label' 
                :class="{error: errorSlotUID == slotData.uid}">
            </td>
            <td>
              <template v-if='isPublishTopic'>
                <template v-if='isUserVarPath(slotData.path)'>
                  <input v-if='slotData.type == "int"' type='number' class='input-small' v-model.number='slotData.value' 
                    :class="{error: errorSlotUID == slotData.uid}">
                  <input v-if='slotData.type == "string"' type='text' class='input-small' v-model.trim='slotData.value' 
                    :class="{error: errorSlotUID == slotData.uid}">
                  <select v-if='slotData.type == "bool"' v-model='slotData.value'>
                    <option>true</option>
                    <option>false</option>
                  </select>
                </template>
                <template v-else-if='isSystemVarPath(slotData.path)'>
                  (the value will be set on backend)
                </template>
              </template>
            </td>
          </tr>
          </template>
        </tbody>
      </table>
    </script>

    <script type="text/x-template" id="node_schema_panel_template">
      <div class='my-well flex-v-container'>
        <h4 class="center">Cloud Tree Node Schema</h4>
        <div class="row scrollable-content flex-grow" style="margin-bottom:30px;">
          <template v-if="isVarObject">
            <variable-slots-edit-comp ref="varsEditor" :node-i-d="nodeID" :obj-data-list="objDataList" :is-publish-topic="isPublishTopic" @nodeSchemaUpdated="onNodeSchemaUpdated"></variable-slots-edit-comp>
          </template>
          <template v-else>
            <sedona-slots-edit-comp ref="slotsEditor" :node-i-d="nodeID" :obj-data-list="objDataList" :initial-selected-slot-id="selectedSlotUID" @nodeSchemaUpdated="onNodeSchemaUpdated"></sedona-slots-edit-comp>
          </template>
        </div>
        <p class="tree-btns">
          <!-- 
          <button class="btn btn-mini" type="button" @click="selectAll">Select All</button>
          <button class="btn btn-mini btn-inverse" type="button" @click="inverseSelect">Inverse Select</button>
          <button class="btn btn-mini btn-warning" type="button" @click="selectNone">Select None</button>
          <button class="btn btn-mini btn-danger pull-right" type="button" @click="deleteSelectedSlot">Del Selected</button>
          -->
          <transition name="fade" mode="out-in">
          <button class="btn btn-mini" type="button" :key="editMode" @click="saveNodeSchema">
          <i :class="[editMode ? 'icon-edit' : 'icon-plus',]"></i> {{ editMode ? 'Update' : 'Add' }}
          </button>
          </transition>
        </p>
      </div>
    </script>

    <script type="text/x-template" id="name_replacement_preprocessor_template">
      <div> 
        <form class="form-inline">
          <label>When slot's name is :</label>
          <input type='text' class='input-small' v-model='payload.old_name' :disabled="!enabled"></td>
          <label> then rename it to :</label>
          <input type='text' class='input-small' v-model='payload.new_name' :disabled="!enabled"></td>
          <a v-if="tooltip && tooltip.length > 0" v-once class="pull-right" href="#" data-toggle="tooltip" data-placement="top" :title="tooltip"><i class="icon-question-sign"></i></a>
        </form>
      </div>
    </script>

    <script type="text/x-template" id="preselect_preprocessor_template">
      <div>
        <form class="form-inline">
          <label>Preselect slot when its name match :</label>
          <input type='text' class='input-small' :disabled="!enabled"></td>
          <a v-if="tooltip && tooltip.length > 0" v-once class="pull-right" href="#" data-toggle="tooltip" data-placement="top" :title="tooltip"><i class="icon-question-sign"></i></a>
        </form>
      </div>
    </script>

    <script type="text/x-template" id="schema_preview_panel_template">
      <div class='row' >
        <div class="alert alert-info">
          <button type='button' class='close' data-dismiss='alert'>x</button>
          <strong>Note:</strong> all property values (number, boolean, string etc) are fake, only for demo purpose.
        </div>
        <schema-preview-entry v-for="(schema_json, index) in schema_jsons" :key='index' :topic='schema_json[0]' :json='schema_json[1]'></schema-preview-entry>
      </div>
    </script>

    <script type="text/x-template" id="preprocessor_list_template">
      <div class='row scrollable-content content-panel' >
        <div id='preprocessorBtnGroup' class='btn-group dropdown pull-right'>
          <a class='btn dropdown-toggle' data-toggle='dropdown' href='#'>
            Add New Preprocessor
            <span class='caret'></span>
          </a>
          <ul class="dropdown-menu">
            <li v-for="preprocessor in type_list">
              <a href="#" @click.prevent="addPreprocessorElem(preprocessor)"><i :class="preprocessor.icon"></i> {{ preprocessor.label }}</a>
            </li>
          </ul>
        </div>
        <table class="table table-striped table-hover">
          <thead>
            <tr>
              <th></th>
              <th></th>
              <th>Preprocessor</th>
              <th class="operation-box">Operation</th>
            </tr>
          </thead>
          <tbody>
            <tr v-for='(inst, index) in instance_list'>
              <td  @click='inst.enabled = !inst.enabled'>
                <input type='checkbox' v-model='inst.enabled'>
              </td>
              <td> <i :class="iconFromComp(inst.component)"></i> </td>
              <td>
                <component :is="inst.component" ref="preprocessor" :payload="inst.payload" :enabled="inst.enabled"></component>
              </td>
              <td class="operation-box">
                <button class="btn btn-mini btn-link" type="button" @click="instance_list.push(_.clone(inst))">Clone</button>
                <button class="btn btn-mini btn-danger btn-link" type="button" @click="instance_list.splice(index, 1)">Delete</button>
              </td>
            </tr>
          </tbody>
        </table>
      </div>
    </script>

    <script type="text/x-template" id="data_mapping_template">
      <div class='row tabbable'>
        <ul class="nav nav-tabs">
          <li v-for="tab in tabs" v-once :class="{ active : tab.id == active_tab_id }"><a :href="'#' + tab.id" :data-tab-id='tab.id' data-toggle='tab' @click='onShowTab'>{{ tab.label }}</a></li>
        </ul>
        <div class='row tab-content'>
          <div :id='tabs[0].id' class='row tab-pane' :class="{ active: tabs[0].id == active_tab_id }" 
            style="height:calc(100vh - 200px);" >
            <data-schema-panel id="dataSchemaPanel" ref="dataSchemaPanel" class="split" 
              v-on:nodeChanged="onNodeChanged">
            </data-schema-panel>
            <node-schema-panel id="nodeSchemaPanel" ref="nodeSchemaPanel" class="split" 
              v-on:nodeSchemaUpdated="onNodeSchemaUpdated" v-on:objectAdded="onObjectAdded" >
            </node-schema-panel>
            <device-data-panel id="deviceDataPanel" ref="deviceDataPanel" class="split" v-on:objectsToSchema="addObjectsToSchema" ></device-data-panel>
          </div>
          <schema-preview-panel :id='tabs[1].id' ref="schemaPreviewPanel" class='tab-pane' :class="{ active: tabs[1].id == active_tab_id }" :schema_jsons="previewJsons" style="max-height:calc(100vh - 200px);"></schema-preview-panel>
          <preprocess-list-panel :id='tabs[2].id' ref="preprocessorListPanel" class='tab-pane' :class="{ active: tabs[2].id == active_tab_id }" style="max-height:calc(100vh - 200px);"></preprocess-list-panel>
        </div>
      </div>
    </script>

    <script type="text/x-template" id="broker_manager_template">
      <div class="row-fluid" v-cloak>
        <div id="broker_list" class="span6" style="overflow:auto;">
          <div class="row" style="overflow:auto; max-height:calc(100vh - 170px); margin-bottom:10px;">
            <table class="table table-striped table-hover table-bordered">
              <thead>
                <tr>
                  <sortable-table-header :sortField="sortField" fieldName="name" @sortBy="sortBy"></sortable-table-header>
                  <sortable-table-header :sortField="sortField" fieldName="type" @sortBy="sortBy"></sortable-table-header>
                  <th>State</th>
                  <th>Action</th>
                </tr>
              </thead>
              <tbody>
                <tr v-for="broker in brokerList" @click="activeBroker=broker" :class="{info:activeBroker && activeBroker.uniqueKey==broker.uniqueKey}">
                  <td>{{broker.name}}</td>
                  <td>{{broker.type | typeLabel}}</td>

                  <td v-if="broker.state_code == -1"><span class="badge">-</span></td>
                  <td v-else-if="broker.state_code == 0" data-toggle="tooltip" :title="broker.state_msg"><span class="badge badge-success">Success</span></td>
                  <td v-else data-toggle="tooltip" :title="broker.state_msg"><span class="badge badge-important">Error</span></td>

                  <td>
                    <button class="btn" :class="[broker.config.enabled ? 'btn-info' : 'btn-inverse',]" @click="broker.config.enabled = !broker.config.enabled">
                      {{ broker.config.enabled ? 'Enabled' : 'Disabled' }}
                    </button>
                  </td>
                </tr>
              </tbody>
            </table>
          </div>
          <div class="row">
            <div class="pull-right">
              <button class="btn" type="button" @click="addBroker">Add</button>
              <button class="btn" type="button" @click="cloneBroker" :disabled="activeBroker==null">Clone</button>
              <button class="btn btn-danger" type="button" @click="deleteBroker" :disabled="activeBroker==null">Delete</button>
            </div>
          </div>
        </div>
        <div id="broker_detail" class="well span6">
          <keep-alive>
          <component v-if="activeBroker != null" :is="activeBroker.type" :key="activeBroker.uniqueKey" 
            :brokerName="activeBroker.name" :configData="activeBroker.config" :nameError="nameError"
            @brokerUpdated="onBrokerUpdated" ></component>
          </keep-alive>
        </div>

        <modal-dialog id="new_broker_dlg" title="New Broker" :error="newBroker.error.length != 0" v-on:submitChange="createBroker">
          <div class="control-group" :class="{error: !isNewNameValid(newBroker.name)}">
            <label class="control-label" for="inputBrokerName">Name:</label>
            <div class="controls">
              <input type="text" id="inputBrokerName" placeholder="Broker's Name" v-model.trim="newBroker.name" autofocus>
              <span class="help-inline" v-show="newBroker.error.length != 0">{{newBroker.error}}</span>
            </div>
          </div>
          <div class="control-group">
            <label class="control-label">Type:</label>
            <div class="controls">
              <select v-model="newBroker.type">
                <option v-for="type in brokerTypeList" :value="type">{{type | typeLabel}}</option>
              </select>
            </div>
          </div>
        </modal-dialog>
      </div>
    </script>

    <script type="text/x-template" id="sortable_table_header_template">
      <th @click="ascendingSort = !ascendingSort">
        <slot>
        {{fieldName | capitalize}}
        </slot>
        <a v-show="sortField == fieldName" class="pull-right" href="#"><i :class="[ascendingSort ? 'icon-chevron-down' : 'icon-chevron-up']"></i></a>
      </th>
    </script>
    
    <script type="text/x-template" id="gcp_iot_core_topic_template">
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>GCP IoT Core Topic</strong></label>
        </div>

        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputTopicName">Topic Name:</label>
          <div class="controls">
            <input type="text" id="inputTopicName" placeholder="Topic Name" v-model.trim="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: path_error.length > 0}">
          <label class="control-label" for="inputTopicPath">Topic Path:</label>
          <div class="controls">
            <template v-if="category == 0"> 
              <div class="input-prepend">
                <span class="add-on">{{pathPrefix}}</span>
                <input type="text" id="inputTopicPath" class="span5" placeholder="Topic Path" v-model.trim="subPath">
              </div>
              <span class="help-inline" v-show="path_error.length > 0">{{ path_error }}</span>
            </template>
            <template v-if="category == 1"> 
              <input type="text" id="inputTopicPath" placeholder="Topic Path" :value="path" readonly>
            </template>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label">Topic Category:</label>
          <div class="controls">
            <select v-model.number="category">
              <option value="0">Publish</option>
              <option value="1">Subscribe</option>
            </select>
          </div>
        </div>

        <div class="control-group" v-if="proxied_devices.length > 0" :class="{error: proxiedDeviceError.length > 0}">
          <label class="control-label">Proxied Device:</label>
          <div class="controls">
            <select v-model="proxiedDeviceIdCopy">
              <option value=""></option>
              <option v-for="device in proxied_devices" :value="device.device_id">{{device.device_id}}</option>
            </select>
            <span class="help-inline" v-show="proxiedDeviceError.length > 0">{{ proxiedDeviceError }}</span>
          </div>
        </div>


        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" :class="{disabled: name_error.length != 0 || path_error.length != 0}" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="azure_iot_hub_topic_template">
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>Azure IoT Hub Topic</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputTopicName">Topic Name:</label>
          <div class="controls">
            <input type="text" id="inputTopicName" placeholder="Topic Name" v-model.trim="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label">Topic Path:</label>
          <div class="controls">
            <input type="text" placeholder="Topic Path" :value="path" readonly>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label">Topic Category:</label>
          <div class="controls">
            <select v-model.number="category">
              <option value="0">Publish</option>
              <option value="1">Subscribe</option>
            </select>
          </div>
        </div>

        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="mqtt_topic_template">
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>MQTT Topic</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputTopicName">Topic Name:</label>
          <div class="controls">
            <input type="text" id="inputTopicName" placeholder="Topic Name" v-model.trim.lazy="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: path_error.length > 0}">
          <label class="control-label" for="inputTopicPath">Topic Path:</label>
          <div class="controls">
            <input type="text" id="inputTopicPath" placeholder="Topic Path" v-model.trim="config.path">
            <span class="help-inline" v-show="path_error.length > 0">{{ path_error }}</span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label">Topic Category:</label>
          <div class="controls">
            <select v-model.number="category">
              <option value="0">Publish</option>
              <option value="1">Subscribe</option>
            </select>
          </div>
        </div>

        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="aws_iot_core_topic_template">
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label"><strong>AWS IoT Core Topic</strong></label>
        </div>
        <div class="control-group" :class="{error: name_error.length > 0}">
          <label class="control-label" for="inputTopicName">Topic Name:</label>
          <div class="controls">
            <input type="text" id="inputTopicName" placeholder="Topic Name" v-model.trim.lazy="name">
            <span class="help-inline" v-show="name_error.length > 0">{{ name_error }}</span>
          </div>
        </div>
        <div class="control-group" :class="{error: path_error.length > 0}">
          <label class="control-label" for="inputTopicPath">Topic Path:</label>
          <div class="controls">
            <input type="text" id="inputTopicPath" placeholder="Topic Path" v-model.trim="config.path">
            <span class="help-inline" v-show="path_error.length > 0">{{ path_error }}</span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label">Topic Category:</label>
          <div class="controls">
            <select v-model.number="category">
              <option value="0">Publish</option>
              <option value="1">Subscribe</option>
            </select>
          </div>
        </div>

        <div class="control-group">
          <div class="controls">
            <button type="submit" class="btn btn-success" @click.prevent="onSaveLocal">Ok</button>
            <button class="btn" @click.prevent="onReset">Reset</button>
          </div>
        </div>
      </form>
    </script>

    <script type="text/x-template" id="topic_manager_template">
      <div class="row-fluid" v-cloak>
        <div id="topic_list" class="span6" style="overflow:auto;">
          <div class="row" style="overflow:auto; max-height:calc(100vh - 170px); margin-bottom:10px;">
            <table class="table table-striped table-hover table-bordered">
              <thead>
                <tr>
                  <sortable-table-header :sortField="sortField" fieldName="name" @sortBy="sortBy"></sortable-table-header>
                  <sortable-table-header :sortField="sortField" fieldName="Connection/Device" @sortBy="sortBy"></sortable-table-header>
                  <sortable-table-header :sortField="sortField" fieldName="category" @sortBy="sortBy"></sortable-table-header>
                  <th>Action</th>
                </tr>
              </thead>
              <tbody>
                <tr v-for="topic in topicList" @click="activeTopic=topic" :class="{info: activeTopic && activeTopic.uniqueKey==topic.uniqueKey}">
                  <td>{{topic.name}}</td>
                  <td>{{brokerLabel(topic)}}</td>
                  <td>{{(topic.category === undefined || topic.category == 0) ? "Publish" : "Subscribe" }}</td>
                  <td>
                    <button class="btn" :class="[topic.enabled ? 'btn-info' : 'btn-inverse',]" @click="topic.enabled = !topic.enabled">
                      {{ topic.enabled ? 'Enabled' : 'Disabled' }}
                    </button>
                  </td>
                </tr>
              </tbody>
            </table>
          </div>
          <div class="row">
            <div class="pull-right">
              <button class="btn" type="button" @click="addTopic">Add</button>
              <button class="btn btn-danger" type="button" @click="deleteTopic">Delete</button>
            </div>
          </div>
        </div>
        <div id="topic_detail" class="well span6">
          <keep-alive>
          <component v-if="activeTopic != null && activeTopic.broker != null" :is="activeTopicType" 
            :key="activeTopic.uniqueKey" :topicName="activeTopic.name" :configData="activeTopic.config" 
            :topicCategory="activeTopic.category === undefined ? 0 : activeTopic.category" 
            :nameError="nameError" :proxiedDeviceError="proxiedDeviceError"
            :broker="activeTopic.broker" 
            :proxiedDeviceId="activeTopic.proxied_device ? activeTopic.proxied_device : ''"
            @topicUpdated="onTopicUpdated" @topicReset="onTopicReset"></component>
          </keep-alive>
        </div>

        <modal-dialog title="New Topic" :error="newTopic.error.length != 0" 
          v-on:submitChange="createTopic" data-backdrop="static">
          <div v-if="brokerList.length > 0">

            <div class="control-group">
              <label class="control-label">Connection:</label>
              <div class="controls">
                <select v-model="newTopic.broker">
                  <option v-for="broker in brokerList" :value="broker.name">{{broker.name}}</option>
                </select>
              </div>
            </div>

            <div class="control-group" :class="{error: !isNewTopicValid(newTopic)}">
              <label class="control-label" for="inputNewTopicName">Name:</label>
              <div class="controls">
                <input type="text" id="inputNewTopicName" placeholder="Topic's Name" v-model.trim="newTopic.name" autofocus>
                <span class="help-inline" v-show="newTopic.error.length != 0">{{newTopic.error}}</span>
              </div>
            </div>
            <div class="control-group">
              <label class="control-label">Category:</label>
              <div class="controls">
                <select v-model.number="newTopic.category">
                  <option value="0">Publish</option>
                  <option value="1">Subscribe</option>
                </select>
              </div>
            </div>
          
            <div class="control-group" v-if="proxied_devices.length > 0">
              <label class="control-label">Proxied Device:</label>
              <div class="controls">
                <select v-model="newTopic.proxied_device">
                  <option value=""></option>
                  <option v-for="device in proxied_devices" :value="device.device_id">{{device.device_id}}</option>
                </select>
              </div>
            </div>

          </div>
          <div v-else class="alert alert-block">
            There is no broker available, please create some broker from BrokerManager first.
          </div>
        </modal-dialog>
      </div>
    </script>

    <script type="text/x-template" id="config_upload_template">
      <modal-dialog title="Upload MQTT Config" data-backdrop="static">
        <div class="alert">
          <button type="button" class="close" data-dismiss="alert">&times;</button>
          By uploading your MQTT config, the current config data will be overwritten. If you want to keep a copy of current config, please download it NOW before uploading: click "Utilities" button at bottom, then select "Download Config" item.
        </div>
        <div id="mqtt_config_uploader_panel" style="overflow:scroll;">
          <div class="dropzone needsclick dz-clickable">
            <div class="dz-message needsclick">
              Drag and Drop MQTT config file here or click to upload.
            </div>
          </div>
          <div class="dropzone dropzone-previews" style="display:none;"> </div>
        </div>
        <div class="alert" :class="[msg.startsWith('Error:') ? 'alert-error' : 'alert-info', ]" v-show="msg.length != 0" style="margin-top:4px;">
          {{ msg }}
        </div>

        <template slot="buttons">
          <button class="btn" data-dismiss="modal" aria-hidden="true">Close</button>
        </template>
      </modal-dialog>
    </script>

    <script type="text/x-template" id="gcp_proxied_devices_manager_template">
      <div style="height:340px;overflow:auto;">
        <div class="row">
          <div class="alert input-validation-error" v-show="newDeviceError.length != 0">
            {{ newDeviceError }}
          </div>
          <input type="text" class="pull-left" ref="newEdit" placeholder="New Device ID" 
            v-model="newDevice.device_id" @keyup.enter="addDevice">
          <button type="button" class="pull-left btn btn-info btn-small" 
            style="margin-left:10px;" @click.stop="addDevice">Add</button>
        </div>
        <div class="row div-hr"></div>
        <div class="row" style="height:300px;overflow:auto;">
          <table class="table table-striped">
            <thead>
              <tr>
                <th>Device ID</th>
                <th style="text-align:right;">Action</th>
              </tr>
            </thead>

            <tbody>
              <tr v-for="d in devicesSnapshot" @click="setActiveDevice(d)" 
                :class="[isDeviceActive(d) ? 'info' : '']">
                <td v-if="isDeviceActive(d)">
                  <div class="alert input-validation-error" v-show="editDeviceError.length != 0">
                    {{ editDeviceError }}
                  </div>
                  <input type="text" class="pull-left" ref="activeEdit" placeholder="Device ID"
                    v-model="editDevice.device_id" @keyup.enter="updateDevice(d.uid)">
                  <button type="button" class="pull-left btn btn-info btn-small" 
                    style="margin-left:10px;" @click.stop="updateDevice(d.uid)">Update</button>
                  <button type="button" class="pull-left btn btn-small" 
                    style="margin-left:10px;" @click.stop="cancelUpdate()">Cancel</button>
                </td>
                <td v-else>
                  {{d.device_id}}
                </td>

                <td style="text-align:right;">
                  <button type="button" class="btn btn-mini btn-link" @click.stop="deleteDevice(d.uid)">
                    <i class="icon-trash"></i>
                  </button>
                </td>
              </tr> 

            </tbody>
          </table>
        </div>
      </div>
    </script>

    <script type="text/x-template" id="cert_generator_template">
      <div class="modal hide fade cert-generator" style="width:600px;" data-backdrop="static">
        <div class="modal-header">
          <button type="button" class="close" data-dismiss="modal" aria-hidden="true">×</button>
          <h3>Certificate Generator</h3>
        </div>

        <div class="modal-body" style="max-height:530px;">
          <div class="alert alert-warning" v-show="error.length > 0">
            <button type='button' class='close' data-dismiss='alert'>x</button>
            {{ error }}
          </div>

          <form class="form-inline" @submit.prevent="onGenerate">
            <label>Cert Name: </label>
            <input type='text' placeholder='-, _, 0-9, a-z, A-Z, max 32 chars' v-model="name">
            <button type="submit" class="btn btn-info">Generate</button>
            <button type="button" class="btn" v-show="cert_content.length > 0" @click="copyCert">Copy Cert</button>
          </form>
          <div class="row" style="margin-top: 20px;"> 
            <div class="well">
              <pre>{{ cert_content }}</pre>
            </div>
          </div>
        </div>

        <div class="modal-footer">
          <button type="button" class="btn" data-dismiss="modal" aria-hidden="true">Close</button>
        </div>
      </div>
    </script>

    <script type="text/x-template" id="data_binding_editor_template">
      <modal-dialog class="data-binding-dlg" title="Change DataBinding" @submitChange="onSubmit" data-backdrop="static">
        <div class="alert alert-warning" v-show="error.length > 0">
          {{ error }}
        </div>
        <div class="row">
          <h5>1. Choose data point to be replaced:</h5>
          <select style="margin-left:12px; width:518px;" v-model="srcBinding">
            <option v-for="(binding, index) in possibleTopBindings" :key="index" :value="binding">{{binding}}</option>
          </select>
        </div>
        <div class="row" style="max-height:500px; margin-top:6px;">
          <h5>2. Pick new data point:</h5>  
          <device-tree :showBtns="false" :listenMsgBusEvent="false" ref="deviceDataTree" style="margin-left:12px;height:400px;"></device-tree>
        </div>
      </modal-dialog>
    </script>

    <script type="text/x-template" id="modal_dialog_template">
      <div class="modal hide fade">
        <div class="modal-header">
          <button type="button" class="close" data-dismiss="modal" aria-hidden="true">×</button>
          <h3>{{title}}</h3>
        </div>
        <div class="modal-body">
          <form class="form-horizontal" @submit.prevent="enterToSubmit && onSubmit()" style="margin-bottom:0">
            <slot></slot>
          </form>
        </div>
        <div class="modal-footer">
          <slot name="buttons">
            <button type="button" class="btn btn-primary" @click="onSubmit">{{saveBtnLabel}}</button>
            <button type="button" class="btn" data-dismiss="modal" aria-hidden="true">Cancel</button>
          </slot>
        </div>
      </div>
    </script>

    <script type="text/javascript" src="<?php A('../../js/jquery-1.10.0.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/bootstrap.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/underscore-min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/spin.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/renderjson.js') ?>"></script>

<?php
  if (isset($_GET["dev"])) {
?>
    <script type="text/javascript" src="<?php A('../../js/dropzone.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/vue.js') ?>"></script>
<?php
  } else {
?>
    <script type="text/javascript" src="<?php A('../../js/dropzone.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/vue.min.js') ?>"></script>
<?php
  }
?>

    <script type="text/javascript" src="<?php A('../../js/plugin-utils.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/vue-alert-component.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/vue-file-selector-component.js') ?>"></script>

    <!-- used by data mapping -->
    <script type="text/javascript" src="<?php A('../../js/split.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/jstree/jstree.min.js') ?>"></script>

    <!-- used by azure iot hub -->
    <script type="text/javascript" src="<?php A('js/hmac-sha256-min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('js/enc-base64-min.js') ?>"></script>

    <script type="text/javascript" src="<?php A('js/data_service_config.js') ?>"></script>

    <script language="javascript" type="text/javascript">
      "use strict";
      window.service_version = "<?php echo $controller->getServiceVersion(); ?>";

      window.in_iframe = window !== window.parent;
      if (window.in_iframe) {
        let elem = document.getElementById("home_nav_btn");
        if(elem) elem.setAttribute("href", "#");
      }
    </script>

  </body>
  </html>
