"use strict";
(function() {
  
  var UserLibClass = function() 
  {
    
    // define javascript methods that will be called when widget need to be initialized or data updated. 
    // in these methods' context, beside jQuery, following variables are accessible:
    //   * this.elem       - parent DOM element 
    //   * User Properties - you can define user property on AdapterWidget in CPT:
    //                        1. select a AdapterWidget object 
    //                        2. right click in the property editor
    //                        3. choose "New User Property" menu item
    //                       NOTE: all user defined properties must start with '@'. 
    //                       To read a user property's value: this.readData("@UserPropertyName");
    //                       To write a user property's value: this.writeData("@UserPropertyName", value);
    //                       To invoke an action: this.invokeAction("ActionSlotPath") or this.invokeAction("ActionSlotPath", value)
    //
    //
    // [optional] if this method defined, it should return a array of required javascript and css files
    // several js libraries(jquery, underscore, backbone etc) have been loaded, so no need to specify 
    // them here again. for all loaded js libraries, refers to file CPT/grweb/public/app/graphic.php
    // below is an example for highcharts:
    this.requiredScripts = function() {   
      // *Note*, you need to put the javascript files under CPT/grweb/public/user_codes folder
      //         manually, otherwise these files cannot be deployed to device
      //return ["../user_codes/highcharts/js/highcharts.js",
      //        "../user_codes/highcharts/js/highcharts-more.js",
      //        "../user_codes/highcharts/js/exporting.js"];
    };
    
    this.getReadDelayAfterWrite = function() {
      var delay = this.readData("@ReadDelayAfterWrite");
      if (typeof this.isDashboard !== 'function' || !this.isDashboard()) {
        //CPT Graphics does not support enum with customized values, 
        //so we need to convert from index to value here. Yes, it sucks 
        delay = [15000, 30000, 60000][delay];
      }
      else {
        if (delay === null || Number.isNaN(_.str.toNumber(delay)))
          delay = 15000;
        else
          delay = _.str.toNumber(delay)*1000;
      }
      return delay;
    };
    
    this.onRadioChanged = function(event) {
      var newVal = $(event.target).val();
      var path = this.readData("@SetPoint");
      if (path === null)
        return;
      
      if (newVal == _.str.toNumber(this.readData("@data")))
        return;
        
      var _this = this;
      var delay = this.getReadDelayAfterWrite();
    
      if (this.updateTimer !== null) {
        clearTimeout(this.updateTimer);
        this.updateTimer = null;
      }
      
      var enableDataUpdate = _.after(2, function() { 
        _this.stopUpdate = false; 
        if (typeof _this.isDashboard !== 'function' || !_this.isDashboard())
          _this.update("@data");
      });
      this.updateTimer = setTimeout(enableDataUpdate, delay);
      
      this.stopUpdate = true;
      var settings = {'complete': enableDataUpdate};
      this.invokeAction(path, newVal, null, settings);
    };
    
    this.setupUI = function() {
      if (this.updateTimer !== null) {
        clearTimeout(this.updateTimer);
        this.updateTimer = null;
      }
      
      $(this.elem).empty();
      
      var desc = this.readData("@Description");
      if (desc)
        $(this.elem).append("<div><p style='margin-bottom: 2px;' >" + desc + "</p></div>");
        
      var choices = this.parseEnumData("@ChoiceList");
      for(var i=0; i<choices.length; ++i)
      {
        var c = choices[i];
      
        $("<div><input type='radio' name='" + this.containerId + "_radio' value='" + c.value + "' >" + c.label + "</div>").appendTo($(this.elem)).on("change", _.bind(this.onRadioChanged, this));
      }
    }
    
    // [required] after above required files loaded successfully, this method will be called to initialize the widget
    this.init = function() { 
      this.stopUpdate = false;
      this.updateTimer = null;
      this.setupUI();
      
      $(this.elem).on('settingsChanged', _.bind(this.setupUI, this));
      
      this.update("@data");
    };
    
    // [required] when user property changed, this callback method will be called
    // it is a good place to update widget with new data, you can read user property's value by:
    // this.readData("@UserPropertyName")
    this.update = function(userPropertyName) {
      if (this.stopUpdate || userPropertyName != "@data")
        return;
      
      var val = this.readData(userPropertyName);
      if (val === null)
        return;
        
      var parsedVal = parseFloat(val);
      if (isNaN(parsedVal))
        return;
        
      //fix float number error 
      parsedVal = (parsedVal*100000).toFixed(0)/100000.0;
      
      var radio = $(this.elem).find("input[value='"+parsedVal+"']");
      if (radio.length > 0) {
        radio.prop('checked', true);
        $(this.elem).trigger("widget.ok");
      } else
        $(this.elem).trigger("widget.error", "Out of range");
    };
    
    // [required] put possible clean up codes here
    this.cleanup = function() {
    };
                      
  };

  var userLib = new UserLibClass();
  var external_scripts = _.has(userLib, 'requiredScripts') ? userLib.requiredScripts() : [];
  if (external_scripts == null)
    external_scripts = [];
  external_scripts.push("../js/underscore.string.min.js");

  //TODO: generate settings from user_lib
  freeboard.loadWidgetPlugin({
    'external_scripts': external_scripts,
    'fill_size': false,
    'type_name': 'CPT_Selection_RadioSelection_Widget',
    'display_name': 'RadioSelection',
    'description': 'RadioSelection Widget converted from CPT Graphics'    ,
    'settings': [
      {
        'name': '@ChoiceList',
        'display_name': '@ChoiceList',
        'type': 'calculated',
        'default_value': '',
        'editor': '',
        'description': 'format: CHOICE1:VALUE1,CHOICE2:VALUE2,...'
      },
      {
        'name': '@Description',
        'display_name': '@Description',
        'type': 'calculated',
        'default_value': '',
        'editor': ''
      },
      {
        'name': '@ReadDelayAfterWrite',
        'display_name': '@ReadDelayAfterWrite',
        'type': 'option',
        'options': [{'name': 'Normal', 'value': '15'},{'name': 'Slow', 'value': '30'},{'name': 'Very Slow', 'value': '60'}],
        'default_value': '15',
        'editor': ''
      },
      {
        'name': '@SetPoint',
        'display_name': '@SetPoint',
        'type': 'calculated',
        'default_value': '',
        'editor': 'action'
      },
      {
        'name': '@data',
        'display_name': '@data',
        'type': 'calculated',
        'default_value': '',
        'editor': ''
      },
      {
        'name': 'rows',
        'display_name': 'Rows',
        'type': 'option',
        'default_value': '1',
        'options': [{'name': '1 row', 'value': 1}, {'name': '2 rows', 'value': 2}, {'name': '3 rows', 'value': 3}, {'name': '4 rows', 'value': 4}, {'name': '5 rows', 'value': 5}, {'name': '6 rows', 'value': 6}, {'name': '7 rows', 'value': 7}, {'name': '8 rows', 'value': 8}]
      }      
    ], 
    newInstance: function(settings, newInstanceCallback)
    {
      var wa = new WidgetAdapter(settings);
      wa = _.extend(wa, new UserLibClass());
      newInstanceCallback(wa);
    }
  });

})();
