<?php
// vim: ts=2 sw=2

include_once "lang.php";
include_once "device_config_file.php";
 
define("WEB_APP_VERSION", '1.24.09.13'); // format: 1.yy.mm.dd

function randStr($length=16)
{
  return substr(str_shuffle("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, $length);
}

function hashPassword($password, $salt)
{
  return hash('sha256', $password . $salt);
}

function arrayMapToString($array, $connector)
{
  $result = array();
  foreach($array as $k => $v) {
    $result[] = "$k='$v'";
  }
  return implode($connector, $result);
}

function arrayGet($array, $key, $default=null) {
  return isset($array[$key]) ? $array[$key] : $default;
}

function startsWith($haystack, $needle) {
    return $needle === "" || strpos($haystack, $needle) === 0;
}

function endsWith($haystack, $needle) {
  return $needle === "" || substr($haystack, -strlen($needle)) === $needle;
}

function parentPath($level) {
  $curPath = __FILE__;
  $parts = explode(DIRECTORY_SEPARATOR, $curPath);
  for($i=0; $i<$level; ++$i)
    array_pop($parts);
  return realpath(implode(DIRECTORY_SEPARATOR, $parts));
}

function cptBaseDir() {
  if (strpos(__FILE__, "/ghost_scripts") !== false) {
    $rootPath = $_SERVER['DOCUMENT_ROOT'];
    return build_file_path($rootPath, 'sdcard', 'cpt');
  } else {
    return parentPath(2);
  }
}

function sync() {
    // shell_exec("sync");
    // shell_exec("echo 3 > /proc/sys/vm/drop_caches");
    // return;

    $output = array();
    $return_val = 0;
    exec("sync", $output, $return_val);
    error_log("sync output : " . implode("\n", $output) . " return: " . $return_val);

    unset($output);

    $return_val = 0;
    exec("echo 3 > /proc/sys/vm/drop_caches", $output, $return_val);
    error_log("drop caches output : " . implode("\n", $output) . " return: " . $return_val);
}

function build_file_path() {
  return join(DIRECTORY_SEPARATOR, func_get_args());
}

function get_absolute_path($path) {
  $path = str_replace(array('/', '\\'), DIRECTORY_SEPARATOR, $path);
  $parts = array_filter(explode(DIRECTORY_SEPARATOR, $path), 'strlen');
  $absolutes = array();
  foreach ($parts as $part) {
    if ('.' == $part) continue;
    if ('..' == $part) {
      array_pop($absolutes);
    } else {
      $absolutes[] = $part;
    }
  }
  $result = implode(DIRECTORY_SEPARATOR, $absolutes); 
  if (startsWith($path, DIRECTORY_SEPARATOR))
    return DIRECTORY_SEPARATOR . $result;
  else
    return $result;
}

function webrootPath() {
  $confFile = new DeviceConfigFile;

  if (!$confFile->load())
    return NULL;

  $path = $confFile->value("Webroot Path", NULL);
  if (is_null($path)) {
    if (file_exists("/var/www"))
      $path = "/var/www";
    else if (file_exists("/www"))
      $path = "/www";
  }
  return rtrim($path, "/");
}

function sdcardMountPath($confFile = NULL, $defaultVal = "/var/www/mountable_sdcard") {
  if (is_null($confFile)) {
    $confFile = new DeviceConfigFile;

    if (!$confFile->load())
      return $defaultVal;
  }

  $sdcardPath = $confFile->value("SDcard Path", $defaultVal);
  return $sdcardPath;
}

function sdcardBackupPath() {
  if (in_array(platformName(), array("FG", "FG+", "FG-")))
    return backupBasePath();

  $sdcardPath = sdcardMountPath();

  //TODO: how to integrate with configs in cpt_site_config file ? 
  //      maybe we should get rid of the cpt_site_config.php file ?
  $sdBackupBasePath = build_file_path($sdcardPath, 'cpt-backups');
  return $sdBackupBasePath;
}

function sdcardFtpPath() {
  if (in_array(platformName(), array("FG", "FG+", "FG-")))
    return build_file_path('', 'public', 'sdcard');

  return build_file_path('', 'mountable_sdcard');
}

function sdcardBackupFtpPath() {
  return build_file_path(sdcardFtpPath(), 'cpt-backups');
}

function backupBasePath() {
    $rootPath = $_SERVER['DOCUMENT_ROOT'];
    if (file_exists(build_file_path($rootPath, 'cpt_site_config.php')))
    {
      include_once(build_file_path($rootPath, 'cpt_site_config.php'));
      return CptSiteConfig::$backupBasePath;
    }
    else
      return build_file_path($rootPath, 'sdcard', 'cpt-backups');
}

function sedonaPath() {
    $confFile = new DeviceConfigFile;
    if ($confFile->load()) {
      $path = $confFile->value("SVM Path", NULL); 
      if (!is_null($path)) 
        return $path;
    }

    $rootPath = $_SERVER['DOCUMENT_ROOT'];
    if (file_exists(build_file_path($rootPath, 'cpt_site_config.php')))
    {
      include_once(build_file_path($rootPath, 'cpt_site_config.php'));
      return CptSiteConfig::$sedonaPath;
    }
    else
    {
      $sedonaPath = build_file_path('', 'mnt', 'sedona');
      if (!file_exists($sedonaPath))
        $sedonaPath = build_file_path('', 'mnt');

      return $sedonaPath;
    }
}

function imgPath() {
  return build_file_path(cptBaseDir(), "img");
}

function dashboardPath() {
  return build_file_path(cptBaseDir(), "dashboard");
}

function isDashboardExisted() {
  return file_exists(dashboardPath());
}

function pluginsPath() {
  return build_file_path(cptBaseDir(), "plugins");
}

function collectPlugins() {
  $plugins = array();
  $pluginNames = array();
  $pluginsPath = pluginsPath();
  if (file_exists($pluginsPath) && is_dir($pluginsPath)) {
    $pathIterIter = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($pluginsPath, FilesystemIterator::SKIP_DOTS), RecursiveIteratorIterator::CHILD_FIRST);
    $pathIterIter->setMaxDepth(0);
    foreach($pathIterIter as $path) {
      if (!$path->isDir())
        continue;

      $pluginId = NULL;
      $indexFile = NULL;
      $label = NULL;
      $enabled = true;
      $isSupported = true;
      $openNewPage = true;
      $pluginManifestFile = build_file_path($path->getPathName(), "manifest.json");
      if (file_exists($pluginManifestFile)) {
        $manifest = json_decode(file_get_contents($pluginManifestFile), true);
        if (is_array($manifest)) {
          $pluginId = isset($manifest['id']) ? $manifest['id'] : NULL;
          $indexFile = isset($manifest['indexFile']) ? $manifest['indexFile'] : NULL;
          $utilityFile = isset($manifest['utilityFile']) ? $manifest['utilityFile'] : NULL;
          $label = isset($manifest['label']) ? $manifest['label'] : NULL;
          $enabled = isset($manifest['enabled']) ? $manifest['enabled'] : true;
          $show = isset($manifest['show']) ? $manifest['show'] : true;
          $openNewPage = isset($manifest['openNewPage']) ? $manifest['openNewPage'] : true;
          if (isset($manifest['supportedPlatforms'])) {
            $platformName = platformName();
            $isSupported = in_array($platformName, $manifest['supportedPlatforms']);
          }
        }
      }

      if (!$enabled || !$isSupported)
        continue;

      if (empty($indexFile)) {
        if (file_exists(build_file_path($path->getPathName(), "index.php")))
          $indexFile = "index.php";
        else if (file_exists(build_file_path($path->getPathName(), "index.html"))) 
          $indexFile = "index.html";
        else if (file_exists(build_file_path($path->getPathName(), "index.htm")))
          $indexFile = "index.htm";
        else
          continue;
      }

      $baseName = $path->getBaseName();
      $indexFilePath = "plugins/" . $baseName . '/' . $indexFile;

      if (empty($pluginId))
        $pluginId = $baseName;

      if (!empty($utilityFile))
        $utilityFile = "plugins/" . $baseName . '/' . $utilityFile;

      $pluginName = empty($label) ? $baseName : $label;
      $plugins[] = array(
          "id" => $pluginId,
          "name" => $pluginName, 
          "openNewPage" => $openNewPage,
          "show" => $show,
          "indexFile" => $indexFilePath,
          "utilityFile" => $utilityFile
      );
      $pluginNames[] = $pluginName;
    }
  }
  array_multisort($pluginNames, SORT_ASC, $plugins);
  return $plugins;
}

function collectFeatures()
{
  // plugins 
  $platName = platformName();
  $plugins = collectPlugins();
  $features = array();
  foreach ($plugins as $plugin) {
    if ($platName == "FW" && $plugin['name'] == "Network Manager") {
      continue;
    }
    $features[$plugin['id']] = array('name' => $plugin['name']);
  }

  // platform related features 
  if ($platName == "FW") {
    foreach ($features as &$feature)
      $feature['parent'] = 'FWUtility';

    $features['FWUtility'] = array('name' => 'Utility');
    $features['FirmwareFlash'] = array('name' => 'Firmware Flash', 'parent' => 'FWUtility');

    $features['FWNetworkConfig'] = array('name' => 'Network Config', 'parent' => 'FWUtility');
    $features['FWAdvNetworkConfig'] = array('name' => 'Advanced Network Config', 'parent' => 'FWUtility');
  }
  return $features;
}

function rmIfExists($dirPath) {
    if (!is_dir($dirPath))
        return;

    // execCmd('rm -rf ' . $dirPath);
    foreach(new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dirPath, FilesystemIterator::SKIP_DOTS), RecursiveIteratorIterator::CHILD_FIRST) as $path) {
        $path->isFile() ? unlink($path->getPathname()) : rmdir($path->getPathname());
    }
    rmdir($dirPath);
}

function escapeJsonChars($val) {
  return addcslashes($val, "\\\"\n\r");
}

function clearCache() {
  // if opcache is enabled, reset all opcache 
  if (function_exists('opcache_reset'))
    opcache_reset();

  // purge dashboard asset manifest file
  $manifestFilePath = build_file_path(dashboardPath(), 'js_assets.json');
  if (file_exists($manifestFilePath))
    unlink($manifestFilePath);
}

function loadFWConfig($config_path) {
  $lines = explode("\n", file_get_contents($config_path));
  $config = array();
  foreach($lines as $line) {
    if (strlen($line) == 0)
      continue;

    $parts = explode(":", $line, 2);
    if (count($parts) != 2)
      continue;

    $config[$parts[0]] = $parts[1];
  }
  return $config;
}

function mergeFWConfig($fw_config_old, $fw_config_new) {
  $old_config = loadFWConfig($fw_config_old);
  $new_config = loadFWConfig($fw_config_new);
  //we need to keep ReleaseDate, SvmVersion, McuVersion old configs, so we
  //ignore these in new config
  unset($new_config['ReleaseDate']);
  unset($new_config['SvmVersion']);
  unset($new_config['McuVersion']);

  $result = array();
  $merged_config = array_merge($old_config, $new_config);
  foreach($merged_config as $key => $value) {
    array_push($result, $key . ":" . $value);
  }
  file_put_contents($fw_config_old, implode("\n", $result));
}

function doReboot($async=false) {
    sync();

    if ($async)
      //FW doesn't support sleep less than 1 second
      shell_exec('(sleep 1 && reboot >/dev/null 2>&1 ) &');
    else
      execCmd('reboot');
}

function recurseCopy($src, $dst) {
  if (is_file($src)) {
    $dst_dir = dirname($dst);
    if (!file_exists($dst_dir))
      mkdir($dst_dir, 0777, true);
    return copy($src, $dst);
  }

  if (!file_exists($dst))
    mkdir($dst, 0777, true);

  $dir = opendir($src);
  while(false !== ( $file = readdir($dir)) ) {
    if (( $file != '.' ) && ( $file != '..' )) {
      if ( is_dir(build_file_path($src, $file)) ) {
        if (!recurseCopy(build_file_path($src, $file), build_file_path($dst, $file))) {
          closedir($dir);
          return false;
        }
      }
      else {
        if (!copy(build_file_path($src, $file), build_file_path($dst, $file))) {
          closedir($dir);
          return false;
        }
      }
    }
  }
  closedir($dir);
  return true;
}

function execCmd($cmd) {
  exec(escapeshellcmd($cmd));
}

function execShellCmd($cmd) {
  //the input cmd is not escaped yet, it's the caller's responsibility to escape it
  $output = shell_exec("( $cmd ) && echo 'SUCCESS'");
  return !(is_null($output) || preg_match("/SUCCESS\s*$/", $output) != 1);
}

function cleanupGhostScripts() {
    $docRootPath = rtrim($_SERVER['DOCUMENT_ROOT'], DIRECTORY_SEPARATOR);
    $ghostScriptsPath = build_file_path($docRootPath, 'ghost_scripts');

    rmIfExists($ghostScriptsPath);
}

function controlCmd($service_name, $action, $monit) {
  $cmds = array();
  $pname = platformName();
  if ($action == 'enable') {
    if ($pname == 'FS')
      $cmds[] = 'for s in /etc/init.d/D??' . $service_name . ' ; do $s start && mv $s /etc/init.d/S${s#/etc/init.d/D}; done'; 
    else if ($pname == 'FW')
      $cmds[] = '/etc/init.d/' . $service_name . ' enable && /etc/init.d/' . $service_name . ' restart'; 

    if ($monit)
      $cmds[] = '[ -e /usr/bin/monit ] && /usr/bin/monit monitor ' . $service_name;
  } else if ($action == 'disable') {
    if ($monit)
      $cmds[] = '[ -e /usr/bin/monit ] && /usr/bin/monit unmonitor ' . $service_name;

    if ($pname == 'FS')
      $cmds[] = 'for s in /etc/init.d/S??' . $service_name . ' ; do $s stop && mv $s /etc/init.d/D${s#/etc/init.d/S}; done'; 
    else
      $cmds[] = '/etc/init.d/' . $service_name . ' disable && /etc/init.d/' . $service_name . ' stop'; 
  } else  {
    error_log("action '$action' is not supported");
  }

  return implode(';', $cmds);
}

function doServiceControl($service_name, $action, $monit) {
  $cmd = controlCmd($service_name, $action, $monit);
  if (!is_null($cmd)) {
    shell_exec($cmd);

    $pname = platformName();
    if ($pname == "FW" && ($service_name == "ntpd" || $service_name == "chronyd")
      && ($action == "enable" || $action == "disable")) {
      $config = getPlatformConfig();
      if (!empty($config)) {
        $config['NtpEnable'] = $action == "enable" ? 1 : 0;
        savePlatformConfig($config);

        if (file_exists("/mnt/sedona/writeConfig"))
          shell_exec("/mnt/sedona/writeConfig");
      }
    }
  }
}

function restartSvm() {
    $pfName = platformName();
    if ($pfName == "FI") {
      error_log("no svm running on FI platform");
      return;
    }
    else if ($pfName == "FW") {
      include_once "UDPClient.class.php";
      //restart svm by sedona action
      $client = new UDPClient("127.0.0.1", "1001");
      $client->setDebug(false);
      $client->invokeAction("/app/objects/.restart", "void", "");
    }
    else {
      //reboot command was added since FG+(from cheah)
      //for FG and FG-, use the old way, otherwise use reboot command
      if ($pfName == "FG" || $pfName == "FG-")
          execCmd("killall FGBase svm.exe");
      else
          doReboot(true);
          //execCmd("/sbin/reboot");
    }
}

// echo arrayMapToString(array('a' => 1, 'b' =>2, 'c' => 100), ",");

function firmwareUpgradePath() {
    $name = platformName();
    if ($name == "FI")
      return build_file_path('', 'media', 'boot', 'fi');
    else if ($name == "FS")
      return build_file_path('', 'media', 'boot', 'fs');
    else if ($name == "FW")
      return build_file_path('', 'tmp');
    else {
      $rootPath = $_SERVER['DOCUMENT_ROOT'];
      if (file_exists(build_file_path($rootPath, 'cpt_site_config.php')))
      {
        include_once(build_file_path($rootPath, 'cpt_site_config.php'));
        return CptSiteConfig::$firmwareUpgradePath;
      }
      else
      {
        if ($name == "FG+" || $name == "FG")
            return build_file_path('', 'sdcard', 'fg320');
        else 
        {
            error_log("firmware upgrade is not supported on this platform: " . $name);
            return null;
        }
      }
    }
}

function firmwareFileName() {
  // in general, fetch firmware file's name from field "FirmwareFileName" in 
  // "/mnt/sedona/easyio.config" file, if that field is missing, then use the 
  // following default value:
  // FS32: fs_rom.zz
  // FG32: fg320.tar.gz 
  // FG32+: fg320plus.md5, rootfs.yaffs2, u-boot.bin, zImage (not supported)
  // FW: fw140.tar.gz

  $fileName = NULL;
  $confFile = new DeviceConfigFile;
  if ($confFile->load()) {
    $fileName = $confFile->value("FirmwareFileName", NULL);
    if (!is_null($fileName))
      return $fileName;
  }

  $name = platformName();
  if ($name == "FI")
    return "fi_rom.zz";
  else if ($name == "FS")
    return "fs_rom.zz";
  else if ($name == "FG" || $name == "FG+")
    return "fg320.tar.gz";
  else if ($name == "FW")
    return "fw140.tar.gz";
  else
    return "";
}

function does_sdcard_exist() {
  $sdcardPath = sdcardMountPath();
  if (is_null($sdcardPath)) {
    $filePath = realpath("/sdcard/No SD Card");
    if (file_exists($filePath))
      return false;
    else
      return true;
  } else {
    return (file_exists($sdcardPath) && !file_exists(rtrim($sdcardPath, '/') . "/No SD Card"));
  }
}

function isPlatformNewerThan($dt_str) {
  if (is_string($dt_str))
    $dt = date_create_from_format("Y-m-d H:i:s", $dt_str);
  else {
    if ($dt_str instanceof DateTime)
      $dt = $dt_str;
    else
      throw new Exception("invalid datetime parameter");
  }
  
  $plat_dt = platformReleaseDate();
  return $plat_dt >= $dt;
}

function platformReleaseDate() {
  $plat_dt = NULL;
  $confFile = new DeviceConfigFile;
  if ($confFile->load()) {
    $plat_dt = $confFile->value("ReleaseDate", NULL);
    if (!is_null($plat_dt))
      $plat_dt = date_create_from_format("Y-m-d H:i:s", $plat_dt);
  }
  return $plat_dt;
}

function platformName() {
  //first check /mnt/sedona/easyio.conf file 
  $confFile = new DeviceConfigFile;
  if ($confFile->load()) {
    $name = $confFile->value("Model", NULL);
    if (!is_null($name))
      return strtoupper($name);
  }

  // if can not find easyio.conf, use legacy hacky way
  if (file_exists("/mnt/fw.config"))
    return "FW";

  $matches = array();
  $fg_linux_ver = '2.6.29.2';
  $fs_linux_ver = '3.4';
  $versionStr = file_get_contents("/proc/version"); 
  if (!preg_match('/linux version ([0-9.]+)/i', $versionStr, $matches))
    return "InvalidPlatform";
  else {
    if (count($matches) < 2 || !isset($matches[1]))
      return "InvalidPlatform";
    else {
      $result = strcmp($matches[1], $fg_linux_ver);
      if ($result > 0) {
        $result2 = strcmp($matches[1], $fs_linux_ver);
        if ($result2 >= 0)
          return "FS";
        else
          return "FG+"; # platform released after FG
      }
      else if ($result == 0)
        return "FG"; # FG platform
      else
        return "FG-"; # platform released before FG
    }
  }
}

# "svm --version" output has been improved to show platformId and firmwareVer
# in version after Oct 9, 2022.
function svmVerOutput() {
  $output = null;
  $cacheSVMOutputFilePath = "/tmp/svm-ver.cache";
  if (file_exists($cacheSVMOutputFilePath)) {
    $output = file_get_contents($cacheSVMOutputFilePath);
    return explode(PHP_EOL, $output);
  }

  $svmFilePath = "/mnt/sedona/svm";
  if (!file_exists($svmFilePath))
    $svmFilePath .= ".exe";
  if (!file_exists($svmFilePath)) {
    error_log("can not find svm executable");
    return $output;
  }

  $output = array();
  $return_val = 0;
  $cmd = $svmFilePath . " --ver";
  exec($cmd, $output, $return_val);
  file_put_contents($cacheSVMOutputFilePath, implode(PHP_EOL, $output));
  return $output;
}

function svmVerProp($name) {
  $output = svmVerOutput();
  if (empty($output) || !is_array($output)) return null;
  
  foreach($output as $line) {
    $parts = explode(':', $line, 2);
    if (count($parts) != 2) continue;
    if ($parts[0] == $name)
      return trim($parts[1]);
  }
  return null;
}

function platformId() {
  $result = svmVerProp("platId");
  if (!is_null($result)) return $result;
  return platformIdFromSedona();
}

function oemPrefix() {
  $platid = platformId();
  if (strlen($platid) <= 0) return '';
  $parts = explode("-", $platid);
  if (count($parts) < 2) return '';
  array_pop($parts);
  return implode('-', $parts);
}

function platformIdFromSedona() {
  $cachePlatformIdFilePath = "/tmp/platformid.cache";
  if (file_exists($cachePlatformIdFilePath))
    return file_get_contents($cachePlatformIdFilePath);

  include_once "UDPClient.class.php";
  $client = new UDPClient("127.0.0.1", "1001");
  $client->setDebug(false);
  $client->get("/app/objects/service/plat.platformId");
  $content = $client->getContent();
  $platid = "";
  if (preg_match('/"value":\s*"([^"]+)"/', $content, $matches) === 1 && count($matches) == 2) {
    $platid = $matches[1];
    file_put_contents($cachePlatformIdFilePath, $platid);
    return $platid;
  } else 
    return "";
}

function isOEM() {
  $platid = platformId();
  if (strlen($platid) <= 0) return false;
  $parts = explode("-", $platid);
  if (count($parts) < 2) return false;
  return $parts[0] == "slc";
}

function appName() {
  return isOEM() ? "SLC" : "CPT";
}

function firmwareVersion() {
  $result = svmVerProp("firmwareVer");
  if (!is_null($result)) return $result;
  return firmwareVersionFromSedona();
}

function firmwareVersionFromSedona() {
  $cacheVerFilePath = "/tmp/firmware_version.cache";
  if (file_exists($cacheVerFilePath))
    return file_get_contents($cacheVerFilePath);

  include_once "UDPClient.class.php";
  $client = new UDPClient("127.0.0.1", "1001");
  $client->setDebug(false);
  $client->get("/app/objects/service/plat.Version");
  $content = $client->getContent();
  $ver = "";
  if (preg_match('/"value":\s*"(V[^"]+)"/', $content, $matches) === 1 && count($matches) == 2) {
    $ver = $matches[1];
    file_put_contents($cacheVerFilePath, $ver);
    return $ver;
  } else 
    return "";
}

function isGoogleFirmware() {
  // this function based on the firmware version naming convention for google's
  // firmware: there is a suffix '_GS' 
  $ver = firmwareVersion();
  return endsWith($ver, "_GS");
}

function getHardwareVersion($firmwareVer) {
  if (preg_match('/^V(\d+\.\d+)/', $firmwareVer, $matches) === 1 && count($matches) == 2)
    return $matches[1];
  else
    return null;
}

function getBuildNumber($firmwareVer) {
  if (preg_match('/^V[^b]+b(\d+)/', $firmwareVer, $matches) === 1 && count($matches) == 2)
    return $matches[1];
  else
    return null;
}

function firmwarePatchLevel() {
  if (!file_exists("/mnt/sedona/patch_level"))
    return "";

  $level = file_get_contents("/mnt/sedona/patch_level");
  if (!$level)
    $level = "";
  return $level;
}

function isFirmwareGreaterThan($minFirmwareVer) {
  $curVer = firmwareVersion();
  $curHardwareVer = floatval(getHardwareVersion($curVer));
  $curBuildNum = floatval(getBuildNumber($curVer));

  $minHardwareVer = floatval(getHardwareVersion($minFirmwareVer));
  $minBuildNum = floatval(getBuildNumber($minFirmwareVer));

  return ($curHardwareVer > $minHardwareVer
    || ($curHardwareVer == $minHardwareVer && $curBuildNum > $minBuildNum));
}

function file_force_contents($filename, $data, $flags = 0){
  if(!is_dir(dirname($filename)))
      mkdir(dirname($filename).DIRECTORY_SEPARATOR, 0777, true);
  return file_put_contents($filename, $data, $flags);
}

function setupIndex($docRootPath, $cptBasePath, $graphicRoot=true) {
  if (empty($docRootPath))
    $docRootPath = rtrim($_SERVER['DOCUMENT_ROOT'], DIRECTORY_SEPARATOR);
  
  if (empty($cptBasePath))
    $cptBasePath = cptBaseDir();

  $indexPath = join(DIRECTORY_SEPARATOR, array($docRootPath, 'index.html'));
  $indexLegacyPath = join(DIRECTORY_SEPARATOR, array($docRootPath, 'index_legacy.html'));

  if ($graphicRoot) {
    $graphicIndexPath  = join(DIRECTORY_SEPARATOR, array($cptBasePath, 'index.html'));
    if (!file_exists($indexLegacyPath) && file_exists($indexPath))
      copy($indexPath, $indexLegacyPath);
    copy($graphicIndexPath, $indexPath);
  }
  else {
    if (file_exists($indexLegacyPath))
      copy($indexLegacyPath, $indexPath);
  }
}


function isAssoc($array) {
  if (!is_array($array))
      return false;

  $keys = array_keys($array);
  return array_keys($keys) !== $keys;
}

function list2json($listData) {
  $json = array();
  foreach($listData as $item) {
      if (is_string($item))
        $item = addcslashes($item,"\\\"\n\r");
      else if (is_array($item)) {
        if (isAssoc($item))
          $item = map2json($item);
        else
          $item = list2json($item);
        $json[] = $item;
        continue;
      }
      $json[] = '"' . $item . '"';
  }
  return '[' . implode(',', $json) . ']';
}

function map2json($mapData) {
  if (function_exists("json_encode"))
    return json_encode($mapData);

  $json = array();
  foreach($mapData as $k => $v) {
    if (is_string($k))
      $k = addcslashes($k,"\\\"\n\r");

    if (is_string($v))
      $v = addcslashes($v,"\\\"\n\r");
    else if (is_bool($v)) {
      $v = $v ? 'true' : 'false';
      $json[] = "\"$k\": $v";
      continue;
    }
    else if (is_array($v)) {
      if (isAssoc($v))
        $v = map2json($v);
      else
        $v = list2json($v);

      $json[] = "\"$k\": $v";
      continue;
    }
    $json[] = "\"$k\": \"$v\"";
  }
  return '{' . implode(',', $json) . '}';
}
  
function cptToolsPath() {
  $rootPath = $_SERVER['DOCUMENT_ROOT'];
  return build_file_path($rootPath, 'sdcard', 'CPT-Tools.zip');
}
function sptToolsPath() {
  $rootPath = $_SERVER['DOCUMENT_ROOT'];
  return build_file_path($rootPath, 'sdcard', 'SPT-Tools.zip');
}

function updatePhpAdminAuth($salt, $checksum) {
  $root_path = parentPath(3);
  $file_path = build_file_path($root_path, 'phpliteadmin.config.php');
  $cpt_auth_content = <<<EOS
<?php
  \$cpt_password_salt = "$salt";
  \$cpt_password_checksum = "$checksum";
?>
EOS;
  file_force_contents($file_path, $cpt_auth_content);
}

function readfile_chunked($filename, $retbytes=true) { 
    $chunksize = 1*(32*1024); // how many bytes per chunk 
    $buffer = ''; 
    $cnt =0; 
    $handle = fopen($filename, 'rb'); 
    if ($handle === false) { 
       return false; 
    } 
    while (!feof($handle)) { 
        $buffer = fread($handle, $chunksize); 
        echo $buffer; 
        ob_flush(); 
        flush(); 
        if ($retbytes) { 
            $cnt += strlen($buffer); 
        } 
    } 
    $status = fclose($handle); 
    if ($retbytes && $status) { 
        return $cnt; // return num. bytes delivered like readfile() does. 
    } 
   return $status; 
} 

function getDeviceData() {
  $data = array();

  //populate device config data
  $confFile = new DeviceConfigFile;
  if ($confFile->load()) {
    foreach($confFile->keys() as $key) {
      $value = $confFile->value($key, NULL);
      if (is_null($value))
        continue;
      $key = '_' . str_replace(" ", "_", trim(strtolower($key)));
      $data[$key] = $value;
    }
  }
  
  //add calculated device data 
  $data["platform_name"] = platformName();
  $data["sdcard_exists"] = does_sdcard_exist();
  if ($data["sdcard_exists"]) {
    $sdcardPath = sdcardMountPath();
    if (is_null($sdcardPath)) // then is the case of FG or FG+
      $sdcardPath = "/sdcard"; //sdcard is mounted at /sdcard for FG or FG+ 
    $data["sdcard_free_space"] = disk_free_space($sdcardPath);
    $data["sdcard_backup_ftp_path"] = sdcardBackupFtpPath();
    $data["sdcard_ftp_path"] = sdcardFtpPath();
  }
  $data["flash_free_space"] = disk_free_space("/");

  return $data;
}

function getPlatformConfig() {
  $config = array();
  //first, read easyio.config file into $config hash
  $conf_file_path = "/mnt/sedona/easyio.config";
  $conf_lock_path = "/mnt/sedona/easyio.config.lock";
  if (!file_exists($conf_file_path)) {
    error_log("can not find " . $conf_file_path);
    return $config;
  }

  $fp = fopen($conf_lock_path, "r+");
  if (flock($fp, LOCK_SH)) {  // acquire a shared lock
    $contents = file_get_contents($conf_file_path);
    flock($fp, LOCK_UN);    // release the lock
    fclose($fp);
  } else {
    error_log("Get: Couldn't get the lock " . $conf_lock_path);
    fclose($fp);
    return $config;
  }

  foreach(explode(PHP_EOL, $contents) as $line) {
    if (empty($line))
      continue;

    $parts = explode(":", $line, 2);
    if (count($parts) != 2) {
      error_log("invalid config line: " . $line);
      continue;
    }
    $config[$parts[0]] = $parts[1]; 
  }

  if (empty($config))
    error_log("can not read content from config file: " . $conf_file_path);
  return $config;
}

function savePlatformConfig($config) {
  $lines = array();
  foreach($config as $key => $value)
    $lines[] = $key . ':' . $value;
  $conf_file_path = "/mnt/sedona/easyio.config";
  $conf_lock_path = "/mnt/sedona/easyio.config.lock";

  $fp = fopen($conf_lock_path, "w+");
  if (flock($fp, LOCK_EX)) {  // acquire an exclusive lock
    file_put_contents($conf_file_path, implode(PHP_EOL, $lines));
    flock($fp, LOCK_UN);    // release the lock
  } else {
    error_log("Save: Couldn't get the lock " . $conf_lock_path);
  }
  fclose($fp);
}

$netmaskList = array(
  "0.0.0.0", "128.0.0.0", "192.0.0.0", "224.0.0.0",
  "240.0.0.0", "248.0.0.0", "252.0.0.0", "254.0.0.0",
  "255.0.0.0", "255.128.0.0", "255.192.0.0", "255.224.0.0",
  "255.240.0.0", "255.248.0.0", "255.252.0.0", "255.254.0.0",
  "255.255.0.0", "255.255.128.0", "255.255.192.0", "255.255.224.0",
  "255.255.240.0", "255.255.248.0", "255.255.252.0", "255.255.254.0",
  "255.255.255.0", "255.255.255.128", "255.255.255.192", "255.255.255.224",
  "255.255.255.240", "255.255.255.248", "255.255.255.252", "255.255.255.254",
);
function cdir2MaskIP($cdir) {
  global $netmaskList;
  $index = intval($cdir);
  if ($index >= 0 && $index < count($netmaskList))
    return $netmaskList[$index];
  else
    return null;
}
function maskIP2CDIR($netmask) {
  global $netmaskList;
  return array_search($netmask, $netmaskList);
}

function render_template($path, array $args){
  ob_start();
  include($path);
  $var=ob_get_contents(); 
  ob_end_clean();
  return $var;
}

function clientIP() {
  $ip = $_SERVER['REMOTE_ADDR'];
  if (isset($_SERVER['HTTP_CLIENT_IP']) && preg_match('/^([0-9]{1,3}\.){3}[0-9]{1,3}$/', $_SERVER['HTTP_CLIENT_IP'])) {
    $ip = $_SERVER['HTTP_CLIENT_IP'];
  }
  $parts = explode(':', $ip);
  $ip = end($parts);
  $ip = filter_var($ip, FILTER_VALIDATE_IP);
  if ($ip === false) {
    return '';
  }
  return $ip;
}

function isAppCompatible($newAppPath, $curAppPath) {
  if (!file_exists($newAppPath) || !file_exists($curAppPath))
    return true;
  
  $newKits = readKitListFromApp($newAppPath); 
  $curKits = readKitListFromApp($curAppPath);

  if(includeEIOKit(array_keys($curKits)) && includeEIOKit(array_keys($newKits))) {
    return true;
  }

  if(includeFXEKit(array_keys($curKits)) && includeFXEKit(array_keys($newKits))) {
    return true;
  }

  if(includeOEMKit(array_keys($curKits)) && includeOEMKit(array_keys($newKits))) {
    return true;
  }

  return false;
}

function isScodeCompatible($kitsPathInBackup) {
  if (doesSvmSupport4Bix()) {
    //4Bix svm is backward compatible, no need to continue the check 
    return true;
  }

  return !isScode4Bix($kitsPathInBackup);
}

function doesSvmSupport4Bix() {
  $config = getPlatformConfig();
  return (array_key_exists("BixSize", $config) && $config["BixSize"] == "4");
}

function isScode4Bix($scodeFilePath) {
  $vers = scodeVersion($scodeFilePath);
  if (count($vers) != 2)
    return false;
  
  return $vers[0] >= 2;
}

function scodeVersion($scodeFilePath) {
  $result = array();
  $handle = fopen($scodeFilePath, "rb");
  if (!$handle)
      return $result;

  $magic = bin2hex(fread($handle, 4));
  if (strncasecmp($magic, "5ED0BA07", 4) != 0 && strncasecmp($magic, "07BAD05E", 4) != 0) {
    //invalid magic
    fclose($handle);
    return $result;
  }

  $major = ord(fread($handle, 1));
  $minor = ord(fread($handle, 1));
  $result[] = $major;
  $result[] = $minor;

  fclose($handle);
  return $result;
}

function includeEIOKit($kitNames) {
  foreach($kitNames as $name) {
    if (startsWith($name, "easyio"))
      return true;
  }
  return false;
}

function includeFXEKit($kitNames) {
  foreach($kitNames as $name) {
    if (startsWith($name, "cse") || startsWith($name, "cwe"))
      return true;
  }
  return false;
}

function includeOEMKit($kitNames) {
  foreach($kitNames as $name) {
    if (startsWith($name, "core") || startsWith($name, "slc"))
      return true;
  }
  return false;
}

function readKitListFromApp($appPath) {
  $result = array();
  $handle = fopen($appPath, "rb");
  if (!$handle) 
    return $result;

  $magic = fread($handle, 4); 
  if (strncmp($magic, "sapp", 4) != 0 || feof($handle)) {
    fclose($handle);
    return $result;
  }

  $version = fread($handle, 4);
  if (strncmp(bin2hex($version), "00000003", 4) != 0 || feof($handle)) {
    fclose($handle);
    return $result;
  }

  $kitNum = ord(fread($handle, 1));
  for($i=0; $i<$kitNum; ++$i) {
    $name = "";
    while(!feof($handle)) {
      $char = fgetc($handle);
      if ($char == "\0")
        break;
      else
        $name = $name . $char;
    }
    $checksum = fread($handle, 4);
    $result[$name] = bin2hex($checksum);
  }

  fclose($handle);
  return $result;
}

function utcnow($format="Y-m-d H:i:s") {
  return date_create("now", new DateTimeZone("UTC"))->format($format);
}

function networkInterface() {
  $pfName = platformName();
  if ($pfName == "FW")
    return "br-lan";
  else
    return "eth0";
}

// Get the highest priority network interface
function getPrimaryNetworkInterface() {
  $interface = rtrim(`ip route | grep default | awk '{print $5}' | head -n 1`);
  if (strlen($interface) > 0) {
    return $interface;
  }
  return networkInterface();
}

function deviceIP() {
  $ip = null;
  $interface = getPrimaryNetworkInterface();
  $ip = rtrim(`ifconfig $interface | sed -n '/inet addr:/ s/^\s*inet addr:\([.0-9]\+\).*\$/\\1/p'`);
  // $pfName = platformName();
  // if ($pfName == "FS" || $pfName == "FI")
  //   $payload['ip'] = `nmcli -t -f ipv4.addresses connection show wired-eth0 | sed -n 's/^[^:]\+://; s#/[0-9]\+##; p'`;
  // else
  //   $payload['ip'] = `ifconfig eth0 | sed -n '/inet addr:/ s/\s*inet addr:\([.0-9]\+\).*/\1/p'`;
  return $ip;
}

function macAddr() {
  $interface = networkInterface();
  $mac = "";
  $macCacheFile = "/tmp/device.mac";
  if (file_exists($macCacheFile))
    $mac = file_get_contents($macCacheFile);
  else {
    $mac = `ip link show $interface | awk '/link\/ether/ { printf $2 }'`;
    file_put_contents($macCacheFile, $mac);
  }
  return $mac;
}

function isWifiSupported() {
	$output = shell_exec('find /lib/modules/$(uname -r)/ -type f -iname "*.ko"');
	if (!$output) return false;

  return preg_match('/\b88XXau.ko\b/i', $output) == 1;
}

function mqttVersion() {
  $mqttPath = "/usr/local/bin/mqtt-service";
  if(!file_exists($mqttPath)){
    return null;
  } 
  $output = shell_exec($mqttPath . " --version");  
  if (is_null($output))
    return null;
  $output = rtrim($output);
  $parts = explode(" ", $output);
  if (count($parts) <= 1)
    return null;

  return array_pop($parts);
}

function componentVersion($componentPath) {
  if (!$componentPath || !file_exists($componentPath)) {
    return null;
  } 
  $cmd = $componentPath . " -v 2>&1";
  $output = shell_exec($cmd);
  if ($output && strpos($output,":") > 0 ) {
    $ver = strtolower(trim(explode(":", $output)[1]));
    if($ver && strpos($ver,"-v") > 0){
      $ver = trim(explode("-v", $ver)[1]);
    }
    return $ver;
  } else if($output && strpos($output,"-v") > 0) {
    $ver = trim(explode("-v", $output)[1]);
    return $ver;
  } else if($output && strpos($output," ") > 0) {
    $ver = trim(explode(" ", $output)[1]);
    return $ver;
  }
  return $output;
}

function sedonaComponentVersion() {
  $pfName = platformName();
  $components = [];
  $ver = firmwareVersion();
  $curBuildNum = getBuildNumber($ver);
  if ($pfName == "FW") {
    if ($curBuildNum >= 23) {
      $components = ["fwbacnet", "fwmbasync", "fwmbasyncslave", "fwmbtcp", "fwmbtcpslave", "fwp2p"];
    }
  } else if ($pfName == "FS") {
    if ($curBuildNum >= 61) {
      $components = ["BacnetApp", "MbAsync", "MbAsyncSlave", "MbTCP", "MbTCPGw", "MbTCPSlave", "P2P"];
    }
  }
  $result = [];
  foreach($components as $component) {
    $val = componentVersion('/mnt/sedona/'. $component);
    if($val){
      $result[$component] = $val;
    }
  }
  return $result;
}

function webAppVersion() {
  return WEB_APP_VERSION;
}
?>
