<?php
//vim: ts=2 sw=2

include_once "jwt_key_store.php";
include_once "db.php";

  class Migration {

    public static function instance() {
      static $inst = NULL;
      if ($inst === NULL) {
        $inst = new Migration();
      }
      return $inst;
    }

    private function __construct() {
    }

    public function run() {
      $migrationApplied = false;

      // apply main db migrations
      $db = DBConn::instance()->db();
      $migrationApplied = $this->applyMigrations($db) || $migrationApplied;

      $this->applyDataFixes($db);

      // apply history db migrations
      $hdb = DBConn::instance()->history_db();
      $migrationApplied = $this->applyMigrations($hdb, "history") || $migrationApplied;

      // apply live db migrations
      $ldb = DBConn::instance()->live_db();
      $migrationApplied = $this->applyMigrations($ldb, "live") || $migrationApplied;

      // create data sync triggers 
      if ($migrationApplied) {
        $model = new DeltaChange();
        $model->setupTriggers();
      }

      $this->applyPatch();
    }

    protected function getMigrationFiles($user_version, $subfolder) {
      $result = array();
      $glob_pattern = is_null($subfolder) ? build_file_path(cptBaseDir(), 'app', 'migrations', '*.php') : build_file_path(cptBaseDir(), 'app', 'migrations', $subfolder, '*.php');
      $migrations = glob($glob_pattern);
      if (!is_array($migrations))
        return $result;

      foreach( $migrations as $mig ) {
        $parts = array();
        if (preg_match('/\/([0-9]+)_[^\/]*.php$/', $mig, $parts)) {
          $order = intval($parts[1]);
          if ($order < $user_version)
            continue;
          $result[$order] = $mig;
        }
      }
      return $result;
    }

    protected function getUserVersion($db) {
      $rs = $db->query("PRAGMA user_version");
      if (!$rs) {
        error_log("can not get user_version");
        return null;
      }

      $user_version = intval($rs->fetchColumn());
      return $user_version;
    }

    protected function applyMigrations($db, $subfolder=null) {
      $result = false;
      $user_version = $this->getUserVersion($db);
      if (is_null($user_version)) return $result;

      $migrations = $this->getMigrationFiles($user_version, $subfolder);
      if (empty($migrations)) return $result;

      $startTime = microtime(true);
      $fp = fopen("/tmp/cpt-migration.lock", "w+");
      if (flock($fp, LOCK_EX)) {
        try {
          $user_version = $this->getUserVersion($db);
          $result = $this->doApplyMigrations($db, $user_version, $subfolder);
        } catch (Throwable $e) { // php 7+
          error_log("failed to run migrations: " . $e->getMessage());
          error_log($e->getTraceAsString());
        } catch (Exception $e) { // php 5.x
          error_log("failed to run migrations: " . $e->getMessage());
          error_log($e->getTraceAsString());
        } finally {
          flock($fp, LOCK_UN);
        }
      } else {
        error_log("failed to get file lock");
      }
      fclose($fp);
      error_log(" * migration time cost: " . (microtime(true)-$startTime));
      return $result;
    }

    protected function doApplyMigrations($db, $user_version, $subfolder) {
      $migrations = $this->getMigrationFiles($user_version, $subfolder);
      if (empty($migrations))
        return false;

      ksort($migrations, SORT_NUMERIC);

      error_log('current user_version: ' . $user_version);
      error_log('find ' . count($migrations) . ' new db migrations, start to apply ...');
      $db->beginTransaction();
      //TODO: catch exceptions in migration file and rollback the transaction
      foreach($migrations as $order => $path) {
        error_log('apply db migration: ' . $path . ' ...');
        include($path);
        $user_version = $order;
      }
      $user_version += 1;
      error_log('update $user_version to ' . $user_version);
      $db->query('PRAGMA user_version = ' . $user_version);
      $db->commit();
      error_log('db migration done.');
      return true;
    }

    protected function applyDataFixes($db) {
      if (!$db)
        return;

      $flag_file = build_file_path(cptBaseDir(), 'RESET_ADMIN_PASSWORD');
      if (!file_exists($flag_file))
        return;
      
      error_log("start to reset password of 'admin' ...");
      $db->beginTransaction();
      $db->exec(<<<EOD
        UPDATE users SET salt = 'hNsq25IWKmRfSCOu', checksum = 'dc7b9f203aa5cf1bca33d5fc126cd783f98590e9' WHERE name = 'admin';
EOD
      );
      unlink($flag_file);
      $db->commit();
      error_log("password reset is done");
    }

    //NOTE: no need to purge the config data before backup, now FI related
    //config has been put in a persistent folder that can survive even across
    //firmare flashing, check fi_config_file.php for more details
    // protected function purgeOldBackupData() {
    //   $macAddrPath = build_file_path(__DIR__, "device.mac");
    //   if (!file_exists($macAddrPath))
    //     return;

    //   try {
    //     $backupMacAddr = file_get_contents($macAddrPath);
    //     unlink($macAddrPath);
    //     $curMacAddr = macAddr();  

    //     if ($backupMacAddr != $curMacAddr) { 
    //       // the backup used for restore is made on a difference device, need
    //       // to purge the device specific data before continue
    //       $confStore = new ConfigStore();
    //       $confStore->deleteConfig("msg_queue_addr");
    //       $confStore->deleteConfig("device_id");
    //       $confStore->deleteConfig("checksum_key");
    //     }
    //   } catch (Throwable $t) { // php 7.1+
    //     error_log("failed to purge old backup data: " . $t->getMessage());
    //     error_log($t->getTraceAsString());
    //   } catch (Exception $e) {
    //     error_log("failed to purge old backup data: " . $e->getMessage());
    //     error_log($e->getTraceAsString());
    //   }
    // }
    
    public function gen_self_key_pair() {
      $pname = platformName();
      if ($pname == "FI") return ;

      $store = new JWTKeyStore($pname);
      $res = $store->gen_self_key_pair();
      if ($res) return;
      error_log("failed to generate self key pair");
    }

    protected function applyPatch() {
      $patchFilePath = build_file_path(cptBaseDir(), 'app', 'HOT_PATCH.php');
      if (!file_exists($patchFilePath)) return;

      try {
        include_once($patchFilePath);
        unlink($patchFilePath);
        error_log("patch applied");
      } catch (Exception $e) {
        error_log("failed to run patch: " . $e->getMessage());
        error_log($e->getTraceAsString());
      }
    }
  }

?>
