<?php
//vim: ts=2 sw=2

set_include_path(get_include_path() . PATH_SEPARATOR . '..');

include_once "utils.php";

class DeviceDataCollector {
  function __construct() {
    $this->service_mapping = array(
      'sedona' => array(
        'label' => 'Sedona',
        'bin' => '/mnt/sedona/svm',
        'ver_cmd' => '--ver',
        'manifests' => array(
          'version' => '/^Sedona VM\s+(?P<version>.+)$/',
          'buildDate' => array('pat' => '/^buildDate:\s+(?P<buildDate>.+)$/'),
          'endian' => '/^endian:\s+(?P<endian>\w+)$/',
          'blockSize' => '/^blockSize:\s+(?P<blockSize>\d+)$/',
          'refSize' => '/^refSize:\s+(?P<refSize>\d+)$/'
        ),
      ),
      'mqtt' => array(
        'label' => 'MQTT Service',
        'bin' => '/usr/local/bin/mqtt-service',
        'ver_cmd' => '--ver',
        'manifests' => array(
          'version' => '/^mqtt-service\s+(?P<version>.+)$/',
        ),
      ),
    );

    $pname = platformName();
    if ($pname == 'FW') {
      $this->service_mapping['sedona']['bin'] = "/mnt/sedona/svm";
    }
  }

  protected function run_cmd($cmd) {
    $output = shell_exec($cmd);
    if (is_null($output)) {
      error_log("failed to run command: $cmd");
      return null;
    }
    $lines = explode("\n", $output);
    if (count($lines) <= 0)
      return null;
    return $lines;
  }

  protected function service_status($cmd) {
    $parts = explode(" ", $cmd);
    if (count($parts) < 1)
      return 'invalid';

    $bin_path = $parts[0];
    if (!file_exists($bin_path))
      return 'not_available';

    $name = basename($bin_path);
    $output = shell_exec("pidof $name");
    return strlen($output) > 0 ? 'up' : 'down';
  }

  protected function collect_service_data($service_id, $cmd, $manifest_defs) {
    $data = array();
    $lines = $this->run_cmd($cmd);
    if (is_null($lines))
      return $data;

    $data['status'] = $this->service_status($cmd);

    foreach($manifest_defs as $name => $manifest) {
      $pat = null;
      if (is_string($manifest)) {
        $pat = $manifest;
      } else if (is_array($manifest)) {
        $pat = $manifest['pat'];
      } 
      if (is_null($pat)) {
        error_log("can not find 'pat' for $service_id.$name");
        continue;
      }

      foreach($lines as $line) {
        preg_match($pat, $line, $matches);
        if (!isset($matches[$name]))
          continue;

        $data[$name] = $matches[$name];
      }
    }

    return $data;
  }

  protected function collect_firmware_data() {
    $data = array();
    $data['version'] = firmwareVersion();

    $config = getPlatformConfig();
    $keys = array('ReleaseDate', 'Model');
    foreach($keys as $key) {
      if (array_key_exists($key, $config))
        $data[$key] = $config[$key];
    }
    return $data;
  }

  protected function collect_os_data() {
    $data = array();
    $output = shell_exec("uname -a");
    $parts = explode(" ", $output);
    $len = count($parts);
    if ($len >= 1) $data['name'] = $parts[0]; 
    if ($len >= 2) $data['host'] = $parts[1]; 
    if ($len >= 3) $data['version'] = $parts[2]; 
    return $data;
  }

  protected function collect_services_data() {
    $result = array();

    foreach($this->service_mapping as $id => $config) {
      if (!isset($config['bin']) || !isset($config['label']) || !isset($config['ver_cmd']))
        continue;

      if (!file_exists($config['bin']))
        continue;

      $cmd = $config['bin'] . " " . $config['ver_cmd'];
      $data = array_merge(array('label' => $config['label']), 
        $this->collect_service_data($id, $cmd, $config['manifests']));
      
      $result[$id] = $data;
    }

    return $result;
  }
   
  protected function collect_meta_data() {
    $data = array();
    $data['last_updated'] = date_create("now", new DateTimeZone("UTC"))->format("c");
    return $data;
  }

  public function run() {
    $result = array();

    //collect basic manifest data
    $result['os'] = $this->collect_os_data();
    $result['firmware'] = $this->collect_firmware_data();

    //collect services manifest data
    $result['services'] = $this->collect_services_data();

    $result['metadata'] = $this->collect_meta_data();
    return $result;
  }

};

?>
