<?php
//vim: ts=2 sw=2

include_once "db.php";
include_once "base_controller.php";

class LiveSyncController extends BaseController {
  protected $MAX_RECORD_NUM_PER_REQ = 200;

  protected function signinRequired() {
    return true;
  }

  protected function adminRequired() {
    return true;
  }

  protected function doAjaxGet() {
    $response = array();

    $action = null;
    if (isset($_GET['action']))
      $action = $_GET['action'];
    else
      $this->renderAjaxError($response, sprintf(L("'%s' parameter is missing"), 'action'));

    if ($action == "fetch_all") {
      $this->fetchAll();
    } else if ($action == "fetch_delta") {
      $this->fetchDelta();
    } else if ($action == "stats") {
      $this->collectStats();
    } else {
      error_log("invalid action: " . $action);
      $this->renderAjaxError($response, sprintf(L("%s is not supported"), $action));
      return;
    }
  }

  public function doAjaxPost() {
    $action = null;
    if (isset($_POST['action']))
      $action = $_POST['action']; 
    if ($action == "saveCheckpoint") {
      $this->saveCheckpoint();
    } else {
      $this->renderAjaxSuccess2();
    }
  }

  protected function parseCount() {
    $count = $this->MAX_RECORD_NUM_PER_REQ;
    if (isset($_GET['count']))
      $count = min(intval($_GET['count']), $this->MAX_RECORD_NUM_PER_REQ);
    return $count;
  }

  protected function parseTableOrDie() {
    if (isset($_GET['table']))
      return $_GET['table'];
    else
      $this->renderAjaxError($response, sprintf(L("'%s' parameter is missing"), 'table'));
  }

  protected function getCheckpointVersion($checkpoint) {
    $parts = explode("_", $checkpoint, 2);
    if (count($parts) != 2) return null;

    $fiVersion = intval($parts[0]);
    if ($fiVersion <= 0) return null;
    return $fiVersion;
  }

  protected function getCheckpointSessionId($checkpoint) {
    $parts = explode("_", $checkpoint, 2);
    if (count($parts) != 2) return null;

    $fiSessionId = $parts[1];
    return $fiSessionId;
  }

  protected function validateCheckpointOrDie() {
    if (!isset($_REQUEST['checkpoint'])) 
      return ;

    $fiVersion = $this->getCheckpointVersion($_REQUEST['checkpoint']);
    $fiSessionId = $this->getCheckpointSessionId($_REQUEST['checkpoint']);
    if (is_null($fiVersion) || is_null($fiSessionId))
      $this->renderAjaxError2(L("checkpoint format or value is invalid"));

    $model = new ConfigStore;
    $checkpoint = $model->value("checkpoint", "");
  
    $deviceVersion = $this->getCheckpointVersion($checkpoint);
    $deviceSessionId = $this->getCheckpointSessionId($checkpoint);
    if (is_null($deviceVersion) || is_null($deviceSessionId))
      $this->renderAjaxError2("incompatible data sync detected:{$checkpoint}", 300);

    if ($deviceVersion >= $fiVersion && $deviceSessionId == $fiSessionId) return ;

    // NOTE: the error text here is important, FI may use it to detect the
    // incompatible data sync issue, so don't modify it 
    $this->renderAjaxError2("incompatible data sync detected:{$checkpoint}", 300);
  }

  //fetchAll can be use to fetch all rows of a table, this should be done under following cases: 
  // * the first time full sync 
  // * periodically(daily?) full sync afterwards to prevent bugs in delta sync
  //
  // curl test cli: 
  //  $ curl -s -H "X-Requested-With: XMLHTTPRequest" "http://192.168.1.128/sdcard/cpt/app/sync_controller.php?action=fetch_all&from_cid=2&table=alarms&count=2" | jq .
  protected function fetchAll() {
    $this->validateCheckpointOrDie();

    $response = array();
    $table = $this->parseTableOrDie();

    $from_id = 0;
    if (isset($_GET['from_id']))
      $from_id = intval($_GET['from_id']);
    else
      $this->renderAjaxError($response, sprintf(L("'%s' parameter is missing"), 'from_id'));

    $count = $this->parseCount();

    //TODO: in future we can support white/black column list in request
    
    $model = new DeltaChange($table);
    $rs = $model->fetchAll($from_id, $count);
    $response['data'] = $rs;
    $this->renderAjaxSuccess2($response);
  }

  protected function fetchDelta() {
    $this->validateCheckpointOrDie();

    $response = array();
    $table = $this->parseTableOrDie();

    if (isset($_GET['from_date']))
      $this->renderAjaxError($response, "data sync API has been updated and your client is incompatible.");

    $from_cid = -1;
    if (isset($_GET['from_cid'])) {
      $from_cid = intval($_GET['from_cid']);
      if ($from_cid < 0 || ($from_cid == 0 && $_GET['from_cid'] != "0"))
        $this->renderAjaxError($response, sprintf(L("'%s' parameter is invalid"), $_GET['from_cid']));
    }
    else
      $this->renderAjaxError($response, sprintf(L("'%s' parameter is missing"), 'from_cid'));

    $count = $this->parseCount();

    $model = new DeltaChange($table);
    $data = array();
    $end_cid = $model->fetchEndChangeLogId($from_cid, $count);
    if (!is_null($end_cid)) {
      $data['added'] = $model->fetchAdded($from_cid, $end_cid);
      // $data['updated'] = $model->fetchDelta('U', $from_cid, $end_cid);
      $data['deleted'] = $model->fetchDeleted($from_cid, $end_cid);
      $data['sync_cid'] = intval($end_cid);
    } else 
      $data['sync_cid'] = intval($from_cid);
    
    $data['sync_count'] = $count;
    $data['sync_time'] = time();
    $data['max_record_num'] = $this->MAX_RECORD_NUM_PER_REQ;
    // error_log("### " . json_encode($data, JSON_PRETTY_PRINT));
    $response['data'] = $data;

    $this->renderAjaxSuccess2($response);
  }

  protected function collectStats() {
    $this->validateCheckpointOrDie();

    $response = array();
    $table = $this->parseTableOrDie();
    $count = $this->parseCount();
    $model = new DeltaChange($table);
    $rs = $model->stats($count);
    if ($rs === FALSE)
      $this->renderAjaxError($response, sprintf(L("failed to collect stats for table '%s'"), $table));
    else {
      $response['data'] = $rs;
      $this->renderAjaxSuccess2($response);
    }
  }

  protected function saveCheckpoint() {
    $response = array();
    if (!isset($_REQUEST['checkpoint'])) {
      $this->renderAjaxError($response, L("'checkpoint' param is missing"));
      return;
    }
    $checkpoint = $_REQUEST['checkpoint'];

    if (!isset($_REQUEST['last_checkpoint'])) {
      $this->renderAjaxError($response, L("'last_checkpoint' param is missing"));
      return;
    }
    $lastCheckpoint = $_REQUEST['last_checkpoint'];

    $model = new ConfigStore;
    $oldCheckpoint = $model->value("checkpoint", "");
    if ($oldCheckpoint != $lastCheckpoint && !startsWith($checkpoint, "1_")) {
      $fiVersion = $this->getCheckpointVersion($lastCheckpoint);
      $fiSessionId = $this->getCheckpointSessionId($lastCheckpoint);
      $deviceVersion = $this->getCheckpointVersion($oldCheckpoint);
      $deviceSessionId = $this->getCheckpointSessionId($oldCheckpoint);
      if (is_null($fiVersion) || is_null($deviceVersion) 
        || $fiVersion > $deviceVersion
        || $fiSessionId != $deviceSessionId) {
        $this->renderAjaxError2("incompatible data sync detected:{$oldCheckpoint}", 300);
        return;
      } else {
        // if FI is restored with an old backup, then FI's checkpoint might be
        // behind device's checkpoint. under this case, device side will ignore
        // this saveCheckPoint request, that means the device's checkpoint,
        // especially the version will never be decreased
        $this->renderAjaxSuccess2();
        return;
      }
    }

    $model->upsert("checkpoint", $checkpoint);
    $this->renderAjaxSuccess2();
  }

}

$controller = new LiveSyncController();
$controller->run();

?>
