<?php
// vim: ts=2 sw=2

include_once "utils.php";

class DBConn {
  protected $db = NULL;
  protected $history_db = NULL;
  protected $live_db = NULL;

  protected $dbpath = NULL;

  // using 'var' to define a property equals define the property using 'public'
  var $history_dbname = 'easyio.db';

  private static $instance = NULL;

  public static function instance() {
    if (self::$instance === NULL) {
      self::$instance = new self();
    }
    return self::$instance;
  }

  public static function reset() {
    self::$instance = NULL;
  }

  private function __construct() {
    $db_file_created = true;
    try {
      $dbname = 'cpt-web.db';
      $this->dbpath = build_file_path(cptBaseDir(), 'app', $dbname);

      if (!file_exists($this->dbpath))
        $db_file_created = false;

      $this->db = new PDO("sqlite:{$this->dbpath}");

    } catch (PDOException $e) {
      print "Error: " . $e->getMessage() . "<br/>";
      die();
    }
    if (file_exists($this->dbpath) && !$db_file_created)
      chmod($this->dbpath, 0666);
  }

  public function __destruct() {
    $this->db = NULL;
    $this->history_db = NULL;
    $this->live_db = NULL;
  }

  public function db() {
    return $this->db;
  }

  public function history_db() {
    if (!isset($this->history_db)) {
      try {
        $history_dbpath = $_SERVER['DOCUMENT_ROOT'] . '/sdcard/' . $this->history_dbname;
        $this->history_db = new PDO("sqlite:{$history_dbpath}");
      } catch (PDOException $e) {
        $message = "open history db faild";
        print "Errpr: " . $message . "<br/>";
        //print "Error: " . $e->getMessage() . "<br/>";
        die();
      }
    }
    return $this->history_db;
  }

  public function live_db() {
    if (!isset($this->live_db)) {
      try {
        $live_dbpath = DBConn::liveDBFilePath();
        $this->live_db = new PDO("sqlite:{$live_dbpath}");
      } catch (PDOException $e) {
        print "Error: " . $e->getMessage() . "<br/>";
        die();
      }
    }
    return $this->live_db;
  }

  public function getColumnNames($dbh, $table) {
    $col_names = array() ;

    $stmt = $dbh->prepare("SELECT sql FROM sqlite_master WHERE tbl_name = ':table'") ;
    $stmt->bindValue(':table', $table);
    $stmt->execute() ;
    $row = $stmt->fetch() ;

    $sql = $row[0] ;
    $r = preg_match("/\(\s*(\S+)[^,)]*/", $sql, $m, PREG_OFFSET_CAPTURE) ;
    while ($r) {
      array_push( $col_names, $m[1][0] ) ;
      $r = preg_match("/,\s*(\S+)[^,)]*/", $sql, $m, PREG_OFFSET_CAPTURE, $m[0][1] + strlen($m[0][0]) ) ;
    }
    return $col_names;
  }

  public static function tableExists($db, $table) {
    try {
      $table = escapeshellarg($table);
      $result = $db->query("SELECT 1 FROM $table LIMIT 1");
      return $result !== FALSE;
    } catch (Exception $e) {
      error_log("table '$table' doesn't exist: " . print_r($db->errorInfo(), TRUE));
      return false;
    }
  }

  public static function placeholders($text, $count=0, $separator=",") {
    $result = array();
    if($count > 0){
      for($x=0; $x<$count; $x++){
        $result[] = $text;
      }
    }

    return implode($separator, $result);
  }


  public static function liveDBFilePath() {
    return build_file_path(cptBaseDir(), 'app', 'live.db');
  }
};

class BaseModel {
  public function __construct() { }

  protected function get_db() {
    return DBConn::instance()->db();
  }

  public function create($attrs) {
    $tblName = $this->tableName();
    if (is_null($tblName))
      return NULL;
    if (!isset($attrs))
      return NULL;

    $db = $this->get_db();
    $cols = array();
    $vals = array();
    foreach(array_keys($attrs) as $k) {
      $k = escapeshellarg($k);
      $cols[] = $k;
      $vals[] = "?";
    }
    $cols = implode(',', $cols);
    $vals = implode(',', $vals);
    $tblName = escapeshellarg($tblName);
    $command = "INSERT INTO {$tblName} ($cols) VALUES ($vals);";
    $stmt = $db->prepare($command);
    $index = 1;
    foreach(array_keys($attrs) as $k) {
      $stmt->bindValue($index, $attrs[$k]);
      ++$index;
    }
    if ($stmt->execute())
      return $db->lastInsertId();
    else
      return false;
  }

  public function update($attrs, $selectors) {
    $tblName = $this->tableName();
    if (is_null($tblName))
      return NULL;

    if (!isset($attrs) || !isset($selectors))
      return NULL;

    $db = $this->get_db();

    $assignments = array();
    foreach(array_keys($attrs) as $k) {
      $assignments[] = "$k=?";
    }
    $assignment = implode(',', $assignments);
    $conditions = array();
    foreach(array_keys($selectors) as $k) {
      $conditions[] = "$k=?";
    }
    $condition = implode(' AND ', $conditions);
    $tblName = escapeshellarg($tblName);
    $command = "UPDATE {$tblName} SET {$assignment} WHERE {$condition}";
    $stmt = $db->prepare($command);
    $index = 1;
    foreach(array_keys($attrs) as $k) {
      $stmt->bindValue($index, $attrs[$k]);
      ++$index;
    }

    foreach(array_keys($selectors) as $k) {
      $stmt->bindValue($index, $selectors[$k]);
      ++$index;
    }
    $stmt->execute();
    return true;
  }

  public function read() {
  }

  public function delete() {
  }

  protected function tableName() {
    return NULL;
  }

  protected function find_by($colname, $value) {
    $tblName = $this->tableName();
    if (is_null($tblName))
      return NULL;

    $db = $this->get_db();
    $query = "SELECT * FROM {$tblName} WHERE {$colname} = ?";
    $stmt = $db->prepare($query);
    $stmt->bindValue(1, $value);
    $stmt->execute();
    return $stmt->fetchAll();
  }

  protected function find_by2($name, $clientIP) {
    $tblName = $this->tableName();
    if (is_null($tblName))
      return NULL;
    $db = $this->get_db();
    $tblName = escapeshellarg($tblName);
    $name = escapeshellarg($name);
    $clientIP = escapeshellarg($clientIP);
    $query = "SELECT * FROM {$tblName} WHERE name={$name} AND client_ip={$clientIP}";
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll();
  }
};

class Permission extends BaseModel {
  protected function tableName() {
    return 'permissions';
  }

  public function permissions($user_id=null) {
    $db = $this->get_db();
    $stmt = null;
    if (is_null($user_id)) {
      $query = <<<SQL
SELECT u.id as user_id, u.name as name, u.home_page as home_page, u.dev_readable as dev_readable, u.dev_writable as dev_writable, p.path, p.readable, p.writable
FROM users AS u LEFT OUTER JOIN permissions AS p ON u.id = p.user_id
ORDER BY user_id DESC;
SQL;
      $stmt = $db->prepare($query);
    } else {
      $query = <<<SQL
SELECT u.id as user_id, u.name as name, u.home_page as home_page, u.dev_readable as dev_readable, u.dev_writable as dev_writable, p.path, p.readable, p.writable
FROM users AS u LEFT OUTER JOIN permissions AS p ON u.id = ? and u.id = p.user_id;
SQL;
      $stmt = $db->prepare($query);
      $stmt->bindValue(1, $user_id);
    }
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function delPermissionsByUserId($user_id) {
    $db = $this->get_db();
    $command = "DELETE FROM permissions WHERE user_id = :user_id";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':user_id', $user_id);
    $stmt->execute();
  }

  public function findByUserAndPath($user_id, $path) {
    $db = $this->get_db();
    $query = "SELECT * FROM {$this->tableName()} WHERE user_id = :user_id AND path = :path;";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user_id', $user_id);
    $stmt->bindValue(':path', $path);
    $stmt->execute();
    return $stmt->fetchAll();
  }
};

class User extends BaseModel {
  protected function tableName() {
    return 'users';
  }

  public function findById($id) {
    $rows = parent::find_by('id', $id);
    if (sizeof($rows) == 0)
      return false;

    $this->attrs = $rows[0];
    return true;
  }

  public function find($name) {
    $rows = parent::find_by('name', $name);
    if (sizeof($rows) == 0)
      return false;

    $this->attrs = $rows[0];
    return true;
  }

  public function attr($name) {
    return $this->attrs[$name];
  }

  public function authenticate($password) {
    $checksum = hash('sha256', $password . $this->attr('salt'));
    return $checksum == $this->attr('checksum');
  }

  public function authenticate2($authHash, $authRand) {
    $checksum = hash('sha256', $this->attr('checksum') . $authRand);
    return $checksum == $authHash;
  }

  public function changePassword($password) {
    $checksum = hash('sha256', $password . $this->attr('salt'));
    return $this->changePassword2($checksum);
  }

  public function changePassword2($checksum) {
    $result = false;
    $result = $this->update(array('checksum' => $checksum), array('id' => $this->attr('id')));
    if ($result && $this->attr("name") == "admin")
      updatePhpAdminAuth($this->attr('salt'), $checksum);
    return $result;
  }

  public function changeHomePage($home_page) {
    if (function_exists('filter_var'))
      $home_page = filter_var($home_page, FILTER_SANITIZE_URL);

    if ($home_page == $this->attr('home_page'))
        return;
    return $this->update(array('home_page' => $home_page), array('id' => $this->attr('id')));
  }

  public function createAccount($name, $password, $utility_enabled, $system_enabled, $account_management_enabled, $password_change_enabled, $dashboard_enabled, $dashboard_as_landing_page) {
    if ($this->find($name))
      return false;

    $salt = randStr();
    $checksum = hash('sha256', $password . $salt);
    return $this->createAccount2($name, $checksum, $salt, $utility_enabled, $system_enabled, $account_management_enabled, $password_change_enabled, $dashboard_enabled, $dashboard_as_landing_page);
  }

  public function createAccount2($name, $checksum, $salt, $utility_enabled, $system_enabled, $account_management_enabled, $password_change_enabled, $dashboard_enabled, $dashboard_as_landing_page) {
    if ($this->find($name))
      return false;

    $vals = array(
      'name' => $name,
      'salt' => $salt,
      'checksum' => $checksum,
      'utility_enabled' => $utility_enabled,
      'system_enabled' => $system_enabled,
      'account_management_enabled' => $account_management_enabled,
      'password_change_enabled' => $password_change_enabled,
      'dashboard_enabled' => $dashboard_enabled,
      'dashboard_as_landing_page' => $dashboard_as_landing_page);
    if ($dashboard_enabled == 't')
      $vals['dev_readable'] = 't';
    $this->create($vals);
    return true;
  }

  private function hashPassword($password, $salt) {
    return hash('sha256', $password . $salt);
  }

  public function isAdmin() {
    return $this->attr('is_admin') == 't' ;
  }

  public function shouldChangePassword() {
    $checksum = hash('sha256', isOEM() ? "hellospt" : "hellocpt" . $this->attr('salt'));
    if($this->attr('password_change_enabled') == 't') {
      return $checksum == $this->attr('checksum');
    }
    return false;
  }

  public function isUtilityEnabled() {
    return $this->isAdmin() || $this->attr('utility_enabled') == 't';
  }

  public function isSystemEnabled() {
    return $this->isAdmin() || $this->attr('system_enabled') == 't';
  }

  public function isAccountManagementEnabled() {
    return $this->isAdmin() || $this->attr('account_management_enabled') == 't';
  }

  public function isPasswordChangeEnabled() {
    return $this->isAdmin() || $this->attr('password_change_enabled') == 't';
  }

  public function isDashboardEnabled() {
    return $this->attr('dashboard_enabled') == 't';
  }

  public function isDashboardLanding() {
    return $this->attr('dashboard_as_landing_page') == 't';
  }

  public function isLinkedWithFI() {
    return $this->attr('fi_linked') == 't';
  }

  public function can($action, $grPath=null) {
    if ($this->isAdmin())
      return true;

    if (is_null($grPath))
    {
      if ($action == 'read')
        return strcasecmp($this->attr('dev_readable'), 't') == 0;
      else if ($action == 'write')
        return strcasecmp($this->attr('dev_writable'), 't') == 0;
      else
        return false;
    }

    $perm = new Permission();
    $perm_datas = $perm->findByUserAndPath($this->attr('id'), $grPath);
    if (count($perm_datas) == 0)
      return true;

    foreach($perm_datas as $data) {
      if (strcasecmp($data['readable'], 'f') == 0  || strcasecmp($data['readable'], 'false') == 0)
        return false;
      if ($action == 'write' && (strcasecmp($data['writable'], 'f') == 0  || strcasecmp($data['writable'], 'false') == 0))
        return false;
    }
    return true;
  }

  public function canDashboard($action, $dashboardId) {
    if ($this->isAdmin())
      return true;

    $perm_datas = DashboardPermission::findByUserAndDashboard($this->attr('id'), $dashboardId);
    if (count($perm_datas) == 0)
      return false;

    foreach($perm_datas as $data) {
      if (!isset($data[$action]))
        continue;

      return strcasecmp($data[$action], 't') == 0;
    }
    return true;
  }

  public function updatePermissions($perms, $dev_readable='f', $dev_writable='f') {
    $user_id = $this->attr("id");
    $new_datas = array();
    if (isset($perms))
    {
      foreach($perms as $perm) {
        $attrs = array();
        $attrs['user_id'] = $user_id;
        $attrs['path'] = $perm['path'];
        if (!isset($attrs['path'])) {
          return "invalid path";
        }

        $readable = (strcasecmp($perm['readable'], 'true') == 0) || (strcasecmp($perm['readable'], 't') == 0);
        $attrs['readable'] = $readable ? 't' : 'f';

        $writable = (strcasecmp($perm['writable'], 'true') == 0) || (strcasecmp($perm['writable'], 't') == 0);
        $attrs['writable'] = $writable ? 't' : 'f';
        $new_datas[] = $attrs;
      }
    }

    $db = $this->get_db();
    $db->beginTransaction();

    // update dev perms
    $dev_perm_data = array('dev_readable' => $dev_readable, 'dev_writable' => $dev_writable);
    $result = $this->update($dev_perm_data, array('id' => $this->attr('id')));
    if (!$result)
    {
      $db->rollBack();
      return "update user's dev permission failed";
    }

    // update gr files permission
    $model = new Permission();
    $model->delPermissionsByUserId($user_id);
    //TODO: optimize below loop to use query statement
    foreach($new_datas as $attrs)
      $model->create($attrs);
    $db->commit();
    return '';
  }

  public function updateDashboardPermissions($perms) {
    $user_id = $this->attr("id");
    $new_datas = array();
    foreach($perms as $perm) {
      $attrs = array();
      $attrs['user_id'] = $user_id;
      $attrs['dashboard_id'] = $perm['dashboardId'];
      $attrs['canManageDataSource'] = $perm['canManageDataSource'];
      $attrs['canManageWidget'] = $perm['canManageWidget'];
      $attrs['canViewDashboard'] = $perm['canViewDashboard'];
      $attrs['canWriteDataSource'] = $perm['canWriteDataSource'];

      if (!isset($attrs['dashboard_id'])) {
        return L("invalid dashboard_id");
      }

      $new_datas[] = $attrs;
    }

    $db = $this->get_db();
    $db->beginTransaction();
    $model = new DashboardPermission();
    $model->delPermissionsByUserId($user_id);
    //TODO: optimize below loop to use query statement
    foreach($new_datas as $attrs)
      $model->create($attrs);
    $db->commit();
    return '';
  }


  public function grBlackList() {
    $user_id = $this->attr("id");
    $db = $this->get_db();
    $query = <<<SQL
SELECT p.path as path
FROM users AS u JOIN permissions AS p ON u.id = ? and u.id = p.user_id and p.readable = 'f' and p.writable = 'f';
SQL;
    $stmt = $db->prepare($query);
    $stmt->bindValue(1, $user_id);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
  }

  public function updateAccount($values) {
    $user_id = $this->attr("id");

    $db = $this->get_db();
    $db->beginTransaction();

    if (array_key_exists('password', $values)) {
      $this->changePassword($values['password']);
      unset($values['password']);
    }
    if (array_key_exists('authHash', $values)) {
      $this->changePassword2($values['authHash']);
      unset($values['authHash']);
    }

    //when user is dashboard enabled, then it should also have 'dev_readable'
    //perm, otherwise data polling will not work in dashboard
    if (array_key_exists('dashboard_enabled', $values)) {
      if ($values['dashboard_enabled'] == 't')
        $values['dev_readable'] = 't';
    }

    unset($values['id']);
    //FIXME: this may cause SQL injection issue, since the keys in $values can
    //possibly provided by user
    if (!empty($values))
      $this->update($values, array('id' => $user_id));

    $db->commit();
  }

  public function deleteAccount() {
    $user_id = $this->attr("id");

    $db = $this->get_db();
    $db->beginTransaction();

    $tblName = $this->tableName();
    $command = "DELETE FROM {$tblName} WHERE id = :id ;";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':id', $user_id);
    $stmt->execute();

    $perm = new Permission();
    $perm->delPermissionsByUserId($user_id);

    $dperm = new DashboardPermission();
    $dperm->delPermissionsByUserId($user_id);

    $faccess = new FeatureAccessControl();
    $faccess->delRulesByUserId($user_id);

    $db->commit();
  }

  public function accounts($user_id=null) {
    $db = $this->get_db();
    $stmt = null;
    if (is_null($user_id)) {
      $query = <<<SQL
SELECT id as user_id, name, utility_enabled, system_enabled, account_management_enabled, password_change_enabled, dashboard_enabled, dashboard_as_landing_page
FROM users ORDER BY user_id DESC;
SQL;
      $stmt = $db->prepare($query);
    } else {
      $query = <<<SQL
SELECT id as user_id, name, utility_enabled, system_enabled, account_management_enabled, password_change_enabled, dashboard_enabled, dashboard_as_landing_page
FROM users WHERE id = ? ORDER BY user_id DESC;
SQL;
      $stmt = $db->prepare($query);
      $stmt->bindValue(1, $user_id);
    }
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }

  public static function validatePassword($password) {
    $minLength = 12;
    $maxLength = 32;
    if (strlen($password) < $minLength)
      return sprintf(L("password should be longer than %d characters"), $minLength);

    if (strlen($password) >= $maxLength)
      return sprintf(L("password should be shorter than %d characters"), $maxLength);

    # must include lowercase character
    if (!preg_match('/[a-z]/', $password))
      return L("password must include lowercase characters");

    # must include uppercase character
    if (!preg_match('/[A-Z]/', $password))
      return L("password must include uppercase characters");

    # must include number
    if (!preg_match('/[0-9]/', $password))
      return L("password must include number characters");

    # must include punctuation
    if (!preg_match('/[!-.:-@^-`{-~]/', $password))
      return L("password must include punctuation characters");

    return null;
  }

  public static function linkAccountWithId($ids, $link) {
    $in_expr = implode(",", array_fill(0, count($ids), '?'));
    $val = $link ? 't' : 'f';
    $query = "UPDATE users SET fi_linked = '{$val}' WHERE id IN ( " . $in_expr . " ) ;";

    $db = DBConn::instance()->db();
    $stmt = $db->prepare($query);
    foreach($ids as $k => $id)
      $stmt->bindValue(($k+1), $id);

    if (!$stmt->execute())
      return L("failed to link accounts");

    return null;
  }

  public static function linkAccounts($accounts, $link) {
    $in_expr = implode(",", array_fill(0, count($accounts), '?'));
    $val = $link ? 't' : 'f';
    $query = "UPDATE users SET fi_linked = '{$val}' WHERE name IN ( " . $in_expr . " ) ;";

    $db = DBConn::instance()->db();
    $stmt = $db->prepare($query);
    foreach($accounts as $k => $name)
      $stmt->bindValue(($k+1), $name);

    if (!$stmt->execute())
      return L("failed to link accounts");

    return null;
  }

  protected $attrs = NULL;
};

function output_history_data($stmt, &$formatter, $download_file_name, $orig_columns=null, $bool_text_mappings=null, $dbh=null, $table=null, $dt_tz=null) {
  $first = true;
  $utc_tz = new DateTimeZone('UTC');
  $formatter->writeHeader($download_file_name);
  $dt_format = 'Y-m-d H:i:s';
  $stmt->setFetchMode(PDO::FETCH_ASSOC);
  $bool_col_indexes = array();
  while($row = $stmt->fetch()) {
    //use '_dt' to overwrite 'dt' column
    if (array_key_exists('dt', $row) && array_key_exists('_dt', $row))
    {
      $row['dt'] = $row['_dt'];
      unset($row['_dt']);
    }

    //get headers row
    if ($first) {
      $first = false;
      $formatter->writeRow(array_keys($row));

      if (!is_null($bool_text_mappings)) {
          $col_count = count(array_keys($row));
          for($i=0; $i< $col_count; ++$i) {
              $meta = $stmt->getColumnMeta($i);
              if (!is_null($meta) && array_key_exists('sqlite:decl_type', $meta) && $meta['sqlite:decl_type'] == 'BOOLEAN') {
                  $bool_col_indexes[] = $i;
              }
          }
      }
    }

    //convert dt to given timezone
    if (isset($dt_tz) && $dt_tz->getName() != 'UTC' && array_key_exists('dt', $row)) {
      $dt_in_utc = DateTime::createFromFormat($dt_format, $row['dt'], $utc_tz);
      $row['dt'] = $dt_in_utc->setTimeZone($dt_tz)->format($dt_format);;
    }

    $col_values = array_values($row);
    if (!is_null($bool_text_mappings)) {
        // convert bool to user given boolean text if there is any
        for($i=0; $i<count($bool_col_indexes); ++$i) {
          $bool_index = $bool_col_indexes[$i];
          $bool_val = $col_values[$bool_index];
          if (!array_key_exists($bool_val, $bool_text_mappings))
              continue;

          $bool_text = $bool_text_mappings[$bool_val];
          if (!is_null($bool_text))
              $col_values[$bool_index] = $bool_text;
        }
    }

    $formatter->writeRow($col_values);
  }

  // error_log("### elapsed after fetch: " . $etime->elapsed());

  //when there is no data, try to return row of columns
  if ($formatter->isEmpty() && (!is_null($orig_columns))) {
    if ($orig_columns != '*') {
      $formatter->writeRow(explode(',', $orig_columns));
    } else {
      if (!is_null($dbh) && !is_null($table)) {
        $col_names = DBConn::instance()->getColumnNames($dbh, $table);
        $formatter->writeRow($col_names);
      }
    }
  }

  $formatter->writeFooter();
  $formatter->done();
};


class Dashboard extends BaseModel {
  protected $attrs = array();

  protected function tableName() {
    return 'dashboards';
  }

  public function findById($id) {
    $rows = parent::find_by('id', $id);
    if (sizeof($rows) == 0)
      return false;

    $this->attrs = $rows[0];
    return true;
  }

  public function isNull() {
    return !isset($this->attrs) || empty($this->attrs);
  }

  public function attr($name) {
    if (isset($this->attrs[$name]))
      return $this->attrs[$name];
    else
      return null;
  }

  public function toJson() {
    return map2json($this->attrs);
  }

  public function createDashboard($name, $content) {
    $vals = array(
      'name' => $name,
      'content' => $content,
      'parent_id' => 1
    );
    return $this->create($vals);
  }

  public function saveDashboard($name=NULL, $content=NULL) {
    $vals = array();
    if ($name)
      $vals['name'] = $name;

    if ($content)
      $vals['content'] = $content;

    return $this->update($vals, array('id' => $this->attr('id')));
  }

  public function renameDashboard($id, $new_name) {
    if (!$this->findById($id))
      return false;

    return $this->saveDashboard($new_name);
  }

  public static function updateLayout($id, $parent_id, $sibling_ids) {
    $stmt = null;
    $db = DBConn::instance()->db();
    $db->beginTransaction();

    $query = 'UPDATE dashboards SET parent_id=? WHERE id=?;';
    $stmt = $db->prepare($query);
    $stmt->bindValue(1, $parent_id);
    $stmt->bindValue(2, $id);
    $stmt->execute();

    $query = 'UPDATE dashboards SET parent_id=?, order_val=? WHERE id=?;';
    $stmt = $db->prepare($query);
    foreach($sibling_ids as $index=>$sid) {
      $stmt->bindValue(1, $parent_id);
      $stmt->bindValue(2, $index+1);
      $stmt->bindValue(3, $sid);
      $stmt->execute();
    }

    return $db->commit();
  }

  public static function deleteDashboard($id_list) {
    $db = DBConn::instance()->db();
    $db->beginTransaction();

    $command = "DELETE FROM dashboards WHERE id IN ( :ids ) ;";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':ids', implode(",", $id_list));
    if (!$stmt->execute())
    {
      $db->rollBack();
      return false;
    }

    foreach($id_list as $dashboardId)
      DashboardPermission::delPermissionsByDashboardId($dashboardId);

    $db->commit();
    return true;
  }

  public static function dashboardList() {
    $db = DBConn::instance()->db();
    $stmt = null;
    $query = <<<SQL
SELECT id, name, parent_id FROM dashboards ORDER BY parent_id, order_val ASC;
SQL;
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }

  public static function indexDashboardByUser($user_id) {
    $db = DBConn::instance()->db();
    $query = <<<SQL
      SELECT id FROM dashboards WHERE parent_id = 1
          AND id NOT IN (SELECT dashboard_id FROM dashboard_permissions WHERE user_id = :user_id AND canViewDashboard = 'f') ORDER BY order_val, id ASC LIMIT 1;
SQL;
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user_id', $user_id);
    $stmt->execute();

    $dashboardId = $stmt->fetchColumn();
    $stmt->closeCursor();
    $dashboard = new Dashboard();
    if (isset($dashboardId))
      $dashboard->findById($dashboardId);
    return $dashboard;
  }

};


class DashboardPermission extends BaseModel {
  protected $attrs = array();

  protected function tableName() {
    return 'dashboard_permissions';
  }

  public function attr($name) {
    return $this->attrs[$name];
  }

  public function permissions($user_id=null, $dashboard_id=null) {
    $db = $this->get_db();
    $stmt = null;

    $query = <<<SQL
SELECT u.id as userId, u.name as userName, d.id as dashboardId, d.name as pageName, p.canManageDataSource, p.canManageWidget, p.canViewDashboard, p.canWriteDataSource
FROM users AS u JOIN dashboard_permissions AS p ON u.id = p.user_id JOIN dashboards as d ON d.id = p.dashboard_id
SQL;
    $where_conditions = array();
    if (!is_null($user_id))
      $where_conditions[] = ' u.id = :user_id ';
    if (!is_null($dashboard_id))
      $where_conditions[] = ' d.id = :dashboard_id ';
    if (!empty($where_conditions))
      $query = $query . ' WHERE ' . implode(' AND ', $where_conditions);
    $query = $query . ' ORDER BY userId DESC;';
    $stmt = $db->prepare($query);
    if (!is_null($user_id))
      $stmt->bindValue(':user_id', $user_id);
    if (!is_null($dashboard_id))
      $stmt->bindValue(':dashboard_id', $dashboard_id);

    $stmt->execute();
    $perms = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $users = array();
    $dashboards = array();
    if (is_null($user_id))
      $users = $db->query('SELECT id AS userId, name AS userName, role_id FROM users')->fetchAll();
    else {
      $user_stmt = $db->prepare("SELECT id AS userId, name AS userName, role_id FROM users WHERE id = ?");
      $user_stmt->bindValue(1, $user_id);
      $user_stmt->execute();
      $users = $user_stmt->fetchAll();
    }

    if (is_null($dashboard_id))
      $dashboards = $db->query('SELECT id AS dashboardId, name AS pageName FROM dashboards WHERE id != 1 ')->fetchAll();
    else {
      $dashboards_stmt = $db->prepare('SELECT id AS dashboardId, name AS pageName FROM dashboards WHERE id = ? ');
      $dashboards_stmt->bindValue(1, $dashboard_id);
      $dashboards_stmt->execute();
      $dashboards = $dashboards_stmt->fetchAll();
    }

    $result = array();
    foreach($users as $user) {
      $userId = $user['userId'];
      $userName = $user['userName'];
      $roleId = intval($user['role_id']);
      foreach($dashboards as $dashboard) {
        $dashboardId = $dashboard['dashboardId'];
        $pageName = $dashboard['pageName'];
        $permExisted = false;
        foreach($perms as $perm) {
          if ($perm['userId'] == $userId && $perm['userName'] == $userName && $perm['pageName'] == $pageName)
          {
            $permExisted = true;
            $row = array();
            foreach($perm as $key => $val)
              $row[$key] = $val;
            $result[] = $row;
            break;
          }
        }
        if (!$permExisted) {
          $canManageDataSource = $roleId == 1;
          $canManageWidget = $roleId == 1 || $roleId == 2;
          $canWriteDataSource = $roleId == 1 || $roleId == 2 || $roleId == 3;
          $canViewDashboard = true;
          $result[] = array(
            'userId' => $userId,
            'userName' => $userName,
            'dashboardId' => $dashboardId,
            'pageName' => $pageName,
            'canManageDataSource' => ($canManageDataSource ? 't' : 'f'),
            'canManageWidget' => ($canManageWidget ? 't' : 'f'),
            'canWriteDataSource' => ($canWriteDataSource ? 't' : 'f'),
            'canViewDashboard' => ($canViewDashboard ? 't' : 'f')
          );
        }
      }
    }

    return $result;
  }

  public function delPermissionsByUserId($user_id) {
    $db = $this->get_db();
    $command = "DELETE FROM dashboard_permissions WHERE user_id = :user_id";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':user_id', $user_id);
    $stmt->execute();
  }

  public static function delPermissionsByDashboardId($dashboard_id) {
    $db = DBConn::instance()->db();
    $command = "DELETE FROM dashboard_permissions WHERE dashboard_id = :dashboard_id";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':dashboard_id', $dashboard_id);
    $stmt->execute();
  }

  public static function findByUserAndDashboard($user_id, $dashboard_id) {
    $db = DBConn::instance()->db();
    $query = "SELECT * FROM dashboard_permissions WHERE user_id = :user_id AND dashboard_id = :dashboard_id;";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user_id', $user_id);
    $stmt->bindValue(':dashboard_id', $dashboard_id);
    $stmt->execute();
    return $stmt->fetchAll();
  }
};

class AuthKeys extends BaseModel {
  protected $attrs = array();

  protected function tableName() {
    return 'auth_keys';
  }

  public function attr($name) {
    return $this->attrs[$name];
  }

  public function find($key) {
    $rows = parent::find_by('key', $key);
    if (sizeof($rows) == 0)
      return false;

    $this->attrs = $rows[0];
    return true;
  }

  public function createKey($note, $expired_at) {
    $key = randStr(16);
    $vals = array(
      'key' => $key,
      'note' => $note,
      'expired_at' => $expired_at);
    $this->create($vals);
    return $key;
  }

  public function updateKey($key, $note, $expired_at) {
    return $this->update(array('note' => $note, 'expired_at' => $expired_at),
      array('key' => $key));
  }

  public function deleteKey($key) {
    $db = $this->get_db();
    $command = "DELETE FROM auth_keys WHERE key = :key";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':key', $key);
    $stmt->execute();
  }

  public function keys() {
    $db = $this->get_db();
    $query = "SELECT id, key, note, expired_at FROM auth_keys ORDER BY created_at DESC;";
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }

  public function isExpired() {
    $tz = new DateTimeZone('UTC');
    $now = new DateTime('NOW', $tz);
    $expired_at = DateTime::createFromFormat('Y-m-d H:i:s', $this->attr('expired_at'), $tz);
    return $now >= $expired_at;
  }

  public static function isValid($key) {
    $instance = new AuthKeys();
    if (!$instance->find($key))
      return false;

    return $instance->isExpired();
  }
};

class FailedAuthCounter extends BaseModel {
  protected $attrs = array();

  protected function tableName() {
    return 'failed_auth_counter';
  }

  public function attr($name) {
    return $this->attrs[$name];
  }

  public static function lockoutDuration($failedAuthCount) {
    return min(pow(2, $failedAuthCount), 3600); //lock out at most for 1hr
  }

  public function getCounter($name, $clientIP) {
    $rows = parent::find_by2($name, $clientIP);
    if (sizeof($rows) == 0)
      return 0;

    $this->attrs = $rows[0];
    return intval($this->attrs['counter']);
  }

  public function incrCounter($name, $clientIP) {
    if (is_null($name))
      return false;

    $tblName = $this->tableName();
    $rows = parent::find_by2($name, $clientIP);
    if (sizeof($rows) == 0) {
      $this->create(array('name' => $name, 'client_ip' => $clientIP, 'counter' => 1, 'lockout_time' => 0));
    } else {
      $this->attrs = $rows[0];
      $counter = intval($this->attrs['counter']);
      $startTime = intval($this->attrs['lockout_time']);
      $params = array('counter' => $counter+1);

      $now = time();
      $maxLockoutTime = FailedAuthCounter::lockoutDuration($counter);
      // when receive another failed login and old lockout already expired,
      // reset lockout_time from now again together increase lock counter,
      // that means current account will be locked for another round of
      // max_lockout_time 
      if ($startTime+$maxLockoutTime < $now)
        $params['lockout_time'] = $now;

      $this->update($params,
        array('name' => $name, 'client_ip' => $clientIP));
    }
    return true;
  }

  public function delRecord($name, $clientIP) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    $command = "DELETE FROM {$tblName} WHERE name= :name and client_ip= :client_ip";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':name', $name);
    $stmt->bindValue(':client_ip', $clientIP);
    $stmt->execute();
  }

  public function getLockoutTime($name, $clientIP) {
    $rows = parent::find_by2($name, $clientIP);
    if (sizeof($rows) == 0)
      return 0;

    $this->attrs = $rows[0];
    return intval($this->attrs['lockout_time']);
  }

  public function updateLockoutTime($name, $clientIP, $now) {
    return $this->update(array('lockout_time' => $now),
      array('name' => $name, 'client_ip' => $clientIP));
  }

};

class FeatureAccessControl extends BaseModel {
  protected $attrs = array();

  protected function tableName() {
    return 'feature_access_control';
  }

  public function attr($name) {
    return $this->attrs[$name];
  }

  protected function processRule($rules_text, $feature_list) {
    $result = array();
    foreach($feature_list as $feature_id) {
      $result[$feature_id] = 'f'; //by default all features are disabled for security reason
    }

    if (!isset($rules_text))
      return $result;

    foreach(explode(",", $rules_text) as $section) {
      $pair = explode(":", $section);
      if (sizeof($pair) != 2)
        continue;

      $result[$pair[0]] = $pair[1];
    }
    return $result;
  }

  public function rules($user_id) {
    $features = collectFeatures();
    $db = $this->get_db();
    $stmt = null;
    if (is_null($user_id)) {
      $query = <<<SQL
SELECT u.id as user_id, u.name as user_name, u.is_admin as is_admin, f.id as rule_id, f.rules as rules
FROM users AS u LEFT OUTER JOIN feature_access_control AS f ON u.id = f.user_id
ORDER BY user_id;
SQL;
      $stmt = $db->prepare($query);
    } else {
      $query = <<<SQL
SELECT u.id as user_id, u.name as user_name, u.is_admin as is_admin, f.id as rule_id, f.rules as rules
FROM users AS u LEFT OUTER JOIN feature_access_control AS f ON u.id = ? and u.id = f.user_id;
SQL;
      $stmt = $db->prepare($query);
      $stmt->bindValue(1, $user_id);
    }
    $stmt->execute();
    $rs = $stmt->fetchAll();
    foreach($rs as &$r) {
      $r['rules'] = $this->processRule($r['rules'], array_keys($features));
    }
    return $rs;
  }

  public function updateRules($rules) {
    $db = $this->get_db();
    $db->beginTransaction();
    foreach($rules as $rule) {
      $rule_list = array();
      foreach($rule['rules'] as $id => $val) {
        $rule_list[] = "$id:$val";
      }
      $rule_text = implode(',', $rule_list);

      $result = NULL;
      if (empty($rule['rule_id'])) { // insert
        $result = $this->create(array('user_id'=>$rule['user_id'], 'rules'=>$rule_text));
      } else { // update
        $result = $this->update(array('user_id'=>$rule['user_id'], 'rules'=>$rule_text),
          array('id'=>$rule['rule_id']));
      }
      if (!$result) {
        $db->rollBack();
        return false;
      }
    }
    $db->commit();
    return true;
  }

  public function canUserAccess($user_id, $feature_id) {
    $u = new User();
    $u->findById($user_id);
    if ($u->isAdmin())
      return true;

    $rows = $this->find_by("user_id", $user_id);
    if (sizeof($rows) == 0)
      return false;

    $this->attrs = $rows[0];
    $rules = $this->attrs["rules"];
    foreach(explode(",", $rules) as $rule) {
      $pair = explode(":", $rule);
      if (sizeof($pair) != 2)
        continue;

      if (trim($pair[0]) == $feature_id)
        return trim($pair[1]) == "t";
    }
    return false;
  }

  public function delRulesByUserId($user_id) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    $command = "DELETE FROM $tblName WHERE user_id = :user_id";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':user_id', $user_id);
    $stmt->execute();
  }
};

class ConfigStore extends BaseModel {
  protected $attrs = array();
  protected function tableName() {
    return 'config_store';
  }
  public function attr($name) {
    return $this->attrs[$name];
  }

  public function upsert($key, $value) {
    $rows = $this->find_by('key', $key);
    $result = true;
    if (sizeof($rows) == 0) { // insert
      $result = $this->create(array("key" => $key, "value" => $value));
    } else { // update
      $result = $this->update(array("value" => $value), array('key' => $key));
    }

    return !is_null($result) && $result !== false;
  }

  public function value($key, $default=null) {
    $rows = parent::find_by('key', $key);

    if (sizeof($rows) == 0)
      return $default;

    if (!isset($rows[0]['value']))
      return $default;
    return $rows[0]['value'];
  }

  public function deleteConfig($key) {
    $db = $this->get_db();
    $tableName = $this->tableName();
    $command = "DELETE FROM $tableName WHERE key = :key";
    $stmt = $db->prepare($command);
    $stmt->bindValue(':key', $key);
    $stmt->execute();
  }
};

//a table must fulfill following requirements to use ChangeLog to track its changes:
// * has 'id' column, auto increment identifier
class DeltaChange extends BaseModel {
  protected $table = null;
  protected $main_tables = array('users');
  protected $live_data_tables = array('alarms', 'notes', 'comp_info');
  protected $table_columns = array('users' => ['id', 'name', 'fi_linked']);

  public function __construct($table=null) {
    parent::__construct();
    $this->table = $table;
  }

  protected function get_db() {
    if (in_array($this->table, $this->main_tables))
      return DBConn::instance()->db();
    else
      return DBConn::instance()->live_db();
  }

  protected function tableName() {
    return $this->table;
  }

  public function fetchAll($from_id, $count) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    if (is_null($tblName) || !DBConn::tableExists($db, $tblName))
      return array();

    $cols = '*';
    if (isset($this->table_columns[$tblName]))
      $cols = implode(',', $this->table_columns[$tblName]);
    $tblName = escapeshellarg($tblName);
    $query = "SELECT {$cols} FROM {$tblName} WHERE id >= ? ORDER BY id ASC LIMIT ?";
    $stmt = $db->prepare($query);
    $stmt->bindValue(1, $from_id);
    $stmt->bindValue(2, $count);
    $stmt->execute();
    $rs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return $rs;
  }

  public function fetchEndChangeLogId($from_cid, &$count) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    if (is_null($tblName) || !DBConn::tableExists($db, $tblName))
      return null;

    $tblName = escapeshellarg($tblName);
    $query = <<<EOM
      SELECT MAX(id) AS end_cid, count(1) AS count FROM (
        SELECT id FROM change_logs 
        WHERE id > ? AND table_name = {$tblName} ORDER BY id ASC LIMIT ?
      );
EOM;
    $stmt = $db->prepare($query);
    $stmt->bindValue(1, $from_cid);
    $stmt->bindValue(2, $count);
    $stmt->execute();
    $rs = $stmt->fetch(PDO::FETCH_ASSOC);
    $count = intval(arrayGet($rs, "count", 0));
    return arrayGet($rs, "end_cid");
  }

  public function fetchAdded($from_cid, $end_cid) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    if (is_null($tblName) || !DBConn::tableExists($db, $tblName))
      return array();

    $cols = '*';
    if (isset($this->table_columns[$tblName]))
      $cols = implode(',', $this->table_columns[$tblName]);
    $tblName = escapeshellarg($tblName);
    $query = <<<EOM
      SELECT {$cols} FROM {$tblName}
        WHERE id IN (
          SELECT DISTINCT row_id FROM change_logs
            WHERE id > ? AND id <= ? AND table_name = {$tblName} AND op_type IN ('C', 'U')
        ) ORDER BY id ASC
EOM;
    $stmt = $db->prepare($query);
    if (!$stmt) {
      error_log("failed to prepare query: " . $query . " err: " . print_r($db->errorInfo(), true));
      return array();
    }
    $stmt->bindValue(1, $from_cid, PDO::PARAM_INT);
    $stmt->bindValue(2, $end_cid, PDO::PARAM_INT);
    $stmt->execute();
    $rs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return $rs;
  }

  public function fetchDeleted($from_cid, $end_cid) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    if (is_null($tblName) || !DBConn::tableExists($db, $tblName))
      return array();
    $tblName = escapeshellarg($tblName);
    $query = <<<EOM
      SELECT DISTINCT row_id AS id FROM change_logs
        WHERE id > ? AND id <= ? AND table_name = {$tblName} AND op_type = 'D'
        AND NOT EXISTS ( SELECT 1 FROM {$tblName} WHERE id = row_id )
        ORDER BY row_id ASC 
EOM;
    $stmt = $db->prepare($query);
    $stmt->bindValue(1, $from_cid);
    $stmt->bindValue(2, $end_cid);
    $stmt->execute();
    $rs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return $rs;
  }

  //given table's data will be summarized based on id and returned
  //
  //this stats has one drawback: it wil not handle modified record, to handle
  //this, 'last_modified_at' column should be added to the table or consider
  //the change_logs data in the stats
  public function stats($chunk_size=100) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    if (is_null($tblName) || !DBConn::tableExists($db, $tblName))
      return array();
    $tblName = escapeshellarg($tblName);
    $query = <<<EOM
      SELECT min(id) AS min_id, max(id) AS max_id, count(1) AS count
        FROM {$tblName} GROUP BY id/{$chunk_size} ORDER BY min_id;
EOM;
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }

  protected function createTriggers($db, $tables) {
    foreach($tables as $tblName) {
      if (!DBConn::tableExists($db, $tblName))
        continue;

      $db->exec(<<<EOD
      CREATE TRIGGER IF NOT EXISTS {$tblName}_create_changelog_trigger
      AFTER INSERT ON {$tblName}
      BEGIN
        INSERT INTO change_logs(table_name, row_id, op_type) VALUES ('{$tblName}', NEW.id, 'C');
      END;

      CREATE TRIGGER IF NOT EXISTS {$tblName}_update_changelog_trigger
      AFTER UPDATE ON {$tblName}
      BEGIN
        INSERT INTO change_logs(table_name, row_id, op_type) VALUES ('{$tblName}', NEW.id, 'U');
      END;

      CREATE TRIGGER IF NOT EXISTS {$tblName}_delete_changelog_trigger
      AFTER DELETE ON {$tblName}
      BEGIN
        INSERT INTO change_logs(table_name, row_id, op_type) VALUES ('{$tblName}', OLD.id, 'D');
      END;
EOD
      );
    }
  }

  public function setupTriggers() {
    $this->createTriggers(DBConn::instance()->db(), $this->main_tables);
    $this->createTriggers(DBConn::instance()->live_db(), $this->live_data_tables);
  }

  //keep change_logs table size in a reasonable size
  public static function houseKeep() {
    $num_of_records_to_keep = 10000;
    $dbs = array(DBConn::instance()->db(), DBConn::instance()->live_db());
    foreach($dbs as $db) {
      $db->exec(<<<EOM
        DELETE FROM change_logs
        WHERE id < (
            SELECT MIN(id)
            FROM (SELECT id
                  FROM change_logs
                  ORDER BY id DESC
                  LIMIT $num_of_records_to_keep))
  EOM
      );
    }
  }

};

class HistoryDelta extends BaseModel {

  protected function get_db() {
    return DBConn::instance()->history_db();
  }

  public function listTables() {
    $db = $this->get_db();
    $query = <<<EOM
      SELECT tbl_name, sql FROM sqlite_master
        WHERE type = 'table' AND tbl_name not like 'sqlite_%';
EOM;
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }

  public function fetchDelta($tbl_name, $from_date, $max_records) {
    $db = $this->get_db();
    if (empty($tbl_name) || !DBConn::tableExists($db, $tbl_name))
      return null;

    $tbl_name = escapeshellarg($tbl_name);
    $query = "SELECT * FROM {$tbl_name} WHERE dt > ? ORDER BY dt LIMIT ?;";
    $stmt = $db->prepare($query);
    if (!$stmt) {
      error_log("failed to build query, ignore it");
      return null;
    }
    $stmt->bindValue(1, $from_date);
    $stmt->bindValue(2, $max_records);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }
};

class Alarms extends BaseModel {
  protected function get_db() {
    return DBConn::instance()->live_db();
  }

  protected function tableName() {
    return "alarms";
  }

  public function createAlarm($uuid, $priority, $value, $text, $tags=null, $cls=null, $state=null) {
    if (strlen($uuid) == 0)
     return L("alarm component's uuid can not be empty");

    if ($priority <= 0 || $priority > 255)
     return L("alarm priority value must be > 0 and <= 255");

    $vals = array(
      "uuid" => $uuid, "priority" => $priority,
      "value" => $value, "text" => $text,
      "date" => utcnow(),
    );
    if (!is_null($tags))
      $vals["tags"] = $tags;
    if (!is_null($cls))
      $vals["class"] = $cls;
    if (!is_null($state))
      $vals["state"] = $state;

    $result = $this->create($vals);
    if (is_null($result))
      return L("failed to create alarm");
    else
      return null;
  }

  public function deleteAlarm($id_list) {
    $db = $this->get_db();
    $db->beginTransaction();

    $in_expr = implode(",", array_fill(0, count($id_list), '?'));
    $query = "DELETE FROM alarms WHERE id IN ( " . $in_expr . " ) ;";
    $stmt = $db->prepare($query);
    foreach($id_list as $k => $id)
      $stmt->bindValue(($k+1), $id);
    if (!$stmt->execute()) {
      $db->rollBack();
      return L("failed to delete alarms");
    }

    $note_query = "DELETE FROM notes WHERE alarm_id IN ( " . $in_expr . " ) ;";
    $note_stmt = $db->prepare($note_query);
    foreach($id_list as $k => $id)
      $note_stmt->bindValue(($k+1), $id);

    if (!$note_stmt->execute()) {
      $db->rollBack();
      return L("failed to delete alarm notes");
    }

    $db->commit();
    return null;
  }

  public function ackAlarm($id_list, $user) {
    $db = $this->get_db();
    $in_expr = implode(",", array_fill(0, count($id_list), '?'));
    $query = "UPDATE alarms SET ackn=1, ackn_user=:user, adate=:now WHERE id IN ( " . $in_expr . " ) ;";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user', $user);
    $stmt->bindValue(':now', utcnow());
    foreach($id_list as $k => $id)
      $stmt->bindValue(($k+1+2), $id); //NOTE: need to add offset 2 here

    if (!$stmt->execute()) {
      return L("failed to ack alarms");
    }
    return null;
  }

  public function createNote($id_list, $user, $text) {
    $db = $this->get_db();
    $query = 'INSERT INTO notes (alarm_id, user, date, text) VALUES (:alarm_id, :user, :now, :text)';
    $stmt = $db->prepare($query);

    $now = utcnow();
    $db->beginTransaction();
    foreach($id_list as $alarm_id) {
      $stmt->bindValue(':alarm_id', $alarm_id);
      $stmt->bindValue(':user', $user);
      $stmt->bindValue(':now', $now);
      $stmt->bindValue(':text', $text);
      if (!$stmt->execute()) {
        return L("failed to create note");
      }
    }
    $db->commit();
    return null;
  }

};

class CompInfo extends BaseModel {
  // refs: https://sqlite.org/limits.html SQLITE_MAX_VARIABLE_NUMBER
  protected $max_placeholder_vars_num = 900;

  public function __construct() {
    parent::__construct();
  }

  protected function get_db() {
    return DBConn::instance()->live_db();
  }

  protected function tableName() {
    return "comp_info";
  }

  protected function applyAdded($added) {
    if (count($added) <= 0) return true;

    $chunk_size = $this->max_placeholder_vars_num/count($added[0]);
    $chunks = array_chunk($added, $chunk_size);

    $db = $this->get_db();
    $tblName = $this->tableName();

    foreach($chunks as $chunk) {
      $values = array();
      $placeholder_groups = array();
      foreach($chunk as $d) {
        $placeholder_groups[] = '(' . DBConn::placeholders('?', sizeof($d)) . ')';
        array_push($values, ...array_values($d));
      }

      $colsStr = implode(", ", array_keys($chunk[0]));
      $query = "INSERT INTO {$tblName} ({$colsStr}) VALUES " . implode(',', $placeholder_groups);
      $stmt = $db->prepare($query);
      if (!$stmt) {
        error_log("failed to prepare insert query: " . $query . " err: " . print_r($db->errorInfo(), true));
        return false;
      }
      if (!$stmt->execute($values)) {
        $err = $stmt->errorInfo();
        error_log("failed to insert comp info: " . print_r($err, true));
        return false;
      }
    }
    return true;
  }

  protected function applyUpsert($records) {
    if (count($records) <= 0) return true;

    $chunk_size = $this->max_placeholder_vars_num/count($records[0]);
    $chunks = array_chunk($records, $chunk_size);

    $db = $this->get_db();
    $tblName = $this->tableName();
    $tblName = escapeshellarg($tblName);
    foreach($chunks as $chunk) {
      $values = array();
      $placeholder_groups = array();
      foreach($chunk as $d) {
        $placeholder_groups[] = '(' . DBConn::placeholders('?', sizeof($d)) . ')';
        array_push($values, ...array_values($d));
      }

      $colsStr = implode(", ", array_keys($chunk[0]));
      $query = "INSERT INTO {$tblName} ({$colsStr}) VALUES " . implode(',', $placeholder_groups);
      $query .= " ON CONFLICT (id) DO UPDATE ";

      $assignments = array();
      $conditions = array();
      foreach(array_keys($chunk[0]) as $col) {
        if ($col == "id") continue;
        $assignments[] = " {$col} = excluded.{$col} ";
        $conditions[] = " {$col} != excluded.{$col} ";
      }
      $query .= " SET " . implode(", ", $assignments);
      $query .= " WHERE " . implode(" OR ", $conditions);

      $stmt = $db->prepare($query);
      if (!$stmt) {
        error_log("failed to prepare insert query: " . $query . " err: " . print_r($db->errorInfo(), true));
        return false;
      }
      if (!$stmt->execute($values)) {
        $err = $stmt->errorInfo();
        error_log("failed to insert comp info: " . print_r($err, true));
        return false;
      }
    }
    return true;
  }

  protected function applyUpdated($updated) {
    if (count($updated) <= 0) return true;
    
    $headers = array_keys($updated[0]);
    foreach($headers as $header) {
      $assignments[] = "$header = :$header";   
    }

    $db = $this->get_db();
    $tblName = $this->tableName();
    $assignmentStr = implode(",", $assignments);
    $query = "UPDATE {$tblName} SET {$assignmentStr} WHERE id = :id";
    $stmt = $db->prepare($query);
    if (!$stmt) {
      error_log("failed to prepare update query: " . $query . " err: " . print_r($db->errorInfo(), true));
      return false;
    }
    
    foreach($updated as $d) {
      foreach($headers as $header) {
        if ($header == "id" || $header == "pid")
          $stmt->bindValue(":{$header}", $d[$header], PDO::PARAM_INT);
        else
          $stmt->bindValue(":{$header}", $d[$header]);
      }

      if (!$stmt->execute()) {
        error_log("failed to update comp info: " . print_r($d, true) . " err: " . print_r($db->errorInfo(), true));
        continue;
      }
    }
    return true;
  }

  protected function applyDeleted($deletedIds) {
    if (count($deletedIds) <= 0) return true;

    $chunk_size = $this->max_placeholder_vars_num;
    $chunks = array_chunk($deletedIds, $chunk_size);
    $db = $this->get_db();
    $tblName = $this->tableName();
    $tblName = escapeshellarg($tblName);
    foreach($chunks as $chunk) {
      $query = "DELETE FROM {$tblName} WHERE id IN (" . DBConn::placeholders('?', count($chunk)) . ")";
      $stmt = $db->prepare($query);
      if (!$stmt) {
        error_log("failed to prepare delete query: " . $query . " err: " . print_r($db->errorInfo(), true));
        return false;
      }

      if (!$stmt->execute($chunk)) {
        error_log("failed to delete comp info: " . implode(",", $chunk));
      }
    }
    return true;
  }

  protected function doLoadCompInfo($lines) {
    if (count($lines) <= 1) return false;

    $headerLine = array_shift($lines);
    $headers = explode(",", $headerLine);
    $db = $this->get_db();
    $tblName = $this->tableName();
    $query = "SELECT id,? FROM {$tblName}";
    $stmt = $db->prepare($query);
    if (!$stmt) {
      error_log("doLoadCompInfo, failed to build query, ignore it");
      return false;
    }
    $stmt->bindValue(1, $headerLine);
    $stmt->execute();
    $oldLines = array_map('reset', $stmt->fetchAll(PDO::FETCH_GROUP|PDO::FETCH_ASSOC));

    $records = array();
    $newIds = array();
    foreach($lines as $line) {
      $values = explode(",", trim($line));
      if (count($values) != count($headers)) continue;

      $data = array_combine($headers, $values);
      if ($data === False) {
        error_log("got invalid data: '$line', ignore it");
        continue;
      }
      $id = $data['id'];

      if ($id == '0')  // for app object, make pid as 0 in db
        $data['pid'] = '0';

      $newIds[] = $id;
      $records[] = $data;
    }
    
    $deletedIds = array_diff(array_keys($oldLines), $newIds);
    $db->beginTransaction();
    $result = $this->applyUpsert($records) && $this->applyDeleted($deletedIds);
    if ($result) 
      $db->commit();
    else
      $db->rollBack();
    return $result;
  }

  public function loadCompInfo() {
    $comp_info_path = "/tmp/compInfo.csv";
    $loading_path = "/tmp/compInfo.csv.loading";

    if (file_exists($comp_info_path)) {
      if (!rename($comp_info_path, $loading_path)) {
        error_log("failed to rename {$comp_info_path} to {$loading_path}");
        return false;
      }
    }
    if (!file_exists($loading_path))
      return false;

    //NOTE: there might be race condition here, when importing data, the
    //comp_info.csv file is updated
    $content = file_get_contents($loading_path);
    $lines = explode("\n", $content);
    if (count($lines) <= 1) {
      error_log("{$loading_path} seems empty, ignore it");
      return false;
    }
    
    if (!$this->doLoadCompInfo($lines))
    {
      error_log("failed to load comp_info");
      return false;
    } else if (!unlink($loading_path)) {
      error_log("failed to remove {$loading_path}");
      return false;
    }

    return true;
  }

  public function query($uuid) {
    $db = $this->get_db();
    $tblName = $this->tableName();
    if (is_null($tblName) || !DBConn::tableExists($db, $tblName))
      return array();

    $query = "SELECT * FROM {$tblName} WHERE uuid = :uuid ORDER BY id";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':uuid', $uuid);
    $stmt->execute();
    $rs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    return $rs;
  }
};

?>
