<?php
// vim: ts=2 sw=2

include_once "db.php";

function collectBackups($path, $type, $returnJson) {
    $backups = array();
    if (!file_exists($path))
        return $backups;

    $type = strtolower($type);
    if ($handle = openDir($path)) {
        while (false !== ($entry = readdir($handle))) {
            if ($entry == "." || $entry == "..")
                continue;

            $fullPath = build_file_path($path, $entry);
            $withNetworkConf = withNetworkConfBackup($fullPath);
            $backupEntry = array("name" => $entry, "type" => $type,
                "withNetworkConf" => $withNetworkConf,
                "ts" => filemtime($fullPath));

            if ($returnJson)
                $backups[] = map2json($backupEntry);
            else
                $backups[] = $backupEntry;
        }
        closedir($handle);
    }
    return $backups;
}

function doListBackups($returnJson = true) {
    $basePath = backupBasePath();
    $backups = array();

    //for old platforms
    if (in_array(platformName(), array("FG+", "FG", "FG-")))
        $backups = collectBackups($basePath, "sdcard", $returnJson);
    else { // for newer platform check backups in flash and sdcard
        $backups = collectBackups($basePath, "flash", $returnJson);

        $sdBackupPath = sdcardBackupPath();
        if (!is_null($sdBackupPath) && does_sdcard_exist())
            $backups = array_merge($backups, collectBackups($sdBackupPath, 'sdcard', $returnJson));
    }
    return $backups;
}

function checkBackupError($backupPath, $backupDir) {
    rmIfExists($backupDir);
    $freeSpace = disk_free_space($backupPath);
    $minBackupSize = 4 * 1024 * 1024; //MB

    if ($freeSpace < $minBackupSize)
        return sprintf(L("failed to backup '%s', please make sure there is enough disk space."), 'graphics');
    else
        return sprintf(L("failed to backup '%s'."), 'graphics');
}

function doBackup($name, $cptBasePath = NULL, $backupHistoryDB = false, $type = "sdcard", $dataOnly = false, $backupNetworkConf = false) {
    $platformName = platformName();
    if (in_array($platformName, array("FG", "FG+", "FG-")) && !does_sdcard_exist())
        return L("can not make backup when there is no sdcard.");

    $type = strtolower($type);
    if ($type == "flash" && in_array($platformName, array("FG+", "FG", "FG-")))
        return L("can not backup to flash.");

    $backupPath = backupBasePath();
    if ($type == "sdcard") {
        $sdBackupPath = sdcardBackupPath();
        if (!is_null($sdBackupPath))
            $backupPath = $sdBackupPath;
    }
    $backupDir = build_file_path($backupPath, $name);
    if (file_exists($backupDir))
        return L("backup folder already exists, please choose a new folder.");

    // make sure all things have been saved
    sync();

    if (!mkdir($backupDir, 0777, true))
        return sprintf(L("failed to create backup folder '%s'."), $name);

    //write backup manifest data
    $manifest = array();
    $manifest['version'] = '2.0';
    $manifest['createdAt'] = utcnow();
    $manifest['dataOnly'] = $dataOnly;
    $manifest['networkConf'] = $backupNetworkConf;

    file_put_contents(build_file_path($backupDir, 'manifest.json'), map2json($manifest));

    //backup sedona files(app.sab & kits.scode)
    $sedonaPath = sedonaPath();
    if (!copy(build_file_path($sedonaPath, 'app.sab'),
            build_file_path($backupDir, 'app.sab'))) {
        rmIfExists($backupDir);
        return sprintf(L("failed to backup '%s'."), 'app.sab');
    }

    if (!copy(build_file_path($sedonaPath, 'kits.scode'),
            build_file_path($backupDir, 'kits.scode'))) {
        rmIfExists($backupDir);
        return sprintf(L("failed to backup '%s'."), 'kits.scode');
    }

    if (!copy(build_file_path($sedonaPath, 'cpt_session.xml'),
            build_file_path($backupDir, 'cpt_session.xml')))
        error_log("can not backup cpt_session.xml.");

    //backup app files
    if (empty($cptBasePath))
        $cptBasePath = cptBaseDir();
    $graphicDir = rtrim($cptBasePath, DIRECTORY_SEPARATOR);
    if (is_dir($graphicDir)) {
        if ($dataOnly) {
            $dataFiles = array();
            $dataFiles[] = build_file_path($graphicDir, "app", "cpt-web.db");
            $dataFiles[] = build_file_path($graphicDir, "app", "grdata");
            $dataFiles[] = build_file_path($graphicDir, "app", "graphics", "assets", "imported");
            $dataFiles[] = build_file_path($graphicDir, "dashboard", "uploads");
            $dataFiles[] = build_file_path($graphicDir, "user_data");
            foreach($dataFiles as $dataFile) {
                if (!file_exists($dataFile))
                    continue;

                $dstPath = build_file_path($backupDir, 'cpt',
                    substr($dataFile, strlen($graphicDir) + 1));
                if (!recurseCopy($dataFile, $dstPath)) {
                    error_log("failed to backup ".$dataFile.
                        " to ".$dstPath);
                    return checkBackupError($backupPath, $backupDir);
                }
            }
        } else {
            if (!recurseCopy($graphicDir, build_file_path($backupDir, 'cpt')))
                return checkBackupError($backupPath, $backupDir);
        }

        runPluginBackupCallbacks($backupDir, $graphicDir);

        if (!backupFirmwareCustomData($backupDir))
            return L("failed to backup firmware data");

        if ($backupHistoryDB) {
            $parentPath = dirname($graphicDir);
            $historyDBPath = build_file_path($parentPath, "easyio.db");
            if (file_exists($historyDBPath)) {
              $result = doBackupDB($historyDBPath, build_file_path($backupDir, 'easyio.db'));
              if ($result !== true) {
                rmIfExists($backupDir);
                return L($result);
              }
            } else
                error_log("can not find history db file at ".$historyDBPath);
            // return sprintf(L("can not find history db file '%s'."), 'easyio.db');
        }

        $macAddrPath = build_file_path($backupDir, "cpt", "app", "device.mac");
        file_put_contents($macAddrPath, macAddr());
    }

    // backup service control config
    $serviceBasePath = '/etc/init.d';
    if ($platformName == "FW") {
        $serviceBasePath = '/etc/rc.d';
    }
    $pattern = '/^([DS]\d+)(mqtt-service|pure-ftpd|sshd|openvpn|openvpn_client|smb|chrony|ntp|dropbear|samba|dnsmasq|chronyd|ntpd)/i';
    $files = glob($serviceBasePath.
        '/*');
    foreach($files as $file) {
        $fileName = basename($file);
        if (preg_match($pattern, $fileName)) {
            $dstPath = build_file_path($backupDir, 'root', $file);
            if (!recurseCopy($file, $dstPath)) {
                return sprintf(L("failed to backup network config file: '%s'."), $file);
            }
        }
    }
    $nginxBasePath = '/etc/nginx';
    $file = build_file_path($nginxBasePath, 'nginx.conf');
    $backupFile = build_file_path($backupDir, 'root', $nginxBasePath, 'nginx.conf');
    if (!recurseCopy($file, $backupFile)) {
        return sprintf(L("failed to backup file: '%s'."), $file);
    }

    if ($backupNetworkConf) {
        $networkConfs = array(
            '/etc/config/wireless',
            '/etc/config/network',
            '/etc/config/dhcp',
            '/etc/config/samba',
            '/mnt/fw.config',
            '/mnt/sedona/ethernetauth.config',
            '/etc/wpa_supplicant/wpa_supplicant_wired.conf'
        );
        foreach($networkConfs as $conf) {
            if (file_exists($conf)) {
                $dstPath = get_absolute_path(build_file_path($backupDir, 'root', $conf));
                if (!recurseCopy($conf, $dstPath))
                    return sprintf(L("failed to backup network config file: '%s'."), $conf);
            } else {
                error_log("can not find config file: ".$conf);
            }
        }
    }

    error_log("backup ".$name.
        " done");

    sync();
}

function isDataOnlyBackup($backupDir) {
    if (!function_exists("json_decode"))
        return false;

    $filePath = build_file_path($backupDir, 'manifest.json');
    if (file_exists($filePath)) {
        $manifest = json_decode(file_get_contents($filePath), true);
        if (!is_null($manifest) && isset($manifest['dataOnly']))
            return $manifest['dataOnly'];
    }
    return !file_exists(build_file_path($backupDir, 'cpt', 'js'));
}

function withNetworkConfBackup($backupDir) {
    if (!function_exists("json_decode"))
        return false;

    $filePath = build_file_path($backupDir, 'manifest.json');
    if (file_exists($filePath)) {
        $manifest = json_decode(file_get_contents($filePath), true);
        if (!is_null($manifest) && isset($manifest['networkConf']))
            return $manifest['networkConf'];
    }
    return false;
}

function needClearAccounts($backupDir) {
    // EASY-1237: need clear accounts in db for password encryption algorithm has changed
    if (!function_exists("json_decode")) return true;

    $filePath = build_file_path($backupDir, 'manifest.json');
    if (file_exists($filePath)) {
        $manifest = json_decode(file_get_contents($filePath), true);
        if (!is_null($manifest) && isset($manifest['version'])) {
            if ($manifest['version'] == '2.0')
                return false;
        }
    }

    return true;
}

function doRestore($name, $cptBasePath = NULL, $withHistoryDB = true, $type = "sdcard", $withNetworkConf = false, $prompt = true, & $errCode) {
    $platformName = platformName();
    if (in_array($platformName, array("FG", "FG+", "FG-")) && !does_sdcard_exist())
        return L("can not restore backup when there is no sdcard.");

    $type = strtolower($type);
    if ($type == "sdcard") {
        if (!does_sdcard_exist())
            return L("no sdcard available");

        $sdBackupPath = sdcardBackupPath();
        if (!file_exists($sdBackupPath)) {
            error_log("can not find sdcard mount path: ".$sdBackupPath);
            return L("can not find sdcard path");
        }

        $backupPath = $sdBackupPath;
    } else
        $backupPath = backupBasePath();

    $backupDir = build_file_path($backupPath, $name);

    if (!file_exists($backupDir))
        return sprintf(L("backup folder(%s) does not exist."), $backupDir);

    // make sure all things have been saved
    sync();

    if (!$prompt && needClearAccounts($backupDir)) {
        $errCode = 1001;
        return sprintf(L("backup '%s' is used for old firmware."), $name);
    }

    $sedonaAppRestored = false;

    $sedonaPath = sedonaPath();
    $appPathInBackup = build_file_path($backupDir, 'app.sab');
    $kitsPathInBackup = build_file_path($backupDir, 'kits.scode');
    if (file_exists($appPathInBackup) && file_exists($kitsPathInBackup)) {
        if (!isAppCompatible($appPathInBackup,
                build_file_path($sedonaPath, 'app.sab')))
            return L("app.sab is not compatible, give up.");

        if (!isScodeCompatible($kitsPathInBackup))
            return L("kits.scode is not compatible, give up.");

        if (!copy($appPathInBackup,
                build_file_path($sedonaPath, 'app.sab.writing')))
            return sprintf(L("failed to restore '%s'."), 'app.sab');
        if (!copy($kitsPathInBackup,
                build_file_path($sedonaPath, 'kits.scode.writing')))
            return sprintf(L("failed to restore '%s'."), 'kits.scode');

        error_log("copied app.sab and kits.scode");

        if (!rename(build_file_path($sedonaPath, 'app.sab.writing'),
                build_file_path($sedonaPath, 'app.sab.stage')) || !rename(build_file_path($sedonaPath, 'kits.scode.writing'),
                build_file_path($sedonaPath, 'kits.scode.stage')))
            return sprintf(L("failed to rename '%s' to '%s'."), 'writing', 'stage');
        $sedonaAppRestored = true;
    } else
        error_log("can not find sedona app in backup folder, skip it");

    $cpt_session_file = build_file_path($backupDir, 'cpt_session.xml');
    if (file_exists($cpt_session_file))
        copy($cpt_session_file,
            build_file_path($sedonaPath, 'cpt_session.xml'));

    if (empty($cptBasePath))
        $cptBasePath = cptBaseDir();
    $graphicDir = rtrim($cptBasePath, DIRECTORY_SEPARATOR);
    $parts = explode(DIRECTORY_SEPARATOR, $graphicDir);
    array_pop($parts);
    $cptParentDir = implode(DIRECTORY_SEPARATOR, $parts);

    $dstPath = null;

    $dataOnly = isDataOnlyBackup($backupDir);
    if ($dataOnly) {
        $dstPath = build_file_path($cptParentDir, 'cpt');
        if (!file_exists($dstPath))
            return L("failed to restore backup: 'cpt' folder missing.");
    } else {
        $dstPath = build_file_path($cptParentDir, 'cpt-writing');
        rmIfExists($dstPath);
        if (!recurseCopy(build_file_path($cptParentDir, 'cpt'), $dstPath))
            return L("failed to restore backup, please make sure there is enough free disk space.");
    }

    $cptFolderInBackup = build_file_path($backupDir, 'cpt');
    if (is_dir($cptFolderInBackup)) {
        $dataFiles = array();
        $dataFiles[] = build_file_path($cptFolderInBackup, "app", "graphics");
        $dataFiles[] = build_file_path($cptFolderInBackup, "app", "grdata");
        $dataFiles[] = build_file_path($cptFolderInBackup, "app", "cpt-web.db");
        $dataFiles[] = build_file_path($cptFolderInBackup, "app", "live.db");
        $dataFiles[] = build_file_path($cptFolderInBackup, "app", "settings.ini");
        $dataFiles[] = build_file_path($cptFolderInBackup, "app", "device.mac");
        $dataFiles[] = build_file_path($cptFolderInBackup, "dashboard", "uploads");
        $dataFiles[] = build_file_path($cptFolderInBackup, "user_codes");
        $dataFiles[] = build_file_path($cptFolderInBackup, "user_data");

        foreach ($dataFiles as $dataFile) {
            if (!file_exists($dataFile))
                continue;

            $dstFilePath = build_file_path($dstPath, substr($dataFile, strlen($cptFolderInBackup) + 1));
            rmIfExists($dstFilePath);
            if (!recurseCopy($dataFile, $dstFilePath)) {
                error_log("failed to restore ".$dataFile." to ".$dstFilePath);
                return L("failed to restore ".$dataFile." to ".$dstFilePath);
            }
        }

        error_log("cpt folder copied ");

        if (!$dataOnly) {
            rmIfExists(build_file_path($cptParentDir, 'cpt-old'));

            if (is_dir($graphicDir) && !rename($graphicDir, build_file_path($cptParentDir, 'cpt-old')))
                return sprintf(L("failed to rename %s to %s."), 'cpt', 'cpt-old');
            if (!rename($dstPath, $graphicDir))
                return sprintf(L("failed to rename %s to %s."), 'cpt.writing', 'cpt');
        }

        if (needClearAccounts($backupDir)) {
            // reset accounts in /cpt/app/cpt-web.db
            DBConn::reset(); // db file has been replaced, so reset the access
            $userDB = new User();
            $users = $userDB->accounts();
            $default_password = (function_exists("isOEM") && isOEM()) ? "hellospt" : "hellocpt";
            foreach ($users as $item) {
                $name = $item['name'];
                $u = new User();
                $u->find($name);
                if ($name == 'admin' || $name == 'Engineer' || 
                    $name == 'Manager' || $name == 'Operator' || 
                    $name == 'Viewer') {
                    $u->changePassword($default_password);
                } else if ($name == 'Kiosk') {
                    $u->changePassword('Kiosk');
                } else {
                    $u->deleteAccount();
                }
            }
        }
    } else
        error_log("can not find 'cpt' folder in this backup, skip it");

    runPluginRestoreCallbacks($graphicDir, $backupDir);

    if (!restoreFirmwareCustomData($backupDir))
        return L("failed to restore firmware data");

    if ($withHistoryDB) {
        $historyDBPath = build_file_path($backupDir, 'easyio.db');
        if (file_exists($historyDBPath)) {
            if (!copy($historyDBPath, build_file_path($cptParentDir, 'easyio.db.writing')))
                return sprintf(L("failed to restore '%s'."), 'easyio.db');

            if (!rename(build_file_path($cptParentDir, 'easyio.db.writing'), build_file_path($cptParentDir, 'easyio.db.stage')))
                return sprintf(L("failed to rename db '%s' to '%s'."), 'writing', 'stage');
        }
    }

    // restore service control config
    $serviceBasePath = '/etc/init.d';
    if ($platformName == "FW") {
        $serviceBasePath = '/etc/rc.d';
    }
    $serviceBackupDir = build_file_path($backupDir, 'root', $serviceBasePath);
    $files = glob($serviceBackupDir.
        '/*');
    if ($platformName == "FW") {
        $serviceFiles = glob('/etc/init.d/*');
        $pattern = '/(mqtt-service|sshd|dropbear|samba|dnsmasq|chronyd|ntpd)$/i';
        foreach($serviceFiles as $file) {
            $serviceName = basename($file);
            if (preg_match($pattern, $serviceName)) {
                $files = glob($serviceBackupDir.
                    '/S??'.$serviceName);
                if (count($files) > 0) {
                    doServiceControl($serviceName, "enable", false);
                } else {
                    doServiceControl($serviceName, "disable", false);
                }
            }
        }
    } else {
        foreach($files as $file) {
            $fileName = basename($file);
            $serviceName = substr($fileName, 3);
            $fileNameD = 'D'.substr($fileName, 1);
            $fileNameS = 'S'.substr($fileName, 1);
            $d_file = build_file_path($serviceBasePath, $fileNameD);
            $s_file = build_file_path($serviceBasePath, $fileNameS);
            if ($fileName == $fileNameS && file_exists($d_file)) {
                doServiceControl($serviceName, "enable", false);
            }
            if ($fileName == $fileNameD && file_exists($s_file)) {
                doServiceControl($serviceName, "disable", false);
            }
        }
    }
    $nginxBasePath = '/etc/nginx';
    $dstFile = build_file_path($nginxBasePath, 'nginx.conf');
    $file = build_file_path($backupDir, 'root', $nginxBasePath, 'nginx.conf');
    if (file_exists($file)) {
        if (!recurseCopy($file, $dstFile)) {
            return sprintf(L("failed to restore file: '%s'."), $file);
        }
        if (!chmod($dstFile, 0644)) {
            return sprintf(L("failed to chmod file: '%s'."), $file);
        }
    }

    if ($withNetworkConf) {
        //call external handler app to do customized restore process
        $postRestoreHandler = "/mnt/sedona/post_restore_handler";
        if (file_exists($postRestoreHandler)) {
            $backupConfigRootDir = build_file_path($backupDir, 'root');
            $output = rtrim(`( $postRestoreHandler $backupConfigRootDir 2>&1 && echo OK ) || echo ERROR`);
            if (!endsWith($output, 'OK'))
                error_log("failed to execute post restore handler");
        } else {
            error_log("can not find post restore handler at: ".$postRestoreHandler);
        }

        // //merge fw.config file first
        // $fw_config = "/mnt/fw.config";
        // $new_fw_config = get_absolute_path(build_file_path($backupDir, 'root', $fw_config));
        // if (file_exists($fw_config) && file_exists($new_fw_config))
        //   mergeFWConfig($fw_config, $new_fw_config);
        //
        // $networkConfs = array('/etc/config/wireless', '/etc/config/network', '/etc/config/dhcp', '/etc/config/samba');
        // foreach($networkConfs as $conf) 
        // {
        //   $confPath = get_absolute_path(build_file_path($backupDir, 'root', $conf));
        //   if (file_exists($confPath) && startsWith($confPath, $backupDir))
        //     if (!recurseCopy($confPath, $conf))
        //       return sprintf(L("failed to restore config file: '%s'."), $conf); 
        // }
    }

    clearCache();

    error_log("restored to ".$name);

    sync();

    $afterRestoreScript = "/usr/local/bin/after_restore.sh";
    if (file_exists($afterRestoreScript)) {
        $envParams = array();
        if ($withNetworkConf)
            $envParams[] = "network_config_restored=1";
        if ($withHistoryDB)
            $envParams[] = "history_db_restored=1";
        if ($sedonaAppRestored)
            $envParams[] = "sedona_app_restored=1";

        $cmd = implode(" ", $envParams).
        " ".$afterRestoreScript;
        $output = shell_exec($cmd);
    }

    if ($withNetworkConf)
        doReboot(true);
    else if ($sedonaAppRestored)
        restartSvm();
}

function doCompressBackup($name, $type) {
    $type = strtolower($type);
    if ($type == "sdcard") {
        if (!does_sdcard_exist())
            return L("no sdcard available");

        $sdBackupPath = sdcardBackupPath();
        if (!file_exists($sdBackupPath)) {
            error_log("can not find sdcard mount path: ".$sdBackupPath);
            return L("can not find sdcard path");
        }

        $backupPath = $sdBackupPath;
    } else
        $backupPath = backupBasePath();

    $backupDir = build_file_path($backupPath, $name);
    if (!file_exists($backupDir))
        return; // just return when backup folder is missing

    //tar -C /www/public/sdcard/cpt-backups -czvf /tmp/foo.tgz 20170914_backup 
    $prefix = platformName();
    if (isOEM())
        $prefix = oemPrefix();
    if ($prefix == "FG+") {
        // for FG+, there is not enough space under '/tmp' folder
        $root_path = parentPath(3);
        $tarFilePath = $root_path.
        "/{$prefix}-{$name}.tgz";
    } else
        $tarFilePath = "/tmp/{$prefix}-{$name}.tgz";

    //tar: tar -cf - t2 | gzip -c > test.tgz
    $cmd = "tar -C {$backupPath} -cf - ".escapeshellarg($name).
    " | gzip -c > ".escapeshellarg($tarFilePath);
    if (!execShellCmd($cmd)) {
        error_log("failed to build backup tgz");
        return;
    }

    return $tarFilePath;
}

function isUpgradeFix($backupDir) {
    if (!function_exists("json_decode"))
        return false;

    $filePath = build_file_path($backupDir, 'manifest.json');
    if (file_exists($filePath)) {
        $manifest = json_decode(file_get_contents($filePath), true);
        if (!is_null($manifest) && isset($manifest['dataOnly']) && isset($manifest['upgradeFix']))
            return $manifest['dataOnly'] && $manifest['upgradeFix'];
    }
    return false;
}

function doHandleUploadBackup($tempName, $tarBallName, $size, $type, $restoreNow, $withNetworkConf, $prompt = true, & $response) {
    $type = strtolower($type);
    if ($type == "sdcard") {
        if (!does_sdcard_exist())
            return L("no sdcard available");

        $sdBackupPath = sdcardBackupPath();
        if (!file_exists($sdBackupPath)) {
            error_log("can not find sdcard mount path: ".$sdBackupPath);
            return L("can not find sdcard path");
        }

        $backupPath = $sdBackupPath;
    } else
        $backupPath = backupBasePath();

    if (!file_exists($backupPath)) {
        if (!mkdir($backupPath, 0777, true)) {
            error_log("can not create backup folder: ".$backupPath);
            return L("can not create backup folder");
        }
    }

    $freeSpace = disk_free_space($backupPath);
    $compressRatio = 5;
    $size = (int) $size;
    if ($freeSpace <= $size * $compressRatio) {
        error_log("not enough disk free space({$freeSpace} bytes left)");
        return L("not enough disk free space({$freeSpace} bytes left)");
    }

    $ext = pathinfo($tarBallName, PATHINFO_EXTENSION);
    $name = basename($tarBallName, ".$ext");

    if (!is_uploaded_file($tempName)) {
        error_log("uploaded file {$name} verfication failed.");
        return L("uploaded file {$name} verfication failed.");
    }

    $cmd = null;
    if ($ext == "tgz")
        $cmd = "gunzip -c {$tempName} | tar -C {$backupPath} -xf -";
    elseif($ext == "zip")
    $cmd = "unzip -o -d {$backupPath} {$tempName}";
    else {
        error_log("invalid backup tarball format: $ext");
        return L("invalid backup tarball format: $ext");
    }

    if (!execShellCmd($cmd)) {
        error_log("failed to uncompress backup tarball");
        return L("failed to uncompress backup tarball");
    }

    //check if this is a upgrade tarball, if yes, apply it now and clean it up
    $output = null;
    if ($ext == "tgz")
        $output = shell_exec("gunzip -c {$tempName} | tar -tf -");
    elseif($ext == "zip")
    $output = shell_exec("unzip -l {$tempName} | awk 'NR == 4 {print $NF}'");

    if (!empty($output) && strlen($output) > 0) {
        $output = explode("\n", $output);
        $backupName = rtrim($output[0], DIRECTORY_SEPARATOR);
        $backupDir = build_file_path($backupPath, $backupName);
        $isUpgrade = isUpgradeFix($backupDir);
        if ($isUpgrade) {
            error_log("Found upgrade backup {$backupName}, apply this ugprade ...");
            $errCode = 1000;
            $error = doRestore($backupName, NULL, false, $type, $withNetworkConf, true, $errCode);
            error_log("Delete upgrade backup folder {$backupName}");
            rmIfExists($backupDir);

            if (!empty($error)) {
                error_log("Failed to apply upgrade: {$error}");
                return L("failed to apply upgrade: {$error}");
            } else {
                if (isset($response)) {
                    $response['msg'] = L("Success to apply upgrade {$backupName}.");
                    $response['reloadRequired'] = true;
                }
                error_log("Success to apply upgrade {$backupName}.");
            }
        } else if ($restoreNow) {
            $errCode = 1000;
            $error = doRestore($backupName, NULL, true, $type, $withNetworkConf, $prompt, $errCode);
            if ($errCode == 1001) {
                if (isset($response)) {
                    $response['error']['code'] = $errCode;
                    $response['error']['text'] = $error;
                    $response['backup']['name'] = $backupName;
                    $response['backup']['type'] = $type;
                    $response['backup']['withNetworkConf'] = $withNetworkConf;
                }
            } else if (!empty($error)) {
                error_log("Failed to restore backup: {$error}");
                return L("failed to restore backup: {$error}");
            } else {
                if (isset($response)) {
                    $response['msg'] = L("Success to restore to backup {$backupName}.");
                    $response['reloadRequired'] = true;
                }
                error_log("Success to restore to backup {$backupName}.");
            }
        }
    }

    return '';
}

function doDeleteBackup($name, $type) {
    $type = strtolower($type);
    if ($type == "sdcard") {
        if (!does_sdcard_exist())
            return L("no sdcard available");

        $sdBackupPath = sdcardBackupPath();
        if (!file_exists($sdBackupPath)) {
            error_log("can not find sdcard mount path: ".$sdBackupPath);
            return L("can not find sdcard path");
        }

        $backupPath = $sdBackupPath;
    } else
        $backupPath = backupBasePath();

    $backupDir = build_file_path($backupPath, $name);
    if (!file_exists($backupDir))
        return; // just return when backup folder is missing
    // return "backup folder(" . $backupDir . ") does not exist.";

    //security check
    if (!startsWith(realpath($backupDir), $backupPath))
        return L("invalid backup folder.");

    rmIfExists($backupDir);

    error_log("backup ".$name.
        " is deleted");

    sync();
}

function doRenameBackup($old_name, $new_name, $type) {
    $type = strtolower($type);
    if ($type == "sdcard") {
        if (!does_sdcard_exist())
            return L("no sdcard available");

        $sdBackupPath = sdcardBackupPath();
        if (!file_exists($sdBackupPath)) {
            error_log("can not find sdcard mount path: ".$sdBackupPath);
            return L("can not find sdcard path");
        }

        $backupPath = $sdBackupPath;
    } else
        $backupPath = backupBasePath();

    $oldBackupDir = build_file_path($backupPath, $old_name);
    if (!file_exists($oldBackupDir))
        return L("invalid backup folder.");

    $newBackupDir = build_file_path($backupPath, $new_name);
    if (file_exists($newBackupDir))
        return sprintf(L("rename failed. '%s' already exists."), $new_name);

    //security check
    if (!startsWith(realpath($oldBackupDir), $backupPath) || !startsWith($newBackupDir, $backupPath))
        return L("invalid backup folder.");

    if (!rename($oldBackupDir, $newBackupDir))
        return sprintf(L("can not rename backup from '%s' to '%s'"), $old_name, $new_name);

    error_log("backup '$old_name' is renamed to '$new_name'");

    sync();
}

function runPluginBackupCallbacks($backupDir, $graphicDir) {
    error_log("run plugin backup callback");
    $pluginsBackupDir = build_file_path($backupDir, "cpt", "plugins");
    if (!file_exists($pluginsBackupDir))
        mkdir($pluginsBackupDir, 0777);
    $plugins = collectPlugins();
    foreach($plugins as $plugin) {
        if (!is_array($plugin) || !isset($plugin['utilityFile']))
            continue;

        $utilityFilePath = build_file_path($graphicDir, $plugin['utilityFile']);
        if (!file_exists($utilityFilePath))
            continue;

        $utilityInstance = include $utilityFilePath;
        if (is_null($utilityInstance))
            continue;

        if (method_exists($utilityInstance, "backup"))
            $utilityInstance->backup($pluginsBackupDir);
    }
}

function runPluginRestoreCallbacks($graphicDir, $backupDir) {
    error_log("run plugin restore callback");
    $plugins = collectPlugins();
    foreach($plugins as $plugin) {
        if (!is_array($plugin) || !isset($plugin['utilityFile']))
            continue;

        $utilityFilePath = build_file_path($graphicDir, $plugin['utilityFile']);
        if (!file_exists($utilityFilePath))
            continue;

        $utilityInstance = include $utilityFilePath;
        if (is_null($utilityInstance))
            continue;

        if (method_exists($utilityInstance, "restore"))
            $utilityInstance->restore($backupDir);
    }
}

function backupFirmwareCustomData($backupDir) {
    $fwCustomDataDir = "/mnt/data";
    // NOTE: Copy firmware data folder to tmp folder first to avoid exceptions during compressing caused by data changes.
    $tmpCustomDataDir = "/tmp/firmware_data";
    if (file_exists($fwCustomDataDir)) {
        rmIfExists($tmpCustomDataDir);
        recurseCopy($fwCustomDataDir, $tmpCustomDataDir);
        $tarFilePath = build_file_path($backupDir, "firmware_data.tar");
        $output = shell_exec("tar -C $tmpCustomDataDir -cf $tarFilePath . && echo 'SUCCESS'");
        rmIfExists($tmpCustomDataDir);
        if (is_null($output) || preg_match("/SUCCESS\s*$/", $output) != 1) {
            error_log("failed to backup ".$fwCustomDataDir.
                " to ".$tarFilePath);
            return false;
        }
    }
    return true;
}

function restoreFirmwareCustomData($backupDir) {
    $fwCustomDataBackupTarFile = build_file_path($backupDir, "firmware_data.tar");
    if (file_exists($fwCustomDataBackupTarFile)) {
        $fwCustomDataDir = "/mnt/data";
        if (!file_exists($fwCustomDataDir))
            mkdir($fwCustomDataDir, 0644, TRUE);

        $output = shell_exec("tar -C /mnt/data -xf $fwCustomDataBackupTarFile && echo 'SUCCESS'");
        if (is_null($output) || preg_match("/SUCCESS\s*$/", $output) != 1) {
            error_log("failed to restore ".$fwCustomDataBackupTarFile.
                " to ".$fwCustomDataDir);
            return false;
        }
    }
    return true;
}

function doBackupDB($src, $dest)
{
  $pdo = null;
  $pdoDest = null;
  if ($src === null) {
    return "Source database path cannot be null";
  }
  if ($dest === null) {
    return "Destination database path cannot be null";
  }
  if (!file_exists($src)) {
    return "Source database does not exist: " . $src;
  }
  if (file_exists($dest)) {
    return "Destination database already exists: " . $dest;
  }
  error_log("Starting backup database from: " . $src . " to: " . $dest);
  try {
    $pdo = new PDO("sqlite:$src");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("VACUUM into '$dest'");
    $pdo = null;
    if (!file_exists($dest)) {
      return "Failed to backup database";
    }
    $pdoDest = new PDO("sqlite:$dest");
    $pdoDest->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $stmt = $pdoDest->query("PRAGMA integrity_check");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $pdoDest = null;
    $checkResult = $result["integrity_check"];
    if ($checkResult === "ok") {
      return true;
    } else {
      return "Failed to backup database: integrity check failed";
    }
  } catch (Exception $e) {
    $pdo = null;
    $pdoDest = null;
    return "Failed to backup database: " . $e->getMessage();
  }
}

?>
