<?php
/**
 * TinyRedisClient - the most lightweight Redis client written in PHP
 *
 * Usage example:
 * $client = new TinyRedisClient("host:port");
 * $client->set("key", "value");
 * $value = $client->get("key");
 *
 * Full list of commands you can see on http://redis.io/commands
 *
 * @author Petr Trofimov <petrofimov@yandex.ru>
 * @see https://github.com/ptrofimov
 */
class TinyRedisClient
{
    /** @var string */
    private $server;
    /** @var resource */
    private $socket;

    public function __construct($server = 'localhost:6379')
    {
        $this->server = $server;
    }

    public function __call($method, array $args)
    {
        array_unshift($args, $method);
        $cmd = '*' . count($args) . "\r\n";
        foreach ($args as $item) {
            $cmd .= '$' . strlen($item) . "\r\n" . $item . "\r\n";
        }
        $sock = $this->getSocket();
        if (!$sock)
          return;

        fwrite($sock, $cmd);

        return $this->parseResponse();
    }

    public function close() 
    {
      if ($this->socket) {
        stream_socket_shutdown($this->socket, STREAM_SHUT_RDWR);
        $this->socket = null;
      }
    }

    private function getSocket()
    {
        $timeout = 10;
        if (!$this->socket) {
          $this->socket = stream_socket_client($this->server, $errno, $errmsg, $timeout);
          if (!$this->socket)
            error_log("failed to connect redis server {$this->server}: $errmsg($errno)");
          else
            stream_set_timeout($this->socket, $timeout); //set timeout for read/write
        }
        return $this->socket;
    }

    private function parseResponse()
    {
        $line = fgets($this->getSocket());
        list($type, $result) = array($line[0], substr($line, 1, strlen($line) - 3));
        if ($type == '-') { // error message
            throw new Exception($result);
        } elseif ($type == '$') { // bulk reply
            if ($result == -1) {
                $result = null;
            } else {
                $line = fread($this->getSocket(), $result + 2);
                $result = substr($line, 0, strlen($line) - 2);
            }
        } elseif ($type == '*') { // multi-bulk reply
            $count = ( int ) $result;
            for ($i = 0, $result = array(); $i < $count; $i++) {
                $result[] = $this->parseResponse();
            }
        }

        return $result;
    }
}
