<?php
//vim: ts=2 sw=2

include_once "utils.php";

class BaseController {

  public $flash = NULL;

  function __construct() {}

  private function isHttps() {
    return (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'
    || $_SERVER['SERVER_PORT'] == 443);
  }

  protected function makeUrl($url) {
    if (startsWith($url, 'http://') || startsWith($url, 'https://'))
        return $url;
    $pageURL = 'http';
    if ($this->isHttps()) {$pageURL .= "s";}
    $pageURL .= "://" . $_SERVER["HTTP_HOST"];
    if ($_SERVER["SERVER_PORT"] != "80") {
      $port_str = ":".$_SERVER["SERVER_PORT"];
      if (!endsWith($pageURL, $port_str))
        $pageURL .=  $port_str;
    }
    if (startsWith($url, '/'))
      $pageURL .= $url;
    else
      $pageURL .= $this->cptBaseUrl() . '/' . $url;
    return $pageURL;
  }

  protected function cptBaseUrl() {
    $rootPath = $_SERVER['DOCUMENT_ROOT'];
    $curPath = substr(__FILE__, strlen($rootPath));
    $parts = explode(DIRECTORY_SEPARATOR, $curPath);
    array_pop($parts);
    array_pop($parts);

    //NOTE: since we are dealing with URL, so only '/' is valid, even on 
    //windows platform. no need to use DIRECTORY_SEPARATOR 
    $result = implode('/', $parts);
    if (!startsWith($result, '/'))
        $result = '/' . $result;
    return $result;
  }

  protected function isAjax() {
    return (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest');
  }

  protected function signinRequired() {
    return false;
  }

  protected function adminRequired() {
    return false;
  }
  
  protected function authKeyRequired() {
    return false;
  }

  protected function sameHostAuthFree() {
    return false;
  }

  protected function isSignedin() {
    return isset($_SESSION['user_id']);
  }

  protected function curUserId() {
    if (isset($_SESSION['user_id']))
      return $_SESSION['user_id'];
    else
      return null;
  }

  public function curUser() {
    if (!isset($_SESSION['user_name']))
      return NULL;

    $user_name = $_SESSION['user_name'];
    if (!isset($user_name))
      return NULL;

    $u = new User();
    if (!$u->find($user_name))
      return NULL;
    return $u;
  }
  protected function redirect($url, $permanent = false) {
    if ($this->isAjax())
    {
        $url = addcslashes($url,"\\\"\n\r");
        die('{"redirect": "' . $url . '"}');
    }
    else
    {
        header('Location: ' . $url, true, $permanent ? 301 : 302);
        die();
    }
  }

  protected function handleCORS() {
    //NOTE: to make CORS working, must put following config into appweb.conf file: 
    // CrossOrigin origin=* credentials=yes
    if (!isset($_SERVER['HTTP_ORIGIN'])) 
      return false;

    header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
    header("Access-Control-Allow-Credentials: true");
    header("Access-Control-Allow-Methods: POST,GET,OPTIONS");
    header("Access-Control-Allow-Headers: Origin,X-Requested-With,Keep-Alive,User-Agent,Content-Type,Content-Range,Range,Accept,If-Modified-Since,Cache-Control");
    header("Access-Control-Max-Age: 86400");

    $method = $_SERVER['REQUEST_METHOD'];
    if ($method == "OPTIONS") 
      return true;
    else
      return false;
  }

  protected function handleRequest() {
    if ($this->isAjax()) {
      $this->ajaxRun();
    } else {
      $this->normalRun();
    }
  }

  protected function isAdmin() {
    return isset($_SESSION['user_id']) &&  $_SESSION['user_id'] == 1 && isset($_SESSION['user_name']) && $_SESSION['user_name'] == 'admin';
  }

  public function run() {
    $this->sanitizeData();

    session_start(Array('name' => 'CPTSESSID'));

    # disable auth_key under standalone mode
    if (isset($_GET['auth_key']))
      unset($_GET['auth_key']);

    if ($this->handleCORS())
      return;

    if ($this->sameHostAuthFree() && $this->isLocalRequest()) {
      $this->handleRequest();
      return;
    }

    if ($this->signinRequired() && !$this->isSignedin()) {
      if (!$this->isAjax())
        $_SESSION['url_required'] = $_SERVER[REQUEST_URI];

      unset($_SESSION['user_id']);
      unset($_SESSION['user_name']); 
      unset($_SESSION['last_visit_time']); 

      $this->redirect($this->makeUrl('app/signin.php'));
    }

    if ($this->adminRequired() && !$this->isAdmin())
    {
      $_SESSION['flash'] = "Please signin as 'admin' user to continue.";
      $this->redirect($this->makeUrl('app/perm_error.php'));
    }

    $this->handleRequest();
  }

  public function flash() {
    if (!is_null($this->flash))
      return $this->flash;

    if(isset($_SESSION['flash'])) {
      $this->flash = $_SESSION['flash'];
      unset($_SESSION['flash']);
      return $this->flash;
    } else
      return NULL;
  }

  protected function normalRun() {
    $method = $_SERVER['REQUEST_METHOD'];
    if ($method == "POST") {
      $this->doPost();
    } else if ($method == "GET") {
      $this->doGet();
    } else {
        error_log("request method '" . $method . "' is not supported");
    }
  }

  protected function ajaxRun() {
    $response = array();
    $method = $_SERVER['REQUEST_METHOD'];
    if ($method == "POST") {
      $this->doAjaxPost();
    } else if ($method == "GET") {
      $this->doAjaxGet();
    } else {
      $response['error'] = array('text' => "$method is not supported");
      echo <<<EOS
{"error": {"text": "{$response['error']['text']}"}}
EOS;
    }
  }

  protected function renderAjaxError($response, $error) {
    $response['error'] = array('text' => addcslashes($error, "\\\"\n\r"));
    die(<<<EOS
{"error": {"text": "{$response['error']['text']}"}}
EOS
    );
  }

  protected function renderAjaxSuccess($response, $escapeStr=true) {
    $json = array();
    foreach($response as $k => $v) {
      if ($escapeStr && is_string($v))
          $v = addcslashes($v, "\\\"\n\r");
          // $v = addslashes($v); // should not escape single quote for json string
      $json[] = "\"$k\": \"$v\"";
    }
    die('{' . implode(',', $json) . '}');
  }

  protected function renderErrorResponse($error) {
    if ($this->isAjax()) {
      $response = array();
      $this->renderAjaxError($response, $error);
    } else {
      die($error);
    }
  }

  protected function escapeJsonChars($val) {
    return addcslashes($val, "\\\"\n\r");
  }

  protected function map2json($mapData) {
    $json = array();
    foreach($mapData as $k => $v) {
      if (is_string($v))
          $v = addcslashes($v,"\\\"\n\r");
      if (is_string($k))
          $k = addcslashes($k,"\\\"\n\r");
      $json[] = "\"$k\": \"$v\"";
    }
    return '{' . implode(',', $json) . '}';
  }

  protected function sanitizeData() {
    if (function_exists('filter_input_array')) {
      $_GET   = filter_input_array(INPUT_GET, FILTER_SANITIZE_STRING);
      $_POST  = filter_input_array(INPUT_POST, FILTER_SANITIZE_STRING);
      // $_REQUEST = (array)$_POST + (array)$_GET + (array)$_REQUEST;
    }
  }
  protected function sanitizeParam($param) {
    if (function_exists('filter_var'))
      return filter_var($param, FILTER_SANITIZE_STRING);
    else
      return $param;
  }
  protected function sanitizeURL($url) {
    if (function_exists('filter_var'))
        return filter_var($url, FILTER_SANITIZE_URL);
    else
        return $url;
  }
  protected function isLocalRequest() {
    return $_SERVER['SERVER_ADDR'] == $_SERVER['REMOTE_ADDR']; 
  }

  protected function doGet() { }
  protected function doPost() { }

  protected function doAjaxGet() { }
  protected function doAjaxPost() { }
}

?>
