"use strict";
$(function () {
  var netmaskList = [
    //{{{1
    '0.0.0.0',
    '128.0.0.0',
    '192.0.0.0',
    '224.0.0.0',
    '240.0.0.0',
    '248.0.0.0',
    '252.0.0.0',
    '254.0.0.0',
    '255.0.0.0',
    '255.128.0.0',
    '255.192.0.0',
    '255.224.0.0',
    '255.240.0.0',
    '255.248.0.0',
    '255.252.0.0',
    '255.254.0.0',
    '255.255.0.0',
    '255.255.128.0',
    '255.255.192.0',
    '255.255.224.0',
    '255.255.240.0',
    '255.255.248.0',
    '255.255.252.0',
    '255.255.254.0',
    '255.255.255.0',
    '255.255.255.128',
    '255.255.255.192',
    '255.255.255.224',
    '255.255.255.240',
    '255.255.255.248',
    '255.255.255.252',
    '255.255.255.254',
  ]; //}}}1

  var validateIP = function (ipaddr) {
    if (!ipaddr) return false;

    if (ipaddr == '0.0.0.0' || ipaddr == '255.255.255.255') return false;

    var pat = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/;
    var parts = ipaddr.match(pat);
    if (parts == null || parts.length != 5) return false;

    parts.shift();
    if (
      _.any(parts, function (p) {
        return parseInt(p) > 255;
      })
    )
      return false;

    return true;
  };
  var validateMask = function (netmask) {
    if (!validateIP(netmask)) return false;
    return _.contains(netmaskList, netmask);
  };

  var deviceRebootStatusChecker = function (
    url,
    waitTime,
    timeoutTime,
    successCallback,
    timeoutCallback
  ) {
    let reqTimeout = 5000;
    let checker = function () {
      let queryStartTime = 0;
      console.log('start to check device status ...');
      $.ajax({
        url: url,
        type: 'GET',
        dataType: 'json',
        data: { type: 'wired' },
        timeout: reqTimeout,
        beforeSend: function () {
          queryStartTime = Date.now();
        },
        success: function () {
          if (successCallback) successCallback();
        },
        error: function (jq) {
          if (jq.status == 200) {
            if (successCallback) successCallback();
            return;
          }

          console.log('device seems down, will check later');
          let timeCost = Date.now() - queryStartTime;
          timeoutTime -= timeCost;
          if (timeoutTime <= 0) {
            if (timeoutCallback) timeoutCallback();
            return;
          }
          // check again after a while
          setTimeout(checker, 3000);
          timeoutTime -= 3000;
        },
      });
    };

    return function () {
      setTimeout(checker, waitTime);
    };
  };

  var progressUIMixin = {
    //{{{1
    data: function () {
      return { spinner: null };
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },
    },
  }; //}}}1

  //define components
  Vue.component('modal-dialog', {
    //{{{1
    template: '#modal_dialog_template',
    props: {
      title: String,
      saveBtnLabel: {
        type: String,
        default: 'Save',
      },
      enterToSubmit: {
        type: Boolean,
        default: true,
      },
    },
    data: function () {
      return {};
    },
    mounted: function () {
      let self = this;
      this.$el.onshow = function () {
        $(self.$el).find('input').focus();
      };
    },
    methods: {
      onSubmit: function () {
        this.$emit('submitChange');
      },
    },
  }); //}}}1

  var networkConfigComp = {
    //{{{1
    template: '#network-config-template',
    mixins: [progressUIMixin],
    props: {
      connType: {
        type: String,
        default: 'wired',
      },
    },
    data: function () {
      return {
        ipError: false,
        netmaskError: false,
        gatewayError: false,
        dns1Error: false,
        dns2Error: false,

        currentView: 'manual',
        sections: [
          { id: 'manual', name: 'Manually' },
          { id: 'auto', name: 'Using DHCP' },
        ],

        manualSettings: {
          ip: '',
          netmask: '255.255.255.0',
          gateway: '',
          dns1: '',
          dns2: '',
          dnsValues: '',
        },
        autoSettings: {
          ip: '',
          netmask: '',
          gateway: '',
          dns1: '',
          dns2: '',
          dnsValues: '',
        },

        connUUID: null,
      };
    },
    computed: {
      ip: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.ip
            : this.autoSettings.ip;
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.ip = v;
        },
      },
      netmask: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.netmask
            : this.autoSettings.netmask;
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.netmask = v;
        },
      },
      gateway: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.gateway
            : this.autoSettings.gateway;
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.gateway = v;
        },
      },
      dns1: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.dns1
            : this.autoSettings.dns1;
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.dns1 = v;
        },
      },
      dns2: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.dns2
            : this.autoSettings.dns2;
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.dns2 = v;
        },
      },
      dnsValues: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.dnsValues
            : this.autoSettings.dnsValues;
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.dnsValues = v;
        },
      },

      cdir: function () {
        return netmaskList.indexOf(this.netmask);
      },
      method: {
        get: function () {
          return this.currentView;
        },
        set: function (v) {
          if (v == 'manual') this.currentView = 'manual';
          else if (v == 'auto') this.currentView = 'auto';
        },
      },
      connLabel: function () {
        return this.connType == 'general' ? 'Ethernet' : 'WIFI';
      },
    },
    mounted: function () {
      this.loadData(this.connUUID);
    },
    methods: {
      reportSuccess: function (msg) {
        this.$refs.alertPanel.display('Done', msg, 'success');
      },
      reportError: function (msg) {
        this.$refs.alertPanel.display('Error', msg, 'error');
      },
      loadData: function (conn_uuid) {
        var self = this;
        if (self.connType == 'wifi.network' && !conn_uuid) {
          console.log('connUUID is invalid, skip network data loading');
          return;
        }

        let data = { type: self.connType };
        if (conn_uuid) {
          this.connUUID = conn_uuid;
          data['conn_uuid'] = this.connUUID;
        }
        $.ajax({
          url: 'network_config.php',
          type: 'GET',
          dataType: 'json',
          data,
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            if (resp.data) {
              var addresses = resp.data['ipv4.addresses'];
              var parts = addresses.split('/');
              if (parts.length == 2) {
                self.$set(self.manualSettings, 'ip', parts[0]);
                var index = Number(parts[1]);
                if (index >= 0 && index < netmaskList.length)
                  self.$set(self.manualSettings, 'netmask', netmaskList[index]);
              }
              self.$set(
                self.manualSettings,
                'gateway',
                resp.data['ipv4.gateway']
              );

              var ipv4DNS = resp.data['ipv4.dns'];
              if (ipv4DNS) {
                ipv4DNS = ipv4DNS.trim();
                if (ipv4DNS.indexOf(' ') > 0) {
                  ipv4DNS = ipv4DNS.replace(/ /g, ',');
                }
                var dnsList = ipv4DNS
                  .split(',')
                  .map(function (p) {
                    return p.trim();
                  })
                  .filter(function (p) {
                    return validateIP(p);
                  });
                self.$set(self.manualSettings, 'dnsValues', dnsList.join('\n'));
                if (dnsList.length > 0)
                  self.$set(self.manualSettings, 'dns1', dnsList[0]);
                if (dnsList.length > 1)
                  self.$set(self.manualSettings, 'dns2', dnsList[1]);
              }

              var method = resp.data['ipv4.method'];
              if (method == 'auto') {
                let dhcpOptions = {};
                for (const key in resp.data) {
                  if (!key.startsWith("DHCP4.OPTION")) {
                    continue;
                  }
                  let [subkey, subvalue] = resp.data[key].split("=");
                  if (subkey && subvalue) {
                    dhcpOptions[subkey.trim()] = subvalue.trim();
                  }
                } 

                let {ip_address = "", routers = "", subnet_mask = "", domain_name_servers = ""} = dhcpOptions;
                self.$set(self.autoSettings, "ip", ip_address);
                self.$set(self.autoSettings, 'gateway', routers.split(',')[0]);
                self.$set(self.autoSettings, 'netmask', subnet_mask);
                if (domain_name_servers) {
                  domain_name_servers = domain_name_servers.replace(/,/g, "\n").replace(/ /g, "\n");
                } 
                self.$set(self.autoSettings, "dnsValues", domain_name_servers);
              }

              self.method = method;
            }
          },
          error: function (data) {
            console.warn('get error');
            console.warn(data);
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },
      validateInputs: function () {
        if (this.method == 'auto') return true;

        this.ipError = !validateIP(this.ip);
        this.netmaskError = !validateMask(this.netmask);
        this.gatewayError = !validateIP(this.gateway);

        // if (this.dns1.trim().length > 0)
        //   this.dns1Error = !validateIP(this.dns1);
        // else this.dns1Error = false;
        // if (this.dns2.trim().length > 0)
        //   this.dns2Error = !validateIP(this.dns2);
        // else this.dns2Error = false;
        this.dns1Error = false;
        this.dns2Error = false;
        if (this.dnsValues.length > 0) {
          this.dns1Error =
            this.dnsValues
              .split('\n')
              .map(function (p) {
                return p.trim();
              })
              .filter(function (p) {
                return p && !validateIP(p);
              }).length > 0;
        }
        return !(
          this.ipError ||
          this.netmaskError ||
          this.gatewayError ||
          this.dns1Error ||
          this.dns2Error
        );
      },
      saveNetworkSettings: function () {
        if (!this.validateInputs()) {
          console.warn('invalid input');
          return false;
        }

        var self = this;
        var ipv4_data = {
          method: this.method,
        };
        if (this.method == 'manual') {
          var dns = this.dnsValues
            .split('\n')
            .map(function (p) {
              return p.trim();
            })
            .filter(function (p) {
              return validateIP(p);
            })
            .join(',');

          let cdir = netmaskList.indexOf(this.manualSettings.netmask);
          _.extend(ipv4_data, {
            ip: this.manualSettings.ip,
            cdir: cdir,
            gateway: this.manualSettings.gateway,
            dns,
          });
        }

        let data = {
          type: self.connType,
          config: { ipv4: ipv4_data },
        };
        if (self.connUUID) data['conn_uuid'] = this.connUUID;

        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data,
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data) {
            if (data.redirect) redirect(data.redirect);

            var msg = 'Network settings have been saved successfully.';
            if (self.method == 'manual') msg = msg + ' reloading ...';
            self.reportSuccess(msg);
             
            // use window.parent.location to support both iframe or non-iframe modes
            let l = window.parent.location;
            let url = l.href;
            let new_url = url
            if (self.method == "manual") {
              new_url = url.replace(l.hostname, self.manualSettings.ip);
            }
            let checker = deviceRebootStatusChecker(
              new_url,
              5000,
              60 * 1000,
              function () {
                self.getSpinner().stop();
                self.reportSuccess('device is ready');
                if (l.href != new_url) l.assign(new_url);
                else l.reload();
              },
              function () {
                self.reportError('device status checking failed');
                self.getSpinner().stop();
              }
            );
            checker();
          },
          error: function () {
            self.getSpinner().stop();
            self.reportError('failed to save settings.');
          },
        });
        return true;
      },
    },
    components: {
      alert: alertComp,
    },
  }; //}}}1

  var wiredComp = {
    template: '#wired-network-template',
    data: function () {
      return {};
    },
    methods: {
      saveNetworkSettings: function () {
        this.$refs.networkConfig.saveNetworkSettings();
      },
    },
    components: {
      'network-config': networkConfigComp,
    },
  };

  var generalComp = {
    template: '#general-network-template',
    mixins: [progressUIMixin],
    data: function () {
      return {
        hostname: '',
        hostnameError: false,

        primaryNetwork: 'wifi',
      };
    },
    mounted: function () {
      this.loadData();
    },
    methods: {
      loadData: function () {
        let self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'GET',
          dataType: 'json',
          data: { type: 'general' },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            if (resp.data) {
              if (_.has(resp.data, 'hostname'))
                self.hostname = resp.data['hostname'];

              let primary = resp.data['primary_network'];
              if (primary == 'wifi' || primary == 'wired')
                self.primaryNetwork = primary;
            }
          },
          error: function (data) {
            console.warn(data);
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },
      validateInputs: function () {
        this.hostnameError =
          !this.hostname || !/^[-a-zA-Z0-9]{1,19}$/.test(this.hostname);

        return !this.hostnameError;
      },
      saveNetworkSettings: function () {
        if (!this.validateInputs()) return false;

        let self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            type: 'general',
            //GOTCHA: now the firmware will always set the default hostname after
            //reboot, when the issue has been fixed later, we can enable modifyHostName
            hostname: this.hostname,
            primary_network: this.primaryNetwork,
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data) {
            if (data.redirect) redirect(data.redirect);

            let msg = 'Network settings has been saved.';
            msgbus.$emit('alert', 'Done', msg, 'success');
          },
          error: function () {
            msgbus.$emit(
              'alert',
              'Error',
              'Failed to save network settings.',
              'error'
            );
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },
    },
  };

  const wifi_scan_interval = 30 * 1000;
  var wifiComp = {
    //{{{1;
    template: '#wifi-network-template',
    mixins: [progressUIMixin],
    data: function () {
      return {
        known_list: [],
        other_list: [],

        timer_id: null,
        auto_rescan: false,

        wifi_auth: {
          ssid: '',
          password: '',
          error: '',
          showPassword: false,
        },

        network_config_error: false,
      };
    },
    mounted: function () {
      this.loadData();
    },
    computed: {
      active_wifi: {
        get: function () {
          let index = this.known_list.findIndex((ap) => ap.isActive);
          if (index == -1) return null;
          return this.known_list[index];
        },
      },
      active_wifi_config_title: function () {
        if (!this.active_wifi || !this.active_wifi.SSID)
          return 'WIFI Network Config';

        return this.active_wifi.SSID + ' Network Config';
      },
    },
    methods: {
      signalImg: function (signal) {
        let index;
        if (signal <= 30) index = '1';
        else if (signal <= 80) index = '2';
        else index = '3';
        return `./img/wifi-strength-${index}.svg`;
      },
      isActive: function (ssid) {
        return this.active_wifi && this.active_wifi.SSID === ssid;
      },
      loadData: function () {
        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'GET',
          dataType: 'json',
          data: { type: 'wifi' },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            self.known_list = [];

            if (resp.error) {
              if (resp.error.text)
                msgbus.$emit('alert', 'Error', resp.error.text, 'error');
              return;
            }

            if (resp.data && Array.isArray(resp.data) && resp.data.length > 0) {
              let reloadNeeded = false;
              for (const ap of resp.data) {
                self.known_list.push({
                  SSID: ap['802-11-wireless.ssid'],
                  UUID: ap['uuid'],
                  isActive: ap['GENERAL.STATE'] == 'activated',
                });

                // if wifi state is under activating, load data later
                if (ap['GENERAL.STATE'] == 'activating') reloadNeeded = true;
              }
              if (reloadNeeded)
                setTimeout(function () {
                  self.loadData();
                }, 5 * 1000);
            }
          },
          error: function (data) {
            console.warn(data);
          },
          complete: function () {
            self.getSpinner().stop();

            self.scanWifi();
          },
        });
      },
      scanWifi: function () {
        console.log('scan wifi');
        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'GET',
          dataType: 'json',
          data: { type: 'wifi.scan' },
          beforeSend: function () {
            if (self.timer_id) {
              clearTimeout(self.timer_id);
              self.timer_id = null;
            }
            self.getSpinner().spin(self.$el);
          },
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            self.other_list = [];
            if (resp.error) {
              msgbus.$emit('alert', 'Error', resp.error.text, 'error');
              return;
            }

            if (!resp.data && !Array.isArray(resp.data)) return;

            let ap_list = [];
            for (const ap of resp.data) {
              let index = self.known_list.findIndex(
                (kap) => kap['SSID'] == ap['SSID']
              );

              ap['SIGNAL'] = parseInt(ap['SIGNAL']);
              if (index != -1) {
                self.$set(self.known_list[index], 'SIGNAL', ap['SIGNAL']);
                continue;
              }
              ap_list.push(ap);
            }
            ap_list.sort((a, b) => {
              return a['SIGNAL'] > b['SIGNAL'] ? -1 : 1;
            });
            self.other_list = ap_list;
          },
          error: function (data) {
            console.warn(data);
          },
          complete: function () {
            self.getSpinner().stop();

            if (!self.auto_rescan) return;
            self.timer_id = setTimeout(function () {
              self.scanWifi();
            }, wifi_scan_interval);
          },
        });
      },

      doWifiConn: function (op_type, conn_uuid) {
        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            type: op_type,
            conn_uuid: conn_uuid,
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            if (resp.error) {
              msgbus.$emit('alert', 'Error', resp.error.text, 'error');
              return;
            }

            setTimeout(function () {
              self.loadData();
            }, 500);
          },
          error: function (data) {
            console.warn(data);
          },
          complete: function () {
            self.getSpinner().stop();
          },
        });
      },
      upWifi: function (conn_uuid) {
        this.doWifiConn('wifi.up', conn_uuid);
      },
      forgetWifi: function (conn_uuid) {
        this.doWifiConn('wifi.forget', conn_uuid);
      },

      isPasswordValid: function (password) {
        return password && password.length > 0;
      },
      connectWifi: function (ssid) {
        this.$set(this.wifi_auth, 'ssid', ssid);
        this.$set(this.wifi_auth, 'password', '');
        this.$set(this.wifi_auth, 'error', '');
        this.$set(this.wifi_auth, 'showPassword', false);

        $(this.$el).find('#wifi_auth_dlg').modal('show');
        setTimeout(() => {
          $(this.$el).find("#inputPassword").focus();
        }, 300)
      },
      doConnectWifi: function () {
        var self = this;
        let auth = {
          ssid: this.wifi_auth.ssid,
          password: this.wifi_auth.password,
        };
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            type: 'wifi.connect',
            auth: auth,
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
            $(self.$el).find('#wifi_auth_dlg').modal('hide');
          },
          success: function (resp) {
            if (resp.redirect) redirect(resp.redirect);

            if (resp.error && resp.error.text) {
              msgbus.$emit('alert', 'Error', resp.error.text, 'error');
              return;
            }

            let msg = `connected to wifi ${self.wifi_auth.ssid} ...`;
            msgbus.$emit('alert', 'Done', msg, 'success');
          },
          error: function (data) {
            msgbus.$emit(
              'alert',
              'Error',
              `failed to connect to ${self.wifi_auth.ssid}`,
              'error'
            );
          },
          complete: function () {
            setTimeout(function () {
              self.getSpinner().stop();
              self.loadData();
            }, 5000);
          },
        });
      },
      disconnectWifi: function (conn_uuid) {
        this.doWifiConn('wifi.disconnect', conn_uuid);
      },

      showNetworkConfigDlg: function (conn_uuid) {
        this.network_config_error = false;
        $(this.$el).find('#wifi_network_config_dlg').modal('show');
        this.$refs.networkConfig.loadData(conn_uuid);
      },
      saveNetworkConfig: function () {
        let success = this.$refs.networkConfig.saveNetworkSettings();
        // if (success) $(this.$el).find('#wifi_network_config_dlg').modal('hide');
        this.network_config_error = !success;
      },
    },
    components: {
      'network-config': networkConfigComp,
    },
  }; //}}}1

  var fileSelectorInputMixin = {
    //{{{1
    methods: {
      changeFile: function (title, prop) {
        var inst = this.$root.$refs['file_selector_inst'];
        var oldVal = this.$data[prop];
        var self = this;
        inst.doModal(title, oldVal, function (newVal) {
          self.$set(self.$data, prop, newVal);
        });
      },
    },
  }; //}}}1

  var md5Comp = {
    //{{{1
    template: '#security-802_1x-md5-template',
    props: ['enabled', 'initialSettings'],
    data: function () {
      return {
        username: '',
        password: '',

        usernameError: false,
        passwordError: false,
      };
    },
    mounted: function () {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function () {
          return {
            identity: this.username,
            password: this.password,
          };
        },
        set: function (newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.username = newVal['802-1x.identity'];
          else this.username = '';

          if (_.has(newVal, '802-1x.password'))
            this.password = newVal['802-1x.password'];
          else this.password = '';
        },
      },
    },
    methods: {
      validate: function () {
        this.usernameError = this.username.length <= 0;
        //TODO: check if password is secure enough
        this.passwordError = this.password.length < 8;
        return !this.usernameError && !this.passwordError;
      },
    },
  }; //}}}1

  var tlsComp = {
    //{{{1
    mixins: [fileSelectorInputMixin],
    template: '#security-802_1x-tls-template',
    props: ['enabled', 'initialSettings'],
    data: function () {
      return {
        identity: '',
        cert: '',
        caCert: '',
        key: '',
        keyPassword: '',

        identityError: false,
        certError: false,
        caCertError: false,
        keyError: false,
        keyPasswordError: false,
      };
    },
    mounted: function () {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function () {
          return {
            identity: this.identity,
            'client-cert': this.cert,
            'ca-cert': this.caCert,
            'private-key': this.key,
            'private-key-password': this.keyPassword,
          };
        },
        set: function (newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.identity = newVal['802-1x.identity'];
          else this.identity = '';

          if (_.has(newVal, '802-1x.client-cert'))
            this.cert = newVal['802-1x.client-cert'];
          else this.cert = '';

          if (_.has(newVal, '802-1x.ca-cert'))
            this.caCert = newVal['802-1x.ca-cert'];
          else this.caCert = '';

          if (_.has(newVal, '802-1x.private-key'))
            this.key = newVal['802-1x.private-key'];
          else this.key = '';

          if (_.has(newVal, '802-1x.private-key-password'))
            this.keyPassword = newVal['802-1x.private-key-password'];
          else this.keyPassword = '';
        },
      },
    },
    methods: {
      validate: function () {
        this.identityError = this.identity.length <= 0;
        this.certError = this.cert.length <= 0;
        // this.caCertError = this.caCert.length <= 0;
        this.keyError = this.key.length <= 0;
        this.keyPasswordError = this.keyPassword.length <= 8;

        return _.any([
          this.identityError,
          this.certError,
          /*this.caCertError,*/ this.keyError,
          this.keyPassword,
        ]);
      },
    },
  }; //}}}1

  var ttlsComp = {
    //{{{1
    mixins: [fileSelectorInputMixin],
    template: '#security-802_1x-ttls-template',
    props: ['enabled', 'initialSettings'],
    data: function () {
      return {
        anonymousIdentity: '',
        anonymousIdentityError: false,

        caCert: '',
        caCertError: false,

        identity: '',
        identityError: false,

        password: '',
        passwordError: false,

        phase2AuthEap: 'pap',
        phase2AuthEapList: ['PAP', 'MSCHAP', 'MSCHAPv2', 'CHAP', 'MD5', 'GTC'],
      };
    },
    mounted: function () {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function () {
          let conf = {
            'anonymous-identity': this.anonymousIdentity,
            'ca-cert': this.caCert,
            identity: this.identity,
            password: this.password,
          };

          if (_.contains(['pap', 'mschap', 'chap'], this.phase2AuthEap))
            conf['phase2-auth'] = this.phase2AuthEap;
          else conf['phase2-autheap'] = this.phase2AuthEap;
          return conf;
        },
        set: function (newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.identity = newVal['802-1x.identity'];
          else this.identity = '';

          if (_.has(newVal, '802-1x.password'))
            this.password = newVal['802-1x.password'];
          else this.password = '';

          if (_.has(newVal, '802-1x.ca-cert'))
            this.caCert = newVal['802-1x.ca-cert'];
          else this.caCert = '';

          if (_.has(newVal, '802-1x.anonymous-identity'))
            this.anonymousIdentity = newVal['802-1x.anonymous-identity'];
          else this.anonymousIdentity = '';

          var phase2Auth = '';
          if (_.has(newVal, '802-1x.phase2-autheap'))
            phase2Auth = newVal['802-1x.phase2-autheap'];
          if (phase2Auth.length == 0 && _.has(newVal, '802-1x.phase2-auth'))
            phase2Auth = newVal['802-1x.phase2-auth'];

          if (phase2Auth.length == 0) phase2Auth = 'pap';
          this.phase2AuthEap = phase2Auth;
        },
      },
    },
    methods: {
      validate: function () {
        this.anonymousIdentityError = this.anonymousIdentity.length <= 0;
        this.caCertError = this.caCert.length <= 0;
        this.identityError = this.identity.length <= 0;
        this.passwordError = this.password.length <= 8;

        return _.any([
          this.identityError,
          this.caCertError,
          this.passwordError,
          this.anonymousIdentityError,
        ]);
      },
    },
  }; //}}}1

  var peapComp = { //{{{1
    mixins: [fileSelectorInputMixin],
    template: '#security-802_1x-peap-template',
    props: ['enabled', 'initialSettings'],
    data: function() {
      return {
        anonymousIdentity: '',
        anonymousIdentityError: false,

        caCert: '',
        caCertError: false,

        identity: '',
        identityError: false,

        password: '',
        passwordError: false,

        phase2Auth: 'mschapv2',
        phase2AuthList: ['MSCHAPv2', 'MD5', 'GTC'],
      };
    },
    mounted: function() {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function() {
          return {
            'anonymous-identity': this.anonymousIdentity,
            'ca-cert': this.caCert,
            'identity': this.identity,
            'password': this.password,
            'phase2-auth': this.phase2Auth,
          };
        },
        set: function(newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.identity = newVal['802-1x.identity'];
          else
            this.identity = '';

          if (_.has(newVal, '802-1x.password'))
            this.password = newVal['802-1x.password'];
          else
            this.password = '';

          if (_.has(newVal, '802-1x.ca-cert'))
            this.caCert = newVal['802-1x.ca-cert'];
          else
            this.caCert = '';

          if (_.has(newVal, '802-1x.anonymous-identity'))
            this.anonymousIdentity = newVal['802-1x.anonymous-identity'];
          else
            this.anonymousIdentity = '';
        
          if (_.has(newVal, '802-1x.phase2-auth'))
            this.phase2Auth = newVal['802-1x.phase2-auth'];
          else
            this.phase2Auth = 'mschapv2';
        },
      },
    },
    methods: {
      validate: function() {
        this.anonymousIdentityError = this.anonymousIdentity.length <= 0;
        this.caCertError = this.caCert.length <= 0;
        this.identityError = this.identity.length <= 0;
        this.passwordError = this.password.length <= 8;

        return _.any([this.identityError, this.caCertError, this.passwordError, this.anonymousIdentityError]);
      },
    },
  }; //}}}1

  var securityComp = { //{{{1
    template: '#security-802_1x-template',
    mixins: [progressUIMixin],
    data: function() {
      return {
        enabled: false,
        disabled: true,
        currentView: 'md5',
        sections: [
          {id: 'md5', name: 'MD5'},
          {id: 'tls', name: 'TLS'},
          {id: 'ttls', name: 'Tunneled TLS'},
          {id: 'peap', name: 'Protected EAP(PEAP)'}
        ],
        initialSettings: {},
      };
    },
    mounted: function() {
      this.loadSecuritySettings();
    },
    methods: {
      loadSecuritySettings: function() {
        //TODO: load security settings from server
        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'GET',
          dataType: 'json',
          data: { type: '802.1x' },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.data) {
              self.initialSettings = resp.data;
              if (_.has(self.initialSettings, '802-1x.eap')) {
                if (self.initialSettings['802-1x.eap'].length > 0) {
                  self.currentView = self.initialSettings['802-1x.eap'];
                }
              }
              self.enabled = self.initialSettings['802-1x.enabled'];
              self.disabled = self.initialSettings['802-1x.readOnly'] == true;
              self.$refs['sub_config_panel'].configData = self.initialSettings;
            } else {
              self.enabled = false;
              self.disabled = true;
            }
          },
          error: function(data, status) {
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          },
        });
      },
      saveSecuritySettings: function() {
        var panel = this.$refs['sub_config_panel'];
        if (!panel) {
          console.warn('can not find current panel');
          return;
        }

        // if (this.enabled && !panel.validate())
        //   return

        var data = { type: '802.1x' };
        if (this.enabled)
          _.extend(data, {
            eap: this.currentView,
            config: panel.configData,
          });
        else
          _.extend(data, { eap: '' });

        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: data,
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data) {
            if (data.redirect)
              redirect(data.redirect);

            var msg = "security settings have been saved successfully.";
            msgbus.$emit("alert", "Done", msg, 'success'); 

            if (window.iot_platform == 'FW') 
              _.delay(function() { window.history.back(); }, 1500);
          },
          error: function(data, status) {
            msgbus.$emit("alert", "Error", "failed to save security settings.", 'error'); 
          },
          complete: function(data, status) {
            self.getSpinner().stop();
          },
        });
      },
    },
    components: {
      'md5':  md5Comp,
      'tls':  tlsComp,
      'ttls': ttlsComp,
      'peap': peapComp,
    },
  }; //}}}1

  var ipv6Comp = { //{{{1
    template: '#ipv6-network-template',
    data: function() {
      return {
        ipError: false,
        subnetPrefixLenError: false,
        gatewayError: false,
        dns1Error: false,
        dns2Error: false,

        manualSettings: {
          ip: '',
          subnetPrefixLen: 64,
          gateway: '',
          dns1: '',
          dns2: '',
          dnsValues: '',
        },
        autoSettings: {
          ipList: [],
          gateway: '',
          dnsList: [],
        },

        currentView: 'manual',
        sections: [
          {id: 'manual', name: 'Manual'},
          {id: 'auto', name: 'Automatic'},
        ],

        spinner: null,
      };
    },
    mounted: function() {
      var self = this;
      $.ajax({
        url: 'network_config.php',
        type: 'GET',
        dataType: 'json',
        data: { type: 'ipv6' },
        beforeSend: function() {
          self.getSpinner().spin(self.$el);
        },
        success: function(resp) {
          if (resp.redirect)
            redirect(resp.redirect);

          if (resp.data) {
            var addresses = _.compact(resp.data['ipv6.addresses'].split(','));
            if (addresses.length >= 1) {
              var parts = addresses[0].split('/');
              if (parts.length == 2) {
                if (self.validateIP(parts[0]))
                  self.manualSettings.ip = parts[0];

                var prefixLen = Number(parts[1]);
                if (prefixLen >= 1 && prefixLen <= 128)
                  self.manualSettings.subnetPrefixLen = prefixLen;
              }
            }
            if (_.has(resp.data, 'ipv6.gateway'))
              self.manualSettings.gateway = resp.data['ipv6.gateway'];

            var ipv6DNS = resp.data['ipv6.dns'];
            if (ipv6DNS) {
              ipv6DNS = ipv6DNS.trim();
              if (ipv6DNS.indexOf(' ') > 0) {
                ipv6DNS = ipv6DNS.replace(/ /g, ',');
              }
              var dnsList = ipv6DNS
                .split(',')
                .map(function (p) {
                  return p.trim();
                })
                .filter(function (p) {
                  return self.validateIP(p);
                });
              self.$set(self.manualSettings, 'dnsValues', dnsList.join('\n'));
              if (dnsList.length > 0 && self.validateIP(dnsList[0]))
                self.manualSettings.dns1 = dnsList[0];
              if (dnsList.length > 1 && self.validateIP(dnsList[1]))
                self.manualSettings.dns2 = dnsList[1];
            }

            var method = resp.data['ipv6.method'];
            if (method == 'auto') {
              self.autoSettings.gateway = resp.data['IP6.GATEWAY'];
              _.each(
                _.select(_.keys(resp.data), function (k) {
                  return k.startsWith('IP6.ADDRESS');
                }),
                function (elem, index) {
                  self.autoSettings.ipList.push(resp.data[elem]);
                }
              );
              _.each(
                _.select(_.keys(resp.data), function (k) {
                  return k.startsWith('IP6.DNS');
                }),
                function (elem, index) {
                  self.autoSettings.dnsList.push(resp.data[elem]);
                }
              );
            }

            self.method = method;
          }
        },
        error: function (data, status) {
          console.warn('get error');
          console.warn(data);
        },
        complete: function () {
          self.getSpinner().stop();
        },
      });
    },
    computed: {
      ip: {
        get: function () {
          return this.currentView == 'manual' ? this.manualSettings.ip : '';
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.ip = v;
        },
      },
      subnetPrefixLen: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.subnetPrefixLen
            : '';
        },
        set: function (v) {
          if (this.currentView == 'manual')
            this.manualSettings.subnetPrefixLen = v;
        },
      },
      gateway: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.gateway
            : '';
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.gateway = v;
        },
      },
      dns1: {
        get: function () {
          return this.currentView == 'manual' ? this.manualSettings.dns1 : '';
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.dns1 = v;
        },
      },
      dns2: {
        get: function () {
          return this.currentView == 'manual' ? this.manualSettings.dns2 : '';
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.dns2 = v;
        },
      },
      dnsValues: {
        get: function () {
          return this.currentView == 'manual'
            ? this.manualSettings.dnsValues
            : '';
        },
        set: function (v) {
          if (this.currentView == 'manual') this.manualSettings.dnsValues = v;
        },
      },

      method: {
        get: function () {
          return this.currentView;
        },
        set: function (v) {
          if (v == 'manual') this.currentView = 'manual';
          else if (v == 'auto') this.currentView = 'auto';
        },
      },
    },
    methods: {
      getSpinner: function () {
        if (!this.spinner) this.spinner = new Spinner();
        return this.spinner;
      },

      validateIP: function (ipaddr) {
        if (!ipaddr) return false;

        var pat = /^[0-9a-f]{0,4}$/;
        var parts = ipaddr.split(':');
        if (parts.length < 3 || parts.length > 8) return false;

        return _.all(parts, function (part) {
          return pat.test(part);
        });
      },
      validateInputs: function () {
        this.ipError = !this.validateIP(this.ip);
        this.subnetPrefixLenError =
          this.subnetPrefixLen > 128 || this.subnetPrefixLen < 1;
        this.gatewayError = !this.validateIP(this.gateway);
        // if (this.dns1.trim().length > 0)
        //   this.dns1Error = !this.validateIP(this.dns1);
        // else this.dns1Error = false;
        // if (this.dns2.trim().length > 0)
        //   this.dns2Error = !this.validateIP(this.dns2);
        // else this.dns2Error = false;

        this.dns1Error = false;
        this.dns2Error = false;
        var self = this;
        if (this.dnsValues.length > 0) {
          this.dns1Error =
            this.dnsValues
              .split('\n')
              .map(function (p) {
                return p.trim();
              })
              .filter(function (p) {
                return p && !self.validateIP(p);
              }).length > 0;
        }
        return !(
          this.ipError ||
          this.subnetPrefixLenError ||
          this.gatewayError ||
          this.dns1Error ||
          this.dns2Error
        );
      },

      saveIPv6NetworkSettings: function () {
        if (this.method == 'manual' && !this.validateInputs()) {
          console.warn('invalid input');
          return;
        }

        var self = this;
        var ipv6_data = {
          method: this.method,
        };
        if (this.method == 'manual') {
          var dns = this.dnsValues
            .split('\n')
            .map(function (p) {
              return p.trim();
            })
            .filter(function (p) {
              return self.validateIP(p);
            })
            .join(',');

          _.extend(ipv6_data, {
            ip: this.ip,
            subnetPrefixLen: this.subnetPrefixLen,
            gateway: this.gateway,
            dns,
          });
        }
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            type: 'ipv6',
            config: {
              ipv6: ipv6_data,
            },
          },
          beforeSend: function () {
            self.getSpinner().spin(self.$el);
          },
          success: function (data) {
            if (data.redirect) redirect(data.redirect);

            if (data.error) {
              msgbus.$emit('alert', 'Error', data.error.text, 'error');
              return;
            }

            var msg =
              'settings have been saved successfully. controller is under restart, please reload page later.';
            //stop trying to redirect to new IPv6 address, since interface param is missing, it will not work
            //if (self.method == 'manual')
            //  msg = msg + " redirecting to new address ...";
            msgbus.$emit('alert', 'Done', msg, 'success', 10000);
            /*
            if (self.method == 'manual')
              _.delay(function() {
                var l = window.location;
                var url = l.href;
                window.location.assign(url.replace(l.hostname, '['+self.ip+']'));
              }, 10000);
              */
          },
          error: function (data, status) {
            msgbus.$emit('alert', 'Error', 'failed to save settings.', 'error');
          },
          complete: function (data, status) {
            self.getSpinner().stop();
          },
        });
      },
    },
  }; //}}}1

  //create&init root object
  var app = new Vue({
    //{{{1
    el: '#network_settings_content',
    data: function () {
      let sections = [];
      if (window.iot_platform == 'FS') {
        sections = [
          { sec_id: 'general', name: 'General' },
          { sec_id: 'wired', name: 'Wired' },
          { sec_id: 'security', name: '802.1x Security' },
          { sec_id: 'ipv6', name: 'IPv6' },
        ];
        if (window.wifi_supported)
          sections.splice(2, 0, { sec_id: 'wifi', name: 'WIFI' });
      } else if (window.iot_platform == 'FW')
        sections = [{ sec_id: 'security', name: '802.1x Security' }];
      else if (window.iot_platform == 'FI')
        sections = [
          { sec_id: 'general', name: 'General' },
          { sec_id: 'wired', name: 'Wired' },
        ];

      if (sections.length == 0) return {};

      let anchor = window.location.hash.substring(1);
      let sec = sections.find(function (s) {
        return s.sec_id == anchor;
      });
      let active_sec_id = sec && sec.sec_id ? sec.sec_id : sections[0].sec_id;
      return {
        sections: sections,
        active_sec_id,
      };
    },
    methods: {
      isSectionActive: function (sec) {
        return sec.sec_id == this.active_sec_id;
      },
    },
    components: {
      general: generalComp,
      security: securityComp,
      ipv6: ipv6Comp,
      alert: alertComp,
      'file-selector': fileSelectorComp,
      wired: wiredComp,
      wifi: wifiComp,
    },
  }); //}}}1
});

// vim: foldmethod=marker
