<?php

set_include_path(get_include_path() . PATH_SEPARATOR . '../../app');
include_once "base_controller.php";

class DiagnoseUtility extends BaseController {
  protected function signinRequired() {
    return true;
  }
  protected function adminRequired() {
    return true;
  }

  protected function featureId() {
    return basename(__DIR__);
  }

  public function interfaces() {
    return explode("\n", `ip link ls up | sed -n '/^[0-9]\+/p' | cut -d: -f2 | sed 's/ //g' | sort`);
  }

  public function doAjaxGet() {
    $response = array();
    if (!isset($_GET['action']))
      $this->renderAjaxError($response, "'action' parameter is missing");
    
    $action = $_GET['action'];
    if ($action == 'listCaptures')
      $this->listCaptures();
    else
      $this->renderAjaxError($response, "'action' parameter is invalid");
  }

  public function doListCaptures() {
    $files = array();
    foreach(glob("/mnt/logs/trace_*.pcap*") as $file) {
      if(!is_file($file)) 
        continue;

      $files[] = array(
        'name'=> basename($file), 
        'size' => filesize($file), 
        'mtime' => filemtime($file)
      );
    }
    return $files;
  }

  public function listCaptures() {
    $response['captures'] = $this->doListCaptures();
    $this->renderAjaxSuccess2($response);
  }

  public function doAjaxPost() {
    $response = array();
    $action = null;
    if (isset($_POST['action']))
      $action = $_POST['action']; 

    if ($action == "startCapture")
      $this->startCapture();
    else if ($action == "deleteCaptures")
      $this->deleteCaptures();
    else
      $this->renderAjaxError($response, "invalid action: " . $action);
  }

  protected function startCapture() {
    $response = array();

    $interface = null;
    if (isset($_REQUEST['interface'])) 
      $interface = $_REQUEST['interface'];
    if (!in_array($interface, $this->interfaces()))
      $this->renderAjaxError($response, "invalid capture interface: " . $interface);

    $filter = null;
    if (isset($_REQUEST['filter'])) 
      $filter =  $_REQUEST['filter'];
    if (empty($filter) || strlen($filter) > 256)
      $this->renderAjaxError($response, "invalid capture filter: " . $filter);

    $timeLength = null;
    if (isset($_REQUEST['timeLength'])) 
      $timeLength = intval($_REQUEST['timeLength']);
    if ($timeLength == 0 || $timeLength > 1296000)
      $this->renderAjaxError($response, "invalid capture timeLength: " . $timeLength);
      
    $count = null;
    if (isset($_REQUEST['count'])) 
      $count = intval($_REQUEST['count']);
    if ($count == 0 || $count > 256)
      $this->renderAjaxError($response, "invalid capture count: " . $count);

    $cmd = "/etc/init.d/K??tcpdump stop";
    $cmd .= " ; CAPTURE_INTERFACE=" . escapeshellarg($interface);
    $cmd .= " CAPTURE_TIME_LEN=" . escapeshellarg($timeLength);
    $cmd .= " CAPTURE_COUNT=" . escapeshellarg($count);
    $cmd .= " nohup /etc/init.d/K??tcpdump start " . escapeshellarg($filter) . " >/dev/null 2>&1 &";
    `$cmd`;
    
    $this->renderAjaxSuccess2();
  }

  protected function deleteCaptures() {
    $response = array();
    if (!isset($_REQUEST['captureFiles']))
      $this->renderAjaxError($response, "invalid 'captureFiles' parameter");

    $doneList = array();
    $captureFiles = $_REQUEST['captureFiles'];
    foreach($captureFiles as $file) {
      if (unlink("/mnt/logs/" . $file))
        $doneList[] = $file;
    }
    $response['deletedCaptures'] = $doneList;
    $this->renderAjaxSuccess2($response);
  }

  protected function doGet() {
    if (!isset($_GET['action'])) 
      return;
      
    $action = $_GET['action'];
    if ($action == "download")
      $this->downloadCapture();
    else
      die(sprintf(L("Action %s is invalid."), $action));
  }

  protected function downloadCapture() {
    if (!isset($_GET['name']))
      die("No name specified");

    $name = $_GET['name'];
    $path = "/mnt/logs/$name";
    if (file_exists($path)) {
      header('Content-Description: File Transfer');
      header('Content-Type: application/octet-stream');
      header('Content-Disposition: attachment; filename="'.basename($path).'"');
      header('Expires: 0');
      header('Cache-Control: must-revalidate');
      header('Pragma: public');
      header('Content-Length: ' . filesize($path));
      readfile($path);
      exit;
    }
  }

}

$controller = new DiagnoseUtility();
$controller->run();

?>
<!DOCTYPE html>
<html>
  <head>
    <meta charset="utf-8" />
    <title>Diagnose Utility</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <link rel="stylesheet" href="<?php A('../../css/bootstrap.min.css') ?>" media="screen" />
  </head>
  <body>
    <div class="container" id="diagnose_utility_content">
      <div class="navbar">
        <div class="navbar-inner">
          <a id="home_nav_btn" class="brand" href="<?php echo $controller->utilityUrl() ?>"><?php echo $controller->utilityLabel() ?></a>
          <ul class="nav">
            <li class="divider-vertical"></li>
            <li v-for="sec in sections" :class="{active:isSectionActive(sec)}"><a href="#" @click="active_sec_id=sec.sec_id">{{ sec.name }}</a></li>
          </ul>
        </div>
      </div>

      <div class="row">
        <alert></alert>
        <keep-alive>
          <component :is="active_sec_id"></component>
        </keep-alive>
      </div>
    </div>

    <script type="text/x-template" id="alert-template">
    <div class="span12">
      <div class="alert" :class="label_class" v-show="show">
        <button type="button" class="close" @click="show=false">&times;</button>
        <strong>{{ title }}</strong> {{ msg }}
      </div>
    </div>
    </script>

    <script type="text/x-template" id="new_traffic_capture_template">
      <div class="row">
        <div class="span12">
        <form class="form-horizontal" @submit.prevent="save">
          <div class="control-group">
            <label class="control-label">Interface:</label>
            <div class="controls">
              <select v-model="interface">
<?php foreach($controller->interfaces() as $interface) {
                if (empty(trim($interface))) continue;
                echo "<option value='$interface'>$interface</option>";
}?>
              </select>
            </div>   
          </div>
          <div class="control-group" :class="{error:filterError}">
            <label class="control-label">Filter:</label>
            <div class="controls">
              <input type="text" id="inputFilter" v-model.trim="filter" placeholder="Capture Filter">
              <span class="help-inline" v-show="filterError">Invalid Capture Filter</span>
            </div>
          </div>

          <div class="control-group" :class="{error:timeLengthError}">
            <label class="control-label">Time Length:</label>
            <div class="controls">
              <div class="input-append">
                <input type="text" id="inputTimeLength" v-model.number="timeLength" placeholder="Capture Time Length in second">
                <span class="add-on">second</span>
              </div>
              <span class="help-inline" v-show="timeLengthError">Time length must >= 0 && <= 1296000</span>
            </div>
          </div>
          <div class="control-group" :class="{error:countError}">
            <label class="control-label">Capture Count:</label>
            <div class="controls">
              <input type="text" id="inputCount" v-model.number="count" placeholder="Capture Count">
              <span class="help-inline" v-show="countError">Capture count > 0 && <= 256</span>
            </div>
          </div>

          <div class="control-group">
            <div class="controls">
              <button type="submit" class="btn btn-primary">Start Capture</button>
            </div>
          </div>
        </form>
        </div>
      </div>
    </script>

    <script type="text/x-template" id="traffic_capture_list_template">
      <div>
        <div style="margin-bottom:12px">
          <button class="btn" type="button" @click="reload">Reload</button>
          <button class="btn" type="button" @click="toggleSelectAll">Select All</button>
          <button class="btn" type="button" @click="selectInverse">Inverse Select</button>
          <button class="btn btn-danger pull-right" type="button" @click="deleteSelected">Delete</button>
        </div>
          <table class="table table-hover table-striped">
            <thead>
              <tr>
                <th></th>
                <th>Name</th>
                <th>Timestamp</th>
                <th><p class="text-right">Size</p></th>
              </tr>
            </thead>
            <tbody>
              <tr v-for="file,index in captureFiles" :key="index">
                <td><input type="checkbox" v-model="file.selected"></td>
                <td><a :href="'?action=download&name='+file.name">{{ file.name }}</a></td>
                <td>{{ formatDate(file.mtime) }}</td>
                <td><p class="text-right">{{ file.size }}</p></td>
              </tr>
            </tbody>
          </table>
      </div>
    </script>

    <script type="text/x-template" id="traffic_capture_template">
      <div class="span12">
        <div class="well well-small">
          <div class="page-header"> <h4 class="text-center">New Traffic Capture</h4> </div>
          <new-traffic-capture />
        </div>

        <div class="well well-small">
          <div class="page-header"> <h4 class="text-center">Manage Traffic Captures</h4> </div>
          <traffic-capture-list />
        </div>
      </div>
    </script>

    <script type="text/javascript" src="<?php A('../../js/jquery-1.10.0.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/bootstrap.min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/underscore-min.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/spin.min.js') ?>"></script>

<?php
  if (isset($_GET["dev"])) {
?>
    <script type="text/javascript" src="<?php A('../../js/vue.js') ?>"></script>
<?php
  } else {
?>
    <script type="text/javascript" src="<?php A('../../js/vue.min.js') ?>"></script>
<?php
  }
?>

    <script type="text/javascript" src="<?php A('../../js/plugin-utils.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/vue-alert-component.js') ?>"></script>
    <script type="text/javascript" src="<?php A('../../js/vue-modal-component.js') ?>"></script>
    <script type="text/javascript" src="<?php A('js/diagnose_utility.js') ?>"></script>

    <script language="javascript" type="text/javascript">
      "use strict";
<?php 
      $captures = $controller->doListCaptures();
      echo "window.captureFiles = " . json_encode($captures) . ";";
?>
      window.in_iframe = window !== window.parent;
      if (window.in_iframe) {
        let elem = document.getElementById("home_nav_btn");
        if(elem) elem.setAttribute("href", "#");
      }
    </script>

  </body>
</html>
