"use strict";
(function () {
  var L = function (str) {
    return l18n(str, 'dashboard/plugins/cpt/cpt.datasources.js');
  };
  var isBlank = function (str) {
    return String(str).replace(/^\s+|\s+$/g, '').length == 0;
  };
  var isLoginUrl = function (url) {
    return /\/signin\.php$|\/login$/.test(url);
  };

  var makeAdoptClient = function () {
    return {
      host: null,
      useHTTPS: true,
      token: null,
      isProxied: false,
      isFI: false,
      proxyPrefix: '/r/ws/ip/',

      socket: null,
      subList: [],
      reqSeq: 100,
      pendingReqs: {},

      onDataReceived: null,

      dataStore: [],

      clearState() {
        this.socket = null;
        this.subList = [];
        this.pendingReqs = {};
      },
      reset(host, useHTTPS, token, isProxied, isFI) {
        if (
          host == this.host &&
          useHTTPS == this.useHTTPS &&
          token == this.token &&
          isProxied == this.isProxied &&
          isFI == this.isFI
        ) {
          console.debug('[adopt reset] nothing changed, skip reset');
          return;
        }

        this.stop();

        this.host = host;
        this.useHTTPS = useHTTPS;
        this.token = token;
        this.isProxied = isProxied;
        this.isFI = isFI;
      },
      start() {
        this.getSocketForSure();
      },
      stop() {
        if (this.socket) this.socket.close(1000, 'client session done');
        this.clearState();
      },
      genURL() {
        let protocol = this.useHTTPS ? 'wss' : 'ws';
        if (this.isProxied || this.isFI) {
          return (
            protocol +
            '://' +
            window.location.host +
            this.proxyPrefix +
            this.host +
            `/adopt/?token=${this.token}`
          );
        } else {
          return protocol + '://' + this.host + `/adopt/?token=${this.token}`;
        }
      },
      retryPendings(tryNow) {
        if (!this.socket) return;

        let retryInterval = tryNow === true ? 0 : 5000;

        //resend timedout messages
        let now = Date.now();
        for (let sid in this.pendingReqs) {
          let req = this.pendingReqs[sid];
          if (now - req.time < retryInterval) continue;

          if (req.retries >= 3) {
            console.log(
              '[adopt housekeeper] give up retry for request: ' +
                JSON.stringify(req.data)
            );
            delete this.pendingReqs[sid];
          } else {
            ++req.retries;
            req.time = now;
            this.socket.send(JSON.stringify(req.data));
            console.log(
              `[adopt housekeeper] retry ${
                req.retries
              } times on request(${JSON.stringify(req.data)})`
            );
          }
        }
      },
      houseKeeping() {
        //make sure websocket is created
        if (!this.socket) {
          let url = this.genURL();
          console.debug(`[adopt connect] connecting to ${url} ...`);
          this.socket = new WebSocket(url, 'ADOPT.easyio.com');

          //set up event handlers
          this.socket.onopen = (e) => {
            console.log('[adopt open] connection established');

            //try pending requests
            this.retryPendings(true);
          };
          this.socket.onmessage = (event) => {
            //console.log(`[adopt message] data received: ${event.data}`)
            let msg = JSON.parse(event.data);
            if (msg.id === undefined) {
              console.warn(
                `[adopt onmessage] got invalid msg(missing id): ${event.data}`
              );
              return;
            }

            if (msg.id == 0) {
              //CoV
              this.handleCoV(msg);
            } else {
              if (msg.code === undefined) {
                console.warn(
                  `[adopt onmessage] got invalid msg(missing code): ${event.data}`
                );
                return;
              }

              let lastReq = this.pendingReqs[msg.id];
              if (!lastReq)
                console.warn(
                  `[adopt onmessage] cann't find ${msg.id} in pendingReq`
                );
              else if (msg.code != 200) {
                // handle error response
                if ('data' in lastReq) {
                  let lastData = lastReq.data;
                  console.warn(
                    `[adopt onmessage] previous request(${lastData}) failed: ${event.data}`
                  );
                } else {
                  console.error(
                    `[adopt onmessage] cann't find 'data' in pendingReq`
                  );
                }
              } else {
                this.handleResponse(msg, lastReq);
              }

              delete this.pendingReqs[msg.id];
            }
          };
          this.socket.onclose = (event) => {
            if (event.wasClean) {
              console.log(
                `[adopt close] connection closed cleanly, code=${event.code} reason=${event.reason}`
              );
            } else {
              console.warn(`[adopt close] connection died`);
            }
            this.clearState();
          };
          this.socket.onerror = (error) => {
            console.warn(`[adopt error] ${error.message}`);
          };
        } else this.retryPendings();
      },

      //check if path matches path wildcard pattern
      //following patterns are supported:
      // 1. /EasyIO/*
      // 2. /EasyIO/*.out
      // 3. /EasyIO/**
      // 4. /EasyIO/**.out
      matchPath(pattern, path) {
        if (pattern.includes('*')) {
          //remove possible slot from the end
          let index = pattern.indexOf('.');
          if (index != -1) pattern = pattern.slice(0, index);

          let patParts = pattern.split('/');
          let pathParts = path.split('/');

          //if path is shorter than pattern, then not match
          if (pathParts.length < patParts.length) return false;

          let matched = patParts.every((patPart, index) => {
            if (patPart == '*') return true;
            else if (patPart == '**') {
              return true;
            } else {
              return pathParts[index] == patPart;
            }
          });

          if (!matched) return false;

          if (pathParts.length > patParts.length) {
            matched = patParts[patParts.length - 1] == '**';
          }
          return matched;
        } else return pattern == path;
      },
      mergeData(deltaData) {
        deltaData.forEach((data) => {
          let index = this.dataStore.findIndex((d) => d.path == data.path);
          if (index == -1) {
            this.dataStore.push(data);
            return;
          }

          //merge slots
          if ('slots' in this.dataStore[index]) {
            if ('slots' in data) {
              data.slots.forEach((slot) => {
                let sIndex = this.dataStore[index].slots.findIndex(
                  (s) => s.name == slot.name
                );
                if (sIndex == -1) this.dataStore[index].slots.push(slot);
                else this.dataStore[index].slots.splice(sIndex, 1, slot);
              });
            }
          } else this.dataStore.splice(index, 1, data);
        });
      },
      handleCoV(msg) {
        let cov = msg.data;
        if (typeof msg.data == 'object') {
          if (this.onDataReceived !== null) {
            let deltaData = this.convertToDSFormat(cov);
            this.mergeData(deltaData);
            this.onDataReceived(JSON.parse(JSON.stringify(this.dataStore)));
          }
        } else {
          console.warn(
            `[adopt CoV] invalid CoV event data: ${JSON.stringify(msg)}`
          );
        }
      },
      handleResponse(msg, req) {
        switch (req.data.cmd) {
          case 'subscribe':
            req.data.data.paths.forEach((path) => {
              if (this.subList.includes(path)) return;
              this.subList.push(path);
            });
            if (typeof msg.data == 'object' && this.onDataReceived !== null) {
              let deltaData = this.convertToDSFormat(msg.data);
              this.mergeData(deltaData);
              this.onDataReceived(JSON.parse(JSON.stringify(this.dataStore)));
            } else {
              //subscribe message will return the initial data
              //req.data.data.paths.forEach( path => this.readComp(path) );
            }
            break;
          case 'unsubscribe':
            req.data.data.paths.forEach((path) => {
              if (this.subList.includes(path)) {
                this.subList.splice(this.subList.indexOf(path), 1);
              }

              let modified = false;
              for (let i = this.dataStore.length - 1; i >= 0; --i) {
                //GOTCHA: if user subscribe following objects first:
                //          1. /EasyIO/Add2
                //          2. /EasyIO/**
                //        then he unsubscribe "/EasyIO/**", "/EasyIO/Add2" will
                //        be unsubscribed since it is covered by "/EasyIO/**",
                //        there will be no "/EasyIO/Add2" CoV until the page is
                //        refreshed.
                if (!this.matchPath(path, this.dataStore[i].path)) continue;
                this.dataStore.splice(i, 1);
                modified = true;
              }
              if (modified && this.onDataReceived != null)
                this.onDataReceived(JSON.parse(JSON.stringify(this.dataStore)));
            });
            break;
          case 'readComp':
            if (typeof msg.data == 'object' && this.onDataReceived !== null) {
              let deltaData = this.convertToDSFormat(msg.data);
              this.mergeData(deltaData);
              this.onDataReceived(JSON.parse(JSON.stringify(this.dataStore)));
            }
            break;
          default:
            break;
        }
      },
      getSocketForSure() {
        this.houseKeeping();
        return this.socket;
      },
      send(sock, data) {
        this.pendingReqs[data.id] = {
          time: Date.now(),
          retries: 0,
          data: data,
        };
        if (this.reqSeq >= 65535) this.reqSeq = 100;

        if (sock && sock.readyState == WebSocket.OPEN) {
          sock.send(JSON.stringify(data));
        }
      },

      // commands
      readComp(path) {
        let sock = this.getSocketForSure();
        let data = {
          id: this.reqSeq++,
          cmd: 'readComp',
          data: {
            path: path,
          },
        };
        this.send(sock, data);
      },
      subscribe(pathList) {
        let sock = this.getSocketForSure();

        //do unsubscribe if possible
        let unsubList = this.subList.filter((path) => !pathList.includes(path));
        if (unsubList.length > 0) this.unsubscribe(unsubList);

        //prevent to subscribe the same path multiple times
        let paths = pathList
          .filter((path) => !this.subList.includes(path))
          .filter((path) => {
            for (let id in this.pendingReqs) {
              let req = this.pendingReqs[id].data;
              if (req.cmd == 'subscribe' && req.data.paths.includes(path))
                return false;
            }
            return true;
          });
        if (paths.length == 0) return;

        let data = {
          id: this.reqSeq++,
          cmd: 'subscribe',
          data: {
            paths: paths,
            what: 0x04 | 0x2,
            returnData: true,
          },
        };
        this.send(sock, data);
      },
      unsubscribe(pathList) {
        let sock = this.getSocketForSure();
        let paths = pathList
          .filter((path) => this.subList.includes(path))
          .filter((path) => {
            for (let id in this.pendingReqs) {
              let req = this.pendingReqs[id].data;
              if (req.cmd == 'unsubscribe' && req.data.paths.includes(path))
                return false;
            }
            return true;
          });
        if (paths.length == 0) return;

        let data = {
          id: this.reqSeq++,
          cmd: 'unsubscribe',
          data: {
            paths: paths,
            what: 0x04 | 0x2,
          },
        };
        this.send(sock, data);
      },

      parseDataType(val) {
        let type = null;
        switch (typeof val) {
          case 'number':
            if (Number.isInteger(val)) type = 'int';
            else type = 'float';
            break;
          case 'boolean':
            type = 'bool';
            break;
          case 'string':
            type = 'Str';
            break;
          case 'object':
            if (_.isNull(val)) type = 'bool';
            break;
          default:
            break;
        }
        return type;
      },
      short2LongType(shortName) {
        let short2long = {
          b: 'bool',
          c: 'byte',
          s: 'short',
          i: 'int',
          l: 'long',
          f: 'float',
          d: 'double',
          S: 'Str',
          B: 'Buf',
        };

        if (shortName in short2long) return short2long[shortName];
        else return shortName;
      },
      convertToDSFormat(compData) {
        if (!_.isArray(compData)) compData = [compData];

        //convert slots data
        compData
          .filter((comp) => comp.props != undefined)
          .forEach((comp) => {
            comp['slots'] = [];
            for (let prop in comp.props) {
              if (['meta', 'uuid'].includes(prop)) continue;
              let val = comp.props[prop];
              let type = this.parseDataType(val);
              if (type == null) continue;
              comp.slots.push({
                name: prop,
                slotType: 'property',
                type: type,
                value: val,
              });
            }
            delete comp.props;
          });

        compData
          .filter((comp) => comp.actions != undefined)
          .forEach((comp) => {
            for (let action in comp.actions) {
              let val = comp.actions[action];
              let type = _.isNull(val) ? 'void' : this.short2LongType(val);
              comp.slots.push({ name: action, slotType: 'action', type: type });
            }
            delete comp.actions;
          });

        //generate 'childNum'
        compData
          .filter((comp) => comp.kids != undefined)
          .forEach((comp) => {
            comp['childNum'] = _.keys(comp.kids).length;
            delete comp.kids;
          });

        return compData;
      },
    };
  };

  
  var INT_NULL = -1883242496;
  var parseIntEx = function (str) {
    if (str === INT_NULL.toString() || str == 'null' || str == 'nil') {
      return 'null';
    } else {
      return parseInt(str);
    }
  };
  //NOTE: some functions are defined in 'this.someFunc' format; some function
  //are defined in 'someFunc' format. when call the first type of function,
  //must use 'self.someFunc()'; the other type of function, just use
  //'someFunc()'.
  var cptWebServiceDatasource = function (settings, updateCallback) {
    var self = this;
    var maxUrlLength = 1000;
    var updateTimer = null;
    var currentSettings = settings;
    var reqUrlPayLoads = [];
    var jwt_token = null;
    var proxyPrefix = '/r/h/ip/';
    var adoptClient = null;

    var authViewModel = {
      username: ko.observable(''),
      password: ko.observable(''),
      error_msg: ko.observable(''),
    };

    //set breakpoint whenever 'name' property changed
    // currentSettings._name = currentSettings.name;
    // Object.defineProperty(currentSettings, "name", {
    //   get: function() {
    //     return currentSettings._name;
    //   },
    //   set: function(value) {
    //     debugger;
    //     currentSettings._name = value;
    //   }
    // });

    function controllerHostLabel() {
      if (
        !currentSettings.controllerHost ||
        currentSettings.controllerHost.length === 0
      ) {
        if (isProxied()) {
          let path = window.location.pathname;
          if (!path.startsWith(proxyPrefix)) return window.location.host;

          let parts = path.substring(proxyPrefix.length).split('/');
          if (parts.length <= 0) return window.location.host;

          return parts[0];
        } else return window.location.host;
      } else return currentSettings.controllerHost;
    }

    function commonPath(pathList) {
      if (pathList.length == 1) {
        var parts = pathList[0].split('/');
        parts.pop();
        return parts.join('/');
      }
      var commonParts = _.reduce(
        pathList,
        function (memo, path) {
          var parts = path.split('/');
          if (_.isEmpty(memo)) return parts;
          else {
            for (var i = 0; i < memo.length; ++i) {
              if (parts.length <= i) return parts;

              if (memo[i] != parts[i]) return _.first(memo, i);
            }
            return memo;
          }
        },
        []
      );
      return commonParts.join('/');
    }

    function parseValueString(item) {
      var value = item['value'];
      if (!_.isString(value)) return value;

      var slotType = item['slotType'];
      if (slotType == 'action') return null;

      var type = item['type'];
      if (type == 'double' || type == 'float') return parseFloat(value);
      else if (type == 'int') return parseIntEx(value);
      else if (type == 'byte' || type == 'short') return parseInt(value);
      else if (type == 'long') {
        // if the value is too big, use BigInt instead of parseInt
        if (value.length >= 16) return BigInt(value);
        else return parseInt(value);
      } else if (type == 'bool') {
        if (value == 'true') return true;
        else if (value == 'false') return false;
        else return null;
      } else return value;
    }

    function processSlotData(slotDatas) {
      return _.object(
        _.map(slotDatas, function (item) {
          return item['name'];
        }),
        _.map(slotDatas, function (item) {
          return parseValueString(item);
        })
      );
    }

    function processCompData(compDatas) {
      _.each(compDatas, function (compData, index) {
        if (_.isArray(compData)) {
          var objData = processCompData(compData);
          objData['path'] = commonPath(_.keys(objData));
          compDatas[index] = objData;
        } else {
          compData = _.extend(compData, processSlotData(compData.slots));
          delete compData['slots'];
        }
      });

      //convert component data list into hash indexed by 'path'
      return _.object(
        _.map(compDatas, function (compData) {
          var path = compData['path'];
          delete compData['path'];
          return path;
        }),
        compDatas
      );
    }

    function parseCompTypeData(dsName, compListData) {
      return _.reduce(
        compListData,
        function (memo, compDatas) {
          return _.extend(
            memo,
            _.object(
              _.pluck(compDatas, 'path'),
              _.map(_.pluck(compDatas, 'slots'), function (slots) {
                return _.object(_.pluck(slots, 'name'), slots);
              })
            )
          );
        },
        window.g_compTypeData[dsName]
      );
    }

    function buildCompTypeData(data) {
      if (_.isUndefined(window.g_compTypeData)) window.g_compTypeData = {};

      var name = currentSettings.name;
      if (_.isUndefined(window.g_compTypeData[name]))
        window.g_compTypeData[name] = {};

      data = _.flatten(data);
      if (_.every(data, _.isArray)) parseCompTypeData(name, data);
      else parseCompTypeData(name, [data]);
    }

    function updateRefresh() {
      if (updateTimer) {
        clearTimeout(updateTimer);
        updateTimer = null;
      }
      self.doUpdateNow();
    }

    function buildReqUrlPayLoads() {
      if (reqUrlPayLoads.length > 0) return;

      var urls = _.pluck(currentSettings.compPathList, 'compPath');
      if (_.isEmpty(urls)) return;

      for (var i = 0; i < urls.length; ++i) {
        var url = urls[i];
        if (reqUrlPayLoads.length == 0) {
          reqUrlPayLoads.push(url);
          continue;
        }

        var oneBatch = _.last(reqUrlPayLoads);
        if (oneBatch.length + url.length + 1 <= maxUrlLength) {
          oneBatch += '~' + url;
          reqUrlPayLoads[reqUrlPayLoads.length - 1] = oneBatch;
        } else reqUrlPayLoads.push(url);
      }
      return reqUrlPayLoads;
    }

    function updateJWTToken(token) {
      if (_.isUndefined(window.g_jwtTokenData)) window.g_jwtTokenData = {};

      window.g_jwtTokenData[currentSettings.name] = token;
    }

    this.scheduleNextUpdate = function () {
      if (updateTimer) {
        clearTimeout(updateTimer);
        updateTimer = null;
      }

      var refreshTime =
        reqUrlPayLoads.length > 0 ? 1 * 1000 : currentSettings.refresh * 1000;
      updateTimer = setTimeout(self.updateNow, refreshTime);
    };

    this.renewJWTToken = _.debounce(
      function () {
        var hostLabel = controllerHostLabel();
        var requestURL = '';
        if (isProxied() || isFI()) {
          if (hostLabel == document.location.host)
            requestURL = '../app/token_manager.php';
          else
            requestURL =
              proxyPrefix + hostLabel + '/sdcard/cpt/app/token_manager.php';
        } else {
          var protocol = getProtocol();
          requestURL =
            protocol + '://' + hostLabel + '/sdcard/cpt/app/token_manager.php';
        }

        var headers = { 'X-Requested-With': 'XMLHttpRequest' };
        if (jwt_token) headers['Authorization'] = 'Bearer ' + jwt_token;

        $.ajax({
          url: requestURL,
          dataType: 'json',
          type: 'POST',
          xhrFields: { withCredentials: true },
          headers: headers,
          data: { action: 'renew' },
          success: function (resp) {
            if (resp.token) {
              jwt_token = resp.token;
              updateJWTToken(jwt_token);
              self.doUpdateNow();
            } else {
              console.warn(resp);
              if (resp.redirect && isLoginUrl(resp.redirect)) {
                if (isFI())
                  console.warn(
                    'device is not provisioned or its secure key is outdated.'
                  );
                showAuthDialog();
              } else self.scheduleNextUpdate();
            }
          },
          error: function (xhr, status, error) {
            console.warn(
              'failed to renew token from url: ' +
                requestURL +
                ' status: ' +
                status +
                'error: ' +
                error
            );
            self.scheduleNextUpdate();
          },
        });
      },
      100,
      true
    );

    this.doUpdateByADOPT = function () {
      if (jwt_token == null) {
        //this case should only happen when login controller with cookie
        self.renewJWTToken();
        return;
      }

      if (adoptClient === null && jwt_token != null) {
        adoptClient = makeAdoptClient();
        adoptClient.onDataReceived = (data) => {
          data = data
            .filter(
              (comp) => Array.isArray(comp.slots) && comp.slots.length > 0
            )
            .map((comp) => {
              delete comp['cid'];
              delete comp['type'];
              return comp;
            });
          buildCompTypeData(data);
          updateCallback(processCompData(data));
        };
        adoptClient.reset(
          controllerHostLabel(),
          getProtocol() == 'https',
          jwt_token,
          isProxied(),
          isFI()
        );
        adoptClient.start();
      }

      if (!adoptClient) {
        self.scheduleNextUpdate();
        return;
      }

      var urls = _.pluck(currentSettings.compPathList, 'compPath');
      if (_.isEmpty(urls)) {
        self.scheduleNextUpdate();
        return;
      }

      adoptClient.subscribe(urls);
      self.scheduleNextUpdate();
    };

    this.doUpdateByHTTP = function () {
      if (reqUrlPayLoads.length == 0) buildReqUrlPayLoads();

      if (reqUrlPayLoads.length == 0) return;

      var payload = reqUrlPayLoads.shift();

      var hostLabel = controllerHostLabel();
      var requestURL = '';
      if (isProxied() || isFI()) {
        if (hostLabel == document.location.host)
          requestURL = '../app/data_api.php?url=/app/objects' + payload;
        else
          requestURL =
            proxyPrefix +
            hostLabel +
            '/sdcard/cpt/app/data_api.php?url=/app/objects' +
            payload;
      } else {
        var protocol = getProtocol();
        requestURL =
          protocol +
          '://' +
          hostLabel +
          '/sdcard/cpt/app/data_api.php?url=/app/objects' +
          payload;
      }

      var headers = { 'X-Requested-With': 'XMLHttpRequest' };
      if (jwt_token) headers['Authorization'] = 'Bearer ' + jwt_token;

      $.ajax({
        url: requestURL,
        dataType: 'json',
        type: 'GET',
        xhrFields: { withCredentials: true },
        headers: headers,
        success: function (data) {
          if (data.response) {
            buildCompTypeData(data.response.data);

            updateCallback(processCompData(data.response.data));

            self.scheduleNextUpdate();
          } else {
            console.warn(data);
            if (data.redirect && isLoginUrl(data.redirect)) {
              if (isFI())
                console.warn(
                  'device is not provisioned or its secure key is outdated.'
                );
              showAuthDialog();
            } else self.scheduleNextUpdate();
          }
        },
        error: function (xhr, status, error) {
          console.warn(
            'failed to get data from ' +
              requestURL +
              ' status: ' +
              status +
              'error: ' +
              error
          );
          self.scheduleNextUpdate();
        },
      });
    };

    this.doUpdateNow = function () {
      if (currentSettings.useADOPT) self.doUpdateByADOPT();
      else self.doUpdateByHTTP();
    };

    this.updateNow = _.debounce(self.doUpdateNow, 200, true);

    this.onDispose = function () {
      clearTimeout(updateTimer);
      updateTimer = null;
    };

    this.onSettingsChanged = function (newSettings) {
      currentSettings = _.defaults(newSettings, currentSettings);

      // update adoptClient
      if (adoptClient) {
        if (newSettings.useADOPT) {
          adoptClient.reset(
            controllerHostLabel(),
            getProtocol() == 'https',
            jwt_token,
            isProxied(),
            isFI()
          );
        } else {
          adoptClient.stop();
          adoptClient = null;
        }
      }

      updateRefresh();
    };

    function showAuthDialog(askPassword) {
      if (!isBlank(currentSettings.userName))
        authViewModel.username(currentSettings.userName);
      else authViewModel.username('admin');

      if (askPassword === true) authViewModel.password('');
      else if (!isBlank(currentSettings.password))
        authViewModel.password(currentSettings.password);

      if (
        isBlank(authViewModel.username()) ||
        isBlank(authViewModel.password())
      ) {
        var dom = $(
          "<div><div style='color:red; margin-top:5px; margin-bottom:5px;' data-bind='html: error_msg'></div><div style='margin-bottom:15px;'><div style='width:80px; float:left;'><label for='user_name'>UserName:</label></div><input type='text' id='user_name' data-bind='value: username' autofocus></div><div><div style='width:80px; float:left;'><label for='password'>Password:</label></div><input type='password' id='password' data-bind='value: password'></div></div>"
        );
        ko.applyBindings(authViewModel, dom[0]);
        freeboard.showDialog(
          dom,
          L('{appname} Web Service Authentication({host})').supplant({
            appname: window.appName,
            host: controllerHostLabel(),
          }),
          L('Ok'),
          L('Cancel'),
          loginCptWebService
        );
      } else loginCptWebService();
    }

    function isProxied() {
      return (
        document.location.pathname.substr(0, proxyPrefix.length) == proxyPrefix
      );
    }

    function isFI() {
      //need to make sure platformName global variable is defined in the HTML
      //container file, this is dashboard/index.php file for dashboard case
      return window.platformName === 'FI';
    }

    function getProtocol() {
      // the logics to pick protocol:
      //   * if current protocol is HTTPS, then use HTTPS
      //   * otherwise check useHTTPS option
      // access a http resource from a https web page will be blocked by
      // browser due to security concerns
      return document.location.protocol == 'https:' ||
        currentSettings.useHTTPS === true
        ? 'https'
        : 'http';
    }

    function loginCptWebService() {
      //to support the SameSite policy in browser, cross site login with cookie
      //is removed because:
      //  * cross site cookie must be secure(HTTPS)
      //  * HTTPS requires a signed certificate deployed on controller
      //this is very hard to set up on building controller since it normally
      //has no domain name, even no internet IP address.
      //
      //instead now login with JWT token is introduced, we should upgrade all
      //controller(FS/FW/FT) web servers to support JWT authentication.

      // if behind FI proxy, no need to login since we should already logged in
      if (isProxied() || isFI()) return;

      authViewModel.error_msg('');
      var username = authViewModel.username();
      var password = authViewModel.password();
      if (!username || !password) {
        authViewModel.error_msg(L('invalid username or password'));
        return true;
      }

      freeboard.showLoadingIndicator(true);

      var protocol = getProtocol();
      var url =
        protocol + '://' + controllerHostLabel() + '/sdcard/cpt/app/signin.php';

      if (currentSettings.loginWithJWT) {
        $.ajax({
          url: url,
          type: 'POST',
          dataType: 'json',
          xhrFields: { withCredentials: true },
          headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-EIO-Auth-Upgrade': 'JWT',
          },
          data: {
            'user[name]': username,
            'user[password]': password,
          },
        })
          .done(function (data, status, xhr) {
            let token = xhr.getResponseHeader('X-EIO-TOKEN');
            if (token && token.length != 0) {
              jwt_token = token;
              updateJWTToken(jwt_token);
            }

            freeboard.showLoadingIndicator(false);
            if (data.error) {
              authViewModel.error_msg(data.error.text);
              authViewModel.password('');
              showAuthDialog(true);
              return;
            } else {
              console.debug(controllerHostLabel() + ' signin done');
              _.each(
                _.uniq(
                  window.cptWebServiceLoginCallbacks[
                    currentSettings.controllerHost
                  ]
                ),
                function (callback) {
                  callback();
                }
              );
            }
          })
          .fail(function (xhr, status, error) {
            freeboard.showLoadingIndicator(false);
            var msg = `status: ${status};`;
            if (error.length != 0) msg += ` error: ${error}`;
            else
              msg +=
                " maybe the controller hasn't been upgraded to support JWT";
            console.warn(msg);
          });
      } else {
        $.ajax({
          url: url,
          type: 'GET',
          dataType: 'json',
          xhrFields: { withCredentials: true },
          headers: { 'X-Requested-With': 'XMLHttpRequest' },
          data: { 'user[name]': username },
        }).done(function (data) {
          if (!data.authToken) {
            console.error('failed to fetch authToken');
            freeboard.showLoadingIndicator(false);
            return;
          }
          var sections = data.authToken.split('_');
          if (sections.length != 2) {
            console.warn('invalid authToken: ' + data.authToken);
            freeboard.showLoadingIndicator(false);
            return;
          }
          var token1 = sections[0];
          var token2 = sections[1];
          // to calculate password hash, there are 2 steps:
          // 1. calculate SHA-1 hash for password + token1
          // 2. calculate SHA-1 hash for result of step1 + token2
          var shaObj1 = new jsSHA(password + token1, 'TEXT');
          var shaObj2 = new jsSHA(
            shaObj1.getHash('SHA-1', 'HEX') + token2,
            'TEXT'
          );
          var authHash = shaObj2.getHash('SHA-1', 'HEX');
          $.ajax({
            url: url,
            type: 'POST',
            dataType: 'json',
            xhrFields: { withCredentials: true },
            headers: { 'X-Requested-With': 'XMLHttpRequest' },
            data: {
              'user[name]': username,
              'user[authHash]': authHash,
            },
          }).done(function (data) {
            freeboard.showLoadingIndicator(false);
            if (data.error) {
              authViewModel.error_msg(data.error.text);
              authViewModel.password('');
              showAuthDialog(true);
              return;
            } else {
              console.debug('signin done');
              _.each(
                _.uniq(
                  window.cptWebServiceLoginCallbacks[
                    currentSettings.controllerHost
                  ]
                ),
                function (callback) {
                  callback();
                }
              );
            }
          });
        });
      }
      return false;
    }

    if (!window.cptWebServiceLoginFuncs) window.cptWebServiceLoginFuncs = {};
    if (!window.cptWebServiceLoginFuncs[currentSettings.controllerHost]) {
      //NOTE: if data source host is the current host, then authentication is not required
      if (
        currentSettings.controllerHost.length !== 0 &&
        currentSettings.controllerHost != window.location.host
      ) {
        // var showDlg = _.once(showAuthDialog)
        window.cptWebServiceLoginFuncs[currentSettings.controllerHost] =
          function (callback) {
            if (!window.cptWebServiceLoginCallbacks)
              window.cptWebServiceLoginCallbacks = {};
            if (
              !window.cptWebServiceLoginCallbacks[
                currentSettings.controllerHost
              ]
            )
              window.cptWebServiceLoginCallbacks[
                currentSettings.controllerHost
              ] = [];
            window.cptWebServiceLoginCallbacks[
              currentSettings.controllerHost
            ].push(callback);
            // showDlg();
          };
      }
    }

    if (window.cptWebServiceLoginFuncs[currentSettings.controllerHost])
      window.cptWebServiceLoginFuncs[currentSettings.controllerHost](
        updateRefresh
      );
    else updateRefresh();
  };

  freeboard.loadDatasourcePlugin({
    type_name: 'CPTWebService',
    display_name: L('{appname} Web Service').supplant({
      appname: window.appName,
    }),
    description: L('interface to access {appname} Web Service data').supplant({
      appname: window.appName,
    }),
    external_scripts: ['plugins/cpt/sha1.js'],
    settings: [
      {
        name: 'controllerHost',
        display_name: L('Controller Host'),
        type: 'text',
        default_value: '',
        description: L(
          'for example: 192.168.10.11, 192.168.10.11:8080. leave blank will use current host.'
        ),
      },
      {
        name: 'userName',
        display_name: L('UserName'),
        type: 'text',
        default_value: 'admin',
        description: L('username to login controller.'),
      },
      {
        name: 'password',
        display_name: L('Password'),
        type: 'password',
        default_value: '',
        description: L('password for above user'),
      },
      {
        name: 'useHTTPS',
        display_name: L('Use HTTPS'),
        type: 'boolean',
        default_value: '',
        description: L(
          'use HTTPS protocol to communicate with this controller. when page URL is HTTPS, then always use HTTPS'
        ),
      },
      {
        name: 'loginWithJWT',
        display_name: L('Login By JWT'),
        type: 'boolean',
        default_value: true,
        description: L(
          'Login with JWT token, it should be enabled at most of time, unless to connect to old versions for backward compatible.'
        ),
      },
      // {
      //   name: "useADOPT",
      //   display_name: L("Use ADOPT Protocol"),
      //   type: "boolean",
      //   default_value: '',
      //   description: L("Use ADOPT protocol that supports CoV event for more efficient data update. The controller MUST have ADOPT protocol enabled.")
      // },
      {
        name: 'compPathList',
        display_name: L('Component Path List'),
        type: 'array',
        description: L(
          "for example: /Folder/Ramp ; /Folder/* (all data under '/Folder')"
        ),
        settings: [
          {
            name: 'compPath',
            display_name: L('Component Path'),
            type: 'text',
          },
        ],
      },
      {
        name: 'refresh',
        display_name: L('Refresh Every'),
        type: 'number',
        suffix: L('seconds'),
        default_value: 5,
      },
    ],
    newInstance: function (settings, newInstanceCallback, updateCallback) {
      newInstanceCallback(
        new cptWebServiceDatasource(settings, updateCallback)
      );
    },
  });
})();
// vim: ts=2 sw=2 softtabstop=1
