<?php
// vim: ts=2 sw=2

include_once "../app/db.php";
include_once "../app/base_controller.php";
include_once "../app/feature_control.php";
include_once "js_asset_collector.php";

class DashboardController extends BaseController {
  protected function signinRequired() {
    return true;
  }

  public function doGet() {
    $u = $this->curUser();
    if (!$u->isDashboardEnabled())
      die("permission denied");

    if (isset($_GET['dashboard_id'])) {
      $dashboardId = $_GET['dashboard_id'];
      $perm = DashboardPermission::findByUserAndDashboard($this->curUserId(), $dashboardId);
      if (!empty($perm) && $perm[0]['canViewDashboard'] != 't')
        $this->redirect($this->makeUrl('dashboard/index.php'));
    }
  }

  public function curDashboard() {
    if (isset($_GET['dashboard_id'])) {
      $dashboardId = $_GET['dashboard_id'];
      $dashboard = new Dashboard();
      $dashboard->findById($dashboardId);
      return $dashboard;
    }
    else {
      return Dashboard::indexDashboardByUser($this->curUserId());
    }
  }
  
  public function curDashboardPerms($u, $dashboardId) {
    if ($u->isAdmin())
      return array('canManageDataSource'=>'t', 'canWriteDataSource'=>'t', 'canManageWidget'=>'t', 'canViewDashboard'=>'t');

    $permModel = new DashboardPermission();
    $perms = $permModel->permissions($u->attr('id'), $dashboardId);
    if (empty($perms))
      return array('canManageDataSource'=>'f', 'canWriteDataSource'=>'f', 'canManageWidget'=>'f', 'canViewDashboard'=>'t');
    else
      return $perms[0];
  }

  protected function isActionAllowed($actionName, $dashboardId=null) {
    $u = $this->curUser();
    if ($u->isAdmin())
      return true;

    $perms = array('canManageDataSource'=>'f', 'canWriteDataSource'=>'f', 'canManageWidget'=>'f', 'canViewDashboard'=>'t');
    if (isset($dashboardId))
      $perms = $this->curDashboardPerms($u, $dashboardId);
    return isset($perms[$actionName]) && $perms[$actionName] == 't';
  }

  public function doAjaxGet() {
    $response = array();

    $u = $this->curUser();
    if (!$u->isDashboardEnabled()) {
      $this->renderAjaxError($response, "permission denied");
      return;
    }

    $action = $_GET['action'];
    if ($action == 'list')
      $this->listDashboard();
    if ($action == 'load')
      $this->loadDashboard();
    else
      $this->renderAjaxError($response, sprintf(L("%s is not supported"), $action));
  }
  
  public function cptWidgets() {
    return glob('plugins/cpt/widgets/*/*_widget.js');
  }
  
  protected function listDashboard() {
    //only enable dashboard for FG+
    if (!FeatureControl::instance()->isEnabled("Dashboard"))
      die('{}');

    $dashboards = Dashboard::dashboardList();
    
    $uid = $this->curUserId();
    $permModel = new DashboardPermission();
    $perms = $permModel->permissions($uid);
    $response = array();
    foreach($dashboards as $dashboard) {
      $dashboardId = $dashboard['id'];

      // root node will always be visible
      if ($dashboardId != '1') {
        $isViewable = false;
        foreach($perms as $perm) {
          if (isset($perm['dashboardId']) && $perm['dashboardId'] != $dashboardId)
            continue;
          if (isset($perm['canViewDashboard']) && $perm['canViewDashboard'] == 't') {
            $isViewable = true;
            break;
          }
        }
        if (empty($perms))
          $isViewable = true;
        if (!$isViewable)
          continue;
      }
      $response[] = map2json($dashboard);
    }
    die('{"response": [' . implode(",", $response) . ']}');
  }
  
  public function doAjaxPost() {
    $response = array();

    $u = $this->curUser();
    if (!$u->isDashboardEnabled()) {
      $this->renderAjaxError($response, "permission denied");
      return;
    }

    $action = $_POST['action'];
    if ($action == 'save')
      $this->doSave();
    else if ($action == 'rename')
      $this->doRename();
    else if ($action == 'delete')
      $this->doDelete();
    else if ($action == 'layout')
      $this->doLayout();
    else
      $this->renderAjaxError($response, sprintf(L('%s is not supported'), $action));
  }
  
  protected function doSave() {
    $response = array();

    if (!isset($_POST['name']))
      $this->renderAjaxError($response, sprintf(L("missing '%s' parameter"), 'name'));

    $board = new Dashboard();
    if (isset($_POST['id'])) { // update
      $id = $_POST['id'];

      if (!$this->isActionAllowed('canManageWidget', $id))
        $this->renderAjaxError($response, L('permission denied'));

      if (!$board->findById($id))
        $this->renderAjaxError($response, sprintf(L("can not find dashboard for id: %s"), $id));
      
      $board->saveDashboard($_POST['name'], $this->getInputData('content', false));
      $response['id'] = $id;
      die('{"response": ' . map2json($response). '}');
    } else { // create
      if (!$this->isActionAllowed('canManageWidget'))
        $this->renderAjaxError($response, L('permission denied'));

      if (!isset($_POST['name']) || !isset($_POST['content']))
        $this->renderAjaxError($response, sprintf(L("missing parameters: both '%s' and '%s' are required."), 'name', 'content'));
      
      $id = $board->createDashboard($_POST['name'], $this->getInputData('content', false));
      if ($id) {
        $response['id'] = $id;
        $this->redirect($this->makeUrl('dashboard/index.php?dashboard_id='.$id));
      }
      else
        $this->renderAjaxError($response, L("failed to create dashboard"));
    }
  }
  
  protected function loadDashboard() {
    $response = array();
    $id = $_GET['id'];
    $board = new Dashboard();
    if ($board->findById($id)) {
      $uid = $this->curUserId();
      $permModel = new DashboardPermission();
      $perms = $permModel->permissions($uid, $id);
      if (!empty($perms)) {
        die('{"response": ' . $board->attr("content") . ', "perms": ' . map2json($perms[0]) . '}');
      } else {
        die('{"response": ' . $board->attr("content") . '}');
      }
    } else {
      $this->renderAjaxError($response, sprintf(L("can not find dashboard for id: %s"), $id));
    }
  }
  
  protected function doRename() {
    $response = array();
    if (!isset($_POST['new_name']) || !isset($_POST['id']))
      $this->renderAjaxError($response, "missing parameters");

    $id = $_POST['id'];
    $new_name = $_POST['new_name'];
    
    if (!$this->isActionAllowed('canManageWidget', $id))
      $this->renderAjaxError($response, L('permission denied'));

    $board = new Dashboard(); 
    if ($board->renameDashboard($id, $new_name))
      $this->renderAjaxSuccess($response);
    else
    {
      $error = sprintf(L("failed to rename dashboard(#%s) to %s"), $id, $new_name);
      $this->renderAjaxError($response, $error);
    }
  }
  
  protected function doDelete() {
    $response = array();
    if (!isset($_POST['id_list']))
      $this->renderAjaxError($response, L("invalid parameter"));
    
    $id_list = explode(",", $_POST['id_list']);

    foreach($id_list as $id) {
      if (!$this->isActionAllowed('canManageWidget', $id))
        $this->renderAjaxError($response, L('permission denied'));
    }

    if (Dashboard::deleteDashboard($id_list))
      $this->renderAjaxSuccess($response); 
    else
      $this->renderAjaxError($response, L("failed to delete dashboards: ") . $_POST['id_list']);
  }

  protected function doLayout() {
    $response = array();
    if (!isset($_POST['id']) || !isset($_POST['parent_id']) || !isset($_POST['sibling_ids']))
      $this->renderAjaxError($response, L("invalid parameter"));

    $id = $_POST['id'];
    $parent_id = $_POST['parent_id'];
    $sibling_ids = explode(',', $_POST['sibling_ids']);
    
    if (!$this->isActionAllowed('canManageWidget', $id))
      $this->renderAjaxError($response, L('permission denied'));

    if (Dashboard::updateLayout($id, $parent_id, $sibling_ids))
      $this->renderAjaxSuccess($response); 
    else
    {
      $error = sprintf(L("failed to update layout for dashboard(#%s)"), $id);
      $this->renderAjaxError($response, $error);
    }
  }

  public function plugins() {
    $access_model = new FeatureAccessControl();

    $plugins = collectPlugins();
    foreach($plugins as &$plugin) { 
      if (!isset($plugin['indexFile']))
        continue;

      if (!$access_model->canUserAccess($_SESSION['user_id'], $plugin['id'])) 
        $plugin['show'] = false;

      $plugin['url'] = $this->makeUrl($plugin['indexFile']);
    }
    return $plugins;
  }
}

$controller = new DashboardController();
$controller->run();

$u = $controller->curUser();
$d = $controller->curDashboard();
if (!$d->isNull()) {
  // die(L("Can not open dashboard, maybe you don't have permission to open it"));
  $dperms = $controller->curDashboardPerms($u, $d->attr('id'));
  $canManageDataSource = $u->isAdmin() || $dperms['canManageDataSource'] == 't';
  $canWriteDataSource = $u->isAdmin() || $dperms['canWriteDataSource'] == 't';
  $canManageWidget = $u->isAdmin() || $dperms['canManageWidget'] == 't';
  $canViewDashboard = $u->isAdmin() || $dperms['canViewDashboard'] == 't';
}
else {
  $canManageDataSource = $u->isAdmin() ? true : false;
  $canWriteDataSource = $u->isAdmin() ? true : false;
  $canManageWidget = $u->isAdmin() ? true : false;
  $canViewDashboard = $u->isAdmin() ? true : false;
}

$contentJson = str_replace("</script>", "<\/script>", 
  // append cache param to assets that starts with './uploads/' for cache busting
  appendAssetCacheParam($d->attr('content'), "./uploads/")); 

$accessModel = new FeatureAccessControl();
$shouldChangePassword = $_SESSION['should_change_pwd'];
$isKiosk = $_SESSION['user_name'] == 'Kiosk';
unset($_SESSION['should_change_pwd']);
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title><?php echo sprintf(L('%s Dashboard'), appName()) ?></title>
	  <meta name="mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-capable" content="yes" />
    <meta name="apple-mobile-web-app-status-bar-style" content="black" />
    <meta name="viewport" content = "width = device-width, initial-scale = 1, user-scalable = no" />
    <link href="<?php A('css/freeboard.min.css') ?>" rel="stylesheet" />
    <link href="<?php A('../js/jstree/themes/default/style.min.css') ?>" rel="stylesheet" />
    <!-- <link href="../css/cssprogress.min.css" rel="stylesheet" /> -->
    <link href="<?php A('../css/iziToast.min.css') ?>" rel="stylesheet" />
    <link href="<?php A('css/page.css') ?>" rel="stylesheet" />

    <?php $controller->faviconHTMLInfo() ?>
</head>
<body>

<div id="sidebar_toggle_btn">+</div>

<div id="board-content">
  <img id="dash-logo" data-bind="attr:{src: header_image}, visible:header_image()">

  <div id="dashboard_panel">
    <div class="gridster responsive-column-width">
        <ul data-bind="grid: true">
        </ul>
    </div>
  </div>

  <div id="sidebar">
    <div class="content">
      <div id="jstree"> </div>

<?php if (isset($u) && $u->isAccountManagementEnabled() && platformName() != "FI"): ?>
      <div id="admin_panel">
         <span id="account_control_btn" class="text-button" ><i class="icon-user icon-white"></i> <?php echo L("Accounts") ?></span>
         <span id="permission_control_btn" class="text-button" style="margin-bottom:20px;"><i class="icon-wrench icon-white"></i> <?php echo L("Permissions") ?></span>

<?php if (platformName() == "FW") { ?>
         <span id="backup_btn" class="text-button" ><i class="icon-folder-close icon-white"></i> <?php echo L("Backup") ?></span>
         <span id="restore_btn" class="text-button" ><i class="icon-retweet icon-white"></i> <?php echo L("Restore") ?></span>
         <span id="upload_btn" class="text-button" ><i class="icon-upload icon-white"></i> <?php echo L("Upload") ?></span>

  <?php if (isset($u) && $u->isAdmin()) { ?>
         <span class="text-button" ><i class="icon-wrench icon-white"></i> <a href="../app/utility.php?action=run_phpliteadmin" target="_blank"><?php echo L("DB Manager") ?></a></span>
  <?php } ?>

<?php } ?>

      </div>
<?php endif; ?>

<?php 
    if (platformName() == "FI" ) {
      // add plugin links
      $plugins = $controller->plugins();
      if (!empty($plugins)) {
        echo '<hr class="hr"/>';
        foreach($plugins as $plugin) {
          if (!$plugin['show'])
            continue;
           
          if ($plugin['openNewPage'])
            echo "<div style='text-align:center'><a href='". $plugin['url'] . "' target='_blank' >" . $plugin['name'] ."</a></div>";
          else
            echo "<div style='text-align:center'><a href='". $plugin['url'] . "' >" . $plugin['name'] ."</a></div>";
        }
      }
    }
?>

      <hr class="hr"/>

      <div class="session-control">
<?php if (platformName() == "FW") { ?>
  <?php if (isset($u) && $accessModel->canUserAccess($u->attr('id'), 'FWUtility')) : ?>
        <a href="<?php echo $controller->utilityUrl() ?>" target="_blank"><?php echo L('Utility') ?></a>
        |
  <?php endif; ?>
  <?php if ($controller->isProxied()) : ?>
        <a href='/'>EMS</a> | <a href="/logout"><?php echo L('Signout') ?></a>
  <?php else : ?>
        <a href="../app/signout.php"><?php echo L('Signout') ?></a>
  <?php endif; ?>
<?php } else { ?>
  <?php if (platformName() == "FI") : ?>
        <a href='/'>EMS</a> | <a href="/logout"><?php echo L('Signout') ?></a>
  <?php elseif ($isKiosk) : ?>
        <a href="../app/signout.php"><?php echo L('Signout') ?></a>
  <?php else : ?>
        <a href="../app/graphic.php"><?php echo L('Graphic') ?></a> | <a href="../app/signout.php"><?php echo L('Signout') ?></a>
  <?php endif; ?>
<?php } ?>
      </div>

      <hr class="hr"/>
      <div class="UIOptions" style="margin-top:20px">
        <div>
          <input id="toggleSidebarTouch" type="checkbox" style="width:auto; margin-left:30px;" checked>
          <label for="toggleSidebarTouch"><?php echo L('Enable Slideout') ?></label>
        </div>

        <div>
          <input id="toggleNotification" type="checkbox" style="width:auto; margin-left:30px;" checked>
          <label for="toggleNotification"><?php echo L('Enable Notification') ?></label>
        </div>

        <div>
          <input id="toggleConfirmOnClose" type="checkbox" style="width:auto; margin-left:30px;" checked>
          <label for="toggleConfirmOnClose"><?php echo L('Confirm On Close') ?></label>
        </div>
      </div>
    </div>
  </div>
</div>

<header id="main-header" data-bind="if:allow_edit">
<?php if ($u->isAdmin() || $canManageDataSource || $canManageWidget): ?>
    <div id="admin-bar">
        <div id="admin-menu">
            <div id="board-tools">
                <h1 id="board-logo" class="title bordered">freeboard</h1>
                <div id="board-actions">
                    <ul class="board-toolbar vertical">
                        <?php if ($u->isAdmin()): ?>
                        <li><i class="icon-file icon-white"></i>
                            <label data-bind="click: function() { manager.newDashboard(); }" data-pretty="true"><?php echo L('New') ?></label>
                        </li>
                        <?php endif; ?>

                        <li><i class="icon-download-alt icon-white"></i>
                            <label data-bind="click: function() { manager.saveDashboard(); }" data-pretty="true"><?php echo L('Save') ?></label>
                        </li>

                        <?php if ($u->isAdmin()): ?>
                        <li><i class="icon-download-alt icon-white"></i>
                            <label data-bind="click: function() { manager.saveDashboardAs(); }" data-pretty="true"><?php echo L('Save As...') ?></label>
                        </li>
                        <?php endif; ?>

                        <li id="add-pane" data-bind="click: createPane"><i class="icon-plus icon-white"></i><label><?php echo L('Add Pane') ?></label></li>
                    </ul>
                </div>
            </div>
            <div id="datasources">
                <h2 class="title"><?php echo L('DATASOURCES') ?></h2>

                <div class="datasource-list-container">
                    <table class="table table-condensed sub-table" id="datasources-list" data-bind="if: datasources().length">
                        <thead>
                        <tr>
                            <th><?php echo L('Name') ?></th>
                            <th><?php echo L('Last Updated') ?></th>
                            <th>&nbsp;</th>
                        </tr>
                        </thead>
                        <tbody data-bind="foreach: datasources">
                        <tr>
                            <td>
                              <?php if ($canManageDataSource): ?>
                                <span class="text-button datasource-name" data-bind="text: name, pluginEditor: {operation: 'edit', type: 'datasource'}"></span>
                              <?php else: ?>
                                <span class="text-button datasource-name" data-bind="text: name"></span>
                              <?php endif; ?>
                            </td>
                            <td data-bind="text: last_updated"></td>
                            <td>
                                <ul class="board-toolbar">
                                    <li data-bind="click: updateNow"><i class="icon-refresh icon-white"></i></li>
                                  <?php if ($canManageDataSource): ?>
                                    <li data-bind="pluginEditor: {operation: 'delete', type: 'datasource'}">
                                        <i class="icon-trash icon-white"></i></li>
                                  <?php endif; ?>
                                </ul>
                            </td>
                        </tr>
                        </tbody>
                    </table>
                </div>
                <?php if ($canManageDataSource): ?>
                <span class="text-button table-operation" data-bind="pluginEditor: {operation: 'add', type: 'datasource'}"><?php echo L('ADD') ?></span>
                <?php endif; ?>
            </div>
        </div>
    </div>
	<div id="column-tools" class="responsive-column-width">
		<ul class="board-toolbar left-columns">
			<li class="column-tool add" data-bind="click: addGridColumnLeft"><span class="column-icon right"></span><i class="icon-arrow-left icon-white"></i></li>
			<li class="column-tool sub" data-bind="click: subGridColumnLeft"><span class="column-icon left"></span><i class="icon-arrow-right icon-white"></i></li>
		</ul>
		<ul class="board-toolbar right-columns">
			<li class="column-tool sub" data-bind="click: subGridColumnRight"><span class="column-icon right"></span><i class="icon-arrow-left icon-white"></i></li>
			<li class="column-tool add" data-bind="click: addGridColumnRight"><span class="column-icon left"></span><i class="icon-arrow-right icon-white"></i></li>
		</ul>
	</div>
    <div id="toggle-header" data-bind="click: toggleEditing">
        <i id="toggle-header-icon" class="icon-wrench icon-white"></i></div>
<?php endif; ?>
</header>

<div style="display:hidden">
    <ul data-bind="template: { name: 'pane-template', foreach: panes}">
    </ul>
</div>

<script type="text/html" id="pane-template">
    <li data-bind="pane: true">
        <header>
            <h1 data-bind="text: title"></h1>
            <ul class="board-toolbar pane-tools">
                <li data-bind="pluginEditor: {operation: 'add', type: 'widget'}">
                    <i class="icon-plus icon-white"></i>
                </li>
                <li data-bind="pluginEditor: {operation: 'edit', type: 'pane'}">
                    <i class="icon-wrench icon-white"></i>
                </li>
                <li data-bind="pluginEditor: {operation: 'delete', type: 'pane'}">
                    <i class="icon-trash icon-white"></i>
                </li>
            </ul>
        </header>
        <section data-bind="foreach: widgets">
            <div class="sub-section" data-bind="css: 'sub-section-height-' + height()">
                <div class="widget" data-bind="widget: true, css:{fillsize:fillSize}"></div>
                <div class="sub-section-tools">
                    <ul class="board-toolbar">
                        <!-- ko if:$parent.widgetCanMoveUp($data) -->
                        <li data-bind="click:$parent.moveWidgetUp"><i class="icon-chevron-up icon-white"></i></li>
                        <!-- /ko -->
                        <!-- ko if:$parent.widgetCanMoveDown($data) -->
                        <li data-bind="click:$parent.moveWidgetDown"><i class="icon-chevron-down icon-white"></i></li>
                        <!-- /ko -->
                        <li data-bind="pluginEditor: {operation: 'edit', type: 'widget'}"><i class="icon-wrench icon-white"></i></li>
                        <li data-bind="pluginEditor: {operation: 'delete', type: 'widget'}"><i class="icon-trash icon-white"></i></li>
                    </ul>
                </div>
            </div>
        </section>
    </li>
</script>

<script type="text/html" id="name_dialog_tpl">
  <div>
    <div style='color:red; margin-top:5px; margin-bottom:5px;' data-bind='html: error_msg'></div>
    <div style='margin-bottom:15px;'>
      <label for='name'><?php echo L('Name:') ?></label>
      <input type='text' id='name' data-bind='value: name' autofocus>
    </div>
  </div>
</script>

<script type="text/html" id="user_perm_tpl">
  <div>
    <div style='color:red; margin-top:5px; margin-bottom:5px;' data-bind='html: error_msg'></div>
    <div style='margin-bottom:15px;'>
      <label><?php echo L("UserName:") ?></label>
      <select id="user_name_chooser" data-bind="options: users, optionsText: 'userName', value: selectedUser">
        <option data-bind="value: userName, text: userName"/> 
      </select>
      <table class="panel-table table-striped">
        <tbody data-bind="foreach: selectedUser().perms">
          <tr>
            <td data-bind="text: pageName"></td>
            <td><label><input type='checkbox' data-bind="checked: canManageDataSource, disable: userName=='Kiosk'"/><?php echo L('CanManageDataSource') ?></label></td>
            <td><label><input type='checkbox' data-bind="checked: canManageWidget, disable: userName=='Kiosk'"/><?php echo L('CanManageWidget') ?></label></td>
            <td><label><input type='checkbox' data-bind='checked: canWriteDataSource'/><?php echo L('CanWriteDataSource') ?></label></td>
            <td><label><input type='checkbox' data-bind='checked: canViewDashboard'/><?php echo L('CanViewDashboard') ?></label></td>
          </tr>
        </tbody>
      </table>
    </div>
  </div>
</script>

<script type="text/html" id="user_account_tpl">
  <div>
    <div style='color:red; margin-top:5px; margin-bottom:5px;' data-bind='html: error_msg'></div>
    <div id='password_edit_panel' style='margin-bottom:15px;'>
      <div>
        <label><?php echo L("UserName:") ?></label>
        <select data-bind="options: users, optionsText: 'userName', value: selectedUser">
          <option data-bind="value: userId, text: userName"/> 
        </select>
      </div>
      <div>
        <label><?php echo L("New Password:") ?></label>
        <div class="input-prepend input-append">
          <input id="newPassword" type="password" data-bind="value: selectedUser().newPassword">
          <span class="add-on" id="show-new-password">
            <i class="icon-eye-open icon-white"></i>
          </span>
        </div>
      </div>
      <div>
        <label><?php echo L("Verify Password:") ?></label>
        <div class="input-prepend input-append">
          <input id="verifyPassword" type="password" data-bind="value: selectedUser().verifyPassword">
          <span class="add-on" id="show-verify-password">
            <i class="icon-eye-open icon-white"></i>
          </span>
        </div>
      </div>
    </div>
  </div>
</script>

<script type="text/html" id="backup_panel_tpl">
  <div>
    <div class="alert" data-bind="css: {hide: msg().length == 0, 'alert-error': is_error(), 'alert-success': !is_error()}, html: msg"></div>
    <div id='backup_edit_panel' style='margin-bottom:15px;'>
      <div>
        <label><?php echo L("Backup Name:") ?></label>
        <div><input type="text" data-bind="value: name" autofocus></div>
      </div>
      <!--
      <div>
        <input class="backup-network-conf" type="checkbox" data-bind="checked: backupNetworkConf" /><?php echo L("Backup Network Configs") ?>
      </div>
      -->
    </div>
  </div>
</script>

<script type="text/html" id="restore_panel_tpl">
  <div>
    <div class="alert" data-bind="css: {hide: msg().length == 0, 'alert-error': is_error(), 'alert-success': !is_error()}, html: msg"></div>
    <div id='restore_edit_panel' style='margin-bottom:15px;'>
      <div>
        <table class="panel-table table-striped">
          <thead>
            <tr>
              <td>NAME</td>
              <td>CREATED AT</td>
            </tr>
          </thead>
          <tbody data-bind="foreach: backups">
            <tr>
              <td data-bind="text: name"></td>
              <td data-bind="text: moment(ts,'X').format('YYYY-MM-DD HH:mm')"></td>
              <td>
                <div data-bind="visible: withNetworkConf">
                <input type="checkbox" data-bind="checked: restoreNetworkConf"><?php echo L("Network Config") ?>
                </div>
              </td>
              <td style="text-align: right;">
                <span class='text-button' data-bind="click: $parent.restoreBackup"><?php echo L("Restore") ?></span>
                |
                <span class='text-button' data-bind="click: $parent.downloadBackup, css:{'disabled-button': preset}"><?php echo L("Download") ?></span>
                |
                <span class='text-button' data-bind="click: $parent.deleteBackup, css:{'disabled-button': preset}"><?php echo L("Delete") ?></span>
              </td>
            </tr>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</script>

<script type="text/html" id="upload_panel_tpl">
  <div>
    <div class="alert" data-bind="css: {hide: msg().length == 0, 'alert-error': is_error(), 'alert-success': !is_error()}, html: msg"></div>
    <div style='margin-bottom:15px;'>
      <div>
        <form  enctype="multipart/form-data">
          <label for="inputBackupFile">Upload File:</label>
          <input type="file" id="inputBackupFile" name="backupTarBall" style="width:400px;">
          <div class="cssProgress hide">
            <div class="progress2">
              <div class="cssProgress-bar cssProgress-success cssProgress-stripes" data-percent='0' >
                <span class="cssProgress-label">0%</span>
              </div>
            </div>
          </div>
          <div style="margin-top:20px;">
            <input id="inputNetworkConfig" style="width: initial;" type="checkbox" data-bind="checked: restoreNetworkConf">
            <label for="inputNetworkConfig"><?php echo L("Restore Network Config") ?></label>

            <span class="text-button" data-bind="click: onUpload, disable: state == 'uploading'" style="float:right;">Upload</span>
            <span class="text-button" data-bind="click: onUploadRestore, disable: state == 'uploading'" style="float:right;">Upload & Restore</span>
          </div>
        </form>
      </div>
    </div>
  </div>
</script>

  <?php frontendLangJson() ?>

  <script type='application/json' id='contentJson'>
    <?php echo $contentJson ?>
  </script>

  <script type="text/javascript" src="<?php A('../js/l18n.js') ?>"></script>
  <script src="<?php A('js/freeboard.thirdparty.min.js') ?>"></script>
  <script src="<?php A('../js/jstree/jstree.min.js') ?>"></script>
  <script src="<?php A('js/widget_adapter.min.js') ?>"></script>
  <script type="text/javascript">
    let baseWidgets = [
      "<?php A('js/freeboard_plugins.min.js') ?>",
      "<?php A('plugins/cpt/cpt.datasources.js') ?>",
      "<?php A('plugins/mqtt/paho.mqtt.plugin.js') ?>"
    ];

    window.cptWidgets = [ 
      <?php 
        $widgets = $controller->cptWidgets(); 
        foreach($widgets as $widget)
          echo '"'.assetCacheParam($widget).'",' . PHP_EOL;
      ?> 
    ];

    window.widgetUrls = baseWidgets.concat(window.cptWidgets);
    window.cur_account_info = {
        'name': '<?php echo htmlspecialchars($u->attr('name'), ENT_QUOTES|ENT_HTML401) ?>',
        'is_admin': <?php echo $u->isAdmin() ? 1 : 0 ?>,
      };
    window.shouldChangePassword = <?php echo $shouldChangePassword ? 'true' : 'false' ?>;
    let assetManifest = '<?php echo json_encode(getAssetManifest()) ?>';
    assetManifest = JSON.parse(assetManifest);
    window.assetCacheParam = function(assetPath) {
      if (!(assetPath in assetManifest))
        return assetPath;
      return assetPath + "?stamp=" + assetManifest[assetPath].mtime + "_" + assetManifest[assetPath].size;
    };

    window.isOEM = <?php echo isOEM() ? 'true' : 'false' ?>;
    window.appName = "<?php echo appName() ?>";

    <?php if ($d->isNull() ): ?>
      <?php if (isset($_GET['dashboard_id'])): ?>
        alert("<?php echo L("Can not open dashboard, not enough permission or dashboard not exists.") ?>");
      <?php endif; ?>
    <?php else: ?>

      <?php 
        $dashboardId = $d->attr('id'); 
        if(isset($dashboardId)):
      ?>
        window.curDashboardId = "<?php echo $dashboardId; ?>";
      <?php else: ?>
        window.curDashboardId = null;
      <?php endif; ?>

    <?php endif; ?>

    window.perms = {
      canManageDataSource: <?php echo ($canManageDataSource ? 'true' : 'false'); ?>,
      canWriteDataSource : <?php echo ($canWriteDataSource ? 'true' : 'false'); ?>,
      canManageWidget : <?php echo ($canManageWidget ? 'true' : 'false'); ?>,
      canViewDashboard : <?php echo ($canViewDashboard ? 'true' : 'false'); ?>
    };

    window.platformName = "<?php echo platformName() ?>";
    
    $(window).on("uiReady", function() {
      var enableTouch = Cookies.get("UIOption.enableTouchSlideout")=='true';
      $("#toggleSidebarTouch").prop('checked', enableTouch);
      window.sidebarControl.enableTouch(enableTouch);

      $("#toggleSidebarTouch").click(function() {
        window.sidebarControl.enableTouch(this.checked);
        Cookies.set("UIOption.enableTouchSlideout", this.checked)
      });
      
      var enableNotification = Cookies.get("UIOption.enableNotification")=='true';
      $("#toggleNotification").prop('checked', enableNotification);
      var notificationCls = '';
      if (!enableNotification) 
        notificationCls = 'hide'
      iziToast.settings({class: notificationCls})

      $("#toggleNotification").click(function() {
        if (!this.checked)
          iziToast.destroy();

        var notificationCls = '';
        if (!this.checked)
          notificationCls = 'hide';
        iziToast.settings({class: notificationCls})
        Cookies.set("UIOption.enableNotification", this.checked)
      });

      var enableConfirmOnClose = Cookies.get("UIOption.enableConfirmOnClose");
      if (typeof enableConfirmOnClose == 'undefined')
        enableConfirmOnClose = false;
      else
        enableConfirmOnClose = enableConfirmOnClose == 'true';
      $("#toggleConfirmOnClose").prop('checked', enableConfirmOnClose);

      $("#toggleConfirmOnClose").click(function() {
        Cookies.set("UIOption.enableConfirmOnClose", this.checked)
      });
    });
  </script>

  <script src="<?php A('../js/jquery.form.min.js') ?>"></script>
  <script src="<?php A('../js/iziToast.min.js') ?>"></script>
  <script src="<?php A('../js/moment.min.js') ?>"></script>

  <script src="<?php A('js/slideout.min.js') ?>"></script>
  <script src="<?php A('js/page.min.js') ?>"></script>

  <?php 
    if (file_exists("css/custom.css")) {
  ?>
  <link href="<?php A('css/custom.css') ?>" rel="stylesheet" />
  <?php } ?>
  
  <script language="javascript" type="text/javascript">
      "use strict";
    let interval = 60000;
    var liveChecker = function() {
      $.ajax({url: "../app/landing_page.php", method: "GET", dataType: "json", ajax: true})
        .done(function() {
          let toast = document.querySelector(".connErrorToast")
          if (toast)
            iziToast.hide({}, toast);
        })
        .fail(function() {
          let toast = document.querySelector(".connErrorToast")
          if (toast)
            iziToast.hide({}, toast);
          iziToast.error({
            title: "Error", 
            message: "Controller Connection Lost", 
            class:"connErrorToast", 
            position:"topRight", 
            timeout: false})
        }).always(function() {
          setTimeout(liveChecker, interval);
        });
    }

    setTimeout(liveChecker, interval);
  </script>
</body>
</html>
