<?php
//vim: ts=2 sw=2

include_once "jwt/JWT.php";
include_once "jwt/ExpiredException.php";

include_once "db.php";
include_once "utils.php";
include_once "jwt_key_store.php";

use \Firebase\JWT\JWT;
use \Firebase\JWT\ExpiredException;

// the JWT authentication & authorization workflow: 
//
//   1. signin request with header "X-EIO-Auth-Upgrade: JWT" is considered JWT req
//   2. after signin is done, pass back JWT in response by header "X-EIO-TOKEN: JWT_STRING"
//   3. later requests will carry JWT by request header "Authorization: Bearer JWT_STRING"
//
class SessionManager
{
  protected $cookiePath = '';
  protected $token = NULL;
  protected $token_string = NULL;
  protected $expired = false;

  function __construct($cookiePath)
  {
    $this->cookiePath = $cookiePath;
  }

  protected function isJWTAuthUpgrade()
  {
    return (isset($_SERVER['HTTP_X_EIO_AUTH_UPGRADE'])
      && strtoupper($_SERVER['HTTP_X_EIO_AUTH_UPGRADE']) == "JWT");
  }

  protected function isJWTAuth()
  {
    return !is_null($this->jwtTokenString());
  }

  protected function getPubKey($payload)
  {
    $key = null;
    $iss = arrayGet($payload, "iss", "EIO_SELF");
    if ($iss == "EIO_SELF" || $iss == "EIO_FI") {
      $store = new JWTKeyStore(platformName());
      $key = $store->pub_for($iss);
    } else if ($iss == "EIO_P2P") {
      error_log("P2P JWT token is not supported yet");
    }
    return $key;
  }

  protected function getUserName($payload)
  {
    return arrayGet($payload, "usr");
  }

  protected function getTokenIssuer($payload)
  {
    return arrayGet($payload, "iss", "EIO_SELF");
  }

  protected function getTokenFromCookie()
  {
    if (!isset($_COOKIE['EMSTOKEN']))
      return null;

    return $_COOKIE['EMSTOKEN'];
  }

  protected function getTokenFromHeader()
  {
    if (!isset($_SERVER['HTTP_AUTHORIZATION']))
      return null;
    $val = $_SERVER['HTTP_AUTHORIZATION'];

    $parts = explode(" ", trim($val));
    if (count($parts) < 2)
      return null;

    if ($parts[0] != "Bearer")
      return null;

    return $parts[1];
  }

  protected function jwtTokenString()
  {
    if (!is_null($this->token_string))
      return $this->token_string;

    $this->token_string = $this->getTokenFromCookie();
    if (is_null($this->token_string))
      $this->token_string = $this->getTokenFromHeader();

    return $this->token_string;
  }

  protected function jwtToken()
  {
    //TODO: avoid decode JWT multiple times
    if (!is_null($this->token))
      return $this->token;

    $token_string = $this->jwtTokenString();
    if (is_null($token_string))
      return null;

    $tks = explode('.', $token_string);
    if (count($tks) != 3) {
      error_log("invalid auth token format");
      return null;
    }

    try {
      $payload = (array)JWT::jsonDecode(JWT::urlsafeB64Decode($tks[1]));
    } catch (Exception $e) {
      error_log("failed to decode JWT payload: " . $e->getMessage() . "\n");
      return null;
    }

    if (is_null($payload)) {
      error_log("failed to decode JWT payload");
      return null;
    }

    $key = $this->getPubKey($payload);
    try {
      $this->token = (array)JWT::decode($token_string, $key, array('RS256'));
    } catch (ExpiredException $e) {
      $this->expired = true;
      error_log("JWT expired: " . $e->getMessage() . "\n");
      return null;
    } catch (Exception $e) {
      error_log("JWT decode error: " . $e->getMessage() . "\n");
      return null;
    }
    return $this->token;
  }

  //called at the very beginning of request handler:
  // * create cookie if that's missing
  // * start session(create/resume session data based on session_id)
  public function start($is_CORS, $is_https)
  {
    if ($this->isJWTAuth()) {
      // get session_id from JWT 
      $token = $this->jwtToken();
      $signedIn = false;
      if (!is_null($token) && !is_null($token['jti'])) {
        session_id($token['jti']);
        $signedIn = true;
      } else {
      }

      // resume session with the above session_id
      session_start();

      if ($signedIn) {
        if (!isset($_SESSION['user_name'])) {
          return $this->initSessionUserData($token);
        }
      }
    } else {
      //NOTE: *XCPTSESSID* (DEPRECATED) cookie only for CORS requests, abandoned due to SameSite issue 
      //      *XCPTUSSESSID* unsecure cookie to workaround the CORS auth cookie issue before we have JWT 
      //      *CPTSESSID* cookie for same-site requests
      //      *HTTPSSESSID* for https, include Secure.
      $ses_name = $is_CORS ? 'XCPTUSSESSID' : ($is_https ? 'HTTPSSESSID' : 'CPTSESSID');

      $cookie_exists = isset($_COOKIE[$ses_name]);
      //GOTCHA: if there are multiple php application deployed, then by default
      //they all use the cookie 'PHPSESSID', that will cause conflicts issue
      //among them, rename the cookie to fix it. 
      session_start(array('name' => $ses_name));

      if (!$cookie_exists) {
        $this->setup_cookie($ses_name, null, $is_CORS, $is_https);
      }
    }
    return false;
  }

  public function initSessionUserData($token)
  {
    $name = $this->getUserName($token);
    $u = new User();
    if (!$u->find($name)) return false;

    $issuer = $this->getTokenIssuer($token);
    //NOTE: admin user is always linked with FI, otherwise no chance to login
    //to device with JWT at all. 
    if ($issuer == "EIO_FI" && $name != 'admin' && !$u->isLinkedWithFI())
      return false;

    $_SESSION['user_name'] = $name;
    $_SESSION['user_id'] = intval($u->attr('id'));
    return true;
  }

  public function createJWT($user, $remember_me)
  {
    $now = time();
    $store = new JWTKeyStore(platformName());
    $key = $store->key_for('EIO_SELF');
    $expiry = $now + ($remember_me ? 30 * 86400 : 86400);
    $payload = array(
      'iss' => 'EIO_SELF',
      'iat' => $now,
      'exp' => $expiry,
      'jti' => session_id(),
      'usr' => $user->attr("name"),
      'uid' => $user->attr("id"),
    );
    $jwt = JWT::encode($payload, $key, "RS256");
    return $jwt;
  }

  //called just after user login
  public function afterLogin($user, $remember_me, $is_CORS, $is_https)
  {
    $now = time();
    if ($this->isJWTAuthUpgrade()) {
      $jwt = $this->createJWT($user, $remember_me);
      //NOTE: for CORS request, the header name 'X-EIO-TOKEN' must be listed in
      //'Access-Control-Allow-Headers' field in web server's settings
      header("X-EIO-TOKEN: $jwt");
    } else {
      session_regenerate_id(true);
      $ses_name = session_name();
      $expiry_str = null;
      if ($remember_me)
        $expiry_str = "Expires=" . date('D, d M Y H:i:s', $now + (86400 * 30)) . ";"; // 1 month from now

      $this->setup_cookie($ses_name, $expiry_str, $is_CORS, $is_https);
    }
  }

  protected function setup_cookie($ses_name, $expiry = null, $is_CORS = false, $is_https = false)
  {
    $ses_id = session_id();

    $cookie_str = "Set-Cookie: $ses_name=$ses_id; Path={$this->cookiePath};";
    if (!is_null($expiry))
      $cookie_str .= " $expiry ";

    //for cors request, the cookie must be set with 'SameSite=None' and
    //'Secure', otherwise cors request will fail.
    //more details, refer to: https://blog.heroku.com/chrome-changes-samesite-cookie#what-does-this-mean-what-are-third-party-cookies-what-are-cross-site-request
    if ($is_CORS) {
      //FIXME: old version of chrome or android webview app(for example
      //SystemView) will reject the cookie if SameSite=None, this is not
      //supported before: https://www.chromium.org/updates/same-site/incompatible-clients
      //we will need to ask user to disable 'same-site-by-default-cookies' in
      //chrome until JWT support is added 
      // header("Set-Cookie: $ses_name=$ses_id; Path=/; SameSite=None; Secure");
    } else {
      $cookie_str .= " SameSite=Lax; ";
      // header("Set-Cookie: $ses_name=$ses_id; Path=/; SameSite=Lax;");
    }
    $cookie_str .= " HttpOnly; ";
    if ($is_https) {
      $cookie_str .= " Secure; ";
    }
    header($cookie_str);
  }

  public function expired()
  {
    if (!$this->isJWTAuth())
      return false;

    return is_null($this->jwtToken()) || $this->expired;
  }
}
