<?php 
//vim: ts=2 sw=2

include_once "lang.php";
include_once "utils.php";

function cmp($a, $b) {
  if ($a['order'] == $b['order'])
    return 0;
  else 
    return $a['order'] > $b['order'];
}

class ServiceUtils {

  function __construct() {
    $this->service_mapping = array(
      array(
        'name' => 'http',
        'label' => 'HTTP',
        'proc_name' => 'nginx',
        'service_name' => 'http',
        'order' => 1
      ),
      array(
        'name' => 'https',
        'label' => 'HTTPS',
        'proc_name' => 'nginx',
        'service_name' => 'https',
        'order' => 2
      ),
      array(
        'name' => 'mqtt',
        'label' => 'MQTT',
        'proc_name' => 'mqtt-service',
        'monit' => false,
        'order' => 30
      ),
    );

    $pname = platformName();
    if ($pname == 'FS') {
      array_push($this->service_mapping, 
        array(
          'name' => 'ftp',               //service name
          'label' => 'Ftp',
          'proc_name' => 'pure-ftpd',  // service process pattern 
          'order' => 10
        ),
        array(
          'name' => 'ssh',
          'label' => 'SSH',
          'proc_name' => 'sshd',
          'order' => 20 
        ),
        array(
          'name' => 'openvpn',
          'label' => 'OpenVPN Server',
          'proc_name' => 'openvpn',
          'monit' => false,
          'order' => 25 
        ),
        array(
          'name' => 'openvpn_client',
          'label' => 'OpenVPN Cloud',
          'proc_name' => 'openvpn_client',
          'service_name' => 'openvpn_client',
          'monit' => false,
          'order' => 25 
        ),
        array( 
          'name' => 'smb',
          'label' => 'Samba',
          'proc_name' => 'smbd',
          'service_name' => 'smb',
          'order' => 40
        )
      );

      // use chrony if it exists, otherwise still use ntp for backward compatible
      if (file_exists("/etc/chrony.conf")) {
        array_push($this->service_mapping, 
          array(
            'name' => 'chronyd',
            'label' => 'NTP',
            'proc_name' => 'chronyd',
            'service_name' => 'chrony',
            'order' => 26 
          )
        );
      } else {
        array_push($this->service_mapping, 
          array(
            'name' => 'ntpd',
            'label' => 'NTP',
            'proc_name' => 'ntpd',
            'service_name' => 'ntp',
            'order' => 26 
          )
        );
      }

    } else if ($pname == "FW") {
      array_push($this->service_mapping, 
        array(
          'name' => 'ssh',
          'label' => 'SSH',
          'proc_name' => (file_exists("/usr/sbin/sshd") ? 'sshd' : 'dropbear'), 
          'order' => 20 
        ),
        array( 
          'name' => 'smb',
          'label' => 'Samba',
          'proc_name' => 'smbd',
          'service_name' => 'samba',
          'order' => 40
        ),
        array( 
          'name' => 'dns',
          'label' => 'DNS',
          'proc_name' => 'dnsmasq',
          'service_name' => 'dnsmasq',
          'order' => 21 
        )
      );

      // use chrony if it exists, otherwise still use ntp for backward compatible
      if (file_exists("/etc/chrony/chrony.conf")) {
        array_push($this->service_mapping, 
          array(
            'name' => 'chronyd',
            'label' => 'NTP',
            'proc_name' => 'chronyd',
            'service_name' => 'chronyd',
            'order' => 26 
          )
        );
      } else {
        array_push($this->service_mapping, 
          array(
            'name' => 'ntpd',
            'label' => 'NTP',
            'proc_name' => 'ntpd',
            'service_name' => 'ntpd',
            'order' => 26 
          )
        );
      }
    }

    //sort service tuples by 'order' value
    usort($this->service_mapping, "cmp");
  }

  public function isProcessRunning($proc_name) {
    $cmd = 'ps | grep "' . $proc_name . '" | grep -v grep';
    return !is_null(shell_exec($cmd));
  }

  public function ntpServiceName() {
    $pname = platformName();
    $confPath = $pname == "FW" ? "/etc/chrony/chrony.conf" : "/etc/chrony.conf";
    if (file_exists($confPath)) {
      return $pname == "FW" ? "chronyd" : "chrony";
    } else {
      return $pname == "FW" ? "ntpd" : "ntp";
    }
  }

  public function isServiceEnabled($name) {
    $cmd = null;
    if ($name == 'http') {
      $cmd = 'cat /etc/nginx/nginx.conf | grep "^\s*include http_server.conf;\s*$"';
    } else if ($name == 'https') {
      $cmd = 'cat /etc/nginx/nginx.conf | grep "^\s*include https_server.conf;\s*$"';
    } else {
      $pname = platformName();
      if ($pname == "FS")
        $cmd = 'ls /etc/init.d/S??' . $name ;
      else if ($pname == "FW")
        $cmd = 'ls /etc/rc.d/S??' . $name ;
      else {
        error_log("$pname is not supported yet");
        return;
      }
    }

    return !is_null(shell_exec($cmd));
  }

  public function isServiceAvailable($name) {
    $found = false;
    foreach($this->service_mapping as $service) {
      $service_name = isset($service['service_name']) ? $service['service_name'] : $service['proc_name'];
      if ($name != $service_name)
        continue;

      $found = true;
      break;
    }
    if (!$found) return false;

    $serviceCheckList = array('openvpn', 'openvpn_client');
    if (!in_array($name, $serviceCheckList))
      return true;

    $cmd = null;
    $pname = platformName();
    if ($pname == 'FS') {
      $cmd = 'ls /etc/init.d/???' . $name ;
    } else if ($pname == 'FW') {
      $cmd = 'ls /etc/rc.d/???' . $name ;
    } else {
      return false;
    }

    return !is_null(shell_exec($cmd));
  }

  public function collectServicesStatus() {
    $status = array();
    foreach($this->service_mapping as $service) {
      $service_name = isset($service['service_name']) ? $service['service_name'] : $service['proc_name'];
      if (!$this->isServiceAvailable($service_name)) 
        continue;

      $status[] = array(
        'name' => $service['name'],
        'label' => $service['label'],
        'running' => $this->isProcessRunning($service['proc_name']),
        'enabled' => $this->isServiceEnabled($service_name),
      );
    }
    return $status;
  }

  public function doAction($name, $action) {
    $valid_actions = array('enable', 'disable');
    if (!in_array($action, $valid_actions)) {
      return sprintf(L("invalid service action: %s"), $action);
    }

    foreach($this->service_mapping as $service) {
      if ($service['name'] != $name)
        continue;

      return $this->doServiceAction($service, $action);
    }
    return sprintf(L("can not find service by name %s"), $name);
  }

  protected function controlWeb($service_name, $action) {
    $cmd = null;
    if ($action == 'enable') {
      $cmd = 'sed -i -re "s/^\s*#(\s*include ' . $service_name . '_server.conf;\s*)$/\1/" /etc/nginx/nginx.conf';

      // if enable http, disable the local http server
      if ($service_name == 'http')
        $cmd = $cmd . ' && ' . 'sed -i -re "s/^(\s*include local_http_server.conf;\s*)$/#\1/" /etc/nginx/nginx.conf';
    } else if ($action == 'disable') {
      if ( ($service_name == 'http' && !$this->isServiceEnabled('https')) 
        || ($service_name == 'https' && !$this->isServiceEnabled('http')) )
        return "can not disable both http and https service";

      $cmd = 'sed -i -re "s/^(\s*include ' . $service_name . '_server.conf;\s*)$/#\1/" /etc/nginx/nginx.conf';

      // if disable http, enable the local http server
      if ($service_name == 'http')
        $cmd = $cmd . ' && ' . 'sed -i -re "s/^\s*#(\s*include local_http_server.conf;\s*)$/\1/" /etc/nginx/nginx.conf';
    }
    else {
      error_log("action '$action' is not supported for web service");
      return;
    }

    $pname = platformName();
    $cmd = $cmd . ' && ' . ($pname == 'FS' ? '/etc/init.d/S??nginx restart' : '/etc/rc.d/S??nginx restart');

    shell_exec($cmd);
  }

  protected function doServiceAction($service, $action) {
    if ($service['name'] == 'http' || $service['name'] == 'https') {
      return $this->controlWeb($service['service_name'], $action);
    }
    $service_name = isset($service['service_name']) ? $service['service_name'] : $service['proc_name'];
    $monit = isset($service['monit']) ? $service['monit'] : false;
    doServiceControl($service_name, $action, $monit);
  }
}

?>
