#! /usr/bin/env php
<?php
// vim: ts=2 sw=2

// declare(ticks = 1);

include_once "../cpt/app/UDPClient.class.php";

abstract class ConnectionState {
  const disconnected = 0;
  const connecting = 1;
  const connected = 2;
  const terminated = 3;
}

class MQTTClient {
  private $host = 'test.mostquitto.org';
  private $port = 1883;
  private $client = null;
  private $connState = ConnectionState::disconnected;
  private $reconnectTimes = 0;

  private $sedonaClient = null;
  private $sedonaObjects = null;
  
  function __construct($enableTls=false) {
    $this->client = new Mosquitto\Client();

    //setup callbacks
    $this->client->onConnect(array($this, "onConnect"));
    $this->client->onDisconnect(array($this, "onDisconnect"));
    $this->client->onMessage(array($this, "onMessage"));
    
    if ($enableTls)
      $this->setupTls();
    
    //init sedona objects list
    $this->sedonaObjects = array("/EasyIO/*");
    
    // pcntl_signal(SIGTERM, array($this, "sig_handler"));
    // pcntl_signal(SIGHUP,  array($this, "sig_handler"));
  }
  
  function setupTls() {
    if (is_null($this->client))
      return false;

    $curDir = dirname(__FILE__);

    $caFile =  $curDir . DIRECTORY_SEPARATOR . 'ca' . DIRECTORY_SEPARATOR . 'server.crt';
    $certFile = $curDir . DIRECTORY_SEPARATOR . 'client.crt';
    $keyFile = $curDir . DIRECTORY_SEPARATOR . 'client.key';
    $this->client->setTlsCertificates($caFile, $certFile, $keyFile);
    $this->client->setTlsOptions(Mosquitto\Client::SSL_VERIFY_PEER, "tlsv1.2");
    return true;
  }
  function getSedonaClient() {
    if (is_null($this->sedonaClient))
    {
      $host = "127.0.0.1";
      $port = "1001";
      $this->sedonaClient = new UDPClient($host, $port);
      $this->sedonaClient->setDebug(false);
    }

    if (is_null($this->sedonaClient))
      error_log("can not create sedona client");
    return $this->sedonaClient;
  }
  
  function connect($host, $port=1883) {
    if (is_null($this->client))
      return false;
    
    $this->host = $host;
    $this->port = $port;

    if ($this->connState == ConnectionState::connecting || $this->connState == ConnectionState::connected)
      return true;

    $this->connState = ConnectionState::connecting;
    $this->client->connect($host, $port);
    return true;
  }
  
  function disconnect() {
    if (is_null($this->client))
      return false;

    if ($this->connState == ConnectionState::connected) {
      echo "disconnect conn\n";
      $this->client->disconnect();
    }
    return true;
  }
  
  function reconnect() {
    echo "Reconnecting ...\n";
    $this->reconnectTimes += 1;
    $maxRetryTimes = 10;
    if ($this->reconnectTimes > $maxRetryTimes) {
      // echo "reconnect too many times(>${maxRetryTimes}), give up\n";
      // $this->connState = ConnectionState::terminated;
      // return;
      sleep(5);
    }
      
    return $this->connect($this->host, $this->port);
  }
  
  function doWork() {
    $start = microTime(true);
    $content = $this->fetchSedonaData();
    echo "time cost: " . (microTime(true) - $start) . "\n";
    $this->publishToCloud($content);
  }
  
  function eventLoop() {
    $interval = 5; // in seconds 
    $counter = -1;
    while($this->connState != ConnectionState::terminated) {
      if ($this->connState == ConnectionState::connected) {
        if ((microTime(true)-$counter) < $interval) {
          try {
            $this->client->loop();
          } catch(Mosquitto\Exception $e) {
            echo "Got exception(code: " . $e->getCode() . ", msg: " . $e->getMessage() . ")\n";
            $this->reconnect();
          }
          continue;
        } else
          $counter = microTime(true);

        $this->doWork();
      } else {
        try {
          $this->client->loopForever();
        } catch(Mosquitto\Exception $e) {
          echo "Got exception(code: " . $e->getCode() . ", msg: " . $e->getMessage() . ")\n";
          $this->reconnect();
        }
      }
    }
  }
  
  function run() {
    if (is_null($this->client))
      return false;
    
    $this->eventLoop();
    
    $this->disconnect();
    return true;
  }
  
  // Callbacks
  function onConnect($rc, $msg) {
    if ($rc == 0) {
      $this->connState = ConnectionState::connected;
      echo "Connection Established\n";
      $this->reconnectTimes = 0;
      $this->client->exitLoop();
      return;
    }

    switch($rc) {
    case 1:
      $this->connState = ConnectionState::terminated;
      error_log("Connection refused(unacceptable protocol version)");
      break;
    case 2:
      $this->connState = ConnectionState::terminated;
      error_log("Connection refused (identifier rejected)");
      break;
    case 3: 
      $this->connState = ConnectionState::terminated;
      error_log("Connection refused (broker unavailable)");
      $this->reconnect();
      break;
    default:
      $this->connState = ConnectionState::terminated;
      error_log("Can not connect due to unknown reason.");
      break;
    }
    echo "Connection Refused.\n";
  }
  
  function onDisconnect($rc) {
    if ($rc == 0) { //requested by client
      $this->connState = ConnectionState::terminated;
      echo "Connection Closed\n";
    } else {//TODO: retry ?
      $this->connState = ConnectionState::terminated;
      echo "Connection Lost\n";
      $this->reconnect();
    }
  }
  
  function onMessage($msg) {
    echo "onMessage\n";
  }
  
  function sig_handler($signo) {
    switch ($signo) {
      case SIGTERM:
        $this->connState = ConnectionState::terminated;
        break;
      case SIGHUP:
        $this->connState = ConnectionState::terminated;
        break;
      case SIGUSR1:
        echo "Caught SIGUSR1...\n";
        break;
      default:
        // handle all other signals
    }
  }
  
  // core functions 
  function fetchSedonaData() {
    $sc = $this->getSedonaClient();
    if (is_null($sc))
      return false;
    
    $url = "/app/objects" . implode('~', $this->sedonaObjects);
    if ($sc->get($url)) {
      return $sc->getContent();
    } else {
      error_log("failed to get sedona data: " .  $sc->getError());
    }

    return true;
  }
  
  function publishToCloud($content) {
    if (!$content)
      return false;
    
    $json = json_decode($content, true);
    if (!$json) {
      error_log("failed to parse sedona data response: " . json_last_error_msg());
      error_log("invalid json content: " . $content);
      return false;
    }

    if (!array_key_exists('resultCode', $json) || !array_key_exists('data', $json) || $json['resultCode'] != 0) {
      error_log("invalid sedona data: " . $content);
      return false;
    }
    
    foreach($json['data'] as $section) {
      foreach($section as $comp) {
        if (!isset($comp['path']))
          continue;

        $compPath = $comp['path'];
        $this->client->publish($compPath, json_encode($comp), 2, true);
      }
    } 
    return true;
  }
};

$startTS = microTime(true);

// $client = new MQTTClient(true);
// $client->connect('test.mosquitto.org', 8883);
$client = new MQTTClient();
$client->connect('test.mosquitto.org');

$client->run();

echo "total run time: " . (microTime(true)-$startTS) . "seconds \n";

?>
