$(function() {

  Dropzone.autoDiscover = false;

  function validateIP(ipaddr) {
    if (!ipaddr)
      return false;

    if (ipaddr == "0.0.0.0" || ipaddr == "255.255.255.255")
      return false;

    let pat = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/;
    let parts = ipaddr.match(pat);
    if (parts == null || parts.length != 5)
      return false;

    parts.shift();
    if (_.any(parts, function(p) { return parseInt(p) > 255; }))
      return false;

    return true;
  }

  let progressUIMixin = { //{{{1
    data: function() {
      return {spinner: null};
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },
    },
  }; //}}}1

  //define components 
  let httpsCertUploadComp = { //{{{1
    props: ['server_url', 'accepted_files', 'param_name'],
    template: '#https_cert_upload_template',
    data: function() {
      return {};
    },
    mounted: function() {
      let self = this;
      if (!this.server_url) {
        console.debug("invalid 'server_url'");
        return;
      }
      if (!this.accepted_files) {
        console.debug("invalid 'accepted_files'");
        return;
      }

      if (!this.param_name) {
        console.debug("invalid 'param_name'"); 
        return;
      }

      $(this.$el).dropzone({
        url: this.server_url,
        paramName: this.param_name,
        maxFileSize: 1,
        maxFiles: 1,
        clickable: true, 
        createImageThumbnails: false,
        acceptedFiles: this.accepted_files,
        addedfile: _.bind(self.onFileAdded, self),
      });
    },
    methods: {
      getDropzone: function() {
        return Dropzone.forElement(this.$el);
      },
      onFileAdded: function(file) {
        let dz = this.getDropzone();
        if (!dz.previewContainer)
          dz.previewContainer = this.$el;

        _.every(_.initial(dz.files, dz.options.maxFiles), function(f) {
          dz.removeFile(f);
        });

        _.bind(dz.defaultOptions.addedfile, dz)(file);
      },
    },
  }; //}}}1

  let webConfigComp = { //{{{1
    mixins: [progressUIMixin],
    template: '#web_config_template',
    data: function() {
      return {
        cert: '',
        certError: false,
      };
    },
    methods: {
      downloadHTTPSCert: function() {
        let self = this;
        $.ajax({
          url: 'service_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            action: 'download_https_cert',
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect) {
              window.location.href = resp.redirect;
              return;
            }

            if (resp.error) 
              msgbus.$emit("alert", "Error", resp.error, 'error');
          },
          error: function(data, status) {
            console.warn("failed to create client cert: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
      applyHTTPSCert: function() {
        let self = this;
        $.ajax({
          url: 'service_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            action: 'apply_https_certs'
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              window.location.href = resp.redirect;

            if (resp.error) {
              msgbus.$emit("alert", "Error", resp.error, 'error');
              return;
            }

            msgbus.$emit("alert", "Success", "Uploaded cert and key are applied.", 'success');
          },
          error: function(data, status) {
            console.warn("failed to apply uploaded certificates: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
    },
    components: {
      'https-cert-upload': httpsCertUploadComp,
    },
  }; //}}}1

  let openvpnServerComp = { //{{{1
    template: '#openvpn_server_template',
    mixins: [progressUIMixin],
    data: function() {
      return {
        proto: 'udp',
        port: 1194,
        ip: '10.8.0.0',
        netmask: '255.255.255.0',

        ca: false,
        serverCert: false,

        portError: false,
        ipError: false,
        netmaskError: false,

        caError: false,
        serverCertError: false,

        // data for modal dialog
        caCN: '',
        serverCertCN: '',

        caCNError: false,
        serverCertCNError: false,

      };
    },
    mounted: function() {
      let self = this;
      $.ajax({
        url: 'openvpn_controller.php',
        type: 'GET',
        dataType: 'json',
        beforeSend: function() {
          self.getSpinner().spin(self.$el);
        },
        success: function(resp) {
          if (resp.redirect)
            redirect(resp.redirect);

          if (resp.error) {
            msgbus.$emit("alert", "Error", resp.error, 'error');
            return;
          }

          if (!resp.data) {
            console.warn("no openvpn data returned");
            return;
          }

          if (_.has(resp.data, 'port'))
            self.port = resp.data['port'];
          if (_.has(resp.data, 'proto'))
            self.proto = resp.data['proto'];

          if (_.has(resp.data, 'server'))
            self.server = resp.data['server'];

          if (_.has(resp.data, 'ca'))
            self.ca = resp.data['ca'];
          if (_.has(resp.data, 'serverCert'))
            self.serverCert = resp.data['serverCert'];

        },
        error: function(data, status) {
          console.warn("get error response from openvpn_controller");
          console.warn(data);
        },
        complete: function() {
          self.getSpinner().stop();
        }
      });
    },
    computed: {
      server: {
        get: function() {
          return this.ip + " " + this.netmask;
        },
        set: function(newValue) {
          let parts = newValue.split(' ');
          if (parts.length < 2) 
            return;
          this.ip = parts[0].trim();
          this.netmask = parts[1].trim();
        }
      },
    },
    methods: {
      collectCAData: function() {
        this.$refs.gen_ca_modal.show();
      },
      collectServerCertData: function() {
        if (!this.ca || this.caError)
          msgbus.$emit("alert", "Error", "Can not generate server certificate before CA is ready.", 'error');
        else
          this.$refs.gen_cert_modal.show();
      },
      generateCA: function() {
        this.caCNError = !/^[a-zA-Z0-9-_]{1,64}$/.test(this.caCN);
        if (this.caCNError)
          return;

        let self = this;
        $.ajax({
          url: 'openvpn_controller.php',
          type: 'POST',
          dataType: 'json',
          timeout: 10*60*1000,
          data: {
            action: 'genCA',
            cn: this.caCN
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.error) {
              msgbus.$emit("alert", "Error", resp.error, 'error');
            } else {
              self.ca = true;
              self.caError = false;
              self.$refs.gen_ca_modal.hide();
            }
          },
          error: function(data, status) {
            console.warn("failed to create CA: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
      generateServerCert: function() {
        this.serverCertCNError = !/^[a-zA-Z0-9-_]{1,64}$/.test(this.serverCertCN);
        if (this.serverCertCNError)
          return;

        let self = this;
        $.ajax({
          url: 'openvpn_controller.php',
          type: 'POST',
          dataType: 'json',
          data: {
            action: 'genServerCert',
            cn: this.serverCertCN,
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.error) {
              msgbus.$emit("alert", "Error", resp.error, 'error');
            }
            else {
              self.serverCert = true;
              self.serverCertError = false;
              self.$refs.gen_cert_modal.hide();
            }
          },
          error: function(data, status) {
            console.warn("failed to create server cert: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },

      validateInputs: function() {
        this.portError = this.port <= 0 || this.port >= 65535;
        this.ipError = !validateIP(this.ip);
        this.netmaskError = !validateIP(this.netmask);

        this.caError = !this.ca;
        this.serverCertError = !this.serverCert;

        return !(this.portError || this.ipError || this.netmaskError || this.caError || this.serverCertError);
      },
      saveOpenVPNConfig: function() {
        if (!this.validateInputs()) {
          console.debug("invalid input");
          return;
        }

        let self = this;
        $.ajax({
          url: 'openvpn_controller.php',
          type: 'POST',
          dataType: 'json',
          data: {
            action: 'saveConfig',
            port: this.port,
            proto: this.proto,
            server: this.server
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.error)
              msgbus.$emit("alert", "Error", resp.error, 'error');
            else
              msgbus.$emit("alert", "Done", 'config is saved', 'success');

          },
          error: function(data, status) {
            console.warn("failed to save openvpn config");
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
    },
    components: {
      'modalComp': modalComp,
    },
  }; //}}}1

  let openvpnClientOVPNGeneratorComp = { //{{{1
    mixins: [progressUIMixin],
    template: '#openvpn_client_ovpn_generation_template',
    data: function() {
      return {
        clientName: '',
        ip: '',
        port: 1194,

        clientNameError: false,
        ipError: false,
        portError: false,
      };
    },
    computed: {
      remote: {
        get: function() {
          return this.ip + " " + this.port;
        },
        set: function(newValue) {
        },
      }
    },
    methods: {
      validateInputs: function() {
        this.ipError = !validateIP(this.ip);
        this.clientNameError = !/^[0-9a-zA-Z-_]{1,64}$/.test(this.clientName);
        this.portError = this.port <= 0 || this.port >= 65535;

        return !(this.ipError || this.clientNameError || this.portError);
      },
      generateClientOVPN: function() {
        if (!this.validateInputs()) {
          console.debug("invalid input");
          return;
        }

        let self = this;
        $.ajax({
          url: 'openvpn_controller.php',
          type: 'POST',
          dataType: 'json',
          data: {
            action: 'genClientCert',
            cn: this.clientName,
            remote: this.remote,
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect) {
              window.location.href = resp.redirect;
              return;
            }

            if (resp.error) 
              msgbus.$emit("alert", "Error", resp.error, 'error');
          },
          error: function(data, status) {
            console.warn("failed to create client cert: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      }
    }
  }; //}}}1

  let openvpnConfigComp = { //{{{1
    mixins: [progressUIMixin],
    template: '#openvpn_config_template',
    data: function() {
      return {
      };
    },
    methods: {
    },
    components: {
      'openvpn-server': openvpnServerComp,
      'openvpn-client-ovpn-generator': openvpnClientOVPNGeneratorComp,
    }
  }; //}}}1

  let authKeyEditorComp = { //{{{1
    mixins: [progressUIMixin],
    template: '#auth_key_editor_template',
    props: ['editing', 'initial_expired_at', 'initial_note'],
    datepickr: null,
    data: function() {
      return {
        note: this.initial_note,
        expired_at: this.initial_expired_at,
        
        note_error: false,
      };
    },
    methods: {
      showHideCalendar: function(evt) {
        let self = this;
        if (this.datepickr) {
          this.datepickr.set('defaultDate', this.expired_at);
        } else {
          let today = new Date();
          this.datepickr = flatpickr(evt.target, {
            minDate: today,
            defaultDate: self.expired_at,
            onChange: function(selectedDates, dateStr, instance) {
              self.expired_at = dateStr;
            },
          });
        }
        this.datepickr.toggle();
      },
      saveModification: function() {
        if (this.expired_at.length <= 0)
          return;
        this.note_error = this.note.length > 64;
        if (this.note_error)
          return;

        this.$emit("editCompleted", this.note, this.expired_at);
      },
    },
  }; //}}}1

  let authKeyComp = { //{{{1
    mixins: [progressUIMixin],
    template: '#auth_key_config_template',
    datepickr: null,
    data: function() {
      let today = new Date();
      let default_dt = new Date(today.getFullYear()+1, today.getMonth(), today.getDate());
      return {
        keys: [],
        expired_at: flatpickr.formatDate(default_dt, "Y-m-d"),
        note: "",

        expired_at_error: false,
        note_error: false,
      };
    },
    mounted: function() {
      let self = this;
      $.ajax({
        url: '../../app/auth_key_controller.php',
        type: 'GET',
        dataType: 'json',
        beforeSend: function() {
          self.getSpinner().spin(self.$el);
        },
        success: function(resp) {
          if (resp.redirect)
            redirect(resp.redirect);

          if (resp.error) {
            msgbus.$emit("alert", "Error", resp.error, 'error');
            return;
          }

          if (_.isArray(resp.data))
            self.keys = resp.data;
        },
        error: function(data, status) {
          console.warn("failed to load existing auth keys: " + status);
          console.warn(data);
        },
        complete: function() {
          self.getSpinner().stop();
        }
      });
    },
    methods: {
      showCalendar: function(evt) {
        if (!this.datepickr) {
          let today = new Date();
          this.datepickr = flatpickr(evt.target, {
            minDate: today,
            defaultDate: new Date(today.getFullYear()+1, today.getMonth(), today.getDate()),
            onChange: _.bind(this.onCalendarChanged, this),
          });
        }
        this.datepickr.open();
      },
      onCalendarChanged: function(selectedDates, dateStr, instance) {
        this.expired_at = dateStr;
        this.validateInputs();
      },
      sortBy: function(col_name) {
        let self = this;
        _.each(this.keys, function(key) {
          if (_.has(key, 'editing'))
            self.$set(key, 'editing', false);
        });
        this.keys = _.sortBy(this.keys, function(key) {
          return key[col_name];
        });
      },
      validateInputs: function() {
        this.expired_at_error = this.expired_at.length == 0;
        this.note_error = this.note.length > 64;

        return !(this.expired_at_error || this.note_error);
      },
      createAuthKey: function() {
        if (!this.validateInputs())
          return; 

        let self = this;
        $.ajax({
          url: '../../app/auth_key_controller.php',
          type: 'post',
          dataType: 'json',
          data: {
            action: "create",
            note: self.note,
            expired_at: self.expired_at,
          },
          beforesend: function() {
            self.getspinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.error) {
              msgbus.$emit("alert", "error", resp.error, 'error');
              return;
            }

            if (resp.new_key) {
              self.keys.push({
                key: resp.new_key, 
                note: self.note, 
                expired_at: self.expired_at,
                editing: false
              });
            }
          },
          error: function(data, status) {
            console.warn("failed to load existing auth keys: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },

      toggleAuthKeyEditor: function(key) {
        let val = true;
        if (_.has(key, 'editing'))
          val = !key.editing;
        this.$set(key, 'editing', val);
      },
      delAuthKey: function(key, index) {
        let self = this;
        $.ajax({
          url: '../../app/auth_key_controller.php',
          type: 'post',
          dataType: 'json',
          data: {
            action: "delete",
            key: key.key,
          },
          beforesend: function() {
            self.getspinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.error) {
              msgbus.$emit("alert", "error", resp.error, 'error');
              return;
            }
            
            self.$delete(self.keys, index);
          },
          error: function(data, status) {
            console.warn("failed to delete auth key: " + key.key + " status: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
      onEditCompleted: function(key, note, expired_at) {
        let self = this;
        $.ajax({
          url: '../../app/auth_key_controller.php',
          type: 'post',
          dataType: 'json',
          data: {
            action: "update",
            key: key.key,
            note: note,
            expired_at: expired_at,
          },
          beforesend: function() {
            self.getspinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.error) {
              msgbus.$emit("alert", "error", resp.error, 'error');
              return;
            }
            
            self.$set(key, 'note', note);
            self.$set(key, 'expired_at', expired_at);
            self.$set(key, 'editing', false);
          },
          error: function(data, status) {
            console.warn("failed to modify auth key: " + key.key + " status: " + status);
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      }
    },
    components: {
      'auth-key-editor': authKeyEditorComp,
    },
  }; //}}}1

  let dataAPIConfigComp = { //{{{1
    mixins: [progressUIMixin],
    template: '#data_api_config_template',
    data: function() {
      return {};
    },
    components: {
      'auth-key-config': authKeyComp,
    },
  }; //}}}1

  //create&init root object
  var app = new Vue({ //{{{1
    el: '#service_config_content',
    data: function() {
      let sections = [];
      if (window.iot_platform == 'FS')
        sections = [
          {sec_id: 'web', name: 'Web'},
          {sec_id: 'openvpn', name: 'OpenVPN'},
          {sec_id: 'data-api', name: 'DataAPI'},
        ];
      else if (window.iot_platform == 'FW')
        sections = [
          {sec_id: 'web', name: 'Web'},
          {sec_id: 'data-api', name: 'DataAPI'},
        ];
      else
        sections = [];

      return {
        sections: sections,
        active_sec_id: _.first(sections).sec_id,
      }
    },
    methods: {
      isSectionActive: function(sec) { return sec.sec_id == this.active_sec_id; }
    },
    components: {
      'web': webConfigComp,
      'openvpn': openvpnConfigComp,
      'data-api': dataAPIConfigComp,
      'alert': alertComp,
    }
  }); //}}}1

});

// vim: foldmethod=marker
